/*
 * Copyright (C) 2003  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "PDFContentView.h"
#include "ExtendedImageView.h"
#include "ViewPDF.h"

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSNotification.h>
#include <Foundation/NSValue.h>
#include <AppKit/NSClipView.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSScroller.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSImageView.h>


/*
 * Non-Public methods.
 */
@interface PDFContentView(Private)
- (void) _createScrollView;
- (void) _createImageView;
- (void) _createMessageView;
- (void) _updateScrollers;
- (void) _notifyDragScroll: (id)notification;
@end

/*
 * This view displays an image within a ScrollView. Additionaly,
 * some controls are provided to manipulate the display properties
 * (eg. the zoom factor).
 */
@implementation PDFContentView

/*
 * Designated initializer.
 */
- (id) initWithFrame: (NSRect)frame
{
   if ((self = [super initWithFrame: frame]))
   {
      [self _createScrollView];
      [self _createImageView];
      [self _createMessageView];

      [self setAutoresizesSubviews: YES];
      [self setAutoresizingMask: (NSViewWidthSizable |
                                  NSViewHeightSizable)];

      messageDisplayed = NO;
      imageRep  = nil;
      pageNum = 1;
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc PDFContentView, retain count is %d", [self retainCount]);

   [[NSNotificationCenter defaultCenter] removeObserver: self];

   [scrollView release];
   [imageView release];
   [messageView release];
   [imageRep release];
   
   [super dealloc];
}


/*
 * Returns the NSImageView that is embedded in this
 * view.
 */
- (id) imageView
{
   return imageView;
}


/*
 * When resized, center the image in the scrollview.
 */
- (void) resizeWithOldSuperviewSize: (NSSize)oldBoundsSize 
{
   [super resizeWithOldSuperviewSize: oldBoundsSize];
}



/*
 * Set the document to be displayed. Note that the
 * document is _not_ retained by this method.
 */
- (void) setDocument: (PDFDocument*)_pdfDoc
{
   NSAssert(_pdfDoc, @"pdfDoc is nil");
   pdfDoc = _pdfDoc;

   NSSize   imageSize;
   NSImage* image;

   imageSize = NSMakeSize([pdfDoc pageWidth: 1],
                          [pdfDoc pageHeight: 1]);
      
   imageRep = [[PDFImageRep alloc] initWithDocument: pdfDoc];
   [imageRep setSize: imageSize];

   image = [[[NSImage alloc] init] autorelease];
   [image addRepresentation: imageRep];

   [image setCacheMode: NSImageCacheAlways];
   [image setBackgroundColor: [NSColor whiteColor]];

   NSAssert([self imageView], @"No ImageView");
   [[self imageView] setImage: image];
   // image is retained by the imageview
}


/*
 * Set the page to be displayed. The content of this
 * page is scaled by the specified factor when drawn
 * to the screen. A factor of 1.0 is equal to 100
 * percent. Note that updateView has to be called
 * to let the changes take effect.
 */
- (void) setPage: (int)_pageNum scaledBy: (float)factor
{
   NSSize scaledSize;
   NSSize unscaledSize;
   double hDPI, vDPI;

   NSAssert(pdfDoc, @"No PDFDocument is associated with the content view");
   
   unscaledSize = NSMakeSize([pdfDoc pageWidth: pageNum],
                             [pdfDoc pageHeight: pageNum]);
   
   scaledSize = NSMakeSize(unscaledSize.width * factor,
                           unscaledSize.height * factor);
   
   hDPI = (scaledSize.width / unscaledSize.width) * PDFBaseResolution;
   vDPI = (scaledSize.height / unscaledSize.height) * PDFBaseResolution;
   
   pageNum = _pageNum;
   [imageRep setPageNum: pageNum];
   [imageRep setResolution: (hDPI < vDPI ? hDPI : vDPI)];
   [[[self imageView] image] recache];
   [[[self imageView] image] setSize: [imageRep size]];

   [self removeMessage];
}


/*
 * Set the size of the internal image view. This can be used
 * to scale the displayed page becaue  the internal image view
 * is configured to scale it's image when resized.
 */
- (void) setPageSize: (NSSize)size
{
   NSRect viewFrame;

   viewFrame = NSMakeRect(0, 0, size.width, size.height);
   //?? [imageView setFrame: viewFrame];
}


/*
 * Repaint the the view, update the scrollers ...
 */
- (void) updateView
{
   NSLog(@"[PDFContentView updateView]");

   [self sizeToFitPage];
   [imageView setNeedsDisplay: YES];
   [self setNeedsDisplay: YES];
}


/*
 * Get the embedded image scroll view.
 */
- (ExtendedScrollView*) scrollView
{
   return scrollView;
}


/*
 * Display a message in the internal ImageView.
 * This can be used to display informations for
 * the user (eg. when work is progress). The
 * message will disapper when removeMessage
 * or setImage is called.
 */
- (void) displayMessage: (NSString*)message
{
   NSRect mFrame;
   NSRect iFrame;

   [messageView setStringValue: message];
   [messageView sizeToFit];
   
   iFrame = [imageView frame];
   mFrame = [messageView frame];

   mFrame = NSMakeRect(NSMidX(iFrame) - (NSWidth(mFrame) / 2),
                       NSMidY(iFrame) - (NSHeight(mFrame) / 2),
                       NSWidth(mFrame),
                       NSHeight(mFrame));

   [messageView setFrame: mFrame];
                             
   [imageView addSubview: messageView];
   messageDisplayed = YES;
}


/*
 * Remove a displayed message from the ImageView.
 * Use this only if displayMessage has been
 * called before.
 */
- (void) removeMessage
{
   if (messageDisplayed)
   {
      [messageView removeFromSuperview];
      messageDisplayed = NO;
   }
}


/*
 * Resize the internal ImageView so that is shows the
 * complete current page.
 */
- (void) sizeToFitPage
{
   NSSize imageSize;
   NSRect viewFrame;

   imageSize = [[imageView image] size];
   viewFrame = NSMakeRect(0, 0, imageSize.width, imageSize.height);

   [imageView setFrame: viewFrame];
}


/*
 * Create the ScrollView that holds the image.
 */
- (void) _createScrollView
{
   NSRect scrollViewFrame;

   scrollViewFrame = [self frame];
   scrollView = [[ExtendedScrollView alloc] initWithFrame: scrollViewFrame];

   [scrollView setHasHorizontalScroller: YES];
   [scrollView setHasVerticalScroller: YES];
   [scrollView setAutoresizesSubviews: YES];
   [scrollView setAutoresizingMask: (NSViewWidthSizable |
                                     NSViewHeightSizable)];
   [scrollView setDrawsBackground: NO];

   [self addSubview: scrollView];
}


- (void) _createImageView
{
   NSRect   frame;

   frame = NSMakeRect(0, 0,
                      [scrollView contentSize].width,
                      [scrollView contentSize].height);
   imageView = [[ExtendedImageView alloc] initWithFrame: frame];
   //imageView = [[NSImageView alloc] initWithFrame: frame];
   [imageView setImageAlignment: NSImageAlignCenter];
   //[imageView setImageScaling: NSScaleNone];
   [imageView setImageScaling: NSScaleProportionally];

   [scrollView setDocumentView: imageView];

   // get informed about scrolling through mouse dragging
   [[NSNotificationCenter defaultCenter] addObserver: self
                                         selector: @selector(_notifyDragScroll:)
                                         name: N_ScrollingRequested
                                         object: imageView];
}


- (void) _createMessageView
{
   NSRect   frame;

   frame = NSMakeRect(0, 0, 0, 0);
   messageView = [[NSTextField alloc] initWithFrame: frame];
   [messageView setFont: [NSFont boldSystemFontOfSize: 16.0]];
   [messageView setBordered: NO];
   [messageView setBezeled: NO];
   [messageView setSelectable: NO];
   [messageView setDrawsBackground: NO];
   [messageView setEditable: NO];
}


- (void) _notifyDragScroll: (id)notification
{
   NSPoint  newOrigin;
   NSSize   scrollAmount;
   NSSize   contentSize;
   NSRect   vRect;

   scrollAmount =
      [[[notification userInfo] objectForKey: UserInfoKeyScrollAmount] sizeValue];

   vRect       = [scrollView documentVisibleRect];
   contentSize = [scrollView contentSize];

   newOrigin = NSMakePoint(vRect.origin.x + scrollAmount.width,
                           vRect.origin.y + scrollAmount.height);

   // check bounds
   /*
   if (newOrigin.x < 0)
   {
      newOrigin.x = 0;
   }
   if (newOrigin. x > (contentSize.height - vRect.size.height))
   {
      newOrigin.x = contentSize.height - vRect.size.height;
   }
   */

   [[scrollView contentView] scrollToPoint:
       [[scrollView contentView] constrainScrollPoint:newOrigin]];
}

@end
