/*
 * Copyright (C) 2004  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef _H_SEARCHSERVICE
#define _H_SEARCHSERVICE

#include <Foundation/NSObject.h>
#include <Foundation/NSLock.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSTimer.h>
#include <PDFKit/PDFDocument.h>
#include "Match.h"

@protocol SearchServiceDelegate;

/**
 * A SearchService can search the contents of a PDFDocument
 * for the occurence of a text fragment. Searching is performed
 * in the background.
 *
 * A SearchService object cannot be reused.
 */
@interface SearchService : NSObject
{
   NSString*                  text;
   int                        page;
   int                        pageCount;
   NSRect                     position;
   NSTimer*                   resultPicker;
   BOOL                       stopPicking;
   BOOL                       releaseWhenDone;
   NSMutableArray*            results;
   NSLock*                    resultsLock;
   PDFDocument*               document;
   id                         delegate;
   BOOL                       shouldStop;
}

/** Initialize the receiver and go. The delegate object gets
 *  notified about the search progress.  */
- (id) initWithText: (NSString*)aTextFragment
        forDocument: (PDFDocument*)aDocument
     startingAtPage: (int)aPage
         atPosition: (NSRect)aPagePosition
           delegate: (id)aDelegate;

/** Set the receiver's delegate object.  */
- (void) setDelegate: (id)aDelegate;

/** Determine if the receiver is running.  */
- (BOOL) isRunning;

/** Stop a running search. Raises an NSInternalInconsitencyException
 *  if the receiver has already been stopped. It is not guaranteed
 *  that the search terminated when this method returns. You should
 *  consider the search in progress as long as the delegate does not
 *  get the serviceDidFinishSearch message. The flag specified whether
 *  the service should release itself when it is done.  */
- (void) stopAndReleaseWhenDone: (BOOL)flag;

/** Get the document that is searched by this service.  */
- (PDFDocument*) document;

/** Get the number of pages already searched by the receiver.  */
- (int) countSearchedPages;

@end


/**
 * A SearchServiceDelegate will be notified by a SearchService
 * about the progress of a search.
 */
@protocol SearchServiceDelegate

/** The service is about to start the search.  */
- (void) serviceWillStartSearch: (SearchService*)aService;

/** A service is about to start searching a certain page. The
 *  delegate can return NO to skip this page. Otherwise, return
 *  YES.  */
- (BOOL) service: (SearchService*)aService willStartSearchingPage: (int)aPage;

/** A service did complete searching a certain page.  */
- (void) service: (SearchService*)aService didFinishedSearchingPage: (int)aPage;

/** A service found one ore more matches in the searched document.  */
- (void) service: (SearchService*)aService didFoundMatches: (NSArray*)matches;

/** The service did complete the search. It's safe to release the 
 *  service now.  */
- (void) serviceDidFinishSearch: (SearchService*)aService;

@end

#endif
