/* -*- C++ -*-
 
  This file is part of ViPEC
  Copyright (C) 1991-2000 Johan Rossouw (jrossouw@alcatel.altech.co.za)
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
*/

#include <CircuitLine.h>

#include <Setup.h>
#include <Logger.h>
#include <Schematic.h>
#include <CircuitNode.h>

#include <qpoint.h>
#include <qpainter.h>

#include <iostream>

using namespace std;

//-----------------------------------------------------------------
CircuitLine::CircuitLine(CircuitNode& start, CircuitNode& stop)
    : start_(&start),
    stop_(&stop)
{
  start.addLine(this);
  stop.addLine(this);
}

//-----------------------------------------------------------------
CircuitLine::~CircuitLine()
{
  start_->removeLine(this);
  stop_->removeLine(this);
}

//-----------------------------------------------------------------
void CircuitLine::draw(QPainter* p)
{
  p->save();
  if ( isSelected() )
    {
      p->setPen( Qt::red );
    }
  else
    {
      p->setPen( Qt::blue );
    }
  p->moveTo( start_->pos() );
  p->lineTo( stop_->pos() );
  p->restore();
}

//-----------------------------------------------------------------
QPoint& CircuitLine::startPoint()
{
  return start_->pos();
}

//-----------------------------------------------------------------
const QPoint& CircuitLine::startPoint() const
  {
    return start_->pos();
  }

//-----------------------------------------------------------------
QPoint& CircuitLine::endPoint()
{
  return stop_->pos();
}

//-----------------------------------------------------------------
const QPoint& CircuitLine::endPoint() const
  {
    return stop_->pos();
  }

//-----------------------------------------------------------------
CircuitNode* CircuitLine::start() const
  {
    return start_;
  }

//-----------------------------------------------------------------
CircuitNode* CircuitLine::stop() const
  {
    return stop_;
  }

//-----------------------------------------------------------------
bool CircuitLine::replaceNode(CircuitNode* oldNode, CircuitNode* newNode)
{
  ASSERT( oldNode );
  ASSERT( newNode );
  bool replaced = FALSE;
  if ( oldNode == start_ )
    {
      start_->removeLine( this );
      start_ = newNode;
      replaced = TRUE;
    }
  else if ( oldNode == stop_ )
    {
      stop_->removeLine( this );
      stop_ = newNode;
      replaced = TRUE;
    }
  if ( replaced )
    {
      QString message = QString("Replaced node at %1 with node at %2").arg((ulong)oldNode,0,16).arg((ulong)newNode,0,16);
      Logger::debug(message);
    }
  return replaced;
}

//-----------------------------------------------------------------
void CircuitLine::breakAndInsertNode( Schematic* schematic, CircuitNode* node )
{
  stop_->removeLine( this );
  CircuitLine* line = new CircuitLine( *node, *stop_ );
  stop_ = node;
  node->addLine( this );
  schematic->addLine( line );
}
