/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <string>
#include <vector>
#include <iostream>
#include <fstream>
/* -- */
#include "parser/Line.hpp"
/* -- */
using namespace std;

Line::Line() : _line_count(0) {
}

Line::Line(const Line &l) : _tokens(l._tokens), _line_count(l._line_count) {
}

Line::Line(const string &string_line) {
    fill_line(string_line);
}

Line::~Line(){
    if(!_tokens.empty()){
        _tokens.clear();
    }
}

void Line::fill_line(const std::string &string_line) {
    // We split the string_line in tokens
    string current_token = "";

    char char_read = '\0';
    const int size_max = string_line.size();

    int index_begin_string = 0;

    for(int i = 0 ; char_read != '\n' ; ++ i) {

        if(i > size_max) {
            break;
        }

        char_read = string_line[i];

        // End of the line
        if(char_read == '\n') { /* Don't forget the last token */
            current_token = string_line.substr(index_begin_string, i-index_begin_string);
            if(!current_token.empty()) {
                _tokens.push_back(current_token);
                current_token.clear(); // Reinitialisation
            }
            continue;
        }

        // '%' is a "special" token used to specify (Pajé format) that wa are in a definition line
        if(char_read == '%') {
            _tokens.push_back("%");
            current_token.clear(); // Reinitialisation
            index_begin_string = i+1;
            continue;
        }

        // Composite token
        if(char_read == '\'' || char_read == '"') {
            int composite_token_size = build_composite_token(string_line, i);
            i += composite_token_size;
            current_token.clear(); // Reinitialisation
            index_begin_string = i+1;
            continue;
        }

        if((char_read == ' ') || (char_read == '\t')){
            // We store the token
            current_token = string_line.substr(index_begin_string, i-index_begin_string);
            if(!current_token.empty()) {
                _tokens.push_back(current_token);
                current_token.clear(); // Reinitialisation
            }
            index_begin_string = i+1;
            continue;
        }

    }

    if(!current_token.empty()) {
        _tokens.push_back(current_token);
        current_token.clear(); // Reinitialisation
    }
}


bool Line::starts_with(const string &s) const {
    if(!_tokens.empty())
        return _tokens[0] == s;
    else // tokens is empty
        return false;
}

bool Line::item(unsigned int i, string &e) const {
    if (i >= _tokens.size())
        return false;
    e = _tokens[i];
    return true;
}

unsigned int Line::length() const {
    return _tokens.size();
}

void Line::set_line_number(const unsigned int n) {
    _line_count = n;
}

void Line::clear(){
    _tokens.clear();
}

void Line::print() const {
    const unsigned int token_size = _tokens.size();
    cout << "-" ;
    for(unsigned int i = 0 ; i < token_size ; i ++)
        cout << _tokens[i] << " " ;
    cout << "-" << endl;
}

unsigned int Line::get_line_count() const {
    return _line_count;
}

int Line::build_composite_token(const string &line, int index) {
    // line[index] corresponds to the delimiter ie " or '
    // We search the first one after this one
    const unsigned int delimiter = line.find_first_of(line[index], index+1);

    if(delimiter == string::npos) {
        throw "overflow in buffer";
    }

    const string token = line.substr(index+1, delimiter-index-1);

    _tokens.push_back(token);

    return delimiter-index;
}
