(**
   Implements a load-like percentage display.
**)

MODULE VOLoad;

(*
    Implements a load-like percentage display.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       F   := VOFrame,
       G   := VOGUIObject,
       O   := VOObject,
       P   := VOPrefs,
       S   := VOSegment,
       U   := VOUtil,
       V   := VOValue,

       co  := IntStr,
       str := Strings;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (P.PrefsDesc)
                 frame* : LONGINT; (* the frame to use *)
               END;

  Load*     = POINTER TO LoadDesc;
  LoadDesc* = RECORD (G.ImageDesc)
                 prefs    : Prefs;
                 frame    : F.Frame;
                 labelTxt : S.Segment;
                 labelSfx : U.Text;
                 top,
                 bottom   : LONGINT;
                 onColor,
                 maxColor,
                 mediumColor,
                 offColor : LONGINT;
                 current  : V.ValueModel;
                 framed,
                 vertical,
                 labeled  : BOOLEAN;
               END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DIn;
  END Init;

  PROCEDURE (l : Load) Init*;

  BEGIN
    l.Init^;

    l.prefs:=prefs;

    l.top:=100;
    l.bottom:=0;

    l.current:=NIL;

    l.labelTxt:=NIL;
    l.labelSfx:=NIL;

    l.vertical:=TRUE;
    l.framed:=TRUE;
    l.labeled:=FALSE;

    l.onColor:=D.shineColor;
    l.maxColor:=l.onColor;
    l.mediumColor:=l.onColor;
    l.offColor:=D.shadowColor;
    l.SetBackground(D.blackColor);
  END Init;

  (**
    Tell, if the load should be display horizontally or
    vertically.

    Default is vertical.
  **)

  PROCEDURE (l : Load) SetVertical*(vertical : BOOLEAN);

  BEGIN
    l.vertical:=vertical;
    IF l.visible THEN
      l.Redraw;
    END;
  END SetVertical;

  (**
    Tell, if the object should display its label.
  **)

  PROCEDURE (l : Load) ShowLabel*(useLabel : BOOLEAN);

  BEGIN
    l.labeled:=useLabel;
    IF l.visible THEN
      l.Redraw;
    END;
  END ShowLabel;

  (**
    Tell, if the object should be framed. Defaults to TRUE.
  **)

  PROCEDURE (l : Load) ShowFrame*(framed : BOOLEAN);

  BEGIN
    (* We cannot switch back to using no frame if we already generated one *)
    ASSERT(l.frame=NIL);

    l.framed:=framed;
  END ShowFrame;

  (**
    Set the integer model which represents the current value.
  **)

  PROCEDURE (l : Load) SetModel*(model : V.ValueModel);

  BEGIN
    IF l.current#NIL THEN
      l.UnattachModel(l.current);
    END;
    l.current:=model;
    l.AttachModel(l.current);
    IF l.visible THEN
      l.Redraw;
    END;
  END SetModel;

  PROCEDURE (l : Load) CalcSize*(display : D.Display);

  VAR
    window : D.Window;

  BEGIN
    IF ~(G.inited IN l.flags) THEN
      IF display.colorMode=D.colorMode THEN
        l.onColor:=display.AllocateNamedColor("green",l.onColor);
        l.maxColor:=display.AllocateNamedColor("red",l.maxColor);
        l.mediumColor:=display.AllocateNamedColor("yellow",l.mediumColor);
      ELSIF display.colorMode=D.monochromeMode THEN
        l.onColor:=D.blackColor;
        l.maxColor:=l.onColor;
        l.mediumColor:=l.onColor;
        l.offColor:=D.whiteColor;
        l.SetBackground(D.whiteColor);
      END;
    END;

    NEW(l.frame);
    l.frame.Init;

    l.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    IF l.framed THEN
      l.frame.SetInternalFrame(l.prefs.frame);
    ELSE
      l.frame.SetInternalFrame(F.none);
    END;
    l.frame.CalcSize(display);

    l.width:=l.frame.leftBorder+2*display.spaceWidth+l.frame.rightBorder;
    l.height:=l.frame.topBorder+2*display.spaceHeight+l.frame.bottomBorder;

    IF l.labeled THEN
      NEW(l.labelTxt);
      l.labelTxt.Init;
      l.labelTxt.SetStringWidth(3);
      l.CopyBackground(l.labelTxt);
      l.labelTxt.SetForeground(l.onColor);
      l.labelTxt.CalcSize(display);

      IF l.vertical THEN
        l.width:=G.MaxLong(l.width,l.labelTxt.oWidth+l.frame.leftBorder+l.frame.rightBorder);
        INC(l.height,display.spaceHeight DIV 2 + l.labelTxt.oHeight);
      ELSE
        INC(l.width,display.spaceWidth DIV 2 + l.labelTxt.oWidth);
        l.height:=G.MaxLong(l.height,l.labelTxt.oHeight+l.frame.topBorder+l.frame.bottomBorder);
      END;
    END;

    l.minWidth:=l.width;
    l.minHeight:=l.height;

    window:=display.GetNewWindow();
    window.AddFreeList(l);

    l.CalcSize^(display);
  END CalcSize;

  (**
    Set the bottom and top value the object should use for displaying.
  **)

  PROCEDURE (l : Load) SetRange*(bottom,top : LONGINT);

  BEGIN
    IF (bottom#l.bottom) OR (top#l.top) THEN
      l.bottom:=bottom;
      l.top:=top;

      IF l.visible THEN
        l.Redraw;
      END;
    END;
  END SetRange;

  (**
    Refresh loadmeter and text.
  **)

  PROCEDURE (l : Load) DrawLoad;

  VAR
    boxWidth,
    boxHeight,
    lines,
    current,
    top,left,
    mark,help,
    maxMark,
    mediumMark,
    value      : LONGINT;
    text       :ARRAY 4 OF CHAR;

  BEGIN
    IF l.current.IsNull() THEN
      value:=l.bottom;
    ELSE
      value:=l.current.GetLongint();
    END;

    boxHeight:=l.height-l.frame.topBorder-l.frame.bottomBorder;
    boxWidth:=l.width-l.frame.leftBorder-l.frame.rightBorder;

    IF l.labeled THEN
      IF l.vertical THEN
        DEC(boxHeight,l.display.spaceHeight DIV 2+l.labelTxt.oHeight);
      ELSE
        DEC(boxWidth,l.display.spaceWidth DIV 2+l.labelTxt.oWidth);
      END;
    END;

    IF l.vertical THEN
      lines:=boxHeight DIV 4;
    ELSE
      lines:=boxWidth DIV 4;
    END;

    help:=((value-l.bottom)*100) DIV (l.top-l.bottom+1);

    mark:=(help*lines) DIV 100;
    IF (help*lines) MOD 100>=50 THEN
      INC(mark);
    END;

    maxMark:=(75*lines) DIV 100;
    IF (75*lines) MOD 100>=50 THEN
      INC(maxMark);
    END;

    mediumMark:=(50*lines) DIV 100;
    IF (50*lines) MOD 100>=50 THEN
      INC(mediumMark);
    END;

    IF l.labeled THEN
      help:=((value-l.bottom)*100) DIV (l.top-l.bottom+1);
      IF ((value-l.bottom)*100) DIV (l.top-l.bottom+1)>=50 THEN
        INC(help);
      END;
      IF help>=75 THEN
        l.labelTxt.SetForeground(l.maxColor);
      ELSIF help>=50 THEN
        l.labelTxt.SetForeground(l.mediumColor);
      ELSE
        l.labelTxt.SetForeground(l.onColor);
      END;
      co.IntToStr(help,text);
      IF help<100 THEN
        IF help>=10 THEN
          str.Insert(" ",0,text);
        ELSE
          str.Insert("  ",0,text);
        END;
      END;
      l.labelTxt.SetString(text);
    END;

    current:=1;
    IF l.vertical THEN
      IF l.labeled THEN
        top:=l.y+(l.height-(2*lines-1)*2-l.display.spaceHeight DIV 2 -l.labelTxt.oHeight) DIV 2;
      ELSE
        top:=l.y+(l.height-(2*lines-1)*2) DIV 2;
      END;
      left:=l.x+(l.width-boxWidth) DIV 2;
      WHILE current<=lines DO
        IF current>lines-mark THEN
          IF current<=lines-maxMark THEN
            l.draw.PushForeground(l.maxColor);
          ELSIF current<=lines-mediumMark THEN
            l.draw.PushForeground(l.mediumColor);
          ELSE
            l.draw.PushForeground(l.onColor);
          END;
        ELSE
          l.draw.PushForeground(l.offColor);
        END;
        l.draw.DrawLine(left,top,left+boxWidth-1,top);
        l.draw.DrawLine(left,top+1,left+boxWidth-1,top+1);
        l.draw.PopForeground;
        INC(top,4);
        INC(current);
      END;
      IF l.labeled THEN
        l.labelTxt.Draw(l.x+(l.width-l.labelTxt.oWidth) DIV 2,
                        l.y+l.height-1-l.labelTxt.oHeight,l.draw);
      END;
    ELSE
      top:=l.y+(l.height-boxHeight) DIV 2;
      IF l.labeled THEN
        left:=l.x+(l.width-(2*lines-1)*2-l.display.spaceWidth DIV 2 -l.labelTxt.oWidth) DIV 2;
      ELSE
        left:=l.x+(l.width-(2*lines-1)*2) DIV 2;
      END;
      WHILE current<=lines DO
        IF current<=mark THEN
          IF current>maxMark THEN
            l.draw.PushForeground(l.maxColor);
          ELSIF current>mediumMark THEN
            l.draw.PushForeground(l.mediumColor);
          ELSE
            l.draw.PushForeground(l.onColor);
          END;
        ELSE
          l.draw.PushForeground(l.offColor);
        END;
        l.draw.DrawLine(left,top,left,top+boxHeight-1);
        l.draw.DrawLine(left+1,top,left+1,top+boxHeight-1);
        l.draw.PopForeground;
        INC(left,4);
        INC(current);
      END;
    END;
  END DrawLoad;

  PROCEDURE (l : Load) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    l.Draw^(x,y,draw);

    l.frame.Resize(l.width,l.height);
    l.frame.Draw(l.x,l.y,draw);

    draw.PushForeground(l.background);
    draw.FillRectangle(l.x+l.frame.leftBorder,l.y+l.frame.topBorder,
                       l.width-l.frame.leftBorder-l.frame.rightBorder,
                       l.height-l.frame.topBorder-l.frame.bottomBorder);
    draw.PopForeground;

    l.DrawLoad;
  END Draw;

  PROCEDURE (l : Load) Hide*;

  BEGIN
    IF l.visible THEN
      l.frame.Hide;
      l.DrawHide;
      l.Hide^;
    END;
  END Hide;

  PROCEDURE (l : Load) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF l.visible THEN
      l.DrawLoad;
    END;
  END Resync;

  PROCEDURE (l : Load) Free*;

  BEGIN
    l.display.FreeColor(l.onColor);
    l.display.FreeColor(l.mediumColor);
    l.display.FreeColor(l.maxColor);
  END Free;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOLoad.