#
# WAJIG - Debian Package Management Front End
#
# Implementation of all commands
#
# Copyright (c) Graham.Williams@csiro.au
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

#------------------------------------------------------------------------
#
# Standard python modules
#
#------------------------------------------------------------------------
import os
import re
import string
import sys
import tempfile
import signal
#
# When writing to a pipe where there is no reader (e.g., when
# output is directed to head or to less and the user exists from less
# before reading all output) the SIGPIPE signal is generated. Capture
# the signal and hadle it with the default handler.
#
signal.signal(signal.SIGPIPE,signal.SIG_DFL)

#------------------------------------------------------------------------
#
# APT module
#
#------------------------------------------------------------------------
import apt_pkg

#------------------------------------------------------------------------
#
# Wajig modules
#
#------------------------------------------------------------------------
import changes
import perform

#------------------------------------------------------------------------
#
# Global Variables
#
#------------------------------------------------------------------------
installed_file = changes.get_installed_filename()
available_file = changes.get_available_filename()
previous_file  = changes.get_previous_filename()

#------------------------------------------------------------------------
#
# Interface Variables
#
#------------------------------------------------------------------------

verbose = 0

def set_verbosity_level(new_level):
    global verbose
    verbose = new_level

#------------------------------------------------------------------------
#
# COMMAND
#
#------------------------------------------------------------------------
def do_command(command, root=0, update=0):
    """Simply perform the command and optionally update Installed list

    Arguments:
    command	Command to perform
    root	Whether the command needs root access
    update	Whether to also update the list of installed packages

    Returns:"""

    perform.execute(command,root=root)
    if update > 0: changes.update_installed()

#------------------------------------------------------------------------
#
# DESCRIBE
#
# The package descriptions can either be in dpkg --status or in
# apt-cache dumpavail.  They are not in the latter if they are no longer
# available, so I really need to be checking both!
#
# Similarly using the Tag file within apt-pkg python package.
# Some installed packages may no longer be available.
# Thus need to check both Available and Status.
# But then we get repeats so need to remove repeats.
#
#------------------------------------------------------------------------
def do_describe(packages):
    """Print a description of each package.

    Arguments:
    packages	List of packages to describe

    Returns:"""

    #
    # From where do we get the information?
    #   /var/lib/dpkg/available		dpkg's idea of available
    #	/var/cache/apt/available	apt's  idea of available
    #
    # TODO does apt_pkg provide a way to get available apt-cache directly?
    #
    # I though to move to /var/cache/apt/available so I don't need to
    # use dpkg's copy which is not updated by apt-get update.
    # Hoever, "dselect update" does update it after doing
    # "apt-get update". 
    #
    # So it seems "apt-get update" updates /var/cache/apt/pkgcache.bin not
    # /var/cache/apt/available.  Thus new packages are not found in available.
    # How to get access to /var/cache/apt/pkgcache.bin? Go back to
    # the dpkg available list /var/lib/dpkg/available for now and
    # make sure UPDATE uses "dselect update"
    #
    # Compare also with gnome-tasksel which does the following:
    #
    #  apt-get update
    #  apt-cache dumpavail > /tmp/avail
    #  dpkg --update-avail /tmp/avail
    #
    #
    # avail  = apt_pkg.ParseTagFile(open("/var/cache/apt/available","r"));
    # avail  = apt_pkg.ParseTagFile(open("/var/cache/apt/pkgcache.bin","r"));
    avail  = apt_pkg.ParseTagFile(open("/var/lib/dpkg/available","r"));
    status = apt_pkg.ParseTagFile(open("/var/lib/dpkg/status","r"));
    #
    # Record the descriptions
    #
    describe_list = {}
    #
    # Check for information in the Available list
    #
    while avail.Step():
        if (avail.Section.get("Package") in packages):
            package_name = avail.Section.get("Package")
            package_description = avail.Section.get("Description")
            if not describe_list.has_key(package_name):
                describe_list[package_name] = package_description
    #
    # Check for information in the Status list
    #
    while status.Step():
        if (status.Section.get("Package") in packages):
            package_name = status.Section.get("Package")
            package_description = status.Section.get("Description")
            if not describe_list.has_key(package_name):
                describe_list[package_name] = package_description
    #
    # Print out the one line descriptions. Should it be sorted?
    # If not sorted it should be same order as on the command line.
    #
    pkgs = describe_list.keys()
    pkgs.sort()
    #
    # Print the description depending on level of detail requested
    # through the value of "verbose"
    #
    if len(pkgs) == 0:
        print "No packages to describe"
    elif verbose == 0:
        print "%-24s %s" % ("Package", "Description")
        print "="*24 + "-" + "="*51
        for pkg in pkgs:
            # Only print that first line, but check that there
            # is a description available.
            package_short_description = ""
            if describe_list[pkg]:
              line_end = string.find(describe_list[pkg],"\n")
              package_short_description = describe_list[pkg][0:line_end]
            print "%-24s %s" % (pkg, string.capitalize(package_short_description))
    elif verbose == 1:
        for pkg in pkgs:
            print pkg + ": " + string.capitalize(describe_list[pkg]) + "\n"
    else:
        #
        # TODO is there a way of doing this using apt_pkg easily?
        # Otherwise "apt-cache show" seems okay if a little slower.
        #
        package_names = perform.concat(packages)
        command = "apt-cache show " + package_names
        perform.execute(command,root=1)

#------------------------------------------------------------------------
#
# DESCRIBE NEW
#
#------------------------------------------------------------------------
def do_describe_new():
    """Report on packages that are newly available.

    Arguments:

    Returns:"""

    #
    # Load the dictionaries from file then describe each new one.
    #
    changes.load_dictionaries();
    new_pkgs = changes.get_new_available();
    if len(new_pkgs) == 0:
        print "No new packages"
    else:
        do_describe(new_pkgs);

#------------------------------------------------------------------------
#
# DOWNLOAD
#
#------------------------------------------------------------------------
def do_download(packages):
    """Download packages.

    Arguments:
    packages	List of packages to be downloaded

    Returns:"""

    command = "apt-get --download-only install " + perform.concat(packages)
    perform.execute(command, root=1)
    changes.update_installed()

#------------------------------------------------------------------------
#
# FORCE
#
# TODO This is being rewritten to handle list of packages - still in progress
#
#------------------------------------------------------------------------
def do_force(packages):
    """Force the installation of a package.

    This is useful when there is a conflict of the same file from
    multiple packages.

    Arguments:
    packages	The packages to be installed (must be in the cache)

    Returns:"""

    command = "dpkg --install --force overwrite "
    if re.match(".*\.deb$", packages[0]):
        for pkg in packages:
            if os.path.exists(pkg):
                command = command + "'" + pkg + "' "
            elif os.path.exists("/var/cache/apt/archives/" + pkg):
                command = command + "'/var/cache/apt/archives/" + pkg + "' "
            else:
                print "Wajig: The file " + pkg +\
                      " not found in either the" +\
                      " current directory\n" +\
                      "       nor in /var/cache/apt/archives/\n" +\
                      "       Please confirm location and try again."
                return()
        perform.execute(command, root=1)
    else:
        for pkg in packages:
            command = command + " /var/cache/apt/archives/$(/bin/ls /var/cache/apt/archives/" +\
                      " | egrep '^" + pkg + "_' | sort | tail -1)"
        perform.execute(command,root=1)

    changes.update_installed()

#------------------------------------------------------------------------
#
# HOLD
#
#------------------------------------------------------------------------
def do_hold(packages):
    """Place packages on hold (they will not be upgraded).

    Arguments:
    packages	List of packages.

    Returns:"""

    for p in packages:
	# The dpkg needs sudo but not the echo.
	# Maybe sensible to have all sudo's in this file?
        command = "echo \"" + p + " hold\" | " +\
	          perform.setroot + " dpkg --set-selections"
	perform.execute(command)

#------------------------------------------------------------------------
#
# INSTALL
#
#------------------------------------------------------------------------
def do_install(packages):
    """Install packages.

    Arguments:
    packages	List of packages to be installed or path to deb file

    Returns:"""

    #
    # Currently we use the first argument to determine the type of all
    # of the rest. Perhaps we should look at each one in turn?
    #

    #
    # Handle URLs first. We don't do anything smart.  Simply download
    # the .deb file and install it.  If it fails then don't attempt to
    # recover.  The user can do a wget themselves and install the
    # resulting .deb if they need to.
    #
    # Currently only a single URL is allowed. Should this be generalised?
    #
    if re.match("(http|ftp)://", packages[0]):
        if len(packages) > 1:
            print "wajig: Error: install URL allows only one URL, not " +\
                  str(len(packages))
            sys.exit(1)
        tmpdeb = tempfile.mktemp() + ".deb"
        command = "wget --output-document=" + tmpdeb + " " + packages[0]
        if not perform.execute(command):
            command = "dpkg --install " + tmpdeb
            perform.execute(command, root=1)
            if os.path.exists(tmpdeb): os.remove(tmpdeb)
        else:
            print "Wajig: The location " + packages[0] +\
                  " was not found. Check and try again."
            return()
    elif re.match(".*\.deb$", packages[0]):
        command = "dpkg -i "
        for pkg in packages:
            if os.path.exists(pkg):
                command = command + "'" + pkg + "' "
            elif os.path.exists("/var/cache/apt/archives/" + pkg):
                command = command + "'/var/cache/apt/archives/" + pkg + "' "
            else:
                print "Wajig: The file " + pkg +\
                      " not found in either the" +\
                      " current directory\n" +\
                      "       nor in /var/cache/apt/archives/\n" +\
                      "       Please confirm location and try again."
                return()
        perform.execute(command, root=1)
    else:
	command = "apt-get install " + perform.concat(packages)
	perform.execute(command, root=1)
    changes.update_installed()

#------------------------------------------------------------------------
#
# LISTNAMES
#
#------------------------------------------------------------------------
def do_listinstalled(pattern):
    """Print list of installed packages.

    Arguments:

    Returns:"""

    command = "dpkg --get-selections | awk '$2 ~/^install$/ {print $1}'"

    if len(pattern) == 1:
	command = command + " | grep " + pattern[0] + " | sort"
    perform.execute(command)

#------------------------------------------------------------------------
#
# LISTNAMES
#
#------------------------------------------------------------------------
def do_listnames(pattern):
    """Print list of known package names.

    Arguments:

    Returns:"""

    if len(pattern) == 0:
        command = "apt-cache pkgnames | sort"
    else:
	command = "apt-cache pkgnames | grep " + pattern[0] + " | sort"
    perform.execute(command)

#------------------------------------------------------------------------
#
# NEW
#
#------------------------------------------------------------------------
def do_new():
    """Report on packages that are newly available.

    Arguments:

    Returns:"""

    print "%-24s %s" % ("Package", "Available")
    print "="*24 + "-" + "="*16
    #
    # Load the dictionaries from file then list each one and it's version
    #
    changes.load_dictionaries();
    new_pkgs = changes.get_new_available();
    new_pkgs.sort()
    for i in range(0,len(new_pkgs)):
        print "%-24s %s" % (new_pkgs[i], changes.get_available_version(new_pkgs[i]))
                   
#------------------------------------------------------------------------
#
# NEWUPGRADES
#
#------------------------------------------------------------------------
def do_newupgrades():
    """Report on packages that are newly upgraded.

    Arguments:

    Returns:"""

    #
    # Load the dictionaries from file then list each one and it's version
    #
    changes.load_dictionaries();
    new_upgrades = changes.get_new_upgrades();
    if len(new_upgrades) == 0:
        print "No new upgrades"
    else:
        print "%-24s %-24s %s" % ("Package", "Available", "Installed")
        print "="*24 + "-" + "="*24 + "-" + "="*24 
        new_upgrades.sort()
        for i in range(0,len(new_upgrades)):
            print "%-24s %-24s %-24s" % (new_upgrades[i], \
                            changes.get_available_version(new_upgrades[i]), \
                            changes.get_installed_version(new_upgrades[i]))

#------------------------------------------------------------------------
#
# STATUS
#
#------------------------------------------------------------------------
def do_status(packages):
    """List status of the packages identified.

    Arguments:
    packages	List the version of installed packages

    Returns:"""

    print "%-23s %-15s %-15s %-15s %s" %\
          ("Package", "Installed", "Previous", "Now", "State")
    print "="*23 + "-" + "="*15 + "-" + "="*15 + "-" + "="*15 + "-" + "="*5
    sys.stdout.flush()

    #
    # Get status.  Previously used dpkg --list but this truncates package
    # names to 16 characters :-(. Perhaps should now also remove the DS
    # column as that was the "ii" thing from dpkg --list.  It is now
    # "install" or "deinstall" from dpkg --get-selections.
    #
    #   command = "dpkg --list | " +\
    #             "awk '{print $2,$1}' | " +\

    command = "dpkg --get-selections | " +\
	      "join - " + installed_file + " | " +\
              "join -a 1 - " + previous_file + " | " +\
	      "awk 'NF==3 {print $0, \"N/A\"; next}{print}' | " +\
	      "join -a 1 - " + available_file + " | " +\
	      "awk 'NF==4 {print $0, \"N/A\"; next}{print}' | "
    if len(packages) > 0:
        command = command + "egrep '^($"
        for i in packages: 
            command = command + " |" + i
        command = command + " )' |"

    command = command +\
	      "awk '{printf(\"%-20s\\t%-15s\\t%-15s\\t%-15s\\t%-2s\\n\", " +\
	      "$1, $3, $4, $5, $2)}'"
    perform.execute(command)

    #
    # Check whether the package is not in the installed list, and if not
    # list its status appropriately.
    #
    for i in packages: 
        if os.system("egrep '^" + i + " ' " + installed_file +\
	               " >/dev/null"):
            #
	    # Package is not installed
	    #
	    command = \
              "join -a 2 " + previous_file + " " + available_file + " | " +\
	      "awk 'NF==2 {print $1, \"N/A\", $2; next}{print}' | " +\
	      "egrep '^" + i + " '"
	    command = command +\
	      " | awk '{printf(\"%-20s\\t%-15s\\t%-15s\\t%-15s\\n\", " +\
	      "$1, \"N/A\", $2, $3)}'"
	    perform.execute(command)



#------------------------------------------------------------------------
#
# TOUPGRADE
#
#------------------------------------------------------------------------
def do_toupgrade():
    """List packages with Available version more recent than Installed.

    Arguments:

    Returns:"""

    # A simple way of doing this is to just list packages in the installed
    # list and the available list which have different versions.
    # However this does not capture the situation where the available
    # package version predates the installed package version (e.g, 
    # you've installed a more recent version than in the distribution).
    # So now also add in a call to "dpkg --compare-versions" which slows
    # things down quite a bit!
    #
    print "%-24s %-24s %s" % ("Package", "Available", "Installed")
    print "="*24 + "-" + "="*24 + "-" + "="*24 
    #
    # Load the dictionaries from file then list each one and it's version
    #
    changes.load_dictionaries();
    to_upgrade = changes.get_to_upgrade();
    to_upgrade.sort()
    for i in range(0,len(to_upgrade)):
        print "%-24s %-24s %-24s" % (to_upgrade[i], \
                            changes.get_available_version(to_upgrade[i]), \
                            changes.get_installed_version(to_upgrade[i]))
    
#------------------------------------------------------------------------
#
# UPDATE
#
#------------------------------------------------------------------------
def do_update(query=0):
    """Perform.Execute an update of the available packages

    Arguments:
    query	If non-zero then ask user whether this should be done.

    Returns:
    Return status of the command"""

    if verbose > 0:
        print "The Packages files listing the available packages is being updated."
    if query > 0:
	versionfile=open("/etc/debian_version");
	debian_version = string.split(versionfile.read())[0];
	if debian_version == "testing/unstable":
	    print """
You appear to be running the `unstable' or `testing' distribution of
Debian. It is very likely that your Packages files are out of date. 
Doing an UPDATE is recommended.
"""
	else:
	    print """
You appear to be running the `stable' distribution of Debian. It is not 
likely that your Packages files are out of date. Doing an UPDATE is optional
"""
	doit = raw_input("Update? [Y/n] ")
	if not re.match("^(y|Y|yes|Yes|YES)$", doit):
            return
    #
    # Note that we do a "dselect update" rather than an "apt-get update" since the
    # latter does not update dpkg's idea of what's available!
    #
    # if perform.execute("dselect update", root=1) == 0:
    #
    # 02/03/10 Go back to apt-get for a while.  Note that the available
    # list is now obtained from apt cache, not dpkg lib, for the DESCRIBE
    # command. I've also started using my own mirror and getting a syntax
    # error when dpkg grabs apt's availables file.
    #
    # 02/03/26 Go back to dselect - The apt available file is not updated
    # by apt-get so new was not finding the package descriptions.
    #
    if perform.execute("dselect update", root=1) == 0:
    # if perform.execute("apt-get update", root=1) == 0:
	#
	# Only update the available list if the UPDATE succeeded
 	#
        changes.update_available()
	#
	# How many new upgrades are there?
	#
	changes.count_upgrades()
    else:
        print "Wajig: The update failed for some reason."
        print "       Have you configured dselect lately?"
        print "       It may need updating to use apt."

        exit(1);

#------------------------------------------------------------------------
#
# WHATIS
#
# No longer used! Use describe instead.
#
#------------------------------------------------------------------------
def do_whatis0(packages):
    """Look for a one-line description of the given packages.

    Arguments:
    packages	List of packages to get descriptions for

    Returns:"""

    command = "fping packages.debian.org 2>&1 >/dev/null"
    if perform.execute(command) == 0:
        for i in packages:
            results = tempfile.mktemp()
	    command = "wget --output-document=" + results +\
		      " http://packages.debian.org/cgi-bin" +\
		      "/search_packages.pl\?keywords=" + i +\
		      "\&searchon=names\&subword=1\&" +\
		      "version=unstable\&release=all 2> /dev/null"
	    perform.execute(command)
	    command = "cat " + results + " | " +\
		"egrep '<TD><B><A HREF|<TD COLSPAN=2>' | " +\
		"perl -p -e 's|<[^>]*>||g;s|^	 ||g;" +\
	        "s|^&nbsp; |	|;s|&nbsp;||g;s|&quot;|\"|g;'"
	    perform.execute(command)
	    if os.path.exists(results): os.remove(results)
    else:
	print "Perhaps the server is temporarily down."

#------------------------------------------------------------------------
#
# WHICHPKG
#
#------------------------------------------------------------------------
def do_whichpkg(filename):
    """Look for a particular file or pattern in a Debian package.

    Arguments:
    filename	Pattern to find

    Returns:"""

    #
    # Thought the following was it once but no, it only searches installed
    # packages.
    #
    # command = "dpkg -S " + filename
    # perform.execute(command)

    command = "fping packages.debian.org 2>&1 >/dev/null"
    if perform.execute(command) == 0:
        print "%-50s %-17s" % ("File Path", "Package")
        print "="*50 + "-" + "="*17
        sys.stdout.flush()
        results = tempfile.mktemp()
        command = "wget --output-document=" + results +\
		      " http://packages.debian.org/cgi-bin" +\
		      "/search_contents.pl\?word=" + filename +\
                      "\&searchmode=searchfilesanddirs" +\
		      "\&case=insensitive\&version=unstable" +\
		      "\&arch=i386 2> /dev/null"
	perform.execute(command)
        #
        # Test for multiple page output and handle appropriately
        #
        command = "grep page= " + results + " 2>&1 > /dev/null"
        if perform.execute(command) == 0:
            #
            # Multiple pages of output
            #
            command = "cat " + results + " | " +\
                      "egrep -v '^<|^$|^Packages search page' | " +\
                      "egrep -v 'page=' | " +\
                      "perl -p -e 's|<[^>]*>||g;s|<[^>]*$||g;s|^[^<]*>||g;'"
            perform.execute(command)
            #
            # Loop over the other pages
            #
            command = "for i in $(grep page= " + results +\
                      " | perl -p -e 's|.*page=||; s|&.*>||' | sort -u " +\
                      " | grep -v 1); " +\
                      "do " +\
                      "wget --output-document=" + results +\
                      " http://packages.debian.org/cgi-bin" +\
                      "/search_contents.pl?page=$i\&word=" + filename +\
                      "\&version=unstable\&arch=i386" +\
                      "\&case=insensitive\&searchmode=searchfilesanddirs " +\
                      " 2> /dev/null; " +\
                      "cat " + results + " | " +\
                      "egrep -v '^<|^$|^Packages search page' | " +\
                      "egrep -v 'page=' | " +\
                      "perl -p -e 's|<[^>]*>||g;s|<[^>]*$||g;" +\
                      "s|^[^<]*>||g;';" +\
                      "done"
            perform.execute(command)
        else:
            #
            # A single page of output
            #
            command = "cat " + results + " | " +\
                      "egrep -v '^<|^$|^Packages search page' | " +\
                      "perl -p -e 's|<[^>]*>||g;s|<[^>]*$||g;s|^[^<]*>||g;'"
            perform.execute(command)
        if os.path.exists(results): os.remove(results)
    else:
	print "Perhaps the Debian server is temporarily down."

