#ifndef WARPED_H
#define WARPED_H

// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "WarpedConfig.h"

/** This header file defines some of the default data types and
    enumerations used in the system. */

#ifdef HAVE_STDINT_H
// The following is a C-99ism...
#ifndef __STDC_LIMIT_MACROS
#define __STDC_LIMIT_MACROS
#include <stdint.h>
#endif
typedef int64_t warped64_t;
typedef int32_t warped32_t;
const warped64_t warped64Max = INT64_MAX;
const warped64_t warped64Min = INT64_MIN;
const warped64_t warped32Max = INT32_MAX;
const warped64_t warped32Min = INT32_MIN;
#elif defined(SIZEOF_LONG_LONG_) // defined in warped-config.h
#include <limits.h>
typedef long long warped64_t;
typedef long warped32_t;
const warped64_t warped64Max = LLONG_MAX;
const warped64_t warped64Min = LLONG_MIN;
const warped64_t warped32Max = LONG_MAX;
const warped64_t warped32Min = LONG_MIN;
#else
#include <limits.h>
typedef long VTimeMajor_t;
typedef long VTimeMinor_t;
const warped64_t warped64Max = LONG_MAX;
const warped64_t warped64Min = LONG_MIN;
const warped64_t warped32Max = LONG_MAX;
const warped64_t warped32Min = LONG_MIN;
#warning Could not find 64 bit type using 32 bit max - NOT recommended.
#endif

#include <iostream>
#include <clutils/Debug.h>
#include <clutils/StringUtilities.h>

// Console I/O operations - we define this stream interface to trap 
// the I/O produced by different processes and feed the I/O to the 
// central console 

extern ostream *wout, *werr;

// include the default definition of VTime
#include "VTime.h"
// include the default definition of OBJECT_ID
#include "DefaultObjectID.h"

// This definition for bool is used to ease portability between different
// compliers (some of them have pre-defined type "bool" while some don't).
// Correspondingly, the definition for the constants "true" and "false"
// used in the system are here.

// Check if NULL is defined. If not define it here
#ifndef NULL
#define NULL   0
#endif

// Define the ASSERT statement that will be turned on when the compiler
// flag DEVELOPER_ASSERTIONS is turned "on"
#ifndef ASSERT
#ifndef  NO_DEVELOPER_ASSERTIONS
#include <assert.h>
#define ASSERT(x) assert(x)
#else
#define ASSERT(x)
#endif
#endif

// Define the DEBUG statement that will be turned on when the compiler
// flag DEVELOPER_ASSERTIONS is turned "on"
#ifndef DEBUG
#ifndef NO_DEVELOPER_ASSERTIONS
#define DEBUG(x)  x 
#else
#define DEBUG(x)
#endif
#endif



// Define the type of the tag used to seralize and deSerialize stuff
typedef int SerializationTag;

// the delimiter used in serialization and deserialization
const char DELIMITER = ' ';

// Time Warp Event Message Type - it is either a POSITIVE or NEGATIVE msg.
enum EventMessageType {POSITIVE, NEGATIVE};

// The severity enumerations in the system. The order of enumeration is
// from Good to Bad.
enum SEVERITY {NOTE, WARNING, ERROR, ABORT};

/// Enumeration for message suppression
//  Cases where message can be
//  suppressed: Coastforwarding and during lazy and lazyAggr
//  cancellation
enum MsgSuppression {NO, COASTFORWARD, LAZYCANCEL};


/** used as a flag for executing a find operation */
enum findMode {LESS, LESSEQUAL, EQUAL, GREATEREQUAL, GREATER};

// a templatized getMinimum function - returns the object that is the
// least.
template <class type>
inline const type&
MIN_FUNC(const type& x, const type& y) {
  if (x < y) { return x; }
  return y;
}

// a templatized getMaximum function - returns the object that is the
// largest.
template <class type>
inline const type&
MAX_FUNC(const type& x, const type& y) {
  if (x > y) { return x; }
  return y;
}

// the following two classes are passed in as function objects 
// to the hash_map data structure 
class EqualID {
public:
  size_t operator()(const OBJECT_ID *s1, const OBJECT_ID *s2) const {
    return (*s1 == *s2);
  }
};

// hash function for object ids
struct OIDHash{
  size_t operator()(const OBJECT_ID& id) const { return size_t(&id); }
};

// hash function object for object ids
class hashID {
public:
  size_t operator()(const OBJECT_ID *objectID) const {
    OIDHash myHashFunction;
    return (myHashFunction(*objectID));
  }
};

inline
ostream &operator<<( ostream &os, SEVERITY severity ){
  switch( severity ){
  case NOTE:
    os << "NOTE";
    break;
  case WARNING:
    os << "WARNING";
    break;
  case ERROR:
    os << "ERROR";
    break;
  case ABORT:
    os << "ABORT";
    break;
  default:
    os << "<Unknown Severity " + intToString(severity) +" >";
  }

  return os;
}

#endif
