/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This file is derived from the QtDotNet Style v2.0, a Qt Solutions
* component. See original copyright information below.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

/****************************************************************************
**
** Copyright (C) 2003-2007 Trolltech ASA. All rights reserved.
**
** This file is part of a Qt Solutions component.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.trolltech.com/products/qt/opensource.html
**
** If you are unsure which license is appropriate for your use, please
** review the following information:
** http://www.trolltech.com/products/qt/licensing.html or contact the
** Trolltech sales department at sales@trolltech.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "dotnetstyle.h"

#if !defined( WI_NO_STYLE_DOTNET )

#include <QStyleOption>
#include <QPainter>
#include <QMainWindow>
#include <QDockWidget>
#include <QDialogButtonBox>
#include <QToolBar>
#include <QDialog>
#include <QLibrary>
#include <QStatusBar>
#include <QApplication>
#include <qt_windows.h>

using namespace WebIssues;

typedef bool (WINAPI* PtrIsAppThemed)();
typedef HRESULT (WINAPI* PtrGetCurrentThemeName)( OUT LPWSTR pszThemeFileName, int cchMaxNameChars, OUT OPTIONAL LPWSTR pszColorBuff, int                   cchMaxColorChars, OUT OPTIONAL LPWSTR pszSizeBuff, int cchMaxSizeChars );

static PtrIsAppThemed pIsAppThemed = NULL;
static PtrGetCurrentThemeName pGetCurrentThemeName = NULL;

static const int windowsItemFrame       =  2; // menu item frame width
static const int windowsSepHeight       =  9; // separator item height
static const int windowsItemHMargin     =  3; // menu item hor text margin
static const int windowsItemVMargin     =  2; // menu item ver text margin
static const int windowsArrowHMargin	=  6; // arrow horizontal margin
static const int windowsRightBorder     = 15; // right border on windows

DotNetStyle::DotNetStyle() : QWindowsVistaStyle()
{
    QLibrary themeLib( "uxtheme" );
    themeLib.load();
    if ( themeLib.isLoaded() ) {
        pIsAppThemed = (PtrIsAppThemed)themeLib.resolve( "IsAppThemed" );
        pGetCurrentThemeName = (PtrGetCurrentThemeName)themeLib.resolve( "GetCurrentThemeName" );
    }
}

DotNetStyle::~DotNetStyle()
{
}

void DotNetStyle::drawPrimitive( PrimitiveElement element, const QStyleOption* option,
    QPainter* painter, const QWidget* widget ) const
{
    QColor gradientDark = gradientDarkColor( option );
    QColor gradientLight = gradientLightColor( option );
    QColor shadow = shadowColor( option );

    QRect rect = option->rect;

    switch ( element ) {
        case PE_IndicatorDockWidgetResizeHandle:
            break;

        case PE_PanelButtonTool:
            {
                QStyleOption opt2 = *option;
                if ( widget ) {
                    if( QDockWidget* dw = qobject_cast<QDockWidget*>( widget->parentWidget() ) ) {
                        if ( !dw->isFloating() )
                            opt2.palette.setBrush( QPalette::Button, Qt::transparent );
                    }
                }
                if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                    QWindowsVistaStyle::drawPrimitive( element, &opt2, painter, widget );
                else
                    QWindowsXPStyle::drawPrimitive( element, &opt2, painter, widget );
            }
            break;

        case PE_FrameMenu:
            {
                QPen pen = painter->pen();
                QBrush brush = painter->brush();
                if ( const QStyleOptionToolBar* toolbar = qstyleoption_cast<const QStyleOptionToolBar*>( option ) )
                {
                    Qt::Orientation orientation = ( toolbar->toolBarArea == Qt::RightToolBarArea ||
                        toolbar->toolBarArea == Qt::LeftToolBarArea ) ? Qt::Vertical : Qt::Horizontal;
                    painter->setBrush( toolbarGradient( toolbar, toolbar->rect, orientation ) );
                } else {
                    painter->setBrush( m_menuBackgroundColor );
                }
                painter->setPen( m_menuFrameColor );
                painter->drawRect( option->rect.adjusted( 0, 0, -1, -1 ) );
                painter->setPen( pen );
                painter->setBrush( brush );
            }
            break;

        case PE_Widget:
            if ( const QMainWindow* mw = qobject_cast<const QMainWindow*>( widget ) ) {
                if ( m_currentPal != qApp->palette() ) //workaround for style polish on theme color change
                {
                    const_cast<DotNetStyle*>( this )->m_currentPal = qApp->palette();
                    const_cast<DotNetStyle*>( this )->updatePalette();
                }
                painter->save();
                QPoint begin = widget->geometry().topRight();
                QPoint end = widget->geometry().topLeft() + QPoint( 0, widget->geometry().height()/2 );
                begin = widget->mapFromGlobal( begin );
                end = widget->mapFromGlobal( end );
                QLinearGradient menubargradient( begin, end );
                menubargradient.setColorAt( 0, gradientLight );
                menubargradient.setColorAt( 0.8, gradientDark );
                menubargradient.setColorAt( 1, gradientDark );
                QRect windowRect = option->rect;
                if ( QStatusBar* statusBar = mw->findChild<QStatusBar*>() ) {
                    windowRect.adjust( 0, 0, 0, -statusBar->height() );
                    painter->setPen( option->palette.background().color().lighter( 106 ) );
                    painter->drawLine( windowRect.bottomLeft() + QPoint( 0, 1 ),
                        windowRect.bottomRight() + QPoint( 0, 1 ) );
                }
                painter->fillRect( windowRect, menubargradient );
                painter->restore();
            }
            break;

        case PE_IndicatorToolBarSeparator:
            {
                QRect rect = option->rect;
                shadow.setAlpha( 180 );
                painter->setPen( shadow );
                const int margin = 3;
                if ( option->state & State_Horizontal ) {
                    const int offset = rect.width()/2;
                    painter->drawLine( rect.bottomLeft().x() + offset,
                        rect.bottomLeft().y() - margin,
                        rect.topLeft().x() + offset,
                        rect.topLeft().y() + margin );
                    painter->setPen( QPen( option->palette.background().color().light( 110 ) ) );
                    painter->drawLine( rect.bottomLeft().x() + offset + 1,
                        rect.bottomLeft().y() - margin,
                        rect.topLeft().x() + offset + 1,
                        rect.topLeft().y() + margin );
                } else {
                    const int offset = rect.height()/2;
                    painter->setPen( QPen( option->palette.background().color().dark( 110 ) ) );
                    painter->drawLine( rect.topLeft().x() + margin ,
                        rect.topLeft().y() + offset,
                        rect.topRight().x() - margin,
                        rect.topRight().y() + offset );
                    painter->setPen( QPen( option->palette.background().color().light( 110 ) ) );
                    painter->drawLine( rect.topLeft().x() + margin ,
                        rect.topLeft().y() + offset + 1,
                        rect.topRight().x() - margin,
                        rect.topRight().y() + offset + 1 );
                }
            }
            break;

        case PE_IndicatorToolBarHandle:
            painter->save();
            {
                QColor gripColor = shadow.darker( 120 );
                if ( option->state & State_Horizontal ) {
                    for ( int i = rect.height()/5; i <= 4*( rect.height()/5 ) ; i+=4 ) {
                        int y = rect.topLeft().y() + i + 1;
                        int x1 = rect.topLeft().x() + 3;
                        painter->fillRect( x1 + 1, y, 2, 2, Qt::white );
                        painter->fillRect( x1, y - 1, 2, 2, gripColor );
                    }
                }
                else {
                    for ( int i = rect.width()/5; i <= 4*( rect.width()/5 ) ; i+=4 ) {
                        int x = rect.topLeft().x() + i + 1;
                        int y1 = rect.topLeft().y() + 3;
                        painter->fillRect( x , y1 + 1, 2, 2, Qt::white );
                        painter->fillRect( x - 1, y1, 2, 2, gripColor );
                    }
                }
            }
            painter->restore();
            break;

        case PE_PanelMenuBar:
            break;

        case PE_FrameTabWidget:
            if ( qobject_cast<const QMainWindow*>( widget->parentWidget() ) ) {
                QPen pen = painter->pen();
                painter->setPen( highlightOutlineColor( option, false ) );
                painter->drawLine( rect.topLeft(), rect.topRight() );
                painter->setPen( pen );
            } else {
                if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                    QWindowsVistaStyle::drawPrimitive( element, option, painter, widget );
                else
                    QWindowsXPStyle::drawPrimitive( element, option, painter, widget );
            }
            break;

        default:
            if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                QWindowsVistaStyle::drawPrimitive( element, option, painter, widget );
            else
                QWindowsXPStyle::drawPrimitive( element, option, painter, widget );
            break;
    }
}

void DotNetStyle::drawControl( ControlElement element, const QStyleOption* option,
    QPainter* painter, const QWidget* widget ) const
{
    QColor gradientDark = gradientDarkColor( option );
    QColor gradientLight = gradientLightColor( option );
    QColor shadow = shadowColor( option );

    QRect rect = option->rect;

    switch ( element ) {
        case CE_MenuItem:
            painter->save();
            if ( const QStyleOptionMenuItem* menuitem = qstyleoption_cast<const QStyleOptionMenuItem*>( option ) ) {
                int x, y, w, h;
                QStyleOptionMenuItem mbiCopy = *menuitem;
                painter->fillRect( rect, m_menuBackgroundColor );
                painter->fillRect( QRect( 0, rect.top(), 25, rect.bottom() ), menuGradient( option, rect ) );
                menuitem->rect.getRect(& x, &y, &w, &h );

                int tab = menuitem->tabWidth;
                bool dis = !( menuitem->state & State_Enabled );
                bool checked = menuitem->checkType != QStyleOptionMenuItem::NotCheckable ? menuitem->checked : false;
                bool act = menuitem->state & State_Selected;

                int checkcol = qMax( menuitem->maxIconWidth, 20 );
                if ( menuitem->menuItemType == QStyleOptionMenuItem::Separator ) {
                    int yoff = y-1 + h / 2;
                    painter->setPen( m_menuSeparatorColor );
                    painter->drawLine( x + 32, yoff, x + w + 6, yoff );
                    painter->restore();
                    return;
                }

                QRect vCheckRect = visualRect( option->direction, menuitem->rect,
                    QRect( menuitem->rect.x(), menuitem->rect.y(), checkcol, menuitem->rect.height() ) );
                vCheckRect.adjust( 2, 0, 0, 0 );

                if ( act && !dis ) {
                    painter->setPen( highlightOutlineColor( option ) );
                    painter->setBrush( highlightBrush( option, false ) );
                    painter->drawRect( option->rect.adjusted( 0, 0, -2, -2 ) );
                }

                if ( !menuitem->icon.isNull() ) {
                    QIcon::Mode mode = dis ? QIcon::Disabled : QIcon::Normal;
                    if ( act && !dis )
                        mode = QIcon::Active;
                    QPixmap pixmap;
                    if ( checked )
                        pixmap = menuitem->icon.pixmap( pixelMetric( PM_SmallIconSize ), mode, QIcon::On );
                    else
                        pixmap = menuitem->icon.pixmap( pixelMetric( PM_SmallIconSize ), mode );
                    int pixw = pixmap.width();
                    int pixh = pixmap.height();
                    QRect pmr( 0, 0, pixw, pixh );
                    pmr.moveCenter( vCheckRect.center() );
                    painter->setPen( menuitem->palette.text().color() );
                    painter->drawPixmap( pmr.topLeft(), pixmap );
                } else if ( checked ) {
                    painter->save();
                    painter->setPen( highlightOutlineColor( option, act ) );
                    painter->setBrush( highlightMenuCheckBrush( option, act ) );
                    painter->drawRect( vCheckRect.adjusted( -1, 1, 0, -3 ) );
                    painter->restore();

                    QStyleOptionMenuItem newMi = *menuitem;
                    newMi.state = State_None;
                    if ( !dis )
                        newMi.state |= State_Enabled;
                    if ( act )
                        newMi.state |= State_On;
                    newMi.rect = visualRect( option->direction, menuitem->rect,
                        QRect( menuitem->rect.x() + windowsItemFrame, menuitem->rect.y() + windowsItemFrame,
                        checkcol - 2 * windowsItemFrame, menuitem->rect.height() - 2*windowsItemFrame ) );
                    drawPrimitive( PE_IndicatorMenuCheckMark, &newMi, painter, widget );
                }
                painter->setPen( menuitem->palette.buttonText().color() );

                QColor discol;
                if ( dis ) {
                    discol = menuitem->palette.text().color();
                    painter->setPen( discol );
                }

                int xm = windowsItemFrame + checkcol + windowsItemHMargin;
                int xpos = menuitem->rect.x() + xm;
                QRect textRect( xpos + 5, y + windowsItemVMargin, w - xm - windowsRightBorder - tab + 1, h - 2 * windowsItemVMargin );
                QRect vTextRect = visualRect( option->direction, menuitem->rect, textRect );
                QString s = menuitem->text;

                if ( !s.isEmpty() ) {
                    painter->save();
                    int t = s.indexOf( QLatin1Char( '\t' ) );
                    int text_flags = Qt::AlignVCenter | Qt::TextShowMnemonic | Qt::TextDontClip | Qt::TextSingleLine;
                    if ( !styleHint( SH_UnderlineShortcut, menuitem, widget ) )
                        text_flags |= Qt::TextHideMnemonic;
                    text_flags |= Qt::AlignLeft;
                    if ( t >= 0 ) {
                        QRect vShortcutRect = visualRect( option->direction, menuitem->rect,
                        QRect( textRect.topRight(), QPoint( menuitem->rect.right(), textRect.bottom() ) ) );
                        if ( dis && !act ) {
                            painter->setPen( discol );
                        }
                        painter->drawText( vShortcutRect, text_flags, s.mid( t + 1 ) );
                        s = s.left( t );
                    }
                    QFont font = menuitem->font;
                    if ( menuitem->menuItemType == QStyleOptionMenuItem::DefaultItem )
                        font.setBold( true );
                    painter->setFont( font );
                    if ( dis && !act ) {
                        painter->setPen( discol );
                    }
                    painter->drawText( vTextRect, text_flags, s.left( t ) );
                    painter->restore();
                }
                if ( menuitem->menuItemType == QStyleOptionMenuItem::SubMenu ) {
                    int dim = ( h - 2 * windowsItemFrame ) / 2;
                    PrimitiveElement arrow;
                    arrow = ( option->direction == Qt::RightToLeft ) ? PE_IndicatorArrowLeft : PE_IndicatorArrowRight;
                    xpos = x + w - windowsArrowHMargin - windowsItemFrame - dim;
                    QRect  vSubMenuRect = visualRect( option->direction, menuitem->rect, QRect( xpos, y + h / 2 - dim / 2, dim, dim ) );
                    QStyleOptionMenuItem newMI = *menuitem;
                    newMI.rect = vSubMenuRect;
                    newMI.state = dis ? State_None : State_Enabled;
                    drawPrimitive( arrow, &newMI, painter, widget );
                }
            }
            painter->restore();
            break;

        case CE_MenuEmptyArea:
            {
                painter->fillRect( option->rect, option->palette.base() );
                QLinearGradient grad( QPoint( rect.left() , 0 ), QPoint( rect.left() + 23, 0 ) );
                grad.setColorAt( 0, gradientLight.light( 110 ) );
                grad.setColorAt( 1, gradientDark );
                painter->fillRect( QRect( 0, rect.top() + 2, 23, rect.bottom() - 3 ), grad );
            }
            break;

        case CE_MenuBarItem:
            if ( const QStyleOptionMenuItem* mbi = qstyleoption_cast<const QStyleOptionMenuItem*>( option ) ) {
                QRect rect( QPoint( 0, 0 ), widget->window()->size() );
                {
                    if ( widget->window() )
                        drawPrimitive( PE_Widget, option, painter, widget );
                }
                if ( option->state & ( QStyle::State_Sunken | QStyle::State_Selected ) )
                {
                    QColor highlight = shadow;
                    painter->setPen( m_menuFrameColor );
                    if ( option->state & QStyle::State_Sunken ) {

                        painter->setBrush( menuGradient( option, option->rect, Qt::Vertical ) );

                        painter->drawRect( option->rect.adjusted( 0, 2, -1, 0 ) );

                        QColor shade = Qt::black;
                        shade.setAlpha( 50 );
                        painter->setBrush( shade );
                        painter->setPen( Qt::transparent );
                        QRect shadowRect( option->rect.topRight() + QPoint( 0, 4 ), option->rect.bottomRight() + QPoint( 1,0 ) );
                        painter->drawRect( shadowRect );
                    }
                    else {
                        painter->setPen( highlightOutlineColor( option ) );
                        painter->setBrush( highlightBrush( option, false ) );
                        painter->drawRect( option->rect.adjusted( 0, 2, -2, -2 ) );
                    }
                }
                QStyleOptionMenuItem mbiCopy = *mbi;
                QPalette pal = mbi->palette;
                pal.setBrush( QPalette::All, QPalette::ButtonText, mbi->palette.text() );
                mbiCopy.palette = pal;
                QCommonStyle::drawControl( element, &mbiCopy, painter, widget );
            }
            break;

        case CE_MenuBarEmptyArea:
            if ( widget->window() )
                drawPrimitive( PE_Widget, option, painter, widget );
            break;

        case CE_DockWidgetTitle:
            if ( const QDockWidget* dockWidget = qobject_cast<const QDockWidget*>( widget ) ) {
                painter->save();
                if ( m_internalPalette == DotNetStyle::Classic ) {
                    painter->setBrush( option->palette.background().color().dark( 108 ) );
                    painter->setPen( option->palette.dark().color() );
                } else {
                    painter->setBrush( toolbarGradient( option, option->rect, Qt::Horizontal ) );
                    QColor color = gradientDarkColor( option ).darker( 120 );
                    painter->setPen( color );
                }
                painter->drawRect( rect.adjusted( 0, 2, -1, -3 ) );

                if ( const QStyleOptionDockWidget* dwOpt = qstyleoption_cast<const QStyleOptionDockWidget*>( option ) ) {
                    QRect titleRect = option->rect.adjusted( 6, 2, 0, 0 );
                    int buttonMargin = 4;
                    int mw = pixelMetric( QStyle::PM_DockWidgetTitleMargin, dwOpt, widget );
                    if ( dwOpt->closable ) {
                        QPixmap pm = standardIcon( QStyle::SP_TitleBarCloseButton, dwOpt, widget ).pixmap( 10, 10 );
                        titleRect.adjust( 0, 0, -pm.size().width() - mw - buttonMargin, 0 );
                    }
                    if ( dwOpt->floatable ) {
                        QPixmap pm = standardIcon( QStyle::SP_TitleBarMaxButton, dwOpt, widget ).pixmap( 10, 10 );
                        titleRect.adjust( 0, 0, -pm.size().width() - mw - buttonMargin, 0 );
                    }
                    titleRect =  visualRect( option->direction, option->rect, titleRect );

                    if ( !dwOpt->title.isEmpty() ) {
                        QString titleText = painter->fontMetrics().elidedText( dwOpt->title, Qt::ElideRight, titleRect.width() );
                        drawItemText( painter, titleRect,
                                    Qt::AlignLeft | Qt::AlignVCenter, dwOpt->palette,
                                    dwOpt->state & State_Enabled, titleText,
                                    QPalette::WindowText );
                    }
                    painter->restore();
                }
            }
            break;

        case CE_ToolBar:
            if ( const QStyleOptionToolBar* toolbar = qstyleoption_cast<const QStyleOptionToolBar*>( option ) ) {
                painter->save();
                rect.adjust( 1, 0, 0, 0 );

                if ( const QToolBar* tb =  qobject_cast<const QToolBar*>( widget ) ) {
                    if ( option->direction == Qt::LeftToRight )
                        rect.setRight( tb->childrenRect().right() + 2 );
                    else
                        rect.setLeft( tb->childrenRect().left() - 2 );
                }
                Qt::Orientation gradientOrientation =( toolbar->toolBarArea == Qt::RightToolBarArea ||
                    toolbar->toolBarArea == Qt::LeftToolBarArea ) ? Qt::Vertical : Qt::Horizontal;
                QLinearGradient gradient = toolbarGradient( toolbar, rect, gradientOrientation );
                painter->setBrush( gradient );
                painter->setPen( Qt::NoPen );

                bool roundEdges = true;
                if ( widget && widget->parentWidget() ) {
                    if ( const QToolBar* tb =  qobject_cast<const QToolBar*>( widget ) ) {
                        if ( !( tb->childrenRect().width() + 32 < widget->parentWidget()->geometry().width() ) ) {
                            roundEdges = false;
                            rect.adjust( -1, 0, 0, 0 );
                        }
                    }
                }
                if ( roundEdges ) {
                    rect = rect.adjusted( 1, 1, -1, 0 );
                    QRegion region = rect;
                    region -= QRect( rect.left(), rect.bottom(), 2, 1 );
                    region -= QRect( rect.right() - 1, rect.bottom(), 2, 1 );
                    region -= QRect( rect.left(),  rect.bottom() - 1, 1, 1 );
                    region -= QRect( rect.right(), rect.bottom() - 1, 1, 1 );
                    region -= QRect( rect.left(), rect.top(), 2, 1 );
                    region -= QRect( rect.right() - 1, rect.top(), 2, 1 );
                    region -= QRect( rect.left(),  rect.top() + 1, 1, 1 );
                    region -= QRect( rect.right(), rect.top() + 1, 1, 1 );
                    painter->setClipRegion( region );

                    painter->fillRect( rect, gradient );

                    painter->setPen( QPen( shadow, 0 ) );
                    painter->drawLine( rect.bottomLeft() + QPoint( 3,0 ), rect.bottomRight() - QPoint( 2,0 ) );
                    painter->drawPoint( rect.bottomRight() - QPoint( 1, 1 ) );
                    QColor alphashadow = shadow;
                    alphashadow.setAlpha( 128 );
                    painter->setPen( QPen( alphashadow, 0 ) );
                    painter->drawLine( rect.topRight() + QPoint( 0, 2 ), rect.bottomRight() - QPoint( 0, 2 ) );
                } else {
                    painter->fillRect( rect, gradient );
                }
                painter->restore();
            }
            break;

        case CE_TabBarTabShape:
            if ( QTabWidget* tabWidget = qobject_cast<QTabWidget*>( widget->parentWidget() ) ) {
                if ( qobject_cast<QMainWindow*>( tabWidget->parentWidget() ) ) {
                    if ( const QStyleOptionTab* tab = qstyleoption_cast<const QStyleOptionTab*>( option ) ) {
                        painter->save();

                        bool lastTab = tab->position == QStyleOptionTab::End;
                        bool firstTab = tab->position == QStyleOptionTab::Beginning;
                        bool onlyOne = tab->position == QStyleOptionTab::OnlyOneTab;

                        bool isDisabled = !(tab->state & State_Enabled);
                        bool hasFocus = tab->state & State_HasFocus;
                        bool isHot = tab->state & State_MouseOver;
                        bool selected = tab->state & State_Selected;

                        int tabOverlap = onlyOne ? 0 : pixelMetric(PM_TabBarTabOverlap, option, widget);

                        painter->setPen( highlightOutlineColor( option, selected ) );

                        if ( isHot || selected )
                            painter->setBrush( highlightTabBrush( option, selected ) );
                        else
                            painter->setBrush( tabGradient( tab, rect, Qt::Horizontal ) );

                        if ( selected )
                            rect.adjust( firstTab ? 0 : -tabOverlap, 0, ( lastTab ? 0 : tabOverlap ) - 1, -1 );
                        else
                            rect.adjust( 0, 1, lastTab ? -1 : 0, -2 );

                        painter->drawRect( rect );

                        painter->restore();
                        break;
                    }
                }
            }
            if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                QWindowsVistaStyle::drawControl( element, option, painter, widget );
            else
                QWindowsXPStyle::drawControl( element, option, painter, widget );
            break;

        default:
            if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                QWindowsVistaStyle::drawControl( element, option, painter, widget );
            else
                QWindowsXPStyle::drawControl( element, option, painter, widget );
            break;
    }
}

void DotNetStyle::drawComplexControl( ComplexControl control, const QStyleOptionComplex* option,
    QPainter* painter, const QWidget* widget ) const
{
    switch ( control ) {
        case CC_ToolButton:
            if ( const QStyleOptionToolButton* toolbutton = qstyleoption_cast<const QStyleOptionToolButton*>( option ) )
            {
                QRect button;
                button = subControlRect( control, toolbutton, SC_ToolButton, widget );
                if ( ( widget && qobject_cast<QToolBar*>( widget->parentWidget() ) ) &&
                    ( ( toolbutton->state & State_MouseOver && toolbutton->state & State_Enabled )
                    || toolbutton->state & State_On ) ) {
                    QLinearGradient menubargradient2( button.topLeft(), button.bottomLeft() );

                    bool act = ( toolbutton->state & State_Sunken ) || ( toolbutton->state & State_On );
                    painter->setPen( highlightOutlineColor( option, act ) );
                    painter->setBrush( highlightBrush( option, act ) );
                    painter->drawRect( button.adjusted( 0, 0, -2, -1 ) );

                    QStyleOptionComplex comp;
                    comp.init( widget );
                    comp.rect = button;
                    QStyleOptionToolButton label = *toolbutton;
                    int fw = pixelMetric( PM_DefaultFrameWidth, option, widget );
                    label.rect = button.adjusted( fw, fw, -fw, -fw );
                    label.features &= ~QStyleOptionToolButton::Arrow;
                    drawControl( CE_ToolButtonLabel, &label, painter, widget );

                    if ( toolbutton->subControls & SC_ToolButtonMenu ) {
                        QStyleOption tool = *toolbutton;
                        tool.rect = QWindowsStyle::subControlRect( control, toolbutton, SC_ToolButtonMenu, widget );
                        painter->setPen( highlightOutlineColor( option, act ) );
                        painter->setBrush( highlightBrush( option, toolbutton->activeSubControls & SC_ToolButtonMenu ) );
                        painter->drawRect( tool.rect.adjusted( -2, 0, -1, -1 ) );
                        tool.rect.adjust( 2, 3, -2, -2 );
                        drawPrimitive( PE_IndicatorArrowDown, &tool, painter, widget );
                    }
                } else
                    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                        QWindowsVistaStyle::drawComplexControl( control, option, painter, widget );
                    else
                        QWindowsXPStyle::drawComplexControl( control, option, painter, widget );
            }
            break;

        default:
            if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                QWindowsVistaStyle::drawComplexControl( control, option, painter, widget );
            else
                QWindowsXPStyle::drawComplexControl( control, option, painter, widget );
            break;
    }
}

QSize DotNetStyle::sizeFromContents( ContentsType type, const QStyleOption* option,
    const QSize& size, const QWidget* widget ) const
{
    QSize newSize;
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        newSize = QWindowsVistaStyle::sizeFromContents( type, option, size, widget );
    else
        newSize = QWindowsXPStyle::sizeFromContents( type, option, size, widget );

    switch ( type ) {
        case CT_MenuBarItem:
            newSize += QSize( 0, 3 );
            break;

        case CT_MenuItem:
            if ( const QStyleOptionMenuItem* menuitem = qstyleoption_cast<const QStyleOptionMenuItem*>( option ) ) {
                newSize = QWindowsStyle::sizeFromContents( type, option, size, widget );
                if ( menuitem->menuItemType == QStyleOptionMenuItem::Separator ){
                    newSize.setHeight( 2 );
                } else {
                    newSize += QSize( 10, 2 );
                    if ( newSize.height() < 22 )
                        newSize.setHeight( 22 );
                }
            }
            break;

        case CT_Menu:
            newSize = size;
            newSize -= QSize( 1,2 );
            break;

        default:
            break;
    }

    return newSize;
}

QRect DotNetStyle::subElementRect( SubElement element, const QStyleOption* option, const QWidget* widget ) const
{
    QRect rect;

    switch ( element ) {
        case SE_TabWidgetTabContents:
            if ( qobject_cast<QMainWindow*>( widget->parentWidget() ) ) {
                rect = QWindowsStyle::subElementRect( SE_TabWidgetTabPane, option, widget );
                rect.adjust( 0, 1, 0, 0 );
            } else {
                if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                    rect = QWindowsVistaStyle::subElementRect( element, option, widget );
                else
                    rect = QWindowsXPStyle::subElementRect( element, option, widget );
            }
            break;

        case SE_TabWidgetTabBar:
            if ( qobject_cast<QMainWindow*>( widget->parentWidget() ) ) {
                rect = QWindowsStyle::subElementRect( SE_TabWidgetTabBar, option, widget );
                rect.adjust( 0, 1, 0, 0 );
            } else {
                if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                    rect = QWindowsVistaStyle::subElementRect( element, option, widget );
                else
                    rect = QWindowsXPStyle::subElementRect( element, option, widget );
            }
            break;

        default:
            if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
                rect = QWindowsVistaStyle::subElementRect( element, option, widget );
            else
                rect = QWindowsXPStyle::subElementRect( element, option, widget );

            if ( element == SE_DockWidgetCloseButton || element == SE_DockWidgetFloatButton )
                rect.translate( 0, 1 );
            break;
    }

    return rect;
}

QRect DotNetStyle::subControlRect( ComplexControl control, const QStyleOptionComplex* option,
    SubControl subControl, const QWidget* widget ) const
{
    QRect rect;
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        rect = QWindowsVistaStyle::subControlRect( control, option, subControl, widget );
    else
        rect = QWindowsXPStyle::subControlRect( control, option, subControl, widget );

    return rect;
}

QStyle::SubControl DotNetStyle::hitTestComplexControl( ComplexControl control, const QStyleOptionComplex* option,
    const QPoint& pos, const QWidget* widget ) const
{
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        return QWindowsVistaStyle::hitTestComplexControl( control, option, pos, widget );
    else
        return QWindowsXPStyle::hitTestComplexControl( control, option, pos, widget );
}

int DotNetStyle::pixelMetric( PixelMetric metric, const QStyleOption* option, const QWidget* widget ) const
{
    int retval = 0;

    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        retval = QWindowsVistaStyle::pixelMetric( metric, option, widget );
    else
        retval = QWindowsXPStyle::pixelMetric( metric, option, widget );

    switch ( metric ) {
        case PM_ToolBarItemSpacing:
            retval = 2;
            break;
        case PM_ToolBarIconSize:
            retval = 22;
            break;
        case PM_MenuHMargin:
            retval = 1;
            break;
        case PM_MenuVMargin:
            retval = 1;
            break;
        case PM_MenuBarPanelWidth:
            retval = 0;
            break;
        case PM_MenuPanelWidth:
            retval = 1;
            break;
        case PM_DockWidgetTitleBarButtonMargin:
            retval = 6;
            break;
        case PM_DefaultFrameWidth:
        case PM_SpinBoxFrameWidth:
            if ( widget && qobject_cast<QToolBar*>( widget->parentWidget() ) )
                retval = 2;
            break;
        case PM_ButtonShiftVertical:
        case PM_ButtonShiftHorizontal:
            if ( widget && qobject_cast<QToolBar*>( widget->parentWidget() ) )
                retval = 0;
            break;
        case PM_TabBarBaseOverlap:
            retval = 1;
            break;
        default:
            break;
    }
    return retval;
}

QPalette DotNetStyle::standardPalette() const
{
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        return QWindowsVistaStyle::standardPalette();
    else
        return QWindowsXPStyle::standardPalette();
}

void DotNetStyle::updatePalette()
{
    m_internalPalette = Classic;

    const int maxlength = 256;
    WCHAR themeFileName[ maxlength ];
    WCHAR themeColor[ maxlength ];

    if ( pIsAppThemed && pIsAppThemed() && pGetCurrentThemeName( themeFileName, maxlength, themeColor, maxlength, NULL, 0 ) == S_OK ) {
        QString name = QString::fromWCharArray( themeFileName );
        QString color = QString::fromWCharArray( themeColor );
        if ( name.endsWith( "Luna.msstyles" ) ) {
            if ( color == "HomeStead" )
                m_internalPalette = Olive;
            else if ( color == "Metallic" )
                m_internalPalette = Silver;
            else
                m_internalPalette = Blue;
        } else if ( name.endsWith( "Aero.msstyles" ) ) {
                m_internalPalette = Blue;
        }
    }

    switch ( m_internalPalette ) {
        case Blue:
            m_menuSeparatorColor = QColor( 106, 140, 203 );
            m_menuBackgroundColor = QColor( 246, 246, 246 );
            m_menuFrameColor = QColor( 0, 45, 150 );
            break;

        case Silver:
            m_menuSeparatorColor = QColor( 110, 109, 143 );
            m_menuBackgroundColor = QColor( 254, 250, 255 );
            m_menuFrameColor = QColor( 124, 124, 148 );
            break;

        case Olive:
            m_menuSeparatorColor = QColor( 96, 128, 88 );
            m_menuBackgroundColor = QColor( 244, 244, 238 );
            m_menuFrameColor = QColor( 117, 141, 94 );
            break;

        default:
            m_menuFrameColor = m_currentPal.dark().color().lighter( 108 );
            m_menuSeparatorColor = m_currentPal.background().color().darker( 110 );
            m_menuBackgroundColor = m_currentPal.light().color();
            m_menuBackgroundColor.setHsv( m_menuBackgroundColor.hue(), 3 * m_menuBackgroundColor.saturation() / 8, 250 );
        break;
    }
}

void DotNetStyle::polish( QApplication* app )
{
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        QWindowsVistaStyle::polish( app );
    else
        QWindowsXPStyle::polish( app );

    m_currentPal = app->palette();
    updatePalette();
}

void DotNetStyle::unpolish( QApplication* app )
{
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        QWindowsVistaStyle::unpolish( app );
    else
        QWindowsXPStyle::unpolish( app );
}

void DotNetStyle::polish( QWidget* widget )
{
    if ( qobject_cast<QMainWindow*>( widget ) )
        widget->setAttribute( Qt::WA_StyledBackground );

    if ( qobject_cast<QTabBar*>( widget ) )
        widget->setAttribute( Qt::WA_Hover );

    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        QWindowsVistaStyle::polish( widget );
    else
        QWindowsXPStyle::polish( widget );
}

void DotNetStyle::unpolish( QWidget* widget )
{
    if ( qobject_cast<QMainWindow*>( widget ) )
        widget->setAttribute( Qt::WA_StyledBackground, false );

    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        QWindowsVistaStyle::unpolish( widget );
    else
        QWindowsXPStyle::unpolish( widget );
}

void DotNetStyle::polish( QPalette& pal )
{
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        QWindowsVistaStyle::polish( pal );
    else
        QWindowsXPStyle::polish( pal );
}

QPixmap DotNetStyle::standardPixmap( StandardPixmap standardPixmap, const QStyleOption* opt,
    const QWidget* widget ) const
{
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        return QWindowsVistaStyle::standardPixmap( standardPixmap, opt, widget );
    else
        return QWindowsXPStyle::standardPixmap( standardPixmap, opt, widget );
}

int DotNetStyle::styleHint( StyleHint hint, const QStyleOption* option, const QWidget* widget,
    QStyleHintReturn* returnData ) const
{
    int ret = 0;
    if ( QSysInfo::WindowsVersion >= QSysInfo::WV_VISTA && QSysInfo::WindowsVersion < QSysInfo::WV_NT_based )
        ret =  QWindowsVistaStyle::styleHint( hint, option, widget, returnData );
    else
        ret = QWindowsXPStyle::styleHint( hint, option, widget, returnData );
    return ret;
}

void DotNetStyle::fillGradient( const QStyleOption* option, QPainter* painter, const QWidget* widget )
{
    QColor gradientDark = gradientDarkColor( option );
    QColor gradientLight = gradientLightColor( option );

    QPoint begin = widget->rect().topRight();
    QPoint end = widget->rect().topLeft() + QPoint( 0, widget->geometry().height()/2 );

    QLinearGradient menubargradient( begin, end );
    menubargradient.setColorAt( 0, gradientLight );
    menubargradient.setColorAt( 0.8, gradientDark );
    menubargradient.setColorAt( 1, gradientDark );

    painter->fillRect( widget->rect(), menubargradient );
}

QColor DotNetStyle::gradientDarkColor( const QStyleOption* option ) const
{
    switch ( m_internalPalette ) {
        case Olive:
            return QColor( 217, 217, 167 );
        case Blue:
            return QColor( 158, 190, 245 );
        case Silver:
            return QColor( 216, 216, 229 );
        default:
            return option->palette.background().color().lighter( 104 );
    }
}

inline QColor DotNetStyle::gradientLightColor( const QStyleOption* option ) const
{
    switch ( m_internalPalette ) {
        case Olive:
            return QColor( 241, 240, 227 );
        case Blue:
            return QColor( 196, 217, 249 );
        case Silver:
            return QColor( 242, 242, 247 );
        default:
            return option->palette.light().color().lighter( 106 );
    }
}

inline QColor DotNetStyle::shadowColor( const QStyleOption* option ) const
{
    switch ( m_internalPalette ) {
        case Blue:
            return QColor( 59, 97, 156 );
        case Silver:
            return QColor( 124, 124, 148 );
        case Olive:
            return QColor( 96, 128, 88 );
        default:
            return option->palette.background().color();
    }
}

inline QColor DotNetStyle::highlightOutlineColor( const QStyleOption* option, bool act ) const
{
    switch ( m_internalPalette ) {
        case Blue:
            return QColor( 0, 0, 128 );
        case Silver:
            return QColor( 75, 75, 111 );
        case Olive:
            return QColor( 63, 93, 56 );
        default:
            if ( act )
                return option->palette.brush( QPalette::Active, QPalette::Highlight ).color().darker( 110 );
            else
                return option->palette.brush( QPalette::Active, QPalette::Highlight ).color().lighter( 110 );
    }
}

QBrush DotNetStyle::highlightBrush( const QStyleOption* option, bool act ) const
{
    QBrush brush;
    QColor highlight;

    switch ( m_internalPalette ) {
        case Silver:
        case Olive:
        case Blue:
            if ( const QStyleOptionMenuItem* menuitem = qstyleoption_cast<const QStyleOptionMenuItem*>( option ) ) {
                brush = QColor( 255, 238, 194 );
            } else {
                QLinearGradient gradient( option->rect.topLeft(), option->rect.bottomLeft() );
                if ( act ) {
                    gradient.setColorAt( 0, QColor( 254, 149, 82 ) );
                    gradient.setColorAt( 1, QColor( 255, 211, 142 ) );
                } else {
                    gradient.setColorAt( 0, QColor( 255, 252, 200 ) );
                    gradient.setColorAt( 1, QColor( 255, 208, 145 ) );
                }
                brush = gradient;
            }
            break;

        default:
            highlight = option->palette.brush( QPalette::Active, QPalette::Highlight ).color().lighter( 120 );
            if ( act )
                highlight.setHsv( highlight.hue(), 70, 220 );
            else
                highlight.setHsv( highlight.hue(), 40, 230 );
            brush = QBrush( highlight );
            break;
    }

    return brush;
}

QBrush DotNetStyle::highlightMenuCheckBrush( const QStyleOption* option, bool act ) const
{
    QBrush brush;
    QColor highlight;

    switch ( m_internalPalette ) {
        case Silver:
        case Olive:
        case Blue:
            if ( act )
                brush = QColor( 254, 128, 62 );
            else
                brush = QColor( 255, 192, 111 );
            break;

        default:
            highlight = option->palette.highlight().color().lighter( 120 );
            if ( act )
                highlight.setHsv( highlight.hue(), 120, 180 );
            else {
                highlight.setHsv( highlight.hue(), 30, 230 );
            }
            highlight.setAlpha( 170 );
            brush = highlight;
            break;
    }

    return brush;
}

QBrush DotNetStyle::highlightTabBrush( const QStyleOption* option, bool act ) const
{
    if ( act ) {
        QLinearGradient gradient( option->rect.topLeft(), option->rect.bottomLeft() );
        QColor background = option->palette.background().color();
        QColor start = background.lighter( 115 );
        gradient.setColorAt( 0, start );
        gradient.setColorAt( 1, background );
        return gradient;
    } else {
        switch ( m_internalPalette ) {
            case Silver:
            case Olive:
            case Blue:
                return highlightBrush( option, act );

            default:
                QLinearGradient gradient( option->rect.topLeft(), option->rect.bottomLeft() );
                QColor highlight = option->palette.brush( QPalette::Active, QPalette::Highlight ).color().lighter( 120 );
                highlight.setHsv( highlight.hue(), 40, 230 );
                QColor start = highlight.lighter( 115 );
                gradient.setColorAt( 0, start );
                gradient.setColorAt( 1, highlight );
                return gradient;
        }
    }
}

QLinearGradient DotNetStyle::toolbarGradient( const QStyleOption* option, QRect rect, Qt::Orientation orientation ) const
{
    QLinearGradient gradient( rect.topLeft(), orientation == Qt::Vertical ? rect.topRight() : rect.bottomLeft() );
    QColor start, stop;

    switch ( m_internalPalette )
    {
        case Blue:
            start = QColor( 220, 235, 254 );
            stop = QColor( 129, 169, 226 );
            gradient.setColorAt( 0.3, start );
            break;

        case Silver:
            start = QColor( 242, 242, 249 );
            stop = QColor( 153, 151, 181 );
            gradient.setColorAt( 0.3, start );
            break;

        case Olive:
            start = QColor( 241, 245, 217 );
            stop = QColor( 183, 198, 145 );
            gradient.setColorAt( 0.0, start );
            break;

        default:
            start = option->palette.light().color();
            start = start.lighter( 130 );
            start.setAlpha( 180 );
            stop = option->palette.button().color();
            stop.setHsv( stop.hue(), qMin( stop.saturation()*2, 255 ), ( stop.value()*2 )/3 );
            stop.setAlpha( 0 );
            gradient.setColorAt( 0.3, start );
            break;
    }

    gradient.setColorAt( 1, stop );

    return gradient;
}

QLinearGradient DotNetStyle::menuGradient( const QStyleOption* option, QRect rect, Qt::Orientation orientation ) const
{
    QColor start, stop;
    QLinearGradient grad;

    if ( orientation == Qt::Vertical ) {
        grad.setStart( rect.left(), rect.top() );
        grad.setFinalStop( rect.left(), rect.bottom() );
    } else {
        grad.setStart( QPoint( rect.left(), 0 ) );
        grad.setFinalStop( QPoint( rect.left() + 24, 0 ) );
    }

    switch ( m_internalPalette )
    {
        case Blue:
            start = QColor( 220, 235, 254 );
            stop = QColor( 129, 169, 226 );
            grad.setColorAt( 0.3, start );
            break;

        case Olive:
            start = QColor( 246, 248, 224 );
            stop = QColor( 186, 201, 148 );
            grad.setColorAt( 0.0, start );
            break;

        case Silver:
            start = QColor( 215, 215, 226 );
            stop = QColor( 138, 137, 166 );
            grad.setColorAt( 0.4, start );
            break;

        default:
            start = option->palette.brush( QPalette::Active, QPalette::Base ).color();
            stop = gradientDarkColor( option );
            stop.setHsv( stop.hue(), qMin( (int)( stop.saturation()*1.2 ), 255 ), ( 5*stop.value()/7 ) );
            stop.setAlpha( 120 );
            grad.setColorAt( 0, start );
            break;
    }

    grad.setColorAt( 1, stop );

    return grad;
}

QLinearGradient DotNetStyle::tabGradient( const QStyleOption* option, QRect rect, Qt::Orientation orientation ) const
{
    switch ( m_internalPalette )
    {
        case Blue:
        case Olive:
        case Silver:
            return toolbarGradient( option, rect, orientation );

        default:
            QLinearGradient gradient( rect.topLeft(), orientation == Qt::Vertical ? rect.topRight() : rect.bottomLeft() );
            QColor start = gradientLightColor( option );
            QColor stop = gradientDarkColor( option );
            gradient.setColorAt( 0, start );
            gradient.setColorAt( 1, stop );
            return gradient;
    }
}

#endif
