/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef HTMLHISTORYWRITER_H
#define HTMLHISTORYWRITER_H

#include <QStringList>

class QTextBrowser;

namespace WebIssues
{

/**
* Class for writing the issue history in HTML format.
*
* This class is responsible for creating the style sheet, adding HTML markup and writing
* the issue history into a QTextBrowser.
*/
class HTMLHistoryWriter
{
public:
    /**
    * Constuctor.
    *
    * @param browser The text browser to output the history to.
    */
    HTMLHistoryWriter( QTextBrowser* browser );

    /**
    * Destructor.
    */
    ~HTMLHistoryWriter();

public:
    /**
    * Begin the document.
    *
    * This method sets up the style sheet and writes the document header.
    */
    void begin();

    /**
    * End the document.
    *
    * This method writes the document footer.
    */
    void end();

    /**
    * Write a block with a header and a list of changes.
    *
    * @param header Header of the block in HTML format.
    *
    * @param items List of items in HTML format.
    */
    void writeChanges( const QString& header, const QStringList& items );

    /**
    * Write a block with a header, comment text and anchor.
    *
    * The name of the anchor is <tt>id{commentId}</tt>.
    *
    * @param header Header of the block in HTML format.
    *
    * @param text Content of the block in plain text format.
    *
    * @param commentId Identifier of the block's anchor.
    */
    void writeComment( const QString& header, const QString& text, int commentId );

    /**
    * Write a block with a header, attachment item and anchor.
    *
    * The name of the anchor is <tt>id{attachmentId}</tt>.
    *
    * @param header Header of the block in HTML format.
    *
    * @param item The attachment item in HTML format.
    *
    * @param attachmentId Identifier of the block's anchor.
    */
    void writeAttachment( const QString& header, const QString& item, int attachmentId );

public:
    /**
    * Add HTML markup to an attribute name or user name.
    *
    * Names are displayed with bold font.
    */
    static QString name( const QString& text );

    /**
    * Add HTML markup to a date.
    *
    * Dates are displayed with bold font.
    */
    static QString date( const QString& text );

    /**
    * Add HTML markup to an issue name or attribute value.
    *
    * URLs and issue numbers are converted to links.
    */
    static QString value( const QString& text );

    /**
    * Add HTML markup to an attachment description.
    *
    * URLs and issue numbers are converted to links.
    */
    static QString description( const QString& text );

    /**
    * Create a link to an attachment.
    *
    * The URL of the link is <tt>attachment://{attachmentId}</tt>.
    *
    * @param attachmentId Identifier of the attachmet.
    *
    * @param name The name of the attachment.
    */
    static QString attachment( int attachmentId, const QString& name );

private:
    QTextBrowser* m_browser;

    QString m_html;
};

}

#endif
