/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "dashboardview.h"

#include <QTextBrowser>
#include <QScrollBar>

#include "commands/commandmanager.h"
#include "data/rdb/tableiterators.h"
#include "data/datamanager.h"
#include "data/updatebatch.h"
#include "data/updateevent.h"
#include "models/folderwatchprovider.h"
#include "models/htmlwatchwriter.h"
#include "views/viewevent.h"
#include "viewmanager.h"

using namespace WebIssues;

/* TRANSLATOR WebIssues::DashboardView */

DashboardView::DashboardView( QObject* parent, QWidget* parentWidget ) : View( parent )
{
    m_browser = new QTextBrowser( parentWidget );
    m_browser->setContextMenuPolicy( Qt::PreventContextMenu );

    connect( m_browser, SIGNAL( anchorClicked( const QUrl& ) ), this, SLOT( anchorClicked( const QUrl& ) ) );

    setMainWidget( m_browser );

    m_provider = new FolderWatchProvider();

    setAccess( NormalAccess );
}

DashboardView::~DashboardView()
{
    delete m_provider;
}

void DashboardView::initialUpdate()
{
}

void DashboardView::updateEvent( UpdateEvent* e )
{
    if ( e->unit() == UpdateEvent::Projects ) {
        m_provider->updateAll();
        populateDashboard();
        updateWatchedFolders();
    } else if ( e->unit() == UpdateEvent::Folder ) {
        m_provider->updateFolder( e->id() );
        populateDashboard();
    }
}

void DashboardView::viewEvent( ViewEvent* e )
{
    switch ( e->action() ) {
        case ViewEvent::RecalculateWatches:
            m_provider->updateFolder( e->id() );
            populateDashboard();
            updateWatchedFolder( e->id() );
            break;

        case ViewEvent::UpdateFilters:
            m_provider->updateFolders( e->id() );
            populateDashboard();
            break;

        case ViewEvent::UpdateSettings:
            m_provider->updateAll();
            populateDashboard();
            break;
    }
    View::viewEvent( e );
}

void DashboardView::updateWatchedFolders()
{
    UpdateBatch* batch = NULL;

    RDB::TableConstIterator<FolderRow> it( dataManager->folders() );
    while ( it.next() ) {
        int folderId = it.get()->folderId();
        if ( dataManager->folderWatchState( folderId ) != NotWatched ) {
            if ( dataManager->folderUpdateNeeded( folderId ) ) {
                if ( !batch )
                    batch = new UpdateBatch( -20 );
                batch->updateFolder( folderId );
            }
        }
    }

    if ( batch )
        commandManager->execute( batch );
}

void DashboardView::updateWatchedFolder( int folderId )
{
    if ( dataManager->folderWatchState( folderId ) != NotWatched ) {
        if ( dataManager->folderUpdateNeeded( folderId ) ) {
            UpdateBatch* batch = new UpdateBatch( -20 );
            batch->updateFolder( folderId );
            commandManager->execute( batch );
        }
    }
}

void DashboardView::populateDashboard()
{
    if ( m_provider->isEmpty() ) {
        QString info;
        info += "<style>p { margin-left: 3px; margin-right: 3px; }</style>";
        info += "<p><strong>" + tr( "There are no active folder watches." ) + "</strong></p>";
        info += "<p>" + tr( "To create a watch, select a folder in the Projects tab and select 'Folder Watches...'." ) + "</p>";

        m_browser->setHtml( info );
    } else {
        HTMLWatchWriter writer( m_browser );
    
        int pos = m_browser->verticalScrollBar()->sliderPosition();
        m_provider->write( &writer );
        m_browser->verticalScrollBar()->setSliderPosition( pos );
    }
}

void DashboardView::anchorClicked( const QUrl& url )
{
    int pos = m_browser->verticalScrollBar()->sliderPosition();
    m_browser->setSource( QUrl() );
    m_browser->verticalScrollBar()->setSliderPosition( pos );

    if ( url.scheme() == "watch" ) {
        int folderId = url.host().toInt();
        QString encoded = url.path().mid( 1 );
        QString filter = QUrl::fromPercentEncoding( encoded.toAscii() );

        viewManager->openFolderView( folderId, filter );
    }
}
