/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef PREVIEWWIDGET_H
#define PREVIEWWIDGET_H

#include <QtGlobal>

#if ( QT_VERSION < 0x040400 )
#include <QGraphicsView>
#else
#include <QPrintPreviewWidget>
#endif

#include <QPrinter>

class PreviewPaintEngine;

/**
* Printer supporting print preview.
*
* In Qt 4.4 and later it's simply equivalent to QPrinter. Before Qt 4.4,
* it implements the print preview capability.
*/
class PreviewPrinter : public QPrinter
{
public:
    /**
    * Default constructor.
    */
    PreviewPrinter();

    /**
    * Destructor.
    */
    ~PreviewPrinter();

#if ( QT_VERSION < 0x040400 )
public:
    PreviewPaintEngine* previewEngine() const { return m_engine; }

    void setPreviewMode( bool preview );

private:
    QPrinter* m_printer;

    PreviewPaintEngine* m_engine;
#endif // ( QT_VERSION < 0x040400 )
};

/**
* Widget for showing print preview.
*
* In Qt 4.4 and later it's simply equivalent to QPrintPreviewWidget. Before Qt 4.4,
* the print preview widget is emulated.
*/
#if ( QT_VERSION < 0x040400 )
class PreviewWidget : public QGraphicsView
#else
class PreviewWidget : public QPrintPreviewWidget
#endif
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param printer The printer used by this widget.
    * @param parent The parent widget.
    */
    PreviewWidget( PreviewPrinter* printer, QWidget* parent );

    /**
    * Destructor.
    */
    ~PreviewWidget();

#if ( QT_VERSION < 0x040400 )
public:
    enum ViewMode {
        SinglePageView,
        FacingPagesView,
        AllPagesView
    };

    enum ZoomMode {
        CustomZoom,
        FitToWidth,
        FitInView
    };

public:
    qreal zoomFactor() const { return m_zoomFactor; }
    QPrinter::Orientation orientation() const;
    ViewMode viewMode() const { return m_viewMode; }
    ZoomMode zoomMode() const { return m_zoomMode; }

    int currentPage() const { return m_currentPage; }
    int numPages() const;

public slots:
    void print();

    void zoomIn( qreal zoom = 1.1 );
    void zoomOut( qreal zoom = 1.1 );
    void setZoomFactor( qreal zoomFactor );
    void setOrientation( QPrinter::Orientation orientation );
    void setViewMode( ViewMode viewMode );
    void setZoomMode( ZoomMode zoomMode );
    void setCurrentPage( int pageNumber );

    void fitToWidth();
    void fitInView();
    void setLandscapeOrientation();
    void setPortraitOrientation();
    void setSinglePageViewMode();
    void setFacingPagesViewMode();
    void setAllPagesViewMode();

    void updatePreview();

public: // overrides
    void setVisible( bool visible );

signals:
    void paintRequested(QPrinter *printer);
    void previewChanged();

protected: // overrides
    void resizeEvent( QResizeEvent* e );
    void showEvent( QShowEvent* e );

private slots:
    void updateCurrentPage();

private:
    void fit( bool doFitting = false );
    void populateScene();
    void layoutPages();
    void generatePreview();
    int calcCurrentPage();

private:
    PreviewPrinter* m_printer;

    QGraphicsScene* m_scene;

    ViewMode m_viewMode;
    ZoomMode m_zoomMode;
    qreal m_zoomFactor;

    int m_currentPage;

    QList<const QPicture*> m_pictures;
    QList<QGraphicsItem*> m_pages;

    bool m_initialized;
    bool m_fitting;
#endif // ( QT_VERSION < 0x040400 )
};

#endif
