/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "button.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "label.h"
#include "mixer.h"
#include "keysym.h"
#include "debug.h"
#include <sigc++/bind.h>
#include <sigc++/object_slot.h>

namespace wftk {

Button::Button() :
  SingleContainer(0, 0, 0, 0),
  pressed(this, "press"),
  released(this, "release"),
  clicked(this),
  isPressed_(false),
  autoRepeat_(100)
{
  autoRepeat_.alarm.connect(slot(*this, &Button::repeat));
  autoRepeat_.halt();

  setClickToFocus(true);

  getResourceBackground("button");
}

Button::Button(const std::string& text, const Font &font) :
  SingleContainer(0, 0, 0, 0),
  pressed(this, "press"),
  released(this, "release"),
  clicked(this),
  isPressed_(false),
  autoRepeat_(100)
{
  autoRepeat_.alarm.connect(slot(*this, &Button::repeat));
  autoRepeat_.halt();

  setClickToFocus(true);

  getResourceBackground("button");

  pack(new Label(text, font));
}

SigC::Connection Button::ButtonSignal::connect(const SigC::Slot0<void>& slot)
{
  return SigC::Signal0<void>::connect(slot);
}

SigC::Connection Button::ButtonSignal::connect(const SigC::Slot1<void,Button*>& slot)
{
  return SigC::Signal0<void>::connect(SigC::bind(slot, button_));
}

void Button::ButtonSignal::operator()()
{
  if(!sample_.empty())
    Mixer::instance()->playSample(sample_);
  SigC::Signal0<void>::operator()();
}

bool Button::buttonEvent(Mouse::Button button, bool pressed_val, const Point& pos)
{
  if(Widget::buttonEvent(button, pressed_val, pos))
    return true;

  if(button != Mouse::LEFT)
    return false;

  if(pressed_val != isPressed_)
    invalidate();

  ref(); // in case it's unrefed during signal emmission, so we don't get a seg fault

  if(pressed_val && !isPressed_)
    {
      isPressed_ = true;
      pressed();
    }

  if(isPressed_ && !pressed_val)
    {
      isPressed_ = false;
      released();
      clicked();
    }

  unref(); // drop the reference we took earlier

  return true;
}

bool Button::keyEvent(const SDL_keysym& sym, bool pressed_val)
{
  if(Widget::keyEvent(sym, pressed_val))
    return true;

  char input = sym.unicode & 0x7f;

  if(pressed_val)
    {
      if((input =='\n') || (input == '\r') || (input == ' '))
	{
	  isPressed_ = true;
	  pressed();
	  invalidate();		  
	  return true;
	}

      if(sym.sym == (int) KEY_TAB)
	{
	  if(sym.mod & KEY_MOD_SHIFT)
	    Focus::instance()->prevFocus();
	  else
	    Focus::instance()->nextFocus();
	  return true;
	}
    }

  if(!pressed_val && isPressed_ )
    {
      if((input =='\n') || (input == '\r') || (input == ' '))
	{
	  isPressed_ = false;
	  released();
	  clicked();	  
	  invalidate();
	  return true;
	}
    }

  return false;
}

void Button::lostMouse()
{
  if(isPressed_)
    {
      isPressed_ = false;
      released();
      invalidate();
    }
}

void Button::press()
{
  if(isPressed_)
    {
      pressed();
    }
  else
    {
      isPressed_ = true;
      pressed();
      //request screen update
      invalidate();
    }
}

void Button::release()
{
  if(isPressed_)
    {
      isPressed_ = false;
      released();
      clicked();
      //request screen update
      invalidate();
    }
  else
    {
      released();
    }
}

void Button::toggle()
{
 if(isPressed())
   {
      //the button was pressed, pressing again releases
      isPressed_ = false;
      released();
   }
 else
   {
      isPressed_ = true;
      pressed();
   }
}

void Button::repeat()
{
  if(isPressed_)
    {
      isPressed_ = false;
      clicked();
      isPressed_ = true;
    }
}

void Button::setRepeat(unsigned repeat)
{
  if(repeat) {
    autoRepeat_.setInterval(repeat);
    autoRepeat_.run();
  }
  else
    autoRepeat_.halt();
}

void Button::setRepeat(bool repeat)
{
  if(repeat)
    autoRepeat_.run();
  else
    autoRepeat_.halt();
}

} // namespace wftk
