/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "focus.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "widget.h"

#include <assert.h>

using namespace wftk;

//initialize static member
Focus* Focus::instance_ = 0;

bool
Focus::grab(Widget* candidat)
{
  if(candidat == owner_)
    return true;

  if(candidat && !candidat->acceptsFocus())
    return false;

  Widget* old = owner_;

  owner_ = candidat;

  if(candidat)
    candidat->gainedFocus();
  if(old)
    old->lostFocus();

  return true;
}

Widget* Focus::nextFocus()
{
  if(focusable_.empty())
    return 0;

  if(owner_) {
    Foci::iterator I = find(owner_);
    ++I;
    if(I == focusable_.end())
      I = focusable_.begin();
    grab(*I);
  }
  else
    grab(focusable_.front());

  return owner_;
}

Widget* Focus::prevFocus()
{
  if(focusable_.empty())
    return 0;

  if(owner_) {
    Foci::iterator I = find(owner_);
    if(I == focusable_.begin())
      I = focusable_.end();
    --I;
    grab(*I);
  }
  else
    grab(focusable_.front());

  return owner_;
}

void Focus::addFocusable(Widget* w)
{
  assert(w && w->acceptsFocus());
  if(find(w) != focusable_.end())
    return;
  focusable_.push_back(w);
}

void Focus::removeFocusable(Widget* w)
{
  assert(w && !w->acceptsFocus());
  if(owner_ && w == owner_)
    grab(0);
  Foci::iterator I = find(w);
  if(I != focusable_.end())
    focusable_.erase(I);
}

bool
Focus::handleEvent(const SDL_Event* event)
{
  switch(event->type) { 
    case SDL_KEYDOWN:
    case SDL_KEYUP:
      {
        bool pressed = (event->key.state == SDL_PRESSED);
        return keystroke.emit(event->key.keysym, pressed) || (have_focus_
          && owner_ && owner_->keyEvent(event->key.keysym, pressed));
      }
    case SDL_ACTIVEEVENT:
      assert(event->active.state & SDL_APPINPUTFOCUS);
      have_focus_ = (event->active.gain != 0);
      return have_focus_ ? gotFocus.emit() : lostFocus.emit();
    default:
      assert(false);
      return false;
  }
}

Focus::Foci::iterator Focus::find(Widget *sa)
{
  if(!sa || !sa->acceptsFocus())
    return focusable_.end();

  Foci::iterator I = focusable_.begin();

  while(I != focusable_.end()) {
    if(*I == sa)
      break;
    ++I;
  }

  return I;
}
