/* vim: set noet ts=4:
 * $Id: wmpuzzle.c,v 1.24 2002/04/17 17:13:52 godisch Exp $
 *
 * Copyright (c) 2002 Martin A. Godisch <godisch@tcs.inf.tu-dresden.de>
 */

#include <errno.h>
#include <getopt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <X11/xpm.h>

#include "wmgeneral.h"
#include "wmpuzzle.h"

#include "xpm/debian.xpm"
#include "xpm/eagle.xpm"
#include "xpm/earth.xpm"
#include "xpm/linux.xpm"

#define PROGRAM_NAME     "wmpuzzle"
#define PROGRAM_VERSION  "0.1.1"

#define DEFAULT_RC_FILE  ".wmpuzzlerc"

#define TILE_X(N)   ((N) % 4 * 14 + 4)
#define TILE_Y(N)   ((N) / 4 * 14 + 4)
#define BLANK_X     (0)
#define BLANK_Y     (64)
#define LAST_X      (14)
#define LAST_Y      (64)
#define TEMP_X      (28)
#define TEMP_Y      (64)
#define TILE_WIDTH  (14)
#define TILE_HEIGHT (14)

#define MAX_STRING (256)

int  wmpuzzle_mask_width  = 64;
int  wmpuzzle_mask_height = 64;
char wmpuzzle_mask_bits[64*64];
char rcname[MAX_STRING];
char image[MAX_STRING]    = "eagle";
char puzzle[16]           = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
int  blank                = 15;

int main(int argc, char *argv[])
{
	int    n;
	int    but_stat       = -1;
	int    use_keyboard   =  0;
	int    shuffle_count  = -1;
	char   **wmpuzzle_xpm = eagle_xpm;
	char   state[16]      = {0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15};
	XEvent Event;

	if (sizeof(char) != 1) {
		fprintf(stderr, "%s: sizeof(char) == %d != 1\n", PROGRAM_NAME, sizeof(char));
		fprintf(stderr, "%s: please contact the author...\n", PROGRAM_NAME);
		exit(1);
	}

	if (signal(SIGTERM, handler) == SIG_ERR) {
		fprintf(stderr, "%s: cannot set handler for %d\n", PROGRAM_NAME, SIGTERM);
		perror("signal");
	}
	do_opts(argc, argv, &wmpuzzle_xpm, &use_keyboard, &shuffle_count);

	if (strlen(getenv("HOME")) < MAX_STRING - strlen(DEFAULT_RC_FILE) - 5) {
		strcpy(rcname, getenv("HOME"));
		strcat(rcname, "/");
		strcat(rcname, DEFAULT_RC_FILE);
	} else {
		fprintf(stderr, "%s: $HOME is too long, using rcfile %s in current directory\n", PROGRAM_NAME, DEFAULT_RC_FILE);
		strcpy(rcname, DEFAULT_RC_FILE);
	}

	createXBMfromXPM(wmpuzzle_mask_bits, wmpuzzle_xpm, wmpuzzle_mask_width, wmpuzzle_mask_height);
	openXwindow(argc, argv, wmpuzzle_xpm, wmpuzzle_mask_bits, wmpuzzle_mask_width, wmpuzzle_mask_height);
	for (n = 0; n < 16; n++)
		AddMouseRegion(n, TILE_X(n), TILE_Y(n), TILE_X(n) + TILE_WIDTH - 1, TILE_Y(n) + TILE_HEIGHT - 1);
	if (shuffle_count >= 0 || !read_state(state))
		shuffle(shuffle_count == -1 ? 2000 : shuffle_count, state);
	copyXPMArea(TILE_X(15), TILE_Y(15), TILE_WIDTH, TILE_HEIGHT, LAST_X, LAST_Y);
	prepare(state);

	while (1) {
		while (XPending(display)) {
			XNextEvent(display, &Event);
			switch (Event.type) {
			case Expose:
				RedrawWindow();
				break;
			case DestroyNotify:
				write_state();
				XCloseDisplay(display);
				exit(0);
			case ButtonPress:
				but_stat = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
				break;
			case ButtonRelease:
				switch (Event.xbutton.button) {
				case Button1:
					n = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
					if (but_stat >= 0 && n == but_stat && valid(n)) {
						move(n);
						RedrawWindow();
					}
					break;
				case Button2:
					if (read_state(state)) {
						prepare(state);
						RedrawWindow();
					}
					break;
				case Button3:
					write_state();
					break;
				}
				but_stat = -1;
				break;
			case KeyPress:
				if (!use_keyboard)
					break;
				switch (Event.xkey.keycode) {
				case 27: /* r */
					if (read_state(state)) {
						prepare(state);
						RedrawWindow();
					}
					break;
				case 39: /* s */
					write_state();
					break;
				case  98: /* up */
					if (blank >= 12)
						break;
					move(blank + 4);
					break;
				case 100: /* left */
					if (blank % 4 == 3)
						break;
					move(blank + 1);
					break;
				case 102: /* right */
					if (blank % 4 == 0)
						break;
					move(blank - 1);
					break;
				case 104: /* down */
					if (blank < 4)
						break;
					move(blank - 4);
					break;
				}
				but_stat = -1;
				RedrawWindow();
				break;
			case EnterNotify:
				if (use_keyboard)
					XSetInputFocus(display, PointerRoot, RevertToParent, CurrentTime);
				break;
			case LeaveNotify:
				if (use_keyboard)
					XSetInputFocus(display, PointerRoot, RevertToParent, CurrentTime);
				break;
			}
		}
		usleep(50000L);
	}
}

void do_opts(int argc, char **argv, char ***wmpuzzle, int *keyboard, int *shuffle)
{
	char *s;
	int  i;

	static struct option long_opts[] = {
		{"help",     0, NULL, 'h'},
		{"image",    1, NULL, 'i'},
		{"keyboard", 0, NULL, 'k'},
		{"shuffle",  1, NULL, 's'},
		{"version",  0, NULL, 'v'},
		{NULL,       0, NULL, 0  }};
	int opt_index = 0;

	while (1) {
		i = getopt_long(argc, argv, "hi:ks:v", long_opts, &opt_index);
		if (i == -1)
			break;
		switch (i) {
		case 'i':
			if (!strcmp(optarg, "debian"))
				*wmpuzzle = debian_xpm;
			else if (!strcmp(optarg, "eagle"))
				*wmpuzzle = eagle_xpm;
			else if (!strcmp(optarg, "earth"))
				*wmpuzzle = earth_xpm;
			else if (!strcmp(optarg, "linux"))
				*wmpuzzle = linux_xpm;
			else {
				fprintf(stderr, "%s: invalid image `%s'\n", PROGRAM_NAME, optarg);
				exit(1);
			}
			strcpy(image, optarg);
			break;
		case 'k':
			*keyboard = 1;
			break;
		case 's':
			i = strtol(optarg, &s, 10);
			if (*s || i <= 0) {
				fprintf(stderr, "%s: invalid shuffle count `%s'\n", PROGRAM_NAME, optarg);
				exit(1);
			}
			*shuffle = i;
			break;
		case 'h':
			printf("Usage: %s [-h] [-i <image>] [-k] [-s <count>] [-v]\n", PROGRAM_NAME);
			printf("  -h, --help             displays this command line summary.\n");
			printf("  -i, --image <image>    uses <image>, valid images are `debian', `eagle',\n");
			printf("                         `earth', and `linux'.\n");
			printf("  -k, --keyboard         enables the arrow keys on the keyboard.\n");
			printf("  -s, --shuffle <count>  shuffles the image <count> times.\n");
			printf("  -v, --version          displays the version number.\n");
			exit(0);
		case 'v':
			printf("%s version %s\n", PROGRAM_NAME, PROGRAM_VERSION);
			exit(0);
		case ':':
			exit(1);
		case '?':
			exit(1);
		}
	}
}

static void handler(int signo)
{
	write_state();
	exit(0);
}

void move(int n)
{
	int i;

	copyXPMArea(TILE_X(n), TILE_Y(n), TILE_WIDTH, TILE_HEIGHT, TILE_X(blank), TILE_Y(blank));
	copyXPMArea(BLANK_X,   BLANK_Y,   TILE_WIDTH, TILE_HEIGHT, TILE_X(n),     TILE_Y(n));
	puzzle[blank] = puzzle[n];
	puzzle[n] = 15;
	blank = n;
	if (n != 15)
		return;
	for (i = 0; i < 16; i++)
		if (puzzle[i] != i)
			return;
	copyXPMArea(LAST_X, LAST_Y, TILE_WIDTH, TILE_HEIGHT, TILE_X(n), TILE_Y(n));
}

void prepare(const char *state)
{
	int i, j, k;

	for (i = 0; i < 16; i++) {
		for (j = 0; puzzle[j] != state[i]; j++);
		if (i == j)
			continue;
		copyXPMArea(TILE_X(i), TILE_Y(i), TILE_WIDTH, TILE_HEIGHT, TEMP_X,    TEMP_Y);
		copyXPMArea(TILE_X(j), TILE_Y(j), TILE_WIDTH, TILE_HEIGHT, TILE_X(i), TILE_Y(i));
		copyXPMArea(TEMP_X,    TEMP_Y,    TILE_WIDTH, TILE_HEIGHT, TILE_X(j), TILE_Y(j));
		k         = puzzle[i];
		puzzle[i] = puzzle[j];
		puzzle[j] = k;
	}
	copyXPMArea(BLANK_X, BLANK_Y, TILE_WIDTH, TILE_HEIGHT, TILE_X(blank), TILE_Y(blank));
}

char *read_state(char *state)
{
	FILE *F;
	char buffer[MAX_STRING], *p, *s, temp[17];
	int  i, line;

	F = fopen(rcname, "r");
	if (!F) {
		if (errno != ENOENT) {
			fprintf(stderr, "%s: cannot open rcfile %s\n", PROGRAM_NAME, rcname);
			perror("fopen");
		}
		return(NULL);
	}

	for (line = 1;; line++) {
		fgets(buffer, MAX_STRING, F);
		if (feof(F))
			break;
		if ((s = strchr(buffer, '\n')))
			*s = '\0';
		if (buffer[0] == '#' || buffer[0] == '\0')
			continue;
		s = strchr(buffer, ':');
		if (!s) {
			fprintf(stderr, "%s: missing `:' in %s line %d\n", PROGRAM_NAME, rcname, line);
			continue;
		}
		*s = '\0';
		s++;
		if (strcmp(buffer, image))
			continue;
		if (strlen(s) != 16) {
			fprintf(stderr, "%s: invalid state in %s line %d\n", PROGRAM_NAME, rcname, line);
			fclose(F);
			return(NULL);
		}
		s += 15;
		for (i = 15; i >= 0; i--) {
			temp[i] = strtol(s--, &p, 16);
			if (*p) {
				fprintf(stderr, "%s: invalid state in %s line %d\n", PROGRAM_NAME, rcname, line);
				fclose(F);
				return(NULL);
			}
			if (temp[i] == 15)
				blank = i;
			s[1] = '\0';
		}
		memcpy(state, temp, 16);
		fclose(F);
		return(state);
	}
	fclose(F);
	return(NULL);
}

void shuffle(int n, char *state)
{
	int    i = blank, j = blank;
	time_t t;
 
	srand(time(&t));
	for (; n; n--) {
		/* FIXME: optimize this! */
		while (!valid(i) || i == j)
			i = rand() % 16;
		j = state[blank];
		state[blank] = state[i];
		state[i] = j;
		j = blank;
		blank = i;
	}
}

int valid(int n)
{
	return(
		(n / 4 == blank / 4 && (n % 4 == blank % 4 + 1 || n % 4 == blank % 4 - 1)) ||
		(n % 4 == blank % 4 && (n / 4 == blank / 4 + 1 || n / 4 == blank / 4 - 1)));
}

void write_state(void)
{
	FILE *F, *G;
	char *s, buffer[MAX_STRING], entry[MAX_STRING], tmpname[MAX_STRING + 4];
	int  line, written = 0;

	strcpy(tmpname, rcname);
	strcat(tmpname, ".new");
	umask(0077);
	F = fopen(tmpname, "w");
	if (!F) {
		fprintf(stderr, "%s: cannot open tempfile `%s'\n", PROGRAM_NAME, tmpname);
		perror("fopen");
		return;
	}

	G = fopen(rcname, "r");
	if (!G && errno != ENOENT) {
		fprintf(stderr, "%s: cannot open rcfile `%s'\n", PROGRAM_NAME, rcname);
		perror("fopen");
		fclose(F);
		unlink(tmpname);
		return;
	}

	sprintf(entry, "%s:%x%x%x%x%x%x%x%x%x%x%x%x%x%x%x%x\n", image,
		puzzle[0],  puzzle[1],  puzzle[2],  puzzle[3],
		puzzle[4],  puzzle[5],  puzzle[6],  puzzle[7],
		puzzle[8],  puzzle[9],  puzzle[10], puzzle[11],
		puzzle[12], puzzle[13], puzzle[14], puzzle[15]);

	if (G) {
		for (line = 1;; line++) {
			if (line > 1)
				fputs(buffer, F);
			fgets(buffer, MAX_STRING, G);
			if (feof(G))
				break;
			if (buffer[0] == '#' || buffer[0] == '\n')
				continue;
			s = strchr(buffer, ':');
			if (!s) {
				fprintf(stderr, "%s: missing `:' in %s line %d\n", PROGRAM_NAME, rcname, line);
				continue;
			}
			if (!strncmp(buffer, image, s - buffer)) {
				strcpy(buffer, entry);
				written = 1;
			}
		}
		fclose(G);
	}
	if (!written)
		fputs(entry, F);
	fclose(F);

	if (rename(tmpname, rcname)) {
		fprintf(stderr, "%s: cannot overwrite rcfile `%s'\n", PROGRAM_NAME, rcname);
		perror("rename");
		return;
	}
}
