/*  wmusic - a xmms remote-controlling DockApp
 *  Copyright (C) 2000-2001 Bastien Nocera <hadess@hadess.net>
 *  Maintained by John Chapin <john-wmusic@skinnee.cisat.jmu.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define DBL_CLICK_INTERVAL 250	/* double click interval in milliseconds */
#define ARROW_INTERVAL 100	/* arrow update interval in milliseconds */
#define SCROLL_INTERVAL 300	/* scroll update interval in milliseconds */
#define SEPARATOR " ** "	/* The separator for the scrolling title */

#include <dockapp.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <xmms/xmmsctrl.h>

#include "version.h"
#include "config.h"
#include "wmusic-master.xpm"
#include "wmusic-digits.xpm"

/*---------------------------------------------------------------------------*/
/*                             Prototypes                                    */
/*---------------------------------------------------------------------------*/

void copyNumArea(int x, int y, int sx, int sy, int dx, int dy);
void ActionPlay(int x, int y, DARect rect, void *data);
void ActionPause(int x, int y, DARect rect, void *data);
void ActionEject(int x, int y, DARect rect, void *data);
void ActionPrev(int x, int y, DARect rect, void *data);
void ActionNext(int x, int y, DARect rect, void *data);
void ActionStop(int x, int y, DARect rect, void *data);
void ActionFastr(int x, int y, DARect rect, void *data);
void ActionFastf(int x, int y, DARect rect, void *data);
void ToggleWins(int x, int y, DARect rect, void *data);
void ToggleVol(int x, int y, DARect rect, void *data);
void ToggleTitle(int x, int y, DARect rect, void *data);
void ToggleTime(int x, int y, DARect rect, void *data);
void buttonPress(int button, int state, int x, int y);
void buttonRelease(int button, int state, int x, int y);
void DisplayRoutine();
void DrawPos (int pos);
void DrawTime(int time);
void DrawArrow(void);
void DrawChannels(int channels);
void DrawKbps(int bps);
void DrawTitle(char *title);
void ExecuteXmms(void);

/*----------------------------------------------------------------------------*/
/*                             Variables                                      */
/*----------------------------------------------------------------------------*/

/* X11 variables */
char *displayName = "";
GC gc;
XEvent ev;

/* Dockapp variables */
unsigned int xmms_session = 0;
char *xmms_cmd = "xmms";
Bool pl_vis=0, eq_vis=0;
Bool xmms_running;
unsigned int volume_step = 5;

Bool t_scroll=0;
Bool t_time=0;
float title_pos = 0;
unsigned int arrow_pos = 0;
Bool pause_norotate = 0;
Time click_time=0;

Pixmap pixmap, mask;	/* Pixmap that is displayed */
Pixmap pixnum, masknum;	/* Pixmap source */

static DAActionRect buttonRects[] = {
	{{5, 39, 14, 9}, ActionPrev},
	{{19, 39, 14, 9}, ActionNext},
	{{33, 39, 13, 9}, ActionFastr},
	{{46, 39, 13, 9}, ActionFastf},
	{{5, 48, 11, 11}, ActionEject},
	{{16, 48, 21, 11}, ActionPlay},
	{{37, 48, 11, 11}, ActionPause},
	{{48, 48, 11, 11}, ActionStop}
};

static DAActionRect toggleRect[] = {
	{{5, 5, 54, 30}, ToggleWins}
};

static DAActionRect globRect[] = {
	{{0, 0, 64, 64}, ToggleVol}
};

static DAActionRect displayRects[] = {
	{{5, 5, 54, 15}, ToggleTime},
	{{5, 19, 54, 15}, ToggleTitle}
};

static DAProgramOption options[] = {
	{"-c", "--command", "Command to launch xmms", DOString, False,
		{&xmms_cmd} },
	{"-d", "--display", "Display to use", DOString, False, {&displayName} },
	{"-r", "--run", "Run xmms on startup", DONone, False, {NULL} },
	{"-s", "--session", "Xmms session to use", DONatural, False,
		{&xmms_session} },
	{"-t", "--title", "Show title instead of kbps by default", DONone,
		False, {NULL} },
	{"-u", "--hide", "Hide xmms windows on startup", DONone, False,
		{NULL} },
	{"-V", "--volume", "Stepping of the wheel volume control (in percent)",
		DONatural, False, {&volume_step} },
	{"-x", "--wmxmms", "Use wmxmms-like show and hide", DONone, False,
		{NULL} },
	{"-a", "--rotate-arrow", "Do not rotate the arrow, when paused", 
		DONone, False, {NULL} },
	{"-l", "--time-left", "Show time left instead of time remaining by default",
	        DONone, False, {NULL} }
};

/*----------------------------------------------------------------------------*/
/*                              Functions                                     */
/*----------------------------------------------------------------------------*/

void copyNumArea(int x, int y, int sx, int sy, int dx, int dy)
{
	XCopyArea(DADisplay, pixnum, pixmap, gc, x, y, sx, sy, dx, dy);
}

void buttonDraw(DARect rect)
{
	copyNumArea((rect.x)-5, (rect.y)-8, rect.width, rect.height,
			rect.x, rect.y);
	DASetPixmap(pixmap);
}

void ActionPlay(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_play(xmms_session);
	}
}

void ActionPause(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_pause(xmms_session);
	}
}

void ActionEject(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_eject(xmms_session);
	}
}

void ActionPrev(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_playlist_prev(xmms_session);
	}
}

void ActionNext(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_playlist_next(xmms_session);
	}
}

void ActionStop(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_stop(xmms_session);
	}
}

void ActionFastr(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_jump_to_time(xmms_session,
			xmms_remote_get_output_time(xmms_session)-10000);
	}
}

void ActionFastf(int x, int y, DARect rect, void *data)
{
	if (data) {
		buttonDraw(rect);
	} else {
		xmms_remote_jump_to_time(xmms_session,
			xmms_remote_get_output_time(xmms_session)+10000);
	}
}

void ToggleWins(int x, int y, DARect rect, void *data)
{
	if (xmms_running)
	{
		if (options[7].used)
		{
			if (data)
			xmms_remote_main_win_toggle(xmms_session,
					!xmms_remote_is_main_win(xmms_session));
			if (!data)
			xmms_remote_pl_win_toggle(xmms_session,
					!xmms_remote_is_pl_win(xmms_session));
		} else {
			if (xmms_remote_is_main_win(xmms_session))
			{
				pl_vis = xmms_remote_is_pl_win(xmms_session);
				eq_vis = xmms_remote_is_eq_win(xmms_session);

				xmms_remote_main_win_toggle(xmms_session, 0);
				xmms_remote_pl_win_toggle(xmms_session, 0);
				xmms_remote_eq_win_toggle(xmms_session, 0);
			} else {
				xmms_remote_main_win_toggle(xmms_session, 1);
				xmms_remote_pl_win_toggle(xmms_session, pl_vis);
				xmms_remote_eq_win_toggle(xmms_session, eq_vis);
			}
		}
	} else {
		if ( (ev.xbutton.time-click_time) <= DBL_CLICK_INTERVAL )
		{
			click_time=0;
			ExecuteXmms();
		} else {
			click_time=ev.xbutton.time;
		}
	}
}

void ToggleVol(int x, int y, DARect rect, void *data)
{
	int volume;
	int factor;

	if (*(int*)data == 1)
		factor = volume_step;
	else
		factor = -volume_step;

	/* xmms seems to do the bounds checking for us */

	volume = xmms_remote_get_main_volume(xmms_session);
	xmms_remote_set_main_volume(xmms_session, volume+factor);
}

void ToggleTitle(int x, int y, DARect rect, void *data)
{
	if (t_scroll)
		t_scroll = 0;
	else t_scroll =1;
}

void ToggleTime(int x, int y, DARect rect, void *data)
{
	if (t_time)
		t_time = 0;
	else t_time =1;
}

void buttonPress(int button, int state, int x, int y)
{
	if (xmms_running)
	{
		if (button == 1)
		{
			char *tmp="1";
			DAProcessActionRects(x, y, buttonRects, 8, tmp);
			DAProcessActionRects(x, y, displayRects, 8, tmp);
		}
		if (button == 2)
		{
			DAProcessActionRects(x, y, toggleRect, 1, NULL);
		}
		if (button == 3)
		{
			char *tmp="1";
			DAProcessActionRects(x, y, toggleRect, 1, tmp);
		}
		if ((button == 4) || (button == 5))
		{
			if (button == 5)
			{
				/* Wheel scrolls down */
				int tmp=2;
				DAProcessActionRects(x, y, globRect, 1, &tmp);
			} else {
				/* Wheel scrolls up */
				int tmp=1;
				DAProcessActionRects(x, y, globRect, 1, &tmp);
			}
		}
	}
	else
		DAProcessActionRects(x, y, toggleRect, 1, NULL);
}

void buttonRelease(int button, int state, int x, int y)
{
	if (xmms_running)
	{
		if (button == 1)
		{
			copyNumArea(0,51, 54, 20, 5,39);
			DASetPixmap(pixmap);
			DAProcessActionRects(x, y, buttonRects, 8, NULL);
		}
	}
}

void DisplayRoutine()
{
	int time, length, position;
	int rate, freq, nch;
	char *title = NULL;
	Bool do_scroll=t_scroll;
	xmms_running=xmms_remote_is_running(xmms_session);

	/* Compute diplay */
	if (!xmms_running)
	{
		position = 100; /* Will display "00" as the current position */
		do_scroll = 0;   /* Will display channel/kbps info */
		arrow_pos = 5;
		rate = 0;
		time = 0;
		nch = 4;
		freq = 0;
		length = 0;
	} else {
		position = xmms_remote_get_playlist_pos(xmms_session);
		xmms_remote_get_info(xmms_session, &rate, &freq, &nch);
		time = xmms_remote_get_output_time(xmms_session);
		length = xmms_remote_get_playlist_time(xmms_session, position);
		title = xmms_remote_get_playlist_title(xmms_session, position);
		if (!xmms_remote_is_playing(xmms_session) || (pause_norotate && 
					xmms_remote_is_paused(xmms_session) ))
			arrow_pos = 5;
	}

	/*Draw everything */
	if (t_time) DrawTime(length-time);
	else DrawTime(time);
	DrawPos(position);
	DrawArrow();
	DrawChannels(nch);
	if (do_scroll && (title != NULL))
		DrawTitle(title);
	else
		DrawKbps(rate);

	DASetPixmap(pixmap);

	if (title != NULL)
		free(title);
}

void DrawPos (int pos)
{
	char posstr[16];
	char *p = posstr;
	int i=1;

	pos++;
	if (pos > 99) pos=0;
	sprintf(posstr, "%02d", pos);
	
	for (;i<3; i++)
	{
		copyNumArea((*p-'0')*6 + 1, 1, 6, 7, (i*6)+39, 7);
		p++;
	}
}

void DrawTime(int time)
{
	char timestr[16];
	char *p = timestr;
	int i=0;

	time = time / 1000;

	/* 3 cases:
	 *     up to 99 minutes and 59 seconds
	 *     up to 99 hours and 59 minutes
	 *     more
	 */
	if (time < 6000)
	{
		sprintf(timestr, "%02d%02d", time / 60, time % 60);
	} else {
		if (time < 360000)
		{
			sprintf(timestr, "%02d%02d", time / 3600,
					time % 3600 / 60);
		} else {
			sprintf(timestr, "%02d%02d", 0, 0);
		}
	}

	for (;i<4; i++)
	{
		copyNumArea((*p-'0')*7 + 2, 11, 7, 9, i<2 ?(i*7)+7:(i*7)+12, 7);
		p++;
	}
}

void DrawArrow(void)
{
	copyNumArea((arrow_pos*8)+30, 22, 8, 9, 47, 19);
	arrow_pos++;
	if (arrow_pos > 4) arrow_pos = 0;
}

void DrawChannels(int channels)
{
	if (channels == 4)
	{
		copyNumArea(55, 32, 10, 6, 10, 18);
	} else {
		copyNumArea((channels*10), 22, 10, 6, 10, 18);
	}
}

void DrawKbps(int bps)
{
	char kbpstr[16];
	char *p = kbpstr;
	int i=1;

	if (bps > 999000) bps=0;
	sprintf(kbpstr, "%03d", bps / 1000);

	for (;i<4; i++)
	{
		copyNumArea((*p-'0')*6 + 1, 1, 6, 7, (i*6)+1, 26);
		p++;
	}
	copyNumArea(55, 39, 18, 8, 25, 26);
}

void DrawTitle(char *name)
{
	int i, len, max, pos;
	char *title = NULL;
	char c=' ';

	if (name == NULL)
		return;
#ifdef DEBUG
	printf("name: %s\n", name);
#endif

	pos = title_pos;

	len = strlen(name);
	if (len <6)
	{
		max = len;
		title = g_strdup(name);
	} else {
		max = 6;
		title = g_strdup_printf("%s%s", name, SEPARATOR);
		len = strlen(title);
	}
	if (pos >= len)
	{
		title_pos = 0;
		pos = 0;
	}

	/* Start by printing all the characters we can */
	for (i=0;i<max; i++)
	{
		{
			int tmp = i+pos;

			if (tmp<len)
				c = toupper(title[tmp]);
			else
				c = toupper(title[(tmp-len)%6]);
		}
#ifdef DEBUG
		printf("%c", c);
#endif

		if (isalpha(c))
			copyNumArea((c-'A')*6 + 1, 73, 6, 8, (i*6)+7, 26);
		else if (isdigit(c))
			copyNumArea((c-'0')*6 + 1, 83, 6, 8, (i*6)+7, 26);
		else if (isspace(c))
			copyNumArea(61, 83, 6, 8, (i*6)+7, 26);
		else 
		{
			switch (c)
			{
				case '-':
					copyNumArea(67, 83, 6, 8, (i*6)+7, 26);
					break;
				case '.':
					copyNumArea(73, 83, 6, 8, (i*6)+7, 26);
					break;
				case 0x27:
					copyNumArea(79, 83, 6, 8, (i*6)+7, 26);
					break;
				case '(':
					copyNumArea(85, 83, 6, 8, (i*6)+7, 26);
					break;
				case ')':
					copyNumArea(91, 83, 6, 8, (i*6)+7, 26);
					break;
				case '*':
					copyNumArea(97, 83, 6, 8, (i*6)+7, 26);
					break;
				case '/':
					copyNumArea(103, 83, 6, 8, (i*6)+7, 26);
					break;
				default:
					/* Char not recognized, use a space */
					copyNumArea(61, 83, 6, 8, (i*6)+7, 26);
					break;
			}
		}
	}
#ifdef DEBUG
	printf("\n");
#endif

	/* If the title is too short, fill up with spaces */
	for (;i<6; i++)
	{
		copyNumArea(61, 83, 6, 8, (i*6)+7, 26);
	}
	/* You don't want a 2 letters title to scroll, do you ? */
	if (len > 6)
		title_pos = title_pos + 0.5;
	/* Clean up */
	free(title);
}

void ExecuteXmms(void)
{
	char *command;
	int status;

	command=malloc(strlen(xmms_cmd)+5);
	sprintf(command, "%s &", xmms_cmd);
	status = system(command);
	if (status)
	{
		fprintf(stderr, "XMMS can't be launched, exiting...");
		exit(1);
	}
	while (!xmms_remote_is_running(xmms_session))
		usleep(10000L);
	free(command);
}
/*----------------------------------------------------------------------------*/
/*                                   Main                                     */
/*----------------------------------------------------------------------------*/

int main(int argc, char **argv)
{
	unsigned height, width;
	DACallbacks callbacks={NULL, buttonPress, buttonRelease, NULL,
		NULL, NULL, NULL};

	/* Initialization */
	DAParseArguments(argc, argv, options,
		sizeof(options)/sizeof(DAProgramOption),
		"XMMS remote control by Bastien Nocera <hadess@hadess.net>",
		VERSION);

	DAInitialize(displayName, "wmusic", 64, 64, argc, argv);
	DASetCallbacks(&callbacks);
	DASetTimeout(100);

	DAMakePixmapFromData(wmusic_master_xpm, &pixmap,
			&mask, &height, &width);
	DAMakePixmapFromData(wmusic_digits_xpm, &pixnum,
			&masknum, &height, &width);
	gc = DefaultGC(DADisplay, DefaultScreen(DADisplay));
	DASetShapeWithOffset(mask, 0, 0);

	DASetPixmap(pixmap);
	DAShow();

	/* End of initialization */

	if (options[4].used) t_scroll=1;
	if (options[8].used) pause_norotate=1;
	if (options[9].used) t_time=1;

	/* Launch xmms if it's not running and -r was used */
	xmms_running=xmms_remote_is_running(xmms_session);
	if ((!xmms_running) && (options[2].used))
	{
		ExecuteXmms();
		xmms_running=1;
	}

	if(xmms_remote_is_main_win(xmms_session) && (options[6].used))
		ToggleWins(0, 0, DANoRect, NULL);

	/* Update the display */
	DisplayRoutine();

	while (1) {
		if (DANextEventOrTimeout(&ev, 100))
				DAProcessEvent(&ev);
	DisplayRoutine();
	}
	return 0;
}

