/* normalops.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
/* $Id: normalops.cc,v 1.25 2006/02/05 23:06:14 ralf Exp $ */

#include "normalops.h"
#include "listermode.h"
#include "normalmode.h"
#include "worker.h"

/******
  ChangeListerSetOp
*******/

const char *ChangeListerSetOp::name="ChangeListerSetOp";

bool ChangeListerSetOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *ChangeListerSetOp::getName()
{
  return name;
}

ChangeListerSetOp::ChangeListerSetOp() : FunctionProto()
{
  mode=CLS_CURRENT_LISTER;
  hasConfigure = true;
}

ChangeListerSetOp::~ChangeListerSetOp()
{
}

ChangeListerSetOp *ChangeListerSetOp::duplicate() const
{
  ChangeListerSetOp *ta=new ChangeListerSetOp();
  ta->mode=mode;
  return ta;
}

int ChangeListerSetOp::run( WPUContext *wpu, ActionMessage *msg )
{
  Lister *l1;
  switch(mode) {
    case CLS_LEFT_LISTER:
      l1 = msg->getWorker()->getLister(0);
      break;
    case CLS_RIGHT_LISTER:
      l1 = msg->getWorker()->getLister(1);
      break;
    case CLS_OTHER_LISTER:
      l1 = msg->getWorker()->getOtherLister( msg->getWorker()->getActiveLister() );
      break;
    default:
      l1 = msg->getWorker()->getActiveLister();
  }
  if(l1!=NULL) l1->configure();
  return 0;
}

const char *ChangeListerSetOp::getDescription()
{
  return catalog.getLocaleCom(19);
}

int ChangeListerSetOp::configure()
{
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  CycleButton *rcyb;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  
  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(19))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(19));
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
  win->create();
  _freesafe(tstr);

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 2 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 0 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );

  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 348 ), 1 ), 0, 0, cfix );
  rcyb = (CycleButton*)ac1_1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, cincw );
  rcyb->addOption(catalog.getLocale(326));
  rcyb->addOption(catalog.getLocale(327));
  rcyb->addOption(catalog.getLocale(324));
  rcyb->addOption(catalog.getLocale(325));
  rcyb->resize(rcyb->getMaxSize(),rcyb->getHeight());
  ac1_1->readLimits();

  switch(mode) {
    case CLS_RIGHT_LISTER:
      rcyb->setOption(1);
      break;
    case CLS_CURRENT_LISTER:
      rcyb->setOption(2);
      break;
    case CLS_OTHER_LISTER:
      rcyb->setOption(3);
      break;
    default:
      rcyb->setOption(0);
      break;
  }
  
  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cb = (Button*)ac1_2->add( new Button( aguix,
						0,
						0,
						catalog.getLocale( 8 ),
						1,
						0,
						0 ), 1, 0, cfix );
  
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==cb) endmode=1;
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    switch(rcyb->getSelectedOption()) {
      case 1:
        mode=CLS_RIGHT_LISTER;
        break;
      case 2:
        mode=CLS_CURRENT_LISTER;
        break;
      case 3:
        mode=CLS_OTHER_LISTER;
        break;
      default:
        mode=CLS_LEFT_LISTER;
        break;
    }
  }
  
  delete win;

  return endmode;
}

bool ChangeListerSetOp::save(Datei *fh)
{
  if ( fh == NULL ) return false;
  switch(mode) {
    case CLS_RIGHT_LISTER:
      fh->configPutPair( "mode", "right" );
      break;
    case CLS_CURRENT_LISTER:
      fh->configPutPair( "mode", "current" );
      break;
    case CLS_OTHER_LISTER:
      fh->configPutPair( "mode", "other" );
      break;
    default:
      fh->configPutPair( "mode", "left" );
      break;
  }
  return true;
}

void ChangeListerSetOp::setMode(int nv)
{
  switch(nv) {
    case CLS_RIGHT_LISTER:
      mode=nv;
      break;
    case CLS_CURRENT_LISTER:
      mode=nv;
      break;
    case CLS_OTHER_LISTER:
      mode=nv;
      break;
    default:
      mode=CLS_LEFT_LISTER;
  }
}

/*************
 Ende ChangeListerSetOp
**************/

/******
  SwitchListerOp
*******/

const char *SwitchListerOp::name="SwitchListerOp";

bool SwitchListerOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *SwitchListerOp::getName()
{
  return name;
}

SwitchListerOp::SwitchListerOp() : FunctionProto()
{
}

SwitchListerOp::~SwitchListerOp()
{
}

SwitchListerOp *SwitchListerOp::duplicate() const
{
  SwitchListerOp *ta=new SwitchListerOp();
  return ta;
}

int SwitchListerOp::run( WPUContext *wpu, ActionMessage *msg )
{
  Lister *l1;
  l1 = msg->getWorker()->getOtherLister( msg->getWorker()->getActiveLister() );
  if(l1!=NULL) l1->makeActive();
  return 0;
}

const char *SwitchListerOp::getDescription()
{
  return catalog.getLocaleCom(20);
}

/*************
 Ende SwitchListerOp
**************/

/******
  FilterSelectOp
*******/

const char *FilterSelectOp::name="FilterSelectOp";
int FilterSelectOp::instance_counter = 0;
List *FilterSelectOp::filter_history = NULL;
int FilterSelectOp::history_size = 30;
int FilterSelectOp::lastw = -1;
int FilterSelectOp::lasth = -1;

bool FilterSelectOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *FilterSelectOp::getName()
{
  return name;
}

FilterSelectOp::FilterSelectOp() : FunctionProto()
{
  filter=(char*)_allocsafe(2);
  strcpy(filter,"*");
  actual_filter=(char*)_allocsafe(1);
  actual_filter[0]=0;
  
  if ( instance_counter == 0 ) {
    initHistory();
  }
  instance_counter++;
  hasConfigure = true;
}

FilterSelectOp::~FilterSelectOp()
{
  _freesafe(filter);
  _freesafe(actual_filter);
  instance_counter--;
  if ( instance_counter == 0 ) {
    closeHistory();
  }
}

FilterSelectOp *FilterSelectOp::duplicate() const
{
  FilterSelectOp *ta=new FilterSelectOp();
  _freesafe(ta->filter);
  ta->filter=dupstring(filter);
  return ta;
}

int FilterSelectOp::run( WPUContext *wpu, ActionMessage *msg )
{
  Lister *l1;
  ListerMode *lm1;
  NormalMode *nm1;
  if(msg->mode!=msg->AM_MODE_DNDACTION) {
    l1 = msg->getWorker()->getActiveLister();
    if(l1!=NULL) {
      lm1=l1->getActiveMode();
      if(lm1!=NULL) {
        if(lm1->isType("NormalMode")==true) {
          /* Requester oeffnen */
          if(requestfilter()==true) {
            nm1=(NormalMode*)lm1;
            nm1->filterselect(actual_filter,0);
          }
        }
      }
    }
  }
  return 0;
}

const char *FilterSelectOp::getDescription()
{
  return catalog.getLocaleCom(21);
}

int FilterSelectOp::configure()
{
  StringGadget *sg;
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  char *tstr;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;

  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(21))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(21));
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
  win->create();
  _freesafe(tstr);
  
  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 2 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 0 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );

  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 349 ), 1 ), 0, 0, cfix );
  sg = (StringGadget*)ac1_1->add( new StringGadget( aguix, 0, 0, 100,
						    filter,
						    0 ), 1, 0, cincw );
  
  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, cfix );

  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    _freesafe(filter);
    filter=dupstring(sg->getText());
  }
  delete win;
  return (ende==1)?0:1;
}

bool FilterSelectOp::save(Datei *fh)
{
  if ( fh == NULL ) return false;
  fh->configPutPairString( "filter", filter );
  return true;
}

bool FilterSelectOp::requestfilter()
{
  if ( filter_history == NULL ) return false;  // this can only happen when the stupid
                                               // programmer calls closeHistory
                                               // but to be sure...

  int w, h;
  StringGadget *sg;
  FieldListView *lv;
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  int id, row;
  char *tstr;
  bool lvstill;
  int lastrow;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  const int cmin = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW;

  win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 268 ) );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 4 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 270 ), 1 ), 0, 0, cincwnr );
  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 272 ), 1 ), 0, 1, cincwnr );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 1, 2 ), 0, 2 );
  ac1_1->setMinSpace( 0 );
  ac1_1->setMaxSpace( 0 );
  ac1_1->setBorderWidth( 0 );

  lv = (FieldListView*)ac1_1->add( new FieldListView( aguix, 0, 0, 100, 7 * aguix->getCharHeight(), 0 ), 0, 0, cmin );

  lv->setHBarState( 2 );
  lv->setVBarState( 2 );
  lv->setDisplayFocus( true );
  lv->setAcceptFocus( true );

  id = filter_history->initEnum();
  tstr = (char*)filter_history->getFirstElement( id );
  while ( tstr != NULL ) {
    row = lv->insertRow( 0 );
    lv->setText( row, 0, tstr );
    lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    tstr = (char*)filter_history->getNextElement( id );
  }
  filter_history->closeEnum( id );
  lv->showRow( filter_history->size() - 1 );
  lv->redraw();

  sg=(StringGadget*)ac1_1->add( new StringGadget( aguix, 0, 0, 100,
						  strlen(actual_filter)>0?actual_filter:filter,
						  0 ), 0, 1, cincw );
  
  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 3 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, cfix );
  win->contMaximize( true );
  w = ( lastw > 0 ) ? lastw : win->getWidth();
  h = ( lasth > 0 ) ? lasth : win->getHeight();

  win->setDoTabCycling( true );
  win->resize(w,h);
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  sg->takeFocus();
  lastrow = -1;
  lvstill = false;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
      } else if(msg->type==AG_STRINGGADGET_OK) {
        if(msg->stringgadget.sg==sg) ende=1;
      } else if(msg->type==AG_STRINGGADGET_CANCEL) {
        if(msg->stringgadget.sg==sg) ende=-1;
      } else if ( msg->type == AG_KEYPRESSED ) {
        if ( win->isParent( msg->key.window, false ) == true ) {
          switch ( msg->key.key ) {
          case XK_Up:
            if ( sg->isActive() == true ) {
              lv->setActiveRow( lv->getElements() - 1 );
              lv->takeFocus();
              lv->showActive();
              lastrow = lv->getActiveRow();
              tstr = (char*)filter_history->getElementAt( filter_history->size() - 1 - lastrow );
              if ( tstr != NULL ) {
                sg->setText( tstr );
              }
            }
            lvstill = false;
            break;
          case XK_Down:
            if ( lv->getHasFocus() == true ) {
              if ( ( lv->getActiveRow() == ( lv->getElements() - 1 ) ) &&
                 ( lvstill == true ) ) {
                lv->setActiveRow( -1 );
                sg->takeFocus();
              }
            }
            break;
          case XK_Escape:
            ende = -1;
            break;
          case XK_Return:
            if ( cancelb->getHasFocus() == false )
              ende = 1;
            break;
          }
        }
      } else if ( ( msg->type == AG_FIELDLV_ONESELECT ) || ( msg->type == AG_FIELDLV_MULTISELECT ) ) {
        row = lv->getActiveRow();
        if ( row == lastrow ) lvstill = true;
        else lvstill = false;
        lastrow = row;
        if ( lv->isValidRow( row ) == true ) {
          tstr = (char*)filter_history->getElementAt( filter_history->size() - 1 - row );
          if ( tstr != NULL ) {
            sg->setText( tstr );
          }
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    _freesafe(actual_filter);
    actual_filter=dupstring(sg->getText());
    addHistoryItem( actual_filter );
  }
  lastw = win->getWidth();
  lasth = win->getHeight();
  delete win;
  return (ende==1?true:false);
}

void FilterSelectOp::initHistory()
{
  if ( filter_history != NULL ) return;
  char *tstr;
  Datei *fp = new Datei();
  std::string str1;

  filter_history = new List();

  tstr = getenv( "HOME" );
  if ( tstr != NULL ) {
    str1 = tstr;
    str1 += "/.worker/filterselect-history";
    fp->open( str1.c_str(), "r" );
    
    while ( fp->isEOF() == false ) {
      tstr = fp->getLine();
      if ( tstr == NULL ) break;
      if ( strlen( tstr ) > 0 ) {
        filter_history->addElement( tstr );
      } else {
        _freesafe( tstr );
      }
    }
    
    fp->close();
  }
  delete fp;
}

void FilterSelectOp::closeHistory()
{
  if ( filter_history == NULL ) return;
  char *tstr;
  int id;
  Datei *fp = new Datei();
  std::string str1;

  str1 = wconfig->getHome();
  str1 += "/filterselect-history";
  fp->open( str1.c_str(), "w" );
  
  id = filter_history->initEnum();
  tstr = (char*)filter_history->getFirstElement( id );
  while ( tstr != NULL ) {
    fp->putLine( tstr );
    _freesafe( tstr );
    filter_history->removeFirstElement();
    tstr = (char*)filter_history->getFirstElement( id );
  }
  filter_history->closeEnum( id );
  delete filter_history;
  filter_history = NULL;
  
  fp->close();
  delete fp;
}

void FilterSelectOp::addHistoryItem( const char *str )
{
  char *tstr;
  int id;
  
  id = filter_history->initEnum();
  tstr = (char*)filter_history->getFirstElement( id );
  while ( tstr != NULL ) {
    if ( strcmp( str, tstr ) == 0 ) break;
    tstr = (char*)filter_history->getNextElement( id );
  }
  filter_history->closeEnum( id );

  if ( tstr == NULL ) {
    // not already in the list
    if ( filter_history->size() >= history_size ) {
      tstr = (char*)filter_history->getElementAt( filter_history->size() - 1 );
      if ( tstr != NULL ) {
        _freesafe( tstr );
      }
      filter_history->removeElementAt( filter_history->size() - 1 );
    }
    filter_history->addElementAt( 0, dupstring( str ) );
  } else {
    // in the list so put it at the beginning
    filter_history->removeElement( tstr );
    filter_history->addElementAt( 0, tstr );
  }
}

void FilterSelectOp::setFilter( const char *str )
{
  if ( str == NULL ) return;
  if ( filter != NULL ) _freesafe( filter );
  filter = dupstring( str );
}

/*************
 Ende FilterSelectOp
**************/

/******
  FilterUnSelectOp
*******/

const char *FilterUnSelectOp::name="FilterUnSelectOp";

bool FilterUnSelectOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *FilterUnSelectOp::getName()
{
  return name;
}

FilterUnSelectOp::FilterUnSelectOp() : FilterSelectOp()
{
}

FilterUnSelectOp::~FilterUnSelectOp()
{
}

FilterUnSelectOp *FilterUnSelectOp::duplicate() const
{
  FilterUnSelectOp *ta=new FilterUnSelectOp();
  _freesafe(ta->filter);
  ta->filter=dupstring(filter);
  return ta;
}

int FilterUnSelectOp::run( WPUContext *wpu, ActionMessage *msg )
{
  Lister *l1;
  ListerMode *lm1;
  NormalMode *nm1;
  if(msg->mode!=msg->AM_MODE_DNDACTION) {
    l1 = msg->getWorker()->getActiveLister();
    if(l1!=NULL) {
      lm1=l1->getActiveMode();
      if(lm1!=NULL) {
        if(lm1->isType("NormalMode")==true) {
          /* Requester oeffnen */
          if(requestfilter()==true) {
            nm1=(NormalMode*)lm1;
            nm1->filterselect(actual_filter,1);
          }
        }
      }
    }
  }
  return 0;
}

const char *FilterUnSelectOp::getDescription()
{
  return catalog.getLocaleCom(22);
}

int FilterUnSelectOp::configure()
{
  StringGadget *sg;
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  char *tstr;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;

  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(22))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(22));
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
  win->create();
  _freesafe(tstr);
  
  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 2 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 0 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );

  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 349 ), 1 ), 0, 0, cfix );

  sg = (StringGadget*)ac1_1->add( new StringGadget( aguix, 0, 0, 100,
						    filter,
						    0 ), 1, 0, cincw );
  
  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, cfix );

  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    _freesafe(filter);
    filter=dupstring(sg->getText());
  }
  delete win;
  return (ende==1)?0:1;
}

bool FilterUnSelectOp::requestfilter()
{
  if ( filter_history == NULL ) return false;  // this can only happen when the stupid
                                               // programmer calls closeHistory
                                               // but to be sure...

  StringGadget *sg;
  FieldListView *lv;
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  int id, row;
  char *tstr;
  bool lvstill;
  int lastrow;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  const int cmin = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW;

  win = new AWindow( aguix, 10, 10, 10, 10, 0, catalog.getLocale( 269 ) );
  win->create();

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 4 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 271 ), 1 ), 0, 0, cincwnr );
  ac1->add( new Text( aguix, 0, 0, catalog.getLocale( 272 ), 1 ), 0, 1, cincwnr );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 1, 2 ), 0, 2 );
  ac1_1->setMinSpace( 0 );
  ac1_1->setMaxSpace( 0 );
  ac1_1->setBorderWidth( 0 );

  lv = (FieldListView*)ac1_1->add( new FieldListView( aguix, 0, 0, 100,
						      7 * aguix->getCharHeight(), 0 ), 0, 0, cmin );

  lv->setHBarState( 2 );
  lv->setVBarState( 2 );
  lv->setDisplayFocus( true );
  lv->setAcceptFocus( true );

  id = filter_history->initEnum();
  tstr = (char*)filter_history->getFirstElement( id );
  while ( tstr != NULL ) {
    row = lv->insertRow( 0 );
    lv->setText( row, 0, tstr );
    lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
    tstr = (char*)filter_history->getNextElement( id );
  }
  filter_history->closeEnum( id );
  lv->showRow( filter_history->size() - 1 );
  lv->redraw();

  sg = (StringGadget*)ac1_1->add( new StringGadget( aguix, 0, 0, 100,
						    strlen(actual_filter)>0?actual_filter:filter,
						    0 ), 0, 1, cincw );
  
  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 3 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cancelb = (Button*)ac1_2->add( new Button( aguix,
						     0,
						     0,
						     catalog.getLocale( 8 ),
						     1,
						     0,
						     0 ), 1, 0, cfix );

  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  sg->takeFocus();
  lastrow = -1;
  lvstill = false;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
      } else if(msg->type==AG_STRINGGADGET_OK) {
        if(msg->stringgadget.sg==sg) ende=1;
      } else if(msg->type==AG_STRINGGADGET_CANCEL) {
        if(msg->stringgadget.sg==sg) ende=-1;
      } else if ( msg->type == AG_KEYPRESSED ) {
        if ( win->isParent( msg->key.window, false ) == true ) {
          switch ( msg->key.key ) {
          case XK_Up:
            if ( sg->isActive() == true ) {
              lv->setActiveRow( lv->getElements() - 1 );
              lv->takeFocus();
              lv->showActive();
              lastrow = lv->getActiveRow();
              tstr = (char*)filter_history->getElementAt( filter_history->size() - 1 - lastrow );
              if ( tstr != NULL ) {
                sg->setText( tstr );
              }
            }
            lvstill = false;
            break;
          case XK_Down:
            if ( lv->getHasFocus() == true ) {
              if ( ( lv->getActiveRow() == ( lv->getElements() - 1 ) ) &&
                 ( lvstill == true ) ) {
                lv->setActiveRow( -1 );
                sg->takeFocus();
              }
            }
            break;
          case XK_Escape:
            ende = -1;
            break;
          case XK_Return:
            if ( cancelb->getHasFocus() == false )
              ende = 1;
            break;
          }
        }
      } else if ( msg->type == AG_FIELDLV_ONESELECT ) {
        row = lv->getActiveRow();
        if ( row == lastrow ) lvstill = true;
        else lvstill = false;
        lastrow = row;
        if ( lv->isValidRow( row ) == true ) {
          tstr = (char*)filter_history->getElementAt( filter_history->size() - 1 - row );
          if ( tstr != NULL ) {
            sg->setText( tstr );
          }
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    _freesafe(actual_filter);
    actual_filter=dupstring(sg->getText());
    addHistoryItem( actual_filter );
  }
  delete win;
  return (ende==1?true:false);
}

/*************
 Ende FilterUnSelectOp
**************/

/******
  Path2OSideOp
*******/

const char *Path2OSideOp::name="Path2OSideOp";

bool Path2OSideOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *Path2OSideOp::getName()
{
  return name;
}

Path2OSideOp::Path2OSideOp() : FunctionProto()
{
}

Path2OSideOp::~Path2OSideOp()
{
}

Path2OSideOp *Path2OSideOp::duplicate() const
{
  Path2OSideOp *ta=new Path2OSideOp();
  return ta;
}

int Path2OSideOp::run( WPUContext *wpu, ActionMessage *msg )
{
  Lister *l1;
  ListerMode *lm1;
  NormalMode *nm1;
  const char *dir=NULL;
  if(msg->mode!=msg->AM_MODE_DNDACTION) {
    l1 = msg->getWorker()->getActiveLister();
    if(l1!=NULL) {
      lm1=l1->getActiveMode();
      if(lm1!=NULL) {
        if(lm1->isType("NormalMode")==true) {
          nm1=(NormalMode*)lm1;
          dir = nm1->getCurrentDir();
        }
      }
    }
    if(dir!=NULL) {
      l1 = msg->getWorker()->getOtherLister( msg->getWorker()->getActiveLister());
      if(l1!=NULL) {
        l1->switch2Mode(0);
        lm1=l1->getActiveMode();
        if(lm1!=NULL) {
          if(lm1->isType("NormalMode")==true) {
            nm1=(NormalMode*)lm1;
            nm1->enterDir(dir);
          }
        }
      }
    }
  }
  return 0;
}

const char *Path2OSideOp::getDescription()
{
  return catalog.getLocaleCom(23);
}

/*************
 Ende Path2OSideOp
**************/

/******
  QuitOp
*******/

const char *QuitOp::name="QuitOp";

bool QuitOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *QuitOp::getName()
{
  return name;
}

QuitOp::QuitOp() : FunctionProto()
{
  mode=Q_NORMAL_QUIT;
  hasConfigure = true;
}

QuitOp::~QuitOp()
{
}

QuitOp *QuitOp::duplicate() const
{
  QuitOp *ta=new QuitOp();
  ta->mode=mode;
  return ta;
}

int QuitOp::run( WPUContext *wpu, ActionMessage *msg )
{
  msg->getWorker()->quit( mode == Q_QUICK_QUIT ? 1 : 0 );
  return 0;
}

const char *QuitOp::getDescription()
{
  return catalog.getLocaleCom(24);
}

int QuitOp::configure()
{
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  CycleButton *rcyb;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  
  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(24))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(24));
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
  win->create();
  _freesafe(tstr);

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 2 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 0 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );

  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 350 ), 1 ), 0, 0, cfix );
  rcyb = (CycleButton*)ac1_1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, cincw );
  rcyb->addOption(catalog.getLocale(351));
  rcyb->addOption(catalog.getLocale(352));
  rcyb->resize(rcyb->getMaxSize(),rcyb->getHeight());
  ac1_1->readLimits();

  switch(mode) {
    case Q_QUICK_QUIT:
      rcyb->setOption(1);
      break;
    default:
      rcyb->setOption(0);
      break;
  }
  
  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 1 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cb = (Button*)ac1_2->add( new Button( aguix,
						0,
						0,
						catalog.getLocale( 8 ),
						1,
						0,
						0 ), 1, 0, cfix );
  
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==cb) endmode=1;
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    switch(rcyb->getSelectedOption()) {
      case 1:
        mode=Q_QUICK_QUIT;
        break;
      default:
        mode=Q_NORMAL_QUIT;
        break;
    }
  }
  
  delete win;

  return endmode;
}

bool QuitOp::save(Datei *fh)
{
  if ( fh == NULL ) return false;
  switch(mode) {
    case Q_QUICK_QUIT:
      fh->configPutPair( "mode", "quick" );
      break;
    default:
      fh->configPutPair( "mode", "normal" );
      break;
  }
  return true;
}

void QuitOp::setMode(int nmode)
{
  if(nmode==Q_QUICK_QUIT) mode=nmode;
  else mode=Q_NORMAL_QUIT;
}

/*************
 Ende QuitOp
**************/

