/* normalmode.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: normalmode.cc,v 1.149 2002/04/06 19:23:24 ralf Exp $ */

#include "normalmode.h"
#include "locale.h"
#include "wconfig.h"
#include "worker.h"

const char *NormalMode::type="NormalMode";

int NormalMode::maxfilestrlen=1;
int NormalMode::maxdirstrlen=1;
int NormalMode::maxbytesstrlen=1;
int NormalMode::maxwbytesstrlen=1;
int NormalMode::maxnochanges=0;

#ifdef WANT_THREADS
void *NM_slavestart( void *arg )
{
  NormalMode *nm = (NormalMode*)arg;
  nm->slavehandler();
  return NULL;
}
#endif

NormalMode::NormalMode(Lister *tparent):ListerMode(tparent)
{
  x=0;
  y=0;
  w=0;
  h=0;
  ce=NULL;
  cache=new List();
  filters=new List();
  showHidden=true;
  req=new Requester(aguix);
  searchmodeon=false;
  lastsearch=NULL;
  sortmode=SORT_NAME;
  filterlfdnr=0;
  lv=NULL;
  lastrow=-1;
  showfreespace=true;
  namestr=NULL;
  lastfsupdate=0;
  updatetime=10;

  lasteagain = false;
  
#ifdef WANT_THREADS
  // to preventing slave access
  reclist = new ft_rec_list();
  recreslist = new ft_recres_list();

  slave.filetypes = new List();
  slave.filetype_ex.lock();
  slave.running = 0;
  slave.status = THREAD_NOP;
  slave.order = THREAD_REINIT;
  if ( pthread_create( &slave.th, NULL, NM_slavestart, this ) == 0 ) {
    thread_usage = true;
  } else {
    fprintf( stderr, "Worker: Can't create thread, disabling thread-support!\n" );
    thread_usage = false;
  }

  if ( thread_usage == true ) {
    slave.filetype_ex.wait();
  }
  ft_list_update();
  slave.filetype_ex.signal();
  slave.filetype_ex.unlock();
#else
  thread_usage = false;
#endif
  
  finish_done = false;
}

NormalMode::~NormalMode()
{
#ifdef WANT_THREADS
  slave.order = THREAD_EXIT;
  // wake up slave
  reclist->signal();
  if ( thread_usage == true ) {
    pthread_join( slave.th, NULL );
  }
  ft_list_clear();
  delete slave.filetypes;
  ft_rec_list_clear();
  ft_recres_list_clear();
  delete reclist;
  delete recreslist;
#endif

  if(searchmodeon==true) finishsearchmode();
  deleteCache();
  deleteFilters();
  delete cache;
  delete filters;
  delete req;
  if(namestr!=NULL) _freesafe(namestr);
}

void NormalMode::messageHandler(AGMessage *msg)
{
  int tx,ty,tw,th;
  bool ma=false;
  switch(msg->type) {
    case AG_STRINGGADGET_OK:
      if(searchmodeon==false) {
        if(msg->stringgadget.sg==sg) {
          enterDir(sg->getText());
        }
        break;
      }
    case AG_STRINGGADGET_CANCEL:
      if(searchmodeon==false) {
        if(msg->stringgadget.sg==sg) {
          if(getActualDir()!=NULL) sg->setText(getActualDir());
        }
        break;
      }
    case AG_STRINGGADGET_DEACTIVATE:
      if(searchmodeon==true) {
        finishsearchmode();
      }
      break;
    case AG_STRINGGADGET_CONTENTCHANGE:
      if(searchmodeon==true) {
        if(showfirstentry(sg->getText())==true) {
          if(lastsearch!=NULL) _freesafe(lastsearch);
          else debugmsg("oops1\n");
          lastsearch=dupstring(sg->getText());
        } else {
          sg->setText(lastsearch);
        }
      }
      break;
    case AG_SIZECHANGED:
      parentlister->getGeometry(&tx,&ty,&tw,&th);
      reconf(tx,ty,tw,th);
      break;
    case AG_STRINGGADGET_ACTIVATE:
      if(msg->stringgadget.sg==sg) ma=true;
      break;
    case AG_BUTTONCLICKED:
      if(msg->button.button==hb[0]) {
        showcache(-1);
        ma=true;
      } else if(msg->button.button==hb[1]) { 
        showcache(1);
        ma=true;
      }
      break;
    case AG_FIELDLV_PRESSED:
      if(msg->fieldlv.lv==lv) ma=true;
      break;
    case AG_FIELDLV_ONESELECT:
      if(msg->fieldlv.lv==lv) {
        if(lastrow==-1) {
          lastrow=msg->fieldlv.row;
          gettimeofday(&lastclick,NULL);
        } else {
          if(msg->fieldlv.row==lastrow) {
            struct timeval acttime;
            gettimeofday(&acttime,NULL);
            if(isDoubleClick(&acttime,&lastclick)==true) {
              //TODO: Wir haben einen Doppelclick
              // Aus lvc->data==fe->nr bekommen wir das fe (nr beschreibt Position)
              if(ce!=NULL) {
                ArrayList *f1=ce->verz->getFiles();
                FileEntry *tfe=(FileEntry*)f1->getElementAt( lv->getData( lastrow ) );
                startAction(tfe);
              }
              // when had a doubleclick the next click on the same should not be a
              // doubleclick again
              lastrow=-1;
            } else {
              lastrow=msg->fieldlv.row;
              gettimeofday(&lastclick,NULL);
            }
          } else {
            lastrow=msg->fieldlv.row;
            gettimeofday(&lastclick,NULL);
          }
        }
      }
      break;
  }
  if(ma==true) parentlister->makeActive();
  showFreeSpace(true);
}

void NormalMode::on()
{
  parentlister->getGeometry(&x,&y,&w,&h);
  int side=parentlister->getSide();
  AGUIXFont *afont=aguix->getFont(wconfig->getFont(2+side));
  int sgh=afont->getCharHeight()+8;
  int hbw=afont->getCharWidth()+10;
  int m;

  lv=(NMFieldListViewDND*)parentawindow->add(new NMFieldListViewDND(aguix,x,y,w,h-sgh,1));
  lv->setMBG(wconfig->getListerBG());
  m=(wconfig->getHBarTop(side)==true)?1:2;
  lv->setHBarState(m);
  m=(wconfig->getVBarLeft(side)==true)?1:2;
  lv->setVBarState(m);
  m=wconfig->getHBarHeight(side);
  lv->setHBarHeight(m);
  m=wconfig->getVBarWidth(side);
  lv->setVBarWidth(m);
  lv->setFont(wconfig->getFont(2+side));
  hb[0]=(Button*)parentawindow->add(new Button(aguix,x,y+h-sgh,hbw,"<",1,0,2));
  hb[0]->resize(hbw,sgh);
  hb[0]->setFont(wconfig->getFont(2+side));
  hb[1]=(Button*)parentawindow->add(new Button(aguix,x+w-hbw,y+h-sgh,hbw,">",1,0,3));
  hb[1]->resize(hbw,sgh);
  hb[1]->setFont(wconfig->getFont(2+side));
  sg=(StringGadget*)parentawindow->add(new StringGadget(aguix,x+hbw,y+h-sgh,w-2*hbw,"",4));
  sg->resize(w-2*hbw,sgh);
  sg->setFont(wconfig->getFont(2+side));
  parentlister->setActiveMode(this);
  setName();
  lv->setSelectHandler(this);

  setupLVFields();
  
  showcache(0);
}

void NormalMode::off()
{
  finishsearchmode();
  parentawindow->remove(lv);
  parentawindow->remove(hb[0]);
  parentawindow->remove(hb[1]);
  parentawindow->remove(sg);
  parentlister->setActiveMode(NULL);
  parentlister->setName("");
  if ( parentlister->getFocus() == true )
    parentlister->getWorker()->setTitle(NULL);
  lv=NULL;
}

void NormalMode::reconf(int tx,int ty,int tw,int th)
{
  if(lv!=NULL) {
    if((tw!=w)||(th!=h)||(tx!=x)||(ty!=y)) {
      w=tw;
      h=th;
      x=tx;
      y=ty;
      int side=parentlister->getSide();
      AGUIXFont *afont=aguix->getFont(wconfig->getFont(2+side));
      int sgh=afont->getCharHeight()+8;
      int hbw=afont->getCharWidth()+10;
      lv->resize(tw,th-sgh);
      lv->move(tx,ty);
      hb[0]->move(x,y+h-sgh);
      hb[0]->resize(hbw,sgh);
      hb[1]->move(x+w-hbw,y+h-sgh);
      hb[1]->resize(hbw,sgh);
      sg->move(x+hbw,y+h-sgh);
      sg->resize(w-2*hbw,sgh);
    }
  }
}

void NormalMode::activate()
{
  int row;
  ArrayList *l;
  if(ce!=NULL) {
    if(ce->activefe!=NULL) {
      l=ce->verz->getFiles();
      row = 0;
      while( row < lv->getElements() ) {
        if( lv->getData( row ) == ce->activefe->nr) break;
        row++;
      }
      if( row < lv->getElements() ) lv->setActiveRow( row );
    }
  }
  showCacheState();
  parentlister->getWorker()->setTitle(sg->getText());
}

void NormalMode::deactivate()
{
  int row=lv->getActiveRow();
  if ( row >= 0 ) {
    lv->setActiveRow( -1);
  }
  finishsearchmode();
  parentlister->getWorker()->setTitle(NULL);
}

void NormalMode::buildLister(Verzeichnis *verz)
{
  int side=parentlister->getSide();
  int id;
  int *dis;
  int lvs=verz->getUseSize();
  ArrayList *al;

  if(lv==NULL) return;
  lv->setSize(lvs);
  lv->setSizeDNDText(lvs);
  al=verz->getFiles();
  id=al->initEnum();
  FileEntry *fe=(FileEntry*)al->getFirstElement(id);
  
  lv->setActiveRowQ( -1 );
  int tpos;
  
  tpos = 0;
  dis=wconfig->getDisplay(side);
  while(fe!=NULL) {
    if(fe->use==true) {
      verz->setLVC4FE_DND( lv, tpos, fe, dis );

      lv->setData( tpos, fe->nr);
      if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
        lv->setFG( tpos, FieldListView::CC_NORMAL, wconfig->getUnselDir(0));
        lv->setBG( tpos, FieldListView::CC_NORMAL, wconfig->getUnselDir(1));
        lv->setFG( tpos, FieldListView::CC_SELECT, wconfig->getSelDir(0));
        lv->setBG( tpos, FieldListView::CC_SELECT, wconfig->getSelDir(1));
        lv->setFG( tpos, FieldListView::CC_ACTIVE, wconfig->getUnselDirAct(0));
        lv->setBG( tpos, FieldListView::CC_ACTIVE, wconfig->getUnselDirAct(1));
        lv->setFG( tpos, FieldListView::CC_SELACT, wconfig->getSelDirAct(0));
        lv->setBG( tpos, FieldListView::CC_SELACT, wconfig->getSelDirAct(1));
      } else {
        lv->setFG( tpos, FieldListView::CC_NORMAL, wconfig->getUnselFile(0));
        lv->setBG( tpos, FieldListView::CC_NORMAL, wconfig->getUnselFile(1));
        lv->setFG( tpos, FieldListView::CC_SELECT, wconfig->getSelFile(0));
        lv->setBG( tpos, FieldListView::CC_SELECT, wconfig->getSelFile(1));
        lv->setFG( tpos, FieldListView::CC_ACTIVE, wconfig->getUnselFileAct(0));
        lv->setBG( tpos, FieldListView::CC_ACTIVE, wconfig->getUnselFileAct(1));
        lv->setFG( tpos, FieldListView::CC_SELACT, wconfig->getSelFileAct(0));
        lv->setBG( tpos, FieldListView::CC_SELACT, wconfig->getSelFileAct(1));
      }
      if(fe->isCorrupt==true) {
        lv->setFG( tpos, FieldListView::CC_NORMAL, 4 );
        lv->setFG( tpos, FieldListView::CC_ACTIVE, 4 );
      }
      if(fe->select==true) lv->setSelectQ( tpos, true ); else lv->setSelectQ( tpos, false );
      if(fe==ce->activefe) {
        // only make the lvc active if we are active
        //TODO: setActiveRow stt redraw an!!!!
        if(parentlister->isActive()==true) lv->setActiveRowQ( tpos );
      }
      tpos++;
    }
    fe=(FileEntry*)al->getNextElement(id);
  }
  al->closeEnum(id);
  lv->redraw();
}

void NormalMode::buildLister()
{
  if(ce==NULL) return;
  int side=parentlister->getSide();
  int id;
  int *dis;

  if(lv==NULL) return;
  lv->setSize(ce->files[0]+ce->dirs[0]+1);
  lv->setSizeDNDText(ce->files[0]+ce->dirs[0]+1);
  ArrayList *al=ce->verz->getFiles();
  id=al->initEnum();
  FileEntry *fe=(FileEntry*)al->getFirstElement(id);
 
  lv->setActiveRowQ( -1 );
 
  int tpos;
 
  tpos = 0;
  dis=wconfig->getDisplay(side);
  while(fe!=NULL) {
    if(fe->use==true) {
      ce->verz->setLVC4FE_DND( lv, tpos, fe, dis );
      lv->setData( tpos, fe->nr);
      if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
        lv->setFG( tpos, FieldListView::CC_NORMAL,wconfig->getUnselDir(0));
        lv->setBG( tpos, FieldListView::CC_NORMAL,wconfig->getUnselDir(1));
        lv->setFG( tpos, FieldListView::CC_SELECT,wconfig->getSelDir(0));
        lv->setBG( tpos, FieldListView::CC_SELECT,wconfig->getSelDir(1));
        lv->setFG( tpos, FieldListView::CC_ACTIVE,wconfig->getUnselDirAct(0));
        lv->setBG( tpos, FieldListView::CC_ACTIVE,wconfig->getUnselDirAct(1));
        lv->setFG( tpos, FieldListView::CC_SELACT,wconfig->getSelDirAct(0));
        lv->setBG( tpos, FieldListView::CC_SELACT,wconfig->getSelDirAct(1));
      } else {
        lv->setFG( tpos, FieldListView::CC_NORMAL,wconfig->getUnselFile(0));
        lv->setBG( tpos, FieldListView::CC_NORMAL,wconfig->getUnselFile(1));
        lv->setFG( tpos, FieldListView::CC_SELECT,wconfig->getSelFile(0));
        lv->setBG( tpos, FieldListView::CC_SELECT,wconfig->getSelFile(1));
        lv->setFG( tpos, FieldListView::CC_ACTIVE,wconfig->getUnselFileAct(0));
        lv->setBG( tpos, FieldListView::CC_ACTIVE,wconfig->getUnselFileAct(1));
        lv->setFG( tpos, FieldListView::CC_SELACT,wconfig->getSelFileAct(0));
        lv->setBG( tpos, FieldListView::CC_SELACT,wconfig->getSelFileAct(1));
      }
      if(fe->isCorrupt==true) {
        lv->setFG( tpos, FieldListView::CC_NORMAL,4);
        lv->setFG( tpos, FieldListView::CC_ACTIVE,4);
      }
      if(fe->select==true) lv->setSelectQ( tpos, true ); else lv->setSelectQ( tpos, false );
      if(fe==ce->activefe) {
        // only make the lvc active if we are active
        if(parentlister->isActive()==true) lv->setActiveRowQ( tpos );
      }
      tpos++;
    }
    fe=(FileEntry*)al->getNextElement(id);
  }
  al->closeEnum(id);
  lv->redraw();
}

void NormalMode::deleteCache()
{
  NMCacheEntry *tce=(NMCacheEntry*)cache->getFirstElement();
  while(tce!=NULL) {
    delete tce;
    tce=(NMCacheEntry*)cache->getNextElement();
  }
}

int NormalMode::enterDir(const char *dirstr)
{
//  int side=parentlister->getSide();
  int maxs=wconfig->getCacheSize();
  int caches=cache->size();
  char *tstr=HandlePath(dirstr);
  Verzeichnis *tverz=NULL;
  NMCacheEntry *tce=NULL;
  int exentry=-1;
  ArrayList *oldfiles,*newfiles;
  int oldid,newid;
  FileEntry *ofe,*nfe,*oldact;
  int comperg;
  bool reset;
  int returnvalue=0;  // 0 = OK
                      // 1 = Not found
                      // 2 = not found, but lv cleared (was actually)
  char *textstr;

  if(lv==NULL) return 1;
  parentlister->getWorker()->getMainWin()->setCursor(AGUIX::WAIT_CURSOR);
  aguix->Flush();
#ifdef DEBUG
  printf("enterDir: %s\n",dirstr);
#endif
  
  finishsearchmode();
  
#ifdef WANT_THREADS
  // clear the reclist so no wrong pointers are in the list
  ft_rec_list_clear();
#endif
  
  if(caches>0) {
    int i=0;
    tce=(NMCacheEntry*)cache->getFirstElement();
    while(tce!=NULL) {
      if(strcmp(tce->verz->getDir(),tstr)==0) {
        exentry=i;
#ifdef DEBUG
        printf("enterDir: cache hit\n");
#endif
        break;
      }
      tce=(NMCacheEntry*)cache->getNextElement();
      i++;
    }
  }
  tverz=new Verzeichnis();
  if(tverz->readDir(tstr)==0) {
  } else {
    delete tverz;
    tverz=NULL;
  }
  if(tverz!=NULL) {

    if(tverz->getInvalidEntries()->size()>0) {
      // show requester
      ArrayList *invalid_entries=tverz->getInvalidEntries();
      int id=invalid_entries->initEnum();
      FileEntry *tfe;
      
      tfe=(FileEntry*)invalid_entries->getFirstElement(id);
      while(tfe!=NULL) {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(203))+strlen(tfe->name)+1);
        sprintf(textstr,catalog.getLocale(203),tfe->name);
        req->request(catalog.getLocale(125),textstr,catalog.getLocale(11));
        _freesafe(textstr);
        tfe=(FileEntry*)invalid_entries->getNextElement(id);
      }
      invalid_entries->closeEnum(id);
    }

#ifdef DEBUG
     printf("enterDir: readdir ok\n");
#endif
    if(tce==NULL) {
      // no cache entry found
      if(cache->size()<maxs) {
        // da ist noch Platz im Cache
#ifdef DEBUG
        printf("enterDir: new cache entry\n");
#endif
      } else {
        tce=(NMCacheEntry*)cache->getElementAt(0);
        delete tce;
        cache->removeElementAt(0);
#ifdef DEBUG
        printf("enterDir: cache is full\n");
#endif
      }
      tce=new NMCacheEntry();
      tce->verz=tverz;
      cache->addElement(tce);
      exentry=cache->size()-1;
      tce->restartcheck();
      newfiles=tce->verz->getFiles();
      if(newfiles!=NULL) tce->activefe=(FileEntry*)newfiles->getFirstElement();
      //^^^ should never happen, but to be sure
      tce->checkfordcd();
    } else {
#ifdef DEBUG
      printf("enterDir: using existing cache, updating values\n");
#endif
      // put tce at the end of the cache to realize Not-Recently-Used
      cache->removeElement(tce);
      cache->addElement(tce);
      exentry=cache->size()-1;
      // insert old values into to new tverz
      // first sort both for names so we can easily (and fast) get old values
      tverz->sort(SORT_NAME|SORT_DIRMIXED);
      tce->verz->sort(SORT_NAME|SORT_DIRMIXED);
      oldfiles=tce->verz->getFiles();
      newfiles=tverz->getFiles();
      oldid=oldfiles->initEnum();
      newid=newfiles->initEnum();
      ofe=(FileEntry*)oldfiles->getFirstElement(oldid);
      nfe=(FileEntry*)newfiles->getFirstElement(newid);
      oldact=tce->activefe;
      tce->activefe=NULL;
      while(nfe!=NULL) {
        if(oldact!=NULL)
          if(strcmp(nfe->fullname,oldact->fullname)==0)
            tce->activefe=nfe;
        while(ofe!=NULL) {
          comperg=strcmp(ofe->fullname,nfe->fullname);
          if(comperg>0) break;
          else if(comperg==0) {
            // found an entry fit to the actual
            if(ofe->select==true) nfe->select=true;
            else nfe->select=false;
            if(ofe->isDir()==true)
              if(ofe->dirsize>=0) nfe->dirsize=ofe->dirsize;
            if(ofe->filetype!=NULL) nfe->filetype=ofe->filetype;
          }
          ofe=(FileEntry*)oldfiles->getNextElement(oldid);
          if(strcmp(nfe->name,"..")==0) break; // the ".." is always first in new and old so stop here
                                               // because following elements can be "smaller" then ".."
                                               // f.i. "#343" < ".."
        }
        if(ofe==NULL) break; // there are no further old files => no need to continue
        nfe=(FileEntry*)newfiles->getNextElement(newid);
      }
      oldfiles->closeEnum(oldid);
      newfiles->closeEnum(newid);
      delete tce->verz;
      tce->verz=tverz;
#ifdef DEBUG
      printf("enterDir: update values finished\n");
#endif
    }
    tverz->sort(sortmode);
    tce->actsortmode=sortmode;
    tce->reset_firstnullft();
    tce->firstnullftcheck();
    check(tce);
    tce->actfilterlfdnr=filterlfdnr;
    tce->actshowhidden=showHidden;
    if(ce!=NULL) {
      ce->pos=lv->getYOffset();
      ce->xpos=lv->getXOffset();
    }
    ce=tce;
    actcacheentry=exentry;
    recheck(ce,false);
    buildLister();
    lv->setYOffset(ce->pos);
    lv->setXOffset(ce->xpos);
    sg->setText(tstr);
    showCacheState();
#ifdef DEBUG
    printf("enterDir: create successfully finished\n");
#endif
  } else {
    // can't read the dir
#ifdef DEBUG
    printf("enterDir: readdir failed\n");
#endif
    returnvalue=1;
    reset=true;
    if(tce!=NULL) {
      //TODO: Verzeichnis gibt es nicht mehr, also aus Cache entfernen
      // klappt jetzt so
      if(tce==ce) {
#ifdef DEBUG
        printf("enterDir: remove actual cache entry\n");
#endif
        // it's the actual cacheentry
        reset=false;
        returnvalue=2;
        cache->removeElement(ce);
        delete ce;
        ce=NULL;
        actcacheentry=-1;
        lv->setSize(0);
        sg->setText("");
      } else {
#ifdef DEBUG
        printf("enterDir: remove cache entry\n");
#endif
        cache->removeElement(tce);
        delete tce;
        if(ce!=NULL) actcacheentry=cache->getIndex(ce);
      }
    }
    if(reset==true) {
      // wieder herstellen aus actcacheentry
#ifdef DEBUG
      printf("enterDir: restore from %d\n",actcacheentry);
#endif
/*      tce=(NMCacheEntry*)cache->getElementAt(actcacheentry);*/
      if(ce!=NULL) {
        _freesafe(tstr);
        tstr=HandlePath(ce->verz->getDir());
        sg->setText(tstr);
      } else sg->setText("");
    }
  }
  _freesafe(tstr);
#ifdef DEBUG
  printf("enterDir: finished\n\n");
#endif
  if ( parentlister->getFocus() == true )
    parentlister->getWorker()->setTitle( sg->getText() );
  parentlister->getWorker()->getMainWin()->unsetCursor();
  return returnvalue;
}

void NormalMode::showcache(int direction)
{
  NMCacheEntry *tce;
  bool found=false;
  int cs=cache->size();
  int i=0,id;
  if(cs<1) return;

  finishsearchmode();
  
  if(ce!=NULL) {
    ce->pos=lv->getYOffset();
    ce->xpos=lv->getXOffset();
  }
  int t=actcacheentry;
  id=cache->initEnum();
  do {
    i++;
    t+=direction;
    if(t>=cache->size()) t=0;
    else if(t<0) t=cache->size()-1;
    tce=(NMCacheEntry*)cache->getElementAt(id,t);
    if(tce!=NULL) {
      if(enterDir(tce->verz->getDir())==0) {
        break;
        found=true;
      }
    }
  } while(i<cs);
  cache->closeEnum(id);
}

NM_Filter::NM_Filter()
{
  pattern=NULL;
  check=0;
}

NM_Filter::~NM_Filter()
{
  if(pattern!=NULL) _freesafe(pattern);
}

void NM_Filter::setPattern(const char *np)
{
  if(pattern!=NULL) _freesafe(pattern);
  if(np!=NULL) {
    pattern=dupstring(np);
  } else {
    pattern=NULL;
  }
}

char *NM_Filter::getPattern()
{
  return pattern;
}

void NM_Filter::setCheck(int nc)
{
  if((nc<0)||(nc>2)) return;
  check=nc;
}

int NM_Filter::getCheck()
{
  return check;
}

void NormalMode::deleteFilters()
{
  NM_Filter *fil=(NM_Filter*)filters->getFirstElement();
  while(fil!=NULL) 
  {
    delete fil;
    fil=(NM_Filter*)filters->getNextElement();
  }
}

void NormalMode::checkFilters(Verzeichnis *verz)
{
  int fid,id;
  ArrayList *fl=verz->getFiles();
  id=fl->initEnum();
  FileEntry *fe=(FileEntry*)fl->getElementAt(id,1);
  NM_Filter *fil;
  int ins=0,outs=0;
  bool isIn;
  fid=filters->initEnum();
  fil=(NM_Filter*)filters->getFirstElement(fid);
  while(fil!=NULL) {
    if(fil->getCheck()==1) ins++;
    if(fil->getCheck()==2) outs++;
    fil=(NM_Filter*)filters->getNextElement(fid);
  }
  while(fe!=NULL) {
    if((showHidden==false)&&(fe->isHidden==true)) {
      fe->use=false;
    } else {
      if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
        // a dir is always visible
        fe->use=true;
      } else {
        if(ins>0) isIn=false; else isIn=true;
        fil=(NM_Filter*)filters->getFirstElement(fid);
        if(fil==NULL) isIn=true;
        else {
          while(fil!=NULL) {
            if(fil->getCheck()>0) {
              if(fnmatch(fil->getPattern(),fe->name,0)==0) {
                if(fil->getCheck()==1) isIn=true; else isIn=false;
                break;
              }
            }
            fil=(NM_Filter*)filters->getNextElement(fid);
          }
        }
        fe->inFilter=isIn;
        fe->use=isIn;
      }
    }
    fe=(FileEntry*)fl->getNextElement(id);
  }
  filters->closeEnum(fid);
  fl->closeEnum(id);
}

NMCacheEntry::NMCacheEntry()
{
  int i;
  verz=NULL;
  activefe=NULL;
  pos=0;
  xpos=0;
  checkfe=NULL;
  lastfe = NULL;
  for(i=0;i<2;i++) {
    files[i]=-1;
    dirs[i]=-1;
  }
  for(i=0;i<2;i++) {
    files_s[i]=-1;
    dirs_s[i]=-1;
  }
  actsortmode=-1;
  actshowhidden=-1;
  actfilterlfdnr=-1;
  dontcheck=false;
  firstnullft = NULL;
}

NMCacheEntry::~NMCacheEntry()
{
  if(verz!=NULL) delete verz;
}

void NMCacheEntry::subentry(FileEntry *fe)
{
  if(fe->isDir()==true) {
    if(strcmp(fe->name,"..")!=0) {
      dirs[0]--;
      if(fe->dirsize>=0) dirs_s[0]-=fe->dirsize; else dirs_s[0]-=fe->size;
      if(fe->select==true) {
        dirs[1]--;
        if(fe->dirsize>=0) dirs_s[1]-=fe->dirsize; else dirs_s[1]-=fe->size;
      }
    }
  } else {
    files[0]--;
    files_s[0]-=fe->size;
    if(fe->select==true) {
      files[1]--;
      files_s[1]-=fe->size;
    }
  }
}

void NMCacheEntry::calc()
{
  int i;
  if(verz==NULL) return;
  ArrayList *filelist=verz->getFiles();
  int id=filelist->initEnum();
  for(i=0;i<2;i++) {
    files[i]=0;
    dirs[i]=0;
  }
  for(i=0;i<2;i++) {
    files_s[i]=0;
    dirs_s[i]=0;
  }
  FileEntry *fe=(FileEntry*)filelist->getFirstElement(id);
  while(fe!=NULL) {
    if(fe->use==true) {
      if(fe->isDir()==true) {
        if(strcmp(fe->name,"..")!=0) {
          dirs[0]++;
          if(fe->dirsize>=0) dirs_s[0]+=fe->dirsize; else dirs_s[0]+=fe->size;
          if(fe->select==true) {
            dirs[1]++;
            if(fe->dirsize>=0) dirs_s[1]+=fe->dirsize; else dirs_s[1]+=fe->size;
          }
        }
      } else {
        files[0]++;
        files_s[0]+=fe->size;
        if(fe->select==true) {
          files[1]++;
          files_s[1]+=fe->size;
        }
      }
    } else {
      if(fe==activefe) activefe=NULL;
    }
    fe=(FileEntry*)filelist->getNextElement(id);
  }
  filelist->closeEnum(id);
}

void NormalMode::selhandler(FieldListView *tlv, int row)
{
  if(ce==NULL) return;
  if(ce->verz==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  fe=(FileEntry*)filelist->getElementAt(lv->getData( row ));

  if(fe==NULL) return;

  bool state=lv->getSelect( row );
  bool oldstate=fe->select;
  
  if(state!=oldstate) {
    // Aenderung
    
    if( row == 0 ) {
      lv->setSelect( row, false );
    } else {
      if(state==true) {
        // Aus -> An
        if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
          ce->dirs[1]++;
          if(fe->dirsize>=0) ce->dirs_s[1]+=fe->dirsize; else ce->dirs_s[1]+=fe->size;
        } else {
          ce->files[1]++;
          ce->files_s[1]+=fe->size;
        }
      } else {
        // An -> Aus
        if((fe->isDir()==true)&&(fe->isCorrupt==false)) {
          ce->dirs[1]--;
          if(fe->dirsize>=0) ce->dirs_s[1]-=fe->dirsize; else ce->dirs_s[1]-=fe->size;
        } else {
          ce->files[1]--;
          ce->files_s[1]-=fe->size;
        }
      }
      fe->select=state;
    }
  }
  showCacheState();
  if ( lv->getActiveRow() == row ) {
    ce->activefe=fe;
  }
}

NMFieldListViewDND::NMFieldListViewDND(AGUIX *taguix,int tx,int ty,int width,int height,int tdata):FieldListViewDND(taguix,tx,ty,width,height,tdata)
{
  nm=NULL;
}

NMFieldListViewDND::~NMFieldListViewDND()
{
}

void NMFieldListViewDND::setSelectHandler(NormalMode *nnm)
{
  nm=nnm;
}

void NMFieldListViewDND::runSelectHandler( int row )
{
  if(nm!=NULL) nm->selhandler(this,row );
}

void NormalMode::showCacheState()
{
  if(ce==NULL) return;
  if(parentlister->isActive()==false) return;  // do nothing when we are not active
  char *tstr1=catalog.getLocale(107),*tstr2=catalog.getLocale(108);
  char *tstr3=catalog.getLocale(13);
  char *str;
  char buf1[64],buf2[64],buf3[sizeof(loff_t)*8+5],buf4[sizeof(loff_t)*8+5],bufx[sizeof(loff_t)*8];
  loff_t i1=-1,i2=-1;
  loff_t l1,l2;
  int l,tl;
  char *formatstr;
  
  l1=ce->files_s[1]+ce->dirs_s[1];
  l2=ce->files_s[0]+ce->dirs_s[0];
  if(l1>(10*1024*1024)) {
    // MB
    i1=l1/(1024*1024);
    
    bufx[WriteLong2Str(bufx,i1)]='\0';
    sprintf(buf3,"%s MB",bufx);
  } else if(l1>(10*1024)) {
    // KB
    i1=l1/1024;
    sprintf(buf3,"%ld KB",(long)i1);
  } else {
    sprintf(buf3,"%ld B",(long)l1);
  }
  if(l2>(10*1024*1024)) {
    // MB
    i2=l2/(1024*1024);

    bufx[WriteLong2Str(bufx,i2)]='\0';
    sprintf(buf4,"%s MB",bufx);
  } else if(l2>(10*1024)) {
    // KB
    i2=l2/1024;
    sprintf(buf4,"%ld KB",(long)i2);
  } else {
    sprintf(buf4,"%ld B",(long)l2);
  }
  buf1[MakeLong2NiceStr(buf1,l1)]='\0';
  buf2[MakeLong2NiceStr(buf2,l2)]='\0';
  
  // wenn mehr als 100 mal nichts an diesen Variablen
  // geaendert wurde, jeweils um 1 verringern, so dass
  // nicht ewig zu viel Platz verwendet wird, nur weil
  // einmal eine besonders grosse Datei da war
  if(maxnochanges>100) {
    maxnochanges=0;
    // auf >0 muss nicht geprft werden, da danach sowieso nochmal geprft
    // wird
    maxfilestrlen--;
    maxdirstrlen--;
    maxbytesstrlen--;
    maxwbytesstrlen--;
  }

  tl=0;
  l=(int)(log10((double)ce->files[0])+1.0);
  if(l>maxfilestrlen) {
    maxfilestrlen=l;
    tl++;
  }
  l=(int)(log10((double)ce->dirs[0])+1.0);
  if(l>maxdirstrlen) {
    maxdirstrlen=l;
    tl++;
  }
  l=strlen(buf2);
  if(l>maxbytesstrlen) {
    maxbytesstrlen=l;
    tl++;
  }

  formatstr=(char*)_allocsafe(strlen("[%s %ld / %ld]   [%s %ld / %ld]   [%s %s / %s   (%s / %s) ]")+
                              8*sizeof(maxfilestrlen)+1);

  str=(char*)_allocsafe(sizeof(char)*(strlen(tstr1)+strlen(tstr2)+strlen(tstr3)+
                        2*maxfilestrlen+
                        2*maxdirstrlen+
                        2*maxbytesstrlen+
                        2*maxwbytesstrlen+256));
  if(i2>=0) {
    // this only when the field is visible
    l=strlen(buf4);
    if(l>maxwbytesstrlen) {
      maxwbytesstrlen=l;
      tl++;
    }

    sprintf(formatstr,"[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds   (%%%ds / %%%ds) ]",
            maxfilestrlen,
            maxfilestrlen,
            maxdirstrlen,
            maxdirstrlen,
            maxbytesstrlen,
            maxbytesstrlen,
            maxwbytesstrlen,
            maxwbytesstrlen);
    sprintf(str,formatstr,tstr1,ce->files[1],ce->files[0],
            tstr2,ce->dirs[1],ce->dirs[0],tstr3,buf1,buf2,buf3,buf4);
  } else {
    sprintf(formatstr,"[%%s %%%dld / %%%dld]   [%%s %%%dld / %%%dld]   [%%s %%%ds / %%%ds]",
            maxfilestrlen,
            maxfilestrlen,
            maxdirstrlen,
            maxdirstrlen,
            maxbytesstrlen,
            maxbytesstrlen);
    tl=strlen(formatstr);
    for(l=0;l<(9+2*maxwbytesstrlen);l++)
      formatstr[tl+l]=' ';
    formatstr[tl+l]='\0';
    sprintf(str,formatstr,tstr1,ce->files[1],ce->files[0],
            tstr2,ce->dirs[1],ce->dirs[0],tstr3,buf1,buf2);
  }
  _freesafe(formatstr);
  
  if(tl==0) // keine nderungen
    maxnochanges++;
  
  parentlister->setStatebarText(str);
  _freesafe(str);
}

bool NormalMode::isType(const char *str)
{
  if(strcmp(str,type)==0) return true; else return false;
}

const char *NormalMode::getType()
{
  return type;
}

int NormalMode::configure()
{
  Button *okb,*cb,*fb;
  AWindow *win;
  CycleButton *cyb[2];
  ChooseButton *chb,*ucb;
  Text *ttext;
  int tw,ttw,tth,ttx,tty,t1,t2;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  bool tshowhidden;
  List *tfilters;
  int id;
  NM_Filter *fi,*tfi;
  StringGadget *tsg;
  int tut;
  
  ttw=tth=10;
  ttx=tty=5;
  win=new AWindow(aguix);

  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(getName())+1);
  sprintf(tstr,catalog.getLocale(293),getName());
  win->create(NULL,10,10,ttw,tth,0,tstr);
  _freesafe(tstr);

  ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(158),1));
  ttx+=ttext->getWidth()+5;
  cyb[0]=(CycleButton*)win->add(new CycleButton(aguix,ttx,tty,100,1,0,0));
  cyb[0]->addOption(catalog.getLocale(160));
  cyb[0]->addOption(catalog.getLocale(161));
  cyb[0]->addOption(catalog.getLocale(162));
  cyb[0]->addOption(catalog.getLocale(163));
  cyb[0]->addOption(catalog.getLocale(164));
  cyb[0]->addOption(catalog.getLocale( 432 ));
  cyb[0]->addOption(catalog.getLocale( 433 ));
  cyb[0]->resize(cyb[0]->getMaxSize(),cyb[0]->getHeight());
  if((sortmode&0xff)==SORT_SIZE) cyb[0]->setOption(1);
  else if((sortmode&0xff)==SORT_ACCTIME) cyb[0]->setOption(2);
  else if((sortmode&0xff)==SORT_MODTIME) cyb[0]->setOption(3);
  else if((sortmode&0xff)==SORT_CHGTIME) cyb[0]->setOption(4);
  else if((sortmode&0xff)==SORT_TYPE) cyb[0]->setOption(5);
  else if((sortmode&0xff)==SORT_OWNER) cyb[0]->setOption(6);
  else cyb[0]->setOption(0);

  tty+=cyb[0]->getHeight()+5;
  tw=ttx+cyb[0]->getWidth()+5;
  if(tw>ttw) ttw=tw;
  ttx=5;

  chb=(ChooseButton*)win->add(new ChooseButton(aguix,ttx,tty,20,20,
                                              ((sortmode&SORT_REVERSE)==SORT_REVERSE)?1:0,
                                              catalog.getLocale(165),LABEL_RIGHT,1,0));
  tty+=chb->getHeight()+5;

  ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(353),1));
  ttx+=ttext->getWidth()+5;
  cyb[1]=(CycleButton*)win->add(new CycleButton(aguix,ttx,tty,100,1,0,0));
  cyb[1]->addOption(catalog.getLocale(354));
  cyb[1]->addOption(catalog.getLocale(355));
  cyb[1]->addOption(catalog.getLocale(356));
  cyb[1]->resize(cyb[1]->getMaxSize(),cyb[1]->getHeight());
  if((sortmode&SORT_DIRLAST)==SORT_DIRLAST) cyb[1]->setOption(1);
  else if((sortmode&SORT_DIRMIXED)==SORT_DIRMIXED) cyb[1]->setOption(2);
  else cyb[1]->setOption(0);

  tty+=cyb[1]->getHeight()+5;
  tw=ttx+cyb[1]->getWidth()+5;
  if(tw>ttw) ttw=tw;
  ttx=5;

  t1=(strlen(catalog.getLocale(159))+2);
  t1*=aguix->getCharWidth();
  fb=(Button*)win->add(new Button(aguix,ttx,tty,t1,catalog.getLocale(159),1,0,0));
  tw=t1+10;
  if(tw>ttw) ttw=tw;
  tty+=fb->getHeight()+5;

  ucb=(ChooseButton*)win->add(new ChooseButton(aguix,ttx,tty,20,20,
                                              (showfreespace==true)?1:0,
                                              catalog.getLocale(128),LABEL_RIGHT,1,0));
  tty+=ucb->getHeight()+5;
  ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(129),1));
  ttx+=ttext->getWidth()+5;
  tstr=(char*)_allocsafe(sizeof(int)*8);
  sprintf(tstr,"%d",updatetime);
  tsg=(StringGadget*)win->add(new StringGadget(aguix,ttx,tty,ttw-ttx-5,tstr,0));
  _freesafe(tstr);
  tty+=tsg->getHeight()+5;

  t1=(strlen(catalog.getLocale(11))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>ttw) ttw=tw;
  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   tty,
                                   t1,
                                   catalog.getLocale(11),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  ttw-5-t2,
                                  tty,
                                  t2,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));
  tty+=okb->getHeight()+5;
  
  tth=tty;
  
  tshowhidden=showHidden;
  tfilters=new List();
  id=filters->initEnum();
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    tfilters->addElement(fi->duplicate());
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  
  win->resize(ttw,tth);
  win->setMaxSize(ttw,tth);
  win->setMinSize(ttw,tth);
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==cb) endmode=1;
          else if(msg->button.button==fb) {
            configureFilters(&tshowhidden,tfilters);
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    int tsortmode=0;
    switch(cyb[0]->getSelectedOption()) {
      case 1:
        tsortmode=SORT_SIZE;
        break;
      case 2:
        tsortmode=SORT_ACCTIME;
        break;
      case 3:
        tsortmode=SORT_MODTIME;
        break;
      case 4:
        tsortmode=SORT_CHGTIME;
        break;
      case 5:
        tsortmode=SORT_TYPE;
        break;
      case 6:
        tsortmode=SORT_OWNER;
        break;
      default:
        tsortmode=SORT_NAME;
        break;
    }
    if(chb->getState()==1) tsortmode|=SORT_REVERSE;
    switch(cyb[1]->getSelectedOption()) {
      case 1:
        tsortmode|=SORT_DIRLAST;
        break;
      case 2:
        tsortmode|=SORT_DIRMIXED;
        break;
      default:
        break;
    }
    setSortmode(tsortmode);
    setShowHiddenE(tshowhidden);
    // apply tfilters
    setFilters(tfilters);
    setShowFreeSpace((ucb->getState()==1)?true:false);
    tut=atoi(tsg->getText());
    setUpdatetime(tut);
  }
  id=tfilters->initEnum();
  tfi=(NM_Filter*)tfilters->getFirstElement(id);
  while(tfi!=NULL) {
    delete tfi;
    tfi=(NM_Filter*)tfilters->getNextElement(id);
  }
  tfilters->closeEnum(id);
  delete tfilters;
  
  win->close();
  delete win;

  return endmode;
}

void NormalMode::down()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    pos=al1->getIndex(ce->activefe);
    do {
      pos++;
      fe=(FileEntry*)al1->getElementAt(id,pos);
      if(fe==NULL) break;
    } while(fe->use==false);
    if(fe!=NULL) {
      // Naechstes Element gefunden
      ce->activefe=fe;
      row = lv->getActiveRow();
      if( lv->isValidRow( row ) == true ) {
        lv->setActiveRow( row + 1 );
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::up()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    pos=al1->getIndex(ce->activefe);
    do {
      pos--;
      fe=(FileEntry*)al1->getElementAt(id,pos);
      if(fe==NULL) break;
    } while(fe->use==false);
    if(fe!=NULL) {
      // Voriges Element gefunden
      ce->activefe=fe;

      row = lv->getActiveRow();
      if ( lv->isValidRow( row ) == true ) {
        lv->setActiveRow( row - 1 );
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::check(NMCacheEntry *tce)
{
  checkFilters(tce->verz);
  tce->calc();
  cyclicfunc(CYCLICFUNC_MODE_FIRSTNULLFT);
}

void NormalMode::toggleHidden()
{
  setShowHidden((showHidden==true)?false:true);
}

void NormalMode::setShowHidden(bool nhf)
{
  showHidden=nhf;
  if(ce!=NULL) check(ce);
  setName();
}

void NormalMode::toggleHiddenE()
{
  setShowHiddenE((showHidden==true)?false:true);
}

void NormalMode::setShowHiddenE(bool nhf)
{
  finishsearchmode();
  
  setShowHidden(nhf);
  buildLister();
  showCacheState();
  if(lv!=NULL) lv->showActive();
}

bool NormalMode::isDoubleClick(struct timeval *t1,struct timeval *t2)
{
  int dt=0;
  int s,us;
  s=abs(t1->tv_sec-t2->tv_sec);
  us=t1->tv_usec-t2->tv_usec;
  if(us<0) {
    us+=1000000;
    s--;
  }
  dt=us+s*1000000;
  if(dt<350000) return true;
  return false;
}

void NormalMode::startAction(FileEntry *tfe)
{
  WCFiletype *ft;
  if(tfe==NULL) return;
  if(tfe->isCorrupt==true) return; // Ziel existiert sowieso nicht, alles koennen wir nichts
                                   // machen

  finishsearchmode();
  
  if(tfe->isDir()==true) {
    // Verzeichnis "entern"
    enterDir(tfe->fullname);
  } else {
    ft=tfe->filetype;
    if(ft==NULL) ft=wconfig->getnotyettype();
    if(ft!=NULL) {
      ActionMessage amsg;
      amsg.flags=ft->getDoubleClickActionsGF();
//      amsg.mode=amsg.AM_MODE_ONLYACTIVE;
      amsg.mode=amsg.AM_MODE_SPECIAL;
      amsg.extraInfo=tfe;
      parentlister->getWorker()->interpret(ft->getDoubleClickActions(),&amsg);
    }
  }
}

void NormalMode::cyclicfunc(cyclicfunc_mode_t mode)
{
  WCFiletype *ft;
  int trow;
  int side;
  int *dis;
  bool finished;
  bool doflush;

  if(mode==CYCLICFUNC_MODE_NORMAL) {
    if(ce!=NULL) {
      FileEntry *fe=ce->checkfe;

      side = parentlister->getSide();
      dis = wconfig->getDisplay( side );
  
      ft = NULL;

      finished = false;
      doflush = false;
      
#ifdef WANT_THREADS
      if ( thread_usage == true ) {
        if ( ce->firstnullft != NULL )
           finish_done = false;

        // first fill the slave buffer
        reclist->lock();
        while ( ( fe != NULL ) && ( reclist->isFull_locked() == false ) ){
          ft_rec_list::ft_rec_list_t *te = new ft_rec_list::ft_rec_list_t;
          te->name = fe->fullname;
          te->fe = fe;
          te->dontCheckContent = ce->dontcheck;

          reclist->put_locked( te );
          ce->nextcheckfe();
  
          if ( ce->checkfe == NULL )
            ce->lastfe = fe;
  
          fe = ce->checkfe;
        }
        reclist->signal();
        reclist->unlock();
      
        // now check for results
        trow = 0;
        
        recreslist->lock();
        ft_recres_list::ft_recres_list_t *rte;
        ft_rec_list::ft_rec_list_t *te;
        while ( recreslist->isEmpty_locked() == false ) {
          // alle Resultate entnehmen
          rte = recreslist->remove_locked();
          if ( rte != NULL ) {
            // mit oberstem Element von reclist vergleichen
            reclist->lock();
  
            te = reclist->gettop_locked();
            if ( te != NULL ) {
              if ( strcmp( te->name, rte->name ) == 0 ) {
                // correct answer
                te = reclist->remove_locked();
#ifdef DEBUG
//                printf("type for %s:%d\n",te->name,rte->ft_index);
#endif
                if ( rte->ft_index >= 0 ) {
                  ft = (WCFiletype*)wconfig->getFiletypes()->getElementAt( rte->ft_index );
                } else {
                  ft = NULL;
                }
                
                if ( ft != NULL ) {
                  // change fe and lvc to filetype
                  fe = te->fe;
                  fe->filetype = ft;
  
                  while( lv->isValidRow( trow ) == true ) {
                    if ( lv->getData( trow ) == fe->nr ) break;
                    trow++;
                  }
                  if ( lv->isValidRow( trow ) == false ) {
                    // lvc not found
                    // because we start always at the last position
                    // let's search again from 0

#ifdef DEBUG
                    printf( "filetype lvc search rollover\n" );
#endif
  
                    trow = 0;
                    while( lv->isValidRow( trow ) == true ) {
                      if ( lv->getData( trow ) == fe->nr ) break;
                      trow++;
                    }
                  }
                  
                  if ( lv->isValidRow( trow ) == true ) {
                    ce->verz->setLVC4FE_DND( lv, trow, fe, dis );
                    //lv->redraw( trow );
                    doflush = true;
                  } else {
                    // lvc not found
                    // this is not a problem because filters,... could be changed
                  
                    // reset pos for next element
                    trow = 0;
                  }
                  
                  if ( fe == ce->firstnullft ) {
                    // firstnullft is no longer the first element with null ft
                    ce->next_firstnullft();
                  }
                }
                
#if 0
                if ( ( te->fe == ce->lastfe ) && ( ce->checkfe == NULL ) ) {
                  // last fileentry recognized
                  finished = true;
                }
#endif
                
                delete te;
              } else {
#ifdef DEBUG
                printf( "wrong answer: %s\n", rte->name );
#endif
              }
            }
            _freesafe( rte->name );
            delete rte;

            reclist->unlock();
          }
        }
        if ( ( ce->firstnullft == NULL ) && ( finish_done == false ) )
           finished = true;
        
        recreslist->unlock();
      } else {  // thread_usage
#endif
        if ( fe != NULL ) {
          ft = fe->checkFiletype( wconfig->getFiletypes(), ce->dontcheck );
          if ( ft != NULL ) {
            trow = 0;
            while ( lv->isValidRow( trow ) == true ) {
              if ( lv->getData( trow ) == fe->nr ) break;
              trow++;
            }
            if ( lv->isValidRow( trow ) == true ) {
              ce->verz->setLVC4FE_DND( lv, trow, fe, dis );
              //lv->redraw( trow );
              doflush = true;
            }
          }
          ce->nextcheckfe();
          if(ce->checkfe==NULL) {
            finished = true;
          }
        }
#ifdef WANT_THREADS
      }
#endif
      if ( finished == true ) {
        // last fe recognized
        // the following is no longer needed because I have FieldLV now
#if 0
        // Am Ende
        #ifdef DEBUG
        printf("lv->getYOffset\n");
        #endif
 
        int ty=lv->getYOffset();

        #ifdef DEBUG
        printf("buildLister\n");
        #endif

        buildLister();

        #ifdef DEBUG
        printf("lv->setYOffset\n");
        #endif

        lv->setYOffset(ty);
        doflush = true;
#endif
        finish_done = true;
      }
      
      if ( doflush == true ) {
        #ifdef DEBUG
        printf("Flush\n");
        #endif
        lv->redraw();
        aguix->Flush();
      }
      showFreeSpace(false);
    }
  } else if(mode==CYCLICFUNC_MODE_FIRSTNULLFT) {
    NMCacheEntry *tce;
    int id=cache->initEnum();
    tce=(NMCacheEntry*)cache->getFirstElement(id);
    while(tce!=NULL) {
      tce->firstnullftcheck();
      tce=(NMCacheEntry*)cache->getNextElement(id);
    }
    cache->closeEnum(id);
  } else {
#ifdef WANT_THREADS
    // get slave in reinit state

    if ( thread_usage == true ) {
      // first wait for the slave to process previous order
      while ( slave.order != THREAD_NOP ) waittime( 1 );
    }
    
    // put it in reinit mode
    slave.order = THREAD_REINIT;
    // wake up slave
    slave.waitvar.signal();
    reclist->signal();

    // update filetype list
    ft_list_update();
    
    // clear lists
    ft_rec_list_clear();
    ft_recres_list_clear();
    
    // wake up slave
    slave.filetype_ex.signal();
    
    if ( thread_usage == true ) {
      // wait for slave in wait mode
      // slave will go from reinit in wait mode after wake up
      while ( slave.order != THREAD_NOP ) waittime( 1 );
      while ( slave.status != THREAD_WAIT ) waittime( 1 );
    }
    
    slave.order = THREAD_RUN;
    // wake up slave
    slave.waitvar.signal();
    reclist->signal();
#endif

    setupLVFields();
  
    NMCacheEntry *tce;
    int id=cache->initEnum();
    tce=(NMCacheEntry*)cache->getFirstElement(id);
    while(tce!=NULL) {
      tce->restartcheck();
      tce->checkfordcd();
      tce=(NMCacheEntry*)cache->getNextElement(id);
    }
    cache->closeEnum(id);
    if(ce!=NULL) {
      int oldx,oldy;
      oldx=lv->getXOffset();
      oldy=lv->getYOffset();
      buildLister();
      lv->setYOffset(oldy);
      lv->setXOffset(oldx);
    }
  }
}

void NMCacheEntry::nextcheckfe()
{
  if(checkfe==NULL) return;
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      int tpos=fes->getIndex(checkfe)+1;
      for(;;) {
        checkfe=(FileEntry*)fes->getElementAt(tpos++);
        if(checkfe!=NULL) {
          if(checkfe->use==true) {
            // Pruefen, ob Dateityp noch nicht erkannt wurde
            // TODO: Jetzt reicht noch auf NULL pruefen, spaeter muss auf den NotYetChecked geprueft
            // werden
            if(checkfe->filetype==NULL) {
              if((checkfe->isLink==true)&&(checkfe->isCorrupt==false)&&(checkfe->isDir()==false)) break;
              if(checkfe->isDir()==false) break;
            }
          }
        } else break;
      }
    } else checkfe=NULL;
  } else checkfe=NULL;
}

void NMCacheEntry::restartcheck()
{
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      int id=fes->initEnum();
      FileEntry *fe=(FileEntry*)fes->getFirstElement(id);
      while(fe!=NULL) {
        fe->filetype=NULL;
        fe=(FileEntry*)fes->getNextElement(id);
      }
      fes->closeEnum(id);
      checkfe=(FileEntry*)fes->getFirstElement();
    } else checkfe=NULL;
  } else checkfe=NULL;
  nextcheckfe();
  firstnullft = checkfe;
}

void NMCacheEntry::firstnullftcheck()
{
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      checkfe=(FileEntry*)fes->getFirstElement();
    } else checkfe=NULL;
  } else checkfe=NULL;
  nextcheckfe();
}

void NMCacheEntry::reset_firstnullft()
{
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      firstnullft=(FileEntry*)fes->getFirstElement();
      next_firstnullft();
    } else firstnullft=NULL;
  } else firstnullft=NULL;
}

void NMCacheEntry::next_firstnullft()
{
  if(firstnullft==NULL) return;
  if(verz!=NULL) {
    ArrayList *fes=verz->getFiles();
    if(fes!=NULL) {
      int tpos=fes->getIndex(firstnullft)+1;
      for(;;) {
        firstnullft=(FileEntry*)fes->getElementAt(tpos++);
        if(firstnullft!=NULL) {
          if(firstnullft->use==true) {
            // Pruefen, ob Dateityp noch nicht erkannt wurde
            // TODO: Jetzt reicht noch auf NULL pruefen, spaeter muss auf den NotYetChecked geprueft
            // werden
            if(firstnullft->filetype==NULL) {
              if((firstnullft->isLink==true)&&(firstnullft->isCorrupt==false)&&(firstnullft->isDir()==false)) break;
              if(firstnullft->isDir()==false) break;
            }
          }
        } else break;
      }
    } else firstnullft=NULL;
  } else firstnullft=NULL;
}

const char* NormalMode::getName()
{
  return catalog.getLocale(173);
}

int NormalMode::load(Datei *fh)
{
  int chunksize=fh->getInt();
//  fh->putInt(filters->size());  //TODO: was soll das hier

  int cs=Datei::getUCharSize(),
      is=Datei::getIntSize();
  unsigned char c1;
  int size;
  int tsortmode=0;
  int id;
  bool tshowHidden=false;
  List *tfilters;
  NM_Filter *tfi;

  if(chunksize>=(4*cs+is)) {
    tshowHidden=(fh->getUChar()==1)?true:false;
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        tsortmode=SORT_SIZE;
        break;
      case 2:
        tsortmode=SORT_ACCTIME;
        break;
      case 3:
        tsortmode=SORT_MODTIME;
        break;
      case 4:
        tsortmode=SORT_CHGTIME;
        break;
      case 5:
        tsortmode=SORT_TYPE;
        break;
      case 6:
        tsortmode=SORT_OWNER;
        break;
      default:
        tsortmode=SORT_NAME;
        break;
    }
    if(fh->getUChar()==1) tsortmode|=SORT_REVERSE;
    c1=fh->getUChar();
    switch(c1) {
      case 1:
        tsortmode|=SORT_DIRLAST;
        break;
      case 2:
        tsortmode=SORT_DIRMIXED;
        break;
      default:
        break;
    }
    size=fh->getInt();
    chunksize-=4*cs+is;
    
    tfilters=new List();
    for(int i=0;i<size;i++) {
      tfi=new NM_Filter();
      tfi->load(fh);
      tfilters->addElement(tfi);
    }
    // apply new values
    setShowHiddenE(tshowHidden);
    setSortmode(tsortmode);
    setFilters(tfilters);
    id=tfilters->initEnum();
    tfi=(NM_Filter*)tfilters->getFirstElement(id);
    while(tfi!=NULL) {
      delete tfi;
      tfi=(NM_Filter*)tfilters->getNextElement(id);
    }
    tfilters->closeEnum(id);
    delete tfilters;
    // this is added from version 2.0.2 so for file from older version
    if(chunksize>=(cs+is)) {
      showfreespace=(fh->getUChar()==1)?true:false;
      chunksize-=cs;
      setUpdatetime(fh->getInt());
      chunksize-=is;
    }
  }
  while(chunksize>0) {
    fh->getUChar();
    chunksize--;
  }
  return 0;
}

bool NormalMode::save(Datei *fh)
{
  int cs=5*Datei::getUCharSize()+2*Datei::getIntSize();
  int id;

  fh->putInt(cs);
  fh->putUChar((showHidden==true)?1:0);
  switch(sortmode&0xff) {
    case SORT_SIZE:
      fh->putUChar(1);
      break;
    case SORT_ACCTIME:
      fh->putUChar(2);
      break;
    case SORT_MODTIME:
      fh->putUChar(3);
      break;
    case SORT_CHGTIME:
      fh->putUChar(4);
      break;
    case SORT_TYPE:
      fh->putUChar(5);
      break;
    case SORT_OWNER:
      fh->putUChar(6);
      break;
    default:
      fh->putUChar(0);
      break;
  }
  fh->putUChar(((sortmode&SORT_REVERSE)==0)?0:1);
  if((sortmode&SORT_DIRLAST)==SORT_DIRLAST) fh->putUChar(1);
  else if((sortmode&SORT_DIRMIXED)==SORT_DIRMIXED) fh->putUChar(2);
  else fh->putUChar(0);
  fh->putInt(filters->size());
  id=filters->initEnum();
  NM_Filter *fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    fi->save(fh);
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  fh->putUChar((showfreespace==true)?1:0);
  fh->putInt(updatetime);
  return false;
}

void NormalMode::top()
{
  ArrayList *al1;
  int id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  ce->activefe=(FileEntry*)al1->getFirstElement(id);
  lv->setActiveRow( 0 );
  lv->showActive();
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::last()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  pos=al1->size();
  do {
    pos--;
    fe=(FileEntry*)al1->getElementAt(id,pos);
    if(fe==NULL) break;
  } while(fe->use==false);
  if(fe!=NULL) {
    ce->activefe=fe;
    row = lv->getElements() - 1;
    lv->setActiveRow( row );
    lv->showActive();
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::pageup()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    int rpos=lv->getMaxDisplayV()-1;
    FileEntry *lasthit=ce->activefe;
    pos=al1->getIndex(ce->activefe);
    while(rpos>0) {
      do {
        pos--;
        fe=(FileEntry*)al1->getElementAt(id,pos);
        if(fe==NULL) break;
      } while(fe->use==false);
      if(fe==NULL) break;
      lasthit=fe;
      rpos--;
    };
    if(lasthit!=ce->activefe) {
      ce->activefe=lasthit;
      row = lv->getActiveRow();
      if( row >= 0 ) {
        row -= lv->getMaxDisplayV()-1;
        if(row<0) row=0;
        lv->setActiveRow( row);
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::pagedown()
{
  ArrayList *al1;
  int row;
  FileEntry *fe;
  int pos,id;
  if(ce==NULL) return;

  finishsearchmode();
  
  al1=ce->verz->getFiles();
  id=al1->initEnum();
  if(ce->activefe==NULL) {
    ce->activefe=(FileEntry*)al1->getFirstElement(id);
    lv->setActiveRow( 0 );
    lv->showActive();
  } else {
    int rpos=lv->getMaxDisplayV()-1;
    FileEntry *lasthit=ce->activefe;
    pos=al1->getIndex(ce->activefe);
    while(rpos>0) {
      do {
        pos++;
        fe=(FileEntry*)al1->getElementAt(id,pos);
        if(fe==NULL) break;
      } while(fe->use==false);
      if(fe==NULL) break;
      lasthit=fe;
      rpos--;
    };
    if(lasthit!=ce->activefe) {
      ce->activefe=lasthit;
      row = lv->getActiveRow();
      if ( row >= 0 ) {
        row += lv->getMaxDisplayV()-1;
        if(row>=lv->getElements()) row=lv->getElements()-1;
        lv->setActiveRow( row );
        lv->showActive();
      }
    }
  }
  al1->closeEnum(id);
  aguix->Flush();
}

void NormalMode::select()
{
  if(ce==NULL) return;
  if(ce->activefe==NULL) return;

  finishsearchmode();
  
  int row=lv->getActiveRow();
  if( row < 0 ) {
    printf("oops, this should never happen!\n");
    return;
  }
  if(lv->getSelect(row)==false) lv->setSelect(row, true); else lv->setSelect(row,false);
  selhandler(lv,row );
  down();
}

void NormalMode::selectall()
{
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  
  finishsearchmode();
  
  /* LVCs auswaehlen */
  row = 1;
  while( row < lv->getElements() ) {
    lv->setSelect(row,true);
    row++;
  }
  /* FEs auswaehlen */
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if((fe->use==true)&&(strcmp(fe->name,"..")!=0)) fe->select=true;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  filelist->closeEnum(eid);
  ce->files[1]=ce->files[0];
  ce->files_s[1]=ce->files_s[0];
  ce->dirs[1]=ce->dirs[0];
  ce->dirs_s[1]=ce->dirs_s[0];
  lv->redraw();
  showCacheState();
}

void NormalMode::selectnone()
{
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  
  finishsearchmode();
  
  /* LVCs deselecten */
  row = 0;
  while( row < lv->getElements()) {
    lv->setSelect(row,false);
    row++;
  }
  /* FEs deselecten */
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if(fe->use==true) fe->select=false;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  filelist->closeEnum(eid);
  ce->files[1]=0;
  ce->files_s[1]=0;
  ce->dirs[1]=0;
  ce->dirs_s[1]=0;
  lv->redraw();
  showCacheState();
}

void NormalMode::invertall()
{
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  
  finishsearchmode();
  
  /* LVCs invertieren */
  row = 1;
  while(row<lv->getElements()) {
    lv->setSelect(row,lv->getSelect(row)==true?false:true);
    row++;
  }
  /* FEs invertieren */
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if((fe->use==true)&&(strcmp(fe->name,"..")!=0)) fe->select=(fe->select==true?false:true);
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  filelist->closeEnum(eid);
  ce->files[1]=ce->files[0]-ce->files[1];
  ce->files_s[1]=ce->files_s[0]-ce->files_s[1];
  ce->dirs[1]=ce->dirs[0]-ce->dirs[1];
  ce->dirs_s[1]=ce->dirs_s[0]-ce->dirs_s[1];
  lv->redraw();
  showCacheState();
}

void NormalMode::parent()
{
  char oentry[128];
  FileEntry *fe;
  if(ce==NULL) return;
  ArrayList *filelist;
  int row;
  int eid;
  
  char *adir=ce->verz->getDir();
  char *ndir=ParentDir(adir,oentry,127);
  enterDir(ndir);
  /* oentry finden und aktivieren */
  filelist=ce->verz->getFiles();
  eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if(fe->use==true) if(strcmp(fe->name,oentry)==0) break;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }
  if(fe!=NULL) {
    /* Eintrag gefunden */
    ce->activefe=fe;
    row = 0;
    while( row < lv->getElements() ) {
      if(lv->getData(row)==fe->nr) break;
      row++;
    }
    if( row < lv->getElements()) {
      lv->setActiveRow(row);
      lv->showActive();
      aguix->Flush();
    }
  }
  filelist->closeEnum(eid);
  _freesafe(ndir);
}

void NormalMode::enterDirActive()
{
  if(ce==NULL) return;
  if(ce->activefe!=NULL) {
    if(ce->activefe->isDir()==true) enterDir(ce->activefe->fullname);
  }
}

void NormalMode::filterselect(char *filter,int mode)
{
  /* Steht in filter[0] ein /, werden Verzeichnisse betrachtet, sonst Dateien */
  if(ce==NULL) return;
  FileEntry *fe;
  ArrayList *filelist=ce->verz->getFiles();
  int row;
  char *realfilter;
  bool tselect=(mode==0)?true:false;
  bool usedir;
  
  finishsearchmode();
  
  if(filter[0]=='/') {
    usedir=true;
    realfilter=&filter[1];
  } else {
    usedir=false;
    realfilter=filter;
  }
  
  int eid=filelist->initEnum();
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while(fe!=NULL) {
    if((fe->use==true)&&(strcmp(fe->name,"..")!=0))
      if(fe->match(realfilter)==true) 
        if(fe->isDir()==usedir) fe->select=tselect;
    fe=(FileEntry*)filelist->getNextElement(eid);
  }

  row = 0;
  fe=(FileEntry*)filelist->getFirstElement(eid);
  while( row < lv->getElements()) {
    while(fe->nr!=lv->getData(row)) fe=(FileEntry*)filelist->getNextElement(eid);
    lv->setSelect(row,fe->select);
    row++;
  }
  filelist->closeEnum(eid);
  lv->redraw();
  ce->calc();
  showCacheState();
}

char *NormalMode::getActualDir()
{
  if(ce!=NULL) return ce->verz->getDir();
  return NULL;
}

void
NormalMode::copy(struct NM_copyorder *copyorder)
{
  Lister *olister=NULL;
  NormalMode *nm2=NULL,*updatenm=NULL;
  ListerMode *lm=NULL;
  List *copylist;
  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2,*tss;
  ArrayList *filelist;
  FileEntry *fe,*tfe;
  bool found;
  int row;
  int id,feid,lvcid;
  bool enter;
  NM_CopyOp_Dir *cod1,*tcod;
  unsigned long files,dirs,gf,gd;
  loff_t bytes;
  bool skip,cancel=false;
  CopyOpWin *cowin;
  nm_copy_t ce1;
  bool nodatabase;
  int oldx,oldy;
  int erg;
  char *textstr,*buttonstr,*tstr;
  
  if(copyorder==NULL) return;
  if(copyorder->destdir==NULL) return;
  if(getActualDir()==NULL) {
    // no actual dir => no files to copy
    return;
  }
  copyorder->overwrite=copyorder->NM_OVERWRITE_NORMAL;  // force to start in this mode

  finishsearchmode();
  
  if(Datei::fileExistsExt(copyorder->destdir)!=Datei::D_FE_DIR) {
    // no dir as destination
    textstr=(char*)_allocsafe(strlen(catalog.getLocale(67))+strlen(copyorder->destdir)+1);
    sprintf(textstr,catalog.getLocale(67),copyorder->destdir);
    buttonstr=catalog.getLocale(11);
    req->request(catalog.getLocale(125),textstr,buttonstr);
    _freesafe(textstr);
    return;
  }
  
#ifdef WANT_THREADS
  // clear the reclist so the slace doesn't block us because he is still reading
  // from disk
  ft_rec_list_clear();
#endif
  
  if(strcmp(getActualDir(),copyorder->destdir)==0) {
    copyorder->do_rename=true;
    updatenm=this;
  } else {
    olister=parentlister->getWorker()->getOtherLister(parentlister);
    if(olister!=NULL) {
      lm=olister->getActiveMode();
      if(lm!=NULL) {
        if(lm->isType("NormalMode")==true) {
          nm2=(NormalMode*)lm;
          if(nm2->getActualDir()!=NULL) {
            if(strcmp(nm2->getActualDir(),copyorder->destdir)==0) {
              updatenm=nm2;
            }
          }
        }
      }
    }
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  buf=_allocsafe(BUFSIZE);
  
  if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;
  
  // no sense for follow_symlinks when moving
  if(copyorder->move==true) copyorder->follow_symlinks=false;

  /* now presupposition checked
     next search for the files to copy
     therefore collect entries to copy in a list together with the LVC (when on root-level)
     then for each entry do recursive call to copy it
     after success deactivate LVC */

  /* first: create List contains NM_specialsource
     when source==NM_SPECIAL then the given list but check for existing of the FEs
     else source==NM_ONLYACTIVE => take the activefe (from ce) in this list
     else take all selected entries (except the "..")
  */
  cowin=copyorder->cowin;
  if(cowin!=NULL) {
    cowin->open(nodatabase);
    cowin->setmessage(catalog.getLocale(122),0);
    cowin->redraw();
  }
  copylist=new List();
  switch(copyorder->source) {
    case copyorder->NM_SPECIAL:
      ss1=(NM_specialsource*)copyorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if(fe==ss1->entry) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=ss1->entry;
          ss2->row=ss1->row;
          ss2->cod=NULL;
          copylist->addElement(ss2);
        }
        ss1=(NM_specialsource*)copyorder->sources->getNextElement();
      }
      break;
    case copyorder->NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if( row >= 0) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->row=row;
              ss2->cod=NULL;
              copylist->addElement(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            copylist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(copylist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(row >= 0) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=ce->activefe;
                ss2->row=row;
                ss2->cod=NULL;
                copylist->addElement(ss2);
              }
            }
          }
        }
      }
      break;
  }
  if(cowin!=NULL) cowin->redraw();

  // create the NM_CopyOp_Dir for each dir in copylist
  files=dirs=gf=gd=0;
  bytes = 0;
  id=copylist->initEnum();
  
  if(nodatabase==false) {
    ss1=(NM_specialsource*)copylist->getFirstElement(id);
    while((ss1!=NULL)&&(cancel==false)) {
      enter=false;
      if(ss1->entry->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if(ss1->entry->isLink==false) enter=true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(ss1->entry);
        // recursive call
        if(cod1->createSubDirs(copyorder,&gf,&gd)!=0) cancel=true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;
        bytes+=cod1->bytes;

        ss1->cod=cod1;
        // this is a dir so inc the counter
        dirs++;
        gd++;
        if(cowin!=NULL) {
          cowin->set_files_to_copy(gf);
          cowin->set_dirs_to_copy(gd);
          if(cowin->redraw()!=0) cancel=true;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        gf++;
        if ( ( ss1->entry->isLink == true ) &&
             ( copyorder->follow_symlinks == true ) ) {
          if ( ss1->entry->isCorrupt == false ) {
            bytes+=ss1->entry->dsize;
          }
        } else {
          bytes+=ss1->entry->size;
        }
        ss1->cod=NULL;
      }
      ss1=(NM_specialsource*)copylist->getNextElement(id);
    }
  }

  if(cowin!=NULL) {
    cowin->set_files_to_copy(files);
    cowin->set_dirs_to_copy(dirs);
    cowin->set_bytes_to_copy(bytes);
    cowin->starttimer();
  }
  // now database ready, can start copy-process

  copy_deletefe_list=new List();
  copy_lvc_list=new List();

  skip=false;
  ss1=(NM_specialsource*)copylist->getFirstElement(id);
  while((ss1!=NULL)&&(cancel==false)) {
    if(nodatabase==false) {
      tcod=ss1->cod;
    } else {
      enter=false;
      tcod=NULL;
      if(ss1->entry->isDir()==true) {
        if(ss1->entry->isLink==false) enter=true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if(enter==true) tcod=new NM_CopyOp_Dir(ss1->entry);
    }
    if(tcod!=NULL) {
      ce1=copydir(ss1->entry,tcod,copyorder,true,copyorder->destdir);
      if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
        // success
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
      if(copyorder->move==true) {
        if((ce1==NM_COPY_OK)&&(tcod->error_counter==0)) {
          if(ss1->row>=0) lv->showRow(ss1->row);
          removeEntry(ss1->entry,ss1->row,true);
        } else if(ce1==NM_COPY_NEED_DELETE) {
          if(ss1->row>=0) {
            tss=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            tss->row=ss1->row;
            tss->entry=ss1->entry;
            copy_lvc_list->addElement(tss);
          }
        }
      } else {
        if((tcod->error_counter==0)&&(cancel==false)) {
          // only deselect when dir totally copied
          if(ss1->row>=0) {
            lv->setSelect( ss1->row, false );
            lv->showRow(ss1->row);
          }
          ss1->entry->select=false;
          ce->calc();
          showCacheState();
        }
      }
      if(nodatabase==true) delete tcod;
    } else {
      ce1=copyfile(ss1->entry,copyorder,true,copyorder->destdir);
      if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
        // success
        if(copyorder->move==true) {
          if(ce1==NM_COPY_OK) {
            if(ss1->row>=0) lv->showRow(ss1->row);
            removeEntry(ss1->entry,ss1->row,true);
          } else {
            if(ss1->row>=0) {
              tss=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              tss->row=ss1->row;
              tss->entry=ss1->entry;
              copy_lvc_list->addElement(tss);
            }
          }
        } else {
          if(ss1->row>=0) {
            lv->setSelect(ss1->row,false);
            lv->showRow(ss1->row);
          }
          ss1->entry->select=false;
          ce->calc();
          showCacheState();
        }
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
    }
    if(cowin!=NULL) {
      if(cowin->redraw()!=0) cancel=true;
    }
    ss1=(NM_specialsource*)copylist->getNextElement(id);
  }
  copylist->closeEnum(id);
  
  // now delete remaining files/dirs when moving
  if(copyorder->move==true) {
    if(cowin!=NULL) cowin->setmessage("",1);
    feid=copy_deletefe_list->initEnum();
    lvcid=copy_lvc_list->initEnum();
    
    tfe=(FileEntry*)copy_deletefe_list->getFirstElement(feid);
    tss=(NM_specialsource*)copy_lvc_list->getFirstElement(lvcid);
    while((tfe!=NULL)&&(cancel==false)) {
      if(tfe->isDir()==true) {
        if(cowin!=NULL) {
          tstr=(char*)_allocsafe(strlen(catalog.getLocale(141))+strlen(tfe->fullname)+1);
          sprintf(tstr,catalog.getLocale(141),tfe->fullname);
          cowin->setmessage(tstr,0);
          _freesafe(tstr);
        }
      }
      // remove tfe
      if((tfe->isDir()==true)&&(tfe->isLink==false)) erg=rmdir(tfe->fullname);
      else erg=remove(tfe->fullname);
      if(erg==0) {
        // success
        if(tss!=NULL) {
          if(tfe==tss->entry) {
            lv->showRow(tss->row);
            removeEntry(tfe,tss->row,true);
            tss=(NM_specialsource*)copy_lvc_list->getNextElement(lvcid);
          }
        }
      } else {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(tfe->fullname)+1);
        sprintf(textstr,catalog.getLocale(291),tfe->fullname);
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                  catalog.getLocale(8));
        erg=req->request(catalog.getLocale(347),textstr,buttonstr);
        _freesafe(buttonstr);
        _freesafe(textstr);
        if(erg==1) cancel=true;
      }
      tfe=(FileEntry*)copy_deletefe_list->getNextElement(feid);
    }
    
    copy_deletefe_list->closeEnum(feid);
    copy_lvc_list->closeEnum(lvcid);
  }
  
  if(cowin!=NULL) cowin->close();
  
  ss1=(NM_specialsource*)copylist->getFirstElement();
  while(ss1!=NULL) {
    if(ss1->cod!=NULL) delete ss1->cod;
    _freesafe(ss1);
    ss1=(NM_specialsource*)copylist->getNextElement();
  }
  delete copylist;
  delete copy_deletefe_list;

  lvcid=copy_lvc_list->initEnum();
  tss=(NM_specialsource*)copy_lvc_list->getFirstElement(lvcid);
  while(tss!=NULL) {
    _freesafe(tss);
    tss=(NM_specialsource*)copy_lvc_list->getNextElement(lvcid);
  } 
  copy_lvc_list->closeEnum(lvcid);

  delete copy_lvc_list;
  _freesafe(buf);
  if(copyorder->move==true) {
    // now rebuilt the "pos in filelist" <-> "fe->nr" <-> "lvc->data" relation
    oldx=lv->getXOffset();
    oldy=lv->getYOffset();
    removeEntry(NULL,-1,false);
    ce->restartcheck();
    buildLister();
    lv->setXOffset(oldx);
    lv->setYOffset(oldy);
    lv->redraw();
  }
  if(updatenm!=NULL) updatenm->update(false);
  aguix->Flush();
}

NormalMode::nm_newname_t NormalMode::getNewName(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool requestDir,bool acceptrename)
{
  if(requestDir==true)
    return getNewName4Dir(oldfe,copyorder,dest,newname_return,acceptrename);
  else
    return getNewName4File(oldfe,copyorder,dest,newname_return,acceptrename);
}

NormalMode::nm_newname_t NormalMode::getNewName4Dir(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool acceptrename)
{
  char *oldname=oldfe->name;
  char *newname=dupstring(oldname);
  bool nameok=false,skip=false,cancel=false;
  char *return_str=NULL,*buttonstr,*textstr,*newdest;
  int erg;
  Datei::d_fe_t se,lse;
  int round=1;
  nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
  bool trydel,do_rename,strongdirs;

  do_rename=false;
  if(copyorder->do_rename==false) do_rename=false;
  else if(acceptrename==true) do_rename=true;
  
  strongdirs=false;

  do {
    if(!((round==1)&&(do_rename==false))) {
      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                  strlen(catalog.getLocale(225))+1+
                                  strlen(catalog.getLocale(8))+1);
      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                   catalog.getLocale(225),
                                   catalog.getLocale(8));
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
      sprintf(textstr,catalog.getLocale(148),oldname);
      erg=req->string_request(catalog.getLocale(149),textstr,newname,buttonstr,&return_str);
      _freesafe(buttonstr);
      _freesafe(textstr);
      if(return_str!=NULL) {
        _freesafe(newname);
        newname=return_str;
      }
      if(erg==1) {
        skip=true;
        returnvalue=NM_NEWNAME_SKIP;
        break;
      } else if(erg==2) {
        cancel=true;
        returnvalue=NM_NEWNAME_CANCEL;
        break;
      }
    }
    newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
    strcpy(newdest,dest);
    if(strlen(dest)>1) strcat(newdest,"/");
    strcat(newdest,newname);
    se=Datei::fileExistsExt(newdest);
    lse=Datei::lfileExistsExt(newdest);

    switch(lse) {
      case Datei::D_FE_DIR:
        if(strongdirs==true) {
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(190))+strlen(newname)+1);
          sprintf(textstr,catalog.getLocale(190),newname);

          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(273))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                          catalog.getLocale(273),
                                          catalog.getLocale(225),
                                          catalog.getLocale(8));

          erg=req->request(catalog.getLocale(123),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==1) {
            nameok=true;
            returnvalue=NM_NEWNAME_USE;
          } else if(erg==2) {
            skip=true;
            returnvalue=NM_NEWNAME_SKIP;
          } else if(erg==3) {
            cancel=true;
            returnvalue=NM_NEWNAME_CANCEL;
          }
        } else {
          // just use the dir
          nameok=true;
          returnvalue=NM_NEWNAME_USE;
        }
        break;
      case Datei::D_FE_LINK:
      case Datei::D_FE_FILE:
        if((lse==Datei::D_FE_LINK)&&(se==Datei::D_FE_DIR)) {
          // Symlink with dir as destination 
          if(strongdirs==true) {
//            textstr="There is already a symlink called %s, which points to a dir|You can use this link or delete it to|create a real directory";
//            buttonstr="Enter new name|Use this dir|Delete link|Skip|Cancel";

            textstr=(char*)_allocsafe(strlen(catalog.getLocale(275))+strlen(newname)+1);
            sprintf(textstr,catalog.getLocale(275),newname);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(273))+1+
                                        strlen(catalog.getLocale(274))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s|%s|%s",catalog.getLocale(230),
                                            catalog.getLocale(273),
                                            catalog.getLocale(274),
                                            catalog.getLocale(225),
                                            catalog.getLocale(8));

            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            switch(erg) {
              case 1:
                nameok=true;
                returnvalue=NM_NEWNAME_USE;
                break;
              case 2:
                // try to remove the link
                // if success set nameok to true
                // in case of failure show a request and repeat the whole
                if(remove(newdest)==0) {
                  nameok=true;
                  returnvalue=NM_NEWNAME_OK;
                } else {
//                  textstr="Failed to remove this file|Please enter new name!";
//                  buttonstr="Ok|Skip|Cancel";

                  textstr=catalog.getLocale(276);
                  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                              strlen(catalog.getLocale(225))+1+
                                              strlen(catalog.getLocale(8))+1);
                  sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                               catalog.getLocale(225),
                                               catalog.getLocale(8));

                  erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                  _freesafe(buttonstr);
                  if(erg==1) {
                    skip=true;
                    returnvalue=NM_NEWNAME_SKIP;
                  } else if(erg==2) {
                    cancel=true;
                    returnvalue=NM_NEWNAME_CANCEL;
                  }
                }
                break;
              case 3:
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
              case 4:
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
          } else {
            // just use the link to the dir
            nameok=true;
            returnvalue=NM_NEWNAME_USE;
          }
        } else {
          trydel=false;
          if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
            trydel=true;
          } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
            // skip
            skip=true;
            returnvalue=NM_NEWNAME_SKIP;
          } else {
            if(lse==Datei::D_FE_LINK) {
//              textstr="There is already a link named %s";
//              buttonstr="Enter new name|Delete link|Skip|Cancel";

              textstr=(char*)_allocsafe(strlen(catalog.getLocale(278))+strlen(newname)+1);
              sprintf(textstr,catalog.getLocale(278),newname);
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                          strlen(catalog.getLocale(274))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                              catalog.getLocale(274),
                                              catalog.getLocale(225),
                                              catalog.getLocale(8));
            } else {
//              textstr="There is already a file named %s";
//              buttonstr="Enter new name|Delete file|Skip|Cancel";

              textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
              sprintf(textstr,catalog.getLocale(279),newname);
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                          strlen(catalog.getLocale(277))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                              catalog.getLocale(277),
                                              catalog.getLocale(225),
                                              catalog.getLocale(8));
            }
            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            switch(erg) {
              case 1:
                trydel=true;
                break;
              case 2:
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
                break;
              case 3:
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
                break;
            }
          }
          if(trydel==true) {
            if(remove(newdest)==0) {
              nameok=true;
              returnvalue=NM_NEWNAME_OK;
            } else {
//              textstr="Failed to remove this file|Please enter new name!";
//              buttonstr="Ok|Skip|Cancel";

              textstr=catalog.getLocale(276);
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                          strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                           catalog.getLocale(225),
                                           catalog.getLocale(8));

              erg=req->request(catalog.getLocale(347),textstr,buttonstr);
              _freesafe(buttonstr);
              if(erg==1) {
                skip=true;
                returnvalue=NM_NEWNAME_SKIP;
              } else if(erg==2) {
                cancel=true;
                returnvalue=NM_NEWNAME_CANCEL;
              }
            }
          }
        }
        break;
      default:
        nameok=true;
        returnvalue=NM_NEWNAME_OK;
        break;
    }
    _freesafe(newdest);
    round++;
  } while((nameok==false)&&(skip==false)&&(cancel==false));
  if((skip==true)||(cancel==true)) {
    _freesafe(newname);
    *newname_return=NULL;
  } else {
    *newname_return=newname;
  }
  return returnvalue;
}

NormalMode::nm_newname_t NormalMode::getNewName4File(const FileEntry *oldfe,struct NM_copyorder *copyorder,char *dest,char **newname_return,bool acceptrename)
{
  char *oldname=oldfe->name;
  char *newname=dupstring(oldname);
  bool nameok=false,skip=false,cancel=false;
  char *return_str=NULL,*buttonstr,*textstr,*newdest;
  int erg;
  Datei::d_fe_t se,lse;
  int round=1;
  nm_newname_t returnvalue=NM_NEWNAME_CANCEL;
  bool do_rename,trydel;
  char *newsize,*oldsize,*extrastr,*tstr,*newtime,*oldtime;
  int newsizelen,oldsizelen,maxsizelen;
  
  do_rename=false;
  if(copyorder->do_rename==false) do_rename=false;
  else if(acceptrename==true) do_rename=true;

  do {
    if(!((round==1)&&(do_rename==false))) {
//      buttonstr="Ok|Skip|Cancel";
//      textstr="Enter new name for ...";
      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                  strlen(catalog.getLocale(225))+1+
                                  strlen(catalog.getLocale(8))+1);
      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                   catalog.getLocale(225),
                                   catalog.getLocale(8));
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(oldname)+1);
      sprintf(textstr,catalog.getLocale(148),oldname);
      erg=req->string_request(catalog.getLocale(149),textstr,newname,buttonstr,&return_str);
      _freesafe(buttonstr);
      _freesafe(textstr);
      if(return_str!=NULL) {
        _freesafe(newname);
        newname=return_str;
      }
      if(erg==1) {
        skip=true;
        returnvalue=NM_NEWNAME_SKIP;
        break;
      } else if(erg==2) {
        cancel=true;
        returnvalue=NM_NEWNAME_CANCEL;
        break;
      }
    }
    newdest=(char*)_allocsafe(strlen(dest)+1+strlen(newname)+1);
    strcpy(newdest,dest);
    if(strlen(dest)>1) strcat(newdest,"/");
    strcat(newdest,newname);
    se=Datei::fileExistsExt(newdest);
    lse=Datei::lfileExistsExt(newdest);

    switch(lse) {
      case Datei::D_FE_DIR:
//        textstr="there is already a dir called %s";
//        buttonstr="Enter new name|Skip|Cancel";

        textstr=(char*)_allocsafe(strlen(catalog.getLocale(190))+strlen(newname)+1);
        sprintf(textstr,catalog.getLocale(190),newname);

        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));

        erg=req->request(catalog.getLocale(123),textstr,buttonstr);
        _freesafe(buttonstr);
        _freesafe(textstr);
        if(erg==1) {
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else if(erg==2) {
          cancel=true;
          returnvalue=NM_NEWNAME_CANCEL;
        }
        break;
      case Datei::D_FE_LINK:
        trydel=false;
        if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
          trydel=true;
        } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
          // skip
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else {
//          textstr="there is already a link named %s";
//          buttonstr="Enter new name|Delete link|Skip|Cancel";

          textstr=(char*)_allocsafe(strlen(catalog.getLocale(278))+strlen(newname)+1);
          sprintf(textstr,catalog.getLocale(278),newname);
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(274))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s",catalog.getLocale(230),
                                          catalog.getLocale(274),
                                          catalog.getLocale(225),
                                          catalog.getLocale(8));

          erg=req->request(catalog.getLocale(123),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe(textstr);
          switch(erg) {
            case 1:
              trydel=true;
              break;
            case 2:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              break;
            case 3:
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
              break;
          }
        }
        if(trydel==true) {
          if(remove(newdest)==0) {
            nameok=true;
            returnvalue=NM_NEWNAME_OK;
          } else {
//            textstr="Failed to remove this file|Please enter new name!";
//            buttonstr="Ok|Skip|Cancel";

            textstr=catalog.getLocale(276);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));

            erg=req->request(catalog.getLocale(347),textstr,buttonstr);
            _freesafe(buttonstr);
            if(erg==1) {
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
            } else if(erg==2) {
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
            }
          }
        }
        break;
      case Datei::D_FE_FILE:
        if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_ALWAYS)) {
          nameok=true;
          returnvalue=NM_NEWNAME_OVERWRITE;
        } else if((do_rename==false)&&(copyorder->overwrite==copyorder->NM_OVERWRITE_NEVER)) {
          skip=true;
          returnvalue=NM_NEWNAME_SKIP;
        } else {
          FileEntry *newfe=new FileEntry();
          if(newfe->name!=NULL) _freesafe(newfe->name);
          if(newfe->fullname!=NULL) _freesafe(newfe->fullname);
          newfe->name=dupstring(newname);
          newfe->fullname=dupstring(newdest);
          newfe->readInfos();
          
          newsizelen=LongSize(newfe->size);
          oldsizelen=LongSize(oldfe->size);
          maxsizelen=(newsizelen>oldsizelen)?newsizelen:oldsizelen;
          newsize=(char*)_allocsafe(maxsizelen+1);
          oldsize=(char*)_allocsafe(maxsizelen+1);

          memset(newsize,' ',maxsizelen);
          memset(oldsize,' ',maxsizelen);
          
          MakeLong2NiceStr(newsize+maxsizelen-newsizelen,newfe->size);
          MakeLong2NiceStr(oldsize+maxsizelen-oldsizelen,oldfe->size);
          
          newsize[maxsizelen]='\0';
          oldsize[maxsizelen]='\0';
          
          newtime=dupstring(ctime(&(newfe->lastmod)));
          oldtime=dupstring(ctime(&(oldfe->lastmod)));
          newtime[strlen(newtime)-1]='\0';  // remove return
          oldtime[strlen(oldtime)-1]='\0';  // remove return
          
          extrastr=(char*)_allocsafe(strlen(catalog.getLocale(137))+strlen(newsize)+strlen(oldsize)+
                                     strlen(newtime)+strlen(oldtime)+1);
          sprintf(extrastr,catalog.getLocale(137),newsize,oldsize,newtime,oldtime);
          
          _freesafe(newsize);
          _freesafe(oldsize);
          _freesafe(newtime);
          _freesafe(oldtime);
          delete newfe;
//          textstr="There is already a file named %s";
//          buttonstr="Enter new name|Overwrite|Overwrite all|Overwrite none|Skip|Cancel";

          textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
          sprintf(textstr,catalog.getLocale(279),newname);

          tstr=catstring(textstr,"|");
          _freesafe(textstr);
          textstr=catstring(tstr,extrastr);
          _freesafe(tstr);
          _freesafe(extrastr);

          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                      strlen(catalog.getLocale(280))+1+
                                      strlen(catalog.getLocale(135))+1+
                                      strlen(catalog.getLocale(136))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s|%s|%s|%s",catalog.getLocale(230),
                                                catalog.getLocale(280),
                                                catalog.getLocale(135),
                                                catalog.getLocale(136),
                                                catalog.getLocale(225),
                                                catalog.getLocale(8));

          erg=req->request(catalog.getLocale(123),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe(textstr);
          switch(erg) {
            case 1:
              nameok=true;
              returnvalue=NM_NEWNAME_OVERWRITE;
              break;
            case 2:
              nameok=true;
              returnvalue=NM_NEWNAME_OVERWRITE;
              copyorder->overwrite=copyorder->NM_OVERWRITE_ALWAYS;
              break;
            case 3:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              copyorder->overwrite=copyorder->NM_OVERWRITE_NEVER;
              break;
            case 4:
              skip=true;
              returnvalue=NM_NEWNAME_SKIP;
              break;
            case 5:
              cancel=true;
              returnvalue=NM_NEWNAME_CANCEL;
              break;
          }
        }
        break;
      default:
        nameok=true;
        returnvalue=NM_NEWNAME_OK;
        break;
    }
    _freesafe(newdest);
    round++;
  } while((nameok==false)&&(skip==false)&&(cancel==false));
  if((skip==true)||(cancel==true)) {
    _freesafe(newname);
    *newname_return=NULL;
  } else {
    *newname_return=newname;
  }
  return returnvalue;
}

NormalMode::nm_copy_t
NormalMode::copydir(const FileEntry *fe,NM_CopyOp_Dir *cod,struct NM_copyorder *copyorder,bool acceptrename,char *destdir)
{
  ArrayList *filelist;
  NM_CopyOp_Dir *cod1=NULL,*tcod;
  bool nameok,createdir;
  nm_newname_t e1;
  char *destsubdir;
  Datei::d_fe_t de1;
  char *newname=NULL;
  int id1,id2;
  FileEntry *subfe;
  bool isdir,skip,cancel;
  char *buttonstr,*textstr;
  int erg;
  nm_copy_t ce1;
  CopyOpWin *cowin=copyorder->cowin;
  bool nodatabase;
  bool docopy,failedmove=false;
  struct utimbuf utb;

  if(copyorder->copymode==copyorder->NM_COPYMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;

  skip=cancel=false;
  e1=getNewName(fe,copyorder,destdir,&newname,true,acceptrename);
  if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_USE)) {
    // build new name
    destsubdir=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
    strcpy(destsubdir,destdir);
    if(strlen(destsubdir)>1) strcat(destsubdir,"/");
    strcat(destsubdir,newname);

    nameok=false;
    createdir=true;
    if(e1==NM_NEWNAME_USE) {
      // there is already such dir (or a symlink to a dir) -> use it
      nameok=true;
      createdir=false;
    } else if(e1==NM_NEWNAME_OK) {
      // check for fileexists (should not)
      // if nethertheless then skip this dir because the rename-func wasn't able to remove it
      de1=Datei::lfileExistsExt(destsubdir);
      if(de1==Datei::D_FE_NOFILE) {
        // everything is ok
        nameok=true;
      } else if(de1==Datei::D_FE_DIR) {
        // dest is a dir
        // should not happend but ok
        nameok=true;
        createdir=false;
        e1=NM_NEWNAME_USE; // for moving it's important to set this, so it doesn't try to
                           // move
      }
    }
    if(nameok==true) {
      docopy=true;
      failedmove=false;
      if(copyorder->move==true) {
        if(e1==NM_NEWNAME_USE) {
          // because of using the destination dir
          // we need to do as failed rename
          failedmove=true;
        } else {
          if(rename(fe->fullname,destsubdir)==0) {
            // success
            docopy=false;
          } else {
            // failure
            failedmove=true;
          }
        }
      }
      
      if(docopy==true) {
        if(cowin!=NULL) cowin->newfile(fe->name,destsubdir);
        if(createdir==true) {
          if(mkdir(destsubdir,0755)!=0) {
            // failed to create dir -> skip
//            textstr="Failed to create dir|I will skip this dir!";
//            buttonstr="Ok|Cancel";

            if(cowin!=NULL) cowin->stoptimer();
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(126))+strlen(destsubdir)+1);
            sprintf(textstr,catalog.getLocale(126),destsubdir);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                         catalog.getLocale(8));
            erg=req->request(catalog.getLocale(347),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) cancel=true;
            nameok=false;
            if(cowin!=NULL) cowin->conttimer();
          }
        }
        if(nameok==true) {
          // first copy all subdirs
          // then copy all files 
          if(nodatabase==false) {
            id1=cod->subdirs->initEnum();
            cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
            while(cod1!=NULL) {
              ce1=copydir(cod1->fileentry,cod1,copyorder,false,destsubdir);
              if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                // success
                cod->error_counter+=cod1->error_counter;
              } else {
                cod->error_counter+=cod1->error_counter+1;
                if(ce1==NM_COPY_CANCEL) {
                  cancel=true;
                  break;
                }
              }
              cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
            }
            cod->subdirs->closeEnum(id1);
          }

          // next only if read this dir correctly
          if((cod->ok==true)&&(cancel==false)) {
            filelist=cod->verz->getFiles();
            id2=filelist->initEnum();
            subfe=(FileEntry*)filelist->getFirstElement(id2);
            while(subfe!=NULL) {
              if(strcmp(subfe->name,"..")!=0) {
                isdir=false;
                if(subfe->isDir()==true) {
                  if(subfe->isLink==false) isdir=true;
                  else if(copyorder->follow_symlinks==true) isdir=true;
                }
                if(isdir==false) {
                  ce1=copyfile(subfe,copyorder,false,destsubdir);
                  if((ce1==NM_COPY_OK)||(ce1==NM_COPY_NEED_DELETE)) {
                    // success
                  } else {
                    cod->error_counter++;
                    if(ce1==NM_COPY_CANCEL) {
                      cancel=true;
                      break;
                    }
                  }
                } else {
                  if(nodatabase==true) {
                    // no database so copy the dir
                    tcod=new NM_CopyOp_Dir(subfe);
                    ce1=copydir(subfe,tcod,copyorder,false,destsubdir);
                    if(ce1==NM_COPY_OK) {
                      // success
                      cod->error_counter+=tcod->error_counter;
                    } else {
                      cod->error_counter+=tcod->error_counter+1;
                      if(ce1==NM_COPY_CANCEL) {
                        cancel=true;
                      }
                    }
                    delete tcod;
                    if(cancel==true) break;
                  }
                }
              }
              subfe=(FileEntry*)filelist->getNextElement(id2);
            }
            filelist->closeEnum(id2);
          }
          // finally change the permissions of the dir
          if((createdir==true)&&(copyorder->preserve_attr==true)) {
            chmod(destsubdir,(fe->isLink==true)?fe->dmode:fe->mode);
            chown(destsubdir,(fe->isLink==true)?fe->duserid:fe->userid,
                             (fe->isLink==true)?fe->dgroupid:fe->groupid);
            utb.actime=(fe->isLink==true)?fe->dlastaccess:fe->lastaccess;
            utb.modtime=(fe->isLink==true)?fe->dlastmod:fe->lastmod;
            utime(destsubdir,&utb);
          }
        } else skip=true;
      }
    } else skip=true;
    _freesafe(destsubdir);
  } else {
    switch(e1) {
      case NM_NEWNAME_CANCEL:
        cancel=true;
      default:
        skip=true;
        break;
    }
  }
  if((skip==true)&&(cowin!=NULL)) {
    // dir skiped so dec CopyOpWin counter by the files/dirs skiped
    cowin->dec_file_counter(cod->files);
    cowin->dec_dir_counter(cod->dirs);
    cowin->dec_byte_counter(cod->bytes);
  }
  if(cowin!=NULL) {
    cowin->dir_finished();
    if(cowin->redraw()!=0) cancel=true;
  }
  if(newname!=NULL) _freesafe(newname);
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) {
    cod->error_counter++;
    return NM_COPY_SKIP;
  }
  if(copyorder->move==true) {
    if((cod->error_counter==0)&&(failedmove==true)) {
      // put fe in list
      copy_deletefe_list->addElement((FileEntry*)fe);
      return NM_COPY_NEED_DELETE;
    }
  }
  return NM_COPY_OK;
}

NormalMode::nm_copy_t
NormalMode::copyfile(const FileEntry *fe,struct NM_copyorder *copyorder,bool acceptrename,char *destdir)
{
  bool nameok;
  nm_newname_t e1;
  char *newfullname;
  Datei::d_fe_t de1;
  char *newname=NULL;
  int fdw,fdr;
  bool useintern,skip,cancel;
  char *textstr=NULL,*buttonstr;
  int erg;
  ssize_t readbytes,writebytes;
  char *exestr,*cpprog;
  CopyOpWin *cowin=copyorder->cowin;
  bool docopy,failedmove=false,opok=false;
  struct utimbuf utb;

  skip=cancel=false;
  e1=getNewName(fe,copyorder,destdir,&newname,false,acceptrename);

  if((e1==NM_NEWNAME_OK)||(e1==NM_NEWNAME_OVERWRITE)) {
    // build new name
    newfullname=(char*)_allocsafe(strlen(destdir)+1+strlen(newname)+1);
    strcpy(newfullname,destdir);
    if(strlen(newfullname)>1) strcat(newfullname,"/");
    strcat(newfullname,newname);

    nameok=false;
    skip=false;
    if(e1==NM_NEWNAME_OVERWRITE) {
      nameok=true;
    } else if(e1==NM_NEWNAME_OK) {
      // check for fileexists (should not)
      de1=Datei::lfileExistsExt(newfullname);
      if(de1==Datei::D_FE_NOFILE) {
        // everything is ok
        nameok=true;
      } else if(de1==Datei::D_FE_FILE) {
        // dest is a dir
        // should not happend but ok try to overwrite this
        nameok=true;
      }
    }
    if(nameok==true) {
      if(strcmp(fe->fullname,newfullname)!=0) {
        // wenn move true, dann rename probieren
        //        klappt rename nicht, muss flag gesetzt werden und normal weitergemacht
        //        werden und bei korrekten kopieren FE in liste gepackt werden und NEED_DELETE
        //        zurueckgeliefert werden
        //      also: 1.flag docopy am Anfang auf true setzen
        //            2.bei move==true rename probieren
        //            2.1.bei Erfolg docopy auf false, Rueckgabewert ist automatisch OK
        //            2.2.bei Misslingen failedmove auf true setzen
        //            3.bei docopy==true normal kopieren
        //            3.1.bei Misslingen kommt halt skip zurueck
        //            3.2.Bei Erfolg und failedmove==true muss FE in liste gepackt werden
        //                und NEED_DELETE zurueckgeliefert werden
        docopy=true;
        failedmove=false;
        if(copyorder->move==true) {
          if(rename(fe->fullname,newfullname)==0) {
            // success
            docopy=false;
          } else {
            // failure
            failedmove=true;
          }
        }
      
        if(docopy==true) {
          // first try to open the file
          // in case of failure give a requester with choose "Delete"
          // then try to delete it and retry to open
          // if all fails, skip
          useintern=false;
          if(S_ISREG(fe->mode)) useintern=true;
          else if((fe->isLink==true)&&(copyorder->follow_symlinks==true)) useintern=true;

          if(cowin!=NULL) cowin->newfile(fe->name,newfullname);

          if(useintern==true) {
            fdr=open(fe->fullname,O_RDONLY);
            if(fdr==-1) {
              // can't open inputfile
              if(cowin!=NULL) cowin->stoptimer();
              textstr=(char*)_allocsafe(strlen(catalog.getLocale(281))+strlen(fe->fullname)+1);
              sprintf(textstr,catalog.getLocale(281),fe->fullname);
              buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                          strlen(catalog.getLocale(8))+1);
              sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                        catalog.getLocale(8));

              erg=req->request(catalog.getLocale(347),textstr,buttonstr);
              _freesafe(buttonstr);
              _freesafe(textstr);
              if(erg==1) cancel=true;
              else skip=true;
              if(cowin!=NULL) cowin->conttimer();
            } else {
              if(cowin!=NULL) {
                if(fe->isLink==true)
                  cowin->set_bytes_to_copy_actfile(fe->dsize);
                else
                  cowin->set_bytes_to_copy_actfile(fe->size);
              }

              for(int i=0;i<2;i++) {
                fdw=open(newfullname,O_CREAT|O_TRUNC|O_WRONLY,(fe->isLink==true)?fe->dmode:fe->mode);
                if(fdw==-1) {
                  if(errno==ENOSPC) {
                    textstr=(char*)_allocsafe(strlen(catalog.getLocale(197))+strlen(destdir)+1);
                    sprintf(textstr,catalog.getLocale(197),destdir);
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                              catalog.getLocale(8));

                    erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    if(erg==1) cancel=true;
                    else skip=true;
                  } else if(errno==EACCES) {
                    if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
                      // error
                      // give request
//                      textstr="Can't overwrite the file|Shall I try to delete it?";
//                      buttonstr="Delete file|Skip|Cancel";

                      if(cowin!=NULL) cowin->stoptimer();
                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(282))+strlen(newfullname)+1);
                      sprintf(textstr,catalog.getLocale(282),newfullname);
                      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(277))+1+
                                                  strlen(catalog.getLocale(225))+1+
                                                  strlen(catalog.getLocale(8))+1);
                      sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(277),
                                                   catalog.getLocale(225),
                                                   catalog.getLocale(8));

                      erg=req->request(catalog.getLocale(123),textstr,buttonstr);
                      _freesafe(buttonstr);
                      _freesafe(textstr);
                      if(erg==1) skip=true;
                      else if(erg==2) cancel=true;
                      else {
                        if(remove(newfullname)!=0) {
//                          textstr="Failed to remove this file|I will skip this file!";
//                          buttonstr="Ok|Cancel";

                          textstr=(char*)_allocsafe(strlen(catalog.getLocale(138))+strlen(newfullname)+1);
                          sprintf(textstr,catalog.getLocale(138),newfullname);
                          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                      strlen(catalog.getLocale(8))+1);
                          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                                    catalog.getLocale(8));
                          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                          _freesafe(buttonstr);
                          _freesafe(textstr);
                          if(erg==1) cancel=true;
                          else skip = true;
                        }
                      }
                    } else {
                      // "Can't open dest file"
                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(198))+strlen(newfullname)+1);
                      sprintf(textstr,catalog.getLocale(198),newfullname);
                      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                  strlen(catalog.getLocale(8))+1);
                      sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                                catalog.getLocale(8));

                      erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                      _freesafe(buttonstr);
                      _freesafe(textstr);
                      if(erg==1) cancel=true;
                      else skip=true;
                    }
                  } else {
                    textstr=(char*)_allocsafe(strlen(catalog.getLocale(198))+strlen(newfullname)+1);
                    sprintf(textstr,catalog.getLocale(198),newfullname);
                    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                                strlen(catalog.getLocale(8))+1);
                    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                              catalog.getLocale(8));

                    erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                    _freesafe(buttonstr);
                    _freesafe(textstr);
                    if(erg==1) cancel=true;
                    else skip=true;
                  }
                  if(cowin!=NULL) cowin->conttimer();
                } else break;
                if((skip==true)||(cancel==true)) break;
              }
              if(fdw!=-1) {
                do {
                  readbytes=read(fdr,buf,BUFSIZE);
                  writebytes=write(fdw,buf,readbytes);
                  if(writebytes==readbytes) {
                    if(cowin!=NULL) {
                      cowin->add_actbytes_copied(writebytes);
                      if(writebytes==BUFSIZE) {
                        if(cowin->redraw()!=0) cancel=true;
                      }
                    }
                  } else {
                    // something gone wrong
                    // let the user choose to cancel or skip this file
                    // Ask to delete the incomplete destfile
                    // Attention: This method also moves files so be sure to
                    //   NOT delete the source!
                    if( ((writebytes>=0)&&(writebytes<readbytes)) ||
                        (errno==ENOSPC) ) {
                      // ENOSPC isn't always reported so assume ENOSPC
                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(197))+strlen(destdir)+1);
                      sprintf(textstr,catalog.getLocale(197),destdir);
                      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                                  strlen(catalog.getLocale(8))+1);
                      sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                                catalog.getLocale(8));

                      erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                      _freesafe(buttonstr);
                      _freesafe(textstr);
                      if(erg==1) cancel=true;
                      else skip = true;
                    } else {
                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(359))+strlen(newfullname)+1);
                      sprintf(textstr,catalog.getLocale(359),newfullname);
                      buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(225))+1+
                                                  strlen(catalog.getLocale(8))+1);
                      sprintf(buttonstr,"%s|%s",catalog.getLocale(225),
                                                catalog.getLocale(8));

                      erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                      _freesafe(buttonstr);
                      _freesafe(textstr);
                      if(erg==1) cancel=true;
                      else skip = true;
                    }
                  }
                } while ( ( writebytes == BUFSIZE ) &&
                          ( cancel == false ) &&
                          ( skip == false ) );
                close(fdw);

                if ( ( cancel == true ) || ( skip == true ) ) {
                  // error while copying!
                  // ask to remove the incomplete destination file

                  if(cowin!=NULL) cowin->stoptimer();

                  textstr = (char*)_allocsafe( strlen( catalog.getLocale( 393 ) ) + strlen( newfullname ) + 1 );
                  sprintf( textstr, catalog.getLocale( 393 ), newfullname );
                  buttonstr = (char*)_allocsafe( strlen( catalog.getLocale( 394 ) ) + 1 +
                                                 strlen( catalog.getLocale( 395 ) ) + 1 );
                  sprintf( buttonstr, "%s|%s", catalog.getLocale( 394 ),
                                               catalog.getLocale( 395 ) );

                  erg = req->request( catalog.getLocale( 123 ), textstr, buttonstr );
                  _freesafe( buttonstr );
                  _freesafe( textstr );
                  if(erg==1) {
                    if(remove(newfullname)!=0) {
                      // textstr="Failed to remove the destination file|This file is probably incomplete!";

                      textstr=(char*)_allocsafe(strlen(catalog.getLocale(139))+strlen(newfullname)+1);
                      sprintf(textstr,catalog.getLocale(139),newfullname);
                      buttonstr=catalog.getLocale(11);
                      erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                      _freesafe(textstr);
                    }
                  }
                  if(cowin!=NULL) cowin->conttimer();
                } else {
                  // copy complete
                  // try to change the mode because open modify it with umask
                  // it doesn't matter if this fails 
                  if(copyorder->preserve_attr==true) {
                    chmod(newfullname,(fe->isLink==true)?fe->dmode:fe->mode);
                    chown(newfullname,(fe->isLink==true)?fe->duserid:fe->userid,
                                      (fe->isLink==true)?fe->dgroupid:fe->groupid);
                    utb.actime=(fe->isLink==true)?fe->dlastaccess:fe->lastaccess;
                    utb.modtime=(fe->isLink==true)?fe->dlastmod:fe->lastmod;
                    utime(newfullname,&utb);
                  }
                  opok=true;
                }
              }
            }
            if(fdr!=-1) close(fdr);
          } else {
            // use "cp"-prog for copy
            if(e1==NM_NEWNAME_OVERWRITE) {
              if(copyorder->cp_com[1]!=NULL) {
                cpprog=(char*)_allocsafe(strlen(copyorder->cp_com[1])+
                                         32+1);
                sprintf(cpprog,"%s >/dev/null 2>/dev/null",copyorder->cp_com[1]);
              } else {
                cpprog=(char*)_allocsafe(strlen(CP_COMMAND_FORCE)+
                                         32+1);
                sprintf(cpprog,"%s >/dev/null 2>/dev/null",CP_COMMAND_FORCE);
              }
            } else {
              if(copyorder->cp_com[0]!=NULL) {
                cpprog=(char*)_allocsafe(strlen(copyorder->cp_com[0])+
                                         32+1);
                sprintf(cpprog,"%s >/dev/null 2>/dev/null",copyorder->cp_com[0]);
              } else {
                cpprog=(char*)_allocsafe(strlen(CP_COMMAND)+
                                         32+1);
                sprintf(cpprog,"%s >/dev/null 2>/dev/null",CP_COMMAND);
              }
            }
            exestr=(char*)_allocsafe(strlen(cpprog)+
                                     strlen(fe->fullname)+strlen(newfullname)+1);
            sprintf(exestr,cpprog,fe->fullname,newfullname);
//            debugmsg("Execute");
//            printf(" %s\n",exestr);
            erg=system(exestr);
            if(erg!=0) {
//              textstr="Error while executing ...";
              if(cowin!=NULL) cowin->stoptimer();
              textstr=(char*)_allocsafe(strlen(catalog.getLocale(283))+strlen(exestr)+1);
              sprintf(textstr,catalog.getLocale(283),exestr);
              buttonstr=catalog.getLocale(11);
              req->request(catalog.getLocale(347),textstr,buttonstr);
              _freesafe(textstr);
              skip=true;
              if(cowin!=NULL) cowin->conttimer();
            } else opok=true;
            //TODO: muss hier nicht eher add_actbytes_copied aufgerufen werden?
//            if(cowin!=NULL) cowin->set_bytes_to_copy_actfile(fe->size);
            if(cowin!=NULL) cowin->add_actbytes_copied(fe->size);
            _freesafe(exestr);
            _freesafe(cpprog);
          }
        }
      } else {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(113))+strlen(fe->fullname)+1);
        sprintf(textstr,catalog.getLocale(113),fe->fullname);
        buttonstr=catalog.getLocale(11);

        req->request(catalog.getLocale(347),textstr,buttonstr);
        _freesafe(textstr);
      }
    } else skip=true;
    _freesafe(newfullname);
  } else {
    switch(e1) {
      case NM_NEWNAME_CANCEL:
        cancel=true;
      default:
        skip=true;
        break;
    }
  }
  if(cowin!=NULL) {
    cowin->file_finished();
    if(cowin->redraw()!=0) cancel=true;
  }
  if(newname!=NULL) _freesafe(newname);
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) return NM_COPY_SKIP;
  if(copyorder->move==true) {
    if((opok==true)&&(failedmove==true)) {
      // remember: failedmove is true when OS function rename failed
      //           in this case we have to copy the file and delete the source
      //           so add this entry only when copy is complete (opok) AND
      //           rename failed (failedmove)
      // put fe in list
      copy_deletefe_list->addElement((FileEntry*)fe);
      return NM_COPY_NEED_DELETE;
    }
  }
  return NM_COPY_OK;
}

bool
NormalMode::isCorrectCPProg(const char *str)
{
  char *pos,*pos2;
  if(strlen(str)<1) return true;
  pos=strstr(str,"%s");
  if(pos!=NULL) {
    if((unsigned)(pos-str+3)<strlen(str)) {
      pos2=strstr(pos+3,"%s");
      if(pos2!=NULL) return true;
    }
  }
  return false;
}

bool
NormalMode::isCorrectViewProg(const char *str)
{
  char *pos;
  if(strlen(str)<1) return true;
  pos=strstr(str,"%s");
  if(pos!=NULL) return true;
  return false;
}

void
NormalMode::deletef(struct NM_deleteorder *delorder)
{
  List *dellist;
  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2;
  ArrayList *filelist;
  FileEntry *fe;
//  bool found;
  int row;
  int id;
  bool enter;
  NM_CopyOp_Dir *cod1,*tcod;
  unsigned long files,dirs,gf,gd;
  bool skip,cancel=false;
  DeleteOpWin *dowin;
  nm_copy_t ce1;
  bool nodatabase;
  char *buttonstr,*textstr;
  int erg;
  bool empty;
  int oldx,oldy;
  int tnrfiles,tnrdirs;
  
  if(delorder==NULL) return;
  if(getActualDir()==NULL) {
    // no actual dir => no files to copy
    return;
  }

  tnrfiles=ce->files[1];
  tnrdirs=ce->dirs[1];

  if ( ( delorder->source == delorder->NM_ALLENTRIES ) &&
       ( ( tnrfiles + tnrdirs ) < 1 ) ) {
    if ( ce->activefe != NULL ) {
      if ( strcmp( ce->activefe->name, ".." ) != 0 ) {
        if ( ce->activefe->isDir() == true ) tnrdirs++;
        else tnrfiles++;
      }
    }
  }
  
  if ( ( tnrfiles + tnrdirs ) < 1 )
    return;

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();
  
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  textstr=(char*)_allocsafe(strlen(catalog.getLocale(145))+LongSize(tnrfiles)+
                            LongSize(tnrdirs)+1);
  sprintf(textstr,catalog.getLocale(145),tnrfiles,tnrdirs);
  erg=req->request(catalog.getLocale(123),textstr,buttonstr);
  _freesafe(buttonstr);
  _freesafe(textstr);
  if(erg!=0) return;

#ifdef WANT_THREADS
  // clear the reclist so the slace doesn't block us because he is still reading
  // from disk
  ft_rec_list_clear();
#endif
  
  if(delorder->delmode==delorder->NM_DELMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;

  dowin=delorder->dowin;
  if(dowin!=NULL) {
    dowin->open();
    dowin->setmessage(catalog.getLocale(140));
    dowin->redraw();
  }
  dellist=new List();
  switch(delorder->source) {
    case delorder->NM_SPECIAL:
/*OPT:At the moment I doesn't allow specialsources
      ss1=(NM_specialsource*)delorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if(fe==ss1->entry) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=ss1->entry;
          ss2->lvc=ss1->lvc;
          ss2->cod=NULL;
          dellist->addElement(ss2);
        }
        ss1=(NM_specialsource*)delorder->sources->getNextElement();
      }*/
      break;
    case delorder->NM_ONLYACTIVE:
/*OPT:At the moment I doesn't allow onlyactive
      if(ce->activefe!=NULL) {
        lvc=lv->getActiveLVC();
        if(lvc!=NULL) {
          if(lvc->getData()==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->lvc=lvc;
              ss2->cod=NULL;
              dellist->addElement(ss2);
            }
          }
        }
      }*/
      break;
    case delorder->NM_ONLYSELECTED:
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            dellist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            dellist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(dellist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row = lv->getActiveRow();
          if(row>=0) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=ce->activefe;
                ss2->row=row;
                ss2->cod=NULL;
                dellist->addElement(ss2);
              }
            }
          }
        }
      }
      break;
  }
  if(dowin!=NULL) dowin->redraw();

  // create the NM_CopyOp_Dir for each dir in dellist
  files=dirs=gf=gd=0;
  id=dellist->initEnum();
  
  if(nodatabase==false) {
    ss1=(NM_specialsource*)dellist->getFirstElement(id);
    while((ss1!=NULL)&&(cancel==false)) {
      enter=false;
      if(ss1->entry->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if(ss1->entry->isLink==false) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(ss1->entry);
        // recursive call
        if(cod1->createSubDirs(delorder,&gf,&gd)!=0) cancel=true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;

        ss1->cod=cod1;
        // this is a dir so inc the counter
        dirs++;
        gd++;
        if(dowin!=NULL) {
          dowin->set_files_to_delete(gf);
          dowin->set_dirs_to_delete(gd);
          if(dowin->redraw()!=0) cancel=true;;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        gf++;
        ss1->cod=NULL;
      }
      ss1=(NM_specialsource*)dellist->getNextElement(id);
    }
  }

  if(dowin!=NULL) {
    dowin->set_files_to_delete(files);
    dowin->set_dirs_to_delete(dirs);
  }
  // now database ready, can start copy-process
  skip=false;
  ss1=(NM_specialsource*)dellist->getFirstElement(id);
  while((ss1!=NULL)&&(cancel==false)) {
    if(nodatabase==false) {
      tcod=ss1->cod;
    } else {
      enter=false;
      tcod=NULL;
      if(ss1->entry->isDir()==true) {
        if(ss1->entry->isLink==false) enter=true;
      }
      if(enter==true) tcod=new NM_CopyOp_Dir(ss1->entry);
    }
    if(tcod!=NULL) {
      // determine if the dir is empty or not
      if(nodatabase==false) {
        if((tcod->files+tcod->dirs)>0) empty=false;
        else empty=true;
      } else {
        empty=false;
        if(tcod->ok==true) {
          if(tcod->verz->getFiles()->size()<2) empty=true;
        }
      }
      skip=false;
      // skip if the user doesn't want to delete non-empty dirs
      if((delorder->dirdel==delorder->NM_DIRDELETE_NONE)&&(empty==false)) skip=true;
      // if the user doesn't select ALL or NONE and the the dir isn't empty show a request
      if((delorder->dirdel==delorder->NM_DIRDELETE_NORMAL)&&(empty==false)) {
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(146))+
                                  strlen(ss1->entry->fullname)+1);
        sprintf(textstr,catalog.getLocale(146),ss1->entry->fullname);
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(287))+1+
                                    strlen(catalog.getLocale(288))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s|%s|%s",catalog.getLocale(11),
                                           catalog.getLocale(287),
                                           catalog.getLocale(288),
                                           catalog.getLocale(225),
                                           catalog.getLocale(8));
        erg=req->request(catalog.getLocale(123),textstr,buttonstr);
        _freesafe(buttonstr);
        _freesafe(textstr);
        switch(erg) {
          case 1:
            delorder->dirdel=delorder->NM_DIRDELETE_ALL;
            break;
          case 2:
            delorder->dirdel=delorder->NM_DIRDELETE_NONE;
            skip=true;
            break;
          case 3:
            skip=true;
            break;
          case 4:
            cancel=true;
            skip=true;
            break;
        }
      }
      if(skip==false) {
        ce1=deldir(ss1->entry,tcod,delorder);
        if(ce1==NM_COPY_OK) {
          // success
        } else if(ce1==NM_COPY_CANCEL) {
          cancel=true;
        }
        if((tcod->error_counter==0)&&(cancel==false)) {
          // successful delete this dir so remove it form the lv
          // first show this lvc
          if(ss1->row>=0) lv->showRow(ss1->row);
          removeEntry(ss1->entry,ss1->row,true);
        }
      }
      if(nodatabase==true) delete tcod;
    } else {
      ce1=delfile(ss1->entry,delorder,false);
      if(ce1==NM_COPY_OK) {
        // success
        if(ss1->row>=0) lv->showRow(ss1->row);
        removeEntry(ss1->entry,ss1->row,true);
      } else if(ce1==NM_COPY_CANCEL) {
        cancel=true;
      }
    }
    if(dowin!=NULL) {
      if(dowin->redraw()!=0) cancel=true;
    }
    ss1=(NM_specialsource*)dellist->getNextElement(id);
  }
  dellist->closeEnum(id);
  if(dowin!=NULL) dowin->close();
  
  ss1=(NM_specialsource*)dellist->getFirstElement();
  while(ss1!=NULL) {
    if(ss1->cod!=NULL) delete ss1->cod;
    _freesafe(ss1);
    ss1=(NM_specialsource*)dellist->getNextElement();
  }
  delete dellist;

  // now rebuilt the "pos in filelist" <-> "fe->nr" <-> "lvc->data" relation
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();
  removeEntry(NULL,-1,false);
  ce->restartcheck();
  buildLister();
  lv->setXOffset(oldx);
  lv->setYOffset(oldy);
  lv->redraw();
  aguix->Flush();
}

void
NormalMode::removeEntry(FileEntry *fe, int row,bool quick)
{
  ArrayList *filelist;
  int pos,tid;
  FileEntry *tfe;
  int trow;
  bool searchcheckfe, searchfirstnullft;
  
  if(ce==NULL) return;
  if(ce->verz==NULL) return;
  filelist=ce->verz->getFiles();
  if(filelist==NULL) return;
  
#ifdef WANT_THREADS
  // we will remove an entry so clear this list
  // so it won't contain wrong pointers
  ft_rec_list_clear();
#endif
  
  if(fe!=NULL) {
    // if it's the active one then move the active up by one
    if(fe==ce->activefe) {
      pos=filelist->getIndex(ce->activefe);
      tid=filelist->initEnum();
      do {
        pos--;
        tfe=(FileEntry*)filelist->getElementAt(tid,pos);
        if(tfe==NULL) break;
      } while(tfe->use==false);
      filelist->closeEnum(tid);
      if(tfe!=NULL) {
        // Voriges Element gefunden
        ce->activefe=tfe;
        trow = row - 1;
        if(trow>=0) {
        //TODO: activerow auf -1 setzen, wenn nicht gefunden?
          lv->setActiveRow( trow);
          lv->showActive();
        }
      }
    }
    if ( fe == ce->checkfe )
      searchcheckfe = true;
    else searchcheckfe = false;
    if ( fe == ce->firstnullft )
      searchfirstnullft = true;
    else searchfirstnullft = false;
    
    // the following destroy the relation between the pos in filelist and lvc->data
    // but at the end I will rebuilt it
    // remove the lvc
    lv->deleteRow(row);
    // remove the FileEntry
    filelist->removeElement(fe);
    ce->subentry(fe);
    delete fe;

    if ( searchcheckfe == true )
      ce->firstnullftcheck();
    if ( searchfirstnullft == true )
      ce->reset_firstnullft();

    // display it
    showCacheState();
  }
  if(quick==false) {
    // now fix the structures
    // now rebuilt the "pos in filelist" <-> "fe->nr" <-> "lvc->data" relation
    trow=0;
    tid=filelist->initEnum();
    tfe=(FileEntry*)filelist->getFirstElement(tid);
    pos=0;
    while(tfe!=NULL) {
      if(tfe->use==true) {
        if(tfe->nr==lv->getData(trow)) {
          lv->setData(trow,pos);
/*          if(tfe->select==true) tlvc->setSelectQ(true); else tlvc->setSelectQ(false);
          if(tfe==ce->activefe) {
             tlvc->setActiveQ(true);
          }*/
        } else {
          // this should never happen
          debugmsg("error1 in removeEntry\n");
        }
        trow++;
      }
      tfe->nr=pos;
      tfe=(FileEntry*)filelist->getNextElement(tid);
      pos++;
    }
    filelist->closeEnum(tid);
  }
}

NormalMode::nm_copy_t
NormalMode::deldir(const FileEntry *fe,NM_CopyOp_Dir *cod,struct NM_deleteorder *delorder)
{
  ArrayList *filelist;
  NM_CopyOp_Dir *cod1=NULL,*tcod;
  int id1,id2;
  FileEntry *subfe;
  bool isdir,skip,cancel;
//  char *buttonstr,*textstr;
//  int erg;
  nm_copy_t ce1;
  DeleteOpWin *dowin=delorder->dowin;
  bool nodatabase,delerror;

  if(delorder->delmode==delorder->NM_DELMODE_NODATABASE) nodatabase=true;
  else nodatabase=false;

  skip=cancel=false;

  if(dowin!=NULL) dowin->setfilename(fe->fullname);
  // first del all subdirs
  // then del all files 
  if(nodatabase==false) {
    id1=cod->subdirs->initEnum();
    cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
    while(cod1!=NULL) {
      ce1=deldir(cod1->fileentry,cod1,delorder);
      if(ce1==NM_COPY_OK) {
        // success
        cod->error_counter+=cod1->error_counter;
      } else {
        cod->error_counter+=cod1->error_counter+1;
        if(ce1==NM_COPY_CANCEL) {
          cancel=true;
          break;
        }
      }
      cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
    }
    cod->subdirs->closeEnum(id1);
  }

  // next only if read this dir correctly
  if((cod->ok==true)&&(cancel==false)) {
    filelist=cod->verz->getFiles();
    id2=filelist->initEnum();
    subfe=(FileEntry*)filelist->getFirstElement(id2);
    while(subfe!=NULL) {
      if(strcmp(subfe->name,"..")!=0) {
        isdir=false;
        if(subfe->isDir()==true) {
          if(subfe->isLink==false) isdir=true;
        }
        if(isdir==false) {
          ce1=delfile(subfe,delorder,false);
          if(ce1==NM_COPY_OK) {
            // success
          } else {
            cod->error_counter++;
            if(ce1==NM_COPY_CANCEL) {
              cancel=true;
              break;
            }
          }
        } else {
          if(nodatabase==true) {
            // no database so copy the dir
            tcod=new NM_CopyOp_Dir(subfe);
            ce1=deldir(subfe,tcod,delorder);
            if(ce1==NM_COPY_OK) {
              // success
              cod->error_counter+=tcod->error_counter;
            } else {
              cod->error_counter+=tcod->error_counter+1;
              if(ce1==NM_COPY_CANCEL) {
                cancel=true;
              }
            }
            delete tcod;
            if(cancel==true) break;
          }
        }
      }
      subfe=(FileEntry*)filelist->getNextElement(id2);
    }
    filelist->closeEnum(id2);
  }
  if((skip==true)&&(dowin!=NULL)) {
    // dir skiped so dec CopyOpWin counter by the files/dirs skiped
    dowin->dec_file_counter(cod->files);
    dowin->dec_dir_counter(cod->dirs);
  }
  delerror=true;
  if((skip==false)&&(cancel==false)) {
    if(cod->error_counter==0) {
      ce1=delfile(fe,delorder,true);
      if(ce1==NM_COPY_OK) {
        // success
        delerror=false;
      } else {
        cod->error_counter++;
        if(ce1==NM_COPY_CANCEL) {
          cancel=true;
        }
      }
    }
  }
  if(delerror==true) {
    char *textstr=(char*)_allocsafe(strlen(catalog.getLocale(289))+strlen(fe->fullname)+1);
    sprintf(textstr,catalog.getLocale(289),fe->fullname);
    char *buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
    int erg=req->request(catalog.getLocale(125),textstr,buttonstr);
    _freesafe(buttonstr);
    _freesafe(textstr);
    if(erg==1) cancel=true;
    else skip=true;
  }
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) {
    cod->error_counter++;
    return NM_COPY_SKIP;
  }
  return NM_COPY_OK;
}

NormalMode::nm_copy_t
NormalMode::delfile(const FileEntry *fe,struct NM_deleteorder *delorder,bool dir)
{
  bool skip,cancel;
  char *textstr,*buttonstr;
  int erg;
  DeleteOpWin *dowin=delorder->dowin;

  skip=cancel=false;

  if(dir==true) erg=rmdir(fe->fullname);
  else erg=remove(fe->fullname);
  if(erg!=0) {
//    textstr="Failed to remove this file!";
//    buttonstr="Ok|Cancel";

    if(dir==true) {
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(147))+strlen(fe->fullname)+1);
      sprintf(textstr,catalog.getLocale(147),fe->fullname);
    } else { 
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(286))+strlen(fe->fullname)+1);
      sprintf(textstr,catalog.getLocale(286),fe->fullname);
    }
    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
    erg=req->request(catalog.getLocale(347),textstr,buttonstr);
    _freesafe(buttonstr);
    _freesafe(textstr);
    if(erg==1) cancel=true;
    else skip=true;
  }
  if(dowin!=NULL) {
    if(dir==true) {
      dowin->dir_finished();
      if(dowin->redraw()!=0) cancel=true;
    } else dowin->file_finished();
  }
  if(cancel==true) return NM_COPY_CANCEL;
  else if(skip==true) return NM_COPY_SKIP;
  return NM_COPY_OK;
}

void
NormalMode::update(bool reset_dirsizes)
{
  update( reset_dirsizes, false );
}

void
NormalMode::update( bool reset_dirsizes, bool keep_filetypes )
{
  if(ce==NULL) return;
  char *dir,*tstr;
  int oldx,oldy,e;
  
  finishsearchmode();
  
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();
  dir=dupstring(ce->verz->getDir());
  for(;;) {
    e=enterDir(dir);
    if(e==0) break;  // read ok, so finish
    if(strcmp(dir,"/")==0) break;  // there is no dir we can display anymore, so finish
    tstr=ParentDir(dir,NULL);
    _freesafe(dir);
    dir=tstr;
  }
  _freesafe(dir);
  if(ce!=NULL) {
    if ( keep_filetypes == true ) {
      ce->firstnullftcheck();
    } else {
      ce->restartcheck();
    }
    if(reset_dirsizes==true) ce->reset_dirsizes();
  }
  buildLister();
  lv->setXOffset(oldx);
  lv->setYOffset(oldy);
  lv->redraw();
}

void
NormalMode::makedir()
{
  if(ce==NULL) return;
  int id;
  ArrayList *filelist;
  FileEntry *tfe;
  int row;
  char *buttonstr,*textstr,*return_str,*destdir;
  int erg;
  
  finishsearchmode();
  
  textstr=catalog.getLocale(152);
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg=req->string_request(catalog.getLocaleCom(27),textstr,"",buttonstr,&return_str);
  _freesafe(buttonstr);
  if((erg==0)&&(strlen(return_str)>0)) {
    destdir=(char*)_allocsafe(strlen(ce->verz->getDir())+1+strlen(return_str)+1);
    strcpy(destdir,ce->verz->getDir());
    if(strlen(destdir)>1) strcat(destdir,"/");
    strcat(destdir,return_str);

    if(mkdir(destdir,S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH)!=0) {
      textstr=(char*)_allocsafe(strlen(catalog.getLocale(127))+strlen(destdir)+1);
      sprintf(textstr,catalog.getLocale(127),destdir);
      buttonstr=catalog.getLocale(11);
      erg=req->request(catalog.getLocale(347),textstr,buttonstr);
      _freesafe(textstr);
    } else {
      // update the dir
      update(false);
      // find the new dir
      filelist=ce->verz->getFiles();
      id=filelist->initEnum();
      tfe=(FileEntry*)filelist->getFirstElement(id);
      while(tfe!=NULL) {
        if(strcmp(tfe->name,return_str)==0) {
          // found
          ce->activefe=tfe;
          row = 0;
          while( row < lv->getElements() ) {
            if(lv->getData(row)==tfe->nr) {
              // lvc found
              lv->setActiveRow(row);
              break;
            }
            row++;
          }
          break;
        }
        tfe=(FileEntry*)filelist->getNextElement(id);
      }
      filelist->closeEnum(id);
      lv->showActive();
      lv->redraw();
    }
    _freesafe(destdir);
  }
  _freesafe(return_str);
}

void
NormalMode::externprog(struct NM_externorder *extorder)
{
  List *entries[2];  // 0 for active,1 for inactive side
  List *splist[2];  // for specialsource entries

  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2;
  ArrayList *filelist;
  FileEntry *fe;
  int row;
  bool found;
  ListerMode *lm1;
  Lister *l2;
  NormalMode *nm2;
  NM_extern_fe *efe;
  char *tmpname,*tmpoutput,*exestr,*tstr;
  Datei *fp;
  int lastsize[2];
  bool useint;
  bool cancel=false;
  int i;

  if(extorder==NULL) return;
  if(extorder->com_str==NULL) return;  // nothing to do

  finishsearchmode();
  
  splist[0]=new List();
  if(ce!=NULL) {
    verz=ce->verz;
    if(verz!=NULL) {
      filelist=verz->getFiles();

      switch(extorder->source) {
        case extorder->NM_SPECIAL:
          ss1=(NM_specialsource*)extorder->sources->getFirstElement();
          while(ss1!=NULL) {
            fe=(FileEntry*)filelist->getFirstElement();
            found=false;
            while(fe!=NULL) {
              if(fe==ss1->entry) {
                if(strcmp(fe->name,"..")!=0) {
                  found=true;
                  break;
                }
              }
              fe=(FileEntry*)filelist->getNextElement();
            }
            if(found==true) {
              // valid entry found
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ss1->entry;
              ss2->row=ss1->row;
              ss2->cod=NULL;
              splist[0]->addElement(ss2);
            }
            ss1=(NM_specialsource*)extorder->sources->getNextElement();
          }
          break;
        case extorder->NM_ONLYACTIVE:
          if(ce->activefe!=NULL) {
            row =lv->getActiveRow();
            if(row >=0 ) {
              if(lv->getData(row)==ce->activefe->nr) {
                if(strcmp(ce->activefe->name,"..")!=0) {
                  ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                  ss2->entry=ce->activefe;
                  ss2->row=row;
                  ss2->cod=NULL;
                  splist[0]->addElement(ss2);
                }
              }
            }
          }
          break;
        default:  // all selected entries
          fe=(FileEntry*)filelist->getFirstElement();
          row = 0;
          while(fe!=NULL) {
            if(fe->use==true) {
              if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=fe;
                ss2->row=row;
                ss2->cod=NULL;
                splist[0]->addElement(ss2);
              }
              row++;
            }
            fe=(FileEntry*)filelist->getNextElement();
          }
          if(splist[0]->size()==0) {
            // no selected entries => take the active
            if(ce->activefe!=NULL) {
              row = lv->getActiveRow();
              if(row>=0) {
                if(lv->getData(row)==ce->activefe->nr) {
                  if(strcmp(ce->activefe->name,"..")!=0) {
                    ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                    ss2->entry=ce->activefe;
                    ss2->row=row;
                    ss2->cod=NULL;
                    splist[0]->addElement(ss2);
                  }
                }
              }
            }
          }
          break;
      }
    }
  }
  splist[1]=new List();

  nm2=NULL;
  l2=parentlister->getWorker()->getOtherLister(parentlister);
  lm1=l2->getActiveMode();
  if(lm1!=NULL)
    if(lm1->isType("NormalMode")==true) {
      nm2=(NormalMode*)lm1;
      nm2->getSelFiles(splist[1],NM_GETFILES_SELORACT);
    }
  
  // now we have 2 lists which contains the files for the operation on both sides
  // next is to create list of NM_extern_fe, if necessary recursive
  entries[0]=new List();
  entries[1]=new List();
  createExtList(splist[0],extorder->recursive,entries[0]);
  createExtList(splist[1],extorder->recursive,entries[1]);

  tmpname=Datei::createTMPName();
  tmpoutput=Datei::createTMPName();
  fp=new Datei();
  if(fp->open(tmpname,"w")==0) {
    fp->putString("#!/bin/sh\n");
    if(getActualDir()!=NULL) fp->putStringExt("cd \"%s\"\n",getActualDir());
    lastsize[0]=entries[0]->size();
    lastsize[1]=entries[1]->size();
    for(int loopnr=1;;loopnr++) {
      exestr=parseComStr(extorder->com_str,extorder,EXE_STRING_LEN-strlen(tmpoutput)-3,entries);
      if(exestr!=NULL) {
        if(extorder->extstart==extorder->NM_EXT_SHOW_OUTPUT) {
          tstr=(char*)_allocsafe(strlen(exestr)+3+strlen(tmpoutput)+1);
          if(loopnr==1) sprintf(tstr,"%s >%s",exestr,tmpoutput);
          else sprintf(tstr,"%s >>%s",exestr,tmpoutput);
          _freesafe(exestr);
          exestr=tstr;
        }
        fp->putString(exestr);
        fp->putString("\n");
        _freesafe(exestr);
        if(extorder->separate_each_entry==false) break;
        if(getEntriesRealSize(entries[0])==0) break;
        if(entries[0]->size()==lastsize[0]) {
          // endless loop
          //TODO: requester
          break;
        }
      } else {
        cancel=true;
        break;
      }
      lastsize[0]=entries[0]->size();
    }
    if(cancel==false) {
      if((extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL_AND_WAIT4KEY)||
         (extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL)) {
        /* in terminal starten */
        if(extorder->extstart==extorder->NM_EXT_START_IN_TERMINAL_AND_WAIT4KEY) {
          fp->putStringExt("echo %s\n",catalog.getLocale(201));
          fp->putString("read x\n");
        }
//        exestr=(char*)_allocsafe(strlen(TERMINAL_BIN)+strlen(tmpname)+1);
//        sprintf(exestr,TERMINAL_BIN,tmpname);
        exestr=(char*)_allocsafe(strlen(wconfig->getTerminalStr())+strlen(tmpname)+1);
        sprintf(exestr,wconfig->getTerminalStr(),tmpname);
      } else {
        exestr=dupstring(tmpname);
      }
      if(extorder->extstart==extorder->NM_EXT_SHOW_OUTPUT) {
        useint=true;
        if(extorder->view_str!=NULL)
          if(strlen(extorder->view_str)>2)
            if(strstr(extorder->view_str,"%s")!=NULL) useint=false;
        if(useint==true) fp->putStringExt(OUTPUT_BIN,tmpoutput);
        else fp->putStringExt(extorder->view_str,tmpoutput);
        fp->putString("\n");
      }
      fp->close();
      chmod(tmpname,0700);
      runCommand( exestr, tmpname, tmpoutput, extorder->inbackground );
      _freesafe(exestr);
    } else fp->close();
  }
  _freesafe(tmpname);
  _freesafe(tmpoutput);
  // remove entries[]
  for(i=0;i<2;i++) {
    efe=(NM_extern_fe*)entries[i]->getFirstElement();
    while(efe!=NULL) {
      delete efe;
      efe=(NM_extern_fe*)entries[i]->getNextElement();
    }
    delete entries[i];
  }
  // remove splist[]
  for(i=0;i<2;i++) {
    ss2=(NM_specialsource*)splist[i]->getFirstElement();
    while(ss2!=NULL) {
      _freesafe(ss2);
      ss2=(NM_specialsource*)splist[i]->getNextElement();
    }
    delete splist[i];
  }
}

int NormalMode::create_externfe_rec(List *extlist,char *fullname,bool rec,FileEntry *fe,int row)
{
  ArrayList *f1;
  int id;
  NM_extern_fe *efe;
  Verzeichnis *tverz;
  FileEntry *tfe;
  
  if((Datei::fileExistsExt(fullname)==Datei::D_FE_DIR)&&(rec==true)) {
    // create all subentries
    tverz=new Verzeichnis();
    if(tverz->readDir(fullname)==0) {
      f1=tverz->getFiles();
      if(f1!=NULL) {
        id=f1->initEnum();
        tfe=(FileEntry*)f1->getFirstElement(id);
        while(tfe!=NULL) {
          if(strcmp(tfe->name,"..")!=0) {
            create_externfe_rec(extlist,tfe->fullname,rec,NULL,-1);
          }
          tfe=(FileEntry*)f1->getNextElement(id);
        }
        f1->closeEnum(id);
      }
    }
    delete tverz;
    // now append the dir
    efe=new NM_extern_fe(fullname,fe,row);
    efe->setDirFinished(1);  // this indicates the end of the recursive dir
                             // so it can be deactivated in the LV
    extlist->addElement(efe);
  } else {
    efe=new NM_extern_fe(fullname,fe,row);
    extlist->addElement(efe);
  }
  return 0;
}

int NormalMode::createExtList(List *splist,bool rec,List *extlist)
{
  int id;
  NM_specialsource *ss1;
  
  id=splist->initEnum();
  ss1=(NM_specialsource*)splist->getFirstElement(id);
  while(ss1!=NULL) {
    create_externfe_rec(extlist,ss1->entry->fullname,rec,ss1->entry,ss1->row);
    ss1=(NM_specialsource*)splist->getNextElement(id);
  }
  splist->closeEnum(id);
  return 0;
}

char *NormalMode::parseComStr(char *sstr,struct NM_externorder *extorder,int maxlen,List *entries[2])
{
  char *dstr=NULL,buf1[129],ch;
  int spos=0,dpos=0,bracketcount=0,bufpos=0;
  int mode=0,myside;
  bool otherside,nounselect,noext;
  int filemode,tpos,tmode;
  NormalMode *nm2,*tnm;
  Lister *l2;
  ListerMode *lm1;
  int tlen;
  bool ende,takefirst;
  char *tstr;
  NM_extern_fe *efe1;
  List *l;
  int id;
  bool cancel=false;
  
  nm2=NULL;
  l2=parentlister->getWorker()->getOtherLister(parentlister);
  lm1=l2->getActiveMode();
  if(lm1!=NULL)
    if(lm1->isType("NormalMode")==true)
      nm2=(NormalMode*)lm1;
  
  myside=parentlister->getSide();
  dstr=(char*)_allocsafe(maxlen+1);
  if(dstr!=NULL) {
    for(ende=false;ende==false;) {
      ch=sstr[spos++];
      if(ch=='\0') break;
      switch(mode) {
        case 1:
          // we are in a open bracket and waiting for close bracket
          if(ch=='{') {
            // other flag, perhaps useless, but perhaps it is in a String-requester
            // in any case just overtake it in the buffer
            if(bufpos<128) buf1[bufpos++]=ch;
            bracketcount++;
          } else if(ch=='}') {
            // a closeing bracket
            if(bracketcount>1) {
              // this is a bracket in the bracket
              if(bufpos<128) buf1[bufpos++]=ch;
            } else {
              // this is the awaited closebracket
              buf1[bufpos]=0;
              // now buf1 contains a flag which must be parsed
              mode=3;
            }
            bracketcount--;
          } else if(ch=='\\') {
            // backslash are only resolved on toplevel (bracketcount==0)
            // so overtake also this
            if(bufpos<128) buf1[bufpos++]=ch;
            mode=4;
          } else {
            if(bufpos<128) buf1[bufpos++]=ch;
          }
          break;
        case 2:
          dstr[dpos++]=ch;
          mode=0;
          break;
        case 4:
          if(bufpos<128) buf1[bufpos++]=ch;
          mode=1;
          break;
        default:
          // we are in no bracket
          if(ch=='\\') {
            // next character is protected and will be overtaken
            mode=2;
          } else if(ch!='{') {
            // a normal character
            dstr[dpos++]=ch;
          } else {
            mode=1;
            bracketcount++;
            bufpos=0;
          }
          break;
      }
      if(mode==3) {
        // parse buf1
        if(strncmp(buf1,"Rs",2)==0) {
          // Stringrequest
          tstr=StringRequest(buf1,extorder,maxlen-dpos,entries);
          if(tstr!=NULL) {
            tlen=strlen(tstr);
            if((dpos+tlen)<maxlen) {
              strcpy(&dstr[dpos],tstr);
              dpos+=strlen(tstr);
            } else ende=true;
            _freesafe(tstr);
          } else {
            cancel=true;
            ende=true;
          }
        } else if(strncmp(buf1,"p",1)==0) {
          // path
          tlen=0;
          if(extorder->withapo==true) tlen+=2;
          tstr=getActualDir();
          if(tstr!=NULL) {
            tlen+=strlen(tstr);
            if((dpos+tlen)<maxlen) {
              if(extorder->withapo==true) dstr[dpos++]='"';
              strcpy(&dstr[dpos],tstr);
              dpos+=strlen(tstr);
              if(extorder->withapo==true) dstr[dpos++]='"';
            } else ende=true;
          }
        } else if(strncmp(buf1,"op",2)==0) {
          // other side
          tlen=0;
          if(extorder->withapo==true) tlen+=2;
          if(nm2==NULL) tstr=NULL;
          else tstr=nm2->getActualDir();
          if(tstr!=NULL) {
            tlen+=strlen(tstr);
            if((dpos+tlen)<maxlen) {
              if(extorder->withapo==true) dstr[dpos++]='"';
              strcpy(&dstr[dpos],tstr);
              dpos+=strlen(tstr);
              if(extorder->withapo==true) dstr[dpos++]='"';
            } else ende=true;
          }
        } else if(strncmp(buf1,"lp",2)==0) {
          // left side
          if(myside==0) tnm=this;
          else tnm=nm2;
          tlen=0;
          if(extorder->withapo==true) tlen+=2;
          if(tnm==NULL) tstr=NULL;
          else tstr=tnm->getActualDir();
          if(tstr!=NULL) {
            tlen+=strlen(tstr);
            if((dpos+tlen)<maxlen) {
              if(extorder->withapo==true) dstr[dpos++]='"';
              strcpy(&dstr[dpos],tstr);
              dpos+=strlen(tstr);
              if(extorder->withapo==true) dstr[dpos++]='"';
            } else ende=true;
          }
        } else if(strncmp(buf1,"rp",2)==0) {
          // right side
          if(myside==1) tnm=this;
          else tnm=nm2;
          tlen=0;
          if(extorder->withapo==true) tlen+=2;
          if(tnm==NULL) tstr=NULL;
          else tstr=tnm->getActualDir();
          if(tstr!=NULL) {
            tlen+=strlen(tstr);
            if((dpos+tlen)<maxlen) {
              if(extorder->withapo==true) dstr[dpos++]='"';
              strcpy(&dstr[dpos],tstr);
              dpos+=strlen(tstr);
              if(extorder->withapo==true) dstr[dpos++]='"';
            } else ende=true;
          }
        } else if(strncmp(buf1,"dp",2)==0) {
          // destination path
          tlen=0;
          if(extorder->withapo==true) tlen+=2;
          if(extorder->destmode==NULL) tstr=NULL;
          else tstr=extorder->destmode->getActualDir();
          if(tstr!=NULL) {
            tlen+=strlen(tstr);
            if((dpos+tlen)<maxlen) {
              if(extorder->withapo==true) dstr[dpos++]='"';
              strcpy(&dstr[dpos],tstr);
              dpos+=strlen(tstr);
              if(extorder->withapo==true) dstr[dpos++]='"';
            } else ende=true;
          }
        } else {
          // file flag or unknown
          filemode=-1;
          otherside=false;
          nounselect=false;
          noext=false;
          tpos=0;
          tmode=0;
          for(;(tmode<4)&&(tpos<4);) {
            ch=buf1[tpos++];
            if(ch=='o') {
              if(tmode<1) {
                otherside=true;
                tmode=1;
              } else {
                // an "o" at the wrong position
                tmode=6;
              }
            } else if(ch=='u') {
              if(tmode<2) {
                nounselect=true;
                tmode=2;
              } else {
                tmode=6;
              }
            } else if((ch=='f')&&(tmode<3)) {
              filemode=0;
              tmode=3;
            } else if((ch=='a')&&(tmode<3)) {
              filemode=1;
              tmode=3;
            } else if((ch=='F')&&(tmode<3)) {
              filemode=2;
              tmode=3;
            } else if((ch=='A')&&(tmode<3)) {
              filemode=3;
              tmode=3;
            } else if((ch=='E')&&(tmode==3)) {
              noext=true;
              tmode=4;
            } else if(ch=='\0') tmode=4;
          }
          if((filemode>=0)&&(filemode<4)) {
            // we have a valid flag
            takefirst=true;
            if((filemode==1)||(filemode==3)) takefirst=false;
            if(otherside==true) l=entries[1];
            else l=entries[0];
            id=l->initEnum();
            efe1=(NM_extern_fe*)l->getFirstElement(id);
            for(int c=0;;) {
              if(efe1==NULL) break;
              if(efe1->getDirFinished()==0) {
                // this is a valid entry
                if(filemode<2) tstr=efe1->getName(noext);
                else tstr=efe1->getFullname(noext);

                tlen=c;  // first file get no space
                if(extorder->withapo==true) tlen+=2;
                if(tstr!=NULL) {
                  tlen+=strlen(tstr);
                  if((dpos+tlen)<maxlen) {
                    if(c>0) dstr[dpos++]=' ';
                    if(extorder->withapo==true) dstr[dpos++]='"';
                    strcpy(&dstr[dpos],tstr);
                    dpos+=strlen(tstr);
                    if(extorder->withapo==true) dstr[dpos++]='"';
                  } else ende=true;
                }
                if(ende==true) break;
              }
              if(nounselect==false) {
                // remove efe1;
                l->removeFirstElement();
                // now deactivate LVC if not NULL
                if((efe1->getFE()!=NULL)&&( lv->isValidRow( efe1->getRow() ) == true )) {
                  if(otherside==false) {
                    // in my list
                    deselect(efe1->getFE(),efe1->getRow());
                  } else {
                    if(nm2!=NULL) nm2->deselect(efe1->getFE(),efe1->getRow());
                  }
                }
                delete efe1;
                efe1=(NM_extern_fe*)l->getFirstElement(id);
              } else {
                efe1=(NM_extern_fe*)l->getNextElement(id);
              }

              if((c==0)&&(takefirst==true)) break;
              c=1;
            }
            l->closeEnum(id);
          }
        }
        mode=0;
      }
    }
    dstr[dpos]=0;
  }
  if(cancel==true) {
    _freesafe(dstr);
    return NULL;
  }
  return dstr;
}

void NormalMode::deselect(FileEntry *fe, int row)
{
  if((lv->isValidRow(row)==false)||(fe==NULL)) return;
  if(ce==NULL) return;
  if(fe->select==false) return;
  if(lv->isValidRow(row)) {
    lv->setSelect(row,false);
    lv->showRow(row);
  }
  fe->select=false;
  ce->calc();
  showCacheState();
}

char *NormalMode::StringRequest(char *sstr,struct NM_externorder *extorder,int maxlen,List *entries[2])
{
  char *istr=NULL,ch;
  char *defstr=NULL;
  char *ristr=NULL,*rdefstr=NULL;
  int spos=0,bracketcount=0;
  int istart,iend,defstart,defend;
  int mode;
  char *buttonstr,*textstr,*return_str;
  int erg;
  
  spos=2; // buf1 starts with Rs
  istart=iend=defstart=defend=-1;
  mode=0;
  for(;mode<4;) {
    ch=sstr[spos++];
    if(ch=='\0') break;
    switch(mode) {
      case 0:
        // we are in no bracket
        if(ch=='{') {
          mode=1;
          bracketcount++;
          istart=spos;
        }
        break;
      case 1:
        // in the infotext
        if(ch=='{') bracketcount++;
        else if(ch=='}') {
          if(bracketcount==1) {
            iend=spos-2;
            mode=2;
          }
          bracketcount--;
        }
        break;
      case 2:
        // we are in no bracket
        if(ch=='{') {
          mode=3;
          bracketcount++;
          defstart=spos;
        }
        break;
      case 3:
        // in the defaulttext
        if(ch=='{') bracketcount++;
        else if(ch=='}') {
          if(bracketcount==1) {
            defend=spos-2;
            mode=4;
          }
          bracketcount--;
        }
        break;
      default:
        break;
    }
  }
  if((istart>=0)&&(iend>=istart)) {
    // we have an infotext
    istr=dupstring(&sstr[istart]);
    istr[iend-istart+1]=0;
    ristr=parseComStr(istr,extorder,maxlen,entries);
    _freesafe(istr);
  }
  if((defstart>=0)&&(defend>=istart)) {
    // we have a defaulttext
    defstr=dupstring(&sstr[defstart]);
    defstr[defend-defstart+1]=0;
    rdefstr=parseComStr(defstr,extorder,maxlen,entries);
    _freesafe(defstr);
  }
  
  textstr=NULL;
  if(ristr!=NULL) {
    if(strlen(ristr)>0) textstr=ristr;
    else _freesafe(ristr);
  }
  if(textstr==NULL) textstr=dupstring(catalog.getLocale(200));
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg=req->string_request(catalog.getLocale(123),textstr,
                          (rdefstr!=NULL)?rdefstr:"",
                          buttonstr,&return_str);
  _freesafe(buttonstr);
  _freesafe(textstr);
  if(rdefstr!=NULL) _freesafe(rdefstr);
  if(erg!=0) {
    _freesafe(return_str);
    return NULL;
  }
  return return_str;
}

int NormalMode::getEntriesRealSize(List*l)
{
  int id=l->initEnum(),rs;
  NM_extern_fe *efe1;
  efe1=(NM_extern_fe*)l->getFirstElement(id);
  rs=0;
  while(efe1!=NULL) {
    if(efe1->getDirFinished()==0) rs++;
    efe1=(NM_extern_fe*)l->getNextElement(id);
  }
  l->closeEnum(id);
  return rs;
}

void NormalMode::renamef(struct NM_renameorder *renorder)
{
  List *renlist;
  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int id,erg;
  bool skip,cancel,nameok;
  char *newname,*buttonstr,*textstr,*newfullname,*return_str;
  int *dis;
#if 0
  int oldx,oldy;
#endif

  if(getActualDir()==NULL) {
    // no actual dir => no files to rename
    return;
  }

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();

  dis=wconfig->getDisplay(parentlister->getSide());
  
  renlist=new List();
  switch(renorder->source) {
    case renorder->NM_SPECIAL:
      ss1=(NM_specialsource*)renorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if(fe==ss1->entry) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=ss1->entry;
          ss2->row=ss1->row;
          ss2->cod=NULL;
          renlist->addElement(ss2);
        }
        ss1=(NM_specialsource*)renorder->sources->getNextElement();
      }
      break;
    case renorder->NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->row=row;
              ss2->cod=NULL;
              renlist->addElement(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            renlist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(renlist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=ce->activefe;
                ss2->row=row;
                ss2->cod=NULL;
                renlist->addElement(ss2);
              }
            }
          }
        }
      }
      break;
  }
  id=renlist->initEnum();
  
  cancel=skip=false;
  ss1=(NM_specialsource*)renlist->getFirstElement(id);
  while((ss1!=NULL)&&(cancel==false)) {
    // now rename ss1->entry
    fe=ss1->entry;
    if(fe!=NULL) {
      newname=dupstring(fe->name);
      newfullname=NULL;
      for(nameok=false;nameok==false;) {
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(148))+strlen(fe->name)+1);
        sprintf(textstr,catalog.getLocale(148),fe->name);
        erg=req->string_request(catalog.getLocale(149),textstr,newname,buttonstr,&return_str);
        _freesafe(buttonstr);
        _freesafe(textstr);
        _freesafe(newname);
        newname=return_str;
        if(strcmp(newname,fe->name)==0) {
          // same name as orig
          // so skip this entry
          break;
        }
        if(erg==2) {
          cancel=true;
          break;
        } else if(erg==1) {
          break;
        } else {
          newfullname=(char*)_allocsafe(strlen(getActualDir())+1+strlen(newname)+1);
          if(strlen(getActualDir())>1) sprintf(newfullname,"%s/%s",getActualDir(),newname);
          else sprintf(newfullname,"%s%s",getActualDir(),newname);
          if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
            sprintf(textstr,catalog.getLocale(279),newname);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));
            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) break;
            else if(erg==2) {
              cancel=true;
              break;
            }
            _freesafe(newfullname);
            newfullname=NULL;
          } else {
            nameok=true;
          }
        }
      }
      if((nameok==true)&&(newfullname!=NULL)) {
        erg=rename(fe->fullname,newfullname);
        if(erg!=0) {
          textstr=catalog.getLocale(151);
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
          _freesafe(buttonstr);
          if(erg==1) cancel=true;
        } else {
          // deselect and change lvc temp.
          if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
          _freesafe(fe->name);
          _freesafe(fe->fullname);
          fe->name=newname;
          newname=NULL;
          if(fe->name[0]=='.') fe->isHidden=true;
          else fe->isHidden=false;          
          fe->fullname=newfullname;
          newfullname=NULL;
          if(lv->isValidRow(ss1->row)==true) {
            //TODO: Eigentlich muesste hier auch das DND neugesetzt werden, aber ich weiss nicht die Position
            // d.h. ich muesste erst suchen.
            // allerdings kann das auch so bleiben,da am Ende ohnehin buildLister gemacht wird
            verz->setLVC4FE(lv,ss1->row,fe,dis);
            lv->redraw(ss1->row);
            aguix->Flush();
          }
        }
      }
      if(newname!=NULL) _freesafe(newname);
      if(newfullname!=NULL) _freesafe(newfullname);
    }
    ss1=(NM_specialsource*)renlist->getNextElement(id);
  }
  renlist->closeEnum(id);
  
  ss1=(NM_specialsource*)renlist->getFirstElement();
  while(ss1!=NULL) {
    _freesafe(ss1);
    ss1=(NM_specialsource*)renlist->getNextElement();
  }
  delete renlist;
  // now update
  
#if 0
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();
  verz->sort(sortmode,true);
  ce->restartcheck();
  check(ce);
  buildLister();
  lv->setYOffset(oldy);
  lv->setXOffset(oldx);
  lv->showActive();
  lv->redraw();
#else
  update(false);
#endif
}

void NormalMode::dirsize(struct NM_dirsizeorder *dsorder)
{
  List *dslist;
  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int id;
  bool enter;
  int *dis;
  int oldx,oldy;
  
  finishsearchmode();
  
  oldx=lv->getXOffset();
  oldy=lv->getYOffset();

  if(dsorder==NULL) return;
  if(getActualDir()==NULL) return;
  verz=ce->verz;
  filelist=verz->getFiles();
  
  dis=wconfig->getDisplay(parentlister->getSide());
  
  dslist=new List();
  switch(dsorder->source) {
    case dsorder->NM_SPECIAL:
      ss1=(NM_specialsource*)dsorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if(fe==ss1->entry) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=ss1->entry;
          ss2->row=ss1->row;
          ss2->cod=NULL;
          dslist->addElement(ss2);
        }
        ss1=(NM_specialsource*)dsorder->sources->getNextElement();
      }
      break;
    case dsorder->NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->row=row;
              ss2->cod=NULL;
              dslist->addElement(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            dslist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(dslist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=ce->activefe;
                ss2->row=row;
                ss2->cod=NULL;
                dslist->addElement(ss2);
              }
            }
          }
        }
      }
      break;
  }

  // create the NM_CopyOp_Dir for each dir in copylist
  id=dslist->initEnum();
  
  ss1=(NM_specialsource*)dslist->getFirstElement(id);
  while(ss1!=NULL) {
    enter=false;
    if(ss1->entry->isDir()==true) {
      if(ss1->entry->isLink==false) enter=true;
    }
    if(enter==true) {
      if ( getDirSize( ss1->entry, ss1->entry->dirsize ) == 0 ) {
        if(lv->isValidRow(ss1->row)==true) {
#if 1
          verz->setLVC4FE(lv,ss1->row,ss1->entry,dis);
          lv->redraw();
#else
          buildLister();
          lv->setYOffset(oldy);
          lv->setXOffset(oldx);
          lv->redraw();
#endif
          aguix->Flush();
        }
      } else {
        ss1->entry->dirsize = -1;
      }
    }
    ss1=(NM_specialsource*)dslist->getNextElement(id);
  }

  dslist->closeEnum(id);
  
  ss1=(NM_specialsource*)dslist->getFirstElement();
  while(ss1!=NULL) {
    _freesafe(ss1);
    ss1=(NM_specialsource*)dslist->getNextElement();
  }
  delete dslist;
  // now update
  verz->sort(sortmode,true);
  ce->restartcheck();
  check(ce);
  buildLister();
  lv->setYOffset(oldy);
  lv->setXOffset(oldx);
  showCacheState();
  lv->redraw();
}

int
NormalMode::getDirSize( FileEntry*rootfe, loff_t &return_size )
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  bool enter;
  loff_t size,recsize;
  Verzeichnis *verz=new Verzeichnis();
  
  if(verz->readDir(rootfe->fullname)!=0) {
    delete verz;
    return -1;
  }
  
  f1=verz->getFiles();
  if(f1==NULL) {
    delete verz;
    return -1;
  }
  parentlister->setStatebarText(rootfe->fullname);
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
  size=0;
  while(fe!=NULL) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        if(fe->isLink==false) enter=true;
      }
      if(enter==true) {
        recsize = 0;
        if ( getDirSize( fe, recsize ) == 0 ) {
          if(recsize>0) size+=recsize;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        size+=fe->size;
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  delete verz;
  return_size = size;
  return 0;
}

void NormalMode::simdd()
{
  if(getActualDir()==NULL) {
    return;
  }
  if(ce->activefe!=NULL) startAction(ce->activefe);
}

void
NormalMode::startprog(struct NM_startprogorder *sporder)
{
  FileEntry *fe;
  char *tmpname,*tmpoutput,*exestr;
  char *textstr,*buttonstr,*return_str;
  Datei *fp;
  bool useint;
  int erg;

  if(sporder==NULL) return;

  finishsearchmode();
  
  fe=sporder->usefe;
  if(fe==NULL)
    if(ce!=NULL)
      if(ce->activefe!=NULL) fe=ce->activefe;
  if(fe!=NULL) {
    textstr=catalog.getLocale(202);
    buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                strlen(catalog.getLocale(8))+1);
    sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                              catalog.getLocale(8));
    erg=req->string_request(catalog.getLocale(123),textstr,
                            fe->name,buttonstr,&return_str);
    _freesafe(buttonstr);
    if(erg==0) {
      tmpname=Datei::createTMPName();
      tmpoutput=Datei::createTMPName();
      fp=new Datei();
      if(fp->open(tmpname,"w")==0) {
        fp->putString("#!/bin/sh\n");
        if(getActualDir()!=NULL) fp->putStringExt("cd \"%s\"\n",getActualDir());
        if(sporder->global==false) fp->putString("./");
        fp->putString(return_str);
        fp->putString("\n");
        if((sporder->start==sporder->NM_START_IN_TERMINAL_AND_WAIT4KEY)||
           (sporder->start==sporder->NM_START_IN_TERMINAL)) {
          /* in terminal starten */
          if(sporder->start==sporder->NM_START_IN_TERMINAL_AND_WAIT4KEY) {
            fp->putStringExt("echo %s\n",catalog.getLocale(201));
            fp->putString("read x\n");
          }
//          exestr=(char*)_allocsafe(strlen(TERMINAL_BIN)+strlen(tmpname)+1);
//          sprintf(exestr,TERMINAL_BIN,tmpname);
          exestr=(char*)_allocsafe(strlen(wconfig->getTerminalStr())+strlen(tmpname)+1);
          sprintf(exestr,wconfig->getTerminalStr(),tmpname);
        } else {
          exestr=dupstring(tmpname);
        }
        if(sporder->start==sporder->NM_SHOW_OUTPUT) {
          useint=true;
          if(sporder->view_str!=NULL)
            if(strlen(sporder->view_str)>2)
              if(strstr(sporder->view_str,"%s")!=NULL) useint=false;
          if(useint==true) fp->putStringExt(OUTPUT_BIN,tmpoutput);
          else fp->putStringExt(sporder->view_str,tmpoutput);
          fp->putString("\n");
        }
        fp->close();
        chmod(tmpname,0700);
        runCommand( exestr, tmpname, tmpoutput, sporder->inbackground );
        _freesafe(exestr);
      }
      _freesafe(tmpname);
      _freesafe(tmpoutput);
    }
    _freesafe(return_str);
  }
}

void NormalMode::searchentry()
{
  if(searchmodeon==false) startsearchmode();
  else {
    shownextentry();
  }
}

void NormalMode::startsearchmode()
{
  if(getActualDir()==NULL) {
    return;
  }
  oldstr=dupstring(sg->getText());
  lastsearch=dupstring("");
  sg->setText("");
  sg->setForbidPosChange(true);
  sg->setStrongKeyCapture(false);
  sg->activate();
  searchmodeon=true;
}

void NormalMode::finishsearchmode()
{
  if(searchmodeon==false) return;
  sg->setForbidPosChange(false);
  sg->setStrongKeyCapture(true);
  sg->setText(oldstr);
  sg->deactivate();
  _freesafe(oldstr);
  if(lastsearch!=NULL) _freesafe(lastsearch);
  else debugmsg("oops2\n");
  lastsearch=NULL;
  searchmodeon=false;
}

bool NormalMode::showfirstentry(const char *str)
{
  Verzeichnis *verz;
  ArrayList *filelist;
  FileEntry *startfe;
  bool found;
  int id;
  int pos;
  
  found=false;

  if(getActualDir()==NULL) {
    return found;
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  
  id=filelist->initEnum();
  
  startfe=ce->activefe;
  pos=0;
  if(startfe!=NULL) pos=filelist->getIndex(startfe);
  return shownextentry(str,pos);
}

bool NormalMode::shownextentry()
{
  Verzeichnis *verz;
  ArrayList *filelist;
  FileEntry *startfe;
  bool found;
  int id;
  int pos;
  
  found=false;

  if(getActualDir()==NULL) {
    return found;
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  
  id=filelist->initEnum();
  
  startfe=ce->activefe;
  pos=0;
  if(startfe!=NULL) pos=filelist->getIndex(startfe)+1;
  return shownextentry(lastsearch,pos);
}

bool NormalMode::shownextentry(const char *str,int pos)
{
  Verzeichnis *verz;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int id;
  char *realstr;
  
  found=false;

  if(getActualDir()==NULL) {
    return found;
  }
  verz=ce->verz;
  filelist=verz->getFiles();
  
  id=filelist->initEnum();
  
  if(pos>=0) {
    realstr=(char*)_allocsafe(strlen(str)+1+1);
    sprintf(realstr,"%s*",str);
    for(int i=0;(i<2)&&(found==false);i++) {
      fe=(FileEntry*)filelist->getElementAt(id,pos);
      while((fe!=NULL)&&(found==false)) {
        if(fe->use==true) {
          if(fnmatch(realstr,fe->name,0)==0) {
            // hit
            ce->activefe=fe;
            row = 0;
            while(lv->isValidRow(row)==true) {
              if(lv->getData(row)==fe->nr) {
                lv->setActiveRow( row);
                break;
              }
              row++;
            }
            found=true;
          }
        }
        fe=(FileEntry*)filelist->getNextElement(id);
      }
      pos=0; // if not found try from pos 0 again
    }
    _freesafe(realstr);
  }
  if(found==true) {
    lv->showActive();
    lv->redraw();
    aguix->Flush();
  }
  
  filelist->closeEnum(id);
  return found;
}

void NormalMode::enterpath()
{
  sg->activate();
}

void NormalMode::scrolllister(int dir)
{
  lv->setXOffset(lv->getXOffset()+dir);
  aguix->Flush();
}

void NormalMode::setActiveDir2Other()
{
  Lister *l1;
  ListerMode *lm1;
  NormalMode *nm1;

  if(ce==NULL) return;
  if(ce->activefe!=NULL) {
    if(ce->activefe->isDir()==true) {
      l1=parentlister->getWorker()->getOtherLister(parentlister);
      if(l1!=NULL) {
        l1->switch2Mode(0);
        lm1=l1->getActiveMode();
        if(lm1!=NULL) {
          if(lm1->isType("NormalMode")==true) {
            nm1=(NormalMode*)lm1;
            nm1->enterDir(ce->activefe->fullname);
          }
        }
      }
    }
  }
}

void NormalMode::createsymlink(struct NM_createsymlinkorder *cslorder)
{
  List *csllist;
  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int id,erg;
  bool skip,cancel,nameok;
  char *newname,*buttonstr,*textstr,*newfullname,*return_str;
  char *sldest;
  Lister *olister=NULL;
  NormalMode *nm2=NULL,*updatenm=NULL;
  ListerMode *lm=NULL;

  if(getActualDir()==NULL) {
    // no actual dir => no files to rename
    return;
  }

  finishsearchmode();
  
  if(strcmp(getActualDir(),cslorder->destdir)==0) {
    updatenm=this;
  }
  olister=parentlister->getWorker()->getOtherLister(parentlister);
  if(olister==NULL) return;
  lm=olister->getActiveMode();
  if(lm!=NULL) {
    if(lm->isType("NormalMode")==true) {
      nm2=(NormalMode*)lm;
      if(nm2->getActualDir()!=NULL) {
        if(strcmp(nm2->getActualDir(),cslorder->destdir)==0) {
          updatenm=nm2;
        }
      }
    }
  }

  verz=ce->verz;
  filelist=verz->getFiles();

  csllist=new List();
  switch(cslorder->source) {
    case cslorder->NM_SPECIAL:
      ss1=(NM_specialsource*)cslorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if(fe==ss1->entry) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=ss1->entry;
          ss2->row=ss1->row;
          ss2->cod=NULL;
          csllist->addElement(ss2);
        }
        ss1=(NM_specialsource*)cslorder->sources->getNextElement();
      }
      break;
    case cslorder->NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->row=row;
              ss2->cod=NULL;
              csllist->addElement(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            csllist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(csllist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=ce->activefe;
                ss2->row=row;
                ss2->cod=NULL;
                csllist->addElement(ss2);
              }
            }
          }
        }
      }
      break;
  }
  id=csllist->initEnum();
  
  cancel=skip=false;
  ss1=(NM_specialsource*)csllist->getFirstElement(id);
  while((ss1!=NULL)&&(cancel==false)) {
    fe=ss1->entry;
    if(fe!=NULL) {
      newname=dupstring(fe->name);
      newfullname=NULL;
      for(nameok=false;nameok==false;) {
        buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                    strlen(catalog.getLocale(225))+1+
                                    strlen(catalog.getLocale(8))+1);
        sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                     catalog.getLocale(225),
                                     catalog.getLocale(8));
        textstr=(char*)_allocsafe(strlen(catalog.getLocale(207))+strlen(fe->name)+1);
        sprintf(textstr,catalog.getLocale(207),fe->name);
        erg=req->string_request(catalog.getLocaleCom(36),textstr,newname,buttonstr,&return_str);
        _freesafe(buttonstr);
        _freesafe(textstr);
        _freesafe(newname);
        newname=return_str;
        if(erg==2) {
          cancel=true;
          break;
        } else if(erg==1) {
          break;
        } else {
          newfullname=(char*)_allocsafe(strlen(cslorder->destdir)+1+strlen(newname)+1);
          if(strlen(cslorder->destdir)>1) sprintf(newfullname,"%s/%s",cslorder->destdir,newname);
          else sprintf(newfullname,"%s%s",cslorder->destdir,newname);
          if(Datei::lfileExistsExt(newfullname)!=Datei::D_FE_NOFILE) {
            textstr=(char*)_allocsafe(strlen(catalog.getLocale(279))+strlen(newname)+1);
            sprintf(textstr,catalog.getLocale(279),newname);
            buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(230))+1+
                                        strlen(catalog.getLocale(225))+1+
                                        strlen(catalog.getLocale(8))+1);
            sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(230),
                                         catalog.getLocale(225),
                                         catalog.getLocale(8));
            erg=req->request(catalog.getLocale(123),textstr,buttonstr);
            _freesafe(buttonstr);
            _freesafe(textstr);
            if(erg==1) break;
            else if(erg==2) {
              cancel=true;
              break;
            }
            _freesafe(newfullname);
            newfullname=NULL;
          } else {
            nameok=true;
          }
        }
      }
      if((nameok==true)&&(newfullname!=NULL)) {
        // now create symlink newfullname which points to fe->fullname or fe->name
        if(cslorder->local==true) sldest=fe->name;
        else sldest=fe->fullname;
        erg=symlink(sldest,newfullname);
        if(erg!=0) {
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(newfullname)+1);
          sprintf(textstr,catalog.getLocale(205),newfullname);
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==1) cancel=true;
        } else {
          // deselect
          if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
        }
      }
      if(newname!=NULL) _freesafe(newname);
      if(newfullname!=NULL) _freesafe(newfullname);
    }
    ss1=(NM_specialsource*)csllist->getNextElement(id);
  }
  csllist->closeEnum(id);
  
  ss1=(NM_specialsource*)csllist->getFirstElement();
  while(ss1!=NULL) {
    _freesafe(ss1);
    ss1=(NM_specialsource*)csllist->getNextElement();
  }
  delete csllist;
  // now update
  if(updatenm!=NULL) updatenm->update(false);
}

void NormalMode::changesymlink(struct NM_changesymlinkorder *chslorder)
{
  List *csllist;
  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int id,erg;
  bool cancel;
  char *buttonstr,*textstr,*return_str;
  char buffer[4096];
  int tx;

  if(getActualDir()==NULL) {
    // no actual dir => no files to rename
    return;
  }

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();

  csllist=new List();
  switch(chslorder->source) {
    case chslorder->NM_SPECIAL:
      ss1=(NM_specialsource*)chslorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if(fe==ss1->entry) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=ss1->entry;
          ss2->row=ss1->row;
          ss2->cod=NULL;
          csllist->addElement(ss2);
        }
        ss1=(NM_specialsource*)chslorder->sources->getNextElement();
      }
      break;
    case chslorder->NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->row=row;
              ss2->cod=NULL;
              csllist->addElement(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            csllist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(csllist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=ce->activefe;
                ss2->row=row;
                ss2->cod=NULL;
                csllist->addElement(ss2);
              }
            }
          }
        }
      }
      break;
  }
  id=csllist->initEnum();
  
  cancel=false;
  ss1=(NM_specialsource*)csllist->getFirstElement(id);
  while((ss1!=NULL)&&(cancel==false)) {
    fe=ss1->entry;
    if(fe!=NULL) {
      if(fe->isLink==true) {
        tx=readlink(fe->fullname,buffer,4095);
        if(tx>=0) {
          buffer[tx]=0;
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(225))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s|%s",catalog.getLocale(11),
                                       catalog.getLocale(225),
                                       catalog.getLocale(8));
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(206))+strlen(fe->name)+1);
          sprintf(textstr,catalog.getLocale(206),fe->name);
          erg=req->string_request(catalog.getLocaleCom(37),textstr,buffer,buttonstr,&return_str);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==2) {
            // cancel
            cancel=true;
          } else if(erg==0) {
            // ok
            if(strcmp(buffer,return_str)!=0) {
              // only if new value differs
              if(remove(fe->fullname)==0) {
                if(symlink(return_str,fe->fullname)==0) {
                  // deselect
                  if(lv->isValidRow(ss1->row)==true) deselect(fe,ss1->row);
                } else {
                  textstr=(char*)_allocsafe(strlen(catalog.getLocale(205))+strlen(fe->fullname)+1);
                  sprintf(textstr,catalog.getLocale(205),fe->fullname);
                  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                              strlen(catalog.getLocale(8))+1);
                  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                            catalog.getLocale(8));
                  erg=req->request(catalog.getLocale(347),textstr,buttonstr);
                  _freesafe(buttonstr);
                  _freesafe(textstr);
                  if(erg==1) cancel=true;
                }
              } else {
                textstr=(char*)_allocsafe(strlen(catalog.getLocale(291))+strlen(fe->fullname)+1);
                sprintf(textstr,catalog.getLocale(291),fe->fullname);
                buttonstr=catalog.getLocale(11);
                req->request(catalog.getLocale(347),textstr,buttonstr);
                _freesafe(textstr);
              }
            }
          }
          _freesafe(return_str);
        }
      }
    }
    ss1=(NM_specialsource*)csllist->getNextElement(id);
  }
  csllist->closeEnum(id);
  
  ss1=(NM_specialsource*)csllist->getFirstElement();
  while(ss1!=NULL) {
    _freesafe(ss1);
    ss1=(NM_specialsource*)csllist->getNextElement();
  }
  delete csllist;
}

void NormalMode::chmodf(struct NM_chmodorder *cmorder)
{
  List *cmlist,*cmfelist;
  Verzeichnis *verz;
  NM_specialsource *ss1,*ss2;
  ArrayList *filelist;
  FileEntry *fe;
  bool found;
  int row;
  int id;
  bool enter;
  NM_CopyOp_Dir *cod1;
  bool skip,cancel,all;
  mode_t mode4all,newmode;
  int erg;
  char *buttonstr,*textstr;
  
  if(cmorder==NULL) return;
  if(getActualDir()==NULL) {
    // no actual dir => no files to copy
    return;
  }

  finishsearchmode();
  
  verz=ce->verz;
  filelist=verz->getFiles();
  
  cmlist=new List();
  switch(cmorder->source) {
    case cmorder->NM_SPECIAL:
      ss1=(NM_specialsource*)cmorder->sources->getFirstElement();
      while(ss1!=NULL) {
        fe=(FileEntry*)filelist->getFirstElement();
        found=false;
        while(fe!=NULL) {
          if(fe==ss1->entry) {
            if(strcmp(fe->name,"..")!=0) {
              found=true;
              break;
            }
          }
          fe=(FileEntry*)filelist->getNextElement();
        }
        if(found==true) {
          // valid entry found
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=ss1->entry;
          ss2->row=ss1->row;
          ss2->cod=NULL;
          cmlist->addElement(ss2);
        }
        ss1=(NM_specialsource*)cmorder->sources->getNextElement();
      }
      break;
    case cmorder->NM_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->row=row;
              ss2->cod=NULL;
              cmlist->addElement(ss2);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement();
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            cmlist->addElement(ss2);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement();
      }
      if(cmlist->size()==0) {
        // no selected entries => take the active
        if(ce->activefe!=NULL) {
          row=lv->getActiveRow();
          if(lv->isValidRow(row)==true) {
            if(lv->getData(row)==ce->activefe->nr) {
              if(strcmp(ce->activefe->name,"..")!=0) {
                ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                ss2->entry=ce->activefe;
                ss2->row=row;
                ss2->cod=NULL;
                cmlist->addElement(ss2);
              }
            }
          }
        }
      }
      break;
  }

  id=cmlist->initEnum();
  
  ss1=(NM_specialsource*)cmlist->getFirstElement(id);
  while(ss1!=NULL) {
    enter=false;
    if((ss1->entry->isDir()==true)&&(cmorder->recursive==true)) {
      // fe is a dir, check if it is a link and take it only when follow_symlinks==true
      if(ss1->entry->isLink==false) enter=true;
    }
    if(enter==true) {
      // fe is a dir so creating corresponding entry
      cod1=new NM_CopyOp_Dir(ss1->entry);
      // recursive call
      cod1->createSubDirs();

      ss1->cod=cod1;
    } else {
      ss1->cod=NULL;
    }
    ss1=(NM_specialsource*)cmlist->getNextElement(id);
  }
  
  cmfelist=new List();
  buildchmodfelist(cmlist,cmfelist);
  cmlist->closeEnum(id);

  // now database ready, can start copy-process

  cancel=false;
  all=false;
  mode4all=0;
  id=cmfelist->initEnum();
  ss1=(NM_specialsource*)cmfelist->getFirstElement(id);
  while((ss1!=NULL)&&(cancel==false)) {
    skip=false;
    fe=ss1->entry;
    if((fe->isDir()==true)&&(cmorder->ondirs==false)) skip=true;
    else if((fe->isDir()==false)&&(cmorder->onfiles==false)) skip=true;
    if(skip==false) {
      if(all==false) {
        erg=requestNewMode(fe,&newmode);
        if(erg==1) {
          all=true;
          mode4all=newmode;
        } else if(erg==2) skip=true;
        else if(erg==3) cancel=true;
      } else newmode=mode4all;
      if((skip==false)&&(cancel==false)) {
        if(chmod(fe->fullname,newmode)!=0) {
          // error
          buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                                      strlen(catalog.getLocale(8))+1);
          sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                                    catalog.getLocale(8));
          textstr=(char*)_allocsafe(strlen(catalog.getLocale(226))+strlen(fe->fullname)+1);
          sprintf(textstr,catalog.getLocale(226),fe->fullname);
          erg=req->request(catalog.getLocale(347),textstr,buttonstr);
          _freesafe(buttonstr);
          _freesafe(textstr);
          if(erg==1) cancel=true;
        } else {
          if( lv->isValidRow( ss1->row ) == true ) deselect(fe,ss1->row);
        }
      }
    }
    ss1=(NM_specialsource*)cmfelist->getNextElement(id);
  }

  ss1=(NM_specialsource*)cmfelist->getFirstElement(id);
  while(ss1!=NULL) {
    _freesafe(ss1);
    ss1=(NM_specialsource*)cmfelist->getNextElement(id);
  }
  cmfelist->closeEnum(id);
  delete cmfelist;
  ss1=(NM_specialsource*)cmlist->getFirstElement();
  while(ss1!=NULL) {
    if(ss1->cod!=NULL) delete ss1->cod;
    _freesafe(ss1);
    ss1=(NM_specialsource*)cmlist->getNextElement();
  }
  delete cmlist;
  update(false);
}

void NormalMode::buildchmodfelist(List *cmlist,List *cmfelist)
{
  int id;
  NM_specialsource *ss1,*ss2;
  
  id=cmlist->initEnum();
  ss1=(NM_specialsource*)cmlist->getFirstElement(id);
  while(ss1!=NULL) {
    if(ss1->cod!=NULL) {
      buildchmodfelist_rec(ss1->cod,ss1->row,cmfelist);
    } else {
      ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
      ss2->entry=ss1->entry;
      ss2->row=ss1->row;
      cmfelist->addElement(ss2);
    }
    ss1=(NM_specialsource*)cmlist->getNextElement(id);
  }
  cmlist->closeEnum(id);
}

void NormalMode::buildchmodfelist_rec(NM_CopyOp_Dir *cod, int row,List *cmfelist)
{
  ArrayList *filelist;
  NM_CopyOp_Dir *cod1;
  int id1,id2;
  FileEntry *subfe;
  bool isdir;
  NM_specialsource *ss2;

  id1=cod->subdirs->initEnum();
  cod1=(NM_CopyOp_Dir*)cod->subdirs->getFirstElement(id1);
  while(cod1!=NULL) {
    buildchmodfelist_rec(cod1,-1,cmfelist);
    cod1=(NM_CopyOp_Dir*)cod->subdirs->getNextElement(id1);
  }
  cod->subdirs->closeEnum(id1);

  // next only if read this dir correctly
  if(cod->ok==true) {
    filelist=cod->verz->getFiles();
    id2=filelist->initEnum();
    subfe=(FileEntry*)filelist->getFirstElement(id2);
    while(subfe!=NULL) {
      if(strcmp(subfe->name,"..")!=0) {
        isdir=false;
        if(subfe->isDir()==true) {
          if(subfe->isLink==false) isdir=true;
        }
        if(isdir==false) {
          ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
          ss2->entry=subfe;
          ss2->row=-1;
          cmfelist->addElement(ss2);
        }
      }
      subfe=(FileEntry*)filelist->getNextElement(id2);
    }
    filelist->closeEnum(id2);
  }
  ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
  ss2->entry=(FileEntry*)cod->fileentry;
  ss2->row=row;
  cmfelist->addElement(ss2);
}

int NormalMode::requestNewMode(FileEntry *fe,mode_t *return_mode)
{
  mode_t tmode,newmode;
  Button *okb,*cb,*ok2allb,*skipb;
  AWindow *win;
  Text *ttext,*readtext,*writetext,*exetext;
  Text *owntext,*grouptext,*othtext,*spetext;
  Text *uidtext,*gidtext,*stickytext;
  ChooseButton *mcb[12];
  int mw[5],tx[3],maxw;
  int tw,ttw,tth,ttx,tty;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  GUIElement *ba[4];
  
  tmode=(fe->isLink==true)?fe->dmode:fe->mode;
  ttw=tth=10;
  ttx=tty=5;
  win=new AWindow(aguix);
  win->create(NULL,10,10,ttw,tth,0,catalog.getLocaleCom(38));
  tstr=(char*)_allocsafe(strlen(catalog.getLocale(213))+strlen(fe->fullname)+1);
  sprintf(tstr,catalog.getLocale(213),fe->fullname);

  ttext=(Text*)win->add(new Text(aguix,ttx,tty,tstr,1));
  _freesafe(tstr);
  mw[4]=ttext->getWidth();
  
  tty+=ttext->getHeight()+5;
  
  readtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(218),1));
  writetext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(219),1));
  exetext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(220),1));
  tty+=exetext->getHeight()+5;
  
  owntext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(214),1));
  grouptext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(215),1));
  othtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(216),1));
  spetext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(217),1));
  maxw=owntext->getWidth();
  tw=grouptext->getWidth();
  if(tw>maxw) maxw=tw;
  tw=othtext->getWidth();
  if(tw>maxw) maxw=tw;
  tw=spetext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[0]=maxw;
  
  uidtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(221),1));
  gidtext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(222),1));
  stickytext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(223),1));
  
  maxw=uidtext->getWidth();
  tw=readtext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[1]=maxw;
  maxw=gidtext->getWidth();
  tw=writetext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[2]=maxw;
  maxw=stickytext->getWidth();
  tw=exetext->getWidth();
  if(tw>maxw) maxw=tw;
  mw[3]=maxw;
  
  // now move all elements
  owntext->move(ttx,tty);
  grouptext->move(ttx,owntext->getY()+owntext->getHeight()+5);
  othtext->move(ttx,grouptext->getY()+grouptext->getHeight()+5);
  spetext->move(ttx,othtext->getY()+othtext->getHeight()+5+uidtext->getHeight()+5);
  
  ttx+=mw[0]+5;
  tx[0]=ttx+mw[1]/2;  // center of first column
  readtext->move(tx[0]-readtext->getWidth()/2,readtext->getY());
  ttx+=mw[1]+5;
  tx[1]=ttx+mw[2]/2;
  writetext->move(tx[1]-writetext->getWidth()/2,writetext->getY());
  ttx+=mw[2]+5;
  tx[2]=ttx+mw[3]/2;
  exetext->move(tx[2]-exetext->getWidth()/2,exetext->getY());

  tty=othtext->getY()+othtext->getHeight()+5;
  
  // now create the ChooseButtons
  tty = readtext->getY() + readtext->getHeight() + 5;
  mcb[0]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_IRUSR)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[1]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_IWUSR)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[2]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_IXUSR)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( owntext, mcb[0] );
  mcb[0]->move( tx[0] - mcb[0]->getWidth() / 2, mcb[0]->getY() );
  mcb[1]->move( tx[1] - mcb[1]->getWidth() / 2, mcb[1]->getY() );
  mcb[2]->move( tx[2] - mcb[2]->getWidth() / 2, mcb[2]->getY() );
  tty = mcb[0]->getY() + mcb[0]->getHeight() + 5;

  mcb[3]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_IRGRP)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[4]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_IWGRP)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[5]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_IXGRP)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( grouptext, mcb[3] );
  mcb[3]->move( tx[0] - mcb[3]->getWidth() / 2, mcb[3]->getY() );
  mcb[4]->move( tx[1] - mcb[4]->getWidth() / 2, mcb[4]->getY() );
  mcb[5]->move( tx[2] - mcb[5]->getWidth() / 2, mcb[5]->getY() );
  tty = mcb[3]->getY() + mcb[3]->getHeight() + 5;

  mcb[6]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_IROTH)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[7]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_IWOTH)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[8]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_IXOTH)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( othtext, mcb[6] );
  mcb[6]->move( tx[0] - mcb[6]->getWidth() / 2, mcb[6]->getY() );
  mcb[7]->move( tx[1] - mcb[7]->getWidth() / 2, mcb[7]->getY() );
  mcb[8]->move( tx[2] - mcb[8]->getWidth() / 2, mcb[8]->getY() );
  tty = mcb[6]->getY() + mcb[6]->getHeight() + 5;

  uidtext->move( tx[0] - uidtext->getWidth() / 2, tty );
  gidtext->move( tx[1] - gidtext->getWidth() / 2, tty );
  stickytext->move( tx[2] - stickytext->getWidth() / 2, tty );
  tty += uidtext->getHeight() + 5;

  mcb[9]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[0], tty,
                                ((tmode&S_ISUID)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[10]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[1], tty,
                                ((tmode&S_ISGID)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  mcb[11]=(ChooseButton*)win->add(new ChooseButton(aguix,tx[2], tty,
                                ((tmode&S_ISVTX)==0)?0:1,
                                "",LABEL_RIGHT,0,0));
  AGUIX::centerElementsY( spetext, mcb[9] );
  mcb[9]->move( tx[0] - mcb[9]->getWidth() / 2, mcb[9]->getY() );
  mcb[10]->move( tx[1] - mcb[10]->getWidth() / 2, mcb[10]->getY() );
  mcb[11]->move( tx[2] - mcb[11]->getWidth() / 2, mcb[11]->getY() );
  tty = mcb[9]->getY() + mcb[9]->getHeight() + 10;
  
  win->maximizeX();
  w = win->getWidth();

  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   tty,
                                   catalog.getLocale(11),
                                   1,
                                   0,
                                   0));
  ok2allb=(Button*)win->add(new Button(aguix,
                                   0,
                                   tty,
                                   catalog.getLocale(224),
                                   1,
                                   0,
                                   0));
  skipb=(Button*)win->add(new Button(aguix,
                                   0,
                                   tty,
                                   catalog.getLocale(225),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  0,
                                  tty,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));

  ba[0] = okb;
  ba[1] = ok2allb;
  ba[2] = skipb;
  ba[3] = cb;
  tw = AGUIX::scaleElementsW( w, 5, 5, -1, false, false, ba, 4 );
  if ( tw > w ) {
    w = tw;
    win->resize( w, win->getHeight() );
  }
  
  tty+=okb->getHeight()+5;
  
  tth=tty;
  win->resize( w, tth );
  win->setMaxSize( w, tth );
  win->setMinSize( w, tth );
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=3;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==ok2allb) endmode=1;
          else if(msg->button.button==skipb) endmode=2;
          else if(msg->button.button==cb) endmode=3;
          break;
        case AG_KEYRELEASED:
          if(win->isParent(msg->key.window,false)==true) {
            switch(msg->key.key) {
              case XK_1:
              case XK_KP_1:
                mcb[6]->setState((mcb[6]->getState()==1)?0:1);
                break;
              case XK_2:
              case XK_KP_2:
                mcb[7]->setState((mcb[7]->getState()==1)?0:1);
                break;
              case XK_3:
              case XK_KP_3:
                mcb[8]->setState((mcb[8]->getState()==1)?0:1);
                break;
              case XK_4:
              case XK_KP_4:
                mcb[3]->setState((mcb[3]->getState()==1)?0:1);
                break;
              case XK_5:
              case XK_KP_5:
                mcb[4]->setState((mcb[4]->getState()==1)?0:1);
                break;
              case XK_6:
              case XK_KP_6:
                mcb[5]->setState((mcb[5]->getState()==1)?0:1);
                break;
              case XK_7:
              case XK_KP_7:
                mcb[0]->setState((mcb[0]->getState()==1)?0:1);
                break;
              case XK_8:
              case XK_KP_8:
                mcb[1]->setState((mcb[1]->getState()==1)?0:1);
                break;
              case XK_9:
              case XK_KP_9:
                mcb[2]->setState((mcb[2]->getState()==1)?0:1);
                break;
              case XK_Return:
              case XK_F1:
              case XK_KP_Enter:
                endmode=0;
                break;
              case XK_Escape:
              case XK_F4:
                endmode=3;
                break;
              case XK_F2:
                endmode=1;
                break;
              case XK_F3:
                endmode=2;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if((endmode==0)||(endmode==1)) {
    // ok
    newmode=0;
    newmode|=(mcb[0]->getState()==1)?S_IRUSR:0;
    newmode|=(mcb[1]->getState()==1)?S_IWUSR:0;
    newmode|=(mcb[2]->getState()==1)?S_IXUSR:0;
    newmode|=(mcb[3]->getState()==1)?S_IRGRP:0;
    newmode|=(mcb[4]->getState()==1)?S_IWGRP:0;
    newmode|=(mcb[5]->getState()==1)?S_IXGRP:0;
    newmode|=(mcb[6]->getState()==1)?S_IROTH:0;
    newmode|=(mcb[7]->getState()==1)?S_IWOTH:0;
    newmode|=(mcb[8]->getState()==1)?S_IXOTH:0;
    newmode|=(mcb[9]->getState()==1)?S_ISUID:0;
    newmode|=(mcb[10]->getState()==1)?S_ISGID:0;
    newmode|=(mcb[11]->getState()==1)?S_ISVTX:0;
    *return_mode=newmode;
  }
  
  win->close();
  delete win;

  return endmode;
}

/*****************
 * NM_CopyOp_Dir *
 *****************/

NM_CopyOp_Dir::NM_CopyOp_Dir(const FileEntry *fe)
{
  fileentry=fe;
  subdirs=new List();
  verz=new Verzeichnis();
  files=dirs=error_counter=0;
  bytes = 0;
  ok=false;
  
  if(verz->readDir(fe->fullname)==0) ok=true;
}

NM_CopyOp_Dir::~NM_CopyOp_Dir()
{
  int id=subdirs->initEnum();
  NM_CopyOp_Dir *cod=(NM_CopyOp_Dir*)subdirs->getFirstElement(id);
  while(cod!=NULL) {
    delete cod;
    cod=(NM_CopyOp_Dir*)subdirs->getNextElement(id);
  }
  delete subdirs;
  delete verz;
}

int
NM_CopyOp_Dir::createSubDirs(struct NM_copyorder *copyorder,unsigned long *gf,unsigned long *gd)
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  NM_CopyOp_Dir *cod1;
  bool enter;
  bool cancel=false;
  
  /*TODO: vielleicht sollte der Fehler weitergereicht werden, aber cancel=true soll nicht entstehen
    daher erstmal 0 zurueckgeben */
  if(ok==false) return 0;
  f1=verz->getFiles();
  if(f1==NULL) return 0;
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
/*TODO:Vorerst zu langsam, kann aber an MemSystem liegen, daher nochmal ohne
       das pruefen
  if(copyorder->cowin!=NULL) {
    copyorder->cowin->setmessage(verz->getDir(),1);
    copyorder->cowin->redraw();
  }*/
  while((fe!=NULL)&&(cancel==false)) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if(fe->isLink==false) enter=true;
        else if(copyorder->follow_symlinks==true) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(fe);
        // recursive call
        if(cod1->createSubDirs(copyorder,gf,gd)!=0) cancel=true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;
        bytes+=cod1->bytes;

        // add this subdir to the list
        subdirs->addElement(cod1);
        
        // this is a dir so inc the counter
        dirs++;
        (*gd)++;

        if(copyorder->cowin!=NULL) {
          copyorder->cowin->set_files_to_copy(*gf);
          copyorder->cowin->set_dirs_to_copy(*gd);
          if(copyorder->cowin->redraw()!=0) cancel=true;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        (*gf)++;
        if ( ( fe->isLink == true ) &&
             ( copyorder->follow_symlinks == true ) ) {
          if ( fe->isCorrupt == false ) {
            bytes+=fe->dsize;
          }
        } else {
          bytes+=fe->size;
        }
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  return (cancel==true)?1:0;
}

int
NM_CopyOp_Dir::createSubDirs(struct NM_deleteorder *delorder,unsigned long *gf,unsigned long *gd)
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  NM_CopyOp_Dir *cod1;
  bool enter;
  bool cancel=false;
  
  if(ok==false) return 0;
  f1=verz->getFiles();
  if(f1==NULL) return 0;
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
  while((fe!=NULL)&&(cancel==false)) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if(fe->isLink==false) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(fe);
        // recursive call
        if(cod1->createSubDirs(delorder,gf,gd)!=0) cancel=true;
        // add the values from this subdir to this dir
        files+=cod1->files;
        dirs+=cod1->dirs;
        bytes+=cod1->bytes;
        
        // add this subdir to the list
        subdirs->addElement(cod1);
        
        // this is a dir so inc the counter
        dirs++;
        (*gd)++;

        if(delorder->dowin!=NULL) {
          delorder->dowin->set_files_to_delete(*gf);
          delorder->dowin->set_dirs_to_delete(*gd);
          if(delorder->dowin->redraw()!=0) cancel=true;
        }
      } else {
        // is not dir (mostly a file but can also be links ...)
        files++;
        (*gf)++;
        // when deleting only size of file, not the dest matters
        bytes+=fe->size;
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  return (cancel==true)?1:0;
}

int
NM_CopyOp_Dir::createSubDirs()
{
  ArrayList *f1;
  int id;
  FileEntry *fe;
  NM_CopyOp_Dir *cod1;
  bool enter;
  
  if(ok==false) return 0;
  f1=verz->getFiles();
  if(f1==NULL) return 0;
  id=f1->initEnum();
  fe=(FileEntry*)f1->getFirstElement(id);
  while(fe!=NULL) {
    if(strcmp(fe->name,"..")!=0) {
      enter=false;
      if(fe->isDir()==true) {
        // fe is a dir, check if it is a link and take it only when follow_symlinks==true
        if(fe->isLink==false) enter=true;
      }
      if(enter==true) {
        // fe is a dir so creating corresponding entry
        cod1=new NM_CopyOp_Dir(fe);
        // recursive call
        cod1->createSubDirs();
        
        // add this subdir to the list
        subdirs->addElement(cod1);
      }
    }
    fe=(FileEntry*)f1->getNextElement(id);
  }
  f1->closeEnum(id);
  return 0;
}

NM_extern_fe::NM_extern_fe(char *nfn)
{
  fullname[0]=dupstring(nfn);
  fullname[1]=NULL;
  name[0]=NULL;
  name[1]=NULL;
  fe=NULL;
  row = -1;
  dirfinished=0;
}

NM_extern_fe::~NM_extern_fe()
{
  if(fullname[0]!=NULL) _freesafe(fullname[0]);
  if(fullname[1]!=NULL) _freesafe(fullname[1]);
  if(name[0]!=NULL) _freesafe(name[0]);
  if(name[1]!=NULL) _freesafe(name[1]);
}

char *NM_extern_fe::getFullname(bool noext)
{
  if(noext==false) return fullname[0];
  else {
    if(fullname[1]==NULL) fullname[1]=Datei::getNameWithoutExt(fullname[0]);
    return fullname[1];
  }
}

char *NM_extern_fe::getName(bool noext)
{
  if(noext==false) {
    if(name[0]==NULL) name[0]=Datei::getFilenameFromPath(fullname[0]);
    return name[0];
  } else {
    if(name[0]==NULL) name[0]=Datei::getFilenameFromPath(fullname[0]);
    if(name[1]==NULL) name[1]=Datei::getNameWithoutExt(name[0]);
    return name[1];
  }
}

NM_extern_fe::NM_extern_fe(char *nfn,FileEntry *tfe, int trow)
{
  fullname[0]=dupstring(nfn);
  fullname[1]=NULL;
  name[0]=NULL;
  name[1]=NULL;
  this->fe=tfe;
  this->row=trow;
  dirfinished=0;
}

void NM_extern_fe::setDirFinished(int newv)
{
  dirfinished=newv;
}

int NM_extern_fe::getDirFinished()
{
  return dirfinished;
}

FileEntry *NM_extern_fe::getFE()
{
  return fe;
}

int NM_extern_fe::getRow()
{
  return row;
}

int NormalMode::getSelFiles(List *list,nm_getfiles_t selmode)
{
  return getSelFiles(list,selmode,false);
}

int NormalMode::getSelFiles(List* list,nm_getfiles_t selmode,bool unselect)
{
  int added=0;
  ArrayList *filelist;
  int row;
  NM_specialsource *ss2;
  FileEntry *fe;
  int id;

  if(ce==NULL) return -1;
  if(ce->verz==NULL) return -1;
  if(list==NULL) return -1;

  filelist=ce->verz->getFiles();
  id=filelist->initEnum();

  switch(selmode) {
    case NM_GETFILES_ONLYACTIVE:
      if(ce->activefe!=NULL) {
        row=lv->getActiveRow();
        if(lv->isValidRow(row)==true) {
          if(lv->getData(row)==ce->activefe->nr) {
            if(strcmp(ce->activefe->name,"..")!=0) {
              ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
              ss2->entry=ce->activefe;
              ss2->row=row;
              ss2->cod=NULL;
              list->addElement(ss2);
              added++;
              if(unselect==true) deselect(ss2->entry,ss2->row);
            }
          }
        }
      }
      break;
    default:  // all selected entries
      fe=(FileEntry*)filelist->getFirstElement(id);
      row = 0;
      while(fe!=NULL) {
        if(fe->use==true) {
          if((strcmp(fe->name,"..")!=0)&&(fe->select==true)) {
            ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
            ss2->entry=fe;
            ss2->row=row;
            ss2->cod=NULL;
            list->addElement(ss2);
            added++;
            if(unselect==true) deselect(ss2->entry,ss2->row);
          }
          row++;
        }
        fe=(FileEntry*)filelist->getNextElement(id);
      }
      if(selmode!=NM_GETFILES_ONLYSELECT) {
        if(list->size()==0) {
          // no selected entries => take the active
          if(ce->activefe!=NULL) {
            row=lv->getActiveRow();
            if(lv->isValidRow(row)==true) {
              if(lv->getData(row)==ce->activefe->nr) {
                if(strcmp(ce->activefe->name,"..")!=0) {
                  ss2=(NM_specialsource*)_allocsafe(sizeof(NM_specialsource));
                  ss2->entry=ce->activefe;
                  ss2->row=row;
                  ss2->cod=NULL;
                  list->addElement(ss2);
                  added++;
                  if(unselect==true) deselect(ss2->entry,ss2->row);
                }
              }
            }
          }
        }
      }
      break;
  }
  
  filelist->closeEnum(id);
  
  return added;
}

int NM_Filter::load(Datei *fh)
{
  int chunksize=fh->getInt();
  int len;
  char *str;
  if(chunksize>=(2*Datei::getIntSize())) {
    len=fh->getInt();
    chunksize-=Datei::getIntSize();
    str=fh->getString(len);
    if(pattern!=NULL) _freesafe(pattern);
    pattern=str;
    chunksize-=len;
    check=fh->getInt();
    chunksize-=Datei::getIntSize();
  }
  while(chunksize>0) {
    fh->getUChar();
    chunksize--;
  }
  return 0;
}

int NM_Filter::save(Datei *fh)
{
  int cs=2*Datei::getIntSize();
  if(pattern!=NULL) cs+=strlen(pattern);
  fh->putInt(cs);
  if(pattern!=NULL) fh->putInt(strlen(pattern));
  else fh->putInt(0);
  if(pattern!=NULL) fh->putString(pattern);
  fh->putInt(check);
  return 0;
}

void NormalMode::setSortmode(int nmode)
{
//  if(((nmode&0xff)>=SORT_NAME)&&((nmode&0xff)<=SORT_CHGTIME)) {
  if ( ISVALID_SORTMODE( nmode ) ) {
    nmode&=0xff|SORT_REVERSE|SORT_DIRLAST|SORT_DIRMIXED;
    sortmode=nmode;
    // aktualisieren
    recheck(ce,true);
    setName();
  }
}

void NormalMode::setFilters(List *nfilters)
{
  int id;
  NM_Filter *fi;
  id=filters->initEnum();
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    delete fi;
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  filters->removeAllElements();
  id=nfilters->initEnum();
  fi=(NM_Filter*)nfilters->getFirstElement(id);
  while(fi!=NULL) {
    filters->addElement(fi->duplicate());
    fi=(NM_Filter*)nfilters->getNextElement(id);
  }
  nfilters->closeEnum(id);
  filterlfdnr++;
  // aktualisieren
  recheck(ce,true);
  setName();
}

NM_Filter *NM_Filter::duplicate()
{
  NM_Filter *fi=new NM_Filter();
  fi->setCheck(check);
  fi->setPattern(pattern);
  return fi;
}

void NormalMode::configureFilters(bool *tshowhidden,List *tfilters)
{
  int tw,ttw,tth,ty;
  int t1,t2;
  int trow;
  int pos;
  List *usetf;
  int id;
  NM_Filter *fi;

  usetf=new List();
  id=tfilters->initEnum();
  fi=(NM_Filter*)tfilters->getFirstElement(id);
  while(fi!=NULL) {
    usetf->addElement(fi->duplicate());
    fi=(NM_Filter*)tfilters->getNextElement(id);
  }
  tfilters->closeEnum(id);

  ttw=10;
  tth=50;
  AWindow *win=new AWindow(aguix);
  win->create(NULL,10,10,ttw,tth,0,catalog.getLocale(159));

  ty=5;
  ChooseButton *chb=(ChooseButton*)win->add(new ChooseButton(aguix,5,ty,20,20,
                                            (*tshowhidden==true)?1:0,
                                            catalog.getLocale(357),LABEL_RIGHT,1,0));
  ty+=chb->getHeight()+5;
  FieldListView *filv = (FieldListView*)win->add( new FieldListView( aguix,
                                                                     5,
                                                                     ty,
                                                                     ttw,
                                                                     100,
                                                                     0 ) );
  filv->setHBarState(2);
  filv->setVBarState(2);
  filv->setNrOfFields( 3 );
  filv->setFieldWidth( 1, 1 );
  ty+=filv->getHeight()+5;
  t1=(strlen(catalog.getLocale(167))+2);
  t1*=aguix->getCharWidth();
  Button *newb=(Button*)win->add(new Button(aguix,5,ty,t1,catalog.getLocale(167),1,0,0));
  t1=(strlen(catalog.getLocale(169))+2);
  t1*=aguix->getCharWidth();
  Button *delb=(Button*)win->add(new Button(aguix,newb->getX()+newb->getWidth(),ty,t1,
                                            catalog.getLocale(169),1,0,0));
  t1=(strlen(catalog.getLocale(168))+2);
  t1*=aguix->getCharWidth();
  Button *editb=(Button*)win->add(new Button(aguix,delb->getX()+delb->getWidth(),ty,t1,
                                            catalog.getLocale(168),1,0,0));
  ty+=newb->getHeight();
  ttw=editb->getX()+editb->getWidth()+5;
  
  t1=(strlen(catalog.getLocale(194))+2);
  t1*=aguix->getCharWidth();
  if(t1<(ttw-10)) t1=ttw-10;
  Button *uallb=(Button*)win->add(new Button(aguix,5,ty,t1,
                                             catalog.getLocale(194),1,0,0));
  ty+=uallb->getHeight()+5;
  ttw=uallb->getX()+uallb->getWidth()+5;

  t1=(strlen(catalog.getLocale(11))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>ttw) ttw=tw;
  Button *okb=(Button*)win->add(new Button(aguix,
                                             5,
                                             ty,
                                             t1,
                                             catalog.getLocale(11),
                                             1,
                                             0,
                                             0));
  Button *cancelb=(Button*)win->add(new Button(aguix,
                                                 ttw-5-t2,
                                                 ty,
                                                 t2,
                                                 catalog.getLocale(8),
                                                 1,
                                                 0,
                                                 0));
  filv->resize(ttw-10,filv->getHeight());
  editb->resize(ttw-5-editb->getX(),editb->getHeight());
  tth=okb->getY()+okb->getHeight()+5;
  
  id=usetf->initEnum();
  fi=(NM_Filter*)usetf->getFirstElement(id);
  pos=0;
  while(fi!=NULL) {
    trow = filv->addRow();
    setLVC4Filter( filv, trow, fi );
    filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
    fi=(NM_Filter*)usetf->getNextElement(id);
  }
  usetf->closeEnum(id);
  
  win->resize(ttw,tth);
  win->setMaxSize(ttw,tth);
  win->setMinSize(ttw,tth);
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
        else if(msg->button.button==newb) {
          NM_Filter *tfi=new NM_Filter();
          usetf->addElement(tfi);
          
          trow = filv->addRow();
          setLVC4Filter( filv, trow, tfi );
          filv->setPreColors( trow, FieldListView::PRECOLOR_ONLYSELECT );
          filv->setActiveRow( trow );
          filv->showActive();
          if(configureFilter(tfi)==0) {
            setLVC4Filter( filv, trow, tfi );
            filv->redraw();
          }
        } else if(msg->button.button==delb) {
          pos = 0;
          while ( filv->isValidRow( pos ) == true ) {
            if ( filv->getSelect( pos ) == true ) {
              fi = (NM_Filter*)usetf->getElementAt( pos );
              if ( fi != NULL ) {
                delete fi;
                usetf->removeElementAt( pos );
                filv->deleteRow( pos );
                filv->redraw();
                pos--;
              }
            }
            pos++;
          }
        } else if(msg->button.button==editb) {
          pos = 0;
          while ( filv->isValidRow( pos ) == true ) {
            if ( filv->getSelect( pos ) == true ) {
              fi = (NM_Filter*)usetf->getElementAt( pos );
              if ( fi != NULL ) {
                if ( configureFilter( fi ) == 0 ) {
                  setLVC4Filter( filv, pos, fi );
                  filv->redraw();
                }
              }
            }
            pos++;
          }
        } else if(msg->button.button==uallb) {
          pos = 0;
          id = usetf->initEnum();
          fi = (NM_Filter*)usetf->getFirstElement( id );
          while ( fi != NULL ) {
            fi->setCheck( 0 );
            if ( filv->isValidRow( pos ) == true ) {
              setLVC4Filter( filv, pos, fi );
              pos++;
            }
            fi = (NM_Filter*)usetf->getNextElement( id );
          }
          usetf->closeEnum( id );
          filv->redraw();
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    // insert usetf in tfilters
    id=tfilters->initEnum();
    fi=(NM_Filter*)tfilters->getFirstElement(id);
    while(fi!=NULL) {
      delete fi;
      fi=(NM_Filter*)tfilters->getNextElement(id);
    }
    tfilters->closeEnum(id);
    tfilters->removeAllElements();

    id=usetf->initEnum();
    fi=(NM_Filter*)usetf->getFirstElement(id);
    while(fi!=NULL) {
      tfilters->addElement(fi->duplicate());
      fi=(NM_Filter*)usetf->getNextElement(id);
    }
    usetf->closeEnum(id);
    
    *tshowhidden=(chb->getState()==1)?true:false;
  }
  
  win->close();
  delete win;
  id=usetf->initEnum();
  fi=(NM_Filter*)usetf->getFirstElement(id);
  while(fi!=NULL) {
    delete fi;
    fi=(NM_Filter*)usetf->getNextElement(id);
  }
  usetf->closeEnum(id);
  usetf->removeAllElements();
  delete usetf;
}

void NormalMode::setLVC4Filter( FieldListView *filv, int row, NM_Filter *fi )
{
  if ( filv == NULL ) return;
  if ( fi == NULL ) return;
  char *p=fi->getPattern();
  
  filv->setText( row, 0, ( p != NULL ) ? p : "" );
  if ( fi->getCheck() == 1 ) filv->setText( row, 2, catalog.getLocale( 170 ) );
  else if ( fi->getCheck() == 2 ) filv->setText( row, 2, catalog.getLocale( 171 ) );
  else filv->setText( row, 2, "" );
}

int NormalMode::configureFilter(NM_Filter *fi)
{
  int tw,ttw,tth,tty,ttx;
  int t1,t2;
  char *tstr;

  ttw=10;
  tth=10;
  AWindow *win=new AWindow(aguix);
  win->create(NULL,10,10,ttw,tth,0,catalog.getLocale(172));

  tty=5;
  ttx=5;
  Text *ttext=(Text*)win->add(new Text(aguix,ttx,tty,catalog.getLocale(93),1));
  ttx+=ttext->getWidth()+5;
  tstr=fi->getPattern();
  StringGadget *tsg=(StringGadget*)win->add(new StringGadget(aguix,ttx,tty,100,
                                       (tstr!=NULL)?tstr:"",0));
  tty+=tsg->getHeight()+5;
  tw=ttx+tsg->getWidth()+5;
  if(tw>ttw) ttw=tw;

  ttx=5;
  CycleButton *cyb=(CycleButton*)win->add(new CycleButton(aguix,ttx,tty,100,1,0,0));
  cyb->addOption(catalog.getLocale(358));
  cyb->addOption(catalog.getLocale(170));
  cyb->addOption(catalog.getLocale(171));
  if(fi->getCheck()==1) cyb->setOption(1);
  else if(fi->getCheck()==2) cyb->setOption(2);
  else cyb->setOption(0);
  cyb->resize(cyb->getMaxSize(),cyb->getHeight());

  tty+=cyb->getHeight()+5;
  tw=ttx+cyb->getWidth()+5;
  if(tw>ttw) ttw=tw;
  ttx=5;

  t1=(strlen(catalog.getLocale(11))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>ttw) ttw=tw;
  Button *okb=(Button*)win->add(new Button(aguix,
                                             5,
                                             tty,
                                             t1,
                                             catalog.getLocale(11),
                                             1,
                                             0,
                                             0));
  Button *cancelb=(Button*)win->add(new Button(aguix,
                                                 ttw-5-t2,
                                                 tty,
                                                 t2,
                                                 catalog.getLocale(8),
                                                 1,
                                                 0,
                                                 0));
  tth=okb->getY()+okb->getHeight()+5;
  
  win->resize(ttw,tth);
  win->setMaxSize(ttw,tth);
  win->setMinSize(ttw,tth);
  win->show();
  AGMessage *msg;
  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  int ende=0;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cancelb) ende=-1;
      }
      aguix->ReplyMessage(msg);
    }
  }
  if(ende==1) {
    switch(cyb->getSelectedOption()) {
      case 1:
        fi->setCheck(1);
        break;
      case 2:
        fi->setCheck(2);
        break;
      default:
        fi->setCheck(0);
        break;
    }
    fi->setPattern(tsg->getText());
  }
  
  win->close();
  delete win;
  return (ende==1)?0:1;
}

void NormalMode::recheck(NMCacheEntry *tce,bool forcerebuild)
{
  int oldx,oldy;
  int sh=(showHidden==true)?1:0;
  bool recalced=false;
  
  if(tce!=NULL) {
    if((tce->actfilterlfdnr!=filterlfdnr)||(tce->actshowhidden!=sh)) {
      check(tce);
      tce->actfilterlfdnr=filterlfdnr;
      tce->actshowhidden=sh;
      recalced=true;
      tce->firstnullftcheck();
    }
    if(tce->actsortmode!=sortmode) {
      tce->verz->sort(sortmode);
      tce->actsortmode=sortmode;
      cyclicfunc(CYCLICFUNC_MODE_FIRSTNULLFT);
    }
  }
  if(forcerebuild==true) {
    if(lv!=NULL) {
      oldx=lv->getXOffset();
      oldy=lv->getYOffset();
      buildLister();
      if(ce!=NULL)
        if(ce->activefe!=NULL)
          lv->centerActive();
        else {
          lv->setYOffset(oldy);
          lv->setXOffset(oldx);
        }
      showCacheState();
      lv->redraw();
    }
  }
}

void NormalMode::setName()
{
  bool filteractive;
  int fid=filters->initEnum();
  char *tstr;
  NM_Filter *fil=(NM_Filter*)filters->getFirstElement(fid);
  filteractive=false;
  while(fil!=NULL) {
    if(fil->getCheck()!=0) {
      filteractive=true;
      break;
    }
    fil=(NM_Filter*)filters->getNextElement(fid);
  }
  filters->closeEnum(fid);
  tstr=(char*)_allocsafe(2+2+3+strlen(catalog.getLocale(173))+1);
  sprintf(tstr,"       %s",catalog.getLocale(173));
  switch(sortmode&0xff) {
    case SORT_NAME:
      tstr[0]='N';
      break;
    case SORT_SIZE:
      tstr[0]='S';
      break;
    case SORT_CHGTIME:
      tstr[0]='C';
      break;
    case SORT_MODTIME:
      tstr[0]='M';
      break;
    case SORT_ACCTIME:
      tstr[0]='A';
      break;
    case SORT_TYPE:
      tstr[0]='T';
      break;
    case SORT_OWNER:
      tstr[0]='O';
      break;
    default:
      tstr[0]='?';
      break;
  }
  if((sortmode&SORT_REVERSE)==SORT_REVERSE) tstr[1]='R';
  if(showHidden==false) tstr[2]='H';
  if(filteractive==true) tstr[3]='*';
//  if((showHidden==false)||(filteractive==true)) tstr[3]='-';
  parentlister->setName(tstr);
  if(namestr!=NULL) _freesafe(namestr);
  namestr=tstr;
  showFreeSpace(0);
}

void NormalMode::reconfig()
{
  int side=parentlister->getSide();
  int m;
  AGUIXFont *afont=aguix->getFont(wconfig->getFont(2+side));
  int sgh=afont->getCharHeight()+8;

  lv->setMBG(wconfig->getListerBG());
  m=(wconfig->getHBarTop(side)==true)?1:2;
  lv->setHBarState(m);
  m=(wconfig->getVBarLeft(side)==true)?1:2;
  lv->setVBarState(m);
  m=wconfig->getHBarHeight(side);
  lv->setHBarHeight(m);
  m=wconfig->getVBarWidth(side);
  lv->setVBarWidth(m);
  lv->setFont(wconfig->getFont(2+side));
  sg->setFont(wconfig->getFont(2+side));

  lv->resize(w,h-sgh);
  lv->move(x,y);
  hb[0]->move(x,y+h-sgh);
  hb[0]->resize(15,sgh);
  hb[0]->setFont(wconfig->getFont(2+side));
  hb[1]->move(x+w-15,y+h-sgh);
  hb[1]->resize(15,sgh);
  hb[1]->setFont(wconfig->getFont(2+side));
  sg->move(x+15,y+h-sgh);
  sg->resize(w-30,sgh);
}

void NormalMode::showFreeSpace(bool force)
{
  if(force==true) showFreeSpace(1);
  else showFreeSpace(updatetime);
}

void NormalMode::showFreeSpace(int ti)
{
  char *tstr,buffer1[3],buffer2[3],*tstr2;
  long s1,f1;
  int len;
  time_t now;
  int erg;

  if(showfreespace==true) {
    now=time(NULL);
    if ( ( now - lastfsupdate >= ti ) || ( lasteagain == true ) ) {
      if(namestr==NULL) setName();
      if(namestr!=NULL) {
        if(getActualDir()!=NULL) {
          erg = parentlister->getWorker()->PS_readSpace( getActualDir() );
          if ( erg == 0 ) {
            tstr=buffer1;
            s1=parentlister->getWorker()->PS_getSpaceH(&tstr);
            tstr=buffer2;
            f1=parentlister->getWorker()->PS_getFreeSpaceH(&tstr);
            len=2*sizeof(long)*8+strlen(buffer1)+strlen(buffer2)+strlen(catalog.getLocale(211))+1;
            tstr=(char*)_allocsafe(len);
            sprintf(tstr,catalog.getLocale(211),f1,buffer2,s1,buffer1);
            tstr2=catstring(namestr,tstr);
            _freesafe(tstr);
            parentlister->setName(tstr2);
            _freesafe(tstr2);
            
            lasteagain = false;
          } else if ( erg == EAGAIN ) {
            // no valid value so lets try again next call
            lasteagain = true;
          }
          lastfsupdate=now;
        }
      }
//      lastfsupdate=now;
    }
  }
}

void NormalMode::setShowFreeSpace(bool v)
{
  showfreespace=v;
  setName();
  showFreeSpace(true);
}

void NormalMode::setUpdatetime(int nv)
{
  updatetime=nv;
  if(updatetime<1) updatetime=1;
}

bool NormalMode::startdnd(DNDMsg *dm)
{
  bool returnvalue=false;
  if(dm->getStart()->element==lv) {
    returnvalue=true;

    int row=(int)(dm->getStart()->specialinfo);
    if(lv->isValidRow(row)==true) {
      if(ce!=NULL) {
        if(ce->verz!=NULL) {
          FileEntry *fe=(FileEntry*)ce->verz->getFiles()->getElementAt(lv->getData(row));
          if(fe!=NULL) {
            // found entry for dnd
            // now use the type to call the dnd action

            WCFiletype *ft,*fft[4];
            ActionMessage amsg;
            List *filetypes=wconfig->getFiletypes();
            fft[0]=NULL;
            fft[1]=NULL;
            fft[2]=NULL;
            fft[3]=NULL;

            amsg.startLister=parentlister;
            amsg.mode=amsg.AM_MODE_DNDACTION;

            int e=filetypes->initEnum();
            ft=(WCFiletype*)filetypes->getFirstElement(e);
            while(ft!=NULL) {
              switch(ft->getinternID()) {
                case NOTYETTYPE:
                  fft[0]=ft;
                  break;
                case UNKNOWNTYPE:
                  fft[1]=ft;
                  break;
                case VOIDTYPE:
                  fft[2]=ft;
                  break;
                case DIRTYPE:
                  fft[3]=ft;
                  break;
              }
              ft=(WCFiletype*)filetypes->getNextElement(e);
            }
            filetypes->closeEnum(e);
            if(fe->isDir()==false) {
              ft=fe->filetype;
              if(ft==NULL) ft=fft[0]; // not yet checked
            } else {
              ft=fft[3];
            }
            if(ft==NULL) ft=fft[2];
            if(ft!=NULL) {
              if(ft->getDNDActions()->size()<1) ft=fft[2];
              if(ft!=NULL) {
                amsg.extraInfo=dm;
                amsg.flags=ft->getDNDActionsGF();
                parentlister->getWorker()->interpret(ft->getDNDActions(),&amsg);
              }
            }
          }
        }
      }
    }
  }
  return returnvalue;
}

bool NormalMode::isyours(GUIElement *elem)
{
  if(elem==lv) return true;
  return false;
}

FileEntry *NormalMode::getFE4DNDSTART(AGDNDSTART *st)
{
  int row;
  FileEntry *fe=NULL;

  if(lv!=NULL) {
    row=(int)(st->specialinfo);
    if(lv->isValidRow(row)==true) {
      if(ce!=NULL) {
        if(ce->verz!=NULL) {
          fe=(FileEntry*)ce->verz->getFiles()->getElementAt(lv->getData(row));
        }
      }
    }
  }
  return fe;
}

void NormalMode::unsetAllFilters()
{
  int id;
  NM_Filter *fi;
  id=filters->initEnum();
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    fi->setCheck(0);
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  filterlfdnr++;
  // aktualisieren
  recheck(ce,true);
  setName();
}

void NormalMode::setFilter(const char *filter,nm_filter_t mode)
{
  int newmode;
  int id;
  NM_Filter *fi;
  bool found;
  
  switch(mode) {
    case NM_FILTER_EXCLUDE:
      newmode=2;
      break;
    case NM_FILTER_UNSET:
      newmode=0;
      break;
    default:
      newmode=1;
      break;
  }
  id=filters->initEnum();
  found=false;
  fi=(NM_Filter*)filters->getFirstElement(id);
  while(fi!=NULL) {
    if(strcmp(fi->getPattern(),filter)==0) {
      found=true;
      break;
    }
    fi=(NM_Filter*)filters->getNextElement(id);
  }
  filters->closeEnum(id);
  if(found==true) {
    fi->setCheck(newmode);
  } else {
    fi=new NM_Filter();
    fi->setCheck(newmode);
    fi->setPattern(filter);
    filters->addElement(fi);
  }

  filterlfdnr++;
  // aktualisieren
  recheck(ce,true);
  setName();
}

int NormalMode::runCommand( const char*exestr,
                            const char *tmpname,
                            const char *tmpoutput,
                            bool inbackground )
{
  pid_t child,ret,p2;
#if 0
  AGMessage *msg;
#else
  Message *tmsg;
#endif
  int status,retval;
  
  parentlister->getWorker()->getMainWin()->setCursor(AGUIX::WAIT_CURSOR);
  child=fork();
  if(child!=0) {
    // parent
    if(child==-1) {
      parentlister->getWorker()->getMainWin()->unsetCursor();
      return -2;
    }
    retval=0;
    for(;;) {
      // first answer all messages to be able to redraw
#if 0
      // if use this, user can click buttons, scroll listers, select items
      // but nothing will happen (but selecting should work ok)
      // but actually I don't allow this
      while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
#else
      // here only the Expose-messages are handled
      while((tmsg=aguix->wait4mess(MES_GET,false))!=NULL) {
        if(tmsg->type==Expose) aguix->ExposeHandler(tmsg);
        _freesafe(tmsg);
      }
#endif
      // no check for existence of the child
      ret=waitpid(child,&status,WNOHANG);
      if(ret==child) {
        if(WIFEXITED(status)) {
          // normal exit
#ifdef DEBUG
          printf("child normally exit!\n");
#endif
          retval=WEXITSTATUS(status);
          break;
        } else if(WIFSIGNALED(status)) {
#ifdef DEBUG
          printf("child exited by a signal!\n");
#endif
          retval=-1;
          break;
        }
      }
      // wait some time and the retry
      waittime(100);
    }
    parentlister->getWorker()->getMainWin()->unsetCursor();
    return retval;
  }
  // child, disconnect from parent
  setsid();
  chdir("/");

  if ( inbackground == true ) {
    // when exestr should run in background
    // do another fork so we can here wait for finish of exestr to
    // remove tmp-files but sametime the parent will return to worker
    p2 = fork();
    if ( p2 != 0 ) {
      // parent
      if ( p2 < 0 ) {
        // error
        exit(1);
      } else {
        exit(0);
      }
    }
    setsid();
  }
  // execute command
  // we could use exec* to call this
  //   but then we have to change exestr to give a char**argv
  //   (which is not so hard because it contains only the commands to execute
  //    tmpname)
  //   we also have to make another fork because exec* doesn't return
  //     and so we need an other process to remove the tmp-files after command finished
  system(exestr);
  
  // remove temp files
  remove(tmpname);
  remove(tmpoutput);
  
  // and exit
  exit(0);
  // just for some compiler to compile without warning
  return 0;
}

void NMCacheEntry::checkfordcd()
{
  List *dcd=wconfig->getDontCheckDirs();
  char *tstr;
  const char *dirstr;
  int id;

  dontcheck=false;
  if(verz!=NULL) {
    dirstr=verz->getDir();
    if(dcd!=NULL) {
      id=dcd->initEnum();
      tstr=(char*)dcd->getFirstElement(id);
      while(tstr!=NULL) {
        if(strncmp(tstr,dirstr,strlen(tstr))==0) {
          dontcheck=true;
          break;
        }
        tstr=(char*)dcd->getNextElement(id);
      }
      dcd->closeEnum(id);
    }
  }
}

void NMCacheEntry::reset_dirsizes()
{
  if(verz==NULL) return;
  ArrayList *filelist=verz->getFiles();
  int id=filelist->initEnum();
  FileEntry *fe=(FileEntry*)filelist->getFirstElement(id);
  while(fe!=NULL) {
    if(fe->isDir()==true) {
      fe->dirsize=-1;
    }
    fe=(FileEntry*)filelist->getNextElement(id);
  }
  filelist->closeEnum(id);
}

#ifdef WANT_THREADS
void NormalMode::slavehandler()
{
  const char *fname;
  char *filename;
  ft_recres_list::ft_recres_list_t *te;
  bool ende;
  int erg;
  bool dontCheckContent;
  int ftindex;
  
  // to get from REINIT directly to RUN
  bool firstrun = true;

  if ( slave.running != 0 ) {
    fprintf( stderr, "Worker: another thread already running!\n");
    return;
  }
  slave.running = 1;
  slave.status = THREAD_REINIT;
  slave.order = THREAD_NOP;

  slave.filetype_ex.lock();

#ifdef DEBUG
  printf("entering slave handler\n");
#endif

  for( ende = false; ende == false; ) {

    switch ( slave.status ) {
      case THREAD_RUN:
#ifdef DEBUG
        printf("waiting for element\n");
#endif

        reclist->lock();
        // wait for next element or stop
        while ( ( reclist->elemAvail_locked() == false ) && ( slave.order == THREAD_NOP ) )
          reclist->wait();

#ifdef DEBUG
        printf("wait finished\n");
#endif

        if ( slave.order != THREAD_NOP ) {
          // new command
          reclist->unlock();
          
          slave.waitvar.lock();
        
          // switch mode for new command
          if ( slave.order == THREAD_EXIT ) {
            slave.status = THREAD_EXIT;
            ende = true;
          } else if ( slave.order == THREAD_REINIT ) {
            slave.status = THREAD_REINIT;
          } else if ( slave.order == THREAD_WAIT ) {
            slave.status = THREAD_WAIT;
          }
          // mark this command read
          slave.order = THREAD_NOP;
          
          // wake up master
          slave.waitvar.signal();
          slave.waitvar.unlock();
        } else {
          // an element to check
          erg = reclist->slaveread_locked( &fname, &dontCheckContent );
          filename = NULL;
          if ( erg == 0 ) {
            if ( fname != NULL ) {
              filename = dupstring( fname );
            }
          }
          reclist->unlock();

          if ( ( erg == 0 ) && ( filename != NULL ) ) {
#ifdef DEBUG
//            printf("erkenne %s\n",filename);
#endif
            ftindex = slave_checkFiletype( filename, dontCheckContent );
            recreslist->lock();
    
            te = new ft_recres_list::ft_recres_list_t;
            te->name = filename;
            te->ft_index = ftindex;
    
            recreslist->put_locked( te );
            recreslist->unlock();
          }
        }
        break;
      case THREAD_WAIT:
        slave.waitvar.lock();
        slave.waitvar.wait();
 
        // new command?
        if ( slave.order == THREAD_EXIT ) {
          slave.status = THREAD_EXIT;
          ende = true;
        } else if ( slave.order == THREAD_REINIT ) {
          slave.status = THREAD_REINIT;
        } else if ( slave.order == THREAD_RUN ) {
          slave.status = THREAD_RUN;
        }
        slave.order = THREAD_NOP;
        
        // wake up master
        slave.waitvar.signal();
        slave.waitvar.unlock();
        break;
      case THREAD_REINIT:
        // wake up master
        slave.waitvar.signal();
        // wait for change
        slave.filetype_ex.signal();
        slave.filetype_ex.wait();
        
        // change mode
        if ( firstrun == true ) {
          firstrun = false;
          slave.status = THREAD_RUN;
        } else {
          slave.status = THREAD_WAIT;
        }
        break;
      default:
        break;
    }
  }
    
  slave.filetype_ex.unlock();

#ifdef DEBUG
  printf("leaving slave handler\n");
#endif
}

int NormalMode::ft_rec_list::slaveread_locked( const char **name, bool *dontCheckContent )
{
  if ( ( name == NULL ) || ( dontCheckContent == NULL ) ) return 1;
  if ( readp != NULL ) {
    *name = readp->name;
    *dontCheckContent = readp->dontCheckContent;
    readp = readp->next;
  } else return 1;
  return 0;
}

bool NormalMode::ft_rec_list::isFull_locked()
{
  if ( elements >= size ) return true;
  return false;
}

bool NormalMode::ft_rec_list::isEmpty_locked()
{
  if ( elements < 1 ) return true;
  return false;
}

int NormalMode::ft_rec_list::put_locked( ft_rec_list_t *elem )
{
  int pos;

  if ( elem == NULL ) return -1;
  if ( isFull_locked() == true ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
    // if readp is null, slave has read all elems, so set it to this new elem
    if ( readp == NULL )
      readp = tail;
  } else {
    head = tail = elem;
    readp = tail;
  }

  pos = elements++;
  return pos;
}

NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::remove_locked()
{
  ft_rec_list_t *te;

  if ( elements == 0 ) return NULL;
  
  // readp could be the element to remove
  if ( readp == head )
    readp = readp->next;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

NormalMode::ft_rec_list::ft_rec_list()
{
  size = 10;
  head = NULL;
  tail = NULL;
  readp = NULL;
  elements = 0;
}

NormalMode::ft_rec_list::~ft_rec_list()
{
  ft_rec_list_t *te;

  lock();
  while ( isEmpty_locked() == false ) {
    te = remove_locked();
    delete te;
  }
  unlock();
}

void NormalMode::ft_rec_list::lock()
{
  ex.lock();
}

void NormalMode::ft_rec_list::unlock()
{
  ex.unlock();
}

void NormalMode::ft_rec_list::wait()
{
  ex.wait();
}

void NormalMode::ft_rec_list::signal()
{
  ex.signal();
}

bool NormalMode::ft_rec_list::elemAvail_locked()
{
  if ( readp != NULL ) return true;
  return false;
}

NormalMode::ft_recres_list::ft_recres_list()
{
  head = tail = NULL;
  elements = 0;
}

NormalMode::ft_recres_list::~ft_recres_list()
{
  ft_recres_list_t *rte;

  lock();
  while ( isEmpty_locked() == false ) {
    rte = remove_locked();
    if ( rte != NULL ) {
      _freesafe( rte->name );
      delete rte;
    }
  }
  unlock();
}

bool NormalMode::ft_recres_list::isEmpty_locked()
{
  if ( elements == 0 ) return true;
  return false;
}

int NormalMode::ft_recres_list::put_locked( ft_recres_list_t *elem )
{
  if ( elem == NULL ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
  } else {
    head = tail = elem;
  }

  elements++;
  return 0;
}

NormalMode::ft_recres_list::ft_recres_list_t *NormalMode::ft_recres_list::remove_locked()
{
  ft_recres_list_t *te;

  if ( elements == 0 ) return NULL;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

void NormalMode::ft_recres_list::lock()
{
  ex.lock();
}

void NormalMode::ft_recres_list::unlock()
{
  ex.unlock();
}


NormalMode::ft_rec_list::ft_rec_list_t *NormalMode::ft_rec_list::gettop_locked()
{
  return head;
}

void NormalMode::ft_rec_list_clear()
{
  ft_rec_list::ft_rec_list_t *te;
  NMCacheEntry *tce;
  int id;

  reclist->lock();
  while ( reclist->isEmpty_locked() == false ) {
    te = reclist->remove_locked();
    delete te;
  }
  // reset checkfe for all caches
  id = cache->initEnum();
  tce = (NMCacheEntry*)cache->getFirstElement( id );
  while ( tce != NULL ) {
    tce->checkfe = tce->firstnullft;
    tce = (NMCacheEntry*)cache->getNextElement( id );
  }
  cache->closeEnum( id );
  reclist->unlock();
}

void NormalMode::ft_recres_list_clear()
{
  ft_recres_list::ft_recres_list_t *rte;

  recreslist->lock();
  while ( recreslist->isEmpty_locked() == false ) {
    // alle Resultate entnehmen
    rte = recreslist->remove_locked();
    if ( rte != NULL ) {
      _freesafe( rte->name );
      delete rte;
    }
  }
  recreslist->unlock();
}

/*
 * ft_list_clear
 *
 * will delete all filetypes from list
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_clear()
{
  WCFiletype *ft;
  int id;

  slave.filetype_ex.lock();
  
  id = slave.filetypes->initEnum();
  ft = (WCFiletype*)slave.filetypes->getFirstElement( id );
  while ( ft != NULL ) {
    delete ft;
    slave.filetypes->removeFirstElement();
    ft = (WCFiletype*)slave.filetypes->getFirstElement( id );
  }
  slave.filetypes->closeEnum( id );
  
  slave.filetype_ex.unlock();
}

/*
 * ft_list_update
 *
 * will update the filetype-list to the current config
 *
 * only allowed for master thread
 */
void NormalMode::ft_list_update()
{
  List *ftlist;
  WCFiletype *ft;
  int id;
  
  slave.filetype_ex.lock();

  ft_list_clear();

  ftlist = wconfig->getFiletypes();
  id = ftlist->initEnum();
  ft = (WCFiletype*)ftlist->getFirstElement( id );
  while ( ft != NULL ) {
    slave.filetypes->addElement( ft->duplicate() );
    ft = (WCFiletype*)ftlist->getNextElement( id );
  }
  ftlist->closeEnum( id );
  
  slave.filetype_ex.unlock();
}

int NormalMode::slave_checkFiletype( const char *fullname, bool dontCheckContent )
{
  FileEntry *fe1;
  int pos = -1;
  WCFiletype *ft;

  slave.filetype_ex.lock();

  fe1 = new FileEntry();
  fe1->fullname = dupstring( fullname );
  fe1->name = Datei::getFilenameFromPath( fullname );
  fe1->nr = 0;
  if( fe1->readInfos() == 0 ) {
    ft = fe1->checkFiletype( slave.filetypes, dontCheckContent );
    if ( ft != NULL ) {
      pos = slave.filetypes->getIndex( ft );
    }
  }
  delete fe1;

  slave.filetype_ex.unlock();
  return pos;
}
#endif

void NormalMode::setupLVFields()
{
  int side = parentlister->getSide();
  int *sets;
  int i;
  int used_fields;

  sets = wconfig->getDisplay(side);
  for( i = 0, used_fields = 0; i < 9; i++ ) {
    if(sets[i]<0) break;
    used_fields++;
  }
  
  if ( used_fields == 0 ) {
    // oops, what's this?
    lv->setNrOfFields( 1 );  // 0 fields are not supported, blame the author (me ;-) )
  } else {
    lv->setNrOfFields( used_fields * 2 - 1 );
  
    for( i = 0; i < 9; i++ ) {
      if(sets[i]<0) break;
      if ( i > 0 ) {
        lv->setFieldWidth( 2 * i -1, 1 );
      }
      switch(sets[i]) {
        case 0://Name
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
          break;
        case 1://Size
#ifdef LEFTJUSTIFY
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
#else
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_RIGHT );
#endif
          break;
        case 2://Type
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
          break;
        case 3://Perm
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
          lv->setFieldWidth( 2 * i, 9 );
          break;
        case 4://Owner
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
          break;
        case 5://Dest
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
          break;
        case 6://Mod
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
          break;
        case 7://Acc
          lv->setFieldAlign( 2* i, FieldListView::ALIGN_LEFT );
          break;
        case 8://Change
          lv->setFieldAlign( 2 * i, FieldListView::ALIGN_LEFT );
          break;
      }
    }
  }
}

