/* xdaliclock - a melting digital clock
 * Copyright (c) 1991-2010 Jamie Zawinski <jwz@jwz.org>
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#import "DaliClockView.h"
#import "xdaliclock.h"

#import <sys/time.h>

#if MAC_OS_X_VERSION_MAX_ALLOWED <= MAC_OS_X_VERSION_10_4
  /* In 10.5 and later, NSColor wants CGfloat args, but that type
     does not exist in 10.4. */
  typedef float CGFloat;
#endif


static int to_pow2 (int i);
static void check_gl_error (const char *type);

#ifdef USE_IPHONE
static void rgb_to_hsv (CGFloat r, CGFloat g, CGFloat b, 
                        CGFloat *h, CGFloat *s, CGFloat *v);
#endif /* USE_IPHONE */


@interface DaliClockView (ForwardDeclarations)

+ (NSUserDefaultsController *)userDefaultsController;
+ (void)setUserDefaultsController:(NSUserDefaultsController *)ctl;

- (void)setForeground:(NSColor *)fg background:(NSColor *)bg;

- (void)clockTick;
- (void)colorTick;
- (void)dateTick;
@end


@implementation DaliClockView

static NSUserDefaultsController *staticUserDefaultsController = 0;

+ (NSUserDefaultsController *)userDefaultsController
{
  return staticUserDefaultsController;
}


+ (void)initializeDefaults:(NSUserDefaultsController *)controller
{
  static BOOL initialized_p = NO;
  if (initialized_p)
    return;
  initialized_p = YES;

  staticUserDefaultsController = controller;
  [controller retain];

  /* Set the defaults for all preferences handled by DaliClockView.
     (AppController or DaliClockSaverView handles other preferences).

     Depending on class-initialization order, another class (e.g.
     AppController) might have already put defaults in the preferences
     object, so make sure we add ours instead of overwriting it all.
   */
  NSColor *deffg = [NSColor  blueColor];
  NSColor *defbg = [NSColor colorWithCalibratedHue:0.50
                                        saturation:1.00
                                        brightness:0.70   // cyan, but darker
# ifndef USE_IPHONE
                                             alpha:0.30   // and translucent
# else  /* USE_IPHONE */
                                             alpha:1.00
# endif /* USE_IPHONE */
                            ];

  NSDate *defdate = [NSDate dateWithTimeIntervalSinceNow:
                     (NSTimeInterval) 60 * 60 * 12];  // 12 hours from now

  NSDictionary* defaults = [NSDictionary dictionaryWithObjectsAndKeys:
    @"0", @"hourStyle",
    @"0", @"timeStyle",
    @"0", @"dateStyle",
    [NSArchiver archivedDataWithRootObject:deffg], @"initialForegroundColor",
    [NSArchiver archivedDataWithRootObject:defbg], @"initialBackgroundColor",
    @"10.0", @"cycleSpeed",
    @"NO", @"usesCountdownTimer",
    defdate,@"countdownDate",
    nil];

  NSMutableDictionary *dict = [NSMutableDictionary dictionaryWithCapacity:100];

# ifndef USE_IPHONE
  [dict addEntriesFromDictionary: [controller initialValues]];
  [dict addEntriesFromDictionary:defaults];
  [controller setInitialValues:dict];
  [[controller defaults] registerDefaults:dict];
# else  /* USE_IPHONE */
  [dict addEntriesFromDictionary:defaults];
  [[NSUserDefaults standardUserDefaults] registerDefaults:dict];
# endif /* USE_IPHONE */
}


- (void)bindPreferences
{
# ifndef USE_IPHONE
  NSUserDefaultsController *controller = staticUserDefaultsController;

  [self    bind:@"hourStyle"
       toObject:controller
    withKeyPath:@"values.hourStyle" options:nil];
  [self    bind:@"timeStyle"
       toObject:controller
    withKeyPath:@"values.timeStyle"
        options:nil];
  [self    bind:@"dateStyle"
       toObject:controller
    withKeyPath:@"values.dateStyle"
        options:nil];
  [self    bind:@"cycleSpeed"
       toObject:controller
    withKeyPath:@"values.cycleSpeed"
        options:nil];
  [self    bind:@"usesCountdownTimer"
       toObject:controller
    withKeyPath:@"values.usesCountdownTimer"
        options:nil];
  [self    bind:@"countdownDate"
       toObject:controller
    withKeyPath:@"values.countdownDate"
        options:nil];

  NSDictionary *colorBindingOptions =
    [NSDictionary dictionaryWithObject:@"NSUnarchiveFromData"
                                forKey:NSValueTransformerNameBindingOption];
  [self    bind:@"initialForegroundColor"
       toObject:controller
    withKeyPath:@"values.initialForegroundColor"
        options:colorBindingOptions];
  [self    bind:@"initialBackgroundColor"
       toObject:controller
    withKeyPath:@"values.initialBackgroundColor"
        options:colorBindingOptions];

# else  /* USE_IPHONE */

  // WTF, why don't we have bindings for this!!
  NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];

  [self setHourStyle:[defaults integerForKey:@"hourStyle"]];
  [self setTimeStyle:[defaults integerForKey:@"timeStyle"]];
  [self setDateStyle:[defaults integerForKey:@"dateStyle"]];
  [self setCycleSpeed:[defaults floatForKey:@"cycleSpeed"]];
  [self setUsesCountdownTimer:[defaults boolForKey:@"usesCountdownTimer"]];
  [self setCountdownDate:(NSDate *)[defaults objectForKey:@"countdownDate"]];

  /* Is it a problem that I'm using NSKeyedUnarchiver on an object
     that was created with NSArchiver instead of NSKeyedArchiver?
     Does the NSUnarchiveFromData NSValueTransformerNameBindingOption
     work with both NSArchiver and NSKeyedArchiver objects?
   */
  NSData *ifgd = [defaults objectForKey:@"initialForegroundColor"];
  NSData *ibgd = [defaults objectForKey:@"initialBackgroundColor"];
  UIColor *ifg = (ifgd ? [NSKeyedUnarchiver unarchiveObjectWithData:ifgd] : 0);
  UIColor *ibg = (ibgd ? [NSKeyedUnarchiver unarchiveObjectWithData:ibgd] : 0);
  if (! ifg) ifg = [UIColor whiteColor];  // This shouldn't happen...
  if (! ibg) ibg = [UIColor blackColor];
  [self setInitialForegroundColor:ifg];
  [self setInitialBackgroundColor:ibg];
# endif /* USE_IPHONE */
}


- (id)initWithFrame:(NSRect)rect
{
  self = [super initWithFrame:rect];
  if (! self) return nil;

  memset (&config, 0, sizeof(config));

# ifndef USE_IPHONE
  config.max_fps = 12;
#else /* USE_IPHONE */
  config.max_fps = 30;
#endif /* USE_IPHONE */

  [self setOwnWindow:YES];  // by default, we may change window background

# ifndef USE_IPHONE
  // Tell the color selector widget to show the "opacity" slider.
  [[NSColorPanel sharedColorPanel] setShowsAlpha:YES];
# endif /* !USE_IPHONE */


# ifdef USE_IPHONE
  [self setMultipleTouchEnabled:YES];
# endif /* USE_IPHONE */


  // initialize the fonts and bitmaps
  //
  [self setFrameSize:[self frame].size];
  [self bindPreferences];

  [self clockTick];
  [self colorTick];


  // Initialize the OpenGL context.
  //
# ifndef USE_IPHONE
  NSOpenGLPixelFormatAttribute attrs[] = {
//    NSOpenGLPFADoubleBuffer,  // does double buffering help at all?
    NSOpenGLPFAColorSize, 24,
    NSOpenGLPFAAlphaSize, 8,
    NSOpenGLPFADepthSize, 0, //16,  we don't need a depth buffer at all.
    0 };
  NSOpenGLPixelFormat *pixfmt = [[NSOpenGLPixelFormat alloc] 
                                  initWithAttributes:attrs];
  ogl_ctx = [[NSOpenGLContext alloc] 
              initWithFormat:pixfmt
                shareContext:nil];

#  if MAC_OS_X_VERSION_MAX_ALLOWED <= MAC_OS_X_VERSION_10_4
  long svarg;
#  else  /* 10.5+ */
  GLint svarg;
#  endif /* 10.5+ */

  // Sync refreshes to the vertical blanking interval
  svarg = 1;
  [ogl_ctx setValues:&svarg forParameter:NSOpenGLCPSwapInterval];

  // Make window transparency possible
  svarg = NO;
  [ogl_ctx setValues:&svarg forParameter:NSOpenGLCPSurfaceOpacity];


  [ogl_ctx makeCurrentContext];

  // Clear frame buffer ASAP, else there are bits left over from other apps.
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Enable multi-threading, if possible.  This runs most OpenGL commands
  // and GPU management on a second CPU.
  {
#   ifndef  kCGLCEMPEngine
#    define kCGLCEMPEngine 313  // Added in MacOS 10.4.8 + XCode 2.4.
#   endif
    CGLContextObj cctx = CGLGetCurrentContext();
    CGLError err = CGLEnable (cctx, kCGLCEMPEngine);
    if (err != kCGLNoError) {
      NSLog (@"enabling multi-threaded OpenGL failed: %d", err);
    }
  }

  {
    GLboolean d = 0;
    glGetBooleanv (GL_DOUBLEBUFFER, &d);
    if (d)
      glDrawBuffer (GL_BACK);
    else
      glDrawBuffer (GL_FRONT);
  }

  check_gl_error ("init");

#else  /* USE_IPHONE */

  // Get the layer
  CAEAGLLayer *eagl_layer = (CAEAGLLayer *)self.layer;
  eagl_layer.opaque = TRUE;
  eagl_layer.drawableProperties = 
    [NSDictionary dictionaryWithObjectsAndKeys:
      [NSNumber numberWithBool:FALSE], kEAGLDrawablePropertyRetainedBacking,
      kEAGLColorFormatRGBA8,           kEAGLDrawablePropertyColorFormat,
      nil];
  eagl_ctx = [[EAGLContext alloc] initWithAPI:kEAGLRenderingAPIOpenGLES1];

  if (!eagl_ctx || ![EAGLContext setCurrentContext:eagl_ctx]) {
    [self release];
    return nil;
  }

  glGenFramebuffersOES (1, &gl_framebuffer);
  glGenRenderbuffersOES (1, &gl_renderbuffer);
  glBindFramebufferOES (GL_FRAMEBUFFER_OES, gl_framebuffer);
  glBindRenderbufferOES (GL_RENDERBUFFER_OES, gl_renderbuffer);

  [eagl_ctx renderbufferStorage:GL_RENDERBUFFER_OES
            fromDrawable:(CAEAGLLayer*)self.layer];

  glFramebufferRenderbufferOES (GL_FRAMEBUFFER_OES, GL_COLOR_ATTACHMENT0_OES,
                                GL_RENDERBUFFER_OES, gl_renderbuffer);

  if (glCheckFramebufferStatusOES(GL_FRAMEBUFFER_OES) !=
      GL_FRAMEBUFFER_COMPLETE_OES) {
    NSLog (@"incomplete framebuffer");
    abort();
  }

  check_gl_error ("OES_init");
#endif /* USE_IPHONE */

  return self;
}


#ifdef USE_IPHONE
- (void)setFrame:(NSRect)r
{
  [super setFrame:r];
  [self setFrameSize:r.size];
}
#endif /* USE_IPHONE */


/* Called when the View is resized.
 */
- (void)setFrameSize:(NSSize)newSize
{
# ifndef USE_IPHONE
  [ogl_ctx clearDrawable];  // need this to cause the vp change to stick
  [super setFrameSize:newSize];

  /* If the user is interactively resizing the window, don't regenerate
     the pixmap until we're done.  (We will just scale whatever image is
     already on the window instead, which reduces flicker when the
     target bitmap size shifts over a boundary).
   */
  if ([self inLiveResize])  return;
# endif /* !USE_IPHONE */

  int ow = config.width;
  int oh = config.height;

   config.width  = newSize.width * 2;   // use the next-larger bitmap
   config.height = newSize.height * 2;
//  config.width = 1280;    // always use the biggest font image
//  config.height = 1024;

  render_bitmap_size (&config, &config.width, &config.height);

  if (config.render_state && (ow == config.width && oh == config.height))
    return;  // nothing to do

  /* When the window is resized, re-create the bitmaps for the largest
     font that will now fit in the window.
   */
  if (config.bitmap) free (config.bitmap);
  config.bitmap = calloc (1, config.height * (config.width << 3));
  if (! config.bitmap) abort();

  if (pixmap) free (pixmap);
  pixmap = calloc (1, config.height * config.width * 4);
  if (! pixmap) abort();

  if (config.render_state)
    render_free (&config);
  render_init (&config);
}


#ifdef USE_IPHONE

- (void)setViewController:(UIViewController*)ctl
{
  viewController = ctl;
}

+ (Class) layerClass
{
    return [CAEAGLLayer class];
}


/* In the simulator, multi-touch sequences look like this:

     touchesBegan [touchA, touchB]
     touchesEnd [touchA, touchB]

   But on real devices, sometimes you get that, but sometimes you get:

     touchesBegan [touchA, touchB]
     touchesEnd [touchB]
     touchesEnd [touchA]

   Or even

     touchesBegan [touchA]
     touchesBegan [touchB]
     touchesEnd [touchA]
     touchesEnd [touchB]

   So the only way to properly detect a "pinch" gesture is to remember
   the start-point of each touch as it comes in; and the end-point of
   each touch as those come in; and only process the gesture once the
   number of touchEnds matches the number of touchBegins.
 */

struct touch_data {
  int down, up;
  CFMutableDictionaryRef start, end;
};

- (void)touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event
{
  if (! touchData) {
    touchData = (touch_data *) calloc (1, sizeof(touch_data));
    touchData->start = CFDictionaryCreateMutable (kCFAllocatorDefault,10,0,0);
    touchData->end   = CFDictionaryCreateMutable (kCFAllocatorDefault,10,0,0);
  }

  // Keys of the dictionary are UITouch objects; values are CGPoint structs.
  for (UITouch *touch in touches) {
    CGPoint *p = (CGPoint *) CFDictionaryGetValue (touchData->start, touch);
    if (!p) {
      p = (CGPoint *) malloc (sizeof(*p));
      *p = [touch locationInView:self];
      CFDictionarySetValue (touchData->start, touch, p);
      touchData->down++;
    }
  }
}


- (void)touchesEnded:(NSSet *)touches withEvent:(UIEvent *)event
{
  int which = 0;

  for (UITouch *touch in touches) {
    CGPoint *p = (CGPoint *) CFDictionaryGetValue (touchData->end, touch);
    if (!p) {
      p = (CGPoint *) malloc (sizeof(*p));
      CFDictionarySetValue (touchData->end, touch, p);
      touchData->up++;
    }
    *p = [touch locationInView:self];
  }

  if (touchData->down != touchData->up)
    return;  // not done touching yet

  if (touchData->down > 1) {				// Multi-touch
    CGPoint start1, start2, end1, end2;
    int i;
    const void *values[10];
    CFDictionaryGetKeysAndValues (touchData->start, 0, values);
    for (i = 0; i < CFDictionaryGetCount (touchData->start); i++) {
      CGPoint *p = (CGPoint *) values[i];
      if (i == 0) start1 = *p;
      else if (i == 1) start2 = *p;
    }
    CFDictionaryGetKeysAndValues (touchData->end, 0, values);
    for (i = 0; i < CFDictionaryGetCount (touchData->end); i++) {
      CGPoint *p = (CGPoint *) values[i];
      if (i == 0) end1 = *p;
      else if (i == 1) end2 = *p;
    }

    double start_dist = 
      sqrt (((start1.x - start2.x) * (start1.x - start2.x)) +
            ((start1.y - start2.y) * (start1.y - start2.y)));
    double end_dist = 
      sqrt (((end1.x - end2.x) * (end1.x - end2.x)) +
            ((end1.y - end2.y) * (end1.y - end2.y)));

    if (start_dist > end_dist) {			// Pinch in
      which = 1;
    } else if (start_dist < end_dist) {			// Pinch out
      which = 2;
    }
  } else if ([touches count] == 1) {
    UITouch *touch = [touches anyObject];
    if ([touch tapCount] == 1)				// Single tap
      which = 3;
    else						// Double tap
      which = 4;
  }


  switch (which) {
  case 1:						// Smaller digits
    config.display_date_p = 0;
    if ([self timeStyle] == SS)        [self setTimeStyle:HHMM];
    else if ([self timeStyle] == HHMM) [self setTimeStyle:HHMMSS];
    break;

  case 2:						// Bigger digits
    config.display_date_p = 0;
    if ([self timeStyle] == HHMMSS)    [self setTimeStyle:HHMM];
    else if ([self timeStyle] == HHMM) [self setTimeStyle:SS];
    break;

  case 3:						// Display date
    config.display_date_p = 1;
    float delay = 2.0;  // seconds
    [NSTimer scheduledTimerWithTimeInterval:delay
             target:self
             selector:@selector(dateOff)
             userInfo:nil
             repeats:NO];
    [self aboutClick:nil];
    break;

  case 4:						// Toggle 24-hour mode
    config.display_date_p = 0;
    if (config.time_mode == SS) [self setTimeStyle:HHMM];
    [self setHourStyle: ![self hourStyle]];
    break;

  case 0: break;
  default: abort(); break;
  }

  // Empty out the dictionaries for next time.
  [self touchesCancelled:touches withEvent:event];

  // Save the current state into the preferences, such as they are.
  //
  NSUserDefaults *controller = [NSUserDefaults standardUserDefaults];
  [controller setObject:[NSNumber numberWithInt:[self timeStyle]]
              forKey:@"timeStyle"];
  [controller setObject:[NSNumber numberWithInt:[self hourStyle]]
              forKey:@"hourStyle"];
  [controller synchronize];
}


- (void)touchesCancelled:(NSSet *)touches withEvent:(UIEvent *)event
{
  if (touchData) {
    int i;
    const void *values[10];
    CFDictionaryGetKeysAndValues (touchData->start, 0, values);
    for (i = 0; i < CFDictionaryGetCount (touchData->start); i++) {
      free ((void *) values[i]);
    }
    CFDictionaryGetKeysAndValues (touchData->end, 0, values);
    for (i = 0; i < CFDictionaryGetCount (touchData->end); i++) {
      free ((void *) values[i]);
    }
    CFDictionaryRemoveAllValues (touchData->start);
    CFDictionaryRemoveAllValues (touchData->end);
    touchData->down = 0;
    touchData->up = 0;
  }
}


- (void)motionEnded:(UIEventSubtype)motion withEvent:(UIEvent *)event
{
  if (event.type == UIEventSubtypeMotionShake) {
    config.test_hack = '0' + (random() % 11);  /* 0-9 or : */
  }
}


/* We need this to respond to "shake" gestures
 */
- (BOOL)canBecomeFirstResponder {
  return YES;
}
 

#else /* !USE_IPHONE */


/* The top-level window is sometimes transparent.
 */
- (BOOL)isOpaque
{
  return NO;
}


/* Called when the user starts interactively resizing the window.
 */
- (void)viewWillStartLiveResize
{
}


/* Called when the user is done interactively sizing the window.
 */
- (void)viewDidEndLiveResize
{
  // Resize the frame one last time, now that we're finished dragging.
  [self setFrameSize:[self frame].size];
}


/* Announce our willingness to accept keyboard input.
 */
- (BOOL)acceptsFirstResponder
{
  return YES;
}

/* Display date when mouse clicked in window.
 */
- (void)mouseDown:(NSEvent *)ev
{
  config.display_date_p = 1;
}


/* Back to time display shortly after mouse released.
 */
- (void)mouseUp:(NSEvent *)ev
{
  float delay = 2.0;  // seconds
  if (config.display_date_p)
    [NSTimer scheduledTimerWithTimeInterval:delay
                                     target:self
                                   selector:@selector(dateOff)
                                   userInfo:nil
                                    repeats:NO];
}


/* Typing Ctrl-0 through Ctrl-9 and Ctrl-hyphen are a debugging hack.
 */
- (void)keyDown:(NSEvent *)ev
{
  NSString *ns = [ev charactersIgnoringModifiers];
  if (! [ns canBeConvertedToEncoding:NSASCIIStringEncoding])
    goto FAIL;
  const char *s = [ns cStringUsingEncoding:NSASCIIStringEncoding];
  if (! s) goto FAIL;
  if (strlen(s) != 1) goto FAIL;
  if (! ([ev modifierFlags] & NSControlKeyMask)) goto FAIL;
  if (*s == '-' || (*s >= '0' && *s <= '9'))
    config.test_hack = *s;
  else goto FAIL;
  return;
FAIL:
  [super keyDown:ev];
}

#endif /* !USE_IPHONE */


- (void)dateOff
{
  config.display_date_p = 0;

# ifdef USE_IPHONE
  if (aboutBox) {
    [aboutBox removeFromSuperview];
    aboutBox = 0;
  }
# endif /* USE_IPHONE */

}


/* This is called from the timer (and other places) to change the colors.
 */
- (void)setForeground:(NSColor *)new_fg background:(NSColor *)new_bg
{
  if (fg != new_fg) {
    if (fg) [fg release];
# ifdef USE_IPHONE
    fg = [new_fg retain];
# else  /* !USE_IPHONE */
    fg = [[new_fg colorUsingColorSpaceName:NSCalibratedRGBColorSpace] retain];
# endif /* !USE_IPHONE */
  }
  if (bg != new_bg) {
    if (bg) [bg release];
# ifdef USE_IPHONE
    bg = [new_bg retain];
# else  /* !USE_IPHONE */
    bg = [[new_bg colorUsingColorSpaceName:NSCalibratedRGBColorSpace] retain];
# endif /* !USE_IPHONE */
  }

# ifdef USE_IPHONE
//    [self drawRect:[self frame]];
# else  /* !USE_IPHONE */
  [self setNeedsDisplay:TRUE];
# endif /* !USE_IPHONE */
}


/* The big kahuna refresh method.  Draw the current contents of the bitmap
   onto the window.  Sounds easy, doesn't it?
 */
- (void)drawRect:(NSRect)rect
{
  NSRect framerect = [self frame];
  NSRect torect;
  BOOL rotated_p = NO;

  if (config.width <= 0 || config.height <= 0) abort();

  if (!fg || !bg) return; // Called too early somehow?

# ifdef USE_IPHONE
  UIInterfaceOrientation orientation = [viewController interfaceOrientation];
  if (orientation == UIInterfaceOrientationLandscapeLeft ||
      orientation == UIInterfaceOrientationLandscapeRight) {
    CGFloat swap = framerect.size.width;
    framerect.size.width = framerect.size.height;
    framerect.size.height = swap;
    rotated_p = YES;
  }
# endif // USE_IPHONE

  float img_aspect = (float) config.width / (float) config.height;
  float win_aspect = framerect.size.width / (float) framerect.size.height;

  // Scale the image to fill the window without changing its aspect ratio.
  //
  if (win_aspect > img_aspect) {
    torect.size.height = framerect.size.height;
    torect.size.width  = framerect.size.height * img_aspect;
  } else {
    torect.size.width  = framerect.size.width;
    torect.size.height = framerect.size.width / img_aspect;
  }

  // The animation slows down a lot if we use truly gigantic numbers,
  // so limit the number size in screensaver-mode.
  //
  if (constrainSizes) {
    int maxh = (config.time_mode == SS ? 512 : /*256*/ 200);
    if (torect.size.height > maxh) {
      torect.size.height = maxh;
      torect.size.width  = maxh * img_aspect;
    }
  }

  // put a margin between the numbers and the edge of the window.
  torect.size.width  *= 0.95;  // 5% horizontally
  torect.size.height *= 0.80;  // 20% vertically

  // center it in the window
  //
  torect.origin.x = (framerect.size.width  - torect.size.width ) / 2;
  torect.origin.y = (framerect.size.height - torect.size.height) / 2;

  // Make window's background transparent so that the GL background works.
  //
#ifndef USE_IPHONE
  if (ownWindow) {
    [[self window] setBackgroundColor:[NSColor clearColor]];
  }

  // Clear the area under the clock display to transparent.
  // Need this even if we own the window.
  [[self window] setOpaque:NO];
  [[NSColor clearColor] set];
  NSRectFill ([self frame]);
#endif /* !USE_IPHONE */

  // Set the viewport and projection matrix

#ifdef USE_IPHONE
  [EAGLContext setCurrentContext:eagl_ctx];
  glBindFramebufferOES(GL_FRAMEBUFFER_OES, gl_framebuffer);
#else /* !USE_IPHONE */
  [ogl_ctx makeCurrentContext];
  [ogl_ctx setView:self];
#endif /* !USE_IPHONE */

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  if (rotated_p) {
    glViewport (0, 0, framerect.size.width, framerect.size.height);
  glOrtho (0, framerect.size.width, 0, framerect.size.height, -1, 1);
}  else{
    glViewport (0, 0, framerect.size.width, framerect.size.height);
  glOrtho (0, framerect.size.width, 0, framerect.size.height, -1, 1);
}
  check_gl_error ("gOrtho");

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();


  // Set the OpenGL foreground and background colors

  CGFloat fgr, fgg, fgb, fga;
  CGFloat bgr, bgg, bgb, bga;
# ifndef USE_IPHONE
  [fg getRed:&fgr green:&fgg blue:&fgb alpha:&fga];
  [bg getRed:&bgr green:&bgg blue:&bgb alpha:&bga];
# else  /* USE_IPHONE */
  const CGFloat *rgba;
  rgba = CGColorGetComponents ([fg CGColor]);
  fgr = rgba[0]; fgg = rgba[1]; fgb = rgba[2]; fga = rgba[3];
  rgba = CGColorGetComponents ([bg CGColor]);
  bgr = rgba[0]; bgg = rgba[1]; bgb = rgba[2]; bga = 1;  // rgba[3];
# endif /* USE_IPHONE */

  glClearColor (bgr, bgg, bgb, bga);
  glColor4f (fgr, fgg, fgb, fga);
  glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  check_gl_error ("clear");


  // Create an OpenGL luminance_alpha texture.
  //
  int w2 = to_pow2 (config.width);
  int h2 = to_pow2 (config.height);

  // We have to scale up the bitmap to a pixmap because GL_LUMINANCE
  // textures can't use GL_BITMAP, only GL_UNSIGNED_BYTE.  Also, the
  // bitmap that digital.c creates does not have power-of-2 dimensions.
  //
  unsigned char *data = (unsigned char *) calloc (w2 * h2, 2);
  int x, y;
  unsigned char *scanin = config.bitmap;
  for (y = 0; y < config.height; y++) {
    unsigned char *scanout = data + (y * w2 * 2);
    for (x = 0; x < config.width; x++) {
      unsigned char bit = scanin[x>>3] & (1 << (7 - (x & 7)));
      *scanout++ = 0xFF;
      *scanout++ = bit ? 0xFF : 0;	
    }
    scanin += (config.width + 7) >> 3;
  }

  // OpenGLES doesn't have GL_INTENSITY, so we use 2-byte GL_LUMINANCE_ALPHA.
  glTexImage2D (GL_TEXTURE_2D, 0, GL_LUMINANCE_ALPHA, w2, h2, 0,
                GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, data);
  check_gl_error ("glTexImage2D");
  free (data);

  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  check_gl_error ("glTexParameteri");

  glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable (GL_BLEND);
  glEnable (GL_TEXTURE_2D);

  GLfloat qx = torect.origin.x;
  GLfloat qy = torect.origin.y;
  GLfloat qw = torect.size.width;
  GLfloat qh = torect.size.height;
  GLfloat tw = (GLfloat) config.width  / w2;
  GLfloat th = (GLfloat) config.height / h2;

#ifndef USE_IPHONE
  glBegin (GL_QUADS);
  glTexCoord2f (0,  th); glVertex3f (qx,    qy,    0);
  glTexCoord2f (tw, th); glVertex3f (qx+qw, qy,    0);
  glTexCoord2f (tw, 0) ; glVertex3f (qx+qw, qy+qh, 0);
  glTexCoord2f (0,  0);  glVertex3f (qx,    qy+qh, 0);
  glEnd();
  check_gl_error ("quad");
#else /* USE_IPHONE */
  GLfloat vertices[] = {
    qx,    qy,    0,
    qx+qw, qy,    0,
    qx,    qy+qh, 0,
    qx+qw, qy+qh, 0
  };
  GLfloat texCoords[] = {
    0,  th,
    tw, th,
    0,  0,
    tw, 0
  };
  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  check_gl_error ("client state");

  glVertexPointer(3, GL_FLOAT, 0, vertices);
  glTexCoordPointer(2, GL_FLOAT, 0, texCoords);
  glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
  check_gl_error ("draw arrays");

#endif /* USE_IPHONE */

  glFinish();

#ifndef USE_IPHONE
  [ogl_ctx flushBuffer];
  check_gl_error ("finish");
#else /* USE_IPHONE */
  glBindRenderbufferOES (GL_RENDERBUFFER_OES, gl_renderbuffer);
  [eagl_ctx presentRenderbuffer:GL_RENDERBUFFER_OES];
  check_gl_error ("finish");
#endif /* USE_IPHONE */
}


/* When this timer goes off, we re-generate the bitmap/pixmap,
   and mark the display as invalid.
*/
- (void)clockTick
{
  if (clockTimer && [clockTimer isValid]) {
    [clockTimer invalidate];
    clockTimer = 0;
  }

  if (config.max_fps <= 0) abort();

# ifndef USE_IPHONE
  NSWindow *w = [self window];
  if (w && ![w isMiniaturized])   // noop if no window yet, or if iconified.
# endif /* !USE_IPHONE */
  {
    render_once (&config);
    // [[self window] invalidateShadow]; // windows with shadows flicker...

# ifdef USE_IPHONE
    [self drawRect:[self frame]];
# else  /* !USE_IPHONE */
    [self setNeedsDisplay:TRUE];
# endif /* !USE_IPHONE */
  }

  // re-schedule the timer according to current fps.
  // Schedule it in two run loop modes so that the timer fires even
  // during liveResize.
  //
  float delay = 0.9 / config.max_fps;
  clockTimer = [NSTimer timerWithTimeInterval:delay
                                       target:self
                                     selector:@selector(clockTick)
                                     userInfo:nil
                                      repeats:NO];
  NSRunLoop *L = [NSRunLoop currentRunLoop];
  [L addTimer:clockTimer forMode:NSDefaultRunLoopMode];
# ifdef NSEventTrackingRunLoopMode
  [L addTimer:clockTimer forMode:NSEventTrackingRunLoopMode];
# endif
}


/* When this timer goes off, we re-pick the foreground/background colors,
   and mark the display as invalid.
 */
- (void)colorTick
{
  if (colorTimer && [colorTimer isValid]) {
    [colorTimer invalidate];
    colorTimer = 0;
  }

  if (config.max_cps <= 0) return;   // cycling is turned off, do nothing

  if ([self window]) {  // do nothing if no window yet

    CGFloat h, s, v, a;
    NSColor *fg2, *bg2;
    float tick = 1.0 / 360.0;   // cycle H by one degree per tick

# ifdef USE_IPHONE
    const CGFloat *rgba = CGColorGetComponents([fg CGColor]);
    rgb_to_hsv (rgba[0], rgba[1], rgba[2], &h, &s, &v);
    h /= 360.0;
    a = rgba[3];
# else  /* !USE_IPHONE */
    [fg getHue:&h saturation:&s brightness:&v alpha:&a];
# endif /* !USE_IPHONE */

    h += tick;
    while (h > 1.0) h -= 1.0;
    fg2 = [NSColor colorWithCalibratedHue:h saturation:s brightness:v alpha:a];

# ifdef USE_IPHONE
    rgba = CGColorGetComponents([bg CGColor]);
    rgb_to_hsv (rgba[0], rgba[1], rgba[2], &h, &s, &v);
    h /= 360.0;
    a = rgba[3];
# else  /* !USE_IPHONE */
    [bg getHue:&h saturation:&s brightness:&v alpha:&a];
# endif /* !USE_IPHONE */

    h += tick * 0.91;   // cycle bg slightly slower than fg, for randomosity.
    while (h > 1.0) h -= 1.0;
    bg2 = [NSColor colorWithCalibratedHue:h saturation:s brightness:v alpha:a];

    [self setForeground:fg2 background:bg2];

# ifdef USE_IPHONE
    /* Every time we tick the color, write the current colors into
       preferences so that the next time the app starts up, the color
       cycle begins where it left off.  (Maybe this is dumb.)
     */
    NSUserDefaults *controller = [NSUserDefaults standardUserDefaults];
    [controller setObject:[NSArchiver archivedDataWithRootObject:fg]
                forKey:@"initialForegroundColor"];
    [controller setObject:[NSArchiver archivedDataWithRootObject:bg]
                forKey:@"initialBackgroundColor"];
# endif /* !USE_IPHONE */
  }

  /* re-schedule the timer according to current fps.
     Schedule it in two run loop modes so that the timer fires even
     during liveResize.
   */
  float delay = 1.0 / config.max_cps;
  colorTimer = [NSTimer timerWithTimeInterval:delay
                                       target:self
                                     selector:@selector(colorTick)
                                     userInfo:nil
                                      repeats:NO];
  NSRunLoop *L = [NSRunLoop currentRunLoop];
  [L addTimer:colorTimer forMode:NSDefaultRunLoopMode];
# ifdef NSEventTrackingRunLoopMode
  [L addTimer:colorTimer forMode:NSEventTrackingRunLoopMode];
# endif
}


/* When this timer goes off, we switch to "show date" mode.
 */
- (void)dateTick
{
  if (dateTimer && [dateTimer isValid]) {
    [dateTimer invalidate];
    dateTimer = 0;
  }

  if (autoDateInterval <= 0) return;

  BOOL was_on = config.display_date_p;
  config.display_date_p = !was_on;

  /* re-schedule the timer according to current fps.
   */
  float delay = autoDateInterval;
  if (!was_on) delay = 1.0;
  dateTimer = [NSTimer scheduledTimerWithTimeInterval:delay
                                               target:self
                                             selector:@selector(dateTick)
                                             userInfo:nil
                                              repeats:NO];
}


- (void)updateCountdown
{
  config.countdown = (usesCountdownTimer
                      ? [countdownDate timeIntervalSince1970]
                      : 0);
}


# ifdef USE_IPHONE
- (void)showBlurb:(NSString *)text
{
  CGFloat margin = [self frame].size.width * 0.05;
  CGRect frame;
  CGFloat rot = 0;
  CGFloat pt = 14;
  UIFont *font = [UIFont boldSystemFontOfSize:pt];
  CGFloat width;
  CGFloat height = [text sizeWithFont:font
                         constrainedToSize:CGSizeMake(10240,10240)].height;

  height += 20; // Don't know how to find the inner margins of the UITextView.

  // Handle rotation here too.  This is insane.
  //
  switch ([viewController interfaceOrientation]) {
  case UIInterfaceOrientationPortraitUpsideDown:
    rot = M_PI;
    frame = CGRectMake (margin, margin,
                        [self frame].size.width - margin*2,
                        height);
    break;
  case UIInterfaceOrientationLandscapeLeft:
    rot = -M_PI/2;
    width = [self frame].size.height - margin*2;
    frame = CGRectMake ((width - [self frame].size.width) / 2 - margin,
                        ([self frame].size.height - height) / 2,
                        width, height);
    break;
  case UIInterfaceOrientationLandscapeRight:
    rot = M_PI/2;
    width = [self frame].size.height - margin*2;
    frame = CGRectMake (-([self frame].size.width / 2) - margin,
                        ([self frame].size.height - height) / 2,
                        width, height);
    break;
  default: 
    frame = CGRectMake (margin,
                        [self frame].size.height - height - margin,
                        [self frame].size.width - margin*2,
                        height);
    break;
  }

  if (aboutBox)
    [aboutBox removeFromSuperview];

  aboutBox = [[UITextView alloc] initWithFrame:frame];
  aboutBox.transform = CGAffineTransformMakeRotation (rot);
  aboutBox.font = font;
  aboutBox.textAlignment = UITextAlignmentCenter;
  aboutBox.showsHorizontalScrollIndicator = NO;
  aboutBox.showsVerticalScrollIndicator = NO;
  aboutBox.scrollEnabled = NO;
  aboutBox.textColor = [UIColor whiteColor];  // fg?
  aboutBox.backgroundColor = [UIColor clearColor];
  aboutBox.text = text;
  aboutBox.editable = NO;

  [self addSubview:aboutBox];
}
#endif /* USE_IPHONE */


/* The About button in the menu bar or preferences sheet.
 */
- (IBAction)aboutClick:(id)sender
{
  NSBundle *nsb = [NSBundle bundleForClass:[self class]];
  NSAssert1 (nsb, @"no bundle for class %@", [self class]);

  NSDictionary *info = [nsb infoDictionary];
# ifndef USE_IPHONE
  NSString *name = @"Dali Clock";
  NSString *vers = [info objectForKey:@"CFBundleVersion"];

  NSString *ver2 = [info objectForKey:@"CFBundleShortVersionString"];
  NSString *icon = [info objectForKey:@"CFBundleIconFile"];
  NSString *cred_file = [nsb pathForResource:@"Credits" ofType:@"html"];
  NSAttributedString *cred = [[[NSAttributedString alloc]
                                initWithPath:cred_file
                                documentAttributes:(NSDictionary **)NULL]
                               autorelease];
  NSString *icon_file = [nsb pathForResource:icon ofType:@"icns"];
  NSImage *iimg = [[NSImage alloc] initWithContentsOfFile:icon_file];

  NSDictionary* dict = [NSDictionary dictionaryWithObjectsAndKeys:
    name, @"ApplicationName",
    vers, @"Version",
    ver2, @"ApplicationVersion",
    @"",  @"Copyright",
    cred, @"Credits",
    iimg, @"ApplicationIcon",
    nil];

  [[NSApplication sharedApplication]
    orderFrontStandardAboutPanelWithOptions:dict];

# else  /* USE_IPHONE */
  NSString *vers = [info objectForKey:@"CFBundleGetInfoString"];
  vers = [vers stringByReplacingOccurrencesOfString:@", "  withString:@"\n"];
  vers = [vers stringByReplacingOccurrencesOfString:@".\n" withString:@"\n"];
  [self showBlurb:vers];
# endif /* USE_IPHONE */
}


// hint for XCode popup menu
#pragma mark accessors

/* [ken] if you (1) only wanted these for bindings purposes (true) and
   (2) they were dumb methods that just manipulated ivars of the same
   name (not true, they mostly write into config), then we would not
   have to write them.  I use the freeware Accessorizer to give me
   accessor templates.
 */
- (int)hourStyle { return !config.twelve_hour_p; }
- (void)setHourStyle:(int)aHourStyle
{
  config.twelve_hour_p = !aHourStyle;
}


- (int)timeStyle { return config.time_mode; }
- (void)setTimeStyle:(int)aTimeStyle
{
  if (config.time_mode != aTimeStyle) {
    config.time_mode = aTimeStyle;
    config.width++;  // kludge: force regeneration of bitmap
    [self setFrameSize:[self frame].size];
  }
}


- (int)dateStyle { return config.date_mode; }
- (void)setDateStyle:(int)aDateStyle
{
  config.date_mode = aDateStyle;
}


- (float)cycleSpeed { return (float)config.max_cps; }
- (void)setCycleSpeed:(float)aCycleSpeed
{
  if (config.max_cps != aCycleSpeed) {
    config.max_cps = (int)aCycleSpeed;
    if (config.max_cps < 0.0001) {
      [self setForeground:initialForegroundColor
               background:initialBackgroundColor];
    } else {
      [self colorTick];
    }
  }
}


- (int)usesCountdownTimer { return usesCountdownTimer; }
- (void)setUsesCountdownTimer:(int)flag
{
  usesCountdownTimer = flag;
  [self updateCountdown];
}


- (NSDate *)countdownDate { return [[countdownDate retain] autorelease]; }
- (void)setCountdownDate:(NSDate *)aCountdownDate
{
  if (countdownDate != aCountdownDate) {
    [countdownDate release];
    countdownDate = [aCountdownDate retain];
    [self updateCountdown];
  }
}


- (NSColor *)initialForegroundColor
{
  return [[initialForegroundColor retain] autorelease];
}


- (void)setInitialForegroundColor:(NSColor *)anInitialForegroundColor
{
  if (initialForegroundColor != anInitialForegroundColor) {
    if (initialForegroundColor)
      [initialForegroundColor release];
    initialForegroundColor = (anInitialForegroundColor
                              ? [anInitialForegroundColor retain]
                              : 0);
    [self setForeground:initialForegroundColor
             background:initialBackgroundColor];
  }
}


- (NSColor *)initialBackgroundColor
{
  return [[initialBackgroundColor retain] autorelease];
}


- (void)setInitialBackgroundColor:(NSColor *)anInitialBackgroundColor
{
  if (initialBackgroundColor != anInitialBackgroundColor) {
    if (initialBackgroundColor)
      [initialBackgroundColor release];
    initialBackgroundColor = (anInitialBackgroundColor
                              ? [anInitialBackgroundColor retain]
                              : 0);
    [self setForeground:initialForegroundColor
             background:initialBackgroundColor];
  }
}


- (void)setOwnWindow:(BOOL)own_p
{
  ownWindow = own_p;
}


- (void)setConstrainSizes:(BOOL)constrain_p;
{
  constrainSizes = constrain_p;
}


- (void)setAutoDate:(float)interval
{
  autoDateInterval = interval;

  config.display_date_p = 1;
  [self dateTick];
  config.display_date_p = 0;
}

@end


/* return the next larger power of 2. */
static int
to_pow2 (int i)
{
  static const unsigned int pow2[] = { 
    1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024, 
    2048, 4096, 8192, 16384, 32768, 65536 };
  int j;
  for (j = 0; j < sizeof(pow2)/sizeof(*pow2); j++)
    if (pow2[j] >= i) return pow2[j];
  abort();  /* too big! */
}


#ifdef USE_IPHONE
static void
rgb_to_hsv (CGFloat R, CGFloat G, CGFloat B,
	    CGFloat *h, CGFloat *s, CGFloat *v)
{
  CGFloat H, S, V;
  CGFloat cmax, cmin;
  CGFloat cmm;
  int imax;
  cmax = R; cmin = G; imax = 1;
  if  ( cmax < G ) { cmax = G; cmin = R; imax = 2; }
  if  ( cmax < B ) { cmax = B; imax = 3; }
  if  ( cmin > B ) { cmin = B; }
  cmm = cmax - cmin;
  V = cmax;
  if (cmm == 0)
    S = H = 0;
  else
    {
      S = cmm / cmax;
      if       (imax == 1)    H =       (G - B) / cmm;
      else  if (imax == 2)    H = 2.0 + (B - R) / cmm;
      else /*if (imax == 3)*/ H = 4.0 + (R - G) / cmm;
      if (H < 0) H += 6.0;
    }
  *h = (H * 60.0);
  *s = S;
  *v = V;
}
#endif /* USE_IPHONE */


/* report a GL error. */
static void
check_gl_error (const char *type)
{
  char buf[100];
  GLenum i;
  const char *e;

# ifndef  GL_TABLE_TOO_LARGE_EXT
#  define GL_TABLE_TOO_LARGE_EXT 0x8031
# endif
# ifndef  GL_TEXTURE_TOO_LARGE_EXT
#  define GL_TEXTURE_TOO_LARGE_EXT 0x8065
# endif
# ifndef  GL_INVALID_FRAMEBUFFER_OPERATION
#  define GL_INVALID_FRAMEBUFFER_OPERATION 0x0506
# endif

  switch ((i = glGetError())) {
    case GL_NO_ERROR: return;
    case GL_INVALID_ENUM:          e = "invalid enum";      break;
    case GL_INVALID_VALUE:         e = "invalid value";     break;
    case GL_INVALID_OPERATION:     e = "invalid operation"; break;
    case GL_STACK_OVERFLOW:        e = "stack overflow";    break;
    case GL_STACK_UNDERFLOW:       e = "stack underflow";   break;
    case GL_OUT_OF_MEMORY:         e = "out of memory";     break;
    case GL_TABLE_TOO_LARGE_EXT:   e = "table too large";   break;
    case GL_TEXTURE_TOO_LARGE_EXT: e = "texture too large"; break;
    case GL_INVALID_FRAMEBUFFER_OPERATION: e = "invalid framebuffer op"; break;
    default:
      e = buf; sprintf (buf, "unknown GL error 0x%04x", (int) i); break;
  }
  NSLog (@"%s GL error: %s", type, e);
  abort();
}

