#ifndef DPOINT_H
#define DPOINT_H

#include <math.h>

#include <qarray.h>
#include <qpoint.h>
#include <qcolor.h>
#include <qfont.h>
#include <qtextstream.h>
#include <qlist.h>
#include <iostream.h>

class DPoint {
 public:
  double x, y, z;
  QString id; // XML ID field
  QString element;  // irrelevant if not an atom.
  QString elementmask;
  QString symbol;  // only used when saving Molecules as XDC-XML
  QString hosecode; // save HOSE codes made in molecule_tools.cpp
  QString tmphose; // temp HOSE code (if in first sphere)
  int hosescore;  // HOSE score
  int ketos; // number of keto groups attached (if any)
  bool aromatic; // part of an aromatic system?
  QString ringset;  // SMILES ring closure code(s) for this atom
  QColor color;
  QFont font;
  bool hit;
  int substituents;  // how many bonds to this atom (sum of order)
  // for SDG and SSSR
  DPoint *source;
  int serial;
  QList<DPoint> neighbors;
  QArray<int> bondorder;
  QList<DPoint> path;
  // functions
  DPoint() { 
    init();
  }
  DPoint(double a, double b) { 
    x = a; y = b; init();
  }
  DPoint(QPoint q) { 
    x = q.x(); y = q.y(); init();
  }
  void init() {
    element = QString("C"); 
    symbol = QString("");
    color = QColor(0,0,0); 
    font = QFont("Helvetica", 12);
    aromatic = false;
    serial = 0;
    hit = false;
    bondorder.resize(6);
  }
  // these would be more relevant... if "substituents" were private...
  int degree() { return substituents; }
  void setDegree(int neworder) { substituents = neworder; }
  void set(QPoint p) {
    x = (double)p.x();
    y = (double)p.y();
  }
  QPoint toQPoint() {
    int x1 = (int)x;
    int y1 = (int)y;
    double xf = x - (double)x;
    double yf = y - (double)y;
    if (xf > 0.5) x1++;
    if (yf > 0.5) y1++;
    return QPoint(x1, y1); 
  }
  void zero() { x = 0.0; y = 0.0; }
  bool isZero() {
    if ( ( x < 0.01 ) && ( x > -0.01) &&
	 ( y < 0.01 ) && ( y > -0.01) )
      return true;
    else
      return false;
  }
  double distanceTo(DPoint *t) {
    double dx = x - t->x;
    double dy = y - t->y;
    return (double)sqrt(dx * dx + dy * dy);
  }
  void SetFontFromXML(QString xml_tag) {
    cout << "DP::SetFontFromXML:" << xml_tag << endl;
    int i1, i2;

    i1 = xml_tag.find("<font>");
    i2 = xml_tag.find("</font>");
    xml_tag.remove(i2, 999);
    xml_tag.remove(i1, 6);
    i1 = xml_tag.find("#");
    cout << xml_tag.mid(0, i1) << "*" << xml_tag.mid(i1 + 1) << endl;
    font = QFont(xml_tag.mid(0, i1), xml_tag.mid(i1 + 1).toInt());
  }
  void SetColorFromXML(QString xml_tag) {
    cout << "DP::SetColorFromXML:" << xml_tag << endl;
    int i1, i2;
    int d1, d2, d3;

    i1 = xml_tag.find("<color>");
    i2 = xml_tag.find("</color>");
    xml_tag.remove(i2, 999);
    xml_tag.remove(i1, 7);

    QTextStream ts(&xml_tag, IO_ReadOnly);
    
    ts >> d1 >> d2 >> d3;
  
    color = QColor(d1, d2, d3);
  }
  // set DPoint coordinates from XML
  void SetCoordFromXML(QString xml_tag) {
    cout << "DP::SetCoordFromXML:" << xml_tag << endl;
    int i1, i2;
    
    i1 = xml_tag.find("<coordinate2>");
    i2 = xml_tag.find("</coordinate2>");
    xml_tag.remove(i2, 999);
    xml_tag.remove(i1, 13);
    
    QTextStream ts(&xml_tag, IO_ReadOnly);

    ts >> x >> y;
  }
  void SetElementFromXML(QString xml_tag) {
    cout << "DP::SetElementFromXML:" << xml_tag << endl;
    int i1, i2;
    
    i1 = xml_tag.find("<element>");
    i2 = xml_tag.find("</element>");
    xml_tag.remove(i2, 999);
    xml_tag.remove(i1, 9);
    
    element = xml_tag;
  }
  void SetElementMaskFromXML(QString xml_tag) {
    cout << "DP::SetElementMaskFromXML:" << xml_tag << endl;
    int i1, i2;
    
    i1 = xml_tag.find("<elemask>");
    i2 = xml_tag.find("</elemask>");
    xml_tag.remove(i2, 999);
    xml_tag.remove(i1, 9);
    
    elementmask = xml_tag;
  }
  void SetSymbolFromXML(QString xml_tag) {
    cout << "DP::SetSymbolFromXML:" << xml_tag << endl;
    int i1, i2;
    
    i1 = xml_tag.find("<symtype>");
    i2 = xml_tag.find("</symtype>");
    xml_tag.remove(i2, 999);
    xml_tag.remove(i1, 9);
    
    symbol = xml_tag;
  }
  void FromXML(QString xml_tag) {
    int i1, i2, ib, ie;

    i1 = xml_tag.find("<atom ");
    ib = xml_tag.find("\"", i1);
    ie = xml_tag.find("\"", ib + 1);
    id = xml_tag.mid(ib + 1, ie - ib - 1);
    cout << "DP::AtomID:" << id << endl;
    i1 = xml_tag.find("<coordinate2>");
    i2 = xml_tag.find("</coordinate2>") + 14;
    SetCoordFromXML(xml_tag.mid(i1, i2 - i1));
    i1 = xml_tag.find("<color>");
    i2 = xml_tag.find("</color>") + 8;
    SetColorFromXML(xml_tag.mid(i1, i2 - i1));
    i1 = xml_tag.find("<font>");
    i2 = xml_tag.find("</font>") + 7;
    SetFontFromXML(xml_tag.mid(i1, i2 - i1));
    i1 = xml_tag.find("<element>");
    i2 = xml_tag.find("</element>") + 10;
    SetElementFromXML(xml_tag.mid(i1, i2 - i1));
    i1 = xml_tag.find("<elemask>");
    i2 = xml_tag.find("</elemask>") + 10;
    SetElementMaskFromXML(xml_tag.mid(i1, i2 - i1));
    i1 = xml_tag.find("<symtype>");
    i2 = xml_tag.find("</symtype>") + 10;
    if (i1 >= 0)
      SetSymbolFromXML(xml_tag.mid(i1, i2 - i1));
  }
  // return atomic number of atom connected in this group
  int getAtomicNumber() {
    int atom1;
    if (element == "C") atom1 = 6;
    if (element == "CN") atom1 = 6;
    if (element == "NC") atom1 = 6;
    if (element == "CO") atom1 = 6;
    if (element == "OC") atom1 = 6;
    if (element == "HC") atom1 = 6;
    if (element == "CH") atom1 = 6;
    if (element == "H2C") atom1 = 6;
    if (element == "CH2") atom1 = 6;
    if (element == "H3C") atom1 = 6;
    if (element == "CH3") atom1 = 6;
    if (element == "N") atom1 = 7;
    if (element == "NH") atom1 = 7;
    if (element == "HN") atom1 = 7;
    if (element == "NO") atom1 = 7;
    if (element == "ON") atom1 = 7;
    if (element == "NH2") atom1 = 7;
    if (element == "H2N") atom1 = 7;
    if (element == "NO2") atom1 = 7;
    if (element == "O2N") atom1 = 7;
    if (element == "O") atom1 = 8;
    if (element == "OH") atom1 = 8;
    if (element == "HO") atom1 = 8;
    if (element == "S") atom1 = 16;
    if (element == "SH") atom1 = 16;
    if (element == "HS") atom1 = 16;
    return atom1;
  }
};

#endif
