# buffer.tcl
# $Id: buffer.tcl,v 1.16 1999/07/31 00:33:36 chris Exp $
# Handles the line buffer, and also handles I/O.
# I'd love to draw an abstraction barrier here, but find
# this difficult because the buffer is stored in the text
# widget in X mode.
#
#
# XED --- an X-enabled implementation of ed, the standard editor.
# Copyright (C) 1999  Chris Laas
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# Chris Laas
# golem@mit.edu
# 22 Pratt Court
# Cohasset, MA 02025 USA



######################################################################
# Procedures that form an abstraction barrier for interaction with
# the line buffer.

proc both-buffer-init {} {
	global is_tclet playback
	if {!$is_tclet && !$playback} {
		fconfigure stdin -blocking false
		fileevent stdin readable {
			# The nonblocking code _should_ deal with TTIN
			# gracefully, but it doesn't (under linux and netbsd;
			# it works fine under solaris without this frobbing).
			# Consequently, I must trap TTIN in the header of
			# main.tcl, and trap the resultant I/O error here.
			if {![catch {gets stdin line} error]} {
				if [eof stdin] { exit }
				set line_source stdin ; process-line $line
				puts -nonewline stdout $prompt ; flush stdout
			} else {
				if {![string match *I/O* $error]} {
					error $error $errorInfo $errorCode
				}
			}
		}
	}
	# Tcl 8.1 is buggy; it coredumps on large output chunks unless this
	# line is here.  I traced the bug as far as generic/tclIO.c:1795
	# (panic("Blocking channel driver did not block on output")).
	if {!$is_tclet} { fconfigure stdout -blocking true }
}


######################################################################
# Handling for the X buffer and window.
proc x-buffer-init {} {
	both-buffer-init

	global tcl_platform

	# Default options.
	set name [winfo name .]
	option add $name.TabWidth 8 startupFile
	option add $name.WrapLines on startupFile
	option add $name.Scrollbar on startupFile

	switch $tcl_platform(platform) {
		unix {
			option add $name*font fixed startupFile
		}
		windows - macintosh {
			option add $name*font {Courier 10} startupFile
		}
	}

	option add $name.currentBackground DarkGrey   startupFile
	option add $name.subcmdBackground  Plum       startupFile
	option add $name.rangeBackground   SkyBlue    startupFile
	option add $name.destBackground    LightCoral startupFile
	option add $name.insertBackground  LightGreen startupFile

	option add $name.status.errorForeground Red startupFile

	text .t -yscrollcommand ".ysb set" -setgrid 1 -takefocus 0 \
			-width 80 -height 24 \
			-insertwidth 0 -insertborderwidth 0 \
			-bd 1 -relief raised -highlightthickness 0
	set fontw [font measure [option get .t font Font] " "]
	.t configure -tabs [expr $fontw * [option get . tabWidth TabWidth]]
	switch -- [string tolower [option get . wrapLines WrapLines]] {
		1 - true - yes - on            { .t configure -wrap char }
		0 - false - no - off - default { .t configure -wrap none }
	}
	scrollbar .ysb -command ".t yview" -takefocus 0 \
			-bd 1 -relief raised -highlightthickness 0
	entry .e -textvariable entry_line \
			-bd 1 -relief sunken -highlightthickness 0
	label .status -textvariable status_line -anchor w -justify left \
			-width 1 -bd 1 -relief raised

	# Default options derived from cget defaults.
	option add $name.status.errorBackground [.status cget -bg] startupFile
	option add $name.status.foreground      [.status cget -fg] startupFile
	option add $name.status.background      [.status cget -bg] startupFile

	global old_status_width
	set old_status_width [winfo width .status]
	bind .status <Configure> {
		set width [winfo width .status]
		if {$width != $old_status_width} {
			.status configure -wraplength $width
			set old_status_width $width
		}
	}

	.t tag configure current \
		-foreground [option get . currentForeground Foreground] \
		-background [option get . currentBackground Background]
	.t tag configure range \
		-foreground [option get .   rangeForeground Foreground] \
		-background [option get .   rangeBackground Background]
	.t tag configure dest \
		-foreground [option get .    destForeground Foreground] \
		-background [option get .    destBackground Background]
	.t tag configure text \
		-foreground [option get .  insertForeground Foreground] \
		-background [option get .  insertBackground Background]
	bind .t <ButtonPress-1> { set b1_moved 0 ; set x %x ; set y %y }
	bind .t <B1-Motion>     {
		if {abs($x - %x) > 1 || abs($y - %y) > 1} { set b1_moved 1 }
	}
	bind .t <ButtonRelease-1> {
		if !$b1_moved {
			regexp {^([0-9]+)} [.t index @%x,%y] current_addr
		}
	}
	bind .t <ButtonPress-3> {
		regexp {^([0-9]+)} [.t index @%x,%y] pointer_addr1
		read-addr-range $entry_line entry_line trash trash
		set entry_line $pointer_addr1$entry_line
		raise .e ; bind .e <Key> {} ; .e icursor end
	}
	bind .t <B3-Motion> {
		event-delay b3 {
			regexp {^([0-9]+)} [.t index @%x,%y] pointer_addr2
			read-addr-range $entry_line entry_line trash trash
			if {$pointer_addr2 > $pointer_addr1} {
				set entry_line $pointer_addr1,$pointer_addr2$entry_line
			} else {
				set entry_line $pointer_addr2,$pointer_addr1$entry_line
			}
			.e icursor end
			see-addr $pointer_addr2

			# Temporarily disable the see-addr command so that
			# update-textbox-current_addr doesn't override _this_
			# see-addr (see above).  Kludgey, but oh well.
			rename see-addr _see-addr
			proc see-addr {args} {}
			after idle {catch {rename see-addr {} ; rename _see-addr see-addr}}
		}
	}
	bind .t <Shift-ButtonPress-3> {
		regexp {^([0-9]+)} [.t index @%x,%y] pointer_addr1
		regsub {[0-9]*$} $entry_line {} entry_line
		.e icursor end
		append entry_line $pointer_addr1
		raise .e ; bind .e <Key> {}
	}
	bind .t <Shift-B3-Motion> {# nothing}

	bind .t <<Cut>> break
	bind .t <<Paste>> break
	bind .t <<PasteSelection>> break
	bind .t <<Clear>> break

	bind .e <Key-Tab> {
		tkEntryInsert %W \t
		#focus %W
		break
	}

	bind .e <Key> { raise .e ; bind .e <Key> {} }
	bind .e <Key-Return> {
		set line $entry_line ; set entry_line {} ; set line_source X
		process-line $line
	}
	bind .e <Key-Escape> {
		set entry_line {}
		lower .e ; bind .e <Key> { raise .e ; bind .e <Key> {} }
	}

	foreach ignore {
		Shift_L Shift_R Control_L Control_R Meta_L Meta_R Alt_L Alt_R
		Super_L Super_R Hyper_L Hyper_R Caps_Lock Num_Lock
		Up Down Prior Next
	} {
		bind .e <Key-$ignore> continue
	}
	if {$tcl_platform(platform) == "unix"} {
		bind .e <Key-Scroll_Lock> continue
	}

	bind . <Key-Up>    { .t yview scroll -1 unit }
	bind . <Key-Down>  { .t yview scroll  1 unit }
	bind . <Key-Prior> { .t yview scroll -1 page }
	bind . <Key-Next>  { .t yview scroll  1 page }
	bind . <Control-Key-d> { exit }

	wm protocol . WM_DELETE_WINDOW { command-quit {} {} q {} }

	switch -- [string tolower [option get . scrollbar Scrollbar]] {
		1 - true - yes - on {
			grid .t      .ysb -sticky nsew
			grid .status -    -sticky nsew
		}
		0 - false - no - off - default {
			grid .t      -sticky nsew
			grid .status -sticky nsew
		}
	}
	grid    rowconfigure . 0 -weight 1
	grid columnconfigure . 0 -weight 1
	place .e -in .status -relx 0 -rely 0 -relwidth 1 -relheight 1
	lower .e

	# Games with focus...
	rename focus _focus
	proc focus {args} {}
	_focus .e
}

proc x-buffer-get {first last} {
	.t get $first.0 $last.end
}

proc x-buffer-put {line text} {
	.t insert [incr line].0 $text\n
}

proc x-buffer-delete {first last} {
	.t delete $first.0 [incr last].0
}

proc x-buffer-clear {} {
	.t delete 1.0 end
}

proc x-buffer-search-forwards {regexp start stop} {
	if {$stop == {}} {
		set match [.t search -forwards -regexp -- $regexp [expr $start-1].end]
	} else {
		set match [.t search -forwards -regexp -- $regexp [expr $start-1].end \
				[expr $stop+1].0]
	}
	if [regexp {^([0-9]+)} $match addr] {
		return $addr
	} else {
		return ""
	}
}

proc x-buffer-search-backwards {regexp start stop} {
	if {$stop == {}} {
		set match [.t search -backwards -regexp -- $regexp [expr $start+1].0]
	} else {
		set match [.t search -backwards -regexp -- $regexp [expr $start+1].0 \
				[expr $stop-1].end]
	}
	if [regexp {^([0-9]+)} $match addr] {
		return $addr
	} else {
		return ""
	}
}

proc x-buffer-mark-set {c line} {
	.t mark set mark_$c $line.0
	.t mark gravity mark_$c right
}

proc x-buffer-mark-get {c} {
	if [catch { regexp {^([0-9]+)} [.t index mark_$c] result }] {
		return {}
	}
	return $result
}


######################################################################
# Handling for the tty buffer.
proc tty-buffer-init {} {
	both-buffer-init

	tty-buffer-clear
}

proc ttybuf-find-addr {addr} {
	global ttybuf
	while {$addr > $ttybuf(last_accurate_addr2ptr)} {
		set ptr $ttybuf(addr.$ttybuf(last_accurate_addr2ptr))
		set ttybuf(addr.[incr ttybuf(last_accurate_addr2ptr)]) \
				$ttybuf($ptr.next)
	}
	return $ttybuf(addr.$addr)
}
proc ttybuf-find-ptr {ptr} {
	global ttybuf

	set addr $ttybuf($ptr.addr)
	set laptr $ttybuf(last_accurate_ptr2addr)
	if {$ttybuf($ptr.addr) > $ttybuf($laptr.addr)} {
		set addr $ttybuf($laptr.addr)
		while {$laptr != {} && $laptr != $ptr} {
			set ttybuf($ttybuf($laptr.next).addr) [incr addr]
			set ttybuf(last_accurate_ptr2addr) $laptr
			set laptr $ttybuf($laptr.next)
		}
		if {$laptr == {}} {
			set addr {}
		} else {
			set ttybuf(last_accurate_ptr2addr) $laptr
		}
	}
	return $addr
}

proc tty-buffer-get {first last} {
	global ttybuf
	set text {}
	set ptr [ttybuf-find-addr $first]
	for {} {$first <= $last} {incr first} {
		lappend text $ttybuf($ptr.line)
		set ptr $ttybuf($ptr.next)
	}
	return [join $text \n]
}

set ttyptrcount 0
proc tty-buffer-put {addr text} {
	global ttybuf ttyptrcount
	set ptr [ttybuf-find-addr $addr]
	set oldnext $ttybuf($ptr.next)
	set lines [split $text \n]
	if {$lines == {}} { set lines {{}} }
	foreach line $lines {
		set next p[incr ttyptrcount]
		set ttybuf($ptr.next) $next
		set ttybuf($next.line) $line
		set ttybuf($next.addr) [incr addr]
		set ttybuf(addr.$addr) $next
		set ptr $next
	}
	set ttybuf($ptr.next) $oldnext
	set ttybuf(last_accurate_addr2ptr) $addr
	set ttybuf(last_accurate_ptr2addr) $ttybuf(addr.[expr $addr-1])
}

proc tty-buffer-delete {first last} {
	global ttybuf
	set ptr1 [ttybuf-find-addr [incr first -1]]
	set ptr2 [ttybuf-find-addr [incr last]]
	set ttybuf($ptr1.next) $ptr2
	set ttybuf(last_accurate_addr2ptr) $first
	if {$ttybuf($ttybuf(last_accurate_ptr2addr).addr) > $first} {
		set ttybuf(last_accurate_ptr2addr) $ttybuf(addr.$first)
	}
}

proc tty-buffer-clear {} {
	global ttybuf
	array set ttybuf {
		last_accurate_addr2ptr 0 last_accurate_ptr2addr ptr0
		addr.0 ptr0 ptr0.next {} ptr0.addr 0
	}
}

proc tty-buffer-search-forwards {regexp start stop} {
	global ttybuf
	set ptr [ttybuf-find-addr $start]
	if {$stop == {}} {
		set stop $start
		while 1 {
			if [regexp -- $regexp $ttybuf($ptr.line)] {
				return $start
			}
			set ptr $ttybuf($ptr.next)
			if {$ptr != {}} {
				incr start
			} else {
				set ptr $ttybuf(ptr0.next)
				set start 1
			}
			if {$start == $stop} { break }
		}
		return ""
	} else {
		for {} {$start <= $stop} {incr start} {
			if [regexp -- $regexp $ttybuf($ptr.line)] {
				return $start
			}
			set ptr $ttybuf($ptr.next)
		}
		return ""
	}
}

proc tty-buffer-search-backwards {regexp start stop} {
	global ttybuf last_addr
	set ptrs {}
	if {$stop == {}} {
		set stop $start
		while 1 {
			set ptr [ttybuf-find-addr $start]
			if [regexp -- $regexp $ttybuf($ptr.line)] {
				return $start
			}
			if {$start != 1} {
				incr start -1
			} else {
				# OK, this is a cop out.  But I'm lazy.
				set start $last_addr
			}
			if {$start == $stop} { break }
		}
		return ""
	} else {
		set ptr [ttybuf-find-addr $start]
		for {} {$stop <= $start} {incr stop} {
			set ptrs [concat [list $ptr] $ptrs]
			set ptr $ttybuf($ptr.next)
		}
		foreach ptr $ptrs {
			if {[regexp -- $regexp $ttybuf($ptr.line)]} {
				return $start
			}
		}
		return ""
	}
}

proc tty-buffer-mark-set {c addr} {
	global ttybuf
	set ttybuf(mark.$c) [ttybuf-find-addr $addr]
}

proc tty-buffer-mark-get {c} {
	global ttybuf
	if ![info exists ttybuf(mark.$c)] { return {} }
	return [ttybuf-find-ptr $ttybuf(mark.$c)]
}


######################################################################

set this_undo_commands {}
set last_undo_commands {}

proc buffer-init {} {
	global ttymode
	if $ttymode {
		tty-buffer-init
	} else {
		x-buffer-init
	}
}
proc buffer-get {first last} {
	global ttymode
	if $ttymode {
		tty-buffer-get $first $last
	} else {
		x-buffer-get $first $last
	}
}
proc buffer-put {addr text} {
	global ttymode this_undo_commands

	set lines [llength [split $text \n]]
	if {$lines == 0} { set lines 1 }
	set this_undo_commands [concat \
		[list "[expr $addr+1],[expr $addr+$lines] d"] \
		$this_undo_commands]

	if $ttymode {
		tty-buffer-put $addr $text
	} else {
		x-buffer-put $addr $text
	}
}
proc buffer-delete {first last} {
	global ttymode this_undo_commands

	set text_lines [split [buffer-get $first $last] \n]
	if {$text_lines == {}} { set text_lines {{}} }
	set this_undo_commands [concat \
			[list "[expr $first-1] a"] \
			$text_lines \
			[list "."] \
			$this_undo_commands]

	if $ttymode {
		tty-buffer-delete $first $last
	} else {
		x-buffer-delete $first $last
	}
}
proc buffer-clear {} {
	global ttymode last_addr this_undo_commands

	set text_lines [split [buffer-get 1 $last_addr] \n]
	if {$text_lines == {}} { set text_lines {{}} }
	set this_undo_commands [concat \
			[list "0 a"] \
			$text_lines \
			[list "."] \
			$this_undo_commands]

	if $ttymode {
		tty-buffer-clear
	} else {
		x-buffer-clear
	}
}
proc buffer-search-forwards {regexp start {stop {}}} {
	global ttymode
	if $ttymode {
		tty-buffer-search-forwards $regexp $start $stop
	} else {
		x-buffer-search-forwards $regexp $start $stop
	}
}
proc buffer-search-backwards {regexp start {stop {}}} {
	global ttymode
	if $ttymode {
		tty-buffer-search-backwards $regexp $start $stop
	} else {
		x-buffer-search-backwards $regexp $start $stop
	}
}
proc buffer-mark-set {c addr} {
	global ttymode this_undo_commands

	if {[string length $c] == 1} {
		# Don't store temp marks.
		set this_undo_commands [concat \
				[list "[buffer-mark-get $c] k$c"] \
				$this_undo_commands]
	}

	if $ttymode {
		tty-buffer-mark-set $c $addr
	} else {
		x-buffer-mark-set $c $addr
	}
}
proc buffer-mark-get {c} {
	global ttymode
	if $ttymode {
		tty-buffer-mark-get $c
	} else {
		x-buffer-mark-get $c
	}
}


######################################################################
# Output procedures, for spewing information back to the user.
if $ttymode {
	set line_source stdin
} else {
	set line_source X
}
proc puts-response {text} {
	global line_source status_line
	switch $line_source {
		stdin { puts stdout $text }
		X {
			set status_line $text
			.status configure \
				-foreground [option get .status foreground Foreground] \
				-background [option get .status background Background]
			lower .e ; bind .e <Key> { raise .e ; bind .e <Key> {} }
		}
	}
}
proc puts-error {text} {
	global line_source status_line
	switch $line_source {
		stdin { puts stderr $text }
		X {
			set status_line $text
			.status configure \
				-foreground [option get .status errorForeground Foreground] \
				-background [option get .status errorBackground Background]
			lower .e ; bind .e <Key> { raise .e ; bind .e <Key> {} }
		}
	}
}


######################################################################
# Realtime updating of text window.
proc update-textbox-entry_line {name1 name2 op} {
	global input_mode entry_line current_addr default_ranges
	foreach tag {range dest} {
		set tag_ranges [.t tag ranges $tag]
		if {[llength $tag_ranges] != 0} {
			eval ".t tag remove $tag $tag_ranges"
		}
	}
	switch -exact -- [lindex $input_mode 0] {
		command - subcommand {
			set addr1 {} ; set addr2 {} ; set dest_addr {}
			catch {
				read-addr-range $entry_line line addr1 addr2
# 				if {$addr1 == {}} {
# 					read-addr-range \
# 							$default_ranges([string index $entry_line 0]) \
# 							line addr1 addr2
# 				}
				set command [string index $line 0]
				if {$command == "m" || $command == "t"} {
					read-addr-range [string range $line 1 end] line trash dest_addr
				}
			}
			if {$addr1 != {}} {
				.t tag add range $addr1.0 [expr $addr2+1].0
				see-addr $addr2
			}
			if {$dest_addr != {}} {
				.t tag add dest $dest_addr.0 [expr $dest_addr+1].0
				see-addr $dest_addr
			}
		}
		text {
			catch { .t delete text.first text.last } error
			set tag_ranges [.t tag ranges text]
			if {[llength $tag_ranges] != 0} {
				eval ".t tag remove text $tag_ranges"
			}
			if {$entry_line != "."} {
				.t insert [expr $current_addr+1].0 $entry_line\n
				.t tag add text \
						[expr $current_addr+1].0 [expr $current_addr+2].0
				see-addr [expr $current_addr+1]
			}
		}
	}
}
set prev_input_mode {}
proc update-textbox-input_mode {name1 name2 op} {
	global prev_input_mode input_mode current_addr
	switch -glob -- [lindex $prev_input_mode 0]/[lindex $input_mode 0] {
		*/text {
			.t insert [expr $current_addr+1].0 \n
			.t tag add text \
					[expr $current_addr+1].0 [expr $current_addr+2].0
			see-addr [expr $current_addr+1]
		}
		text/* {
			if {[.t tag ranges text] != {}} {
				.t delete text.first text.last
			}
		}
		*/subcommand {
			.t tag configure current \
				-foreground [option get . subcmdForeground Foreground] \
				-background [option get . subcmdBackground Background]
		}
		subcommand/* {
			.t tag configure current \
				-foreground [option get . currentForeground Foreground] \
				-background [option get . currentBackground Background]
		}
	}
	set prev_input_mode $input_mode
}
proc update-textbox-current_addr {name1 name2 op} {
	global current_addr
	see-addr $current_addr
	.t tag remove current 1.0 end
	.t tag add current $current_addr.0 [expr $current_addr+1].0
}
proc update-titlebar-default_filename {name1 name2 op} {
	global default_filename
	if {$default_filename == ""} {
		wm title . "XED"
	} else {
		wm title . "XED: $default_filename"
	}
}
proc see-addr {addr} {
	.t see [expr $addr+2].end
	.t see [expr $addr-2].end
	.t see $addr.end
}
if {!$ttymode} {
	trace variable entry_line w {event-delay el update-textbox-entry_line}
	trace variable input_mode w update-textbox-input_mode
	trace variable current_addr w update-textbox-current_addr
	trace variable default_filename w update-titlebar-default_filename
}


# event-delay:  delay processing of an event until idle time, so that
# the event queue will not get backed up.
proc event-delay {name args} {
	upvar #0 afterid_$name id
	if [info exists id] {
		after cancel $id
	}
	if {[llength $args] == 1} { set args [lindex $args 0] }
	set id [after idle $args]
}
