/******************************************************************************
 * arch-ia64/hypervisor-if.h
 * 
 * Guest OS interface to IA64 Xen.
 */

#ifndef __HYPERVISOR_IF_IA64_H__
#define __HYPERVISOR_IF_IA64_H__

/* Structural guest handles introduced in 0x00030201. */
#if __XEN_INTERFACE_VERSION__ >= 0x00030201
#define __DEFINE_XEN_GUEST_HANDLE(name, type) \
    typedef struct { type *p; } __guest_handle_ ## name
#else
#define __DEFINE_XEN_GUEST_HANDLE(name, type) \
    typedef type * __guest_handle_ ## name
#endif

#define DEFINE_XEN_GUEST_HANDLE(name)   __DEFINE_XEN_GUEST_HANDLE(name, name)
#define XEN_GUEST_HANDLE(name)          __guest_handle_ ## name
#define set_xen_guest_handle(hnd, val)  do { (hnd).p = val; } while (0)
#ifdef __XEN_TOOLS__
#define get_xen_guest_handle(val, hnd)  do { val = (hnd).p; } while (0)
#endif

#ifndef __ASSEMBLY__
/* Guest handles for primitive C types. */
__DEFINE_XEN_GUEST_HANDLE(uchar, unsigned char);
__DEFINE_XEN_GUEST_HANDLE(uint,  unsigned int);
__DEFINE_XEN_GUEST_HANDLE(ulong, unsigned long);
DEFINE_XEN_GUEST_HANDLE(char);
DEFINE_XEN_GUEST_HANDLE(int);
DEFINE_XEN_GUEST_HANDLE(long);
DEFINE_XEN_GUEST_HANDLE(void);

typedef unsigned long xen_pfn_t;
DEFINE_XEN_GUEST_HANDLE(xen_pfn_t);
#endif

/* Arch specific VIRQs definition */
#define VIRQ_ITC        VIRQ_ARCH_0 /* V. Virtual itc timer */

/* Maximum number of virtual CPUs in multi-processor guests. */
/* WARNING: before changing this, check that shared_info fits on a page */
#define MAX_VIRT_CPUS 64

#ifndef __ASSEMBLY__

typedef unsigned long xen_ulong_t;

#define GPFN_MEM          (0UL << 56) /* Guest pfn is normal mem */
#define GPFN_FRAME_BUFFER (1UL << 56) /* VGA framebuffer */
#define GPFN_LOW_MMIO     (2UL << 56) /* Low MMIO range */
#define GPFN_PIB          (3UL << 56) /* PIB base */
#define GPFN_IOSAPIC      (4UL << 56) /* IOSAPIC base */
#define GPFN_LEGACY_IO    (5UL << 56) /* Legacy I/O base */
#define GPFN_GFW          (6UL << 56) /* Guest Firmware */
#define GPFN_HIGH_MMIO    (7UL << 56) /* High MMIO range */

#define GPFN_IO_MASK     (7UL << 56)  /* Guest pfn is I/O type */
#define GPFN_INV_MASK    (31UL << 59) /* Guest pfn is invalid */

#define INVALID_MFN       (~0UL)

#define MEM_G   (1UL << 30)
#define MEM_M   (1UL << 20)

#define MMIO_START       (3 * MEM_G)
#define MMIO_SIZE        (512 * MEM_M)

#define VGA_IO_START     0xA0000UL
#define VGA_IO_SIZE      0x20000

#define LEGACY_IO_START  (MMIO_START + MMIO_SIZE)
#define LEGACY_IO_SIZE   (64*MEM_M)

#define IO_PAGE_START (LEGACY_IO_START + LEGACY_IO_SIZE)
#define IO_PAGE_SIZE  PAGE_SIZE

#define STORE_PAGE_START (IO_PAGE_START + IO_PAGE_SIZE)
#define STORE_PAGE_SIZE	 PAGE_SIZE

#define IO_SAPIC_START   0xfec00000UL
#define IO_SAPIC_SIZE    0x100000

#define PIB_START 0xfee00000UL
#define PIB_SIZE 0x200000

#define GFW_START        (4*MEM_G -16*MEM_M)
#define GFW_SIZE         (16*MEM_M)

struct pt_fpreg {
    union {
        unsigned long bits[2];
        long double __dummy;    /* force 16-byte alignment */
    } u;
};

struct cpu_user_regs {
    /* The following registers are saved by SAVE_MIN: */
    unsigned long b6;  /* scratch */
    unsigned long b7;  /* scratch */

    unsigned long ar_csd; /* used by cmp8xchg16 (scratch) */
    unsigned long ar_ssd; /* reserved for future use (scratch) */

    unsigned long r8;  /* scratch (return value register 0) */
    unsigned long r9;  /* scratch (return value register 1) */
    unsigned long r10; /* scratch (return value register 2) */
    unsigned long r11; /* scratch (return value register 3) */

    unsigned long cr_ipsr; /* interrupted task's psr */
    unsigned long cr_iip;  /* interrupted task's instruction pointer */
    unsigned long cr_ifs;  /* interrupted task's function state */

    unsigned long ar_unat; /* interrupted task's NaT register (preserved) */
    unsigned long ar_pfs;  /* prev function state  */
    unsigned long ar_rsc;  /* RSE configuration */
    /* The following two are valid only if cr_ipsr.cpl > 0: */
    unsigned long ar_rnat;  /* RSE NaT */
    unsigned long ar_bspstore; /* RSE bspstore */

    unsigned long pr;  /* 64 predicate registers (1 bit each) */
    unsigned long b0;  /* return pointer (bp) */
    unsigned long loadrs;  /* size of dirty partition << 16 */

    unsigned long r1;  /* the gp pointer */
    unsigned long r12; /* interrupted task's memory stack pointer */
    unsigned long r13; /* thread pointer */

    unsigned long ar_fpsr;  /* floating point status (preserved) */
    unsigned long r15;  /* scratch */

 /* The remaining registers are NOT saved for system calls.  */

    unsigned long r14;  /* scratch */
    unsigned long r2;  /* scratch */
    unsigned long r3;  /* scratch */
    unsigned long r16;  /* scratch */
    unsigned long r17;  /* scratch */
    unsigned long r18;  /* scratch */
    unsigned long r19;  /* scratch */
    unsigned long r20;  /* scratch */
    unsigned long r21;  /* scratch */
    unsigned long r22;  /* scratch */
    unsigned long r23;  /* scratch */
    unsigned long r24;  /* scratch */
    unsigned long r25;  /* scratch */
    unsigned long r26;  /* scratch */
    unsigned long r27;  /* scratch */
    unsigned long r28;  /* scratch */
    unsigned long r29;  /* scratch */
    unsigned long r30;  /* scratch */
    unsigned long r31;  /* scratch */
    unsigned long ar_ccv;  /* compare/exchange value (scratch) */

    /*
     * Floating point registers that the kernel considers scratch:
     */
    struct pt_fpreg f6;  /* scratch */
    struct pt_fpreg f7;  /* scratch */
    struct pt_fpreg f8;  /* scratch */
    struct pt_fpreg f9;  /* scratch */
    struct pt_fpreg f10;  /* scratch */
    struct pt_fpreg f11;  /* scratch */
    unsigned long r4;  /* preserved */
    unsigned long r5;  /* preserved */
    unsigned long r6;  /* preserved */
    unsigned long r7;  /* preserved */
    unsigned long eml_unat;    /* used for emulating instruction */
    unsigned long pad0;     /* alignment pad */

};
typedef struct cpu_user_regs cpu_user_regs_t;

union vac {
    unsigned long value;
    struct {
        int a_int:1;
        int a_from_int_cr:1;
        int a_to_int_cr:1;
        int a_from_psr:1;
        int a_from_cpuid:1;
        int a_cover:1;
        int a_bsw:1;
        long reserved:57;
    };
};
typedef union vac vac_t;

union vdc {
    unsigned long value;
    struct {
        int d_vmsw:1;
        int d_extint:1;
        int d_ibr_dbr:1;
        int d_pmc:1;
        int d_to_pmd:1;
        int d_itm:1;
        long reserved:58;
    };
};
typedef union vdc vdc_t;

struct mapped_regs {
    union vac   vac;
    union vdc   vdc;
    unsigned long  virt_env_vaddr;
    unsigned long  reserved1[29];
    unsigned long  vhpi;
    unsigned long  reserved2[95];
    union {
        unsigned long  vgr[16];
        unsigned long bank1_regs[16]; // bank1 regs (r16-r31) when bank0 active
    };
    union {
        unsigned long  vbgr[16];
        unsigned long bank0_regs[16]; // bank0 regs (r16-r31) when bank1 active
    };
    unsigned long  vnat;
    unsigned long  vbnat;
    unsigned long  vcpuid[5];
    unsigned long  reserved3[11];
    unsigned long  vpsr;
    unsigned long  vpr;
    unsigned long  reserved4[76];
    union {
        unsigned long  vcr[128];
        struct {
            unsigned long dcr;  // CR0
            unsigned long itm;
            unsigned long iva;
            unsigned long rsv1[5];
            unsigned long pta;  // CR8
            unsigned long rsv2[7];
            unsigned long ipsr;  // CR16
            unsigned long isr;
            unsigned long rsv3;
            unsigned long iip;
            unsigned long ifa;
            unsigned long itir;
            unsigned long iipa;
            unsigned long ifs;
            unsigned long iim;  // CR24
            unsigned long iha;
            unsigned long rsv4[38];
            unsigned long lid;  // CR64
            unsigned long ivr;
            unsigned long tpr;
            unsigned long eoi;
            unsigned long irr[4];
            unsigned long itv;  // CR72
            unsigned long pmv;
            unsigned long cmcv;
            unsigned long rsv5[5];
            unsigned long lrr0;  // CR80
            unsigned long lrr1;
            unsigned long rsv6[46];
        };
    };
    union {
        unsigned long  reserved5[128];
        struct {
            unsigned long precover_ifs;
            unsigned long unat;  // not sure if this is needed until NaT arch is done
            int interrupt_collection_enabled; // virtual psr.ic
            /* virtual interrupt deliverable flag is evtchn_upcall_mask in
             * shared info area now. interrupt_mask_addr is the address
             * of evtchn_upcall_mask for current vcpu
             */
            unsigned char *interrupt_mask_addr;
            int pending_interruption;
            int incomplete_regframe; // see SDM vol2 6.8
            unsigned char vpsr_pp;
            unsigned char reserved5_2[7];
            unsigned long reserved5_1[3];
            int metaphysical_mode; // 1 = use metaphys mapping, 0 = use virtual
            int banknum; // 0 or 1, which virtual register bank is active
            unsigned long rrs[8]; // region registers
            unsigned long krs[8]; // kernel registers
            unsigned long pkrs[8]; // protection key registers
            unsigned long tmp[8]; // temp registers (e.g. for hyperprivops)
        };
    };
};
typedef struct mapped_regs mapped_regs_t;

struct vpd {
    struct mapped_regs vpd_low;
    unsigned long  reserved6[3456];
    unsigned long  vmm_avail[128];
    unsigned long  reserved7[4096];
};
typedef struct vpd vpd_t;

struct arch_vcpu_info {
};
typedef struct arch_vcpu_info arch_vcpu_info_t;

struct arch_shared_info {
    /* PFN of the start_info page.  */
    unsigned long start_info_pfn;

    /* Interrupt vector for event channel.  */
    int evtchn_vector;

    uint64_t pad[32];
};
typedef struct arch_shared_info arch_shared_info_t;

typedef unsigned long xen_callback_t;

struct ia64_tr_entry {
    unsigned long pte;
    unsigned long itir;
    unsigned long vadr;
    unsigned long rid;
};

struct vcpu_extra_regs {
    struct ia64_tr_entry itrs[8];
    struct ia64_tr_entry dtrs[8];
    unsigned long iva;
    unsigned long dcr;
    unsigned long event_callback_ip;
};

struct vcpu_guest_context {
#define VGCF_EXTRA_REGS (1<<1)	/* Get/Set extra regs.  */
    unsigned long flags;       /* VGCF_* flags */

    struct cpu_user_regs user_regs;
    struct vcpu_extra_regs extra_regs;
    unsigned long privregs_pfn;
};
typedef struct vcpu_guest_context vcpu_guest_context_t;
DEFINE_XEN_GUEST_HANDLE(vcpu_guest_context_t);

// dom0 vp op
#define __HYPERVISOR_ia64_dom0vp_op     __HYPERVISOR_arch_0
#define IA64_DOM0VP_ioremap             0       // map io space in machine
                                                // address to dom0 physical
                                                // address space.
                                                // currently physical
                                                // assignedg address equals to
                                                // machine address
#define IA64_DOM0VP_phystomach          1       // convert a pseudo physical
                                                // page frame number
                                                // to the corresponding
                                                // machine page frame number.
                                                // if no page is assigned,
                                                // INVALID_MFN or GPFN_INV_MASK
                                                // is returned depending on
                                                // domain's non-vti/vti mode.
#define IA64_DOM0VP_machtophys          3       // convert a machine page
                                                // frame number
                                                // to the corresponding
                                                // pseudo physical page frame
                                                // number of the caller domain
#define IA64_DOM0VP_zap_physmap         17      // unmap and free pages
                                                // contained in the specified
                                                // pseudo physical region
#define IA64_DOM0VP_add_physmap         18      // assigne machine page frane
                                                // to dom0's pseudo physical
                                                // address space.
// flags for page assignement to pseudo physical address space
#define _ASSIGN_readonly                0
#define ASSIGN_readonly                 (1UL << _ASSIGN_readonly)
#define ASSIGN_writable                 (0UL << _ASSIGN_readonly) // dummy flag
/* Internal only: memory attribute must be WC/UC/UCE.  */
#define _ASSIGN_nocache                 1
#define ASSIGN_nocache                  (1UL << _ASSIGN_nocache)

/* This structure has the same layout of struct ia64_boot_param, defined in
   <asm/system.h>.  It is redefined here to ease use.  */
struct xen_ia64_boot_param {
	unsigned long command_line;	/* physical address of cmd line args */
	unsigned long efi_systab;	/* physical address of EFI system table */
	unsigned long efi_memmap;	/* physical address of EFI memory map */
	unsigned long efi_memmap_size;	/* size of EFI memory map */
	unsigned long efi_memdesc_size;	/* size of an EFI memory map descriptor */
	unsigned int  efi_memdesc_version;	/* memory descriptor version */
	struct {
		unsigned short num_cols;	/* number of columns on console.  */
		unsigned short num_rows;	/* number of rows on console.  */
		unsigned short orig_x;	/* cursor's x position */
		unsigned short orig_y;	/* cursor's y position */
	} console_info;
	unsigned long fpswa;		/* physical address of the fpswa interface */
	unsigned long initrd_start;
	unsigned long initrd_size;
	unsigned long domain_start;	/* va where the boot time domain begins */
	unsigned long domain_size;	/* how big is the boot domain */
};

#endif /* !__ASSEMBLY__ */

/* Address of shared_info in domain virtual space.
   This is the default address, for compatibility only.  */
#define XSI_BASE			0xf100000000000000

/* Size of the shared_info area (this is not related to page size).  */
#define XSI_SHIFT			14
#define XSI_SIZE			(1 << XSI_SHIFT)
/* Log size of mapped_regs area (64 KB - only 4KB is used).  */
#define XMAPPEDREGS_SHIFT		12
/* Offset of XASI (Xen arch shared info) wrt XSI_BASE.  */
#define XMAPPEDREGS_OFS			XSI_SIZE

/* Hyperprivops.  */
#define HYPERPRIVOP_RFI			0x1
#define HYPERPRIVOP_RSM_DT		0x2
#define HYPERPRIVOP_SSM_DT		0x3
#define HYPERPRIVOP_COVER		0x4
#define HYPERPRIVOP_ITC_D		0x5
#define HYPERPRIVOP_ITC_I		0x6
#define HYPERPRIVOP_SSM_I		0x7
#define HYPERPRIVOP_GET_IVR		0x8
#define HYPERPRIVOP_GET_TPR		0x9
#define HYPERPRIVOP_SET_TPR		0xa
#define HYPERPRIVOP_EOI			0xb
#define HYPERPRIVOP_SET_ITM		0xc
#define HYPERPRIVOP_THASH		0xd
#define HYPERPRIVOP_PTC_GA		0xe
#define HYPERPRIVOP_ITR_D		0xf
#define HYPERPRIVOP_GET_RR		0x10
#define HYPERPRIVOP_SET_RR		0x11
#define HYPERPRIVOP_SET_KR		0x12
#define HYPERPRIVOP_FC			0x13
#define HYPERPRIVOP_GET_CPUID		0x14
#define HYPERPRIVOP_GET_PMD		0x15
#define HYPERPRIVOP_GET_EFLAG		0x16
#define HYPERPRIVOP_SET_EFLAG		0x17
#define HYPERPRIVOP_RSM_BE		0x18
#define HYPERPRIVOP_GET_PSR		0x19
#define HYPERPRIVOP_MAX			0x19

#endif /* __HYPERVISOR_IF_IA64_H__ */

/*
 * Local variables:
 * mode: C
 * c-set-style: "BSD"
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: nil
 * End:
 */
