/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#include <gdk/gdkkeysyms.h>

#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixdata.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "xfmedia-internal.h"

#include "xfmedia-xine.h"
#include "mainwin.h"
#include <xfmedia/xfmedia-playlist.h>
#include <xfmedia/xfmedia-playlist-queue.h>
#include <xfmedia/xfmedia-settings.h>
#include "trayicon.h"
#include "keybindings.h"
#include "xfmedia-keybindings.h"
#include "main.h"
#include "mediamarks.h"
#include "playlist-files.h"

#define MAX_SHUFFLE_QUEUE_DEPTH 100

static gboolean
xfmedia_mainwin_playlist_load_info_idled(gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    guint idx_start, idx_end, i;
    gchar *title = NULL, *filename = NULL;
    gint length = -1;
    
    gdk_threads_enter();
    
    if(!xfmedia_playlist_get_visible_range(mwin->plist, &idx_start, &idx_end)) {
        gdk_threads_leave();
        return FALSE;
    }
    
    for(i = idx_start; i <= idx_end; i++) {
        if(xfmedia_playlist_get_metadata_loaded(mwin->plist, i))
            continue;
        
        if(!xfmedia_playlist_get(mwin->plist, i, NULL, NULL, &filename) || !filename)
            continue;
        
        title = NULL;
        if(xfmedia_mainwin_get_file_info(mwin, filename, &title, &length)) {
            if(!title)
                title = xfmedia_filename_to_name(filename);
            xfmedia_playlist_modify_entry(mwin->plist, i, title, length, NULL);
            xfmedia_playlist_set_metadata_loaded(mwin->plist, i, TRUE);
            g_free(title);
        }
        g_free(filename);
        
        while(gtk_events_pending())
            gtk_main_iteration();
    }
    
    gdk_threads_leave();
    
    return FALSE;
}

void
xfmedia_mainwin_playlist_scrolled_cb(XfmediaPlaylist *plist, gpointer user_data)
{
    g_idle_add(xfmedia_mainwin_playlist_load_info_idled, user_data);
    xfmedia_plugins_forward_signal_VOID("playlist-scrolled");
}

void
xfmedia_mainwin_playlist_entry_activated_cb(XfmediaPlaylist *plist, gint idx,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_plugins_forward_signal_INT("playlist-entry-activated", idx);
    
    if(!xfmedia_mainwin_play_file_at_index(mwin, idx)) {
        xfce_message_dialog(GTK_WINDOW(mwin->window), "Xfmedia",
                GTK_STOCK_DIALOG_INFO, _("Unable to play file."),
                _("Xfmedia was unable to play the selected file.  Be sure that the file exists and that you have permission to access it."),
                GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
    }
}

void
xfmedia_mainwin_playlist_shuffle_toggled_cb(XfmediaPlaylist *plist,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    DBG("got shuffle-toggled");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->shuffle_mi),
            !gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mwin->shuffle_mi)));
    xfmedia_plugins_forward_signal_VOID("playlist-shuffle-toggled");
}

void
xfmedia_mainwin_playlist_repeat_toggled_cb(XfmediaPlaylist *plist,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    DBG("got repeat-toggled");
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->repeat_mi),
            !gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(mwin->repeat_mi)));
    xfmedia_plugins_forward_signal_VOID("playlist-repeat-toggled");
}

void
xfmedia_mainwin_playlist_cleared_cb(XfmediaPlaylist *plist, gpointer user_data)
{
    xfmedia_settings_set_int("/xfmedia/playlist/playlist_position", -1);
    xfmedia_plugins_forward_signal_VOID("playlist-cleared");
}

void
xfmedia_mainwin_playlist_opened_cb(XfmediaPlaylist *plist,
        const gchar *filename, gpointer user_data)
{
    xfmedia_plugins_forward_signal_STRING("playlist-opened", filename);
}

void
xfmedia_mainwin_playlist_saved_cb(XfmediaPlaylist *plist, const gchar *filename,
        gpointer user_data)
{
    xfmedia_plugins_forward_signal_STRING("playlist-saved", filename);
}

void
xfmedia_mainwin_playlist_entry_added_cb(XfmediaPlaylist *plist, gint idx,
        gpointer user_data)
{
    xfmedia_plugins_forward_signal_INT("playlist-entry-added", idx);
}

void
xfmedia_mainwin_playlist_entry_changed_cb(XfmediaPlaylist *plist, gint idx,
        gpointer user_data)
{
    xfmedia_plugins_forward_signal_INT("playlist-entry-changed", idx);
}

void
xfmedia_mainwin_playlist_entry_removed_cb(XfmediaPlaylist *plist,
        const gchar *uri, gpointer user_data)
{
    xfmedia_plugins_forward_signal_STRING("playlist-entry-removed", uri);
}

static void
file_info_edited_cb(GtkEditable *e, gpointer user_data)
{
    XfmediaTagInfo *tag_info = user_data;
    gint type = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(e), "xfmedia-tag-info-type"));
    GtkWidget *save_btn = g_object_get_data(G_OBJECT(e), "xfmedia-save-button");
    gchar *str;
    
    str = gtk_editable_get_chars(e, 0, -1);
    xfmedia_tag_set_info(tag_info, type, str);
    g_free(str);
    
    if(!GTK_WIDGET_SENSITIVE(save_btn))
        gtk_widget_set_sensitive(save_btn, TRUE);
}

static void
file_info_save_confirm(GtkWidget *w, XfmediaTagInfo *tag_info)
{
    GtkWindow *parent = GTK_WINDOW(gtk_widget_get_toplevel(w));
    
    if(xfmedia_tag_is_dirty(tag_info)) {
        gint resp = xfce_message_dialog(parent, _("Metadata Changed"),
                GTK_STOCK_DIALOG_QUESTION, 
                _("You have edited this file's metadata."),
                _("Changes will be lost unless they are saved.  Do you wish to save changes?"),
                XFCE_CUSTOM_STOCK_BUTTON, _("_Discard Changes"), GTK_STOCK_NO, GTK_RESPONSE_NO,
                XFCE_CUSTOM_STOCK_BUTTON, _("_Save Changes"), GTK_STOCK_YES, GTK_RESPONSE_YES,
                NULL);
        if(resp == GTK_RESPONSE_YES)
            xfmedia_tag_flush(tag_info);
    }
}

static gboolean
file_info_delete_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    file_info_save_confirm(w, user_data);
    return FALSE;
}

static void
file_info_destroy_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    XfmediaTagInfo *tag_info = g_object_get_data(G_OBJECT(w), "xfmedia-tag-info");
    
    xfmedia_tag_close(tag_info);
    xfmedia_mainwin_enable_autohide(mwin);
}

void
xfmedia_mainwin_playlist_file_info_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    XfmediaTagInfo *tag_info;
    gint idx;
#define TMPBUF 64
    gchar *filename = NULL, tmpbuf[TMPBUF], *tmpbuf1 = NULL, *value;
    const gchar *value_c;
    GtkWidget *win, *lbl, *entry, *topvbox, *hbox, *frame, *frame_bin, *vbox,
              *btn, *cbtn, *save_btn = NULL;
    GtkSizeGroup *sg;
    struct stat st;
    gint pos_stream, pos_time, length_time = 0, old_xine_status, old_xine_speed;
    guint32 fourcc;
    gboolean tag_writable;
    
    idx = xfmedia_playlist_get_selected(mwin->plist);
    if(idx < 0)
        return;
    
    if(!xfmedia_playlist_get(mwin->plist, idx, NULL, NULL, &filename))
        return;
    
    old_xine_status = xfmedia_xine_get_status(mwin->xfx);
    old_xine_speed = xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED);
    
    g_mutex_lock(mwin->extra_xine_mx);
    
    if(!xine_open(mwin->extra_stream, filename)) {
        gchar *secondary;
        
        g_mutex_unlock(mwin->extra_xine_mx);
        
        secondary = g_strdup_printf(_("Xfmedia was unable to open '%s'.  Be sure that the file exists and that you have permission to access it."), filename);
        xfce_message_dialog(GTK_WINDOW(mwin->window), "Xfmedia",
                GTK_STOCK_DIALOG_INFO, _("Unable to open file."),
                secondary, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
        g_free(secondary);
        g_free(filename);
        return;
    }
    
    tag_info = xfmedia_tag_open(mwin, filename);
    tag_writable = xfmedia_tag_is_writable(tag_info);
    if(tag_writable)
        save_btn = gtk_button_new_from_stock(GTK_STOCK_SAVE);
    
    /* this is kinda sucky.  i have to drop the mutex so the xfmedia_tag_*()
     * functions will work if/when they need to fall back to using xine.  this
     * is of course somewhat icky, but we'll leave it for now until there's an
     * easier/better way of getting metadata.
     */
    xine_close(mwin->extra_stream);
    g_mutex_unlock(mwin->extra_xine_mx);
    
    win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(win), _("File Info"));
    gtk_window_set_transient_for(GTK_WINDOW(win), GTK_WINDOW(mwin->window));
    gtk_container_set_border_width(GTK_CONTAINER(win), BORDER);
    g_object_set_data(G_OBJECT(win), "xfmedia-tag-info", tag_info);
    g_signal_connect(G_OBJECT(win), "delete-event",
            G_CALLBACK(file_info_delete_cb), tag_info);
    g_signal_connect(G_OBJECT(win), "destroy",
            G_CALLBACK(file_info_destroy_cb), mwin);
    
    topvbox = gtk_vbox_new(FALSE, BORDER/2);
    gtk_widget_show(topvbox);
    gtk_container_add(GTK_CONTAINER(win), topvbox);
    
    sg = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
    
    frame = xfmedia_create_framebox(_("Filename"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    lbl = exo_ellipsized_label_new(filename);
    exo_ellipsized_label_set_ellipsize(EXO_ELLIPSIZED_LABEL(lbl),
            EXO_PANGO_ELLIPSIZE_START);
    gtk_label_set_selectable(GTK_LABEL(lbl), TRUE);
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_label_set_text(GTK_LABEL(lbl), filename);
    gtk_widget_show(lbl);
    gtk_container_add(GTK_CONTAINER(frame_bin), lbl);
    
    frame = xfmedia_create_framebox(_("Media Info"), &frame_bin);
    gtk_widget_show(frame);
    gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
    
    vbox = gtk_vbox_new(FALSE, BORDER);
    gtk_widget_show(vbox);
    gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Title:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_TITLE);
    if(value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        g_free(value);
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), tag_writable);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    if(tag_writable) {
        g_object_set_data(G_OBJECT(entry), "xfmedia-tag-info-type",
                GINT_TO_POINTER(XINE_META_INFO_TITLE));
        g_object_set_data(G_OBJECT(entry), "xfmedia-save-button", save_btn);
        g_signal_connect(G_OBJECT(entry), "changed",
                G_CALLBACK(file_info_edited_cb), tag_info);
    }
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Artist:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_ARTIST);
    if(value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        g_free(value);
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), tag_writable);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    if(tag_writable) {
        g_object_set_data(G_OBJECT(entry), "xfmedia-tag-info-type",
                GINT_TO_POINTER(XINE_META_INFO_ARTIST));
        g_object_set_data(G_OBJECT(entry), "xfmedia-save-button", save_btn);
        g_signal_connect(G_OBJECT(entry), "changed",
                G_CALLBACK(file_info_edited_cb), tag_info);
    }
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Album:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_ALBUM);
    if(value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        g_free(value);
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), tag_writable);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    if(tag_writable) {
        g_object_set_data(G_OBJECT(entry), "xfmedia-tag-info-type",
                GINT_TO_POINTER(XINE_META_INFO_ALBUM));
        g_object_set_data(G_OBJECT(entry), "xfmedia-save-button", save_btn);
        g_signal_connect(G_OBJECT(entry), "changed",
                G_CALLBACK(file_info_edited_cb), tag_info);
    }
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Genre:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_GENRE);
    if(value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        g_free(value);
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), tag_writable);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    if(tag_writable) {
        g_object_set_data(G_OBJECT(entry), "xfmedia-tag-info-type",
                GINT_TO_POINTER(XINE_META_INFO_GENRE));
        g_object_set_data(G_OBJECT(entry), "xfmedia-save-button", save_btn);
        g_signal_connect(G_OBJECT(entry), "changed",
                G_CALLBACK(file_info_edited_cb), tag_info);
    }
    
    lbl = gtk_label_new(_("Track:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_TRACK_NUMBER);
    if(value && *value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(value));
    } else
        gtk_entry_set_width_chars(GTK_ENTRY(entry), 2);
    if(value)
        g_free(value);
    gtk_editable_set_editable(GTK_EDITABLE(entry), tag_writable);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
    if(tag_writable) {
        g_object_set_data(G_OBJECT(entry), "xfmedia-tag-info-type",
                GINT_TO_POINTER(XINE_META_INFO_TRACK_NUMBER));
        g_object_set_data(G_OBJECT(entry), "xfmedia-save-button", save_btn);
        g_signal_connect(G_OBJECT(entry), "changed",
                G_CALLBACK(file_info_edited_cb), tag_info);
    }
    
    lbl = gtk_label_new(_("Year:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_YEAR);
    if(value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        g_free(value);
    }
    gtk_entry_set_width_chars(GTK_ENTRY(entry), 4);
    gtk_editable_set_editable(GTK_EDITABLE(entry), tag_writable);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
    if(tag_writable) {
        g_object_set_data(G_OBJECT(entry), "xfmedia-tag-info-type",
                GINT_TO_POINTER(XINE_META_INFO_YEAR));
        g_object_set_data(G_OBJECT(entry), "xfmedia-save-button", save_btn);
        g_signal_connect(G_OBJECT(entry), "changed",
                G_CALLBACK(file_info_edited_cb), tag_info);
    }
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Comment:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    value = xfmedia_tag_get_info(tag_info, XINE_META_INFO_COMMENT);
    if(value) {
        gtk_entry_set_text(GTK_ENTRY(entry), value);
        g_free(value);
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), tag_writable);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    g_object_set_data(G_OBJECT(entry), "xfmedia-tag-info-type",
            GINT_TO_POINTER(XINE_META_INFO_COMMENT));
    g_object_set_data(G_OBJECT(entry), "xfmedia-save-button", save_btn);
    g_signal_connect(G_OBJECT(entry), "changed",
            G_CALLBACK(file_info_edited_cb), tag_info);
    
    /* now we're finished with the actual metadata tag stuff, so we need
     * the xine extra stream lock back */
    g_mutex_lock(mwin->extra_xine_mx);
    /* there's really no way this can fail here */
    xine_open(mwin->extra_stream, filename);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new(_("Length:"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_size_group_add_widget(sg, lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    if(xine_get_pos_length(mwin->extra_stream, &pos_stream, &pos_time, &length_time)) {
        length_time /= 1000;
        g_snprintf(tmpbuf, TMPBUF, "%02d:%02d", length_time/60, length_time%60);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    lbl = gtk_label_new(_("File size (b):"));
    gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    if(!stat(filename, &st)) {
        tmpbuf1 = xfmedia_num_prettify((gint)st.st_size);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
    }
    gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    
    if(xine_get_stream_info(mwin->extra_stream, XINE_STREAM_INFO_HAS_AUDIO)) {
        frame = xfmedia_create_framebox(_("Audio Info"), &frame_bin);
        gtk_widget_show(frame);
        gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
        
        vbox = gtk_vbox_new(FALSE, BORDER);
        gtk_widget_show(vbox);
        gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Codec:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        value_c = xine_get_meta_info(mwin->extra_stream,
                XINE_META_INFO_AUDIOCODEC);
        if(value_c)
            gtk_entry_set_text(GTK_ENTRY(entry), value_c);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Samples/sec:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        tmpbuf1 = xfmedia_num_prettify(xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_AUDIO_SAMPLERATE));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Bits/sample:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%d",
        		xine_get_stream_info(mwin->extra_stream,
                	XINE_STREAM_INFO_AUDIO_BITS));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Channels:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%d",
        		xine_get_stream_info(mwin->extra_stream,
                	XINE_STREAM_INFO_AUDIO_CHANNELS));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Bitrate (kbit):"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        tmpbuf1 = xfmedia_num_prettify(xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_AUDIO_BITRATE)/1000);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    }
    
    if(xine_get_stream_info(mwin->extra_stream, XINE_STREAM_INFO_HAS_VIDEO)) {
        frame = xfmedia_create_framebox(_("Video Info"), &frame_bin);
        gtk_widget_show(frame);
        gtk_box_pack_start(GTK_BOX(topvbox), frame, FALSE, FALSE, 0);
        
        vbox = gtk_vbox_new(FALSE, BORDER);
        gtk_widget_show(vbox);
        gtk_container_add(GTK_CONTAINER(frame_bin), vbox);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Codec:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        value_c = xine_get_meta_info(mwin->extra_stream,
                XINE_META_INFO_VIDEOCODEC);
        if(value_c)
            gtk_entry_set_text(GTK_ENTRY(entry), value_c);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("FourCC:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        fourcc = xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_VIDEO_FOURCC);
        tmpbuf[0] = ((gchar *)&fourcc)[0];
        tmpbuf[1] = ((gchar *)&fourcc)[1];
        tmpbuf[2] = ((gchar *)&fourcc)[2];
        tmpbuf[3] = ((gchar *)&fourcc)[3];
        tmpbuf[4] = 0;
        if(g_utf8_validate(tmpbuf, -1, NULL))
            gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), 4);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        hbox = gtk_hbox_new(FALSE, BORDER);
        gtk_widget_show(hbox);
        gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        
        lbl = gtk_label_new(_("Frame size:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_size_group_add_widget(sg, lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%dx%d",
                xine_get_stream_info(mwin->extra_stream,
                    XINE_STREAM_INFO_VIDEO_WIDTH),
                xine_get_stream_info(mwin->extra_stream,
                    XINE_STREAM_INFO_VIDEO_HEIGHT));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Channels:"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        g_snprintf(tmpbuf, TMPBUF, "%d", xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_VIDEO_CHANNELS));
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf));
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
        
        lbl = gtk_label_new(_("Bitrate (kbit):"));
        gtk_misc_set_alignment(GTK_MISC(lbl), 0.0, 0.5);
        gtk_widget_show(lbl);
        gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
        
        entry = gtk_entry_new();
        tmpbuf1 = xfmedia_num_prettify(xine_get_stream_info(mwin->extra_stream,
                XINE_STREAM_INFO_VIDEO_BITRATE)/1000);
        gtk_entry_set_text(GTK_ENTRY(entry), tmpbuf1);
        gtk_entry_set_width_chars(GTK_ENTRY(entry), strlen(tmpbuf1));
        g_free(tmpbuf1);
        gtk_editable_set_editable(GTK_EDITABLE(entry), FALSE);
        gtk_widget_show(entry);
        gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    }
    
    xine_close(mwin->extra_stream);
    g_mutex_unlock(mwin->extra_xine_mx);
    g_free(filename);
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_end(GTK_BOX(topvbox), hbox, FALSE, FALSE, 0);
    
    cbtn = btn = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(file_info_save_confirm), tag_info);
    g_signal_connect_data(G_OBJECT(btn), "clicked",
            G_CALLBACK(gtk_widget_destroy), win,
            NULL, G_CONNECT_AFTER|G_CONNECT_SWAPPED);
    
    if(tag_writable) {
        gtk_widget_set_sensitive(save_btn, FALSE);
        gtk_widget_show(save_btn);
        gtk_box_pack_end(GTK_BOX(hbox), save_btn, FALSE, FALSE, 0);
        g_signal_connect_swapped(G_OBJECT(save_btn), "clicked",
                G_CALLBACK(xfmedia_tag_flush), tag_info);
        g_signal_connect_after(G_OBJECT(save_btn), "clicked",
                G_CALLBACK(gtk_widget_set_sensitive), GINT_TO_POINTER(FALSE));
    }
    
    if(old_xine_status == XINE_STATUS_PLAY
            && old_xine_speed != xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED))
    {
        DBG("xine changed speed while getting stream info, fixing");
        xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_SPEED, old_xine_speed);
    }
    
    gtk_widget_show(win);
    gtk_widget_grab_focus(cbtn);
    
    xfmedia_mainwin_disable_autohide(mwin);
#undef TMPBUF
}

void
xfmedia_mainwin_playlist_add_mm_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint idx;
    gchar *title = NULL, *filename = NULL;
    
    idx = xfmedia_playlist_get_selected(mwin->plist);
    if(idx < 0)
        return;
    
    if(!xfmedia_playlist_get(mwin->plist, idx, &title, NULL, &filename))
        return;
    
    xfmedia_mediamarks_add(mwin, GTK_WINDOW(mwin->window), title, filename);
    
    g_free(title);
    g_free(filename);
}

void
xfmedia_mainwin_playlist_add_autoplay_cb(GtkMenuItem *mi, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GtkWidget *lbl = gtk_bin_get_child(GTK_BIN(mi));
    const gchar *plugin_id = gtk_label_get_text(GTK_LABEL(lbl));
    GList *mrls, *l;
    guint nadded = 0;
    
    if(plugin_id) {
        mrls = xfmedia_xine_get_autoplay_mrls(mwin->xfx, plugin_id);
        for(l = mrls; l; l = l->next) {
            const gchar *filename = l->data;
            gchar *title_utf8 = NULL;
            gint length = -1;
            gboolean metadata_loaded;
            
            if(!filename || !*filename)
                continue;
            
            metadata_loaded = xfmedia_mainwin_get_file_info(mwin, filename,
                    &title_utf8, &length);
            if(!title_utf8)
                title_utf8 = xfmedia_filename_to_name(filename);
            xfmedia_playlist_append_entry(mwin->plist, title_utf8, length,
                    filename, metadata_loaded);
            
            g_free(title_utf8);
            nadded++;
            
            while(gtk_events_pending())
                gtk_main_iteration();
        }
        if(mrls)
            g_list_free(mrls);
    }
}

static void
mainwin_add_url_dialog_response_cb(GtkWidget *dlg, gint response,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GtkWidget *entry;
    gchar *url;
    
    entry = g_object_get_data(G_OBJECT(dlg), "xfmedia-url-entry");
    
    url = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, -1);
    if(url && *url)
        xfmedia_playlist_append_entry(mwin->plist, NULL, -1, url, FALSE);
    g_free(url);
    
    gtk_widget_destroy(dlg);
    xfmedia_mainwin_enable_autohide(mwin);
    
    /* yuck */
    g_signal_emit_by_name(G_OBJECT(mwin->plist), "playlist-scrolled", 0);
    //xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}

void
xfmedia_mainwin_add_url_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    static GtkWidget *dlg = NULL;
    GtkWidget *lbl, *entry, *topvbox, *hbox, *toplevel;
    
    if(dlg) {
        gtk_window_present(GTK_WINDOW(dlg));
        return;
    }
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(mwin->window));
    dlg = gtk_dialog_new_with_buttons(_("Add URL"), GTK_WINDOW(toplevel),
            GTK_DIALOG_NO_SEPARATOR|GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OK,
            GTK_RESPONSE_ACCEPT, NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dlg), GTK_RESPONSE_ACCEPT);
    gtk_container_set_border_width(GTK_CONTAINER(dlg), BORDER);
    g_signal_connect(G_OBJECT(dlg), "response",
            G_CALLBACK(mainwin_add_url_dialog_response_cb), mwin);
    g_signal_connect_swapped(G_OBJECT(dlg), "destroy",
            G_CALLBACK(xfmedia_widget_zero_pointer), &dlg);
    
    topvbox = GTK_DIALOG(dlg)->vbox;
    
    hbox = gtk_hbox_new(FALSE, BORDER);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(topvbox), hbox, FALSE, FALSE, 0);
    
    lbl = gtk_label_new_with_mnemonic(_("_URL:"));
    gtk_widget_show(lbl);
    gtk_box_pack_start(GTK_BOX(hbox), lbl, FALSE, FALSE, 0);
    
    entry = gtk_entry_new();
    gtk_entry_set_activates_default(GTK_ENTRY(entry), TRUE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    gtk_label_set_mnemonic_widget(GTK_LABEL(lbl), entry);
    g_object_set_data(G_OBJECT(dlg), "xfmedia-url-entry", entry);
    
    gtk_widget_set_size_request(dlg, 400, -1);
    gtk_widget_show(dlg);
    xfmedia_mainwin_disable_autohide(mwin);
}

enum
{
    SORT_FILENAME = 0,
    SORT_TITLE,
    SORT_ARTIST_TITLE,
    SORT_ARTIST_ALBUM_TRACKNUM
};

static const gchar const *sort_combo_strings[] = {
    N_("Sort by filename"),
    N_("Sort by title"),
    N_("Sort by artist, title"),
    N_("Sort by artist, album, track number"),
    NULL
};

struct SortTitleData
{
    gchar *filename;
    gchar *title_utf8;
};

struct SortArtistTitleData
{
    gchar *filename;
    gchar *artist_utf8;
    gchar *title_utf8;
};

struct SortArtistAlbumTrackData
{
    gchar *filename;
    gchar *artist_utf8;
    gchar *album_utf8;
    gint track;
};

static gint
__compare_func_filename(gconstpointer a, gconstpointer b)
{
    struct SortTitleData *a1 = (struct SortTitleData *)a;
    struct SortTitleData *b1 = (struct SortTitleData *)b;
    
    return g_ascii_strcasecmp(a1->filename, b1->filename);
}

static gint
__compare_func_title(gconstpointer a, gconstpointer b)
{
    struct SortTitleData *a1 = (struct SortTitleData *)a;
    struct SortTitleData *b1 = (struct SortTitleData *)b;
    
    return g_utf8_collate(a1->title_utf8, b1->title_utf8);
}

static gint
__compare_func_artist_title(gconstpointer a, gconstpointer b)
{
    struct SortArtistTitleData *a1 = (struct SortArtistTitleData *)a;
    struct SortArtistTitleData *b1 = (struct SortArtistTitleData *)b;
    gint cmp;
    
    cmp = g_utf8_collate(a1->artist_utf8, b1->artist_utf8);
    if(cmp == 0)
        cmp = g_utf8_collate(a1->title_utf8, b1->title_utf8);
    
    return cmp;
}

static gint
__compare_func_artist_album_tracknumber(gconstpointer a, gconstpointer b)
{
    struct SortArtistAlbumTrackData *a1 = (struct SortArtistAlbumTrackData *)a;
    struct SortArtistAlbumTrackData *b1 = (struct SortArtistAlbumTrackData *)b;
    gint cmp;
    
    cmp = g_utf8_collate(a1->artist_utf8, b1->artist_utf8);
    if(cmp == 0) {
        cmp = g_utf8_collate(a1->album_utf8, b1->album_utf8);
        if(cmp == 0)
            cmp = a1->track < b1->track ? -1 : (a1->track > b1->track ? 1 : 0);
    }
    
    return cmp;
}

static GList *
add_sorted_file_list(XfmediaMainwin *mwin, GList *file_list, gint sort_type)
{
    GList *sorted_file_list = NULL, *l;
    
    switch(sort_type) {
        case SORT_FILENAME:
            for(l = file_list; l; l = l->next) {
                struct SortTitleData *std = g_new0(struct SortTitleData, 1);
                std->filename = l->data;
                sorted_file_list = g_list_insert_sorted(sorted_file_list, std,
                        (GCompareFunc)__compare_func_filename);
            }
            break;
        
        case SORT_TITLE:
            for(l = file_list; l; l = l->next) {
                struct SortTitleData *std = g_new(struct SortTitleData, 1);
                std->filename = l->data;
                std->title_utf8 = xfmedia_xine_get_meta_info_helper(mwin, l->data, XINE_META_INFO_TITLE);
                sorted_file_list = g_list_insert_sorted(sorted_file_list, std,
                        (GCompareFunc)__compare_func_title);
            }
            break;
        
        case SORT_ARTIST_TITLE:
            for(l = file_list; l; l = l->next) {
                struct SortArtistTitleData *satd = g_new(struct SortArtistTitleData, 1);
                satd->filename = l->data;
                satd->artist_utf8 = xfmedia_xine_get_meta_info_helper(mwin, l->data, XINE_META_INFO_ARTIST);
                satd->title_utf8 = xfmedia_xine_get_meta_info_helper(mwin, l->data, XINE_META_INFO_TITLE);
                sorted_file_list = g_list_insert_sorted(sorted_file_list, satd,
                        (GCompareFunc)__compare_func_artist_title);
            }
            break;
        
        case SORT_ARTIST_ALBUM_TRACKNUM:
            for(l = file_list; l; l = l->next) {
                gchar *tmp;
                struct SortArtistAlbumTrackData *saatd = g_new(struct SortArtistAlbumTrackData, 1);
                saatd->filename = l->data;
                saatd->artist_utf8 = xfmedia_xine_get_meta_info_helper(mwin, l->data, XINE_META_INFO_ARTIST);
                saatd->album_utf8 = xfmedia_xine_get_meta_info_helper(mwin, l->data, XINE_META_INFO_ALBUM);
                tmp = xfmedia_xine_get_meta_info_helper(mwin, l->data, XINE_META_INFO_TRACK_NUMBER);
                saatd->track = atoi(tmp);
                g_free(tmp);
                if(saatd->track < 0)
                    saatd->track = 0;
                sorted_file_list = g_list_insert_sorted(sorted_file_list, saatd,
                        (GCompareFunc)__compare_func_artist_album_tracknumber);
            }
            break;
        
        default:
            g_return_val_if_reached(NULL);
            break;  /* NOTREACHED */
    }
    
    return sorted_file_list;
}

static guint
recurse_add_dirs(XfmediaMainwin *mwin, const gchar *dirname, gint add_index,
        gint sort_type)
{
    GDir *dir;
    const gchar *file, *allowed_extensions;
    gchar fullpath[PATH_MAX], *p;
    GList *file_list = NULL, *dir_list = NULL, *sorted_file_list = NULL, *l;
    guint nadded = 0;
    
    allowed_extensions = xfmedia_playlist_get_allowed_extensions(mwin->plist);
    
    dir = g_dir_open(dirname, 0, NULL);
    if(!dir)
        return 0;
    
    while((file = g_dir_read_name(dir))) {
        gboolean is_dir;
        
        g_snprintf(fullpath, PATH_MAX, "%s%s%s", dirname, G_DIR_SEPARATOR_S, file);
        is_dir = g_file_test(fullpath, G_FILE_TEST_IS_DIR);
        
        if(!is_dir && allowed_extensions) {
            p = g_strrstr(file, ".");
            if(!p || !*(p+1))
                continue;
            if(!strstr(allowed_extensions, p))
                continue;
        }
        
        if(is_dir) {
            dir_list = g_list_insert_sorted(dir_list, g_strdup(fullpath),
                    (GCompareFunc)g_ascii_strcasecmp);
        } else
            file_list = g_list_prepend(file_list, g_strdup(fullpath));
    }
    g_dir_close(dir);
    
    for(l = dir_list; l; l = l->next) {
        gint nadded_local;
        nadded_local = recurse_add_dirs(mwin, (const gchar *)l->data, add_index,
                sort_type);
        if(add_index != -1)
            add_index += nadded_local;
        nadded += nadded_local;
        g_free(l->data);
    }
    if(dir_list)
        g_list_free(dir_list);
    
    if(file_list) {
        sorted_file_list = add_sorted_file_list(mwin, file_list, sort_type);
        if(file_list)
            g_list_free(file_list);
        
        gdk_threads_enter();
        for(l = sorted_file_list; l; l = l->next) {
            struct SortTitleData *std = l->data;
            gchar *title_utf8 = xfmedia_filename_to_name(std->filename);
            if(add_index == -1) {
                xfmedia_playlist_append_entry(mwin->plist, title_utf8, -1,
                        std->filename, FALSE);
            } else {
                xfmedia_playlist_insert_entry(mwin->plist, add_index++,
                        title_utf8, -1, std->filename, FALSE);
            }
            nadded++;
            g_free(title_utf8);
            
            /* this part is kinda sketchy */
            g_free(std->filename);
            g_free(std->title_utf8);
            if(sort_type == SORT_ARTIST_TITLE || sort_type == SORT_ARTIST_ALBUM_TRACKNUM)
                g_free(((struct SortArtistAlbumTrackData *)std)->album_utf8);
            g_free(l->data);
        }
        gdk_threads_leave();
        
        if(sorted_file_list)
            g_list_free(sorted_file_list);
    }
    
    return nadded;
}

struct RecurseData {
    XfmediaMainwin *mwin;
    gchar *basedir;
    gint add_index;
    gint sort_type;
};

static gpointer
recurse_add_dirs_th(gpointer data)
{
    struct RecurseData *recurse_data = data;
    guint nadded;
    
    nadded = recurse_add_dirs(recurse_data->mwin, recurse_data->basedir,
            recurse_data->add_index, recurse_data->sort_type);
    
    if(nadded > 0) {
        gdk_threads_enter();
        g_signal_emit_by_name(G_OBJECT(recurse_data->mwin->plist),
                "playlist-scrolled", 0);
        gdk_threads_leave();
    }
    
    g_free(recurse_data->basedir);
    g_free(recurse_data);
    
    return NULL;
}

#define MY_FOLDER_SELECT_RESPONSE 749837
static void
mainwin_add_dir_chooser_response_cb(GtkWidget *chooser, gint response,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    struct RecurseData *recurse_data;
    gchar *dirname;
    
    if(response == MY_FOLDER_SELECT_RESPONSE) {
        dirname = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
        if(dirname) {
            GtkWidget *combo = gtk_file_chooser_get_extra_widget(GTK_FILE_CHOOSER(chooser));
            recurse_data = g_new0(struct RecurseData, 1);
            recurse_data->mwin = mwin;
            recurse_data->basedir = dirname;
            recurse_data->add_index = -1;
            recurse_data->sort_type = gtk_combo_box_get_active(GTK_COMBO_BOX(combo));
            xfmedia_settings_set_int("/xfmedia/playlist/add_sort_type",
                    recurse_data->sort_type);
            g_free(mwin->chooser_last_dir);
            mwin->chooser_last_dir = g_strdup(dirname);
            //plist->priv->is_dirty = TRUE;
            g_thread_create((GThreadFunc)recurse_add_dirs_th, recurse_data,
                    FALSE, NULL);
        }
    }
    
    if(response == MY_FOLDER_SELECT_RESPONSE || response == GTK_RESPONSE_CANCEL) {
        gtk_widget_destroy(chooser);
        xfmedia_mainwin_enable_autohide(mwin);
    }
}

void
xfmedia_mainwin_add_dir_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    static GtkWidget *chooser = NULL;
    GtkWidget *toplevel, *combo;
    gint i;
    
    if(chooser) {
        gtk_window_present(GTK_WINDOW(chooser));
        return;
    }
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(mwin->window));
    chooser = gtk_file_chooser_dialog_new(_("Add Directory"),
            GTK_WINDOW(mwin->window), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN,
            MY_FOLDER_SELECT_RESPONSE, NULL);
    if(mwin->chooser_last_dir)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(chooser),
                mwin->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    g_signal_connect(G_OBJECT(chooser), "response",
            G_CALLBACK(mainwin_add_dir_chooser_response_cb), mwin);
    g_signal_connect_swapped(G_OBJECT(chooser), "destroy",
            G_CALLBACK(xfmedia_widget_zero_pointer), &chooser);
    
    combo = gtk_combo_box_new_text();
    for(i = 0; sort_combo_strings[i]; i++)
        gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _(sort_combo_strings[i]));
    gtk_combo_box_set_active(GTK_COMBO_BOX(combo),
            xfmedia_settings_get_int("/xfmedia/playlist/add_sort_type"));
    gtk_widget_show(combo);
    gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(chooser), combo);
    
    gtk_widget_show(chooser);
    xfmedia_mainwin_disable_autohide(mwin);
}
#undef MY_FOLDER_SELECT_RESPONSE

static void
mainwin_add_file_chooser_response_cb(GtkWidget *chooser, gint response,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GSList *filenames, *l;
    gboolean defer = TRUE;
    gint nadded = 0;
    
    if(response == GTK_RESPONSE_ACCEPT) {
        gtk_widget_hide(chooser);
        while(gtk_events_pending())
            gtk_main_iteration();
    }
    
    if(response == GTK_RESPONSE_ACCEPT || response == GTK_RESPONSE_APPLY) {
        filenames = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(chooser));
        if(filenames && g_slist_length(filenames) < 10)
            defer = FALSE;
        for(l = filenames; l; l = l->next) {
            if(xfmedia_playlists_is_playlist_file(l->data))
                xfmedia_playlists_load(mwin->plist, l->data);
            else {
                gchar *title_utf8 = xfmedia_filename_to_name(l->data);
                xfmedia_playlist_append_entry(mwin->plist, title_utf8, -1,
                        l->data, FALSE);
                g_free(title_utf8);
                nadded++;
            }
            g_free(l->data);
        }
        if(filenames)
            g_slist_free(filenames);
        if(mwin->chooser_last_dir)
            g_free(mwin->chooser_last_dir);
        mwin->chooser_last_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(chooser));
        //plist->priv->is_dirty = TRUE;
        
        //if(nadded > 0)
        //    xfmedia_playlist_shuffle_add_entries(mwin->plist, -1, nadded);
    }
    
    if(response != GTK_RESPONSE_APPLY) {
        gtk_widget_destroy(chooser);
        xfmedia_mainwin_enable_autohide(mwin);
    }
    if(response != GTK_RESPONSE_CANCEL)
        g_signal_emit_by_name(G_OBJECT(mwin->plist), "playlist-scrolled", 0);
        //xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}

void
xfmedia_mainwin_add_file_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    static GtkWidget *chooser = NULL;
    GtkWidget *btn, *toplevel;
    GtkFileFilter *filter;
    
    if(chooser) {
        gtk_window_present(GTK_WINDOW(chooser));
        return;
    }
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(mwin->window));
    chooser = gtk_file_chooser_dialog_new(_("Add Media File"), GTK_WINDOW(toplevel),
            GTK_FILE_CHOOSER_ACTION_OPEN, NULL);
    btn = xfmedia_custom_button_new(_("Add _Selected"), GTK_STOCK_ADD);
    gtk_widget_show(btn);
    gtk_dialog_add_action_widget(GTK_DIALOG(chooser), btn, GTK_RESPONSE_APPLY);
    gtk_dialog_add_button(GTK_DIALOG(chooser), GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL);
    btn = xfmedia_custom_button_new(_("A_dd and Close"), GTK_STOCK_ADD);
    GTK_WIDGET_SET_FLAGS(btn, GTK_CAN_DEFAULT);
    gtk_widget_show(btn);
    gtk_dialog_add_action_widget(GTK_DIALOG(chooser), btn, GTK_RESPONSE_ACCEPT);
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(chooser), TRUE);
    if(mwin->chooser_last_dir)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(chooser),
                mwin->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    g_signal_connect(G_OBJECT(chooser), "response",
            G_CALLBACK(mainwin_add_file_chooser_response_cb), mwin);
    g_signal_connect_swapped(G_OBJECT(chooser), "destroy",
            G_CALLBACK(xfmedia_widget_zero_pointer), &chooser);
    
    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("All Files"));
    gtk_file_filter_add_pattern(filter, "*");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filter);
    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("Music Files"));
    gtk_file_filter_add_pattern(filter, "*.aif*");
    gtk_file_filter_add_pattern(filter, "*.au");
    gtk_file_filter_add_pattern(filter, "*.flac");
    gtk_file_filter_add_pattern(filter, "*.mka");
    gtk_file_filter_add_pattern(filter, "*.mpa");
    gtk_file_filter_add_pattern(filter, "*.mp2");
    gtk_file_filter_add_pattern(filter, "*.mp3");
    gtk_file_filter_add_pattern(filter, "*.ogg");
    gtk_file_filter_add_pattern(filter, "*.ra");
    gtk_file_filter_add_pattern(filter, "*.wav");
    gtk_file_filter_add_pattern(filter, "*.wma");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filter);
    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("Playlists"));
    gtk_file_filter_add_pattern(filter, "*.m3u");
    gtk_file_filter_add_pattern(filter, "*.pls");
    gtk_file_filter_add_pattern(filter, "*.asx");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filter);
    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("Video Files"));
    gtk_file_filter_add_pattern(filter, "*.avi");
    gtk_file_filter_add_pattern(filter, "*.mkv");
    gtk_file_filter_add_pattern(filter, "*.mpeg");
    gtk_file_filter_add_pattern(filter, "*.mpg");
    gtk_file_filter_add_pattern(filter, "*.mpv");
    gtk_file_filter_add_pattern(filter, "*.ogg");
    gtk_file_filter_add_pattern(filter, "*.ogm");
    gtk_file_filter_add_pattern(filter, "*.rm");
    gtk_file_filter_add_pattern(filter, "*.vob");
    gtk_file_filter_add_pattern(filter, "*.wmv");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filter);
    
    gtk_widget_show(chooser);
    xfmedia_mainwin_disable_autohide(mwin);
}

static void
add_menu_position(GtkMenu *menu, gint *x, gint *y, gboolean *push_in, 
        gpointer user_data)
{
    GtkRequisition req;
    gint xpointer, ypointer;
    
    gtk_widget_size_request(GTK_WIDGET(menu), &req);
    
    /* get pointer position */
    gdk_display_get_pointer(
            gdk_screen_get_display(gtk_widget_get_screen(GTK_WIDGET(menu))),
            NULL, &xpointer, &ypointer, NULL);
    
    /* set x to the current position */
    *x = xpointer;
    /* ...and y to the current position minus the menu height */
    *y = ypointer - req.height;
    
    /* FIXME: huh? */
    *push_in = FALSE;
}

static void
xfmedia_mainwin_add_menu_deactivate_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
	guint sig_id;
    GtkToggleButton *btn;
	
	sig_id = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(w),
			"xfmedia-addmenu-deactivate-sigid"));
	g_signal_handler_disconnect(G_OBJECT(w), sig_id);
    
    btn = g_object_get_data(G_OBJECT(w), "xfmedia-addmenu-button");
    gtk_toggle_button_set_active(btn, FALSE);
    
    xfmedia_mainwin_enable_autohide(mwin);
}

void
xfmedia_mainwin_playlist_add_button_cb(GtkWidget *w, GtkToggleButton *button,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    guint sig_id;
    
    DBG("got add-clicked signal");
    
    if(!gtk_toggle_button_get_active(button))
        return;
    
    sig_id = g_signal_connect(G_OBJECT(mwin->add_menu), "deactivate",
    		G_CALLBACK(xfmedia_mainwin_add_menu_deactivate_cb), mwin);
    g_object_set_data(G_OBJECT(mwin->add_menu),
    		"xfmedia-addmenu-deactivate-sigid", GUINT_TO_POINTER(sig_id));
    g_object_set_data(G_OBJECT(mwin->add_menu), "xfmedia-addmenu-button", button);
    
    gtk_menu_popup(GTK_MENU(mwin->add_menu), NULL, NULL,
            (GtkMenuPositionFunc)add_menu_position, button, 0,
            gtk_get_current_event_time());
    
    xfmedia_mainwin_disable_autohide(mwin);
}

gboolean
xfmedia_position_cb(gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint pos_stream, pos_time, length_time, idx;
    
    if(xfmedia_xine_get_status(mwin->xfx) != XINE_STATUS_PLAY
            || xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE
            || mwin->slider_sticky)
    {
        return TRUE;
    }
    
    if(!xfmedia_xine_get_pos_length(mwin->xfx, &pos_stream, &pos_time, &length_time))
        return TRUE;
    pos_time /= 1000;
    length_time /= 1000;
    
    /* FIXME: i *think* it's ok not to lock gdk when calling into xine like
     * that, but i should check. */
    gdk_threads_enter();
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_remaining") && length_time > 0)
        xfmedia_mainwin_set_time_label(mwin, length_time-pos_time, TRUE, NULL);
    else
        xfmedia_mainwin_set_time_label(mwin, pos_time, FALSE, NULL);
    
    /* sometimes xine fails to get the song length on the first try, so we'll
     * check it here if we have a zero length */
    idx = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
    if(mwin->cur_playing && length_time != 0
            && xfmedia_settings_get_int("/xfmedia/playlist/metadata_load_impetus") <= 1)
    {
        gint cur_length = -1;
        if(idx != -1 && xfmedia_playlist_get(mwin->plist, idx, NULL, &cur_length, NULL)
                && (cur_length != length_time || mwin->cur_playing_length <= 0))
        {
            xfmedia_playlist_modify_entry(mwin->plist, idx, NULL, length_time, NULL);
            if(length_time > 0) {
                gtk_range_set_range(GTK_RANGE(mwin->position_slider), 0, length_time);
                gtk_range_set_range(GTK_RANGE(mwin->fs_position_slider), 0, length_time);
            }
            mwin->cur_playing_length = length_time;
            
        }
    } else if(idx < 0 && length_time != 0
            && mwin->cur_playing_length != length_time)
    {
        if(length_time > 0) {
            gtk_range_set_range(GTK_RANGE(mwin->position_slider), 0, length_time);
            gtk_range_set_range(GTK_RANGE(mwin->fs_position_slider), 0, length_time);
        }
        mwin->cur_playing_length = length_time;
    }
    
    if(length_time > 0) {
        xfmedia_mainwin_disconnect_slider_callback(mwin);
        gtk_range_set_value(GTK_RANGE(mwin->position_slider), (gdouble)pos_time);
        gtk_range_set_value(GTK_RANGE(mwin->fs_position_slider), (gdouble)pos_time);
        xfmedia_mainwin_connect_slider_callback(mwin);
    }
    
    gdk_threads_leave();
    
    return TRUE;
}

void
xfmedia_position_slider_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint status, speed, new_pos;
    
    status = xfmedia_xine_get_status(mwin->xfx);
    speed = xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED);
    
    new_pos = (gint)gtk_range_get_value(GTK_RANGE(w)) * 1000;
    DBG("seeking to %d\n", new_pos);
    if(xfmedia_xine_play(mwin->xfx, 0, new_pos)) {
        if(status != XINE_STATUS_PLAY || speed == XINE_SPEED_PAUSE)
            xfmedia_mainwin_set_pause_buttons(mwin);
    }
}

static gboolean
xfmedia_pos_slider_mmotion_cb(GtkWidget *w, GdkEventMotion *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint cur_value;
    
    if(w == mwin->position_slider)
        cur_value = gtk_range_get_value(GTK_RANGE(mwin->position_slider));
    else if(w == mwin->fs_position_slider)
        cur_value = gtk_range_get_value(GTK_RANGE(mwin->fs_position_slider));
    else {
        DBG("er, this shouldn't happen.");
        return FALSE;
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_remaining")) {
        gint pos_stream, pos_time, length_time = 0;
        if(!xfmedia_xine_get_pos_length(mwin->xfx, &pos_stream, &pos_time, &length_time) || length_time == 0)
            return FALSE;
        length_time /= 1000;
        xfmedia_mainwin_set_time_label(mwin, length_time-cur_value, TRUE, "style='italic'");
    } else
        xfmedia_mainwin_set_time_label(mwin, cur_value, FALSE, "style='italic'");
    
    return FALSE;
}

gboolean
xfmedia_pos_slider_btn_press_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    if(evt->button == 1 || evt->button == 2) {
        mwin->slider_sticky = TRUE;
        
        g_signal_connect(G_OBJECT(mwin->position_slider), "motion-notify-event",
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
        g_signal_connect(G_OBJECT(mwin->fs_position_slider), "motion-notify-event",
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
    }
    
    return FALSE;
}

gboolean
xfmedia_pos_slider_btn_rls_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    if(evt->button == 1 || evt->button == 2) {
        mwin->slider_sticky = FALSE;
        
        g_signal_handlers_disconnect_by_func(G_OBJECT(mwin->position_slider),
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
        g_signal_handlers_disconnect_by_func(G_OBJECT(mwin->fs_position_slider),
                G_CALLBACK(xfmedia_pos_slider_mmotion_cb), mwin);
    }
    
    return FALSE;
}

void
xfmedia_playpause_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_toggle_playpause(mwin);
}

void
xfmedia_stop_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_stop(mwin);
}

void
xfmedia_prev_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_prev(mwin);
}

void
xfmedia_next_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_mainwin_next(mwin);
}

gboolean
xfmedia_pos_lbl_click_cb(GtkWidget *w, GdkEventButton *evt, gpointer user_data)
{
    if(evt->button == 1) {
        xfmedia_settings_set_bool("/xfmedia/general/show_remaining",
                !xfmedia_settings_get_bool("/xfmedia/general/show_remaining"));
        /* FIXME: hackish */
        gdk_threads_leave();
        xfmedia_position_cb(user_data);
        gdk_threads_enter();
    }
    
    return xfmedia_mainwin_menu_cb(w, evt, user_data);
}

void
xfmedia_shuffle_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_check_menu_item_get_active(mi);
    xfmedia_playlist_set_shuffle_state(mwin->plist, set);
    xfmedia_settings_set_bool("/xfmedia/playlist/shuffle", set);
    
    if(!set && !g_queue_is_empty(mwin->shuffle_history)) {
        gint i, tot = g_queue_get_length(mwin->shuffle_history);
        for(i = tot-1; i >= 0; i--)
            xfmedia_playlist_entry_ref_destroy((XfmediaPlaylistEntryRef *)g_queue_pop_nth(mwin->shuffle_history, i));
    }
}

void
xfmedia_repeat_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set;
    
    set = gtk_check_menu_item_get_active(mi);
    xfmedia_playlist_set_repeat_state(mwin->plist, set);
    xfmedia_settings_set_bool("/xfmedia/playlist/repeat", set);
}

void
xfmedia_repeat_one_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    gboolean set;
    
    set = gtk_check_menu_item_get_active(mi);
    xfmedia_settings_set_bool("/xfmedia/playlist/repeat_one", set);
}

void
xfmedia_show_vis_menu_toggle_cb(GtkCheckMenuItem *mi, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean set, has_video;

    set = gtk_check_menu_item_get_active(mi);
    xfmedia_settings_set_bool("/xfmedia/general/show_audio_vis", set);
    
    has_video = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_HAS_VIDEO);
    if(!set) {
        if(!has_video)
            gtk_widget_hide(mwin->video_window);
        xfmedia_xine_set_vis(mwin->xfx, NULL);
    } else {
        if(!has_video && xfmedia_xine_get_status(mwin->xfx) == XINE_STATUS_PLAY) {
            gint width, height;
            GdkGeometry geom;
            
            xfmedia_xine_set_vis(mwin->xfx,
                    xfmedia_settings_get_string("/xfmedia/general/selected_vis"));
            
            width = xfmedia_xine_get_stream_info(mwin->xfx,
                    XINE_STREAM_INFO_VIDEO_WIDTH);
            height = xfmedia_xine_get_stream_info(mwin->xfx,
                    XINE_STREAM_INFO_VIDEO_HEIGHT);
            if(width < 1 || height < 1) {
                width = 480;
                height = 360;
            }
            geom.base_width = width;
            geom.base_height = height;
            geom.min_width = geom.min_height = 1;
            geom.min_aspect = geom.max_aspect = (gdouble)width / height;
            DBG("window size: %dx%d, aspect: %f", width, height, geom.max_aspect);
            gtk_window_set_geometry_hints(GTK_WINDOW(mwin->video_window),
                    GTK_WIDGET(mwin->xfx), &geom,
                    GDK_HINT_MIN_SIZE|GDK_HINT_ASPECT|GDK_HINT_BASE_SIZE);
            gtk_widget_show(mwin->video_window);
            gtk_window_resize(GTK_WINDOW(mwin->video_window),
                    width, height);
        }
    }
}

gboolean
xfmedia_vwin_delete_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    DBG("entering");
    
    if(xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_HAS_VIDEO)) {
        xfmedia_xine_stop(mwin->xfx);
        DBG("checking plist (%p)", mwin->plist);
        xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
    } else {
        xfmedia_xine_set_vis(mwin->xfx, NULL);
        gtk_widget_hide(w);
        gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(mwin->svis_mi), FALSE);
        xfmedia_settings_set_bool("/xfmedia/general/show_audio_vis", FALSE);
    }
    
    /* never let the window get destroyed here */
    return TRUE;
}

gboolean
xfmedia_autohidewin_enter_notify(GtkWidget *w, GdkEventCrossing *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint x, y;
    
    if(GTK_WIDGET_VISIBLE(mwin->autohide_window)) {
        x = xfmedia_settings_get_int("/xfmedia/general/pos_x");
        y = xfmedia_settings_get_int("/xfmedia/general/pos_y");
        
        gtk_window_move(GTK_WINDOW(mwin->window), x, y);
        if(xfmedia_settings_get_bool("/xfmedia/general/window_sticky"))
            gtk_window_stick(GTK_WINDOW(mwin->window));
        else
            gtk_window_unstick(GTK_WINDOW(mwin->window));
        if(xfmedia_settings_get_bool("/xfmedia/general/window_above"))
            gtk_window_set_keep_above(GTK_WINDOW(mwin->window), TRUE);
        gtk_widget_show(mwin->window);
        gtk_widget_hide(mwin->autohide_window);
    }
    
    return FALSE;
}

gboolean
xfmedia_mainwin_enterleave_notify_cb(GtkWidget *w, GdkEventCrossing *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    if(evt->type == GDK_ENTER_NOTIFY) {
        if(mwin->autohide_timer) {
            g_source_remove(mwin->autohide_timer);
            mwin->autohide_timer = 0;
        }
    } else if(evt->type == GDK_LEAVE_NOTIFY) {
        if(evt->detail == GDK_NOTIFY_NONLINEAR_VIRTUAL
                && !mwin->temp_disable_autohide
                && xfmedia_settings_get_bool("/xfmedia/general/dock_autohide")
                && !mwin->autohide_timer)
        {
            mwin->autohide_timer = g_timeout_add(AUTOHIDE_TIMEOUT,
                    (GSourceFunc)xfmedia_mainwin_check_do_autohide, mwin);
        }
    }
    
    return FALSE;
}

GdkFilterReturn
xfmedia_mainwin_decs_filter(GdkXEvent *gxevt, GdkEvent *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    XCrossingEvent *xevt = (XCrossingEvent *)gxevt;

    if(xevt->type == EnterNotify) {
        if(mwin->autohide_timer) {
            g_source_remove(mwin->autohide_timer);
            mwin->autohide_timer = 0;
        }
    } else if(xevt->type == LeaveNotify) {
        if(xevt->detail == NotifyNonlinearVirtual
                && !(xevt->state & (ButtonPressMask|ButtonMotionMask
                                    |Button1MotionMask|Button2MotionMask
                                    |Button3MotionMask|Button4MotionMask
                                    |Button5MotionMask))
                && !mwin->temp_disable_autohide
                && xfmedia_settings_get_bool("/xfmedia/general/dock_autohide")
                && !mwin->autohide_timer)
        {
            mwin->autohide_timer = g_timeout_add(AUTOHIDE_TIMEOUT,
                    (GSourceFunc)xfmedia_mainwin_check_do_autohide, mwin);
        }
    } else if(xevt->type == MotionNotify) {
        if(mwin->autohide_timer) {
            g_source_remove(mwin->autohide_timer);
            mwin->autohide_timer = 0;
        }
    }
    
    return GDK_FILTER_CONTINUE;
}

gboolean
xfmedia_mainwin_configure_cb(GtkWidget *w, GdkEventConfigure *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint x = -1, y = -1;
    
    if(GTK_WIDGET_VISIBLE(w))
        gtk_window_get_position(GTK_WINDOW(w), &x, &y);
    else
        return FALSE;
    
    xfmedia_settings_set_int("/xfmedia/general/pos_x", x);
    xfmedia_settings_set_int("/xfmedia/general/pos_y", y);

    xfmedia_settings_set_int("/xfmedia/general/width", evt->width);
    if(gtk_expander_get_expanded(GTK_EXPANDER(mwin->playlist_pane)))
        xfmedia_settings_set_int("/xfmedia/general/height_large", evt->height);
    else
        xfmedia_settings_set_int("/xfmedia/general/height_small", evt->height);
    
    return FALSE;
}

gboolean
xfmedia_mainwin_winstate_cb(GtkWidget *w, GdkEventWindowState *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    mwin->is_iconified = (evt->new_window_state & GDK_WINDOW_STATE_ICONIFIED);
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon")
            && xfmedia_settings_get_bool("/xfmedia/general/minimize_to_tray")
            && mwin->is_iconified)
    {
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(w), TRUE);
    } else
        gtk_window_set_skip_taskbar_hint(GTK_WINDOW(w), FALSE);
    
    return FALSE;
}

void
xfmedia_mainwin_expander_activate_cb(GtkExpander *widget, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint h_l, h_s, w;
    
    if(gtk_expander_get_expanded(widget)) {
        gtk_window_get_size(GTK_WINDOW(mwin->window), &w, &h_s);
        h_l = xfmedia_settings_get_int("/xfmedia/general/height_large");
        if(h_l == -1) {
            h_l = h_s * 3;
            xfmedia_settings_set_int("/xfmedia/general/height_large", h_l);
        }
        xfmedia_settings_set_int("/xfmedia/general/height_small", h_s);
        xfmedia_settings_set_int("/xfmedia/general/width", w);
        
        gtk_widget_set_size_request(mwin->window, -1, -1);
        gtk_window_resize(GTK_WINDOW(mwin->window), w, h_l);
        xfmedia_settings_set_bool("/xfmedia/playlist/playlist_expanded", TRUE);
        g_idle_add(xfmedia_mainwin_playlist_load_info_idled, mwin);
    } else {
        gtk_window_get_size(GTK_WINDOW(mwin->window), &w, &h_l);
        h_s = xfmedia_settings_get_int("/xfmedia/general/height_small");
        gtk_widget_set_size_request(mwin->playlist_pane, -1, -1);
        gtk_widget_set_size_request(mwin->window, -1, h_s);
        gtk_window_resize(GTK_WINDOW(mwin->window), w, h_s);
        gtk_widget_set_size_request(mwin->window, -1, 1);
        xfmedia_settings_set_int("/xfmedia/general/height_large", h_l);
        xfmedia_settings_set_int("/xfmedia/general/width", w);
        xfmedia_settings_set_bool("/xfmedia/playlist/playlist_expanded", FALSE);
    }
}

static void
about_dlg_zero_pointer(GtkWidget **dlg)
{
    *dlg = NULL;
}

static struct
{
    gchar *name;
    gchar *email;
    gchar *langcode;
} translator_list[] = {
    { "Bernhard Walle", "bernhard.walle@gmx.de", "de" },
    { "Piarres Beobide", "pi@beobide.net", "eu" },
    { "Jaime Buffery", "nestu@lunar-linux.org", "es" },
    { "Jari Rahkonen", "jari.rahkonen@pp1.inet.fi", "fi" },
    { "St\303\251phane Roy", "sroy@j2n.net", "fr" },
    { "Collet Etienne", "xanax@no-log.org", "fr" },
    { "lerouge", "lerouge@gmail.com", "fr" },
    { "Ankit Patel", "ankit644@yahoo.com", "gu" },
    { "Atilla Szerv\303\241c", "sas@321.hu", "hu" },
    { "Rimas Kudelis", "rg@akl.lt", "lt" },
    { "Robert Kurowski", "koorek@o2.pl", "pl" },
    { "Tomas Schertel", "tschertel@gmail.com", "pt_BR" },
    { "Juraz Brosz", "juro@jurajbrosz.info", "sk" },
    { "Phan Vinh Thinh", "teppi@vnlinux.org", "vi" },
    { "Sheng Feng Zhou", "zhoushengfeng@highpoint-tech.com.cn", "zh_CN" },
    { NULL, NULL, NULL }
};

void
xfmedia_about_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    XfceAboutInfo *ainfo;
    static GtkWidget *dlg = NULL;
    GtkWidget *toplevel = NULL;
    gint i;
    gchar *TRANSLATOR = _("Translator"), translator_str[1024];
    
    if(dlg) {
        gtk_window_present(GTK_WINDOW(dlg));
        return;
    }
    
    if(w)
        toplevel = gtk_widget_get_toplevel(w);
    if(!toplevel)
        toplevel = mwin->window;
    
    ainfo = xfce_about_info_new("Xfmedia", VERSION " (" REVISION ")",
            _("Xfmedia is a lightweight media player,\nbased on the xine engine"),
            XFCE_COPYRIGHT_TEXT("2004-2005", "Brian Tarricone"), XFCE_LICENSE_GPL);
    xfce_about_info_set_homepage(ainfo, "http://spuriousinterrupt.org/projects/xfmedia/");
    
    xfce_about_info_add_credit(ainfo, "Brian Tarricone", "bjt23@cornell.edu",
            _("Original Author/Maintainer"));
    
    for(i = 0; translator_list[i].name; i++) {
        g_snprintf(translator_str, 1024, "%s (%s)", TRANSLATOR,
                translator_list[i].langcode);
        xfce_about_info_add_credit(ainfo, translator_list[i].name,
                translator_list[i].email, translator_str);
    }
    
    dlg = xfce_about_dialog_new(GTK_WINDOW(toplevel), ainfo, NULL);
    g_signal_connect(G_OBJECT(dlg), "response",
            G_CALLBACK(gtk_widget_destroy), NULL);
    g_signal_connect_swapped(G_OBJECT(dlg), "destroy",
            G_CALLBACK(about_dlg_zero_pointer), &dlg);
    gtk_widget_show_all(dlg);
}

gboolean
xfmedia_vwin_button_press_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint x, y;
    
    if(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS) {
        gboolean is_fs = xfmedia_xine_is_fullscreen(mwin->xfx);
        
        xfmedia_xine_set_fullscreen(mwin->xfx, !is_fs);
        if(!is_fs) {
            gtk_window_get_position(GTK_WINDOW(mwin->video_window), &x, &y);
            xfmedia_settings_set_int("/xfmedia/general/vwin_pos_x", x);
            xfmedia_settings_set_int("/xfmedia/general/vwin_pos_y", y);
            gtk_widget_hide(mwin->video_window);
        } else {
            x = xfmedia_settings_get_int("/xfmedia/general/vwin_pos_x");
            y = xfmedia_settings_get_int("/xfmedia/general/vwin_pos_y");
            gtk_window_move(GTK_WINDOW(mwin->video_window), x, y);
            gtk_widget_show(mwin->video_window);
        }
        
        return TRUE;
    }
    
    return FALSE;
}

gboolean
xfmedia_vwin_motion_notify_cb(GtkWidget *w, GdkEventMotion *evt,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GdkScreen *gscreen;
    gint sheight, swidth;
    
    if(!xfmedia_xine_is_fullscreen(mwin->xfx))
        return FALSE;
    
    gscreen = gtk_widget_get_screen(mwin->window);
    swidth = gdk_screen_get_width(gscreen);
    sheight = gdk_screen_get_height(gscreen);
    
    if(!GTK_WIDGET_VISIBLE(mwin->fs_controls) && evt->y == sheight - 1) {
        Window fs_window;
        
        fs_window = GDK_WINDOW_XWINDOW(xfmedia_xine_get_fullscreen_window(mwin->xfx));
        gdk_property_change(mwin->fs_controls->window,
                gdk_atom_intern("WM_TRANSIENT_FOR", FALSE),
                gdk_atom_intern("WINDOW", FALSE),
                32, GDK_PROP_MODE_REPLACE,
                (const guchar *)&fs_window, sizeof(Window));
        
        gtk_widget_set_size_request(mwin->fs_controls, swidth, -1);
        gtk_widget_realize(mwin->fs_controls);
        gtk_window_move(GTK_WINDOW(mwin->fs_controls), 0,
                sheight - mwin->fs_controls->allocation.height);
        gtk_widget_show(mwin->fs_controls);
    } else if(GTK_WIDGET_VISIBLE(mwin->fs_controls)
            && evt->y < sheight - mwin->fs_controls->allocation.height)
    {
        gtk_widget_hide(mwin->fs_controls);
    }
    
    return FALSE;
}

gboolean
xfmedia_vwin_mapunmap_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    int visible;
    
    if(evt->type == GDK_MAP)
        visible = 1;
    else if(evt->type == GDK_UNMAP)
        visible = 0;
    else
        return FALSE;
    
    xfmedia_xine_port_send_gui_data(mwin->xfx, XINE_GUI_SEND_VIDEOWIN_VISIBLE,
            &visible);
    
    return FALSE;
}

void
xfmedia_xfmedia_xine_stream_ended_cb(XfmediaXine *xfx, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gboolean ret = FALSE, use_playlist_q;
    
    DBG("got stream-ended signal");
    
    use_playlist_q = xfmedia_playlist_queue_n_entries(mwin->playlist_q);
    
    if(mwin->cur_playing
            && !xfmedia_settings_get_bool("/xfmedia/playlist/repeat_one")
            && xfmedia_settings_get_bool("/xfmedia/playlist/shuffle")
            && !use_playlist_q)
    {
        DBG("putting item on shuffle history");
        g_queue_push_head(mwin->shuffle_history,
                xfmedia_playlist_entry_ref_copy(mwin->cur_playing));
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"))
        xfmedia_tray_icon_set_tooltip(mwin->tray_icon, _("Stopped"));
    xfmedia_mainwin_set_play_buttons(mwin);
    
    xfmedia_mainwin_disconnect_slider_callback(mwin);
    gtk_range_set_value(GTK_RANGE(mwin->position_slider), 0.0);
    gtk_range_set_value(GTK_RANGE(mwin->fs_position_slider), 0.0);
    xfmedia_mainwin_connect_slider_callback(mwin);
    gtk_label_set_markup(GTK_LABEL(mwin->time_label),
            "<span font_family=\"monospace\" size=\"large\" weight=\"bold\"> 00:00</span>");
    gtk_label_set_markup(GTK_LABEL(mwin->fs_time_label),
            "<span font_family=\"monospace\"> 00:00</span>");
    
    xfmedia_plugins_forward_signal_VOID("stream-ended");
    
    if(xfmedia_settings_get_bool("/xfmedia/playlist/auto_advance")) {
        if(xfmedia_settings_get_bool("/xfmedia/playlist/repeat_one")) {
            if(mwin->cur_playing) {
                gint idx = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
                ret = xfmedia_mainwin_play_file_at_index(mwin, idx);
            } else
                ret = xfmedia_xine_play(mwin->xfx, 0, 0);
        }
        
        if(use_playlist_q && !ret) {
            XfmediaPlaylistEntryRef *ref;
            gint idx;
            do {
                ref = xfmedia_playlist_queue_pop_entry(mwin->playlist_q);
                if(ref && (idx = xfmedia_playlist_entry_ref_get_index(ref)) >= 0) {
                    ret = xfmedia_mainwin_play_file_at_index(mwin, idx);
                    xfmedia_playlist_entry_ref_destroy(ref);
                } else {
                    if(ref)
                        xfmedia_playlist_entry_ref_destroy(ref);
                    break;
                }
            } while(!ret);
        }
        
        if(!ret)
            ret = xfmedia_mainwin_play_file_at_index_persistent(mwin, -1);
    }
    
    if(!ret && GTK_WIDGET_VISIBLE(mwin->video_window))
        gtk_widget_hide(mwin->video_window);
}

static gboolean
xfmedia_xine_hide_ui_msg_label(gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_SONG_LABEL);
    return FALSE;
}

void
xfmedia_xfmedia_xine_ui_message_cb(XfmediaXine *xfx, const gchar *message,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    DBG("got UI message: '%s'", message);
    
    xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_UI_MESSAGE);
    gtk_label_set_text(GTK_LABEL(mwin->ui_message_label), message);
    
    g_timeout_add(3000, (GSourceFunc)xfmedia_xine_hide_ui_msg_label, mwin);
}

void
xfmedia_xfmedia_xine_progress_message_cb(XfmediaXine *xfx, const gchar *message, gint percent,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    DBG("got progress message: '%s' at %d%%", message, percent);
    
    if(percent > 99)
        xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_SONG_LABEL);
    else {
        xfmedia_mainwin_set_infobar_type(mwin, XFMEDIA_INFOBAR_PROGRESS_MESSAGE);
        
        gtk_label_set_text(GTK_LABEL(mwin->progress_label), message);
        gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(mwin->progress_bar),
                percent / 100.0);
    }
}

void
xfmedia_xfmedia_xine_format_changed_cb(XfmediaXine *xfx, gint width, gint height,
        gint aspect_code, gboolean is_pan_scan, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GdkGeometry geom;
    
    DBG("got format changed signal: %dx%d, aspect_code: %d, is_pan_scan: %s",
            width, height, aspect_code, is_pan_scan?"yes":"no");
    
    /*geom.base_width = width;
    geom.base_height = height;*/
    geom.min_width = geom.min_height = 1;
    geom.min_aspect = geom.max_aspect = (gdouble)width / height;
    DBG("window size: %dx%d, aspect: %f", width, height, geom.max_aspect);
    gtk_window_set_geometry_hints(GTK_WINDOW(mwin->video_window),
            GTK_WIDGET(mwin->xfx), &geom,
            GDK_HINT_MIN_SIZE|GDK_HINT_ASPECT /*|GDK_HINT_BASE_SIZE*/);
    gtk_window_resize(GTK_WINDOW(mwin->video_window), width, height);
}

void
xfmedia_xfmedia_xine_set_title_cb(XfmediaXine *xfx, const gchar *title,
        gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    DBG("got set title signal: '%s'", title);
    
    xfmedia_mainwin_set_song_label(mwin, title);
    
    /* this is something of a hack.  for DVDs, often when the title changes,
     * we're changing menus, which may indicate that the seekable status
     * has changed as well. */
    if(xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_SEEKABLE)) {
        gtk_widget_set_sensitive(mwin->position_slider, TRUE);
        gtk_widget_set_sensitive(mwin->fs_position_slider, TRUE);
    } else {
        gtk_widget_set_sensitive(mwin->position_slider, FALSE);
        gtk_widget_set_sensitive(mwin->fs_position_slider, FALSE);
    }
}

void
xfmedia_xfmedia_xine_mrl_reference_cb(XfmediaXine *xfx, const gchar *mrl,
        gint alternative, const gchar *title, gint length, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    gint idx = -1;
    
    DBG("got mrl reference signal: (%d) %s", alternative, mrl);
    
    if(alternative != 0 || !mrl || !*mrl)
        return;
    
    if(mwin->cur_playing)
        idx = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
    
    if(title) {
        xfmedia_playlist_modify_entry(mwin->plist, idx, title, length, NULL);
        xfmedia_playlist_set_metadata_loaded(mwin->plist, idx, TRUE);
    }
    
    xfmedia_mainwin_play_uri(mwin, mrl, idx);
}
        

void
xfmedia_mainwin_quit_mi_cb(GtkMenuItem *mi, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
}

gboolean
xfmedia_mainwin_map_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GdkWindow *decs;
    
    if(xfmedia_settings_get_bool("/xfmedia/general/dock_autohide")) {
        decs = xfmedia_get_window_decorations(w->window);
        
        gdk_error_trap_push();
        
        if(decs) {
            mwin->window_decs = decs;
            gdk_window_set_events(decs, gdk_window_get_events(decs)
                    | GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
                    | GDK_POINTER_MOTION_MASK);
            gdk_window_add_filter(decs, xfmedia_mainwin_decs_filter, mwin);
        }
        
        gdk_error_trap_pop();
    }
    
    return FALSE;
}

gboolean
xfmedia_mainwin_unmap_cb(GtkWidget *w, GdkEvent *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    if(mwin->window_decs) {
        gdk_error_trap_push();
        gdk_window_remove_filter(mwin->window_decs,
                xfmedia_mainwin_decs_filter, mwin);
        g_object_unref(G_OBJECT(mwin->window_decs));
        mwin->window_decs = NULL;
        gdk_error_trap_pop();
    }
    
    return FALSE;
}

gboolean
xfmedia_mainwin_keyrelease_cb(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    const gchar *keybind_id;
    
    if(xfmedia_playlist_jump_to_file_box_has_focus(mwin->plist))
        return FALSE;
    else if(xfmedia_playlist_has_focus(mwin->plist)) {
        if(evt->keyval == GDK_Up || evt->keyval == GDK_KP_Up
                || evt->keyval == GDK_Down || evt->keyval == GDK_KP_Down
                || evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter)
        {
            /* ignore these, as they should be for playlist navigation */
            DBG("ignored");
            return FALSE;
        }
    }
    
    keybind_id = xfmedia_keybindings_lookup(evt->keyval, evt->state);
    if(keybind_id)
        xfmedia_keybindings_activate(mwin, keybind_id);
    
    return FALSE;
}

static void
mainwin_popupmenu_deactivate_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    guint sigid = GPOINTER_TO_UINT(g_object_get_data(G_OBJECT(w),
            "xfmedia-deactivate-sigid"));
    
    g_signal_handler_disconnect(G_OBJECT(w), sigid);
    xfmedia_mainwin_enable_autohide(mwin);
}

gboolean
xfmedia_mainwin_menu_cb(GtkWidget *w, GdkEventButton *evt, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    if(evt->button == 3 && mwin->popup_menu) {
        guint sigid = g_signal_connect(G_OBJECT(mwin->popup_menu), "deactivate",
                G_CALLBACK(mainwin_popupmenu_deactivate_cb), mwin);
        g_object_set_data(G_OBJECT(mwin->popup_menu),
                "xfmedia-deactivate-sigid", GUINT_TO_POINTER(sigid));
        xfmedia_mainwin_disable_autohide(mwin);
        gtk_menu_popup(GTK_MENU(mwin->popup_menu), NULL, NULL, NULL, NULL,
                evt->button, evt->time);
        
        return TRUE;
    }
    
    return FALSE;
}

void
xfmedia_mainwin_mediamarks_changed_cb(gpointer data)
{
    XfmediaMainwin *mwin = data;
    GtkWidget *old_menu, *menu;
    
    old_menu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(mwin->mmarks_mi));
    gtk_widget_destroy(old_menu);
    
    menu = xfmedia_mediamarks_create_menu(mwin);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(mwin->mmarks_mi), menu);
}

void
xfmedia_mainwin_sort_title_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_playlist_sort(mwin->plist, XFMEDIA_PLAYLIST_SORT_TITLE);
}

void
xfmedia_mainwin_sort_filename_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_playlist_sort(mwin->plist, XFMEDIA_PLAYLIST_SORT_FILENAME);
}

void
xfmedia_mainwin_randomise_playlist_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_playlist_sort(mwin->plist, XFMEDIA_PLAYLIST_SORT_RANDOM);
}

static void
volume_window_destroy_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    mwin->volume_window = NULL;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mwin->volume_btn), FALSE);
    xfmedia_mainwin_check_do_autohide(mwin);
}

static gboolean
volume_window_destroy(GtkWidget **w)
{
    if(gdk_pointer_is_grabbed()) {
        if(w && *w)
            gtk_grab_remove(*w);
        gdk_pointer_ungrab(GDK_CURRENT_TIME);
        gdk_keyboard_ungrab(GDK_CURRENT_TIME);
    }
    
    if(w && *w)
        gtk_widget_destroy(*w);
    
    return FALSE;
}

static void
volume_slider_changed_cb(GtkRange *range, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_AUDIO_VOLUME,
            (gint)gtk_range_get_value(range));
    
    g_idle_add((GSourceFunc)volume_window_destroy, &mwin->volume_window);
}

static gboolean
volume_window_button_press_cb(GtkWidget *widget, GdkEventButton *evt,
        gpointer user_data)
{
    GdkRectangle rect;
    gint x, y, w, h;
    
    gtk_grab_remove(widget);
    gdk_keyboard_ungrab(evt->time);
    gdk_pointer_ungrab(evt->time);
    
    gtk_window_get_position(GTK_WINDOW(widget), &x, &y);
    gtk_window_get_size(GTK_WINDOW(widget), &w, &h);
    
    rect.x = x;
    rect.y = y;
    rect.width = w;
    rect.height = h;
    
    g_signal_handlers_disconnect_by_func(G_OBJECT(widget),
                                         G_CALLBACK(volume_window_button_press_cb),
                                         user_data);
    
    if(!xfmedia_rectangle_contains_point(&rect, evt->x, evt->y))
        gtk_widget_destroy(widget);
    
    return TRUE;
}

void
xfmedia_mainwin_show_volume_window_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    GtkWidget *window, *slider;
    GtkRequisition req;
    gint monitor, btn_x, btn_y, btn_w, btn_h, scr_y, scr_h, win_x, win_y, par_x,
            par_y, par_w, par_h;
    GdkWindow *parent;
    GdkScreen *gscreen;
    GdkRectangle rect;
    
    if(!gtk_toggle_button_get_active(tb)) {
        if(mwin->volume_window)
            volume_window_destroy(&mwin->volume_window);
        return;
    }
    
    /* up here for the signal connection */
    slider = gtk_vscale_new_with_range(0.0, 100.0, 10.0);
    
    mwin->volume_window = window = gtk_window_new(GTK_WINDOW_POPUP);
    gtk_window_set_transient_for(GTK_WINDOW(window), GTK_WINDOW(mwin->window));
    gtk_window_set_decorated(GTK_WINDOW(window), FALSE);
    gtk_window_set_skip_pager_hint(GTK_WINDOW(window), TRUE);
    gtk_window_set_skip_taskbar_hint(GTK_WINDOW(window), TRUE);
    g_signal_connect(G_OBJECT(window), "destroy",
            G_CALLBACK(volume_window_destroy_cb), mwin);
    g_signal_connect(G_OBJECT(window), "button-press-event",
                     G_CALLBACK(volume_window_button_press_cb), slider);
    g_signal_connect_swapped(G_OBJECT(window), "key-press-event",
                     G_CALLBACK(volume_window_destroy), &mwin->volume_window);
    
    /* see above for gtk_vscale_new() */
    gtk_range_set_value(GTK_RANGE(slider),
            (gdouble)xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_AUDIO_VOLUME));
    gtk_scale_set_draw_value(GTK_SCALE(slider), FALSE);
    gtk_range_set_increments(GTK_RANGE(slider), 1, 10);
    gtk_range_set_update_policy(GTK_RANGE(slider), GTK_UPDATE_DISCONTINUOUS);
    gtk_range_set_inverted(GTK_RANGE(slider), TRUE);
    gtk_widget_show(slider);
    gtk_container_add(GTK_CONTAINER(window), slider);
    g_signal_connect(G_OBJECT(slider), "value-changed",
            G_CALLBACK(volume_slider_changed_cb), mwin);
    
    /* size and position the window */
    
    parent = gtk_widget_get_parent_window(GTK_WIDGET(tb));
    gdk_window_get_position(parent, &par_x, &par_y);
    gdk_drawable_get_size(GDK_DRAWABLE(parent), &par_w, &par_h);
    
    gscreen = gtk_widget_get_screen(GTK_WIDGET(tb));
    monitor = gdk_screen_get_monitor_at_window(gscreen, parent);
    gdk_screen_get_monitor_geometry(gscreen, monitor, &rect);
    scr_y = rect.y;
    scr_h = rect.height;
    gtk_widget_set_size_request(slider, -1, scr_h/12);  /* yes, this is somewhat silly */
    
    gtk_widget_size_request(window, &req);
    
    btn_x = GTK_WIDGET(tb)->allocation.x;
    btn_y = GTK_WIDGET(tb)->allocation.y;
    btn_w = GTK_WIDGET(tb)->allocation.width;
    btn_h = GTK_WIDGET(tb)->allocation.height;
    
    win_x = btn_x + par_x;  /* left of button */
    win_y = btn_y + par_y;  /* top of button */
    
    /* center the window on the button horizontally */
    win_x += (btn_w - req.width) / 2;
    
    /* move it so it appears above the button if it would otherwise be partially
     * off the screen, otherwise move it to the bottom of the button */
    if(win_y + req.height > scr_y + scr_h)
        win_y -= req.height;
    else
        win_y += btn_h;
    
    gtk_window_move(GTK_WINDOW(window), win_x, win_y);
    gtk_widget_show(window);
    
    gdk_keyboard_grab(window->window, TRUE, GDK_CURRENT_TIME);
    gdk_pointer_grab(window->window, TRUE, GDK_BUTTON_PRESS_MASK,
                     NULL, NULL, GDK_CURRENT_TIME);
    gtk_grab_add(window);
}
