// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_device.cc
// *
// * Purpose: Device implementation
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 18.6.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_device.h"
#include "xgsm_util.h"
#include "xgsm_main.h"
#include "xgsm_pref.h"

#include <gtk--/main.h>
#include <typeinfo>
#include <gsmlib/gsm_unix_serial.h>
#include <gsmlib/gsm_util.h>
#include <gsmlib/gsm_sms.h>
#include <libgnome/gnome-i18n.h>
#include <unistd.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/stat.h>
#include <iostream>

using namespace std;
using namespace Xgsm;
using namespace gsmlib;
using namespace SigC;

// entry implementations

Entry::Entry(gsmlib::SortedPhonebookIterator entryIterator) :
  _phonebookEntry(entryIterator->clone()), _phonebookIterator(entryIterator)
{
}

Entry::Entry(gsmlib::SortedSMSStoreIterator entryIterator) :
  _smsStoreEntry(entryIterator->clone()), _smsStoreIterator(entryIterator)
{
}

Entry::Entry(gsmlib::PhonebookEntryBase &phonebookEntry) :
  _phonebookEntry(phonebookEntry.clone()), _phonebookIterator(NULL)
{
}

Entry::Entry(gsmlib::SMSStoreEntry &smsStoreEntry) :
  _smsStoreEntry(smsStoreEntry.clone()), _smsStoreIterator(NULL)
{
}

void Entry::setPbText(string text)
{
  _phonebookEntry->set(pbTelephone(), text, pbIndex());
}

void Entry::setPbTelephone(string telephone)
{
  _phonebookEntry->set(telephone, pbText(), pbIndex());
}

string Entry::smsText() const
{
  switch (_smsStoreEntry->message()->messageType())
  {
  case SMSMessage::SMS_SUBMIT:
  case SMSMessage::SMS_DELIVER:
    return _smsStoreEntry->message()->userData();
    break;
    
  case SMSMessage::SMS_STATUS_REPORT:
  {
    SMSStatusReportMessage &sr =
      (SMSStatusReportMessage&)(_smsStoreEntry->message())();
    string recipientAddress = sr.recipientAddress().toString();
    string dischargeTime = sr.dischargeTime().toString();
    return stringPrintf(_("Status Report: Message # %d delivered to %s at %s"),
                        (int)sr.messageReference(),
                        recipientAddress.c_str(), dischargeTime.c_str());
    break;
  }
  }
  assert(0);                    // never reached
  return "";
}

string Entry::smsTelephone() const
{
  return _smsStoreEntry->message()->address().toString();
}

string Entry::smsDate() const
{
  switch (_smsStoreEntry->message()->messageType())
  {
  case SMSMessage::SMS_SUBMIT:
    return "";
  case SMSMessage::SMS_DELIVER:
  {
    SMSDeliverMessage &sd =
      (SMSDeliverMessage&)(_smsStoreEntry->message())();
    return sd.serviceCentreTimestamp().toString(false);
    break;
  }
  case SMSMessage::SMS_STATUS_REPORT:
  {
    SMSStatusReportMessage &sr =
      (SMSStatusReportMessage&)(_smsStoreEntry->message())();
    return sr.dischargeTime().toString(false);
    break;
  }
  }
  assert(0);                    // never reached
  return "";
}

string Entry::smsType() const
{
  switch (_smsStoreEntry->message()->messageType())
  {
  case SMSMessage::SMS_SUBMIT:
    return "S";
  case SMSMessage::SMS_DELIVER:
    return "D";
  case SMSMessage::SMS_STATUS_REPORT:
    return "R";
  }
  assert(0);                    // never reached
  return "";
}

void Entry::setSMSMessage(string telephone, string message,
                          gsmlib::SMSMessage::MessageType type)
{
  switch (type)
  {
  case SMSMessage::SMS_SUBMIT:
  {
    _smsStoreEntry =
      new SMSStoreEntry(new SMSSubmitMessage(message, telephone));
    break;
  }
  case SMSMessage::SMS_DELIVER:
  {
    SMSDeliverMessage *newSMS = new SMSDeliverMessage();
    Address originatingAddress(telephone);
    newSMS->setOriginatingAddress(originatingAddress);
    newSMS->setUserData(message);
    _smsStoreEntry = new SMSStoreEntry(newSMS);
    break;
  }
  default:
    assert(0);
  }
}

// Request and Response implementations

int Request::_nextRequestId = 0;

Request::Request(const CompletionCallback &callback)
{
  _completionConnection = _signal.connect(callback);
}

gsmlib::MEInfo DeviceInfoResponse::getMeInfo()
{
  assert(_deviceInfo == DeviceInfoRequest::MeInfo);
  return _meInfo;
}

vector<gsmlib::OPInfo> DeviceInfoResponse::getAvailableOPInfo()
{
  assert(_deviceInfo == DeviceInfoRequest::OperatorInfo);
  return _operatorInfo;
}

gsmlib::OPInfo DeviceInfoResponse::getCurrentOPInfo()
{
  assert(_deviceInfo == DeviceInfoRequest::CurrentOperatorInfo);
  return _currentOperatorInfo;
}

bool DeviceInfoResponse::getNetworkCLIP()
{
  assert(_deviceInfo == DeviceInfoRequest::CLIPInfo);
  return _clipInfo;
}

DeviceInfoResponse::ForwardInfoTriple
DeviceInfoResponse::getCallForwardInfo(ForwardReason reason)
{
  assert(_deviceInfo == DeviceInfoRequest::CallForwardingInfo);
  vector<ForwardReason> fr = ((DeviceInfoRequest&)request()).forwardReasons();

  int j = 0;
  for (vector<ForwardReason>::iterator i = fr.begin(); i != fr.end(); ++i, ++j)
    if (*i == reason)
      return _forwardingInfo[j];

  assert(0);                    // reason was not requested!
  return ForwardInfoTriple();   // for the compiler...
}

int DeviceInfoResponse::getBatteryChargeStatus()
{
  assert(_deviceInfo == DeviceInfoRequest::BatteryInfo);
  return _batteryInfo;
}

int DeviceInfoResponse::getBitErrorRate()
{
  assert(_deviceInfo == DeviceInfoRequest::BitErrorInfo);
  return _bitErrorInfo;
}

string DeviceInfoResponse::getServiceCentreAddress()
{
  assert(_deviceInfo == DeviceInfoRequest::SCAInfo);
  return _serviceCentreInfo;
}

vector<string> DeviceInfoResponse::getSupportedCharSets()
{
  assert(_deviceInfo == DeviceInfoRequest::CharSetInfo);
  return _charSetInfo;
}

int DeviceInfoResponse::getSignalStrength()
{
  assert(_deviceInfo == DeviceInfoRequest::SignalInfo);
  return _signalStrengthInfo;
}

// GsmlibInterrupt implementation

pthread_mutex_t GsmlibInterrupt::_interruptedMtx = PTHREAD_MUTEX_INITIALIZER;
pthread_key_t GsmlibInterrupt::_interruptedKey;

GsmlibInterrupt::GsmlibInterrupt()
{
  pthread_key_create(&_interruptedKey, NULL);
  setInterruptObject(this);     // register with gsmlib
}

bool GsmlibInterrupt::interrupted()
{
  pthread_mutex_lock(&_interruptedMtx);
  // get Device::_interrupted
  bool result = *(bool*)pthread_getspecific(_interruptedKey);
  pthread_mutex_unlock(&_interruptedMtx);
  return result;
}

void GsmlibInterrupt::registerInterrupted(bool &interrupted)
{
  pthread_setspecific(_interruptedKey, (void*)&interrupted);
}

void GsmlibInterrupt::interrupt(bool &interrupted)
{
  pthread_mutex_lock(&_interruptedMtx);
  interrupted = true;
  pthread_mutex_unlock(&_interruptedMtx);
}

// GsmlibProgress implementation

pthread_key_t GsmlibProgress::_progressKey;

GsmlibProgress::GsmlibProgress()
{
  pthread_key_create(&_progressKey, NULL);
  setProgressObject(this);      // register with gsmlib
}

void GsmlibProgress::reportProgress(int part, int total)
{
  ProgressInfo *pi = (ProgressInfo*)pthread_getspecific(_progressKey);
  cout << "PROGRESS IN " << part << "/" << total << endl; // FIXME
  if (pi != NULL)
  {
    if (part == -1 && total == -1)
    {
      if (pi->_part >=0 && pi->_total >= 0)
        ++pi->_part;
      else
        return;
    }
    else if (total == -1)
      pi->_part = part;         // part must be != -1
    else
    {
      assert(part >= 0);
      pi->_part = part;
      pi->_total = total;
    }
    cout << "PROGRESS OUT " << pi->_part << "/" << pi->_total << endl; // FIXME
    mainWindow->updateProgress(*pi->_progressId, 
                               (gfloat)pi->_part / pi->_total);
  }
}

void GsmlibProgress::registerProgressId(int &progressId)
{
  ProgressInfo *pi = new ProgressInfo(&progressId);
  pthread_setspecific(_progressKey, (void*)pi);
}

void GsmlibProgress::unregisterProgressId()
{
  ProgressInfo *pi = (ProgressInfo*)pthread_getspecific(_progressKey);
  if (pi != NULL)
    delete pi;
  pthread_setspecific(_progressKey, NULL);
}

// Device implementation

GsmlibInterrupt *Device::_gsmlibInterruptObject = NULL;
GsmlibProgress *Device::_gsmlibProgressObject = NULL;

void *Device::runDevice(Device *device)
{
  device->run();
  DEBUG_START(1);
  cout << "thread " << getpid() << " terminated" << endl;
  DEBUG_END;
  return NULL;
}

void Device::run()
{
  int progressId;
  // set thread-specific pointer to _interrupted member
  _gsmlibInterruptObject->registerInterrupted(_interrupted);

  // loop accept requests and handle them
  // the _requestMtx stays locked till the response is ready
  while (true)
  {
    DEBUG_START(1);
    cout << "thread " << getpid() << " running, waiting for request" << endl;
    DEBUG_END;
    if (_terminate) return;
    pthread_cond_wait(&_newRequestCond, &_requestMtx);
    if (_terminate) return;
    progressId = -1;

    try
    {
      // --- OpenRequest
      if (typeid(_request()) == typeid(OpenRequest))
      {
        // this code can be called multiple times for an already open device
        // it should have no "side effects", only store OpenRequest's params
        // and do one-time initializations
        OpenRequest &req((OpenRequest&)_request());
        OpenResponse *res =
          new OpenResponse(_request, req._eventConnection, this);
        _response = res;

        if (req._dataSource == PhonebookFile)
        {
          // find out absolute filename (only if there is a chance...)
          if (req._fileName != "")
          {
            _sourceName = fileExists(req._fileName, PHONEBOOK_FILE_EXTENSION);
            
            if (_sourceName == "")
              throw GsmException(stringPrintf(_("cannot open file %s"),
                                              req._fileName.c_str()),
                                 ParameterError);
          }
          progressId = mainWindow->initProgress(_sourceName);

          // get phonebook name
          res->_phonebookNames = getStoreNames(PHONEBOOK_FILE_EXTENSION);
        }
        else if (req._dataSource == SMSFile)
        {
          // find out absolute filename (only if there is a chance...)
          if (req._fileName != "")
          {
            _sourceName = fileExists(req._fileName, SMSSTORE_FILE_EXTENSION);
            
            if (_sourceName == "")
              throw GsmException(stringPrintf(_("cannot open file %s"),
                                              req._fileName.c_str()),
                                 ParameterError);
          }
          progressId = mainWindow->initProgress(_sourceName);

          // get SMS store names
          res->_SMSStoreNames = getStoreNames(SMSSTORE_FILE_EXTENSION);
        }
        else
        {
          _sourceName = req._deviceName;
          progressId = mainWindow->initProgress(_sourceName);

          // open device if not done
          if (_meta.isnull())
          {
            mainWindow->message(
              stringPrintf(_("Trying to open device %s..."),
                           _sourceName.c_str()));
            _meta = new MeTa(
              new UnixSerialPort(_sourceName,
                                 baudRateStrToSpeed(req._baudRate),
                                 req._initString, req._swHandshake));
            MEInfo mei = _meta->getMEInfo();
            string model = mei._manufacturer + mei._model;
            mainWindow->message(
              stringPrintf(_("Opened device %s [%s]"),
                           _sourceName.c_str(), model.c_str()));
          }
          mainWindow->updateProgress(progressId, 0.33);
          // get phonebook and SMS store names (from cache if not the first
          // OpenRequest)
          if (_phonebookNames.size() > 0 || _SMSStoreNames.size() > 0)
          {
            res->_phonebookNames = _phonebookNames;
            mainWindow->updateProgress(progressId, 0.66);
            res->_SMSStoreNames = _SMSStoreNames;
          }
          else
          {
            _phonebookNames = res->_phonebookNames =
              _meta->getPhoneBookStrings();
            mainWindow->updateProgress(progressId, 0.66);
            _SMSStoreNames = res->_SMSStoreNames = _meta->getSMSStoreNames();
          }
        }
        _dataSource = req._dataSource;
        mainWindow->finishProgress(progressId);
      }

      // --- DeviceInfoRequest
      else if (typeid(_request()) == typeid(DeviceInfoRequest))
      {
        DeviceInfoRequest &req = (DeviceInfoRequest&)_request();

        progressId = mainWindow->initProgress(_sourceName);
        DeviceInfoRequest::DeviceInfo di = req.deviceInfo();

        // count bits in deviceInfoField - for the progress bar
        int taskSize = 0, taskFinished = 0;
        for (int i = (int)DeviceInfoRequest::MeInfo;
             i <= (int)DeviceInfoRequest::SignalInfo; ++i)
          if ((int)di & (1 << i))
            ++taskSize;
        
        if (di & DeviceInfoRequest::CallForwardingInfo)
          taskSize += req.forwardReasons().size();

        _response = new DeviceInfoResponse(_request);
        DeviceInfoResponse &resp = (DeviceInfoResponse&)_response();
        int actualDeviceInfo = 0;

        // MeInfo
        if (di & DeviceInfoRequest::MeInfo)
        {
          try
          {
            resp._meInfo = _meta->getMEInfo();
            actualDeviceInfo |= DeviceInfoRequest::MeInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request ME info [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // OperatorInfo
        if (di & DeviceInfoRequest::OperatorInfo)
        {
          try
          {
            resp._operatorInfo = _meta->getAvailableOPInfo();
            actualDeviceInfo |= DeviceInfoRequest::OperatorInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request operator info "
                             "[%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // CurrentOperatorInfo
        if (di & DeviceInfoRequest::CurrentOperatorInfo)
        {
          try
          {
            resp._currentOperatorInfo = _meta->getCurrentOPInfo();
            actualDeviceInfo |= DeviceInfoRequest::CurrentOperatorInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request current "
                             "operator info [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // CLIPInfo
        if (di & DeviceInfoRequest::CLIPInfo)
        {
          try
          {
            resp._clipInfo = _meta->getNetworkCLIP();
            actualDeviceInfo |= DeviceInfoRequest::CLIPInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request CLIP info "
                             "[%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // CallForwardingInfo
        if (di & DeviceInfoRequest::CallForwardingInfo)
        {
          try
          {
            vector<ForwardReason> fr = req.forwardReasons();
            for (vector<ForwardReason>::iterator i = fr.begin();
                 i != fr.end(); ++i)
            {
              DeviceInfoResponse::ForwardInfoTriple result;
              _meta->getCallForwardInfo(*i, result._voice,
                                        result._fax, result._data);
              resp._forwardingInfo.push_back(result);
              mainWindow->updateProgress(progressId,
                                         (gfloat)++taskFinished / taskSize);
            }
            actualDeviceInfo |= DeviceInfoRequest::CallForwardingInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request call forwarding "
                             "info [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
        }
        // BatteryInfo
        if (di & DeviceInfoRequest::BatteryInfo)
        {
          try
          {
            resp._batteryInfo = _meta->getBatteryChargeStatus();
            actualDeviceInfo |= DeviceInfoRequest::BatteryInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request battery "
                             "charge status [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // BitErrorInfo
        if (di & DeviceInfoRequest::BitErrorInfo)
        {
          try
          {
            resp._bitErrorInfo = _meta->getBitErrorRate();
            actualDeviceInfo |= DeviceInfoRequest::BitErrorInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request bit error "
                             "rate [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // SCAInfo
        if (di & DeviceInfoRequest::SCAInfo)
        {
          try
          {
            resp._serviceCentreInfo = _meta->getServiceCentreAddress();
            actualDeviceInfo |= DeviceInfoRequest::SCAInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request service centre "
                             "address [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // CharSetInfo
        if (di & DeviceInfoRequest::CharSetInfo)
        {
          try
          {
            resp._charSetInfo = _meta->getSupportedCharSets();
            actualDeviceInfo |= DeviceInfoRequest::CharSetInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request supported "
                             "character sets [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }
        // SignalInfo
        if (di & DeviceInfoRequest::SignalInfo)
        {
          try
          {
            resp._signalStrengthInfo = _meta->getSignalStrength();
            actualDeviceInfo |= DeviceInfoRequest::SignalInfo;
          }
          catch (GsmException &ge)
          {
            if (ge.getErrorClass() == InterruptException)
              throw;
            mainWindow->message(
              stringPrintf(_("Device error: Cannot request signal strength "
                             "info [%s, %s]"),
                           ge.what(), _sourceName.c_str()), true);
          }
          mainWindow->updateProgress(progressId,
                                     (gfloat)++taskFinished / taskSize);
        }

        resp._deviceInfo = (DeviceInfoRequest::DeviceInfo)actualDeviceInfo;
        mainWindow->finishProgress(progressId);
      }

      // --- GetPbEntriesRequest
      else if (typeid(_request()) == typeid(GetPbEntriesRequest))
      {
        GetPbEntriesRequest &req = (GetPbEntriesRequest&)_request();

        progressId = mainWindow->initProgress(_sourceName);
        _gsmlibProgressObject->registerProgressId(progressId);

        SortedPhonebookRef pb = getPhonebook(req.phonebookName());
        EntryListRef entries = new EntryList();

        pb->setSortOrder(req.sortOrder());

        for (SortedPhonebookIterator i = pb->begin(); i != pb->end(); ++i)
          entries->push_back(i);

        _gsmlibProgressObject->unregisterProgressId();
        mainWindow->finishProgress(progressId);

        _response = new GetPbEntriesResponse(_request, entries,
                                             pb->getMaxTextLen(),
                                             pb->getMaxTelephoneLen());
      }

      // --- GetSMSEntriesRequest
      else if (typeid(_request()) == typeid(GetSMSEntriesRequest))
      {
        GetSMSEntriesRequest &req = (GetSMSEntriesRequest&)_request();

        progressId = mainWindow->initProgress(_sourceName);
        _gsmlibProgressObject->registerProgressId(progressId);

        SortedSMSStoreRef sms = getSmsStore(req.SMSStoreName());
        EntryListRef entries = new EntryList();

        sms->setSortOrder(req.sortOrder());

        for (SortedSMSStoreIterator i = sms->begin(); i != sms->end(); ++i)
          entries->push_back(i);

        _gsmlibProgressObject->unregisterProgressId();
        mainWindow->finishProgress(progressId);

        _response = new GetSMSEntriesResponse(_request, entries);
      }

      // --- DeletePbEntriesRequest
      else if (typeid(_request()) == typeid(DeletePbEntriesRequest))
      {
        DeletePbEntriesRequest &req = (DeletePbEntriesRequest&)_request();
        SortedPhonebookRef pb = getPhonebook(req.phonebookName());
        progressId = mainWindow->initProgress(_sourceName);
        int size = req.entryIterators().end() - req.entryIterators().begin()
          + 1;
        int p = 0;
        for (vector<gsmlib::SortedPhonebookIterator>::iterator i =
               req.entryIterators().begin(); i != req.entryIterators().end();
             ++i, ++p)
        {
          //sleep(1);
          pb->erase(*i);
          mainWindow->updateProgress(progressId, (gfloat)p / size);
        }
        pb->sync();
        mainWindow->finishProgress(progressId);
        _response = new DeletePbEntriesResponse(_request);
      }

      // --- DeleteSmsEntriesRequest
      else if (typeid(_request()) == typeid(DeleteSmsEntriesRequest))
      {
        DeleteSmsEntriesRequest &req = (DeleteSmsEntriesRequest&)_request();
        SortedSMSStoreRef sms = getSmsStore(req.smsStoreName());
        progressId = mainWindow->initProgress(_sourceName);
        int size = req.entryIterators().end() - req.entryIterators().begin()
          + 1;
        int p = 0;
        for (vector<gsmlib::SortedSMSStoreIterator>::iterator i =
               req.entryIterators().begin(); i != req.entryIterators().end();
             ++i, ++p)
        {
          //sleep(1);
          sms->erase(*i);
          mainWindow->updateProgress(progressId, (gfloat)p / size);
        }
        sms->sync();
        mainWindow->finishProgress(progressId);
        _response = new DeleteSmsEntriesResponse(_request);
      }

      // --- UpdatePbEntryRequest
      else if (typeid(_request()) == typeid(UpdatePbEntryRequest))
      {
        UpdatePbEntryRequest &req = (UpdatePbEntryRequest&)_request();
        SortedPhonebookRef pb = getPhonebook(req.phonebookName());
        progressId = mainWindow->initProgress(_sourceName);
        *req.entry().phonebookIterator() = req.entry();
        mainWindow->updateProgress(progressId, 0.50);
        pb->sync();
        mainWindow->finishProgress(progressId);
        _response = new UpdatePbEntryResponse(_request);
      }

      // --- AddPbEntryRequest
      else if (typeid(_request()) == typeid(AddPbEntryRequest))
      {
        AddPbEntryRequest &req = (AddPbEntryRequest&)_request();
        SortedPhonebookRef pb = getPhonebook(req.phonebookName());
        progressId = mainWindow->initProgress(_sourceName);
        int p = 0;

        switch (req.operation())
        {
        case AddPbEntryRequest::Add:
        {
          for (EntryList::iterator i = req.newEntries()->begin();
               i != req.newEntries()->end(); ++i, ++p)
          {
            //sleep(1);
            pb->insert(PhonebookEntryBase(i->pbTelephone(), i->pbText()));
            mainWindow->updateProgress(progressId,
                                       (gfloat)p / req.newEntries()->size());
          }
          break;
        }
        case AddPbEntryRequest::Synchronize:
        {
          SortOrder savedSortOrder = pb->sortOrder();
          pb->setSortOrder(ByIndex);
          set<int> entriesToKeep; // indices of entries to keep in destination
          
          // add all new entries to phonebook that not already present
          // remember all phonebook entries that are also in newEntries
          for (EntryList::iterator i = req.newEntries()->begin();
               i != req.newEntries()->end(); ++i, ++p)
          {
            // check if new entry is already present in destination
            SortedPhonebook::iterator entryFound = pb->find(i->pbIndex());
            bool present = false;
            if (entryFound != pb->end())
            {
              entryFound->setUseIndex(true);
              present = ((PhonebookEntryBase)*i) == *entryFound;
            }

            if (present)
              entriesToKeep.insert(i->pbIndex()); // do nothing
            else
            {
              if (entryFound != pb->end())
                pb->erase(entryFound); // erase old entry
              entriesToKeep.insert(
                pb->insert((PhonebookEntryBase)*i)->index()); // then insert
            }

            //sleep(1);
            mainWindow->updateProgress(progressId,
                                       (gfloat)p / req.newEntries()->size());
          }

          // erase all sms store entries not present in newEntries
          for (SortedPhonebook::iterator i = pb->begin(); i != pb->end();)
            if (entriesToKeep.find(i->index()) == entriesToKeep.end())
            {
              pb->erase(i);
              i = pb->begin();
              // FIXME: should be i = pb->erase(i);
            }
            else
              ++i;

          pb->setSortOrder(savedSortOrder);
          break;
        }
        }
        pb->sync();
        mainWindow->finishProgress(progressId);
        _response = new AddPbEntryResponse(_request);
      }

      // --- AddSmsEntryRequest
      else if (typeid(_request()) == typeid(AddSmsEntryRequest))
      {
        AddSmsEntryRequest &req = (AddSmsEntryRequest&)_request();
        SortedSMSStoreRef sms = getSmsStore(req.smsStoreName());
        progressId = mainWindow->initProgress(_sourceName);
        int p = 0;

        switch (req.operation())
        {
        case AddSmsEntryRequest::Add:
        {
          for (EntryList::iterator i = req.newEntries()->begin();
               i != req.newEntries()->end(); ++i, ++p)
          {
            //sleep(1);
            sms->insert(*i);
            mainWindow->updateProgress(progressId,
                                       (gfloat)p / req.newEntries()->size());
          }
          break;
        }
        case AddSmsEntryRequest::Synchronize:
        case AddSmsEntryRequest::Backup:
        {
          SortOrder savedSortOrder = sms->sortOrder();
          sms->setSortOrder(ByDate);
          set<int> entriesToKeep; // indices of entries to keep in destination

          // add all new entries to sms store that not already present
          // remember all sms store entries that are also in newEntries
          for (EntryList::iterator i = req.newEntries()->begin();
               i != req.newEntries()->end(); ++i, ++p)
          {
            // check if new entry is already present in destination
            Timestamp date =
              ((SMSStoreEntry)*i).message()->serviceCentreTimestamp();
            pair<SortedSMSStore::iterator, SortedSMSStore::iterator> range =
              sms->equal_range(date);
            bool present = false;

            for (SortedSMSStore::iterator j = range.first;
                 j != range.second; ++j)
              if (((SMSStoreEntry)*i) == *j)
              {
                // do nothing if the entry is already present
                present = true;
                entriesToKeep.insert(j->index());
                break;
              }

            if (! present)
              entriesToKeep.insert(sms->insert((SMSStoreEntry)*i)->index());
            //sleep(1);
            mainWindow->updateProgress(progressId,
                                       (gfloat)p / req.newEntries()->size());
          }

          // erase all sms store entries not present in newEntries
          // (Synchronize only)
          if (req.operation() == AddSmsEntryRequest::Synchronize)
            for (SortedSMSStore::iterator i = sms->begin(); i != sms->end();)
              if (entriesToKeep.find(i->index()) == entriesToKeep.end())
              {
                sms->erase(i);
                i = sms->begin();
                // FIXME: should be i = sms->erase(i);
              }
              else
                ++i;

          sms->setSortOrder(savedSortOrder);
          break;
        }
        }
        sms->sync();
        mainWindow->finishProgress(progressId);
        _response = new AddSmsEntryResponse(_request);
      }

      // --- SendSMSRequest
      else if(typeid(_request()) == typeid(SendSMSRequest))
      {
        SendSMSRequest &req = (SendSMSRequest&)_request();
        
        //cerr << "Would send:" << endl << req.submitSMS()->toString();
        //sleep(3);
        _meta->sendSMS(req.submitSMS());
        
        _response = new SendSMSResponse(_request);
      }

#ifndef NDEBUG
      else
      {
        cerr << "error: unknown request" << endl;
        assert(0);
      }
#endif
    }
    catch (GsmException &ge)
    {
      _response = new ErrorResponse(_request, ge, _sourceName);
      if (progressId != -1)
      {
        _gsmlibProgressObject->unregisterProgressId();
        mainWindow->finishProgress(progressId);
      }
      mainWindow->message(stringPrintf(_("Device/File error: %s [%s]"),
                                       ge.what(), _sourceName.c_str()), true);
    }
    _request == NULL;           // for the busy() call
  }
}

SortedPhonebookRef Device::getPhonebook(string phonebookName)
{
  if (_dataSource == PhonebookFile)
  {
    if (_filePhonebook.isnull())
    {
      _filePhonebook = new SortedPhonebook(_sourceName, true);
      mainWindow->message(
        stringPrintf(_("Opened phonebook file %s [%d entries]"),
                     _sourceName.c_str(), _filePhonebook->size()));
    }
    return _filePhonebook;
  }
  else
  {
    SortedPhonebookRef result = _metaPhonebooks[phonebookName];
    if (result.isnull())
    {
      result = new SortedPhonebook(_meta->getPhonebook(phonebookName, true));
      mainWindow->message(
        stringPrintf(_("Opened phonebook %s:%s [%d/%d entries]"),
                     _sourceName.c_str(), phonebookName.c_str(),
                     result->size(), result->capacity()));
      _metaPhonebooks[phonebookName] = result;
    }
    return result;
  }
}

SortedSMSStoreRef Device::getSmsStore(string smsStoreName)
{
  if (_dataSource == SMSFile)
  {
    if (_fileSMSStore.isnull())
    {
      _fileSMSStore = new SortedSMSStore(_sourceName);
      mainWindow->message(
        stringPrintf(_("Opened sms store file %s [%d entries]"),
                     _sourceName.c_str(), _fileSMSStore->size()));
    }
    return _fileSMSStore;
  }
  else
  {
    SortedSMSStoreRef result = _metaSMSStores[smsStoreName];
    if (result.isnull())
    {
      result = new SortedSMSStore(_meta->getSMSStore(smsStoreName));
      mainWindow->message(
        stringPrintf(_("Opened sms store %s:%s [%d/%d entries]"),
                     _sourceName.c_str(), smsStoreName.c_str(),
                     result->size(), result->capacity()));
      _metaSMSStores[smsStoreName] = result;
    }
    return result;
  }
}

void Device::openMeTa()
{
}

gint Device::idle()
{
  if (_response.isnull()) return TRUE;
  if (pthread_mutex_trylock(&_requestMtx) != 0) return TRUE;

  DEBUG_START(1);
  cout << "idle func called " << endl;
  DEBUG_END;

  ResponseRef res = _response;
  _response = NULL;
  pthread_mutex_unlock(&_requestMtx);

  // tell all interested parties that device is idle again
  _eventSignal.emit(IdleEvent);
  mainWindow->unregisterBusyDevice(_name);
  _busy = false;

  // signal the caller that it's request was handled
  DEBUG_START(1);
  cout << "emitting signal" << endl;
  DEBUG_END;
  res->request()->signal().emit(res);

  // shut down the device in case of error
  if (typeid(res()) == typeid(ErrorResponse))
    unregisterDevice(this);

  return FALSE;
}

string Device::fileExists(string filename, string extension)
{
  // remove leading and trailing spaces
  const char *f = filename.c_str();
  while (isspace(*f)) ++f;
  const char *p = f + strlen(f);
  while (p > f && isspace(*(p - 1))) --p;
  filename = string(f, p - f);

  string homedir = homeDir();
  // filename is absolute or relative, or homedir cannot be determined
  if (filename.find('/') != string::npos || homedir == "")
  {
    if (Xgsm::isFile(filename))
      return filename;
    else
      return "";
  }
  else
  {
    // create the XGSM_DEFAULT_DIR if it does not exist
    struct stat statBuf;
    string defaultDir = homedir + "/" + XGSM_DEFAULT_DIR;
    if (stat(defaultDir.c_str(), &statBuf) != 0)
      if (mkdir(defaultDir.c_str(), 0775) != 0)
      {
        mainWindow->message(
          stringPrintf(_("cannot create %s"), XGSM_DEFAULT_DIR));
        DEBUG_START(1);
        cout << "could not create .xgsm" << endl;
        DEBUG_END;

        return "";
      }
    if (! S_ISDIR(statBuf.st_mode))
    {
      mainWindow->message(
        stringPrintf(_("%s is not a directory"), XGSM_DEFAULT_DIR));
      return "";
    }
    
    // filename must be in the XGSM_DEFAULT_DIR
    // if it does not have the default extension, add it
    if (filename.length() < extension.length() ||
        filename.substr(filename.length() - extension.length(),
                        extension.length()) != extension)
      filename += extension;
    filename = defaultDir + "/" + filename;
    if (isFile(filename))
      return filename;
    // else create the file
    ofstream f(filename.c_str());
    if (f.bad())
      return "";
    else
      return filename;
  }
}

vector<string> Device::getStoreNames(string extension)
{
  vector<string> result;
  string homedir;
  if ((homedir = homeDir()) != "")
  {
    string xgsmDirName = homedir + "/" + XGSM_DEFAULT_DIR;
    DIR *xgsmDir = opendir(xgsmDirName.c_str());
    if (xgsmDir != NULL)
    {
      struct dirent entry, *r = &entry;
      while (readdir_r(xgsmDir, &entry, &r) == 0)
      {
        if (r == NULL) break;
        string filename = entry.d_name;
        if (filename.length() >= extension.length() &&
            filename.substr(filename.length() - extension.length(),
                            extension.length()) == extension)
        {
          filename = filename.substr(0, filename.length() -
                                     extension.length());
          result.push_back(filename);
        }
      }
      closedir(xgsmDir);
    }
  }
  return result;
}

Device::Device(string name) : _name(name),
  _busy(false), _interrupted(false), _terminate(false)
{
  DEBUG_START(1);
  cout << "created device " << name << endl;
  DEBUG_END;

  // intialize interrupt and progress objects only once
  if ( _gsmlibInterruptObject == NULL)
    _gsmlibInterruptObject = new GsmlibInterrupt();
  if ( _gsmlibProgressObject == NULL)
    _gsmlibProgressObject = new GsmlibProgress();
  
  // initialized inter-thread communication variables
  pthread_mutex_init(&_requestMtx, NULL);
  pthread_cond_init(&_newRequestCond, NULL);

  // lock mutex to avoid the race condition that there is a request
  // before run() starts
  pthread_mutex_lock(&_requestMtx);

  // start the thread
  pthread_create(&_workerThread, NULL,
                 (void *(*) (void *))&Device::runDevice, (void*)this);

  // wait for the thread to start (blocking on the condition variable)
  pthread_mutex_lock(&_requestMtx);
  pthread_mutex_unlock(&_requestMtx);
}

vector<DeviceRef> Device::_openDevices;

void Device::unregisterDevice(Device *device)
{
  DEBUG_START(1);
  cout << "unregister device refcount: " << device->refCount() << endl;
  DEBUG_END;

  // remove from list
  for (vector<DeviceRef>::iterator i = _openDevices.begin();
       i != _openDevices.end(); ++i)
    if (i->getptr() == device)
    {
      _openDevices.erase(i);
      break;
    }
}

bool Device::noBusyDevices()
{
  int busyDevices = 0;
  for (vector<DeviceRef>::iterator i = _openDevices.begin();
       i != _openDevices.end(); ++i)
    if ((*i)->busy()) ++busyDevices;
  return busyDevices == 0;
}

DeviceRef Device::getDevice(string name, bool create)
{
  for (vector<DeviceRef>::iterator i = _openDevices.begin();
       i != _openDevices.end(); ++i)
    if ((*i)->name() == name) return *i;
  DeviceRef newDevice;
  if (create)
  {
    newDevice = new Device(name);
    _openDevices.push_back(newDevice);
  }
  return newDevice;
}

Connection Device::request(RequestRef req)
{
  // register event callback
  if (typeid(req()) == typeid(OpenRequest))
  {
    OpenRequest &orq = (OpenRequest&)req();
    orq._eventConnection = _eventSignal.connect(orq._eventCallback);
  }

  // make sure there is no new request while thread is still busy
  assert(! _busy);
#ifndef NDEBUG
  if (pthread_mutex_trylock(&_requestMtx) != 0)
  {
    assert(0);
  }
  pthread_mutex_unlock(&_requestMtx);
#endif

  // device is busy now
  _busy = true;
  _eventSignal.emit(BusyEvent);
  mainWindow->registerBusyDevice(_name);

  _request = req;
  DEBUG_START(1);
  cout << "request:: signalling worker thread" << endl;
  DEBUG_END;
  pthread_cond_signal(&_newRequestCond);

  // register idle function
  // (actually implemented as 10 milliseconds timeout to save CPU time)
  _idle.disconnect();
  _idle = Gtk::Main::timeout.connect(slot(this, &Device::idle), 10);

  return req->connection();
}

void Device::checkClose()
{
  if (refCount() == 2)
    unregisterDevice(this);
}

Device::~Device()
{
  DEBUG_START(1);
  cout << "device " << _name << " destroyed" << endl;
  DEBUG_END;

  // terminate the thread
  _terminate = true;
  interrupt();
  pthread_cond_signal(&_newRequestCond);
  void *dummy;
  pthread_join(_workerThread, &dummy);

  // destroy inter-thread communication variables
  pthread_mutex_destroy(&_requestMtx);
  pthread_cond_destroy(&_newRequestCond);

  // deregister idle function
  _idle.disconnect();

  // tell our using objects we are closing
  _eventSignal.emit(CloseEvent);
  
  // unregister just in case that we might have been busy and an error ocurred
  mainWindow->unregisterBusyDevice(_name);

  // tell the user
  if (_dataSource == MeTaDevice)
    mainWindow->message(
      stringPrintf(_("Closed device %s"), _sourceName.c_str()));
  else
    mainWindow->message(
      stringPrintf(_("Closed file %s"), _sourceName.c_str()));
}

