/*  xhangglider
 *  Copyright (C) 1999 Yasuhiro Take
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "xhang.h"
#include "pilot.h"

extern int pylon[8], pilot_mode[8];
extern double pylon_x[5], pylon_y[5];
extern double max_y, viewpoint_z;
extern int scr_width, scr_height;
static int turn[8];
static double d_core[8];

int pilot_pylonglide(int p, double vx, double vy, double direction_r,
		     int bank_r);

int pilot_watch_others(int p, double *cx, double *cy)
{
  extern int pilot_num;
  extern double px[8], py[8], pz[8];

  double x, y, d = 0.0, d2;
  int i, ret = 0;

  for (i = 0; i < pilot_num; i++) {
    if (pilot_mode[i] == PILOT_THERMAL) {
      get_core(px[i], py[i], pz[i], &x, &y);

      if (pz[p] < y - 30.0) {
	d2 = (x - px[p]) * (x - px[p]) + (y - py[p]) * (y - py[p]);
	
	if (d2 < d || d == 0.0) {
	  d = d2;
	  *cx = x;
	  *cy = y;

	  ret = 1;
	}
      }
    }
  }

  return ret;
}

double vector2r(double vx, double vy)
{
  double v_cos, v_sin;
  double r1, r2;
  int minusflag, obtuseflag;

  minusflag = obtuseflag = 0;
  
  v_cos = vx / sqrt(vx * vx + vy * vy);
  v_sin = vy / sqrt(vx * vx + vy * vy);

  r1 = (acos(v_cos) * 180.0 / M_PI);
  r2 = (asin(v_sin) * 180.0 / M_PI);

  if (r1 > 90.0)
    obtuseflag = 1;
  if (r2 < 0.0)
    minusflag = 1;

  if (obtuseflag == 0) {
    if (minusflag == 0)
      return r1;
    else
      return r2;
  } else {
    if (minusflag == 0)
      return r1;
    else
      return -r1;
  }
}

int pilot_inboxglide(int p, double x, double y, double z, int bank_r,
		     double direction_r)
{
  double limit_x;
  int old_bank_r;

  old_bank_r = bank_r;

  if (bank_r == 0) {
    limit_x = (scr_width * y / scr_height) * 8 / 10;
    
    if (x < -limit_x && direction_r < 0 && bank_r == 0) {
      if (y >= max_y / 2)
	bank_r--;
      else
	bank_r++;
    }
    
    if (x > limit_x && direction_r > 0 && bank_r == 0) {
      if (y >= max_y / 2)
	bank_r++;
      else
	bank_r--;
    }
  }
  
  if (z < 0.0) {
    if (y < -(z - viewpoint_z) - 5.0 && direction_r < 0 && direction_r >= -90
	&& bank_r <= 0)
      bank_r++;
    if (y < -(z - viewpoint_z) - 5.0 && direction_r > 0 && direction_r <= 90 &&
	bank_r >= 0)
      bank_r--;
  }
  
  if (y > max_y && direction_r <= -90 && bank_r >= 0)
    bank_r--;
  if (y > max_y && direction_r >= 90 && bank_r <= 0)
    bank_r++;
  if (y < 60.0 && old_bank_r == bank_r && pilot_mode[p] != PILOT_LIDGE) {
    if (direction_r < 0 && direction_r >= -90)
      bank_r++;
    if (direction_r >= 90 && direction_r <= 90)
      bank_r--;
  }
  
  return bank_r;
}

int pilot_randomglide(int p, double x, double y, double z, int bank_r,
		      double direction_r)
{
  int bank_pm;

  if (bank_r > 0)
    bank_pm = 1;
  else if (bank_r < 0)
    bank_pm = -1;
  else
    bank_pm = 0;
  
  turn[p]++;
  
  if (bank_r == 0 && rand() % (50 - turn[p]) == 0) {
    bank_r = 1 - rand() % 3;
    turn[p] = 0;
  } else if (bank_r != 0) {
    if (turn[p] >= 100) {
      bank_r -= bank_pm;
      if (bank_r == 0)
	turn[p] = 0;
    } else {
      if (abs(bank_r) < 2)
	bank_r += bank_pm;
      else if (rand() % (50 - turn[p]) == 0) {
	turn[p] = 100;
      }
    }
  }
  
  if (bank_r == 0) {
    bank_r = pilot_inboxglide(p, x, y, z, bank_r, direction_r);
    if (bank_r)
      turn[p] = 0;
  }

  if (z > y && y > 0 && bank_r == 0) {
    bank_r = pilot_pylonglide(p, -(500.0 - y), -x, direction_r, bank_r);
    turn[p] = 0;
  }
  
  return bank_r;
}

int pilot_pylonglide(int p, double vx, double vy, double direction_r,
		     int bank_r)
{
  double r;
  int bank_pm;
  int limit_bank_r;
  int d_bank_r;

  if (bank_r > 0)
    bank_pm = 1;
  else if (bank_r < 0)
    bank_pm = -1;
  else
    bank_pm = 0;

  if (pilot_mode[p] == PILOT_THERMAL) {
    limit_bank_r = 5;
    d_bank_r = 1;
  } else {
    limit_bank_r = 2;
    d_bank_r = 1;
  }
  
  r = vector2r(vx, vy);
  
  if (abs(r - direction_r) <= 5) {
    if (bank_r)
      bank_r -= bank_pm;
  } else if (abs(r - direction_r) >= 10) {
    if (r < 0)
      r += 360;
    if (direction_r < 0)
      direction_r += 360;

    if (r > direction_r) {
      if (r - direction_r <= 180) {
	if (bank_r > -limit_bank_r)
	  bank_r -= d_bank_r;
      } else {
	if (bank_r < limit_bank_r)
	  bank_r += d_bank_r;
      }
    } else {
      if (direction_r - r <= 180) {
	if (bank_r < limit_bank_r)
	  bank_r += d_bank_r;
      } else {
	if (bank_r > -limit_bank_r)
	  bank_r -= d_bank_r;
      }
    }
  }

  return bank_r;
}

int pilot_centering(int p, double x, double y, double z, int bank_r,
		    double direction_r, double core_x, double core_y)
{
  double vx1, vy1, d;
  int bank_pm, i;

  if (bank_r > 0)
    bank_pm = 1;
  else if (bank_r < 0)
    bank_pm = -1;
  else
    bank_pm = 0;

  vx1 = core_x - x;
  vy1 = core_y - y;
  d = vx1 * vx1 + vy1 * vy1;

  i = pilot_pylonglide(p, -vy1, vx1, direction_r, bank_r);
  /*
  if (d > d_core[p] + 9.0 && d >= 64.0 && abs(bank_r) < 4)
    bank_r = i;
  
  if (d < d_core[p] - 9.0 && d < 256.0 && abs(bank_r) > 2)
    bank_r -= bank_pm;
  else if (d < d_core[p] - 4.5 && d < 64.0 && abs(bank_r) > 1)
    bank_r -= bank_pm;
  if (d < d_core[p] && d >= 256.0 && bank_r == 0)
    bank_r = i;
  */

  if (bank_r == 0 && d > d_core[p])
    bank_r = i;
  else {
    if (d > d_core[p] && d >= 100.0 && abs(bank_r) < 4)
      bank_r += bank_pm;
    if (d > d_core[p] && d >= 36.0 && abs(bank_r) < 3)
      bank_r += bank_pm;
    if (d > d_core[p] && d >= 9.0 && abs(bank_r) < 2)
      bank_r += bank_pm;


    if (d < d_core[p] - 9.0 && d < 81.0 && abs(bank_r) > 1)
      bank_r -= bank_pm;
    if (d < d_core[p] - 9.0 && d < 144.0 && abs(bank_r) > 2)
      bank_r -= bank_pm;
    if (d < d_core[p] - 9.0 && d < 256.0 && abs(bank_r) > 3)
      bank_r -= bank_pm;
    
    
  }
  

  return bank_r;
}

int pilot(int p, double x, double y, double z, int bank_r, double direction_r)
{
  int bank_pm, i;
  double cx, cy, vx, vy, d;
  extern char Pylon;
  
  if (bank_r > 0)
    bank_pm = 1;
  else if (bank_r < 0)
    bank_pm = -1;
  else
    bank_pm = 0;

  switch (pilot_mode[p]) {
    case PILOT_GLIDING:
      if ((pylon[p] != 0 || z > 175.0) && rand() % 50 == 0 && Pylon == 1)
	pilot_mode[p] = PILOT_GLIDING_PYLON;
      else
	bank_r = pilot_randomglide(p, x, y, z, bank_r, direction_r);

      if (z < -50.0)
	pilot_mode[p] = PILOT_SEEK_LIFT;

      if (z < 150.0 && lift(x, y, z) > 0.0 && lidgelift(x, y, z) == 0.0) {
	get_core(x, y, z, &cx, &cy);

	if (z < cy - 20.0) {
	  pilot_mode[p] = PILOT_THERMAL;
	  
	  vx = cx - x;
	  vy = cy - y;
	  
	  bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	
	  d_core[p] = vx * vx + vy * vy;
	}
      }
      
      break;
    case PILOT_GLIDING_PYLON:
      vx = pylon_x[pylon[p]] - x;
      vy = pylon_y[pylon[p]] - y;
      
      if (abs((int)vx) + abs((int)vy) < 10) {
	pylon[p]++;
	if (pylon[p] == 5)
	  pylon[p] = 0;
	pilot_mode[p] = PILOT_GLIDING;
	fprintf(stderr, "pilot.%d pylon get! next = %d\n", p, pylon[p]);
      } else {
	if (z < -0.0) {
	  pilot_mode[p] = PILOT_SEEK_LIFT;
	} else {
	  bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	  bank_r = pilot_inboxglide(p, x, y, z, bank_r, direction_r);
	}
      }
      
      break;
    case PILOT_SEEK_LIFT:
      if (lift(x, y, z) > 0.0) {
	if (lidgelift(x, y, z) > 0.0) {
	  if (z < -50.0)
	    pilot_mode[p] = PILOT_LIDGE;
	} else {
	  get_core(x, y, z, &cx, &cy);

	  if (z < cy - 10.0) {
	    pilot_mode[p] = PILOT_THERMAL;
	    
	    vx = cx - x;
	    vy = cy - y;
	    
	    bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	    
	    d_core[p] = vx * vx + vy * vy;
	  }
	}
      } else {
	i = pilot_watch_others(p, &cx, &cy);
	
	if (z < -100.0 &&
	    (i == 0 || x*x + y*y < (cx - x)*(cx - x) + (cy - y)*(cy - y))) {
	  vx = 0.0 - x;
	  vy = -z - y;
	    
	  bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	  bank_r = pilot_inboxglide(p, x, y, z, bank_r, direction_r);
	} else {
	  if (i) {
	    vx = cx - x;
	    vy = cy - y;

	    bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	  } else {
	    bank_r = pilot_randomglide(p, x, y, z, bank_r, direction_r);
	  }
	}
      }
      break;
    case PILOT_LIDGE:
      if (z > -10.0 || (lift(x, y, z) <= 0.1 && rand() % 10 == 0 &&
			z > -20.0)) {
	pilot_mode[p] = PILOT_SEEK_LIFT;
      }
      
      if (x < -40.0 && direction_r < 0) {
	if (bank_r < 3)
	  bank_r++;
      }
      if (x < -20.0 && direction_r > 0) {
	vx = 50.0 - x;
	vy = -z - 10.0 - y;

	bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	bank_r = pilot_inboxglide(p, x, y, z, bank_r, direction_r);
      }
      if (-20.0 < x && x < 20.0) {
	if(direction_r < 0) {
	  vx = -50.0 - x;
	  vy = -z - 10.0 - y;
	} else {
	  vx = 50.0 - x;
	  vy = -z - 10.0 - y;
	}

	bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	bank_r = pilot_inboxglide(p, x, y, z, bank_r, direction_r);
      }
      if (x > 20.0 && direction_r < 0) {
	vx = -50.0 - x;
	vy = -z - 10.0 - y;

	bank_r = pilot_pylonglide(p, -vy, vx, direction_r, bank_r);
	bank_r = pilot_inboxglide(p, x, y, z, bank_r, direction_r);
      }
      if (x > 40.0 && direction_r > 0) {
	if (bank_r > -3)
	  bank_r--;
      }
      break;
    case PILOT_THERMAL:
      i = get_core(x, y, z, &cx, &cy);
      d = (cx - x) * (cx - x) + (cy - y) * (cy - y);
      
      if (i)
	bank_r = pilot_centering(p, x, y, z, bank_r, direction_r, cx, cy);

      if ((i != 0 && (z - viewpoint_z) >= cy - 5.0 && lift(x, y, z) <= 0.1) ||
	  (i == 0 && rand() % 75 == 0)) {
	if (z < 100.0)
	  pilot_mode[p] = PILOT_SEEK_LIFT;
	else
	  pilot_mode[p] = PILOT_GLIDING;
      }

      d_core[p] = d;
      break;
  }

  return bank_r;
}

