#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#include "netio.h"
#include "fifo.h"

fifo_item * cmd_queue = NULL;
int netio_line_id = 0;
int netio_sock = 0;

struct tcmd_queue_item
{
	struct t_lcp3_cmd * cmd;
	unsigned int size;
	int sock;
	unsigned char sent;	// how many times?
	time_t tstamp; // latest send
};
typedef struct tcmd_queue_item		cmd_queue_item;

int netio_cmd_send(struct t_lcp3_cmd *cmd, int psize);

unsigned int lookup_ip(const char * hostname)
{
    struct hostent *he;
    struct in_addr ia;
    if ( ! hostname ) return(0);
	if ( ! inet_aton(hostname, &ia) )
	{
		#ifdef DEBUG
			fprintf(stderr, "Trying hostname lookup (hostname doesn't seem to be an IP-address)\n");
		#endif
		if ( !(he = gethostbyname(hostname)) )
		{
			#ifdef DEBUG
				fprintf(stderr, "hostname lookup for '%s' failed.\n", hostname);
			#endif
			return(0);
		}
		ia.s_addr = *((int*)(he->h_addr_list[0]));
	}
	#ifdef DEBUG
		fprintf(stderr, "Found ip '%s' for hostname '%s'.\n", inet_ntoa(ia), hostname);
	#endif
	return(ia.s_addr);
}

void netio_proc_ack(int cmd)
{
	cmd_queue_item * cmdqi = fifo_inspect(cmd_queue);
	if ( cmdqi && (cmdqi->cmd->cmd != cmd) )
	{
		#ifdef DEBUG
			fprintf(stderr, "got ack for %d but I'm waiting for %d!\n", cmd, cmdqi->cmd->cmd);
		#endif
		return;
	}
	cmd_queue = fifo_pop(cmd_queue, (void**)&cmdqi);
	free(cmdqi->cmd);
	free(cmdqi);
	netio_proc_queue();
}

void netio_send_ack(int cmd)
{
	int psize = get_lcp3_pack_size(CMD3_ACK);
	struct t_lcp3_cmd * scmd = malloc(psize);
	struct t_lcp3_info_ack * iack = (struct t_lcp3_info_ack*)(((char*)scmd)+sizeof(struct t_lcp3_cmd));
	if ( ! scmd )
	{
		fprintf(stderr, "sending ack failed (malloc() == NULL)!\n");
		return;
	}
	scmd->magic = LCP3_MAGICNR;
	scmd->version = LCP3_PROTO_VERSION;
	scmd->cmd = CMD3_ACK;
	scmd->info_is_valid = 0;
	iack->ackcmd = cmd;
	#ifdef DEBUG
		fprintf(stderr, "sending CMD3_ACK for %d\n", cmd);
	#endif
	netio_cmd_send(scmd, psize);
}

int netio_handle_data(int sock, struct t_lcp3_cmd ** recv_data)
{
	static struct t_lcp3_cmd * cmdptr = NULL;
	int infosize, tsize = 0;
	void * data_ptr;
	if ( !cmdptr ) cmdptr = malloc(sizeof(struct t_lcp3_cmd));
	data_ptr = &cmdptr->version;
	cmdptr->magic = 0;

	while ( cmdptr->magic != LCP3_MAGICNR ) read(sock, cmdptr, 4);
	tsize = 4;
	tsize += read(sock, data_ptr, sizeof(struct t_lcp3_cmd)-4);
	infosize = get_lcp3_pack_size(cmdptr->cmd);
	*recv_data = malloc(infosize);
	if ( !*recv_data )
	{
		fprintf(stderr, "unable to malloc() receiving buffer!\n");
		return -1;
	}
	memcpy(*recv_data, cmdptr, sizeof(struct t_lcp3_cmd));
	infosize -= sizeof(struct t_lcp3_cmd);
	if ( infosize )
	{ // read info
		data_ptr = ((char*)(*recv_data)) + sizeof(struct t_lcp3_cmd);
		tsize += read(sock, data_ptr, infosize);
	}
	if ( tsize != infosize + sizeof(struct t_lcp3_cmd) )
	{
		fprintf(stderr, "Bad packet size! Infosize != %d\n", infosize);
		free(*recv_data);
		*recv_data = NULL;
		return 0;
	}
	#ifdef DEBUG
		fprintf(stderr, "got cmd %d\n", cmdptr->cmd);
	#endif
	if ( cmdptr->cmd == CMD3_ACK )
	{
		struct t_lcp3_info_ack * iack = data_ptr;
		netio_proc_ack(iack->ackcmd);
	}
	else if ( cmdptr->cmd < MIN_CBR3 )
		netio_send_ack(cmdptr->cmd);
	return 1;
}

int netio_initiate_tcp_connection(const char * hostname, unsigned short port)
{
	char * pname_tcp = "tcp";
	struct protoent * pent;
	struct sockaddr_in sia;
	if ( cmd_queue )
	{
		cmd_queue_item * cur;
		#ifdef DEBUG
			fprintf(stderr, "throwing away old commands which were left in the fifo...\n");
		#endif
		while ( (cmd_queue = fifo_pop(cmd_queue, (void**)&cur)) )
			if ( cur )
			{
				free(cur->cmd);
				free(cur);
			}
		if ( cur )
		{
			free(cur->cmd);
			free(cur);
		}
	}
	#ifdef DEBUG
		fprintf(stderr, "trying hostname lookup and connect()...\n");
	#endif
	pent = getprotobyname(pname_tcp);
	if ( !pent )
	{
		#ifdef DEBUG
			fprintf(stderr, "getprotobyname(\"tcp\") returned NULL!\n");
		#endif
		return -1;
	}
	if ( (netio_sock = socket(AF_INET, SOCK_STREAM, pent->p_proto)) < 0 )
	{
		#ifdef DEBUG
			fprintf(stderr, "socket(AF_INET, SOCK_STREAM, %d); returned < 0\n", pent->p_proto);
		#endif
		return -1;
	}
	sia.sin_family = AF_INET;
	sia.sin_port = port;
	if ( !(sia.sin_addr.s_addr = lookup_ip(hostname)) )
	{
		#ifdef DEBUG
			fprintf(stderr, "Unable to get valid ip address!\n");
		#endif
		close(netio_sock);
		return -1;
	}
	if ( connect(netio_sock, (struct sockaddr*)&sia, sizeof(struct sockaddr_in)) )
	{
		#ifdef DEBUG
			fprintf(stderr, "Unable to get tcp connection with the server!\n");
		#endif
		close(netio_sock);
		return -1;
	}
	#ifdef DEBUG
		fprintf(stderr, "tcp connection to %s:%d established\n", hostname, ntohs(port));
	#endif
	// connection should be ok
	return(netio_sock);
}

void netio_set_socket(int sock)
{
	netio_sock = sock;
}

void netio_set_line_id(int id)
{
	#ifdef DEBUG
		fprintf(stdout, "netio_set_line_id(): selecting line %d\n", id);
	#endif
	netio_line_id = id;
}

int netio_cmd_send(struct t_lcp3_cmd *cmd, int psize)
{
	int ssize = 0;
	fd_set wfd;
	struct timeval tv;
	FD_ZERO(&wfd);
	FD_SET(netio_sock, &wfd);
	tv.tv_sec = 1;
	tv.tv_usec = 0;
	if ( !select(netio_sock+1, NULL, &wfd, NULL, &tv) )
	{
		#ifdef DEBUG
			fprintf(stderr, "select for write timed out after 1 sec!\n");
		#endif
	}
	else
	{
		ssize = send(netio_sock, cmd, psize, 0);
	}
	if ( ssize != psize )
	{
		#ifdef DEBUG
			fprintf(stderr, "netio_cmd_send(): send error, %d of %d bytes sent\n", ssize, psize);
		#endif
		return 0;
	}
	#ifdef DEBUG
		fprintf(stderr, "sent cmd %d\n", cmd->cmd);
	#endif
	return 1;
}

int netio_proc_queue()
{
	cmd_queue_item * pitem = fifo_inspect(cmd_queue);
	if ( !pitem ) return 0;
	if ( (pitem->sent != 0) && (time(NULL) - pitem->tstamp < 2) )
		return 1;
	#ifdef DEBUG
		fprintf(stderr, "netio_proc_queue(): sending cmd %d\n", pitem->cmd->cmd);
	#endif
	if ( netio_cmd_send(pitem->cmd, pitem->size) )
	{
		pitem->tstamp = time(NULL);
		if ( pitem->sent++ > LCP3_TMO_ACK )
		{
			cmd_queue = fifo_pop(cmd_queue, (void**)&pitem);
			#ifdef DEBUG
				fprintf(stderr, "no ack from server... command %d dropped!\n", pitem->cmd->cmd);
			#endif
			free(pitem->cmd);
			free(pitem);
		}
	}
	return (fifo_inspect(cmd_queue) != NULL);
}

int netio_cmd_queue(int cmd, const void * info, int infosize)
{
	int psize = get_lcp3_pack_size(cmd);
	struct t_lcp3_cmd * cmdbuf = malloc(psize);
	cmd_queue_item * qitem = malloc(sizeof(cmd_queue_item));
	fifo_item * old_cmd_queue = cmd_queue;
	char * pinfo = ((char*)cmdbuf) + sizeof(struct t_lcp3_cmd);
	if ( ! (cmdbuf && qitem) )
	{
		fprintf(stderr, "Unable to malloc() send buffer and/or queue item!\n");
		return 0;
	}
	if ( psize - sizeof(struct t_lcp3_cmd) != infosize )
	{
		fprintf(stderr, "code error: netio_cmd_send(): incorrect infosize (%d) for cmd %d\n", infosize, cmd);
		free(cmdbuf);
		return 0;
	}
	memcpy(pinfo, info, infosize);
	cmdbuf->magic = LCP3_MAGICNR;
	cmdbuf->version = LCP3_PROTO_VERSION;
	cmdbuf->cmd = cmd;
	cmdbuf->info_is_valid = 0;
	if ( netio_line_id )
	{
		cmdbuf->info_is_valid = 1;
		cmdbuf->line_id = netio_line_id;
	}
	if ( cmd == CMD3_ACK )
	{
		fprintf(stderr, "ARGL: you may not netio_cmd_queue(CMD3_ACK, ...)!!\n");
		return 0;
	}
	#ifdef DEBUG
		fprintf(stderr, "queueing cmd %d, len = %d bytes, info: %d\n", cmdbuf->cmd, psize, infosize);
	#endif
	// add to the queue...
	qitem->cmd = cmdbuf;
	qitem->size = psize;
	qitem->tstamp = 0;
	qitem->sent = 0;
	cmd_queue = fifo_push(cmd_queue, qitem);
	if ( cmd_queue == old_cmd_queue )
		return 0;
	return (netio_proc_queue());
}
