#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include "proc_lcp3.h"
#include "netio.h"
#include "runtime.h"
#include "cfg.h"
#include "widgets.h"
#include "interface_glob.h"

#include <gtk/gtk.h>

gint proc_lcp3_callback_id;
gint proc_lcp3_refresh_id;
gint proc_lcp3_cmdqueue_callback_id;
gboolean proc_lcp3_init_ok = FALSE;

void proc_autohangup_hupcheck();

void proc_lcp3_process(gpointer data, gint sock, GdkInputCondition condition)
{
	struct t_lcp3_cmd * cmd;
	int infosize;
	char * info;
	struct t_lcp3_info_auth * info_auth;
	struct t_lcp3_info_enckey * info_enckey;
	struct t_lcp3_info_ack * info_ack;
	struct t_lcp3_info_line * info_line;
	struct t_lcp3_info_tput * info_tput;
	struct t_lcp3_info_status * info_status;
	struct t_lcp3_info_isdn * info_isdn;
	// read from socket:
	if ( !netio_handle_data(sock, &cmd) )
	{
		#ifdef DEBUG
			g_print("netio_handle_data() failed!\n");
		#endif
		return;
	}
	#ifdef DEBUG
		g_print("got cmd %d\n", cmd->cmd);
	#endif
	// load pointers
	infosize = get_lcp3_pack_size(cmd->cmd) - sizeof(struct t_lcp3_cmd);
	if ( infosize )
		info = ((char*)cmd) + sizeof(struct t_lcp3_cmd);
	else
		info = NULL;
	info_auth = (void*)info;
	info_enckey = (void*)info;
	info_ack = (void*)info;
	info_line = (void*)info;
	info_tput = (void*)info;
	info_status = (void*)info;
	info_isdn = (void*)info;
	// process status information
	if ( cmd->info_is_valid )
	{
		if ( (cmd->client_stat == CLT_OFFLINE) && (runtime->client_status == CLT_NOAUTH) )
		{
			#ifdef DEBUG
				g_print("authentication successful\n");
			#endif
			if ( wdg_conserver->realized )
			{
				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wdg_conserver->chk_logged_in), TRUE);
				gtk_label_set_text(wdg_conserver->lb_status, "donwloading lines...");
			}
			cmd_send(CMD3_LINESREQ, NULL, 0);
		}
		proc_status_tracker(cmd->client_stat, cmd->line_id, cmd->line_stat,
							cmd->line_channels, FALSE);
	}
	// process command
	switch ( cmd->cmd )
	{
		case CMD3_AUTHBAD:
			if ( wdg_conserver->realized )
				gtk_label_set_text(wdg_conserver->lb_status, "Authentication failed!");
			break;
		case CMD3_CLIENTSTAT:
			// has already been handled
			break;
		case CMD3_LINE:
			// got info_line --> append the line to our GSList * lines
			{
				struct tline * newline = g_malloc(sizeof(struct tline));
				gchar * row[8];
				if ( !newline )
				{
					if ( wdg_conserver->realized )
						gtk_label_set_text(wdg_conserver->lb_status, "Unable to load lines (mem)!");
					break;
				}
				memset(newline, 0, sizeof(struct tline));
				newline->ahup = g_malloc(sizeof(struct tautohup));
				if ( !newline->ahup )
				{
					if ( wdg_conserver->realized )
						gtk_label_set_text(wdg_conserver->lb_status, "Unable to load autohangup information (mem)!");
					break;
				}
				runtime_read_autohup(newline->ahup);
				newline->num = info_line->line_id;
				newline->name = g_strdup(info_line->linename);
				newline->channels = 1;
				newline->ahup->cap_up = info_line->capability_baud_up;
				newline->ahup->cap_down = info_line->capability_baud_down;
				newline->ahup->max_up = newline->ahup->cap_up / 8;
				newline->ahup->max_down = newline->ahup->cap_down / 8;
				lines = g_slist_append(lines, newline);
				// now insert it into the clist
				runtime_load_line_strings(newline);
				row[0] = newline->snum;
				row[1] = newline->name;
				row[2] = NULL;
				row[3] = newline->sstime;
				row[4] = newline->sdns;
				row[5] = newline->sups;
				row[6] = newline->sdnt;
				row[7] = newline->supt;
				gtk_clist_append(wdg_linemaster->cl_lines, row);
				netio_set_line_id(info_line->line_id);
				cmd_send(CMD3_LINESEL, NULL, 0);
				// selected it if it's the default line
				newline->status = -1;
				if ( info_line->total_lines == newline->num )
				{
					runtime->selected_line = runtime_get_line(config->default_line);
					netio_set_line_id(config->default_line);
					if ( !cmd_send(CMD3_LINESEL, NULL, 0) )
						g_print("selecting default line failed.\n");
					gtk_clist_unselect_all(wdg_linemaster->cl_lines);
					gtk_clist_select_row(wdg_linemaster->cl_lines, config->default_line-1, 0);
					interface_propagate_line_data(runtime_get_line(config->default_line));
					if ( wdg_conserver->realized )
					{
						gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(wdg_conserver->chk_lines), TRUE);
						gtk_label_set_text(wdg_conserver->lb_status, "done.");
						// hmm... :)
						gtk_widget_destroy(wnd_conserver);
					}
					cmd_send(CMD3_GETALL, NULL, 0);
				}
			}
			break;
		case CMD3_STATUS:
			// ignore
			break;
		case CMD3_INVALID:
			#ifdef DEBUG
				g_print("command %d seems to be invalid\n", info_ack->ackcmd);
			#endif
			break;
		case CMD3_ENCINFO:
			// only plaintext supported until now...
			{
				struct t_lcp3_info_auth * iauth = g_malloc(sizeof(struct t_lcp3_info_auth));
				if ( !iauth )
				{
					if ( wdg_conserver->realized )
						gtk_label_set_text(wdg_conserver->lb_status, "unable to g_malloc() auth info!");
					break;
				}
				*(iauth->name) = 0;
				*(iauth->pass) = 0;
				iauth->enctype = LCP3_ENCTYPE_NONE;
				if ( info_enckey->enctype != LCP3_ENCTYPE_OPEN )
				{
					strncpy(iauth->name,config->user_name,LCP3_MAX_NAME_LEN);
					strncpy(iauth->pass,config->user_passwd,LCP3_MAX_PWD_LEN);
				}
				cmd_send(CMD3_AUTHREQ, iauth, sizeof(struct t_lcp3_info_auth));
				if ( wdg_conserver->realized )
					gtk_label_set_text(wdg_conserver->lb_status, "auth info sent...");
			}
			break;
		case CMD3_NOACCESS:
			#ifdef DEBUG
				g_print("we don't have access to command %d\n", info_ack->ackcmd);
			#endif
			break;
		case CMD3_CHANFAIL:
			#ifdef DEBUG
				g_print("isdn channel operation failed! %d channels are open.\n", runtime_get_line(cmd->line_id)->channels);
			#endif
			break;
		case CMD3_ACK:
			break;
		case CMD3_UNREG:
			proc_lcp3_cleanup();
			proc_status_tracker(CLT_DISCON, 0, 0, 0, TRUE);
			if ( runtime->terminate ) gtk_main_quit();
			break;
		case CBR3_TPUT:
			{
				struct tline * line = runtime_get_line(cmd->line_id);
				if ( !line ) break;
				line->up_short = info_tput->ups;
				line->up_long = info_tput->upl;
				line->up_total = info_tput->upt / 1024;
				line->down_short = info_tput->dns;
				line->down_long = info_tput->dnl;
				line->down_total = info_tput->dnt / 1024;
				line->client_time = info_tput->client;
				line->server_time = info_tput->line;
				interface_propagate_line_data(line);
				proc_autohangup_hupcheck();
			}
			break;
		case CBR3_LINESTAT:
			// already handled
			break;
		case CBR3_SRVDEATH:
			proc_lcp3_cleanup();
			break;
		case CBR3_ISDNACT:
			{
				struct tmsg * msg = g_malloc(sizeof(struct tmsg));
				if ( !msg ) break;
				msg->row[0] = g_strdup(info_isdn->time);
				msg->row[1] = g_strdup(info_isdn->caller);
				wdg_messages->msgs = g_list_append(wdg_messages->msgs, msg);
				if ( wdg_messages->realized )
					gtk_clist_append(wdg_messages->cl_messages, msg->row);
			}
			break;
		default:
			#ifdef DEBUG
				g_print("Received invalid command: %d\n", cmd->cmd);
			#endif
	}
	if ( cmd ) free(cmd);
}

void proc_autohangup_hupcheck()
{
	struct tline * line = runtime->selected_line;
	// true if tput below limit, resp. time limit < online time
	gboolean crit_time = FALSE;
	gboolean crit_up = FALSE;
	gboolean crit_down = FALSE;
	gboolean crit_tup = FALSE;
	gboolean crit_tdown = FALSE;
	gboolean crit_all = FALSE;
	gboolean hangup = FALSE;
	if ( !line ) return;
	if ( !line->ahup->activated ) return;
	if ( runtime->client_status != CLT_ONLINE ) return;

	if ( !line->ahup->logic_or )
	{
		crit_time = TRUE;
		crit_up = TRUE;
		crit_down = TRUE;
		crit_tup = TRUE;
		crit_tdown = TRUE;
	}

	// check cirterias
	if ( line->ahup->btime ) crit_time = line->ahup->time < line->client_time;
	if ( line->ahup->bup ) crit_up = line->ahup->up > line->up_short;
	if ( line->ahup->bdown ) crit_down = line->ahup->down > line->down_short;
	if ( line->ahup->btotal_up ) crit_tup = line->ahup->total_up > line->up_total;
	if ( line->ahup->btotal_down ) crit_tdown = line->ahup->total_down > line->down_total;

	// update intervall tracker
	if ( crit_up ) line->ahup_tracker.intv_up_failed++;
	else line->ahup_tracker.intv_up_failed = 0;
	if ( crit_down ) line->ahup_tracker.intv_down_failed++;
	else line->ahup_tracker.intv_down_failed = 0;

	// check activity / logic mode & decide
	if ( line->ahup->activity_mode )
	{
		if ( line->ahup->logic_or )
			crit_all = crit_up || crit_down || crit_tup || crit_tdown;
		else
			crit_all = crit_up && crit_down && crit_tup && crit_tdown;

		if ( crit_all ) // ok, there was some activity
			line->ahup_tracker.activity_detected = time(NULL);
		else if ( time(NULL) - line->ahup_tracker.activity_detected >= line->ahup->time * 60 )
			hangup = TRUE;
	}
	else
	{
		crit_up = line->ahup_tracker.intv_up_failed > 4;
		crit_down = line->ahup_tracker.intv_down_failed > 4;
		if ( line->ahup->logic_or )
			crit_all = crit_time || crit_up || crit_down || crit_tup || crit_tdown;
		else
			crit_all = crit_time && crit_up && crit_down && crit_tup && crit_tdown;
		hangup = crit_all;
	}
	
	if ( hangup )
	{
		#ifdef DEBUG
			g_print("autohangup decides to go offline...\n");
		#endif
		cmd_send(CMD3_OFFLINE, NULL, 0);
	}
}

void proc_autohangup_reset(int linenum)
{
	struct tline * line;
	if ( linenum ) line = runtime_get_line(linenum);
	else line = runtime->selected_line;
	if ( !line ) return;
	#ifdef DEBUG
		g_print("resetting autohangup tracking data.\n");
	#endif
	line->ahup_tracker.intv_up_failed = 0;
	line->ahup_tracker.intv_down_failed = 0;
	line->ahup_tracker.activity_detected = time(NULL);
}

void proc_status_tracker(int client, int linenum, int line, int channels, gboolean force)
{
#define WDG_ENABLE(wdg)	gtk_widget_set_sensitive(GTK_WIDGET(wdg), TRUE);
#define WDG_DISABLE(wdg)	gtk_widget_set_sensitive(GTK_WIDGET(wdg), FALSE);
	struct tline * pline = NULL;	
	if ( (runtime->client_status != client) || force )
	{
		runtime->client_status = client;
		switch ( client )
		{
			case CLT_DISCON:
				gtk_label_set_text(wdg_linemaster->lb_client_status, "disconnected");
				WDG_DISABLE(wdg_linemaster->mn_line_autohangup)
				WDG_DISABLE(wdg_linemaster->cl_lines)
				WDG_ENABLE(wdg_linemaster->mn_server_connect)
				WDG_DISABLE(wdg_linemaster->mn_server_disconnect)
				WDG_DISABLE(wdg_linemaster->mn_line_online)
				WDG_DISABLE(wdg_linemaster->mn_line_offline)
				WDG_DISABLE(wdg_linemaster->mn_line_isdnchannels)
				break;
			case CLT_NOAUTH:
				gtk_label_set_text(wdg_linemaster->lb_client_status, "not auth.");
				WDG_DISABLE(wdg_linemaster->mn_line_autohangup)
				WDG_DISABLE(wdg_linemaster->mn_server_connect)
				WDG_ENABLE(wdg_linemaster->mn_server_disconnect)
				break;
			case CLT_OFFLINE:
				gtk_label_set_text(wdg_linemaster->lb_client_status, "offline");
				WDG_ENABLE(wdg_linemaster->mn_line_autohangup)
				WDG_ENABLE(wdg_linemaster->cl_lines)
				WDG_ENABLE(wdg_linemaster->mn_server_disconnect)
				if ( runtime->selected_line && ((runtime->selected_line->status == CST_LOCKED_DOWN)
						|| (runtime->selected_line->status == CST_UP_USER)) )
					WDG_DISABLE(wdg_linemaster->mn_line_online)
				else
					WDG_ENABLE(wdg_linemaster->mn_line_online)
				WDG_DISABLE(wdg_linemaster->mn_line_offline)
				WDG_DISABLE(wdg_linemaster->mn_server_connect)
				WDG_DISABLE(wdg_linemaster->mn_line_isdnchannels)
				break;
			case CLT_DIALING:
				gtk_label_set_text(wdg_linemaster->lb_client_status, "dialing");
				WDG_DISABLE(wdg_linemaster->cl_lines)
				WDG_DISABLE(wdg_linemaster->mn_line_online)
				break;
			case CLT_ONLINE:
				proc_autohangup_reset(0);
				gtk_label_set_text(wdg_linemaster->lb_client_status, "online");
				WDG_DISABLE(wdg_linemaster->cl_lines)
				WDG_ENABLE(wdg_linemaster->mn_line_isdnchannels)
				WDG_DISABLE(wdg_linemaster->mn_line_online)
				WDG_ENABLE(wdg_linemaster->mn_line_offline)
				break;
			case CLT_CLOSING:
				gtk_label_set_text(wdg_linemaster->lb_client_status, "closing");
				WDG_DISABLE(wdg_linemaster->mn_line_isdnchannels)
				WDG_DISABLE(wdg_linemaster->mn_line_offline)
				break;
			default:
				g_print("ARGL! server sent invalid client status %d!\n", client);
		}
	}
	if ( linenum )
		pline = runtime_get_line(linenum);
#ifdef DEBUG
	else
		g_print("linenum == 0  -->  pline == NULL...\n");
#endif
	if ( pline && ((pline->status != line) || (pline->channels != channels)
					|| force) )
	{
		static char * msg_down = "down";
		static char * msg_dialing = "dialing";
		static char * msg_up_server = "up server";
		static char * msg_up_user = "up manually";
		static char * msg_closing = "closing";
		static char * msg_close_error = "close error";
		static char * msg_locked_up_clt = "locked up clt";
		static char * msg_locked_down = "locked down";
		static char * msg_locked_up_timed = "locked up timed";
		static char * msg_unknown = "?";
		if ( (pline->channels != channels) && (pline == runtime->selected_line) )
		{
			pline->channels = channels;
			switch ( channels )
			{
				case 1:
					gtk_check_menu_item_set_active(wdg_linemaster->mn_line_isdnchannels, FALSE);
					break;
				case 2:
					gtk_check_menu_item_set_active(wdg_linemaster->mn_line_isdnchannels, TRUE);
					break;
				default:
			}
		}
		pline->channels = channels;
		pline->status = line;
		if ( channels > 1 ) WDG_DISABLE(wdg_linemaster->mn_line_isdnchannels)
		else WDG_ENABLE(wdg_linemaster->mn_line_isdnchannels)
		switch ( line )
		{
			case CST_DOWN:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_down);
				if ( runtime->selected_line == pline )
				{
					WDG_ENABLE(wdg_linemaster->mn_line_online)
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_down);
				}
				break;
			case CST_DIALING:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_dialing);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_dialing);
				}
				break;
			case CST_UP_SERVER:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_up_server);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_up_server);
				}
				break;
			case CST_UP_USER:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_up_user);
				if ( runtime->selected_line == pline )
				{
					WDG_DISABLE(wdg_linemaster->mn_line_online)
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_up_user);
				}
				break;
			case CST_CLOSING:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_closing);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_closing);
				}
				break;
			case CST_CLOSE_ERROR:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_close_error);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_close_error);
				}
				break;
			case CST_LOCKED_UP_CLT:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_locked_up_clt);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_locked_up_clt);
				}
				break;
			case CST_LOCKED_DOWN:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_locked_down);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_locked_down);
				}
				break;
			case CST_LOCKED_UP_TIMED:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_locked_up_timed);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_locked_up_timed);
				}
				break;
			default:
				gtk_clist_set_text(wdg_linemaster->cl_lines, linenum-1, 2, msg_unknown);
				if ( runtime->selected_line == pline )
				{
					gtk_label_set_text(wdg_linemaster->lb_line_status, msg_unknown);
				}
		}
	}
//	fprintf(stderr, "new status: client=%d, line=%d, linenum=%d, channels=%d\n",
//			client, line, linenum, channels);
}

gboolean proc_lcp3_refresh_callback(gpointer data)
{
	switch ( runtime->client_status )
	{
		case CLT_DISCON:
		case CLT_NOAUTH:
			return TRUE;
		default:
			cmd_send(CMD3_REFRESH, NULL, 0);
	}
	return TRUE;
}

gboolean proc_lcp3_cmd_queue_callback(gpointer ptr)
{
	time_t now = time(NULL);
	if ( (runtime->client_status != CLT_DISCON)
			&& (now - runtime->watchdog > LCP3_CLT_TIMEOUT) )
	{
		g_print("Timeout detected. Reconnecting...\n");
		runtime->watchdog_barked = TRUE;
		proc_lcp3_cleanup();
		proc_lcp3_init();
		cmd_send(CMD3_PUBKEYREQ, NULL, 0);
	}
	else
	{
		runtime->watchdog = now;
		netio_proc_queue();
	}
	return TRUE;
}

int proc_lcp3_init()
{
	if ( proc_lcp3_init_ok )
	{
		g_print("proc_lcp3_init(): already initialized!\n");
		return 1;
	}
	runtime->client_status = CLT_DISCON;
	if ( (runtime->sock = netio_initiate_tcp_connection(config->server_hostname, config->server_port)) < 0 )
	{
		g_print("proc_lcp3_init(): didn't get a tcp connection...\n");
		return 0;
	}
	proc_lcp3_callback_id = gdk_input_add(runtime->sock, GDK_INPUT_READ, proc_lcp3_process, NULL);
	proc_lcp3_refresh_id = gtk_timeout_add((LCP3_CLT_TIMEOUT-5)*1000, proc_lcp3_refresh_callback, NULL);
	proc_lcp3_cmdqueue_callback_id = gtk_timeout_add(3000, proc_lcp3_cmd_queue_callback, NULL);
	runtime->watchdog_barked = FALSE;
	runtime->watchdog = time(NULL);
	proc_lcp3_init_ok = TRUE;
	return 1;
}

int proc_lcp3_cleanup()
{
	runtime->client_status = CLT_DISCON;
	proc_lcp3_init_ok = FALSE;
	gtk_timeout_remove(proc_lcp3_refresh_id);
	gtk_timeout_remove(proc_lcp3_cmdqueue_callback_id);
	gdk_input_remove(proc_lcp3_callback_id);
	close(runtime->sock);
	gtk_clist_clear(wdg_linemaster->cl_lines);
	runtime_save();
	runtime_clear_lines();
	return 1;
}

inline int cmd_send(int cmd, void * info, int infosize)
{
	char res = netio_cmd_queue(cmd, info, infosize);
	if ( info )
		g_free(info);
	return(res);
}
