/* Xmmsfind - small search plugin for the X Multimedia System 
 * 
 * Consists of two parts:
 * 1 - The actual plugin, starts with xmms and gets the playlist from xmms
 *     and stores it in memory.
 * 2 - The remote plugin that is launched from the prompt or keyboard shortcut
 *     and simply tells the plugin to open the searchbox
 *
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public Licensse as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This program is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.  You
 * should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
#include <stdio.h>
#include <limits.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <xmms/plugin.h>
#include <xmms/util.h>
#include <errno.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <glib.h>
#include <pthread.h>

#include "config.h"
#include "globals.h"
#include "xmmsfind_plugin.h"

#define CMD_BUF 1024

static void *thread_read_fifo (void *args);
gchar fifo_path[_POSIX_PATH_MAX];
pthread_t fifo_thread;
gint session;
GtkWidget *xmmsfind_main_window;
_pl_struct *xmmsfind_playlist;


GeneralPlugin xmmsfind = 
{
      NULL,
      NULL,
      -1,                 /* session: filled in by xmms */
      "Xmmsfind " VERSION,
      xmmsfind_init,
      xmmsfind_about,
      xmmsfind_config,
      xmmsfind_cleanup,
};

static void xmmsfind_cleanup ()
{
    xmmsfind_do_free_playlist ();
    pthread_cancel (fifo_thread);
    if (remove (fifo_path) != 0)
      fprintf (stderr, "Xmmsfind: (cleanup) Unable to remove fifo '%s'", fifo_path);
}

GeneralPlugin *get_gplugin_info ()
{
    return (&xmmsfind);
}

/* Creates the fifo in ~/.xmms/ 
 * returns nonzero on success */
int create_fifo ()
{
    snprintf (fifo_path, _POSIX_PATH_MAX, "%s/.xmms/xmmsfind_fifo-%d", g_get_home_dir(), session);
    
    while (mkfifo (fifo_path, 0666) == -1)
    {
	if (errno == EEXIST)
	{
	    if (remove (fifo_path) != 0)
	    {
		fprintf (stderr, "Xmmsfind: (create_fifo) unable to remove old fifo '%s'\n", fifo_path);
		return 0;
	    }
	}
	else
	{
	    fprintf (stderr, "Xmmsfind: (create_fifo) Unable to create fifo '%s'\n", fifo_path);
	    return 0;
	}
    }
    return 1;
}

/* Called from xmms when the plugin is enabled */
static void xmmsfind_init ()
{
    pthread_attr_t attr;
    
    session = xmmsfind.xmms_session;
    xmmsfind_main_window = NULL;
    xmmsfind_playlist = NULL;

    pthread_attr_init (&attr);
    pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_DETACHED); 
    pthread_attr_setinheritsched (&attr, PTHREAD_EXPLICIT_SCHED);
    pthread_attr_setschedpolicy (&attr, SCHED_OTHER);
    pthread_attr_setscope (&attr, PTHREAD_SCOPE_SYSTEM); 
    if (create_fifo ())
    {
	pthread_create (&fifo_thread, &attr, thread_read_fifo, NULL);
    }
}

/* The main thread. Reads from ~/.xmms/xmmsfind_fifo and waits for 
 * the searchcommand.
 * Then it launches the main search window
 */
static void *thread_read_fifo (void *args)
{
    FILE *fifo;
    gchar cmd[CMD_BUF];
    
    if ((fifo = fopen (fifo_path, "r")) == NULL)
    {
	fprintf (stderr, "Xmmsfind: (thread_read_fifo) Unable to open fifo for reading\n");
	pthread_exit (NULL);
    }
    pthread_testcancel ();
    while (fgets (cmd, CMD_BUF, fifo))
    {
	pthread_testcancel ();
	if (!strncasecmp (cmd, "search", 6))
	{
	    GDK_THREADS_ENTER ();
            #ifdef DEBUG
	      puts ("plugin.c: (thread_read_fifo) Got searchcmd!\n");
            #endif
	    create_xmmsfind_main_window ();
            #ifdef DEBUG
	      puts ("plugin.c: (thread_read_fifo) Searchbox showed!\n");
            #endif
	    GDK_THREADS_LEAVE ();
	}
	else
	  fprintf (stderr, "Xmmsfind: (thread_read_fifo) Unknown command from xmmsfind-remote '%s'\n", cmd);

	/* We have to reopen the fifo since it reached EOF earlier */
	fclose (fifo);                                    
	if ((fifo = fopen (fifo_path, "r")) == NULL)
	{
	    fprintf (stderr, "Xmmsfind: (thread_read_fifo) Unable to reopen fifo for reading '%s'\n", fifo_path);
	    
	    return NULL;
	}
	pthread_testcancel ();
    }
    return NULL;
}

void xmmsfind_about ()
{
    static GtkWidget *win_about = NULL;
    
    if (win_about)
      return;
    
    win_about = xmms_show_message ("About Xmmsfind",
				   "Xmmsfind version " VERSION "\n"
                   "Copyright (C) 2004, Michael Marineau (marineam@engr.orst.edu),\n"
				   "Copyright (C) 2001-2002, Isak Savo (iso01001@student.mdh.se).\n"
				   "Greetings to Jimmy Mkel (jimmy@producks.net) for help with this plugin!\n\n"
				   "Distributed under the GNU GPL\n"
                   "For news and updates, visit http://xmmsfind.sourceforge.net\n",
				   "Got it!", FALSE, NULL, NULL);
    gtk_signal_connect (GTK_OBJECT (win_about), "destroy",
			GTK_SIGNAL_FUNC (gtk_widget_destroyed), &win_about);
    
    gtk_widget_show (win_about);
}
