/* ----------------------------------------------------------------------
 *  A plugin that loads K-Jofol's skins and functions as a remote for XMMS.
 *
 *  There are some very cool skins comming out for K-Jofol.  They look
 *  heaps better than the Winamp ones.  This can be compiled as a stand
 *  alone program which functions as a remote control.  Just run XMMS,
 *  then run this to control XMMS.  It will also compile as a visualisation
 *  plugin.
 *
 *  This is still under development, although it is mostly functional.
 *
 *  Written by Tim Ferguson, 1999.
 *  (timf@csse.monash.edu.au, http://www.csse.monash.edu.au/~timf/)
 * ---------------------------------------------------------------------- */
#include "kj.h"

#if RXMMS
gint xmms_running = 0, xmms_session = 0;
#endif

#define VID_DUMP	0

static void kj_cleanup(void);
static void kj_init(void);
void kj_set_vismode(int mode);

#if XMMS_VIS
#include "xmms/plugin.h"

static void kj_sanalyzer_render_freq(gint16 data[2][256]);
static void kj_sanalyzer_render_pcm(gint16 data[2][512]);
static void kj_xmms_configure(void);

VisPlugin kjint_vp =
{
	NULL,
	NULL,
	0, /* XMMS Session ID, filled in by XMMS */
	"K-Jofol Interface",		/* description */
	0, /* Number of PCM channels wanted */
	0, /* Number of freq channels wanted */
	kj_init, /* init */
	kj_cleanup, /* cleanup */
	kj_about, /* about */
	kj_xmms_configure, /* configure */
	NULL, /* disable_plugin */
	NULL, /* playback_start */
	NULL, /* playback_stop */
	kj_sanalyzer_render_pcm, /* render_pcm */
	kj_sanalyzer_render_freq  /* render_freq */
};

VisPlugin *get_vplugin_info(void)
{
	return &kjint_vp;
}

static gint refresh_count = 0, refresh_max[] = {1, 2, 4, 8};
static gfloat peak_falloff[] = {1.2, 1.3, 1.4, 1.5, 1.6};
static gfloat freq_falloff[] = {0.34, 0.5, 1.0, 1.3, 1.6};

#	if XMMS_VISIMPORT_HACK
#include "xmms_visimport.h"
#	endif

#endif

GdkWindow *root_window;

static GtkWidget *win_main = NULL;
static GdkPixmap *bg_main = NULL;
static GdkGC *gc_main;
static gint timeout_handle;
static gint main_move, mbutton_x, mbutton_y, cur_volume = 0, cur_pitch = 100,
	cur_seek, equalizer_state = 1;
gint cur_rate, cur_freq, cur_time, cur_track = -1, cur_track_len;
static gchar *cur_fname = NULL;
static gint freq_xscale[300];
static gfloat freq_data[256];
static gfloat peak_data[256], peak_speed[256];
static gfloat vu_data[2];

#if 0
static gint equalizer_pos[32] = {
	0,40,80,120,160,200,255,200,160,120,80,40,0,128,128,128,128,128,
	128,128,128,128,128,128,128,128,128,128,128,128,128,128
};
#else
static gint equalizer_pos[32] = {
	128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,
	128,128,128,128,128,128,128,128,128,128,128,128,128,128,128,128
};
#endif

gint kj_running = 0;

k_config config = {
	NULL,
	1, 0, 0, 1, 0, 0, 0, 0
};

k_resource res;

#if VID_DUMP
FILE *img_fp = NULL;
#endif

/* ---------------------------------------------------------------------- */
static GtkAccelGroup *accel_main;
static GtkItemFactory *ifact_main, *ifact_analyser;
void popup_menu_cb(gpointer cb_data, guint action, GtkWidget * w);
static int process_button(int x, int y, int b, int m);

#define NUM_MAIN_POPUP		17
GtkItemFactoryEntry main_popup_items[] = {
	{ "/Open file...", "L", popup_menu_cb, 3, "<Item>" },
	{ "/-", NULL, NULL, 0, "<Separator>" },
	{ "/Equalizer...", "<control>G", popup_menu_cb, 4, "<Item>" },
	{ "/Playlist Editor...", "P", popup_menu_cb, 5, "<Item>" },
	{ "/Toggle XMMS window", "M", popup_menu_cb, 60, "<Item>" },
	{ "/-", NULL, NULL, 0, "<Separator>" },
	{ "/Playing/Previous", "Z", popup_menu_cb, 10, "<Item>" },
	{ "/Playing/Play", "X", popup_menu_cb, 11, "<Item>" },
	{ "/Playing/Pause", "C", popup_menu_cb, 12, "<Item>" },
	{ "/Playing/Stop", "V", popup_menu_cb, 13, "<Item>" },
	{ "/Playing/Next", "B", popup_menu_cb, 14, "<Item>" },
	{ "/Options/Preferences...", "<control>P", popup_menu_cb, 20, "<Item>" },
	{ "/Options/Dock Mode", NULL, popup_menu_cb, 21, "<Item>" },
	{ "/Options/Winshade Mode", NULL, popup_menu_cb, 22, "<Item>" },
	{ "/Visualisation", NULL, NULL, 0, "<Item>" },
	{ "/-", NULL, NULL, 0, "<Separator>" },
	{ "/About...", "<control>A", popup_menu_cb, 0, "<Item>" },
	{ "/Quit", NULL, popup_menu_cb, 2, "<Item>" }
};

#define NUM_ANALYSER_POPUP		35
GtkItemFactoryEntry analyser_popup_items[] = {
	{ "/Visualisation Mode", NULL, NULL, 0, "<Branch>" },
	{ "/Visualisation Mode/Analyser", NULL, popup_menu_cb, 30, "<RadioItem>" },
	{ "/Visualisation Mode/Scope", NULL, popup_menu_cb, 31, "/Visualisation Mode/Analyser" },
	{ "/Visualisation Mode/VU", NULL, popup_menu_cb, 32, "/Visualisation Mode/Analyser" },
	{ "/Visualisation Mode/Off", NULL, popup_menu_cb, 33, "/Visualisation Mode/Analyser" },
	{ "/Analyser Mode", NULL, NULL, 0, "<Branch>" },
	{ "/Analyser Mode/Normal", NULL, popup_menu_cb, 34, "<RadioItem>" },
	{ "/Analyser Mode/Fire", NULL, popup_menu_cb, 35, "/Analyser Mode/Normal" },
	{ "/Analyser Mode/Vertical Lines", NULL, popup_menu_cb, 36, "/Analyser Mode/Normal" },
	{ "/Analyser Mode/-", NULL, NULL, 0, "<Separator>" },
	{ "/Analyser Mode/Lines", NULL, popup_menu_cb, 37, "<RadioItem>" },
	{ "/Analyser Mode/Bars", NULL, popup_menu_cb, 38, "/Analyser Mode/Lines" },
	{ "/Analyser Mode/-", NULL, NULL, 0, "<Separator>" },
	{ "/Analyser Mode/Peaks", NULL, popup_menu_cb, 39, "<ToggleItem>" },
	{ "/Scope Mode", NULL, NULL, 0, "<Branch>" },
	{ "/Scope Mode/Dot Scope", NULL, popup_menu_cb, 40, "<RadioItem>" },
	{ "/Scope Mode/Line Scope", NULL, popup_menu_cb, 41, "/Scope Mode/Dot Scope" },
	{ "/Scope Mode/Solid Scope", NULL, popup_menu_cb, 42, "/Scope Mode/Dot Scope" },
	{ "/Refresh Rate", NULL, NULL, 0, "<Branch>" },
	{ "/Refresh Rate/Full", NULL, popup_menu_cb, 45, "<RadioItem>" },
	{ "/Refresh Rate/Half", NULL, popup_menu_cb, 46, "/Refresh Rate/Full" },
	{ "/Refresh Rate/Quarter", NULL, popup_menu_cb, 47, "/Refresh Rate/Full" },
	{ "/Refresh Rate/Eigth", NULL, popup_menu_cb, 48, "/Refresh Rate/Full" },
	{ "/Analyser Falloff", NULL, NULL, 0, "<Branch>" },
	{ "/Analyser Falloff/Slowest", NULL, popup_menu_cb, 50, "<RadioItem>" },
	{ "/Analyser Falloff/Slow", NULL, popup_menu_cb, 51, "/Analyser Falloff/Slowest" },
	{ "/Analyser Falloff/Medium", NULL, popup_menu_cb, 52, "/Analyser Falloff/Slowest" },
	{ "/Analyser Falloff/Fast", NULL, popup_menu_cb, 53, "/Analyser Falloff/Slowest" },
	{ "/Analyser Falloff/Fastest", NULL, popup_menu_cb, 54, "/Analyser Falloff/Slowest" },
	{ "/Peaks Falloff", NULL, NULL, 0, "<Branch>" },
	{ "/Peaks Falloff/Slowest", NULL, popup_menu_cb, 55, "<RadioItem>" },
	{ "/Peaks Falloff/Slow", NULL, popup_menu_cb, 56, "/Peaks Falloff/Slowest" },
	{ "/Peaks Falloff/Medium", NULL, popup_menu_cb, 57, "/Peaks Falloff/Slowest" },
	{ "/Peaks Falloff/Fast", NULL, popup_menu_cb, 58, "/Peaks Falloff/Slowest" },
	{ "/Peaks Falloff/Fastest", NULL, popup_menu_cb, 59, "/Peaks Falloff/Slowest" }
};


/* ---------------------------------------------------------------------- */
void kj_set_analyser_menu(void)
{
int i;

	for(i = 0; i < 3; i++)
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[6 + i].path))->active = (config.analyser_type == i) ? TRUE : FALSE;

	for(i = 0; i < 2; i++)
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[10 + i].path))->active = (config.analyser_mode == i) ? TRUE : FALSE;

	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[13].path))->active = config.analyser_peaks;

	for(i = 0; i < 3; i++)
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[15 + i].path))->active = (config.scope_mode == i) ? TRUE : FALSE;

	for(i = 0; i < 4; i++)
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[19 + i].path))->active = (config.refresh_rate == i) ? TRUE : FALSE;

	for(i = 0; i < 5; i++)
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[24 + i].path))->active = (config.freq_falloff == i) ? TRUE : FALSE;

	for(i = 0; i < 5; i++)
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[30 + i].path))->active = (config.peak_falloff == i) ? TRUE : FALSE;
}


/* ---------------------------------------------------------------------- */
void popup_menu_cb(gpointer cb_data, guint action, GtkWidget * w)
{
	switch(action)
		{
		case 0: kj_about(); break;
		case 2:
			if(config.quit_xmms_exit)
				{
#if RXMMS
				if(xmms_running) xmms_remote_quit(xmms_session);
#endif
				}
			else
				{
#if XMMS_VIS
				kjint_vp.disable_plugin(&kjint_vp);
#else
				kj_cleanup();
#endif
				}
			break;
		case 3: process_button(0, 0, BUTTON_OPEN, 0); break;
		case 4: process_button(0, 0, BUTTON_EQUALIZER, 0); break;
		case 5: kj_playlist(); break;

		case 10: process_button(0, 0, BUTTON_PREVIOUSSONG, 0); break;
		case 11: process_button(0, 0, BUTTON_PLAY, 0); break;
		case 12: process_button(0, 0, BUTTON_PAUSE, 0); break;
		case 13: process_button(0, 0, BUTTON_STOP, 0); break;
		case 14: process_button(0, 0, BUTTON_NEXTSONG, 0); break;

		case 20: kj_configure(); break;
		case 21: process_button(0, 0, BUTTON_DOCKMODE, 0); break;
		case 22:
			if(res.wsh_rc_fname)
				{
				gchar *wsh;
				wsh = g_strdup(res.wsh_rc_fname);
				load_resource(NULL, wsh, &res);
				g_free(wsh);
				kj_set_resource();
				}
			break;
		case 30: case 31: case 32:
		case 33: kj_set_vismode(action - 30); break;
		case 34: case 35:
		case 36: config.analyser_type = action - 34; break;
		case 37:
		case 38: config.analyser_mode = action - 37; break;
		case 39: config.analyser_peaks = GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[13].path))->active; break;
		case 40: case 41:
		case 42: config.scope_mode = action - 40; break;
		case 45: case 46: case 47:
		case 48: config.refresh_rate = action - 45; break;
		case 50: case 51: case 52: case 53:
		case 54: config.freq_falloff = action - 50; break;
		case 55: case 56: case 57: case 58:
		case 59: config.peak_falloff = action - 55; break;
		case 60: xmms_remote_main_win_toggle(xmms_session, !xmms_remote_is_main_win(xmms_session)); break;
		}
}


#if XMMS_VIS
/* ---------------------------------------------------------------------- */
static void kj_xmms_configure(void)
{
	if(!kj_running) kj_load_config();
	kj_configure();
}
#endif


/* ---------------------------------------------------------------------- */
int inside_area(int x, int y, k_area *a)
{
	return((a->type != A_NONE && x > a->x1 && x < a->x2 &&
		y > a->y1 && y < a->y2) ? 1 : 0);
}


/* ----------------------------------------------------------------------
 *  An attempt to generate a frequency scale that is independant of
 *  analyser width.  Width will be truncated if > 255.
 *      ... MUST FIX THIS ...
 * ---------------------------------------------------------------------- */
static void generate_freq_xscale(gint width)
{
gint i, j, s;
gfloat x;

	if(width > 255) width = 255;

	j = 0;
	x = log(width);
	for(i = 0; i < width; i++)
		{
		if((s = (255 - j)/(x * (width - i))) < 1) s = 1;
		j += s;
		freq_xscale[i] = j;
		}

	freq_xscale[0] = 0;
	freq_xscale[i] = 255;
}


#if XMMS_VIS
/* ---------------------------------------------------------------------- */
static void kj_sanalyzer_render_freq(gint16 data[2][256])
{
k_area *pos = res.areas + REGION_ANALYZER;
int i, d, c, w, h, p;
gfloat scale;
GdkGC *gc;

	if(pos->type == A_NONE) return;

	refresh_count = (refresh_count + 1) % refresh_max[config.refresh_rate];
	if(refresh_count != 0) return;

	gdk_draw_pixmap(bg_main, gc_main, res.background->pix, pos->x1, pos->y1, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);

	gc = gdk_gc_new(bg_main);
	gdk_gc_set_foreground(gc, &res.analyser_colour);
	if((w = pos->x2 - pos->x1) > 255) w = 255;
	h = pos->y2 - pos->y1 - 1;
	scale = ((gfloat)h)/log(256);

	for(i = 0; i < w; i++)
		{
		for(c = freq_xscale[i], d = 0; c < freq_xscale[i + 1]; c++)
			if(data[0][c] > d) d = data[0][c];

		d = (gint)(log(d >> 6) * scale);
		if(d > h-1) d = h;
		if(d > freq_data[i]) freq_data[i] = d;
		else if(freq_data[i] > 0.5) freq_data[i] -= (freq_falloff[config.freq_falloff] * refresh_max[config.refresh_rate]);
		else freq_data[i] = 0;
		p = (config.analyser_mode ? i & ~3 : i);

		if(freq_data[p] > 0)
			{
			switch(config.analyser_type)
				{
				case 2:
					gdk_gc_set_foreground(gc, res.vis_colour + 1 + (int)((freq_data[p] * 24)/(h-1)));
				case 0:
					gdk_draw_line(bg_main, gc, pos->x1 + i, pos->y2-1, pos->x1 + i, pos->y2-1-freq_data[p]);
					break;
				case 1:
					for(c = 0; c <= (int)freq_data[p]; c++)
						{
						gdk_gc_set_foreground(gc, res.vis_colour + 24 -
							(int)(((freq_data[p] - c) * 24)/(h-1)));
						gdk_draw_point(bg_main, gc, pos->x1 + i, pos->y2-1-c);
						}
					break;
				}
			}
		}

	if(config.analyser_peaks)
		{
		gdk_gc_set_foreground(gc, &res.vis_colour[0]);

		for(i = 0; i < w; i++)
			{
			if(freq_data[i] > peak_data[i]) { peak_data[i] = freq_data[i]; peak_speed[i] = 0.01; }
			else if(peak_data[i] > 0)
				{
				peak_data[i] -= (peak_speed[i] * refresh_max[config.refresh_rate]);
				peak_speed[i] *= peak_falloff[config.peak_falloff];
				if(peak_data[i] < 0) peak_data[i] = 0;
				}

			p = (config.analyser_mode ? i & ~3 : i);

			if(peak_data[p] > 0)
				gdk_draw_point(bg_main, gc, pos->x1 + i, pos->y2-1-peak_data[p]);
			}
		}

	gdk_gc_destroy(gc);
	gdk_window_clear_area(win_main->window, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
}


/* ---------------------------------------------------------------------- */
static void kj_sanalyzer_render_pcm(gint16 data[2][512])
{
k_area *pos = res.areas + REGION_ANALYZER;
int i, j, w, h, y, old_y;
float scale;
GdkGC *gc;

	if(pos->type == A_NONE) return;

	refresh_count = (refresh_count + 1) % refresh_max[config.refresh_rate];
	if(refresh_count != 0) return;

	gdk_draw_pixmap(bg_main, gc_main, res.background->pix, pos->x1, pos->y1, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
	gc = gdk_gc_new(bg_main);
	gdk_gc_set_foreground(gc, &res.analyser_colour);

	if(config.vis_mode == 1)
		{
		if((w = pos->x2 - pos->x1) > 512) w = 512;
		h = pos->y2 - pos->y1 - 1;
		scale = (float)h/65535;

		old_y = (h/2) + (data[0][0] * scale);
		old_y = (old_y < 0 ? 0 : (old_y >= h ? h - 1 : old_y));

		for(i = 0; i < w; i++)
			{
			y = (h/2) + (data[0][(i * 512)/w] * scale);
			y = (y < 0 ? 0 : (y >= h ? h - 1 : y));
			switch(config.scope_mode)
				{
				case 0: gdk_draw_point(bg_main, gc, pos->x1 + i, pos->y1 + y); break;
				case 1: gdk_draw_line(bg_main, gc, pos->x1 + i, pos->y1 + y, pos->x1 + i, pos->y1 + old_y); break;
				case 2: gdk_draw_line(bg_main, gc, pos->x1 + i, pos->y1 + (h/2), pos->x1 + i, pos->y1 + y); break;
				}
			old_y = y;
			}
		}
	else
		{
		w = pos->x2 - pos->x1;
		h = (pos->y2 - pos->y1 - 4)/2;

		for(j = 0; j < 2; j++)
			{
			y = 0;
			for(i = 0; i < 512; i++) if(abs(data[j][i]) > y) y = abs(data[j][i]);
			y = (y * w)/32768;
#if 0
			if(y > vu_data[j]) vu_data[j] = y;
			else if(vu_data[j] > 0) vu_data[j] -= (freq_falloff[config.freq_falloff] * refresh_max[config.refresh_rate]);
			else vu_data[i] = 0;
#else
			vu_data[j] = y;
#endif
			}
		gdk_draw_rectangle(bg_main, gc, 1, pos->x1, pos->y1+1, vu_data[0], h);
		gdk_draw_rectangle(bg_main, gc, 1, pos->x1, pos->y2 - h, vu_data[1], h);
		}

	gdk_gc_destroy(gc);
	gdk_window_clear_area(win_main->window, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
}
#endif


/* ---------------------------------------------------------------------- */
void kj_set_vismode(int mode)
{
int i;

	config.vis_mode = mode;

	for(i = 0; i < 4; i++)
		GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(ifact_analyser, analyser_popup_items[1 + i].path))->active = (mode == i) ? TRUE : FALSE;

#if XMMS_VIS
	switch(mode)
		{
		case 0: kjint_vp.num_freq_chs_wanted = 1; kjint_vp.num_pcm_chs_wanted = 0; break;
		case 1: kjint_vp.num_freq_chs_wanted = 0; kjint_vp.num_pcm_chs_wanted = 1; break;
		case 2: kjint_vp.num_freq_chs_wanted = 0; kjint_vp.num_pcm_chs_wanted = 2; break;
		case 3: kjint_vp.num_freq_chs_wanted = 0; kjint_vp.num_pcm_chs_wanted = 0; break;
		}

	{
	k_area *pos = res.areas + REGION_ANALYZER;
	if(mode == 3 && pos->type != A_NONE)
		{
		gdk_draw_pixmap(bg_main, gc_main, res.background->pix, pos->x1, pos->y1, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
		gdk_window_clear_area(win_main->window, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
		}
	}
#endif
}


/* ---------------------------------------------------------------------- */
static void draw_button(GdkWindow *w, k_area *button)
{
	if(button->type != A_BUTTON && button->type != A_TOGGLE) return;
	if((button->pressed || button->active) && res.pressed[button->bm])
		gdk_draw_pixmap(w, gc_main, res.pressed[button->bm]->pix, button->x1, button->y1, 
			button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);
	else
		gdk_draw_pixmap(w, gc_main, res.background->pix, button->x1, button->y1, 
			button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);

	gdk_window_clear_area(win_main->window, button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);
}


/* ---------------------------------------------------------------------- */
static void draw_equalizer(GdkWindow *w, k_area *pos)
{
gint i, xp, band_x;

	if(pos->type == A_NONE || !res.equalizer) return;

	xp = pos->x1;
	for(i = 0; i < res.equ_num_bands; i++)
		{
		band_x = res.equ_band_width * ((equalizer_pos[i] * res.equ_bmp_bands)/256);
		gdk_draw_pixmap(w, gc_main, res.equalizer->pix, band_x, 0, 
			xp, pos->y1, res.equ_band_width, res.equalizer->height);
		xp += res.equ_band_space;
		}

	gdk_window_clear_area(win_main->window, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
}


/* ---------------------------------------------------------------------- */
static void draw_seek(GdkWindow *wd, k_area *pos, k_area *bar, gint seek)
{
static int old_seek = -1;
int x, y;
gulong val;
GdkImage *img_src;
GdkColor c;

	if(cur_seek == old_seek) return;

		/* Ugly hack to do the position slider masking */
	if((pos->type != A_NONE) && res.seek && res.pressed[pos->bm])
		{
		old_seek = cur_seek;

		for(y = pos->y1; y < pos->y2; y++)
			for(x = pos->x1; x < pos->x2; x++)
				{
				val = kj_get_pixel(res.seek, x, y);
				if(((val & 0xff) == ((val >> 8) & 0xff)) && ((val & 0xff) == ((val >> 16) & 0xff)))
					{
					if(cur_seek > (val & 0xff)) img_src = res.seek_sel;
					else img_src = res.seek_unsel;
					c.pixel = gdk_image_get_pixel(img_src, x - pos->x1, y - pos->y1);
					gdk_gc_set_foreground(gc_main, &c);
					gdk_draw_point(wd, gc_main, x, y);
					}
				}

		gdk_window_clear_area(win_main->window, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
		}

	if(bar->type != A_NONE)
		{
		x = ((bar->x2 - bar->x1) * cur_seek)/256;
		gdk_draw_pixmap(wd, gc_main, res.pressed[bar->bm]->pix, bar->x1, bar->y1, 
			bar->x1, bar->y1, x, bar->y2 - bar->y1);
		gdk_draw_pixmap(wd, gc_main, res.background->pix, bar->x1 + x, bar->y1, 
			bar->x1 + x, bar->y1, (bar->x2 - bar->x1) - x, bar->y2 - bar->y1);
		gdk_window_clear_area(win_main->window, bar->x1, bar->y1, bar->x2 - bar->x1, bar->y2 - bar->y1);
		}
}


/* ---------------------------------------------------------------------- */
void kj_draw_image(GdkWindow *wd, k_image *img, int sx, int sy, int dx, int dy, int w, int h)
{
GdkGC *gc;

	gc = gdk_gc_new(wd);
	if(img->mask)
		{
		gdk_gc_set_clip_mask(gc, img->mask);
		gdk_gc_set_clip_origin(gc, dx-sx, dy-sy);
		}

	gdk_draw_pixmap(wd, gc, img->pix, sx, sy, dx, dy, w, h);
	gdk_gc_destroy(gc);
}


/* ---------------------------------------------------------------------- */
static void draw_font(GdkWindow *wd, k_font *font, k_area *pos, char *str)
{
int dst_x, x, y, c, w, h;

	if(!font->img || pos->type == A_NONE) return;

	w = font->size[0];
	h = font->size[1];
	dst_x = pos->x1;

	if(font->transparent)
		gdk_draw_pixmap(wd, gc_main, res.background->pix, pos->x1, pos->y1, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);

	while(*str != '\0')
		{
		x = y = -1;
		c = toupper(*str);
		if(dst_x + w > pos->x2) break;

		if(font->type == 1)
			{
			if(c >= 'A' && c <= 'Z') { x = (c - 'A'); y = 0; }
			else if(c >= '0' && c <= '9') { x = (c - '0'); y = 1; }
			else
				{
				switch(c)
					{
					case '"': x = 26; y = 0; break;
					case ':': x = 12; y = 1; break;
					case '(': x = 13; y = 1; break;
					case ')': x = 14; y = 1; break;
					case '-': x = 15; y = 1; break;
					case '`':
					case '\'': x = 16; y = 1; break;
					case '!': x = 17; y = 1; break;
					case '_': x = 18; y = 1; break;
					case '+': x = 19; y = 1; break;
					case '\\': x = 20; y = 1; break;
					case '/': x = 21; y = 1; break;
					case '[': x = 22; y = 1; break;
					case ']': x = 23; y = 1; break;
					case '^': x = 24; y = 1; break;
					case '&': x = 25; y = 1; break;
					case '%': x = 26; y = 1; break;
					case '.':
					case ',': x = 27; y = 1; break;
					case '=': x = 28; y = 1; break;
					case '$': x = 29; y = 1; break;
					case '#': x = 30; y = 1; break;
					case '':
					case '': x = 0; y = 2; break;
					case '':
					case '': x = 1; y = 2; break;
					case '':
					case '': x = 2; y = 2; break;
					case '':
					case '': x = 20; y = 0; break;
					case '?': x = 3; y = 2; break;
					case '*': x = 4; y = 2; break;
					default: x = 30; y = 0; break;
					}
				}
			}
		else
			{
			y = 0;
			if(*str >= '0' && *str <= '9') x = (*str - '0');
			else if(*str == ':' || *str == '%') x = 10;
			else x = 11;
			}

		if(x < 0 || y < 0) break;
		if(font->transparent && font->img->mask)
			kj_draw_image(wd, font->img, x * w, y * h, dst_x, pos->y1, w, h);
		else gdk_draw_pixmap(wd, gc_main, font->img->pix, x * w, y * h, dst_x, pos->y1, w, h);

		str++;
		dst_x += (w + font->spacing);
		}

	if(!font->transparent && dst_x < pos->x2)
		gdk_draw_pixmap(wd, gc_main, res.background->pix, dst_x, pos->y1, dst_x, pos->y1, pos->x2 - dst_x, pos->y2 - pos->y1);

	gdk_window_clear_area(win_main->window, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
}


/* ---------------------------------------------------------------------- */
static void clear_text_box(GdkWindow *wd, k_area *pos)
{
	gdk_draw_pixmap(wd, gc_main, res.background->pix, pos->x1, pos->y1, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
	gdk_window_clear_area(win_main->window, pos->x1, pos->y1, pos->x2 - pos->x1, pos->y2 - pos->y1);
}


/* ---------------------------------------------------------------------- */
static void draw_text(GdkWindow *w, gboolean force)
{
char buf[100];
gint sec, min;
static int fname_dir = 0, fname_pos = 0;
static gint old_rate = -1, old_freq = -1, old_track = -1, old_min = -1,
	old_sec = -1;

	if(force || cur_rate != old_rate)
		{
		if(cur_rate < 0) clear_text_box(w, res.areas + REGION_MP3KBPS);
		else
			{
			sprintf(buf, "%d", cur_rate/1000);
			draw_font(w, &(res.Font), res.areas + REGION_MP3KBPS, buf);
			}
		old_rate = cur_rate;
		}

	if(force || cur_freq != old_freq)
		{
		if(cur_freq < 0) clear_text_box(w, res.areas + REGION_MP3KHZ);
		else
			{
			sprintf(buf, "%d", cur_freq/1000);
			draw_font(w, &(res.Font), res.areas + REGION_MP3KHZ, buf);
			}
		old_freq = cur_freq;
		}

	if(force || cur_track != old_track)
		{
		fname_pos = fname_dir = 0;
		if(cur_track < 0) clear_text_box(w, res.areas + REGION_TRACK);
		else
			{
			sprintf(buf, "%d", cur_track);
			draw_font(w, &(res.Font), res.areas + REGION_TRACK, buf);
			}
		old_track = cur_track;
		}

	min = cur_time/60000;
	sec = (cur_time/1000) % 60;
	if(force || old_sec != sec || old_min != min)
		{
		if(cur_time < 0) clear_text_box(w, res.areas + REGION_MP3TIME);
		else
			{
			sprintf(buf, "%02d:%02d", min, sec);
			draw_font(w, &(res.TimeFont), res.areas + REGION_MP3TIME, buf);
			}
		old_sec = sec; old_min = min;
		}

	if(res.Font.img && res.areas[REGION_FNAME].type != A_NONE)
		{
		if(cur_fname == NULL) clear_text_box(w, res.areas + REGION_FNAME);
		else
			{
			k_area *pos = res.areas + REGION_FNAME;

			if(strlen(cur_fname) < fname_pos) fname_pos = 0;
			draw_font(w, &(res.Font), pos, cur_fname + fname_pos);
			if(fname_dir)
				{
				if(fname_pos > 0) fname_pos--;
				else fname_dir = 0;
				}
			else
				{
				if((strlen(cur_fname + fname_pos) * res.Font.size[0]) > (pos->x2 - pos->x1))
					fname_pos++;
				else fname_dir = 1;
				}
			}
		}
}


/* ---------------------------------------------------------------------- */
static void draw_volume(GdkWindow *w, k_area *button, gint vol)
{
int x;
char buf[10];

	if(res.volume)
		{
		x = ((vol * (res.VolumeControlImageNb-1))/255) * res.VolumeControlImageXSize;
		kj_draw_image(w, res.volume, x, 0, button->x1, button->y1,
			button->x2 - button->x1, button->y2 - button->y1);

		if(res.areas[REGION_VOLTEXT].type != A_NONE)
			{
			sprintf(buf, "%02d%%", vol * 99/255);
			draw_font(w, &(res.VolumeFont), res.areas + REGION_VOLTEXT, buf);
			}

		gdk_window_clear_area(win_main->window, button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);
		}
}


/* ---------------------------------------------------------------------- */
static void draw_pitch(GdkWindow *w, k_area *button, gint pitch)
{
int x;
char buf[10];

	if(res.pitch)
		{
		x = ((pitch * (res.PitchControlImageNb-1))/255) * res.PitchControlImageXSize;
		kj_draw_image(w, res.pitch, x, 0, button->x1, button->y1,
			button->x2 - button->x1, button->y2 - button->y1);

		if(res.areas[REGION_PITCHTEXT].type != A_NONE)
			{
			sprintf(buf, "%02d", pitch);
			draw_font(w, &(res.PitchFont), res.areas + REGION_PITCHTEXT, buf);
			}

		gdk_window_clear_area(win_main->window, button->x1, button->y1, button->x2 - button->x1, button->y2 - button->y1);
		}
}


/* ---------------------------------------------------------------------- */
static void set_equalizer_state(int on)
{
	equalizer_state = on;
	res.areas[BUTTON_EQUON].active = on;
	res.areas[BUTTON_EQUOFF].active = !on;
	draw_button(bg_main, res.areas + BUTTON_EQUON);
	draw_button(bg_main, res.areas + BUTTON_EQUOFF);
}


/* ---------------------------------------------------------------------- */
static void set_toggle_states(int shuffle, int repeat)
{
	if(shuffle >= 0 && shuffle != res.areas[BUTTON_SHUFFLE].active)
		{
		res.areas[BUTTON_SHUFFLE].active = shuffle;
		draw_button(bg_main, res.areas + BUTTON_SHUFFLE);
		}

	if(repeat >= 0 && repeat != res.areas[BUTTON_REPEAT].active)
		{
		res.areas[BUTTON_REPEAT].active = repeat;
		draw_button(bg_main, res.areas + BUTTON_REPEAT);
		}
}


/* ---------------------------------------------------------------------- */
static void update_qualizer(void)
{
int i, v, update = 0;

	for(i = 0; i < 10; i++)
		{
		v = 128 + (gint)(xmms_remote_get_eq_band(xmms_session, i) * 128.0/20.0);
		if(equalizer_pos[i] != v)
			{
			equalizer_pos[i] = v;
			update = 1;
			}
		}

	if(update) draw_equalizer(bg_main, res.areas + REGION_EQUALIZER);

#if XMMS_VISIMPORT_HACK
	if(equalizer_state != cfg.equalizer_active) set_equalizer_state(cfg.equalizer_active);
#endif
}


/* ---------------------------------------------------------------------- */
static int process_button(int x, int y, int b, int m)
{
gulong val;
int ret = 1, i;
Window xwindow;
static int equ_band = 0;

	// printf("Button: %d at %d,%d  (%d)\n", b, x, y, m);

	if(!m)
		{
#if RXMMS
		if(xmms_running)
			{
			if(b == BUTTON_PLAY) xmms_remote_play(xmms_session);
			else if(b == BUTTON_STOP) xmms_remote_stop(xmms_session);
			else if(b == BUTTON_PAUSE) xmms_remote_pause(xmms_session);
			else if(b == BUTTON_REWIND) xmms_remote_playlist_prev(xmms_session);
			else if(b == BUTTON_FASTFORWARD) xmms_remote_playlist_next(xmms_session);
			else if(b == BUTTON_PREVIOUSSONG) xmms_remote_playlist_prev(xmms_session);
			else if(b == BUTTON_NEXTSONG) xmms_remote_playlist_next(xmms_session);
			else if(b == BUTTON_OPEN) xmms_remote_eject(xmms_session);
			else if(b == BUTTON_EQUALIZER) xmms_remote_eq_win_toggle(xmms_session, 1);
			else if(b == REGION_SEEK || b == REGION_SEEKBAR) xmms_remote_jump_to_time(xmms_session, (cur_seek * cur_track_len)/256);
			}
#endif

		if(b == BUTTON_CLOSE)
			{
			if(config.quit_xmms_exit)
				{
#if RXMMS
				if(xmms_running) xmms_remote_quit(xmms_session);
#endif
				}
			else
				{
#if XMMS_VIS
				kjint_vp.disable_plugin(&kjint_vp);
#else
				kj_cleanup();
#endif
				}
			}
		else if(b == BUTTON_VOLUP || b == BUTTON_VOLDOWN)
			{
			cur_volume +=  (b == BUTTON_VOLUP ? 8 : -8);
			if(cur_volume < 0) cur_volume = 0;
			else if(cur_volume > 255) cur_volume = 255;
#if RXMMS
			if(xmms_running)
				xmms_remote_set_main_volume(xmms_session, (cur_volume * 100)/255);
#endif
			}
		else if(b == BUTTON_OPTIONS) kj_configure();
		else if(b == BUTTON_ABOUT) kj_about();
		else if(b == BUTTON_DOCKMODE)
			{
			if(res.dck_rc_fname)
				{
				gchar *dck;
				dck = g_strdup(res.dck_rc_fname);
				load_resource(NULL, dck, &res);
				g_free(dck);
				kj_set_resource();
				res.dock_mode = 1;
				}
			}
		else if(b == BUTTON_UNDOCK)
			{
			load_resource(NULL, NULL, &res);
			kj_set_resource();
			res.dock_mode = 0;
			gdk_window_move(win_main->window, config.win_x, config.win_y);
			}
		else if(b == BUTTON_MINIMIZE)
			{
			if(win_main->window)
				{
				xwindow = GDK_WINDOW_XWINDOW(win_main->window);
				XIconifyWindow(GDK_DISPLAY(), xwindow, DefaultScreen(GDK_DISPLAY()));
				}
			}
		else if(b == BUTTON_PLAYLIST) kj_playlist();
		else if(b == BUTTON_EQUON)
			{
			set_equalizer_state(1);
#if XMMS_VISIMPORT_HACK
			equalizerwin_on_pushed(1);
#endif
			}
		else if(b == BUTTON_SHUFFLE)
			{
#if RXMMS
			xmms_remote_toggle_shuffle(xmms_session);
#endif
			set_toggle_states(!res.areas[BUTTON_SHUFFLE].active, -1);
			}
		else if(b == BUTTON_REPEAT)
			{
#if RXMMS
			xmms_remote_toggle_repeat(xmms_session);
#endif
			set_toggle_states(-1, !res.areas[BUTTON_REPEAT].active);
			}
		else if(b == BUTTON_EQUOFF)
			{
			set_equalizer_state(0);
#if XMMS_VISIMPORT_HACK
			equalizerwin_on_pushed(0);
#endif
			}
		else if(b == BUTTON_EQURESET)
			{
			for(i = 0; i < 32; i++) equalizer_pos[i] = 128;
			draw_equalizer(bg_main, res.areas + REGION_EQUALIZER);
			for(i = 0; i < 10; i++) xmms_remote_set_eq_band(xmms_session, i, 0);
			}
		else if(b == REGION_ANALYZER)
			kj_set_vismode((config.vis_mode + 1) % 4);
		}

	if(b == BUTTON_VOLUME && res.volume_position)
		{
		val = kj_get_pixel(res.volume_position, x, y);
		if(((val & 0xff) == ((val >> 8) & 0xff)) && ((val & 0xff) == ((val >> 16) & 0xff)))
			{
			cur_volume = val & 0xff;
			draw_volume(bg_main, res.areas + BUTTON_VOLUME, cur_volume);
#if RXMMS
			if(xmms_running)
				xmms_remote_set_main_volume(xmms_session, (cur_volume * 100)/255);
#endif
			}
		else ret = 0;
		}

	if(b == BUTTON_PITCH && res.pitch_position)
		{
		val = kj_get_pixel(res.pitch_position, x, y);
		if(((val & 0xff) == ((val >> 8) & 0xff)) && ((val & 0xff) == ((val >> 16) & 0xff)))
			{
			cur_pitch = val & 0xff;
			draw_pitch(bg_main, res.areas + BUTTON_PITCH, cur_pitch);
			}
		else ret = 0;
		}

	if(b == REGION_SEEK && res.seek)
		{
		val = kj_get_pixel(res.seek, x, y);
		if(((val & 0xff) == ((val >> 8) & 0xff)) && ((val & 0xff) == ((val >> 16) & 0xff)))
			{
			cur_seek = val & 0xff;
			draw_seek(bg_main, res.areas + REGION_SEEK, res.areas + REGION_SEEKBAR, cur_seek);
			}
		else ret = 0;
		}

	if(b == REGION_EQUALIZER)
		{
		if(m == 1) equ_band = ((x - res.areas[REGION_EQUALIZER].x1)/res.equ_band_space);
		equalizer_pos[equ_band] = 255 - (((y - res.areas[REGION_EQUALIZER].y1) * 255)/
			(res.areas[REGION_EQUALIZER].y2 - res.areas[REGION_EQUALIZER].y1));
		draw_equalizer(bg_main, res.areas + REGION_EQUALIZER);
		/* if(m == 0) set XMMS equalizer value */
		if(equ_band < 10)
			xmms_remote_set_eq_band(xmms_session, equ_band, (equalizer_pos[equ_band] - 128) * 21.0/128.0);
		}

	if(b == REGION_SEEKBAR)
		{
		cur_seek = ((x - res.areas[REGION_SEEKBAR].x1) * 256)/
			(res.areas[REGION_SEEKBAR].x2 - res.areas[REGION_SEEKBAR].x1);
		draw_seek(bg_main, res.areas + REGION_SEEK, res.areas + REGION_SEEKBAR, cur_seek);
		}

	return ret;
}


/* ---------------------------------------------------------------------- */
static gint button_press_cb(GtkWidget * w, GdkEventButton * event, gpointer data)
{
int b, mflag = 1;
GdkEventButton *bevent = (GdkEventButton *)event;

	if(event->type != GDK_BUTTON_PRESS) return FALSE;

	if(event->button == 3)
		{
		if(res.areas[REGION_ANALYZER].type != A_NONE && inside_area(event->x, event->y, res.areas + REGION_ANALYZER))
			gtk_menu_popup(GTK_MENU(ifact_analyser->widget), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
		else
			gtk_menu_popup(GTK_MENU(ifact_main->widget), NULL, NULL, NULL, NULL, bevent->button, bevent->time);
		return TRUE;
		}

	if(event->button == 4 || event->button == 5)
		{
		cur_volume += (event->button == 4 ? 8 : -8);
		if(cur_volume < 0) cur_volume = 0;
		else if(cur_volume > 255) cur_volume = 255;
#if RXMMS
		if(xmms_running)
			xmms_remote_set_main_volume(xmms_session, (cur_volume * 100)/255);
#endif
		return TRUE;
		}

	if(event->button != 1) return FALSE;

	mbutton_x = event->x;
	mbutton_y = event->y;

	for(b = 0; b < NUM_MAIN_AREAS; b++)
		{
		if((res.areas[b].type != A_TEXT) && inside_area(mbutton_x, mbutton_y, res.areas + b))
			{
			if(process_button(mbutton_x, mbutton_y, b, 1))
				{
				mflag = 0;
				res.areas[b].focus = 1;
				res.areas[b].pressed = 1;
				draw_button(bg_main, res.areas + b);
				}
			}
		}

	main_move = mflag;
	return TRUE;
}


/* ---------------------------------------------------------------------- */
static void button_release_cb(GtkWidget * w, GdkEventButton * event, gpointer data)
{
int b;

	main_move = 0;

	if(event->button != 1) return;
	for(b = 0; b < NUM_MAIN_AREAS; b++)
		{
		if(res.areas[b].pressed)
			{
			if(res.areas[b].type == A_TOGGLE)
				res.areas[b].active = (res.areas[b].active ? 0 : 1);
			res.areas[b].pressed = 0;
			draw_button(bg_main, &res.areas[b]);
			process_button(event->x, event->y, b, 0);
			}
		res.areas[b].focus = 0;
		}
}


/* ---------------------------------------------------------------------- */
static void motion_notify_cb(GtkWidget *w, GdkEventMotion *event, gpointer data)
{
int b, inside;
gint nx, ny, dx, dy;
GdkModifierType modmask;

	if(main_move)
		{
		gdk_window_get_pointer(NULL, &nx, &ny, &modmask);
		nx -= mbutton_x;
		ny -= mbutton_y;
		dx = nx - config.win_x;
		dy = ny - config.win_y;
		kj_main_move(nx, ny);

		if(config.lock_plist)
			kj_playlist_move(config.plist_x + dx, config.plist_y + dy);
		}
	else
		{
		for(b = 0; b < NUM_MAIN_AREAS; b++)
			{
			if(res.areas[b].focus)
				{
				inside = inside_area(event->x, event->y, res.areas + b);
				if(inside) process_button(event->x, event->y, b, 2);
				if((inside && !res.areas[b].pressed) || (!inside && res.areas[b].pressed))
					{
					res.areas[b].pressed = inside;
					draw_button(bg_main, &res.areas[b]);
					}
				}
			}
		}
}


#if VID_DUMP
/* ---------------------------------------------------------------------- */
void grab_vid_frame(void)
{
GdkImage *scr;
guint32 v;
gint x, y, w, h;
static int frm = 0;

	if(img_fp == NULL)
		img_fp = fopen("/dosc/tmp/img.raw", "wb");

	gdk_window_get_size(win_main->window, &w, &h);
	scr = gdk_image_get(win_main->window, 0, 0, w, h);
	printf("%d-%dx%d\n", ++frm, w, h);

	for(y = 0; y < h; y++)
		for(x = 0; x < w; x++)
			{
			v = gdk_image_get_pixel(scr, x, y);
			fputc((v >> 16) & 0xff, img_fp);
			fputc((v >> 8) & 0xff, img_fp);
			fputc(v & 0xff, img_fp);
			}
}
#endif


/* ---------------------------------------------------------------------- */
static gint timeout_func_cb(gpointer data)
{
#if RXMMS
gint vl, vr;
static int old_volume = -1, first_xmms_run = 1;

	GDK_THREADS_ENTER();

		/* has to be here, else problems with rendering on small skins???? */
	gdk_window_set_back_pixmap(win_main->window, bg_main, 0);

	if(xmms_remote_is_running(xmms_session))
		{
		xmms_running = 1;
		if(first_xmms_run)
			{
			if(config.close_main_startup)
				xmms_remote_main_win_toggle(xmms_session, 0);
			first_xmms_run = 0;
			}

		xmms_remote_get_volume(xmms_session, &vl, &vr);
		cur_volume = ((vl > vr ? vl : vr) * 255)/100;
		if(old_volume != cur_volume) draw_volume(bg_main, res.areas + BUTTON_VOLUME, cur_volume);
		old_volume = cur_volume;

		cur_track = xmms_remote_get_playlist_pos(xmms_session);
		cur_track_len = xmms_remote_get_playlist_time(xmms_session, cur_track);
		cur_fname = xmms_remote_get_playlist_title(xmms_session, cur_track);
		set_toggle_states(xmms_remote_is_shuffle(xmms_session), xmms_remote_is_repeat(xmms_session));

		if(xmms_remote_is_paused(xmms_session))
			{
			if(!res.areas[REGION_SEEK].focus)
				{
				if(cur_seek == 0)
					{
					cur_time = xmms_remote_get_output_time(xmms_session);
					cur_seek = (cur_time * 255)/cur_track_len;
					}
				else cur_seek = 0;
				}
			draw_seek(bg_main, res.areas + REGION_SEEK, res.areas + REGION_SEEKBAR, cur_seek);
			}
		else if(xmms_remote_is_playing(xmms_session))
			{
			cur_time = xmms_remote_get_output_time(xmms_session);
			if(!res.areas[REGION_SEEK].focus && !res.areas[REGION_SEEKBAR].focus)
				cur_seek = (cur_time * 255)/cur_track_len;
			draw_seek(bg_main, res.areas + REGION_SEEK, res.areas + REGION_SEEKBAR, cur_seek);
			xmms_remote_get_info(xmms_session, &cur_rate, &cur_freq, &vl);
			}
		else
			{
			if(cur_seek > 0)
				{
				cur_seek = 0;
				draw_seek(bg_main, res.areas + REGION_SEEK, res.areas + REGION_SEEKBAR, cur_seek);
				}
			if(cur_rate > 0)
				{
				cur_rate = cur_freq = cur_time = -1;
				}
			}

		update_qualizer();
		}
	else xmms_running = 0;
#endif

	draw_text(bg_main, 0);

		/* Dock mode - KJ follows current focus window */
	if(res.dock_mode)
		{
		Window win, junkwin;
		XWindowAttributes win_attributes;
		int v, rx, ry;
		char *window_name;
		static int cur_x = -1, cur_y = -1;

		if(XGetInputFocus(GDK_DISPLAY(), &win, &v)
			&& XFetchName(GDK_DISPLAY(), win, &window_name)
			&& strcmp(window_name, "KJ"))
			{
			// printf("%s  ", window_name);
			if(XGetWindowAttributes(GDK_DISPLAY(), win, &win_attributes))
				{
				XTranslateCoordinates(GDK_DISPLAY(), win, win_attributes.root, 
					-win_attributes.border_width, -win_attributes.border_width,
					&rx, &ry, &junkwin);
				// printf("%d,%d  %dx%d  %d\n", rx, ry, win_attributes.width, win_attributes.height, win_attributes.border_width);

				rx += (res.DockPos[0]-7);
				ry += (res.DockPos[1]-24);
				if(cur_x != rx || cur_y != ry)
					{
					gdk_window_move(win_main->window, rx, ry);
					cur_x = rx;
					cur_y = ry;
					}
				}
			}
		}

	kj_update_playlist();

	// gdk_window_clear(win_main->window);
	gdk_flush();

#if VID_DUMP
		grab_vid_frame();
#endif

	GDK_THREADS_LEAVE();

	return 1;
}


/* ---------------------------------------------------------------------- */
static void kj_cleanup(void)
{
	gtk_timeout_remove(timeout_handle);
	kj_save_config();
	if(win_main) gtk_widget_destroy(win_main);
	kj_playlist_cleanup();
	kj_configure_cleanup();
	if(bg_main) gdk_pixmap_unref(bg_main);
	free_resource(&res);
	kj_running = 0;
	win_main = NULL;
	bg_main = NULL;
#if VID_DUMP
	fclose(img_fp);
#endif

#if !XMMS_VIS
	gtk_exit(0);
#endif
}


/* ---------------------------------------------------------------------- */
void kj_main_move(gint x, gint y)
{
	config.win_x = x;
	config.win_y = y;
	if(win_main) gdk_window_move(win_main->window, x, y);
}


/* ---------------------------------------------------------------------- */
void kj_set_resource(void)
{
gint b;

	if(!win_main) return;

	if(!res.background) { printf("No background image!!\n"); kj_cleanup(); }
	memset(freq_data, 0, 256 * sizeof(gfloat));
	memset(peak_data, 0, 256 * sizeof(gfloat));
	memset(peak_speed, 0, 256 * sizeof(gfloat));
	vu_data[0] = vu_data[1] = 0;

	if(res.areas[REGION_ANALYZER].type != A_NONE)
		generate_freq_xscale(res.areas[REGION_ANALYZER].x2 - res.areas[REGION_ANALYZER].x1);
 
	if(bg_main) gdk_pixmap_unref(bg_main);
	bg_main = gdk_pixmap_new(win_main->window, res.background->width, res.background->height, gdk_visual_get_best_depth());

	if(res.background->mask)
		gtk_widget_shape_combine_mask(win_main, res.background->mask, 0, 0);
	else printf("Warning: no background mask.\n");
	gdk_window_set_hints(win_main->window, 0, 0, res.background->width,
		res.background->height, res.background->width, res.background->height, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	gdk_window_resize(win_main->window, res.background->width, res.background->height);
	gtk_widget_set_usize(win_main, res.background->width, res.background->height);	/* both... seems to work :) */

	gdk_draw_pixmap(bg_main, gc_main, res.background->pix, 0, 0, 0, 0, res.background->width, res.background->height);
	draw_volume(bg_main, res.areas + BUTTON_VOLUME, cur_volume);
	draw_pitch(bg_main, res.areas + BUTTON_PITCH, cur_pitch);
	set_equalizer_state(equalizer_state);
	for(b = 0; b < NUM_MAIN_AREAS; b++)
		if(res.areas[b].active) draw_button(bg_main, res.areas + b);
	draw_text(bg_main, 1);
	draw_equalizer(bg_main, res.areas + REGION_EQUALIZER);

	gdk_window_set_back_pixmap(win_main->window, bg_main, 0);
	gdk_window_clear(win_main->window);
	gdk_flush();
}


/* ---------------------------------------------------------------------- */
static void kj_init(void)
{
	if(win_main) return;

	root_window = gdk_window_foreign_new(GDK_ROOT_WINDOW());

	memset(&res, 0, sizeof(k_resource));
	cur_seek = 0;
	cur_rate = cur_freq = cur_time = -1;

	kj_default_config();
	kj_load_config();

	if(!load_resource(config.res_name, NULL, &res))
		{
		printf("Error opening the resource file `%s'.\n", config.res_name);
#if XMMS_VIS
		return;
#else
		exit(-1);
#endif
		}

	if(!res.background)
		{
		printf("No background bitmap was found in resource file.\n");
		printf("The png lib must be included for .png files found in newer skins.\n");
#if XMMS_VIS
		return;
#else
		exit(-1);
#endif
		}

	win_main = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_app_paintable(win_main, TRUE);
	gtk_window_set_title(GTK_WINDOW(win_main), "KJ");
	gtk_window_set_policy(GTK_WINDOW(win_main), FALSE, FALSE, TRUE);
	gtk_window_set_wmclass(GTK_WINDOW(win_main), "K Interface", "KJ");

	gtk_widget_set_app_paintable(win_main, TRUE);
	gtk_widget_set_events(win_main, GDK_BUTTON_MOTION_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
	gtk_signal_connect(GTK_OBJECT(win_main), "button_press_event", GTK_SIGNAL_FUNC(button_press_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(win_main), "button_release_event", GTK_SIGNAL_FUNC(button_release_cb), NULL);
	gtk_signal_connect(GTK_OBJECT(win_main), "motion_notify_event", GTK_SIGNAL_FUNC(motion_notify_cb), NULL);
	// gtk_signal_connect(GTK_OBJECT(win_main), "destroy", GTK_SIGNAL_FUNC(kj_cleanup), NULL);
	gtk_signal_connect(GTK_OBJECT(win_main), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &win_main);

	gtk_widget_realize(win_main);
	gdk_window_set_decorations(win_main->window, 0);
	gc_main = gdk_gc_new(win_main->window);
	kj_set_resource();

	accel_main = gtk_accel_group_new();
	ifact_main = gtk_item_factory_new(GTK_TYPE_MENU, "<main>", accel_main);
	ifact_analyser = gtk_item_factory_new(GTK_TYPE_MENU, "<analyser>", accel_main);
	gtk_item_factory_create_items(ifact_main, NUM_MAIN_POPUP, main_popup_items, NULL);
	gtk_item_factory_create_items(ifact_analyser, NUM_ANALYSER_POPUP, analyser_popup_items, NULL);
	gtk_accel_group_attach(accel_main, GTK_OBJECT(win_main));
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(gtk_item_factory_get_widget(ifact_main, "/Visualisation")), GTK_WIDGET(gtk_item_factory_get_widget(ifact_analyser, "")));

	gtk_widget_show(win_main);

	if(config.save_win_pos)
		gdk_window_move(win_main->window, config.win_x, config.win_y);

	kj_set_vismode(config.vis_mode);
	kj_set_analyser_menu();

	timeout_handle = gtk_timeout_add(200, timeout_func_cb, NULL);
	main_move = 0;
	kj_running = 1;
}


#if !XMMS_VIS
/* ---------------------------------------------------------------------- */
int main(int argc, char **argv)
{
	gtk_init(&argc, &argv);
	gdk_rgb_init();

	if(argc > 1) config.res_name = argv[1];

	kj_init();

	gtk_main();
	return 0;
}
#endif

