
#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include "id666.h"

#define safe_lseek(fd,ofs,whence)	do { if (lseek(fd, ofs, whence) == (off_t)-1) goto corrupt_spc; } while(0)
#define safe_read(fd,buf,size)		do { if (read(fd, buf, size) != size) goto corrupt_spc; } while(0)
#define str_replace(str1,str2)		do { if (str1) { g_free(str1); str1 = NULL; } if (*str2) str1 = g_strdup(str2); } while(0)

#define read_ext_string(str)	do {							\
		char buffer[257];							\
		if (!type || total < tmp || tmp < 4 || tmp > 256) goto corrupt_spc;	\
		safe_read(fd, buffer, tmp);						\
		buffer[tmp-1] = '\0';							\
		safe_lseek(fd, 3-(tmp&3), SEEK_CUR);					\
		total -= tmp;								\
		str_replace(str, buffer);						\
	} while(0)

#define read_ext_16bit(num)	do {				\
		unsigned char intbuf[2];			\
		if (total < tmp || tmp != 2) goto corrupt_spc;	\
		safe_read(fd, intbuf, 2);			\
		*(num) = (intbuf[0] << 8) | (intbuf[1]);	\
	} while(0)

#define read_ext_32bit(num)	do {								\
		unsigned char intbuf[4];							\
		if (total < tmp || tmp != 4) goto corrupt_spc;					\
		safe_read(fd, intbuf, 4);							\
		*(num) = (intbuf[0] << 24) | (intbuf[1] << 16) | (intbuf[2] << 8) | (intbuf[3]);\
	} while(0)

static void spc_read_ext_id666(int fd, id666_t *info) {
	int total, tmp;
	unsigned char type, xid;

	read_ext_32bit(&total);

	while (total >= 4) {
		safe_read(fd, &type, 1);	total--;
		safe_read(fd, &xid, 1);		total--;

		read_ext_16bit(&tmp);		total -= 2;

		switch (xid) {
			case 0x01:	read_ext_string(info->song_title);	break;
			case 0x02:	read_ext_string(info->game_title);	break;
			case 0x03:	read_ext_string(info->artist);		break;
			case 0x04:	read_ext_string(info->dumper);		break;

			case 0x05:	read_ext_32bit(&info->dumpdate);		break;

			case 0x06:
				if (type) break;
				switch (tmp) {
					case 0:	str_replace(info->emulator, "unknown"); break;
					case 1:	str_replace(info->emulator, "zsnes");	break;
					case 2:	str_replace(info->emulator, "snes9x");	break;
				
					default: {
						char emunum[64];

						snprintf(emunum, sizeof(emunum), "%d", tmp);
						str_replace(info->emulator, emunum);
						break;
					}
				}
				break;

			case 0x07:	read_ext_string(info->comments);	break;

			case 0x10:	read_ext_string(info->ost_title);	break;
			case 0x11:	info->ost_disc = tmp;			break;
			case 0x12:	info->ost_track = tmp >> 8;		break;

			case 0x13:	read_ext_string(info->publisher);	break;
			case 0x14:	info->year = tmp;			break;

			case 0x30:	read_ext_32bit(&info->intro_length);
					info->intro_length /= 2;
					break;
			case 0x31:	read_ext_32bit(&info->loop_length);
					info->loop_length /= 2;
					break;
			case 0x32:	read_ext_32bit(&info->end_length);
					info->end_length /= 2;
					break;
			case 0x33:	read_ext_32bit(&info->fade_length);
					info->fade_length /= 2;
					break;

			case 0x34:	/* Muted channels */
					break;

			default:	/* Who knows. Ignore it. */
					break;
		}
	}
	
  corrupt_spc:
	return;
}

int spc_read_id666(int fd, id666_t *info) {

	int i, istext = 1;
	unsigned char ch, buffer[257];

	memset(info, 0, sizeof(*info));

	if (lseek(fd, 0x23, SEEK_SET) == (off_t)-1) {
		fprintf(stderr, "Unable to seek in SPC, id666 info will not be available.\n");
		return -1;
	}

	safe_read(fd, &ch, 1);

	/* Is there any id666 info? */
	if (ch == 27)
		return -1;

	/* 0x2E: Song title, 32 chars */
	safe_lseek(fd, 0x2e, SEEK_SET);
	safe_read(fd, buffer, 32);
	buffer[32] = '\0';
	str_replace(info->song_title, buffer);

	/* 0x4E: Game title, 32 chars */
	safe_read(fd, buffer, 32);
	buffer[32] = '\0';
	str_replace(info->game_title, buffer);

	/* 0x6E: Dumper, 16 chars */
	safe_read(fd, buffer, 16);
	buffer[16] = '\0';
	str_replace(info->dumper, buffer);

	/* 0x7E: Comments, 32 chars */
	safe_read(fd, buffer, 32);
	buffer[32] = '\0';
	str_replace(info->comments, buffer);

	/* 0x9E: -maybe- a binary or ascii date (screw you, id666), 11 chars */
	safe_read(fd, buffer, 11);
	buffer[11] = '\0';

	if (buffer[0]) {
		if (buffer[0] >= 1 && buffer[0] <= 31) {
			/* Probably a binary date */
			info->dumpdate =	((buffer[3] << 24) | (buffer[2] << 16) | (buffer[1] << 8) |
						(buffer[0]));
		} else {
			/* Probably an ascii date, and we sure assume it is */
			buffer[2] = buffer[5] = buffer[10] = '\0';

			info->dumpdate =	atoi(&buffer[6]) * 10000 + atoi(&buffer[0]) * 100 + atoi(&buffer[3]);
		}
	}

	/* Parse the fade length/loop length.  id666 sucks. */

	/* 0xA9: Loop length, 3 chars or 3 bytes */
	safe_read(fd, buffer, 3);
	buffer[3] = '\0';

	/* 0xAC: Fade length, 5 chars or 4 bytes (DAMN YOU, ID666) */
	safe_read(fd, &buffer[4], 5);
	buffer[9] = '\0';

	if (buffer[0] || buffer[1] || buffer[2]) {
		for (i = 0; i < 3 && buffer[i]; i++)
			if (buffer[i] < '0' || buffer[i] > '9') {
				istext = 0;
				break;
			}

		for (; i < 3; i++)
			if (buffer[i]) {
				istext = 0;
				break;
			}
	}

	if (buffer[4] || buffer[5] || buffer[6] || buffer[7] || buffer[8]) {
		for (i = 4; i < 9 && buffer[i]; i++)
			if (buffer[i] < '0' || buffer[i] > '9') {
				istext = 0;
				break;
			}

		for (; i < 9; i++)
			if (buffer[i]) {
				istext = 0;
				break;
			}
	}

	if (istext) {
		info->loop_length = atoi(&buffer[0]) * 32000;
		info->fade_length = atoi(&buffer[4]) * 32;
	} else {
		info->loop_length =	((buffer[2] << 16) | (buffer[1] << 8) | (buffer[0])) * 32000;
		info->fade_length =	((buffer[7] << 24) | (buffer[6] << 16) | (buffer[5] << 8) |
					 (buffer[4])) * 32;
	}

	/* Oh, and guess what?  Some dumpers leave CRAP instead of a binary
	   or ascii fade length!  WHAT OVERWHELMING JOY! */
	if (info->loop_length < 0 || info->loop_length > (1*60*60*32000)
	 || info->fade_length < 0 || info->fade_length > (30*60*32000)) {
		info->loop_length = 0;
		info->fade_length = 0;
		istext = 1;
	}

	/* Did I mention id666 is worthless? */

	/* 0xB0 in 'binary' format, 0xB1 in 'text' format: Artist, 32 chars */
	safe_lseek(fd, 0xB0 + istext, SEEK_SET);
	safe_read(fd, buffer, 32);
	buffer[32] = '\0';
	str_replace(info->artist, buffer);

	/* 0xD1 in 'binary' format, 0xD2 in 'text' format: emulator, 1 byte */
	safe_read(fd, &ch, 1);
	switch (ch) {
		case 0:	str_replace(info->emulator, "unknown");	break;
		case 1:	str_replace(info->emulator, "zsnes");	break;
		case 2:	str_replace(info->emulator, "snes9x");	break;
		
		default: {
			char emunum[64];

			snprintf(emunum, sizeof(emunum), "%d", ch);
			str_replace(info->emulator, emunum);
			break;
		}
	}

	/* 0x10200: Extended tags, if there are any */
	if (lseek(fd, 0x10200, SEEK_SET) == (off_t)-1)
		return 0;

	for (;;) {
		if (read(fd, buffer, 4) != 4)
			break;

		if (!memcmp(buffer, "xid6", 4)) {
			spc_read_ext_id666(fd, info);
			break;
		}
		
		safe_read(fd, buffer, 4);

		i = (buffer[0] << 24) | (buffer[1] << 16) | (buffer[2] << 8) | (buffer[3]);
		safe_lseek(fd, i, SEEK_CUR);
	}

	return 0;
	
  corrupt_spc:
	fprintf(stderr, "Corrupt id666 tags in spc\n");
	return -1;
}

void spc_clear_id666(id666_t *info) {
#define FREE(x)	if (info->x) { g_free(info->x); info->x = NULL; }
	FREE(dumper);
	FREE(artist);
	FREE(game_title);
	FREE(song_title);
	FREE(ost_title);
	FREE(publisher);
	FREE(emulator);
	FREE(comments);
}
