/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <stdio.h>

#include "cfrequenceparticleengine.h"

#ifdef NULL
#undef NULL
#define NULL 0
#endif

typedef struct _TFrequenceParticle
{
	gdouble  pos_x,  pos_y,  pos_z;
	gdouble move_x, move_y, move_z;
	gdouble grav_x, grav_y, grav_z;

	GLfloat alpha;
}
TFrequenceParticle;

GLfloat CFrequenceParticleEngine::height_scale = 1.0 / log(256.0);

CFrequenceParticleEngine::CFrequenceParticleEngine(guint max_particles)
{
	active_particles = 0;
	free_particles = 0;
	this->max_particles = max_particles;
	list_length = 0;
	color[0] = color[1] = color[2] = color[3] = 1.0;
	scaler = 1.0;
	height = 0.0;
	speed = 0.05;
	max_create = 5;
	freq_band_min = 0;
	freq_band_max = 255;
	srand(time(0));
	draw_particles = true;
}

CFrequenceParticleEngine::~CFrequenceParticleEngine()
{
	GSList *particle = active_particles;
	while (particle != 0) {
		g_free(particle->data);
		particle = g_slist_next(particle);
	}
	g_slist_free(active_particles);

	particle = free_particles;
	while (particle != 0) {
		g_free(particle->data);
		particle = g_slist_next(particle);
	}
	g_slist_free(free_particles);
}

void CFrequenceParticleEngine::calc(gint16 data[2][256])
{
	/*
	  Berechen die aktuelle Hoehe der des Frequenzbalken
	*/
	gint i;
	gint y;

	for(i = freq_band_min, y = 0; i <= freq_band_max; i++)
	{
		if(data[0][i] > y)
			y = data[0][i];
	}
	y >>= 7;
	if(y > 0)
		height = (log(y) * height_scale);
	else
		height = 0;
}

void CFrequenceParticleEngine::draw()
{
	GSList *particle = active_particles;
	GSList *prev_slist = 0;
	gboolean draw_it = TRUE;

	glPushMatrix();
	glEnable(GL_BLEND);
	color[3] = 0.2;
	glBegin(GL_QUADS);
		glColor4fv(color);
		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(0.0f, 0.0f, 0.0f);
		glVertex3f(1.0f, 0.0f, 0.0f);
		glVertex3f(1.0f, 0.0f, 1.0f);
		glVertex3f(0.0f, 0.0f, 1.0f);

		glNormal3f(0.0f, 1.0f, 0.0f);
		glVertex3f(0.0f, height, 0.0f);
		glVertex3f(1.0f, height, 0.0f);
		glVertex3f(1.0f, height, 1.0f);
		glVertex3f(0.0f, height, 1.0f);
	glEnd();

	if (!draw_particles) {
		glPopMatrix();
		return;
	}

	while (particle) {
		// Increase height
//		pos_y(particle) += speed;
		draw_it = TRUE;

		TFrequenceParticle* draw_particle = (TFrequenceParticle*) particle->data;
		gdouble p_alpha = draw_particle->alpha;
		// Check new alpha
		if ((draw_particle->pos_y <= height) && (p_alpha < 1.0)) {
			p_alpha = MIN(p_alpha + 0.3, 1);
		}
		else {
			p_alpha = MAX(p_alpha - 0.3, 0);
			// if particle not visible -> "delete" it
			if (p_alpha <= 0.0) {
				GSList *tmp_element = particle;
				if (prev_slist != NULL) {
					prev_slist->next = particle->next;
					particle = prev_slist;
				}
				else {
					active_particles = particle = particle->next;
				}
				tmp_element->next = free_particles;
				free_particles = tmp_element;
				list_length--;
				draw_it = FALSE;
			}
			else {
				draw_particle->move_x /= 5;
				draw_particle->move_y /= 5;
				draw_particle->move_z /= 5;
			}
		}
		// If not deleted draw particle
		if (draw_it) {
			draw_particle->pos_x += draw_particle->move_x;
			draw_particle->pos_y += draw_particle->move_y;
			draw_particle->pos_z += draw_particle->move_z;

			draw_particle->move_x += draw_particle->grav_x;
			draw_particle->move_y += draw_particle->grav_y;
			draw_particle->move_z += draw_particle->grav_z;

			draw_particle->alpha = color[3] = p_alpha;
			gdouble x = draw_particle->pos_x;
			gdouble y = draw_particle->pos_y;
			gdouble z = draw_particle->pos_z;
			glBegin(GL_TRIANGLE_STRIP);
				glColor4fv(color);
/*				if (texture) {
					texture->bind();
//					glTranslatef(pos_x(particle), pos_y(particle), pos_z(particle));
					glTexCoord2d(1,1);  glVertex3f(x + 0.1f, y + 0.1f, z); // Top Right
					glTexCoord2d(0,1);  glVertex3f(x       , y + 0.1f, z); // Top Left
					glTexCoord2d(1,0);  glVertex3f(x + 0.1f, y       , z); // Bottom Right
					glTexCoord2d(0,0);  glVertex3f(x       , y       , z); // Bottom Left
				}
				else {*/
//					printf("%f / %f / %f // %f\n", x, y, z, p_alpha);
			    		glVertex3f(x + 0.1f, y, z + 0.1f); // Top Right
					glVertex3f(x,        y, z + 0.1f); // Top Left
					glVertex3f(x + 0.1f, y, z); // Bottom Right
					glVertex3f(x,        y, z); // Bottom Left
//				}
			glEnd();
		}
		prev_slist = particle;
		particle = g_slist_next(particle);
	}
	guint counter;
	if (max_particles > 0) {
		if (list_length < max_particles) {
			counter = max_particles - list_length;
			counter = MIN(counter, max_create);
		}
		else { counter = 0; }
	}
	else { counter = max_create; }

	// create new particles
	for (; counter > 0; counter--) {
		TFrequenceParticle* new_particle;

		if (free_particles != 0) { // if there are free particles
			GSList *tmp_element = free_particles;
			free_particles = free_particles->next;
			tmp_element->next = active_particles;
			active_particles = tmp_element;
			new_particle = (TFrequenceParticle*) active_particles->data;
		}
		else { // create new ones
			new_particle  = g_new(TFrequenceParticle, 1);
			active_particles = g_slist_prepend(active_particles, new_particle);
		}
		new_particle->pos_x = MAX((GLfloat) rand() / RAND_MAX - 0.1, 0);
		new_particle->pos_y = 0.0;
		new_particle->pos_z = MAX((GLfloat) rand() / RAND_MAX - 0.1, 0);
		new_particle->alpha = 1.0;
		new_particle->grav_x = 0.002;
		new_particle->grav_y = 0.07;
		new_particle->grav_z = 0.002;
		new_particle->move_x = (GLfloat) rand() / RAND_MAX / 100;
		new_particle->move_y = (GLfloat) rand() / RAND_MAX / 10;
		new_particle->move_z = (GLfloat) rand() / RAND_MAX / 100;
		list_length++;
	}
	glPopMatrix();
}



CFrequenceParticleEngineField::CFrequenceParticleEngineField(guint rows, guint columns, guint max_particles)
{
	this->rows = rows;
	this->columns = columns;

	y_angle = 45.0;
	y_speed = 0.5;

	x_angle = 20.0;
	x_speed = 0.0;

	z_angle = 0.0;
	z_speed = 0.0;

	space = 0.01;

	scaler[0] = (1 - ((rows - 1) * space)) / rows;
	scaler[1] = 1.0;
	scaler[2] = (1 - ((columns - 1) * space)) / columns;

	translator[0] = scaler[0] + space;
	translator[1] = 0.0;
	translator[2] = scaler[2] + space;

	bars = new CFrequenceParticleEngine* [columns];
	for (guint i = 0; i < columns; i++)
		{ bars[i] = new CFrequenceParticleEngine[rows](max_particles); }

	if ((rows == 16) && (columns == 16)) {
		const gint xscale[] =
			{ 0, 1, 2, 3, 5, 7, 10, 14, 20, 28, 40, 54, 74, 101, 137, 187, 255 };

		for (guint column = 0; column < columns; column++) {
			GLfloat blue_base = column * (1.0 / (columns - 1));
			GLfloat red_base = 1.0 - blue_base;

			for (guint row = 0; row < rows; row++) {
				bars[column][row].setFrequenceBand(xscale[column], xscale[column + 1]);
				bars[column][row].setColor(red_base - (row * (red_base / 15.0)), row * (1.0 / 15), blue_base);
			}
		}
	}
}

CFrequenceParticleEngineField::~CFrequenceParticleEngineField()
{
	for (guint i = 0; i < columns; i++) { delete[] bars[i]; }
	delete[] bars;
}

void CFrequenceParticleEngineField::renderFreq(gint16 data[2][256])
{
	// Werte um eine Reihe verschieben
	for (guint row = rows - 1; row > 0; row--)
		for (guint column = 0; column < columns; column++)
			{ bars[column][row].setHeight(bars[column][row - 1].getHeight()); }

	// Neue erste Reihe berechnen
	for (guint column = 0; column < columns; column++)
		{ bars[column][0].calc(data); }
}

void CFrequenceParticleEngineField::draw()
{
	glPushMatrix();

	// Richtige Ausrichtung
	glTranslatef(0, -0.8, -4.2);
	glRotatef(x_angle,1.0,0.0,0.0);
	glRotatef(y_angle,0.0,1.0,0.0);
	glRotatef(z_angle,0.0,0.0,1.0);
	glTranslatef(-1.6, 0, -1.6);
	glScalef(2.8f, 1.0f, 2.8f);

	// Alle Partikelbnder positionieren und malen
	for (guint i = 0; i < rows; i++) {
		glPushMatrix();
		for (guint j = 0; j < columns; j++) {
			glTranslatef(translator[0], 0, 0);
			glPushMatrix();
			glScalef(scaler[0], scaler[1], scaler[2]);
			bars[i][j].draw();
			glPopMatrix();
		}
		glPopMatrix();
		glTranslatef(0, 0, translator[2]);
	}
	glDisable(GL_BLEND);
	glPopMatrix();

	// Drehung
	x_angle += x_speed;
	if(x_angle >= 360.0)
		x_angle -= 360.0;

	y_angle += y_speed;
	if(y_angle >= 360.0)
		y_angle -= 360.0;

	z_angle += z_speed;
	if(z_angle >= 360.0)
		z_angle -= 360.0;
}

void CFrequenceParticleEngineField::setDrawParticles(bool value)
{
	for (guint column = 0; column < columns; column++)
		for (guint row = 0; row < rows; row++)
			bars[column][row].setDrawParticles(value);
}

bool CFrequenceParticleEngineField::getDrawParticles()
{
	if (bars != 0) { return bars[0][0].getDrawParticles(); }
	return true;
}
