/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "cscene.h"

#include "maths.h"
#include "data.h"

extern gint16 static_pcm_data[2][512];
extern gint16 static_freq_data[2][256];
extern float madspin_data[256];

#define MAXPOINTS 512
#define FRAMERATE 500
#define MAXLINES 1
#define STAR_ADJUST 5


CScene::CScene()
	:frame(0.0f), total(0.0f), texsize(0.25f)
{
	srand(123456);

	if (logo_texture.loadData(data_xmms_rgb, -1, PT_SGI))
		{ logo_texture.realize(TRUE); }
	else { g_print("cscene.cpp [CScene] : Error - couldn't create xmms texture\n"); }

	if (growbar_texture.loadData(data_growbar_rgb, -1, PT_SGI))
		{ growbar_texture.realize(TRUE); }
	else { g_print("cscene.cpp [CScene] : Error - couldn't create growbar texture\n"); }

	if (mesh_texture.loadData(data_blur_mesh_rgb, -1, PT_SGI))
		{ mesh_texture.realize(TRUE); }
	else { g_print("cscene.cpp [CScene] : Error - couldn't create mesh texture\n"); }

	if (generate_pcm_texture())
		{ pcm_texture.realize(TRUE); }
	else { g_print("cscene.cpp [CScene] : Error - couldn't create pcm texture\n"); }

	if (madspin_star_one.loadFile(SINGIT_DATA_DIR "/star1.bmp", PT_BMP))
	{
		madspin_star_one.setWrapS(GL_CLAMP);
		madspin_star_one.setWrapT(GL_CLAMP);
		madspin_star_one.setMinFilter(GL_LINEAR);
		madspin_star_one.setMagFilter(GL_LINEAR);
		madspin_star_one.realize(TRUE);
	}
	else { g_print("cscene.cpp [CScene] : Error - couldn't create madspin_star_one texture\n"); }

	if (madspin_star_two.loadFile(SINGIT_DATA_DIR "/star2.bmp", PT_BMP))
	{
		madspin_star_two.setWrapS(GL_CLAMP);
		madspin_star_two.setWrapT(GL_CLAMP);
		madspin_star_two.setMinFilter(GL_LINEAR);
		madspin_star_two.setMagFilter(GL_LINEAR);
		madspin_star_two.realize(TRUE);
	}
	else { g_print("cscene.cpp [CScene] : Error - couldn't create madspin_star_two texture\n"); }

	build_world_rings();
}

CScene::~CScene()
{
	glDeleteLists(world_gl_list, 1);
}

bool CScene::generate_pcm_texture()
{
	const guint bpp = 32;
	const gint wh = 32;
	const guint size = 4 * wh * wh;
	gchar texture_image_p[size];

	for (int v = 0; v < wh; v++) {
		for (int u = 0; u < wh; u++) {
			float x, y, a, c;
			x = u - wh / 2;
			y = v - wh / 2;

			a = 255.0f - 32.0f * hypot(x,y);
			if (a < 0.0f) {
				 a = 0.0f;
			}
			c = a;

			*(texture_image_p + 0 + u*4 + v*4*wh) = (gchar)c;
			*(texture_image_p + 1 + u*4 + v*4*wh) = (gchar)c;
			*(texture_image_p + 2 + u*4 + v*4*wh) = (gchar)c;
			*(texture_image_p + 3 + u*4 + v*4*wh) = (gchar)a;
		}
	}

	return (pcm_texture.loadData(&texture_image_p[0], size, PT_RAW)) ?
		pcm_texture.setRAW(bpp, wh, wh) : false;
}

void CScene::draw_pcm(void)
{
	glShadeModel(GL_SMOOTH);

	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);

	glBlendFunc(GL_SRC_ALPHA, GL_ONE);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	pcm_texture.bind();

	glPushMatrix();

	glTranslatef(0.0f, 0.5f, 0.0f);
	glScalef(1.2f, 0.6f, 1.2f);

	glBegin(GL_TRIANGLES);
	for (int j = 0;j != 2;j++) {
		const int skip = 4;

		float y;
		float dy;
		y = j;
		y -= 0.5f;
		dy = y;
		y *= 1.0f;
		dy /= 20.0f;

		for (int i = 0;i != 512;i += skip) {
			float u;

			y += skip * dy;
			u = static_pcm_data[j][i];
			u /= 16384 / 8;
			u = (u < 0.0f) ? -u : u;
			u += 0.8f;

			glColor4f(1.0f, 0.0f, 0.0f, 1.0f);
			glTexCoord2f(0.5f,0.0f);
			glVertex3f(0.0f, y, -u);
			glColor4f(0.0f, 1.0f, 0.0f, 1.0f);
			glTexCoord2f(0.0f,1.0f);
			glVertex3f(-u, y, u);
			glColor4f(0.0f, 0.0f, 1.0f, 1.0f);
			glTexCoord2f(1.0f,1.0f);
			glVertex3f(u, y, u);
		}
	}
	glEnd();

	glPopMatrix();

	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
}

void CScene::construct_ring(int elements, float ur, float dr, float h2, float *rgba_p)
{
	glShadeModel(GL_FLAT);

	glBegin(GL_QUADS);

	glColor4fv(rgba_p);

	fix16_16 fix_rad = 0;

	float bx = Sin(fix_rad);
	float bz = Cos(fix_rad);

	for (int i = 0;i != elements;i++)
	{
		fix_rad += ANGLE_2PI / elements;
		float nx = Sin(fix_rad);
		float nz = Cos(fix_rad);

		glTexCoord2f(0.0f, 0.4f);
		glVertex3f(ur * bx, -h2, ur * bz);

		glTexCoord2f(1.0f, 0.4f);
		glVertex3f(dr * bx,  h2, dr * bz);

		glTexCoord2f(1.0f, 0.6f);
		glVertex3f(dr * nx,  h2, dr * nz);

		glTexCoord2f(0.0f, 0.6f);
		glVertex3f(ur * nx, -h2, ur * nz);

		bx = nx;
		bz = nz;
	}

	glEnd();
}

void CScene::build_world_rings(void)
{
	enum {
		ELEMENTS = 6
	};

	float rgba[4] = { 1.0f, 1.0f, 1.0f, 0.5f };

	float angle[ELEMENTS][4] =
	{ {  60, 0, 0, 1 },
	  {  60, 0, 1, 0 },
	  {  60, 1, 0, 0 },
	  { 120, 0, 0, 1 },
	  { 120, 0, 1, 0 },
	  { 120, 1, 0, 0 } };

	world_gl_list = glGenLists(1);

	glPushMatrix();

	glNewList(world_gl_list, GL_COMPILE);
	glEnable(GL_BLEND);
	for (int i = 0; i != ELEMENTS; i++)
	{
		glPushMatrix();

		glRotatef(angle[i][0],
			  angle[i][1],
			  angle[i][2],
			  angle[i][3]);
		construct_ring(32,
				27.0f,
				27.0f,
				0.1f,
				rgba);
		glPopMatrix();
	}
	glDisable(GL_BLEND);
	glEndList();

	glPopMatrix();
}

void CScene::draw_world_rings(bool animate)
{
	static float full_angle = 30.0f;

	glEnable(GL_BLEND);

	glPushMatrix();

	glRotatef(full_angle, Sin(full_angle), Cos(full_angle), full_angle);

	if (animate)
		{ full_angle += 0.05f; }

	glCallList(world_gl_list);

	glPopMatrix();

	glDisable(GL_BLEND);
}

void CScene::draw_bg_mesh(void)
{
	enum {
		 VX,  VY,  VZ,  VW, // Punkt
		 VR,  VG,  VB,  VA, // Farbe
		VNX, VNY, VNZ, VNW, // Normale

		VTX_ELEMENTS
	};
	enum {
		ELEMENTS = 128
	};
	static int mesh_init_flag;
	static float vtx[4 * VTX_ELEMENTS * ELEMENTS];
	static fix16_16 alpha_radian[4 * ELEMENTS];

	const float u = 16.0f;
	const float r = 28.0f;

	// Nur einmal initialisieren
	if (mesh_init_flag == 0) {

		// Gesamtgroesse des Raumes
		float src[] = {
			-u, u, r, 1.0f,
			 u, u, r, 1.0f,
			 u,-u, r, 1.0f,
			-u,-u, r, 1.0f
		};
		CMatrix m;

		for (int i = 0; i < ELEMENTS; i++) {
			// Beliebige Rotations-Matrix
			m.Unit();
			m.MulRotX(RandFloat() * PI2);
			m.MulRotY(RandFloat() * PI2);
			m.MulRotZ(RandFloat() * PI2);

			// Ein Quadrat positionieren
			for (int h = 0; h < 4; h++) {
				// Punkt "setzen"
				m.MulVector(&src[h*4], &vtx[VX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
				// Normale bestimmen
				GetOuterProduct(&vtx[VX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS],
					&vtx[VNX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
				UnitVector(&vtx[VNX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
				// Farbe bestimmen
				vtx[VR + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = RandFloat();
				vtx[VG + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = RandFloat();
				vtx[VB + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = RandFloat();
				vtx[VA + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = 0.0f;
				alpha_radian[h + i*4] = Rand(ANGLE_2PI);
			}
		}
		mesh_init_flag = 1;
	}

	glPushMatrix();

	glShadeModel(GL_SMOOTH);

	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);

	glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
	mesh_texture.bind();

	glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
	glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
	glEnable(GL_TEXTURE_GEN_S);
	glEnable(GL_TEXTURE_GEN_T);

	glBegin(GL_QUADS);
	for (int i = 0;i != ELEMENTS;i++)
	{
		vtx[VA + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[0 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[0 + i*4] += ANGLE_2PI/512;
		alpha_radian[0 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(0.0f,1.0f);
		glNormal3fv(&vtx[VNX + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);

		vtx[VA + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[1 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[1 + i*4] += ANGLE_2PI/256;
		alpha_radian[1 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(1.0f,1.0f);
		glNormal3fv(&vtx[VNX + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);

		vtx[VA + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[2 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[2 + i*4] += ANGLE_2PI/512;
		alpha_radian[2 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(1.0f,0.0f);
		glNormal3fv(&vtx[VNX + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);

		vtx[VA + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[3 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[3 + i*4] += ANGLE_2PI/128;
		alpha_radian[3 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(0.0f,0.0f);
		glNormal3fv(&vtx[VNX + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
	}
	glEnd();

	glDisable(GL_TEXTURE_GEN_S);
	glDisable(GL_TEXTURE_GEN_T);

	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);

	glPopMatrix();
}


void CScene::draw_xmms_logo(bool animate)
{
	/* Funtions On */
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);

	/* Options */
	glBlendFunc(GL_SRC_ALPHA, GL_ONE);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glShadeModel(GL_FLAT);

	logo_texture.bind();

	glPushMatrix();

	glTranslatef(0.0f, -8.0f, 0.0f);

	const float w2 = 16.0f;
	const float h2 = 16.0f;

	static fix16_16 radian;

	int times = (int)(Sin(radian) * 16.0f);
	if (times < 1) { times = 1; }

	if (animate) {
		radian += ANGLE_2PI / 512;
		radian &= ANGLE_MASK;
	}
	float y = 0;
	float a = 1.0f;

	glBegin(GL_QUADS);
	for (int i = 0;i != times;i++)
	{
		glColor4f(1.0f, 1.0f, 1.0f, a);

		glTexCoord2f(0.0f,0.0f);
		glVertex3f(w2, y, -h2);

		glTexCoord2f(1.0f,0.0f);
		glVertex3f(-w2, y, -h2);

		glTexCoord2f(1.0f,1.0f);
		glVertex3f(-w2, y, h2);

		glTexCoord2f(0.0f,1.0f);
		glVertex3f(w2, y, h2);

		a *= 0.54f;
		y += 0.07f;
	}

	glEnd();

	glPopMatrix();

	/* Funtions Off */
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
}

void CScene::draw_madspin()
{
	double csab, csapb;
	double snab, snapb;
	double sa, ca, s5;
	double s1r, s1g, s1b, s1a;
	double s2r, s2g, s2b, s2a;
	int point;
	int line;
	int trail = 40;
	float b;
	float aa, a;
	double x, y, z;
	int i;
	int ampl = 200;
	float adj_texsize;

	glShadeModel(GL_SMOOTH);

	glBlendFunc(GL_SRC_ALPHA,GL_ONE); // Reset the default blend mode
	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);


	for (i = 1; i < 50; i++)
	{
		total += madspin_data[i];
	}
	if (total > 4.5)
		total = 4.5;

	total /= 4.5f;

	for (line = MAXLINES; line > 0; line--)
	{
		for(point = 0; point <= MAXPOINTS; point++)
		{
			b = 1.5f + point / 33.33333f;
			aa = trail - line * 1.0f;
			a = (aa + frame) / 33.33333f;

			csab = cos(a / b);
			csapb = cos(a + b);
			snab = sin(a / b);
			snapb = sin(a + b);

			sa = sin(a * .1f);
			ca = cos(a * .1f);
			s5 = sin(a * .05f);

			x = sa * snapb * ampl * csab + ca * (csapb * ampl * b * csab + s5 * (ampl * (csab+2*snapb)));
			y = sa * csapb * ampl * csab + ca * (csapb * ampl * b * snab + s5 * (ampl * (csab+2*csapb)));
			z = ca * csapb * ampl * csab + ca * (snapb * ampl * b * snab + s5 * (ampl * (snab+2*snapb)));

			x /= 255.0f;
			y /= 255.0f;
			z /= 255.0f;

			glPushMatrix();											// Save the main location

			glTranslatef((float)x, (float)y, (float)z);	// Move to location of current point
			madspin_star_one.bind();			// Select the main star texture

			s1r = ((point * 1.0f) / MAXPOINTS);
			s1g = (MAXPOINTS - point) / (MAXPOINTS * 1.0f);
			s1b = ((point * 1.0f) / MAXPOINTS) * 0.5f;
			s1a = (madspin_data[(int)(point / MAXPOINTS * 220)] + total);

			s2r = sin(frame / 400.0f);
			s2g = cos(frame / 200.0f);
			s2b = cos(frame / 300.0f);
			s2a = (madspin_data[(int)(point / MAXPOINTS * 220)]);

			if (s1a > 0.005f)
			{
				glBegin(GL_TRIANGLE_STRIP);
					glColor4f((float)s1r, (float)s1g, (float)s1b, (float)s1a);
					texsize = (((madspin_data[(int)(point / MAXPOINTS * 220)])) /
						(2048.01f - (point * 4.0f))) * (((point - MAXPOINTS)/(-MAXPOINTS)) * 18.0f) + 0.15f;

					adj_texsize = texsize * STAR_ADJUST;

					glTexCoord2d(1,1); // Top Right
					glVertex3f(adj_texsize, adj_texsize, (float)z);
					glTexCoord2d(0,1); // Top Left
					glVertex3f(-adj_texsize, adj_texsize, (float)z);
					glTexCoord2d(1,0); // Bottom Right
					glVertex3f(adj_texsize, -adj_texsize, (float)z);
					glTexCoord2d(0,0); // Bottom Left
					glVertex3f(-adj_texsize, -adj_texsize, (float)z);
				glEnd();
			}

			madspin_star_two.bind();	// Select the secondary glitter star texture
			glRotatef(frame + point, 0.0f, 0.0f, 1.0f);

			if (s2a > 0.005f)
			{
				glBegin(GL_TRIANGLE_STRIP);
					glColor4f((float)s2r, (float)s2g, (float)s2b, (float)s2a);
					texsize = (((madspin_data[(int)(point / MAXPOINTS * 220)])) /
						(2048.01f - (point * 4.0f))) * (((point - MAXPOINTS)/(-MAXPOINTS)) * 18.0f) + 0.35f;

					texsize *= ((rand() % 100) / 100.0f) * 2.0f;

					adj_texsize = texsize * STAR_ADJUST;

					glTexCoord2d(1,1); // Top Right
					glVertex3f(adj_texsize, adj_texsize, (float)z);
					glTexCoord2d(0,1); // Top Left
					glVertex3f(-adj_texsize, adj_texsize, (float)z);
					glTexCoord2d(1,0); // Bottom Right
					glVertex3f(adj_texsize, -adj_texsize, (float)z);
					glTexCoord2d(0,0); // Bottom Left
					glVertex3f(-adj_texsize, -adj_texsize, (float)z);
				glEnd();
			}

			glPopMatrix();	// Move back to main position
		}
	}

	glDisable(GL_BLEND);
	glDisable(GL_TEXTURE_2D);
}

void CScene::draw(guint time, bool animate)
{
	bool run_animation = (animate && !lyric_text_engine.has_lyrics());

	g_get_current_time(&last_rtc_time);

	glDepthMask(GL_TRUE);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); // Clear The Screen And The Depth Buffer

	// Save before rotate state for the song text position
	glPushMatrix();

	if (run_animation) { camera.rotate(); }
	camera.lookAt();

/*
	opaque
*/
	draw_world_rings(run_animation);

/*
	transparent
*/
	glDepthMask(GL_FALSE);

	draw_bg_mesh();

	draw_xmms_logo(run_animation);

	if (!lyric_text_engine.has_lyrics()) {
		draw_pcm();
		draw_madspin();
	}

	// Restore state for the song text position
	glPopMatrix();

/*
	opaque
*/
	glEnable(GL_DEPTH_TEST);
	glDepthMask(GL_TRUE);

	lyric_text_engine.draw(time);

/*
	Some kind of frame counter
*/
	g_get_current_time(&new_rtc_time);
	gfloat elapsed_time = ((float)new_rtc_time.tv_usec - (float)last_rtc_time.tv_usec) / 1000000;
	if (elapsed_time < 0)
		elapsed_time = 0;

	frame += elapsed_time * FRAMERATE;
}
