/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include <stdlib.h>
#include <stdio.h>

#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_syswm.h>
#include <SDL/SDL_ttf.h>

#include "singit/displayer_plugin.h"
#include "displayer_sdl.h"
#include "displayer_sdl_config.h"
#include "displayer_sdl_config_dlg.h"
#include "displayer_sdl_lyrics.h"
#include "sdl_helpers.h"
#include "sdl_handlers.h"

#include "singit/control.h"

#include <stdlib.h>
#include <stdio.h>

#include "G-Force.h"
#include "EgOSUtils.h"
#include "RectUtils.h"
#include "CEgFileSpec.h"
#include "SDL_prim.h"

#define SND_BUF_SIZE 550
#define NUMSAMPLES 200
#define SAMPSKIP 1

static PixPort*		gOutPort = NULL;
static GForce*		gGF = NULL;
static UtilStr*		gCurWinTitle = NULL;
static float		gSoundBuf[ SND_BUF_SIZE ];
int                     numsamp = 0;

extern DisplayerPlugin sdl_dp;

static gboolean CreateScreen(gint nWidth, gint nHeight, gboolean bFullscreen)
{
	static gint save_full[2] = { -1 };
	gint nFlags, w, h;

	if (bFullscreen == TRUE) {
		nFlags = SDL_HWSURFACE | SDL_FULLSCREEN;
		save_full[0] = STATUS->screen->w;
		save_full[1] = STATUS->screen->h;
		w = 640;
		h = 480;
	}
	else {
		nFlags = SDL_SWSURFACE | SDL_RESIZABLE;
		if ((save_full[0] != -1) && (save_full[1] != -1)) {
			w = save_full[0];
			h = save_full[1];
			save_full[0] = -1;
		}
		else {
			w = nWidth;
			h = nHeight;
		}
	}

	STATUS->screen = SDL_SetVideoMode(w, h, bFullscreen ? 16 : 0, nFlags);

	if (STATUS->screen == NULL)
		{ return FALSE; }

	STATUS->is_fullscreen = bFullscreen;

	if (bFullscreen == TRUE) {
		sdl_helper_blanking(BLANK_DISABLE);
		SDL_WM_GrabInput(SDL_GRAB_ON);
		SDL_ShowCursor(0);
	}
	else {
		sdl_helper_blanking(BLANK_ENABLE);
		SDL_WM_GrabInput(SDL_GRAB_OFF);
		SDL_ShowCursor(1);
	}

	return (STATUS->screen != NULL);
}

extern "C" gboolean sdl_handler_init(void)
{
	gchar *title;
	Rect rect;

#ifdef CODEDEBUG
	DEBUG(("sdl_handlers.cpp [sdl_handler_init]\n"));
#endif

	// Initialise the SDL library
	if (SDL_WasInit(SDL_INIT_VIDEO) > 0) {
		g_print("SDL Video was initialized before : no two SDL windows allowed : exit\n");
		STATUS->sdl_inuse = TRUE;
		return FALSE;
	}
	else {
		if (SDL_Init(SDL_INIT_VIDEO) != 0) {
			g_print("Unable to init SDL Video: %s\n", SDL_GetError());
			return FALSE;
		}
	}
	STATUS->song = NULL;

	if (CreateScreen(480, 300, FALSE) == FALSE) {
		g_print("Graphic mode is not available: %s\n", SDL_GetError());
		gtk_idle_add (displayer_sdl_disable_func, NULL);
		return FALSE;
	}

	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("SDL displayer"), NULL);
	SDL_WM_SetCaption(title, PACKAGE_NAME);
	g_free(title);
//	SDL_WM_SetCaption(SINGIT_VERSION_STRING "-" "SDL displayer");

	TTF_Init();
	STATUS->font = TTF_OpenFont(SINGIT_DATA_DIR "/vixar.ttf", 32);
//	TTF_SetFontStyle(STATUS->font, TTF_STYLE_BOLD);

	SDL_Color colors[256];

	for (int pp = 0; pp < 256 ; pp++) {
		colors[pp].r = pp / 4;
		colors[pp].g = pp / 4;
		colors[pp].b = pp / 4;
	}

	gGF = new GForce;
	gOutPort = new PixPort;
	gCurWinTitle = new UtilStr;

	gOutPort->Init(STATUS->screen);
	SetRect(&rect, 0, 0, STATUS->screen->w, STATUS->screen->h);
	gGF -> SetPort(gOutPort, rect, false);
//	gGF -> SetWinPort(0, &rect);
//	gGF -> StoreWinRect();

	return TRUE;
}

extern "C" void sdl_handler_finish(void)
{
#ifdef CODEDEBUG
	DEBUG(("sdl_handlers.cpp [sdl_handler_finish]\n"));
#endif

	if ( gGF ) {
	        delete gGF;
	        gGF = NULL;

		delete gCurWinTitle;
		gCurWinTitle = NULL;
	}

	if (gOutPort) {
		delete gOutPort;
		gOutPort = NULL;
	}

      	EgOSUtils::Shutdown();

	if (STATUS->font) {
		TTF_CloseFont(STATUS->font);
		STATUS->font = NULL;
	}

	// Close SDL (prevents the ticks to continue running)
	TTF_Quit();
	SDL_Quit();
}

void draw_border_rect(SDL_Surface *surface, SDL_Rect *rect, guint border, Uint32 color)
{
	guint x2 = rect->x + rect->w;
	guint y2 = rect->y + rect->h;

	SDL_drawLine(surface, rect->x - border - 1, rect->y - border - 1, x2 + border, rect->y - border - 1, color);
	SDL_drawLine(surface, rect->x - border - 1, rect->y - border - 1, rect->x - border - 1, y2 + border, color);
	SDL_drawLine(surface, rect->x - border - 1, y2 + border, x2 + border, y2 + border, color);
	SDL_drawLine(surface, x2 + border, rect->y - border, x2 + border, y2 + border, color);
}

extern "C" void displayer_sdl_draw_text()
{
	sdl_lyrics_draw_solid(STATUS->screen);
}

extern "C" void sdl_handler_key_down(void *pEvent)
{
	SDL_KeyboardEvent* pKeyEvent = (SDL_KeyboardEvent*)pEvent;
	gint key = pKeyEvent->keysym.sym;
	gint mod = pKeyEvent->keysym.mod;
	gint volume, value;
	gboolean draw;

#ifdef CODEDEBUG
	DEBUG(("sdl_handlers.cpp [sdl_handler_key_down]\n"));
#endif

	switch (key) {
	case SDLK_ESCAPE:
	case SDLK_q:
		if (STATUS->is_fullscreen) {
			draw = CreateScreen(640, 480, !STATUS->is_fullscreen);
		}
		gtk_idle_add (displayer_sdl_disable_func, NULL);
		break;
	case SDLK_a:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_about_dlg
				(sdl_dp.singit_session);
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_b:
		xmms_remote_playlist_next(sdl_dp.xmms_session);
		break;
	case SDLK_c:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			displayer_sdl_config_show();
			GDK_THREADS_LEAVE();
		}
		else { xmms_remote_pause(sdl_dp.xmms_session); }
		break;
	case SDLK_e:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_editor_dlg
				(sdl_dp.singit_session);
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_f:
		if (gGF != NULL) {
			gGF->HandleCommand(cFrameRate);
		}
		break;
	case SDLK_i:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_tag_manager_dlg
				(sdl_dp.singit_session);
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_j:
		if (gGF != NULL) {
			if (mod & KMOD_CTRL) {
				gGF->HandleCommand(cPrevDeltaField);
			}
			else if (mod & KMOD_ALT) {
				gGF->HandleCommand(cPrevWaveShape);
			}
			else { gGF->HandleCommand(cPrevColorMap); }
		}
		break;
	case SDLK_k:
		if (gGF != NULL) {
			if (mod & KMOD_CTRL) {
				gGF->HandleCommand(cNextDeltaField);
			}
			else if (mod & KMOD_ALT) {
				gGF->HandleCommand(cNextWaveShape);
			}
			else { gGF->HandleCommand(cNextColorMap); }
		}
		break;
	case SDLK_l:
		if (gGF != NULL) {
			if (mod & KMOD_CTRL) {
				gGF->HandleCommand(cToggleFieldShow);
			}
			else if (mod & KMOD_ALT) {
				gGF->HandleCommand(cToggleShapeShow);
			}
			else { gGF->HandleCommand(cToggleColorShow); }
		}
		break;
	case SDLK_p:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_prefs_dlg
				(sdl_dp.singit_session);
			GDK_THREADS_LEAVE();
		}
		break;
	case SDLK_s:
		if (gGF != NULL) {
			gGF->HandleCommand(cGetConfigInfo);
		}
		break;
	case SDLK_t:
		if (gGF != NULL) {
			gGF->HandleCommand(cDispTrackTitle);
		}
		break;
	case SDLK_v:
		xmms_remote_stop(sdl_dp.xmms_session);
		break;
	case SDLK_x:
		xmms_remote_play(sdl_dp.xmms_session);
		break;
	case SDLK_y:
		if (GET_DSCD->gblToggleYZ)
			{ xmms_remote_playlist_prev(sdl_dp.xmms_session); }
		break;
	case SDLK_z:
		if (!GET_DSCD->gblToggleYZ)
			{ xmms_remote_playlist_prev(sdl_dp.xmms_session); }
		break;
	case SDLK_h:
	case SDLK_QUESTION:
	case SDLK_WORLD_63:
		if (gGF != NULL)
			{ gGF->ShowHelp(); }
		break;
	case SDLK_KP_PLUS:
		volume = xmms_remote_get_main_volume(sdl_dp.xmms_session);
		if (volume < 100) { volume++; }
		xmms_remote_set_main_volume(sdl_dp.xmms_session, volume);
		break;
	case SDLK_KP_MINUS:
		volume = xmms_remote_get_main_volume(sdl_dp.xmms_session);
		if (volume > 0) { volume--; }
		xmms_remote_set_main_volume(sdl_dp.xmms_session, volume);
		break;
	case SDLK_RETURN:
	case SDLK_KP_ENTER:
		if (mod & KMOD_ALT) {
			draw = CreateScreen(640, 480, !STATUS->is_fullscreen);
			if ((draw == TRUE) && ( gGF )) {
				SDL_ShowCursor(STATUS->is_fullscreen ? SDL_DISABLE : SDL_ENABLE);
				Rect rect;
				gOutPort->Init(STATUS->screen);
				SetRect( &rect, 0, 0, STATUS->screen->w,
					STATUS->screen->h);
				gGF -> SetPort(gOutPort, rect, false);
				sdl_lyrics_resize();
			}
			break;
		}
		if ((STATUS->jump_to_song == 0) ||
			(key == SDLK_RETURN)) { break; }
		if (xmms_remote_get_playlist_pos(sdl_dp.xmms_session) !=
			(STATUS->jump_to_song - 1)) {
		xmms_remote_set_playlist_pos
			(sdl_dp.xmms_session, (STATUS->jump_to_song - 1));
		}
	case SDLK_KP_PERIOD:
		STATUS->jump_to_song = 0;
		break;
	case SDLK_KP0:
	case SDLK_KP1:
	case SDLK_KP2:
	case SDLK_KP3:
	case SDLK_KP4:
	case SDLK_KP5:
	case SDLK_KP6:
	case SDLK_KP7:
	case SDLK_KP8:
	case SDLK_KP9:
	case SDLK_0:
	case SDLK_1:
	case SDLK_2:
	case SDLK_3:
	case SDLK_4:
	case SDLK_5:
	case SDLK_6:
	case SDLK_7:
	case SDLK_8:
	case SDLK_9:
		value = key - SDLK_0;
		if (value < 0 || value > 9)
		value = key - SDLK_KP0;

		if (mod & KMOD_CTRL) {
			GDK_THREADS_ENTER();
			enable_plugin(((value + 9) % 10));
			GDK_THREADS_LEAVE();
		}
		else {
			STATUS->jump_to_song = (STATUS->jump_to_song * 10) + value;
			if (STATUS->jump_to_song > xmms_remote_get_playlist_length(sdl_dp.xmms_session))
				{ STATUS->jump_to_song = value; }
		}
		break;
	default:
		break;
        }
}

extern "C" void sdl_handler_resize(void *pEvent)
{
	gboolean draw = TRUE;
	SDL_ResizeEvent* pResizeEvent = (SDL_ResizeEvent*) pEvent;

	if (!STATUS->is_fullscreen)
		{ draw = CreateScreen(pResizeEvent->w, pResizeEvent->h, false); }

	if (draw == TRUE) {
		if ( gGF ) {
			Rect rect;
			gOutPort->Init(STATUS->screen);
			SetRect( &rect, 0, 0, STATUS->screen->w,
				STATUS->screen->h);
			gGF -> SetPort(gOutPort, rect, false);
		}
		sdl_lyrics_resize();
	}
}

extern "C" void displayer_sdl_render()
{
	long time = EgOSUtils::CurTimeMS();

	gGF -> RecordSample( time, gSoundBuf, .000023, numsamp );
}

extern "C" void displayer_sdl_render_pcm(gint16 pcm_data[2][512], SingitSoundPrecalcs *ssp)
{
	int i, j;
//	long k;
//	SDL_Event sdl_event;

	// Store samples in a buffer that the render thread will eventually get to
	j = 0;
	for (i = 0; i < NUMSAMPLES; i++) {
		gSoundBuf[ i ] = pcm_data[0][j];
#ifdef SAMPSKIP
		j += 1 + SAMPSKIP;
#else
		j++;
#endif
	}
	numsamp = NUMSAMPLES;
/*
	if ((STATUS->screen == SDL_GetVideoSurface()) &&
		(STATUS->timeout != 0))
	{
		displayer_sdl_render();
		SDL_UpdateRect(STATUS->screen, 0, 0, 0, 0);
	}
*/
}

extern "C" void displayer_sdl_new_song_signal(gchar *title, gchar* artist)
{
	if ((title != NULL) || (artist != NULL)) {
		if (title != NULL) {
			gGF -> mSongTitle.Wipe();
			gGF -> mSongTitle.Assign( title );
		}
		if (artist != NULL) {
			gGF -> mArtist.Wipe();
			gGF -> mArtist.Assign( artist );
		}

		// Send the signal
		gGF -> NewSong();
	}
}

extern "C" void sdl_handler_active(void *pEvent)
{
	if (STATUS->is_fullscreen == TRUE)
		{ sdl_helper_gain_focus(); }
}

extern "C" SDL_Color getPaletteColor(const gint index)
{
	SDL_Color result = { 255, 255, 255, 0 };

	if (gGF) { result = gGF->GetPaletteColor(index); }
	return result;
}

extern "C" void displayer_sdl_set_active_song_line(gchar *line)
{
	if (gGF != NULL)
		gGF -> SetActiveSongLine( line );
}
