
#include "VideoDevice.h"

#include "PixPort.h"

VideoDevice::VideoDevice( bool inForceFullRedraws ) {

	Init();

	mForceFullRedraws = inForceFullRedraws;
}

void VideoDevice::Init() {

	mAvgFrameRate	= 200; // 20 FPS
	mDesiredFPS		= 200; // 20 FPS
	mFrameCount		= 0;
	mT = - 0x7FFFFFFF;
	mPrevFrameT = 0;
	mNeedsRefresh = true;
	mMinDuration = 0;
	mFPS_Steady = false;
	mBeginFrameCalled = false;
	mPreFrameCalled = false;

#if EG_MAC
	mCurPriority = 0;
#else
	mCurPriority = 45;
#endif

	mOrigin.h = 0;
	mOrigin.v = 0;

#if EG_MAC
	mPrevDev = nil;
	mPrevPort = 0;
#endif

	InitFPS();

	__invalidate( mOutputRect )
}

void VideoDevice::Erase( bool ) {

	mNeedsRefresh = true;
}

void VideoDevice::AcceptFrame( PixPort& inFrame, const Rect* inDirtyRect, long inT_MS ) {

	Rect srceRect, destRect;
	long curFrameRate;

	// If something is weird, reset the timer
	if ( inT_MS < mT )
		mT = inT_MS - mMinDuration;

	if ( inT_MS >= mT + mMinDuration ) {

		mPrevFrameT = mT;
		mT = inT_MS;

		if ( ! inDirtyRect || mForceFullRedraws )
			SetRect( &srceRect, 0, 0, inFrame.GetX(), inFrame.GetY() );
		else
			srceRect = *inDirtyRect;

		// Setup the port
		PreFrame( &inFrame );

		if ( mNeedsRefresh ) {
			Erase( true );
			mNeedsRefresh = false;
		}

		// Clip the source rect based on the output rect
		OffsetRect( srceRect, &destRect, mOutputRect.left + mOrigin.h, mOutputRect.top + mOrigin.v );
		IntersectRect( &destRect, &mOutputRect, &destRect );
		OffsetRect( destRect, &srceRect, - mOutputRect.left - mOrigin.h, - mOutputRect.top - mOrigin.v );

		// Actually copy pixels from one place to another
		TransferBits( inFrame, srceRect, destRect );

		// Maintain the frame rate
		mFrameCount++;
		if ( mFrameCountStart == 0xDEADBEEF )
			mFrameCountStart = mT;
		else if ( mT - mFrameCountStart >= 600 ) {  // Redo a frame rate every .6 secs
			curFrameRate = 10000 * mFrameCount / ( mT - mFrameCountStart );
			mFrameCountStart = mT;
			mFrameCount = 0;

			// Limit how much severe changes affect avg frame rate (by 14 fps)
			if ( mAvgFrameRate - curFrameRate > 140 )
				curFrameRate = mAvgFrameRate - 140;
			else if ( curFrameRate - mAvgFrameRate > 140 )
				curFrameRate = mAvgFrameRate + 140;

			// Keep a longer running average of frame rate
			mAvgFrameRate = ( 3 * mAvgFrameRate + curFrameRate ) / 4;

			// Calculate a new priority
			FPS_Notify();

			// Is frame rate is steady (recent FPS can't vary more than 4 fps)?
			if ( _ABS( mAvgFrameRate - curFrameRate ) < 40 )
				mFPS_Steady = true;
			else
				mFPS_Steady = false;
		}
	}

}



void VideoDevice::BeginFrame() {

	if ( ! mBeginFrameCalled ) {

		#if EG_MAC
		::GetGWorld( (GWorldPtr*)(&mPrevPort), &mPrevDev );
		#endif

		BeginFrame_Self();

		mBeginFrameCalled = true;
	}
}



void VideoDevice::PreFrame( PixPort* inSrceFrame ) {

	if ( ! mBeginFrameCalled )
		BeginFrame();

	if ( mBeginFrameCalled && ! mPreFrameCalled ) {

		PreFrame_Self( inSrceFrame );

		mPreFrameCalled = true;
	}
}




void VideoDevice::EndFrame() {

	if ( mBeginFrameCalled ) {

		EndFrame_Self();

		#if EG_MAC
		::SetGWorld( (GWorldPtr)(mPrevPort), mPrevDev );
		#endif

		mBeginFrameCalled = false;
		mPreFrameCalled = false;
	}
}





void VideoDevice::Resize( Rect& ) {

}





void VideoDevice::FPS_Notify() {
	long dp;

	// Figure out out much we should change the priority
	long diff = mAvgFrameRate - 10 * mDesiredFPS;
	if ( _ABS( diff ) <= 18 )			// 0-2
		dp = 0;
	else if ( _ABS( diff ) <= 60 )		// 2-9
		dp = 1;
	else if ( _ABS( diff ) <= 150 )		// 10-15
		dp = 2;
	else								// 16+
		dp = 3;

	// Change the priority in the appropriate direction
	if ( diff < 0 )
		dp = - dp;

	// the priority is only from 0 to 100
	if ( mCurPriority + dp < 0 )
		mCurPriority = 0;
	else if ( mCurPriority + dp > 100 )
		mCurPriority = 100;
	else
		mCurPriority += dp;
}






/*


void PluginGlue::DrawFrame( PixPort* inSrcePort, Rect& inDirtyRect ) {


	// If we're fullscreen, follow the API (the screen may need to do something to finish)
	if ( mScreen.IsFullscreen() )
		mOutPort = mScreen.BeginFrame();

	// Someone may have asked to clear the GF window/pane
	if ( mNeedsPaneErased ) {
		ErasePane();
		mNeedsPaneErased = false;
	}

	// Blt our offscreen world to the output device
	Rect r = inDirtyRect;
	::OffsetRect( &r, mDispRect.left, mDispRect.top );
	inSrcePort -> CopyBits( mOutPort, &inDirtyRect, &r );

	// For Sonique and ENTHEOGEN, we've already redirected drawing to a given frame buffer (so there's nothing to copy)
		#if ! SONIQUE && ! ENTHEOGEN
		Rect destRect = r;
		::OffsetRect( &destRect, mPaneRect.left, mPaneRect.top );
		mPort.CopyBits( mOutPort, &r, &destRect );
		#endif

	// If we have an export going, send the frame to the exporter
	#if STANDALONE
	Don't forget to look for a config w/ the same file name
	if ( mStartExport ) {
		mStartExport = false;
		VideoExport_Start( *inSrcePort ); }
	else if ( mVideoExporter.Exporting() )
		mVideoExporter.ExportFrame( *inSrcePort, mNextT_MS );
	#endif

	// If we're fullscreen, follow the API (the screen may need to do something to finish)
	if ( mScreen.IsFullscreen() )
		mScreen.EndFrame();
}


*/


