/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


// Heavily based on code form xmmsctrl.c and controlsocket.c
// from the XMMS package (www.xmms.org)

#include "socket.h"
#include "protocol.h"

#include "singit_debug.h"

#include <stdio.h>

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>

#include <sys/un.h>

#include <errno.h>

static gint session_id = 0;
static gint ctrl_fd = 0;
static gchar *socket_name;

static void *singit_socket_func(void *);
static pthread_t singit_socket_thread;
static gboolean going = TRUE, started = FALSE;

static pthread_mutex_t start_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t start_cond = PTHREAD_COND_INITIALIZER;

gboolean singit_socket_setup(void)
{
	struct sockaddr_un saddr;
	gboolean retval = FALSE;
	gint i;

	if ((ctrl_fd = socket(AF_UNIX, SOCK_STREAM, 0)) != -1)
	{
		for (i = 0;; i++)
		{
			saddr.sun_family = AF_UNIX;
			g_snprintf(saddr.sun_path, 108, "%s/singit_%s.%d",
				   g_get_tmp_dir(), g_get_user_name(), i);
			// if (!xmms_remote_is_running(i))
			if (1)
			{
				if ((unlink(saddr.sun_path) == -1) && errno != ENOENT)
				{
					g_log(NULL, G_LOG_LEVEL_CRITICAL,
					      "setup_singit_socket(): Failed to unlink %s (Error: %s)",
					      saddr.sun_path, strerror(errno));
				}
			}
			else {
//				if (cfg.allow_multiple_instances)
//					continue;
				break;
			}
			if (bind(ctrl_fd, (struct sockaddr *) &saddr, sizeof (saddr)) != -1)
			{
				session_id = i;
				listen(ctrl_fd, 100);
				going = TRUE;

				pthread_create(&singit_socket_thread, NULL, singit_socket_func, NULL);
				socket_name = g_strdup(saddr.sun_path);
				retval = TRUE;
				break;
			}
			else
			{
				g_log(NULL, G_LOG_LEVEL_CRITICAL,
				      "setup_singit_socket(): Failed to assign %s to a socket (Error: %s)",
				      saddr.sun_path, strerror(errno));
				break;
			}
		}
	}
	else
		g_log(NULL, G_LOG_LEVEL_CRITICAL,
			"setup_singit_socket(): Failed to open socket: %s",
			strerror(errno));

	if (!retval)
	{
		if (ctrl_fd != -1)
			close(ctrl_fd);
		ctrl_fd = 0;
	}
	return retval;
}

gint singit_socket_get_session_id(void)
{
	return session_id;
}

void singit_socket_cleanup(void)
{
	if (ctrl_fd)
	{
		pthread_mutex_lock(&start_mutex);
		going = FALSE;
		pthread_cond_signal(&start_cond);
		pthread_mutex_unlock(&start_mutex);
		pthread_join(singit_socket_thread, NULL);
		close(ctrl_fd);
		unlink(socket_name);
		g_free(socket_name);
		ctrl_fd = 0;
	}
}

void singit_socket_start(void)
{
	pthread_mutex_lock(&start_mutex);
	started = TRUE;
	pthread_cond_signal(&start_cond);
	pthread_mutex_unlock(&start_mutex);
}

void singit_socket_packet_write(PacketNode *pkt, gpointer data, gint length)
{
	ServerPktHeader pkthdr;

	pkthdr.version = SINGIT_PROTOCOL_VERSION;
	pkthdr.data_length = length;
	write(pkt->fd, &pkthdr, sizeof (ServerPktHeader));
	if (data && length > 0)
		write(pkt->fd, data, length);
}

void singit_socket_packet_ack(PacketNode * pkt)
{
	singit_socket_packet_write(pkt, NULL, 0);
	close(pkt->fd);
	if (pkt->data)
		g_free(pkt->data);
	g_free(pkt);
}

void *singit_socket_func(void *arg)
{
	fd_set set;
	struct timeval tv;
	struct sockaddr_un saddr;
	gint fd;
	PacketNode *pkt;
	gint len;

	pthread_mutex_lock(&start_mutex);
	while (!started && going)
		pthread_cond_wait(&start_cond, &start_mutex);
	pthread_mutex_unlock(&start_mutex);

	while (going)
	{
		FD_ZERO(&set);
		FD_SET(ctrl_fd, &set);
		tv.tv_sec = 0;
		tv.tv_usec = 100000;
		len = sizeof (saddr);
		if ((select(ctrl_fd + 1, &set, NULL, NULL, &tv) <= 0) ||
		    ((fd = accept(ctrl_fd, (struct sockaddr *) &saddr, &len)) == -1))
			continue;

		pkt = g_malloc0(sizeof (PacketNode));
		read(fd, &pkt->hdr, sizeof (ClientPktHeader));
		if (pkt->hdr.data_length)
		{
			pkt->data = g_malloc0(pkt->hdr.data_length);
			read(fd, pkt->data, pkt->hdr.data_length);
		}
		pkt->fd = fd;
		protocoll_process_unlocked(pkt);
	}

	pthread_exit(NULL);
}

gint singit_socket_connect_to_session(gint session)
{
	gint fd;
	uid_t stored_uid, euid;
	struct sockaddr_un saddr;

	if ((fd = socket(AF_UNIX, SOCK_STREAM, 0)) != -1)
	{
		saddr.sun_family = AF_UNIX;
		stored_uid = getuid();
		euid = geteuid();
		setuid(euid);
		sprintf(saddr.sun_path, "%s/singit_%s.%d",
			g_get_tmp_dir(), g_get_user_name(), session);
		setreuid(stored_uid, euid);

		if (connect(fd, (struct sockaddr *) &saddr, sizeof (saddr)) != -1)
			{ return fd; }
	}

	close(fd);
	return -1;
}
