
#include "MirroredFileSys.h"

#include "CEgIOFile.h"
#include "EgOSUtils.h"
#include "XLongList.h"

#define   MIR_FILE_SYS_LOADED	FILE_SYS_USER_FLAG1

// Define for printf debug
// #define DEBUG_MFS 1
#ifdef DEBUG_MFS
#include <stdio.h>
#endif

MirroredFileSys::MirroredFileSys() :
	GeneralFileSys( true ),
	mCache( true ) {

}

void MirroredFileSys::SetRootDir( const char* inFolder ) {

	mRootFolder.AssignPathName( inFolder );
}

void MirroredFileSys::SetRootDir( CEgFileSpec& inFolder ) {

	mRootFolder.Assign( inFolder );
}

FileResult MirroredFileSys::GetItemID( FileObj inParentID, const UtilStr& inTitle, FileObj& outID )	{

	long flags = mCache.GetInfo( inParentID, MCC4_TO_INT("flag") );
	long result;
	UtilStr str;

	if ( flags ) {
		// If the folder isn't loaded, go to the OS to see what's in it
		if ( ( flags & MIR_FILE_SYS_LOADED ) == 0 )
			ScanFolder( inParentID, nil );

		result = mCache.GetItemID( inParentID, inTitle, outID );

		if ( result != FILE_SYS_SUCCESS )
			ThrowErr( mCache );
	}
	else {
		ThrowErr( FILE_SYS_PATH_NOT_FOUND,
			0, inParentID, "Parent folder not found" );
	}

	return result;
}

FileResult MirroredFileSys::Catalog( FileObj inID, XLongList& outList ) {

	long flags;
	bool ok = false;

	flags = mCache.GetInfo( inID, MCC4_TO_INT("flag") );
	if ( flags ) {
		// If the folder isn't loaded, go to the OS to see what's in it
		if ( ( flags & MIR_FILE_SYS_LOADED ) == 0 )
			ok = ScanFolder( inID, &outList );
		else {
			if ( mCache.Catalog( inID, outList ) )
				ok = true;
			else
				ThrowErr( mCache );
		}

		if ( ok )
			return FILE_SYS_SUCCESS;
	}
	else
		ThrowErr( mCache );

	return FILE_SYS_FAIL;
}

const UtilStr* MirroredFileSys::Read( FileObj inObject ) {

	long flags;
	const UtilStr* buf;

	flags = mCache.GetInfo( inObject, MCC4_TO_INT("flag") );
	if ( flags ) {

		// If the folder isn't loaded, load it from the OS.  Put no limit on the size.
		if ( ( flags & MIR_FILE_SYS_LOADED ) == 0 ) {
			Cache( inObject, 0, 0x7FFFFFFF );
		}

		buf = mCache.Read( inObject );
		if ( buf )
			return buf;
		else
			ThrowErr( mCache ); }
	else
		ThrowErr( FILE_SYS_FNF, inObject );

	return nil;
}

FileResult MirroredFileSys::Read( FileObj inID, UtilStr& outBuf ) {

	const UtilStr* data = Read( inID );

	if ( data ) {
		outBuf.Assign( data );
		return FILE_SYS_SUCCESS;
	}
	else
		return FILE_SYS_FAIL;
}

FileResult MirroredFileSys::Create( FileObj inParent, const UtilStr& inName, FileObj& outID, long inFlags ) {

	// Try to create the item in the cache
	FileResult result = mCache.Create( inParent, inName, outID, inFlags | FILE_SYS_REPLACE );

	// Now create the file on disk...
	if ( result )
		result = WriteData( outID, (void*) "Everone gets laid", 0 );

	if ( result != FILE_SYS_SUCCESS )
		ThrowErr( mCache );

	return result;
}

FileResult MirroredFileSys::WriteData( FileObj inID, void* inData, long inNumBytes ) {

	FileResult result = FILE_SYS_FAIL;
	CEgIOFile oFile;
	CEgFileSpec spec;
	long flags = FileObjToSpec( inID, spec );

	// Only allow writing to files
	if ( flags & FILE_SYS_DATA ) {

		// Make sure we don't desync the cache
		PurgeCached( inID, 0 );

		// Open the file on disk for writing...
		spec.SetType( MCC4_TO_INT("TEXT") );
		oFile.open( &spec );
		if ( oFile.noErr() ) {

			oFile.PutBlock( inData, inNumBytes );

			// Did it write OK?
			if ( oFile.noErr() )
				result = FILE_SYS_SUCCESS;
			else
				ThrowErr( FILE_SYS_WRITE_ERR, oFile.getOSErr(), inID, nil ); }

		// Couldn't find the file on disk?
		else if ( oFile.getErr() == cFileNotFound )
			ThrowErr( FILE_SYS_FNF, inID );

		// Couldn't open it for some other reason?
		else
			ThrowErr( FILE_SYS_OPEN_ERR, inID );

		oFile.close();
	}
	else if ( flags )
		ThrowErr( FILE_SYS_FILE_NEEDED, inID );
	else
		ThrowErr( mCache );

	return result;
}

FileResult MirroredFileSys::ScanFolder( FileObj inFolderID, XLongList* outList ) {

	CEgFileSpec spec, OSFolder;
	long index, flags, type = FileObjToSpec( inFolderID, OSFolder );
	FileObj itemID;
	XLongList oldCatalog;
	FileResult retVal = FILE_SYS_FAIL;

#ifdef DEBUG_MFS
	printf( "Scan: %s (%i)\n", (char*) OSFolder.OSSpec(), type );
#endif

	if ( type & FILE_SYS_FOLDER ) {

		// To find out what's in the cache we have to see what's in the cache but not in the dir...
		mCache.Catalog( inFolderID, oldCatalog );

		// Step thru each file in the given OS dir and create a corresponding file
		index = OSU_FIRST_FILE;

		while ( EgOSUtils::GetNextFile( OSFolder, spec, OSU_DO_FILES | FILE_SYS_FOLDER, &index ) ) {

			if ( spec.Exists() == 2 )
				flags = FILE_SYS_FOLDER;
			else
				flags = FILE_SYS_DATA;

			// Make an new mem cache item with the file's name.
			// Note: we don't give replace permission b/c we don't want to overwrite what's already been loaded
			mCache.Create( inFolderID, *spec.GetFileName(), itemID, flags );

#ifdef DEBUG_MFS
			printf( "Item: %s (%li)\n", (char*) spec.GetFileName()->getCStr(), (long) itemID );
#endif
			if ( outList )
				outList -> Add( itemID );

			// Figure out what files are in the cache but deleted on disk by removing existing items from the old cat list.
			oldCatalog.Remove( itemID );
		}

		// Delete items from the cache that weren't found on the disk...
		while ( oldCatalog.FetchLast( (long*) &itemID ) ) {
			mCache.Delete( itemID );
			oldCatalog.RemoveLast();
		}

		// Store that this folder has been scanned.
		mCache.SetInfo( inFolderID, MCC4_TO_INT("flag"), MIR_FILE_SYS_LOADED );

		retVal = FILE_SYS_SUCCESS;
	}

	else if ( type == 0 )
		ThrowErr( FILE_SYS_FNF, inFolderID );

	else
		ThrowErr( FILE_SYS_FOLDER_NEEDED, inFolderID );

	return retVal;
}

// Try to reimplement FileObjToSpec() -- it's currently pretty wasteful
long MirroredFileSys::FileObjToSpec( FileObj inID, CEgFileSpec& outSpec ) {

	long flags, outFlags = 0;
	UtilStr pathname;

	flags = mCache.GetInfo( inID, MCC4_TO_INT("flag") );

	if ( flags ) {
		if ( mCache.GetPathname( inID, pathname, FILE_SYS_SEP_CHAR ) ) {
			outSpec.AssignPathName( pathname.getCStr(), &mRootFolder );
			outFlags = flags;
		}
		else {
			ThrowErr( mCache );
		}
	}
	else
		ThrowErr( mCache );

	return outFlags;
}

void MirroredFileSys::Cache( FileObj inID, long inNumSubLevels, long inMaxSize ) {

	long len, flags;
	char* ptr;
	bool ok = false;

	flags = mCache.GetInfo( inID, MCC4_TO_INT("flag") );
	if ( flags ) {

		// If the item is a folder, cache the whole folder
		if ( flags & FILE_SYS_FOLDER && inNumSubLevels >= 0 ) {
			XLongList catalog;

			// Go to the OS to get the catalog
			if ( ScanFolder( inID, &catalog ) ) {

				if ( inNumSubLevels > 0 ) {
					for ( long i = 0; i < catalog.Count(); i++ ) {
						Cache( catalog[ i ], inNumSubLevels - 1, inMaxSize );

						// If there's a zillion items, we'll want to let the user know we're not frozen
						EgOSUtils::SpinCursor();
					}
				}
			}
		}
				// If the item is a file, read it in
		else if ( flags & FILE_SYS_DATA ) {
			CEgFileSpec spec;

			if ( FileObjToSpec( inID, spec ) ) {

				// Read the item from disk and store it in memory under the file's name
				mTempFile.open( &spec );
				if ( mTempFile.noErr() ) {
					len = mTempFile.size();
					if ( len <= inMaxSize ) {
						ptr = mTemp.Dim( len );
						mTempFile.GetBlock( ptr, len );
						if ( mTempFile.noErr() ) {
							mCache.WriteData( inID, ptr, len );
							mCache.SetInfo( inID, MCC4_TO_INT("flag"), MIR_FILE_SYS_LOADED );
							ok = true;
						}
					}
				}
				mTempFile.close();
			}

			// If we didn't cache the file, clear the LOADED flag
			if ( ! ok )
				mCache.SetInfo( inID, MCC4_TO_INT("flag"), 0 );
		}
	}
}

void MirroredFileSys::PurgeCached( FileObj inID, long inNumSubLevels ) {

	long flags;

	flags = mCache.GetInfo( inID, MCC4_TO_INT("flag") );

	// If the item is a folder, cache the whole folder
	if ( flags & FILE_SYS_FOLDER && inNumSubLevels >= 0 ) {
		XLongList catalog;

		// Go thru what we may have cached and purge that as well
		if ( mCache.Catalog( inID, catalog ) ) {

			for ( long i = 0; i < catalog.Count(); i++ ) {
				PurgeCached( catalog[ i ], inNumSubLevels - 1 );
			}
		}
	}

	// If the item is a file, we don't need to do anything more
	// to a file other than clear the MIR_FILE_SYS_LOADED flag

	if ( flags & MIR_FILE_SYS_LOADED ) {
		// Clear the MIR_FILE_SYS_LOADED flag
		flags &= ~ MIR_FILE_SYS_LOADED;
		mCache.SetInfo( inID, MCC4_TO_INT("flag"), flags  );
	}
}
