/*
 *	$COPYRIGHT$
 *
 *	$Id: xmpi_view.cc,v 1.5 2000/11/09 01:03:15 bbarrett Exp $
 *
 *	Function:	- XMPI main view
 */

#include <X11/Xlib.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/ScrolledW.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "lam.h"
#include "xmpi.h"
#include "xmpi_view.h"
#include "xpm.h"
#include "xmpi_focus.h"
#include "xmpi_trace.h"
#include "xmpi_misc.h"
#include "xmpi_trace.h"
#include "xmpi_dbase.h"
#include "xmpi_matrix.h"

extern "C" {
#include "Pixmaps/lamlogo.xpm"
#include "Pixmaps/lam-logo.xpm"
#include "Bitmaps/light.xbm"
#include "Bitmaps/red.xbm"
#include "Bitmaps/yellow.xbm"
#include "Bitmaps/green.xbm"
#include "Bitmaps/mesg.xbm"
}

/*
 * private functions
 */
static void gcinit();

static void draw_cb(Widget, XtPointer, XmDrawingAreaCallbackStruct*);

static void placeprocs(int, struct _gps*);

static void plotproc(int, int, int);

static void plotprocinfo(int);

static void drawclear();

static void setlogo();

static int xy2rank(int, int);

/*
 * local macros and defines
 */
#define DRAWWIDTH(x)	((int) ((2 * XMPI_VWXMARGIN) + \
				((3 * (x) + 1) * XMPI_VWRADIUS / 2)))
#define DRAWHEIGHT(y)	((int) ((2 * XMPI_VWYMARGIN) + \
			((2 * (y) + 1) * ((SQRT3 * XMPI_VWRADIUS / 2) + 0.5))))

/*
 * global variables
 */
int fl_dbase = 0;		       /* driven by dbase flag */

/*
 * external variables
 */
extern int xmpi_app_nprocs;	       /* number of procs in appl. */

extern struct _gps *xmpi_app_procs;    /* appl. GPS array */

/*
 * local variables
 */
static int vw_nprocs = 0;	       /* # of processes */

static struct xmproc *vw_procs = 0;    /* array of processes */

static int vw_nxprocs;		       /* # procs horizontally */

static int vw_nyprocs;		       /* # procs vertically */

static int drawwidth;		       /* drawing area width */

static int drawheight;		       /* drawing area height */

static int fl_replotcomm = 0;	       /* comm. clear/replot flag */

static int logowidth;		       /* logo pixmap width */

static int logoheight;		       /* logo pixmap height */

static char **vendor_xpm;	       /* vendor logo XPM format */

static int vendorwidth;		       /* vendor pixmap width */

static int vendorheight;	       /* vendor pixmap height */

static char fmtbuf[40];		       /* formatting buffer */

static unsigned int depth;	       /* screen depth */

static Widget draw_w;		       /* drawing area widget */

static Display *disp;		       /* drawing area display */

static Drawable root;		       /* root window */

static GC fgpen;		       /* foreground pen */

static GC bgpen;		       /* background pen */

static GC whitepen;		       /* white pen */

static GC boldpen;		       /* bold pen */

static GC lcompen;		       /* communicator pen (local) */

static GC rcompen;		       /* communicator pen (remote) */

static Pixmap pixmap;		       /* copy of drawing area */

static Pixmap blkpix = XmUNSPECIFIED_PIXMAP;

static Pixmap syspix = XmUNSPECIFIED_PIXMAP;

static Pixmap runpix = XmUNSPECIFIED_PIXMAP;

static Pixmap mesgpix = XmUNSPECIFIED_PIXMAP;

static Pixmap logopix = XmUNSPECIFIED_PIXMAP;

static Pixmap vendorpix = XmUNSPECIFIED_PIXMAP;

/*
 *	xmpi_vw_create
 *
 *	Function:	- create process viewing subwindow
 *	Accepts:	- parent widget
 *	Returns:	- created widget
 */
Widget
xmpi_vw_create(Widget parent_w)
{
  Widget view_w;		       /* subwindow widget */

  Widget scroll_w;		       /* scrolled window widget */

  view_w = XtVaCreateWidget("view_mgr",
			    xmFormWidgetClass, parent_w,
			    NULL);
/*
 * Build the scrolled window.
 */
  vw_nxprocs = XMPI_VWXNPROCS;
  vw_nyprocs = XMPI_VWYNPROCS;

  drawwidth = DRAWWIDTH(vw_nxprocs);
  drawheight = DRAWHEIGHT(vw_nyprocs);

  scroll_w = XtVaCreateManagedWidget("view_scrwin",
				     xmScrolledWindowWidgetClass, view_w,
				     XmNwidth, drawwidth + XMPI_VWXWIDTH,
				  XmNheight, drawheight + XMPI_VWXHEIGHT,
				     XmNscrollingPolicy, XmAUTOMATIC,
				     XmNspacing, 0,
				     XmNtraversalOn, False,
				     XmNtopAttachment, XmATTACH_FORM,
				     XmNbottomAttachment, XmATTACH_FORM,
				     XmNleftAttachment, XmATTACH_FORM,
				     XmNrightAttachment, XmATTACH_FORM,
				     NULL);

  xmpi_setminmax(DRAWWIDTH(1), DRAWHEIGHT(1), drawwidth, drawheight);
/*
 * Build the drawing area.
 */
  draw_w = XtVaCreateManagedWidget("view_draw",
				   xmDrawingAreaWidgetClass, scroll_w,
				   XmNwidth, drawwidth,
				   XmNheight, drawheight,
				   NULL);

  XtAddCallback(draw_w, XmNinputCallback, 
		(XtCallbackProc) draw_cb, NULL);
  XtAddCallback(draw_w, XmNexposeCallback, 
		(XtCallbackProc) draw_cb, NULL);

  XtManageChild(draw_w);
  XtManageChild(scroll_w);

  XtManageChild(view_w);
/*
 * Create the graphic contexts.
 */
  gcinit();
/*
 * Create the LAM logo and the vendor logo.
 * White out the background just for the LAM logo.  It's good to be the
 * king.  :-)
 */
  xpm_build(draw_w, NDlamlogo, (Pixel) 1, &logopix, &logowidth,
	    &logoheight);

  if ((vendor_xpm = xmpi_sys_logo())) {
    xpm_build(draw_w, (const char**) vendor_xpm, (Pixel) 1, &vendorpix,
	      &vendorwidth, &vendorheight);
  }
  setlogo();

  if (logopix != XmUNSPECIFIED_PIXMAP) {
    xmpi_seticon(logopix);
  }
  return (view_w);
}

/*
 *	xmpi_vw_clear
 *
 *	Function:	- clean up the view and focus windows
 */
void
xmpi_vw_clear()
{
  int i;

  struct xmproc *p;

/*
 * Destroy focus windows and message lists.
 */
  xmpi_fo_destroy_all();

  for (i = 0, p = vw_procs; i < vw_nprocs; ++i, ++p) {
    if (p->xmp_msgs) {
      al_free(p->xmp_msgs);
    }
  }
/*
 * Reset the view.
 */
  drawclear();
  setlogo();

  XCopyArea(disp, pixmap, XtWindow(draw_w), fgpen,
	    0, 0, drawwidth, drawheight, 0, 0);
  fl_replotcomm = 0;

  vw_nprocs = 0;
  if (vw_procs) {
    free((char *) vw_procs);
    vw_procs = 0;
  }
}

/*
 *	gcinit
 *
 *	Function:	- create graphic contexts
 */
static void
gcinit()
{
  Pixel fg, bg;			       /* fore/background colours */

  GC pen;			       /* temporary pen */

  XGCValues gcval;		       /* graphic context info */

  XColor col_def;		       /* colour definition */

  XColor dev_null;		       /* unused functionality */

  Pixmap blkbm;			       /* blocked process bitmap */

  Pixmap sysbm;			       /* system overhead bitmap */

  Pixmap runbm;			       /* running process bitmap */

  disp = XtDisplay(draw_w);
  root = RootWindowOfScreen(XtScreen(draw_w));
  depth = DefaultDepthOfScreen(XtScreen(draw_w));

  XtVaGetValues(draw_w, XmNforeground, &fg, XmNbackground, &bg, NULL);
/*
 * Create the foreground, background, and bold pens.
 */
  gcval.line_width = XMPI_VWLNWIDTH;
  gcval.foreground = bg;
  gcval.background = fg;
  gcval.font = app_res.ap_msgfont->fid;

  fgpen = XCreateGC(disp, root,
		    GCFont | GCLineWidth | GCForeground | GCBackground,
		    &gcval);

  gcval.foreground = fg;
  bgpen = XCreateGC(disp, root, GCForeground, &gcval);

  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
		   XMPI_LBLCOLOUR, &col_def, &dev_null);

  gcval.foreground = col_def.pixel;
  gcval.font = app_res.ap_rankfont->fid;
  boldpen = XCreateGC(disp, root,
		      GCFont | GCForeground | GCBackground, &gcval);
/*
 * Create a white pen so that we can display the white-backed LAM logo nicely
 */
  gcval.foreground = ~0;
  whitepen = XCreateGC(disp, root, GCForeground, &gcval);

/*
 * Create the communicator local group pen.
 */
  gcval.foreground = app_res.ap_lcomcol;
  lcompen = XCreateGC(disp, root,
		      GCFont | GCLineWidth | GCForeground | GCBackground,
		      &gcval);
/*
 * Create the communicator remote group pen.
 */
  gcval.foreground = app_res.ap_rcomcol;
  rcompen = XCreateGC(disp, root,
		      GCFont | GCLineWidth | GCForeground | GCBackground,
		      &gcval);
/*
 * Create the blocked process (red) traffic light pixmap.
 */
  blkbm = XCreateBitmapFromData(disp, root,
			 (const char *) red_bits, red_width, red_height);

  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
		   XMPI_BLKCOLOUR, &col_def, &dev_null);

  gcval.foreground = col_def.pixel;
  gcval.background = fg;
  gcval.clip_mask = blkbm;
  pen = XCreateGC(disp, root,
		  GCForeground | GCBackground | GCClipMask, &gcval);

  blkpix = XCreatePixmapFromBitmapData(disp, root,
	  (char *) light_bits, light_width, light_height, bg, fg, depth);

  XCopyPlane(disp, blkbm, blkpix, pen, 0, 0,
	     red_width, red_height, 0, 0, (unsigned long) 1);

  XFreePixmap(disp, blkbm);
/*
 * Create the system overhead (yellow) traffic light pixmap.
 */
  sysbm = XCreateBitmapFromData(disp, root,
		(const char *) yellow_bits, yellow_width, yellow_height);

  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
		   XMPI_SYSCOLOUR, &col_def, &dev_null);

  gcval.foreground = col_def.pixel;
  gcval.clip_mask = sysbm;
  XChangeGC(disp, pen, GCForeground | GCClipMask, &gcval);

  syspix = XCreatePixmapFromBitmapData(disp, root,
	  (char *) light_bits, light_width, light_height, bg, fg, depth);

  XCopyPlane(disp, sysbm, syspix, pen, 0, 0,
	     yellow_width, yellow_height, 0, 0, (unsigned long) 1);

  XFreePixmap(disp, sysbm);
/*
 * Create the running process (yellow) traffic light pixmap.
 */
  runbm = XCreateBitmapFromData(disp, root,
		   (const char *) green_bits, green_width, green_height);

  XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
		   XMPI_RUNCOLOUR, &col_def, &dev_null);

  gcval.foreground = col_def.pixel;
  gcval.clip_mask = runbm;
  XChangeGC(disp, pen, GCForeground | GCClipMask, &gcval);

  runpix = XCreatePixmapFromBitmapData(disp, root,
	  (char *) light_bits, light_width, light_height, bg, fg, depth);

  XCopyPlane(disp, runbm, runpix, pen, 0, 0,
	     green_width, green_height, 0, 0, (unsigned long) 1);

  XFreePixmap(disp, runbm);

  XFreeGC(disp, pen);
/*
 * Create the message pixmap.
 */
  mesgpix = XCreatePixmapFromBitmapData(disp, root,
	     (char *) mesg_bits, mesg_width, mesg_height, bg, fg, depth);
/*
 * Create the drawing pixmap.
 */
  pixmap = XCreatePixmap(disp, root, drawwidth, drawheight, depth);
}

/*
 *	draw_cb
 *
 *	Function:	- callback for drawing area input/expose events
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
draw_cb(Widget w, XtPointer, XmDrawingAreaCallbackStruct *p_callback)
{
  XEvent *event;		       /* activating event */
  int rank;			       /* process rank */
  struct xmproc *proc;		       /* ptr process entry */

  event = p_callback->event;
/*
 * Toggle the process focus window.
 */
  if ((p_callback->reason == XmCR_INPUT) &&
      (event->xany.type == ButtonRelease)) {

    rank = xy2rank(event->xbutton.x, event->xbutton.y);
    if (rank >= 0) {
      proc = vw_procs + rank;

      xmpi_fo_toggle(w, rank, proc);
      xmpi_fo_update(rank);
    }
  }
/*
 * Handle the expose event.
 */
  else if (p_callback->reason == XmCR_EXPOSE) {
    XCopyArea(event->xany.display, pixmap, event->xany.window,
	      fgpen, 0, 0, drawwidth, drawheight, 0, 0);
  }
}

/*
 *	xmpi_vw_update
 *
 *	Function:	- update the process view
 *	Accepts:	- update focus flag
 */
void
xmpi_vw_update(int fl_focus)
{
  int i;			       /* favourite index */
  int changed = 1;		       /* process info changed? */

  if (vw_nprocs == 0)
    return;
/*
 * Get process status information and plot it.
 */
  if (fl_dbase) {
    changed = xmpi_db_getstat(vw_nprocs, vw_procs);
  } else {
    xmpi_sys_snapshot(xmpi_app_procs, vw_nprocs, vw_procs);
  }

  if ((!xmpi_tr_vcron()) || changed) {
    drawclear();
    fl_replotcomm = 0;

    for (i = 0; i < vw_nprocs; ++i) {
      plotproc(i, -1, 0);
    }

    for (i = 0; i < vw_nprocs; ++i) {
      plotprocinfo(i);
    }

    XCopyArea(disp, pixmap, XtWindow(draw_w), fgpen,
	      0, 0, drawwidth, drawheight, 0, 0);
/*
 * Update all focus windows if needed.
 */
    if (fl_focus) {
      for (i = 0; i < vw_nprocs; ++i) {
	xmpi_fo_update(i);
      }
    }
/*
 * Update matrix window.
 */
    xmpi_mat_update(vw_procs);
  }
}

/*
 *	xmpi_vw_dbsetmode
 *
 *	Function:	- set the database control mode
 *	Accepts:	- dbase flag
 */
void
xmpi_vw_dbsetmode(int dbflag)
{
  fl_dbase = dbflag;
}

/*
 *	xmpi_vw_dbmode
 *
 *	Function:	- query the database control mode
 *	Returns:	- database control modedbase
 */
int
xmpi_vw_dbmode()
{
  return (fl_dbase);
}

/*
 *	xmpi_vw_getnprocs
 *
 *	Function:	- get number of processes
 *	Returns:	- number of processes in view window
 */
int
xmpi_vw_getnprocs()
{
  if (fl_dbase)
    return (xmpi_db_getnprocs());
  else
    return (vw_nprocs);
}

/*
 *	xmpi_vw_getprocs
 *
 *	Function:	- gets process status information
 *	Returns:	- process status array
 */
struct xmproc *
xmpi_vw_getprocs()
{
  return (vw_procs);
}


/*
 *	xmpi_vw_init
 *
 *	Function:	- initialize the view window
 */
void
xmpi_vw_init()
{
  int nprocs;			       /* # processes */

  struct _gps *procs;		       /* gps array */

  int i;			       /* favourite index */

/*
 * Clean out old view data.
 */
  xmpi_vw_clear();
/*
 * Load process information.
 */
  if (fl_dbase) {
    nprocs = xmpi_db_getnprocs();

    procs = (struct _gps *) malloc(nprocs * sizeof(struct _gps));
    if (procs == 0)
      xmpi_fail((char*) "xmpi (malloc)");

    for (i = 0; i < nprocs; ++i) {
      procs[i].gps_grank = i;
    }
  } else {
    nprocs = xmpi_app_nprocs;
    procs = xmpi_app_procs;
  }
/*
 * Fill the viewing subwindow.
 */
  placeprocs(nprocs, procs);
  xmpi_vw_update(0);
/*
 * When not in database mode we keep the GPS array in case the user
 * later decides to dump traces.
 */
  if (fl_dbase)
    free((char *) procs);

}

/*
 *	xmpi_vw_commplot
 *
 *	Function:	- plot a communicator group
 *	Accepts:	- array of GPS
 *			- # processes in local group
 *			- # processes in remote group
 *			- local/remote is main highlight
 *
 */
void
xmpi_vw_commplot(struct _gps *group, int nlocal, 
		 int nremote, int islocal)
{
  int i;			       /* favourite index */

  struct _gps *p;		       /* favourite pointer */

/*
 * If a communicator is already set, turn off processes.
 */
  if (fl_replotcomm) {
    for (i = 0; i < vw_nprocs; ++i) {
      plotproc(i, -1, 0);
    }
  }
/*
 * Loop plotting the communicator's "remote" then "local" group processes.
 */
  if (islocal) {
    for (i = 0, p = group + nlocal; i < nremote; ++i, ++p)
      plotproc(p->gps_grank, i, 0);
    for (i = 0, p = group; i < nlocal; ++i, ++p)
      plotproc(p->gps_grank, i, 1);
  } else {
    for (i = 0, p = group; i < nlocal; ++i, ++p)
      plotproc(p->gps_grank, i, 0);
    for (i = 0, p = group + nlocal; i < nremote; ++i, ++p)
      plotproc(p->gps_grank, i, 1);
  }

  XCopyArea(disp, pixmap, XtWindow(draw_w), fgpen,
	    0, 0, drawwidth, drawheight, 0, 0);
  fl_replotcomm = 1;
}

/*
 *	placeprocs
 *
 *	Function:	- place the processes in the view
 *			- create the processes data structure
 *	Accepts:	- # of processes
 *			- array of process GPS
 */
static void
placeprocs(int nprocs, struct _gps *pgps)
{
  int x, y;			       /* pixel coordinates */

  int startx;			       /* starting x ordinate */

  int side;			       /* sqrt(3)/2 * radius */

  int delta;			       /* distance between procs */

  int i, j;

  Dimension swidth;		       /* scrolled window width */

  Dimension sheight;		       /* scrolled window height */

  struct xmproc *p;

/*
 * Allocate the process data structure.
 */
  if (vw_procs)
    free((char *) vw_procs);

  vw_procs = (struct xmproc *)
    malloc((unsigned) nprocs * sizeof(struct xmproc));
  if (vw_procs == 0)
    xmpi_fail((char*) "xmpi (malloc)");

  vw_nprocs = nprocs;
/*
 * Layout the processes in a "squarish" area.
 */
  for (i = 0, j = 0; j < vw_nprocs; ++i) {
    j += i + i + 1;
  }

  vw_nxprocs = i;
  vw_nyprocs = (vw_nprocs + vw_nxprocs - 1) / vw_nxprocs;
/*
 * Compute size of new drawing area.
 */
  side = (int) ((SQRT3 * XMPI_VWRADIUS / 2) + 0.5);

  drawwidth = DRAWWIDTH((vw_nxprocs < XMPI_VWXNPROCS) ?
			XMPI_VWXNPROCS : vw_nxprocs);
  drawheight = DRAWHEIGHT((vw_nyprocs < XMPI_VWYNPROCS) ?
			  XMPI_VWYNPROCS : vw_nyprocs);
/*
 * Shrink the scrolled window if needed.
 * Set the min/max top window sizes.
 */
  XtVaGetValues(XtParent(draw_w),
		XmNwidth, &swidth, XmNheight, &sheight, NULL);

  if ((swidth > (drawwidth + XMPI_VWXWIDTH)) ||
      (sheight > (drawheight + XMPI_VWXHEIGHT))) {
    xmpi_setsize(drawwidth, drawheight);
  }
  xmpi_setminmax(DRAWWIDTH(1), DRAWHEIGHT(1), drawwidth, drawheight);
/*
 * Resize the drawing area and its pixmap.
 */
  XtVaSetValues(draw_w, XmNwidth, drawwidth,
		XmNheight, drawheight, NULL);

  XFreePixmap(disp, pixmap);

  pixmap = XCreatePixmap(disp, root, drawwidth, drawheight, depth);
/*
 * Reset the focus position.
 */
  xmpi_fo_reset();
/*
 * Fill the processes data structure.
 */
  startx = XMPI_VWRADIUS + XMPI_VWXMARGIN;
  if (vw_nxprocs < XMPI_VWXNPROCS) {
    startx += (drawwidth - DRAWWIDTH(vw_nxprocs)) / 2;
  }
  x = startx;

  y = side + XMPI_VWYMARGIN;
  if (vw_nyprocs < XMPI_VWYNPROCS) {
    y += (drawheight - DRAWHEIGHT(vw_nyprocs)) / 2;
  }
  delta = 3 * XMPI_VWRADIUS / 2;

  for (i = j = 0, p = vw_procs; i < vw_nprocs; ++i, ++p, ++pgps) {
    p->xmp_cx = x;
    p->xmp_cy = y + ((j & 1) ? side : 0);
    p->xmp_state = XMPI_SUNDEF;
    p->xmp_nmsg = 0;
    p->xmp_more = 0;
    p->xmp_msgs = 0;
    p->xmp_curmsg = 0;
    p->xmp_prog[0] = '\0';

    if (++j == vw_nxprocs) {
      x = startx;
      y += side + side;
      j = 0;
    } else {
      x += delta;
    }
  }
}

/*
 *	plotproc
 *
 *	Function:	- plot the process hexagon
 *	Accepts:	- process global rank
 *			- process local rank (or -1)
 *			- flag, 1=process in local group, 0=remote
 */
static void
plotproc(int grank, int lrank, int local)
{
  int x, y;			       /* hexagon center */
  int side;			       /* hexagon size */
  int strwidth;			       /* string width */
  int strwidth2 = 0;		       /* second string width */
  int ascent, descent;		       /* font vert. sizes */
  XPoint points[7];		       /* hexagon points */
  char buf[16];			       /* temporary buffer */

  /*
   * Draw the process hexagon.
   */
  x = vw_procs[grank].xmp_cx;
  y = vw_procs[grank].xmp_cy;

  side = (int) ((SQRT3 * XMPI_VWRADIUS / 2) + 0.5);

  points[0].x = x - XMPI_VWRADIUS;
  points[0].y = y;
  points[1].x = x - (XMPI_VWRADIUS / 2);
  points[1].y = y - side;
  points[2].x = x + (XMPI_VWRADIUS / 2);
  points[2].y = y - side;
  points[3].x = x + XMPI_VWRADIUS;
  points[3].y = y;
  points[4].x = x + (XMPI_VWRADIUS / 2);
  points[4].y = y + side;
  points[5].x = x - (XMPI_VWRADIUS / 2);
  points[5].y = y + side;
  points[6].x = x - XMPI_VWRADIUS;
  points[6].y = y;

  XDrawLines(disp, pixmap, (lrank < 0) ? fgpen :
	     (local == 1) ? lcompen : rcompen,
	     points, 7, CoordModeOrigin);
/*
 * Compute the width/height of the rank strings.
 */
  sprintf(fmtbuf, "%d", grank);
  strwidth = XTextWidth(app_res.ap_rankfont, fmtbuf, strlen(fmtbuf));

  if (lrank >= 0) {
    sprintf(buf, "/%d", lrank);
    strwidth2 = XTextWidth(app_res.ap_rankfont, buf, strlen(buf));
  }
  ascent = app_res.ap_rankfont->ascent;
  descent = app_res.ap_rankfont->descent;
/*
 * Clear the rank strings region.
 */
  y += side - 6 - descent;

  XFillRectangle(disp, pixmap, bgpen,
		 x - (XMPI_VWRADIUS / 2), y - ascent,
		 XMPI_VWRADIUS, ascent + descent);
/*
 * Draw the rank strings.
 */
  XDrawString(disp, pixmap, boldpen,
	      x - ((strwidth + strwidth2) / 2), y,
	      fmtbuf, strlen(fmtbuf));

  if (lrank >= 0) {
    XDrawString(disp, pixmap, (local == 1) ? lcompen : rcompen,
		x + ((strwidth - strwidth2) / 2), y,
		buf, strlen(buf));
  }
}

/*
 *	plotprocinfo
 *
 *	Function:	- plot process status information
 *	Accepts:	- process rank
 */
static void
plotprocinfo(int rank)
{
  int x, y;			       /* hexagon center */
  int nmsg;			       /* # buffered messages */
  int xnum, ynum;		       /* number coordinates */
  int strwidth;			       /* string width */
  Pixmap pix;			       /* light pixmap */
  struct xmproc *p;		       /* favourite pointer */

  p = vw_procs + rank;
  x = p->xmp_cx;
  y = p->xmp_cy;
/*
 * Plot the traffic light.
 */
  if (p->xmp_state == XMPI_SUNDEF)
    return;

  switch (p->xmp_state) {
  case XMPI_SRUN:
    pix = runpix;
    break;
  case XMPI_SBLOCK:
    pix = blkpix;
    break;
  case XMPI_SSYSTEM:
    pix = syspix;
    break;
  }

  XCopyArea(disp, pix, pixmap, fgpen,
	    0, 0, light_width, light_height,
	    x + XMPI_VWXLIGHT, y + XMPI_VWYLIGHT);
  
  /*
   * Plot the message icon if there are messages.
   */
  if (p->xmp_msgs == 0)
    return;
  if (((nmsg = p->xmp_nmsg) <= 0) && (p->xmp_more == 0))
    return;

  XCopyArea(disp, mesgpix, pixmap, fgpen,
	    0, 0, mesg_width, mesg_height,
	    x + XMPI_VWXMESG, y + XMPI_VWYMESG);
  /*
   * Write the number of messages.
   */
  if (nmsg > 99) {
    strcpy(fmtbuf, "99+");
  } else if (nmsg > 0) {
    sprintf(fmtbuf, "%d", nmsg);
    if (p->xmp_more)
      strcat(fmtbuf, "+");
  } else {
    strcpy(fmtbuf, "?");
  }

  xnum = x + XMPI_VWXNMSG;
  ynum = y + XMPI_VWYNMSG;

  strwidth = XTextWidth(app_res.ap_msgfont, fmtbuf, strlen(fmtbuf));
  xnum -= strwidth / 2;
  ynum += app_res.ap_msgfont->ascent / 2;

  XDrawString(disp, pixmap, fgpen, xnum, ynum, fmtbuf, strlen(fmtbuf));
}

/*
 *	drawclear
 *
 *	Function:	- clear the drawing area
 */
static void
drawclear()
{
  XFillRectangle(disp, pixmap, bgpen, 0, 0, drawwidth, drawheight);
}

/*
 *	setlogo
 *
 *	Function:	- copy logo(s) to drawing area
 */
static void
setlogo()
{
  XFillRectangle(disp, pixmap, whitepen, 0, 0, drawwidth, drawheight);
  if (logopix != XmUNSPECIFIED_PIXMAP) {
    XCopyArea(disp, logopix, pixmap, fgpen,
	      0, 0, logowidth, logoheight,
	      (drawwidth - logowidth) / 2,
	      (drawheight - logoheight) / 2);
  }
  if (vendorpix != XmUNSPECIFIED_PIXMAP) {
    XCopyArea(disp, vendorpix, pixmap, fgpen,
	      0, 0, vendorwidth, vendorheight,
	      0,
	      drawheight - vendorheight);
  }
}

/*
 *	xy2rank
 *
 *	Function:	- map mouse coordinates to process rank
 *	Accepts:	- mouse coordinates
 *	Returns:	- process rank or LAMERROR
 */
static int
xy2rank(int x, int y)
{
  int i;			       /* favourite index */

  int r2;			       /* radius squared */

  int dx, dy;			       /* x & y deltas */

  int dist;			       /* distance to hex. center */

  int mindist;			       /* min. distance */

  int minrank;			       /* rank of min. distance */

  struct xmproc *p;		       /* favourite pointer */

/*
 * Loop finding the closest hexagon center.
 * Reject all distances greater than the radius.
 * Reject non-unique minimal distances.
 */
  r2 = XMPI_VWRADIUS * XMPI_VWRADIUS;
  minrank = LAMERROR;
  mindist = -1;

  for (i = 0, p = vw_procs; i < vw_nprocs; ++i, ++p) {

    dx = x - p->xmp_cx;
    dy = y - p->xmp_cy;

    dist = (dx * dx) + (dy * dy);

    if (dist <= r2) {
      if ((dist < mindist) || (mindist < 0)) {
	mindist = dist;
	minrank = i;
      } else if (dist == mindist) {
	minrank = LAMERROR;
      }
    }
  }

  return (minrank);
}
