/*
 *  X11 battery monitor for PMU-based powerbooks/ibooks.
 *
 *  Copyright (c) 2002  Brendan O'Dea <bod@debian.org>
 *
 *  This is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include "misc.h"
#include "Gauge.h"
#include "Meter.h"
#include "Basic.h"
#include "carp.h"
#include "layout.h"
#include "pmu.h"
#include "version.h"

RCS_Id("$Id: main.c,v 1.1 2002/01/20 07:14:25 bod Exp $")

#define DEFAULT_LAYOUT		"power;time;charge"
#define DEFAULT_POLL_INTERVAL	5000
#define MAX_LABEL		128

#define COPYRIGHT \
    "Copyright (c) 2002  Brendan O'Dea <bod@debian.org>\n" \
    "This is free software, licensed under the terms of the GNU General Public\n" \
    "License.  There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n" \
    "PARTICULAR PURPOSE.\n"

#define HELP \
    "`%s' is an X11 battery monitor for PMU-based powerbooks/ibooks.\n" \
    "\n" \
    "Usage: %s [-toolkitoption ...] [-option ...]\n" \
    "\n" \
    "Options:\n" \
    " -layout LAYOUT   use the widgets described by LAYOUT for the interface\n" \
    " -poll INTERVAL   poll /proc/pmu for changes every INTERVAL milliseconds\n" \
    " -help            print this help, then exit\n" \
    " -version         print version number, then exit\n" \
    "\n" \
    "The interface layout is defined by the -layout option, which defines a set of\n" \
    "values from /proc/pmu presented in rows delimited by `;', and columns by `,'.\n" \
    "Available values are: `current', `voltage', `power', `charge' and `time'.\n" \
    "\n" \
    "By default each value is the sum of all batteries, although appending _N will\n" \
    "include only the value from battery N (`charge_0' for example).\n" \
    "\n" \
    "The widget used to present the value may be defined by adding `(WIDGET)' after\n" \
    "the name, where the available widget types are: `meter', `hgauge', `vgauge'\n" \
    "or `basic'.\n" \
    "\n" \
    "The default layout is `" DEFAULT_LAYOUT "'.\n"

static String fallback_res[] = {  
    "*font:			lucidasans-10",
    "*fontColor:		grey50",
    "*Basic.foreground:		grey50",
    "*Meter.needleColor:	red",
    "*Gauge.foreground:		green",
    "*Gauge.warningColor:	yellow",
    "*Gauge.criticalColor:	red",
    "*current.minValue:		-2000",
    "*current.maxValue:		2000",
    "*current.units:		\\040mA",
    "*current.label:		Current",
    "*voltage.minValue:		0",
    "*voltage.maxValue:		17000",
    "*voltage.scale:		1000",
    "*voltage.precision:	2",
    "*voltage.units:		\\040V",
    "*voltage.label:		Voltage",
    "*power.minValue:		-30000",
    "*power.maxValue:		30000",
    "*power.scale:		1000",
    "*power.precision:		2",
    "*power.units:		\\040W",
    "*power.label:		Power",
    "*charge.units:		%",
    "*charge.label:		Charge",
    "*time.label:		Time left",
    0
};

static void update_info(XtPointer, XtIntervalId *);
static void quit(Widget, XEvent *, String *, Cardinal *);
static XtActionsRec actions[] = {
    { "quit", quit },
};

static struct opts {
    Boolean help;	/* show usage info */
    Boolean version;	/* show version */
    String layout;	/* widget layout */
    int poll;		/* poll interval */
} opts;

#define offset(field) XtOffsetOf(struct opts, field)
static XtResource resources[] = {
    {
	"help", "Help",
	XtRBoolean, sizeof(Boolean),
	offset(help),
	XtRImmediate, (XtPointer) False
    },
    {
	"version", "Version",
	XtRBoolean, sizeof(Boolean),
	offset(version),
	XtRImmediate, (XtPointer) False
    },
    {
	"layout", "Layout",
	XtRString, sizeof(String),
	offset(layout),
	XtRString, DEFAULT_LAYOUT
    },
    {
	"poll", "Poll",
	XtRInt, sizeof(int),
	offset(poll),
	XtRImmediate, (XtPointer) DEFAULT_POLL_INTERVAL
    },
};
#undef offset

static XrmOptionDescRec options[] = {
    { "-layout",	"layout",	XrmoptionSepArg, 0 },
    { "-poll",		"poll",		XrmoptionSepArg, 0 },
    { "-help",		"help",		XrmoptionNoArg,	(XtPointer) "true" },
    { "-version",	"version",	XrmoptionNoArg,	(XtPointer) "true" },

    /* allow GNU-style options as well for these standard options */
    { "--help",		"help",		XrmoptionNoArg,	(XtPointer) "true" },
    { "--version",	"version",	XrmoptionNoArg,	(XtPointer) "true" },
};

static Atom wm_delete_window;
static int batteries;
static int ac = 0;
static struct doodads **doodads;

int main(int argc, char *argv[])
{
    XtAppContext app;
    Widget shell = XtVaOpenApplication(
	&app, "XPmumon", options, XtNumber(options), &argc, argv,
	fallback_res, applicationShellWidgetClass,
	0
    );

    int row;
    Widget layout;

    if ((self = strrchr(*argv, '/')))
    	self++;
    else
	self = *argv;

    if (argc > 1)
    {
	fprintf(stderr, HELP, self, self);
	return 2;
    }

    XtVaGetApplicationResources(
	shell, (XtPointer) &opts, resources, XtNumber(resources),
	0
    );

    if (opts.help)
    {
	printf(HELP, self, self);
	return 0;
    }

    batteries = read_info("Battery count");
    if (opts.version)
    {
	printf("%s %s\n\n" COPYRIGHT, self, version());
	return 0;
    }

    doodads = parse_layout(opts.layout, batteries);

    /* create layout box */
    layout = XtVaCreateManagedWidget(
	"layout", formWidgetClass, shell,
	0
    );

    /* create widgets */
    for (row = 0; doodads[row]; row++)
    {
	int col;
	for (col = 0; doodads[row][col].name; col++)
	{
	    WidgetClass class = None;
	    Arg args[3]; /* fromHoriz, fromVert and orientation */
	    int n = 0;

	    if (row)
	    {
		XtSetArg(args[n], XtNfromVert, doodads[row - 1][0].w);
		n++;
	    }

	    if (col)
	    {
		XtSetArg(args[n], XtNfromHoriz, doodads[row][col - 1].w);
		n++;
	    }

	    switch (doodads[row][col].style)
	    {
	    case style_meter:
		class = meterWidgetClass;
		break;

	    case style_hgauge:
		class = gaugeWidgetClass;
		break;

	    case style_vgauge:
		class = gaugeWidgetClass;
		XtSetArg(args[n], XtNorientation, XtorientVertical);
		n++;
		break;

	    case style_basic:
		class = basicWidgetClass;
	    }

	    doodads[row][col].w = XtCreateManagedWidget(
		doodads[row][col].name, class, layout,
		args, n
	    );
	}
    }

    XtRealizeWidget(shell);

    /* setup timer to read /proc/pmu */
    update_info((XtPointer) shell, 0);

    /* handle WM close event */
    wm_delete_window = XInternAtom(XtDisplay(shell), "WM_DELETE_WINDOW", False);
    XSetWMProtocols(XtDisplay(shell), XtWindow(shell), &wm_delete_window, 1);
    XtAppAddActions(app, actions, XtNumber(actions));
    XtOverrideTranslations(shell,
	XtParseTranslationTable("<Message>WM_PROTOCOLS: quit()")
    );

    XtAppMainLoop(app);

    /*NOTREACHED*/
    return 0;
}

static void update_info(XtPointer data, XtIntervalId *id ATTRIB_UNUSED)
{
    Widget w = (Widget) data;
    int batt;
    int row;

    ac = read_info("AC Power");

    for (batt = 0; batt < batteries; batt++)
    {
	struct batt_info *info = read_battery(batt);

	for (row = 0; doodads[row]; row++)
	{
	    int col;
	    for (col = 0; doodads[row][col].name; col++)
	    {
		/* initialise value before processing the first battery */
		if (!batt)
		    doodads[row][col].value = 0;

		/* no battery */
		if (!(info->flags & PMU_BATT_PRESENT))
		    continue;

		/* check if widget deals with this battery (or all) */
		if (doodads[row][col].battery != batt &&
		    doodads[row][col].battery != -1)
		    continue;

		switch (doodads[row][col].type)
		{
		case type_current:
		    doodads[row][col].value += info->current;
		    break;

		case type_voltage:
		    doodads[row][col].value += info->voltage;
		    break;

		case type_power:
		    doodads[row][col].value += info->current *
			info->voltage / 1000;

		    break;

		case type_charge:
		    doodads[row][col].value += round(info->charge  * 100.0 /
			info->max_charge);

		    break;

		case type_time:
		    if (ac)
		    {
			/* use value -1 to indicate charging */
			doodads[row][col].value =
			    (info->flags & PMU_BATT_CHARGING) ? -1 : 0;
		    }
		    else
			doodads[row][col].value += info->time_rem;
		}
	    }
	}
    }

    for (row = 0; doodads[row]; row++)
    {
	int col;
	for (col = 0; doodads[row][col].name; col++)
	{
	    switch (doodads[row][col].style)
	    {
	    case style_basic:
		if (doodads[row][col].type == type_time)
		{
		    if (ac)
		    {
			/* charging */
			if (doodads[row][col].value < 0)
			    strcpy(doodads[row][col].string_value, "C");
			else
			    doodads[row][col].string_value[0] = 0;
		    }
		    else
		    {
			int m = doodads[row][col].value / 60;
			snprintf(doodads[row][col].string_value, SVALUE_SZ,
			    "%d:%02d", m / 60, m % 60);
		    }

		    XtVaSetValues(
			doodads[row][col].w,
			XtNvalue, doodads[row][col].value,
			XtNstringValue, doodads[row][col].string_value,
			0
		    );

		    break;
		}

		/*FALLTHRU*/

	    case style_meter:
	    case style_hgauge:
	    case style_vgauge:
		XtVaSetValues(
		    doodads[row][col].w,
		    XtNvalue, doodads[row][col].value,
		    0
		);
	    }
	}
    }

    XtAppAddTimeOut(XtWidgetToApplicationContext(w), opts.poll,
	update_info, data);
}

static void quit(Widget w, XEvent *event, String *params ATTRIB_UNUSED,
    Cardinal *num_params ATTRIB_UNUSED)
{
    if (event->type == ClientMessage && 
	event->xclient.data.l[0] == (long) wm_delete_window)
    {
	XCloseDisplay(XtDisplay(w));
	exit(0);
    }
}
