/*  This file is part of "xprintmon"
 *  Copyright (C) 2006 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include <config.h>

#include <assert.h>
#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#ifndef HAVE_ASPRINTF
#include <stdarg.h>
#endif

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/AsciiText.h>

#include "global.h"

#define APP_CLASS "XPrintMon"

struct printer {
	struct printer *next;
	char *name;
	size_t namelen;
	char *destination;
	size_t destinationlen;
	char *condition;
	size_t conditionlen;
	int ownjobs; //, totaljobs;
	char *ownjobstr; // , *totaljobstr;
	Widget dialog;
	int number;
	enum lpq_state disabled, stopped;
	timecode lastseen;
};

static void printersFree(struct printer *printers) {
	struct printer *p;
	while( (p = printers) != NULL ) {
		printers = p->next;
		free(p->name);
		free(p->destination);
		free(p->condition);
		free(p->ownjobstr);
//		free(p->totaljobstr);
		free(p);
	}
}

enum panes { P_NAME=0, P_OWNJOB, // P_TOTALJOB,
             P_SPOOL, P_PRINT,
             P_DESTINATION, P_CONDITION,
	     /* must be last */
	     P_COUNT
};

static const String headings[P_COUNT] = {
	"Name", "Own jobs", // "#own", "#total",
	"Queue", "Print",
	"Destination", "Status"
};
static const String resourcenames[P_COUNT] = {
	"name", "ownjob", // "totaljob",
	"spool", "print",
	"destination", "status"
};

struct mainwindow {
	Widget shell;
	Widget form,paned;
	Widget command, details, user;
	Widget lists[P_COUNT];
	struct requests *requests;
	struct printer *printers;
	const char **fields[P_COUNT];
	int listcount;
	XtIntervalId timer;
	timecode timecode;
	struct printer *selected;
	int selectedLine;
	bool activeIcon;
	struct mainwindow_resourcedata {
		int rescanTimeOut;
	} resources;
};

static XtResource mainwindow_resources[] = {
	{ "rescan", "TimeOut", XtRInt, sizeof(int),
	  XtOffsetOf(struct mainwindow_resourcedata, rescanTimeOut), XtRImmediate,
	  (XtPointer)10000 }
};

static void freePrinter(struct printer *printer) {
	free(printer->name);
	free(printer->destination);
	free(printer->condition);
	free(printer->ownjobstr);
//	free(printer->totaljobstr);
	free(printer);
}

static void markActive(struct mainwindow *data, bool active) {
	if( active == data->activeIcon )
		return;
	if( !active ) {
		/* do not set inactive as long as there is something
		 * left active */
		struct printer *p;

		for( p = data->printers ; p != NULL ; p = p->next ) {
			if( p->ownjobs > 0 )
				return;
		}
	}
	data->activeIcon = active;
	if( active )
		XtVaSetValues(data->shell,
				XtNiconPixmap,	bm_activeprinter,
				XtNiconMask,	bm_activeprintermask,
				NULL);
	else
		XtVaSetValues(data->shell,
				XtNiconPixmap,	bm_printer,
				XtNiconMask,	bm_printermask,
				NULL);
}

#ifndef HAVE_ASPRINTF
#warning using asprint replacement
static int asprintf(char **r, const char *fmt, ...) {
	va_list ap;
	/* that's ugly, but we will not need longer values here... */
	char buffer[100];
	int len;

	va_start(ap, fmt);
	len = vsnprintf(buffer, 99, fmt, ap);
	buffer[99] = '\0';
	*r = strdup(buffer);
	if( *r == NULL )
		return -1;
	return len;
}
#endif

static void publishList(struct mainwindow *data, struct printer *printer,
		bool gen, bool genOwnJobs, bool genTotalJobs UNUSED,
		bool genStatus, bool genDest) {

	bool changed[P_COUNT];
	enum panes pane;
	static const char *questionmark = "?";
	static const char *enabled = "on";
	static const char *disabled = "off";
	static const char *vanished = "no longer visible!";

	if( printer->ownjobs > 0 )
		markActive(data, true);
	else
		markActive(data, false);

	for( pane = 0 ; pane < P_COUNT ; pane++ ) {
		changed[pane] = false;
	}
#define field(p) data->fields[p][printer->number]
#define fieldcon(p,v) {changed[p] = gen || field(p) != v; field(p) = v;}

	field(P_NAME)= printer->name;
	changed[P_NAME] = gen;

	switch( printer->disabled ) {
		case LPQ_ENABLED:
			fieldcon(P_SPOOL, enabled);
			break;
		case LPQ_DISABLED:
			fieldcon(P_SPOOL, disabled);
			break;
		case LPQ_UNKNOWN:
		default:
			fieldcon(P_SPOOL, questionmark);
			break;

	}
	switch( printer->stopped ) {
		case LPQ_ENABLED:
			fieldcon(P_PRINT, enabled);
			break;
		case LPQ_DISABLED:
			fieldcon(P_PRINT, disabled);
			break;
		case LPQ_UNKNOWN:
		default:
			fieldcon(P_PRINT, questionmark);
			break;

	}
	changed[P_DESTINATION] = gen || genDest;
	field(P_DESTINATION) = printer->destination;
	changed[P_CONDITION] = gen || genStatus;
	if( printer->condition == NULL )
		field(P_CONDITION) = vanished;
	else
		field(P_CONDITION) = printer->condition;

	changed[P_OWNJOB] = gen || genOwnJobs;
	if( gen || genOwnJobs ) {
		if( !gen )
			free(printer->ownjobstr);
		if( printer->ownjobs >= 0 ) {
			asprintf(&printer->ownjobstr, "%d",
					printer->ownjobs);
			field(P_OWNJOB) = printer->ownjobstr;
		} else {
			printer->ownjobstr = NULL;
		 	field(P_OWNJOB) = questionmark;
		}
	}
	/*
	changed[P_TOTALJOB] = gen || genTotalJobs;
	if( gen || genTotalJobs ) {
		if( !gen )
			free(printer->totaljobstr);
		if( printer->totaljobs >= 0 ) {
			asprintf(&printer->totaljobstr, "%d",
					printer->totaljobs);
			field(P_TOTALJOB) = printer->totaljobstr;
		} else {
			printer->totaljobstr = NULL;
		 	field(P_TOTALJOB) = questionmark;
		}
	}
	*/
	for( pane = 0 ; pane < P_COUNT ; pane++ ) {
		if( !changed[pane] )
			continue;
		XawListChange(data->lists[pane], (String*)(data->fields[pane]),
				data->listcount, 0, True);
		assert( data->selectedLine >= 0 && data->selectedLine <= data->listcount );
		if( data->selectedLine > data->listcount )
			data->selectedLine = 0;
		if( data->selectedLine > 0 )
			XawListHighlight(data->lists[pane], data->selectedLine);
		else
			XawListUnhighlight(data->lists[pane]);

	}
#undef field
}

static inline bool isPrinter(const struct printer *printer, const char *name, size_t len) {
	if( printer->namelen != len )
		return false;
	return memcmp(printer->name,name,len) == 0;
}

static void finishedPrinters(void *privdata, timecode timecode) {
	struct mainwindow *data = privdata;
	struct printer *printer;

	for( printer = data->printers ; printer != NULL ;
			printer = printer->next ) {

//		printf("%s: %lu vs %lu\n", printer->name, printer->lastseen, timecode);
		if( printer->lastseen < timecode ) {
			printer->disabled = LPQ_UNKNOWN;
			printer->stopped = LPQ_UNKNOWN;
			printer->ownjobs = -1;
//			printer->totaljobs = -1;
			printer->condition = NULL;
			printer->conditionlen = 0;
			publishList(data, printer, false,
					true, true, true, false);
		}
	}
}

static void foundPrinter(void *privdata, timecode timecode,
		const char *name, size_t namelen,
		int jobs,
		enum lpq_state spoolstate, enum lpq_state printerstate,
		const char *destination, size_t destinationlen,
		const char *remark, size_t remarklen) {

	struct mainwindow *data = privdata;
	struct printer *printer;

	for( printer = data->printers ;
			printer != NULL && !isPrinter(printer,name,namelen);
			printer = printer->next ) {
	}
	if( printer == NULL ) {
		struct printer *last = data->printers;
		enum panes pane;
		const char **n;

		while( last != NULL && last->next != NULL )
			last = last->next;
		printer = calloc(1,sizeof(struct printer));
		if( printer == NULL )
			return;
		printer->name = strndup(name, namelen);
		printer->namelen = namelen;
		printer->ownjobs = jobs;
//		printer->totaljobs = -1;
		printer->disabled = spoolstate;
		printer->stopped = printerstate;
		printer->destination = strndup(destination, destinationlen);
		printer->destinationlen = destinationlen;
		printer->condition = strndup(remark, remarklen);
		printer->conditionlen = remarklen;
		if( printer->name == NULL
				|| printer->destination == NULL
				|| printer->condition == NULL ) {
			freePrinter(printer);
			return;
		}
		printer->next = NULL;
		printer->number = data->listcount;
		printer->lastseen = timecode;
		for( pane = 0 ; pane < P_COUNT ; pane++ ) {
			n = realloc(data->fields[pane],
				(data->listcount+1)*sizeof(String));
			if( n == NULL ) {
				fprintf(stderr, "Out of memory!\n");
				exit(EXIT_FAILURE);
			}
			data->fields[pane] = n;
		}
		if( last != NULL )
			last->next = printer;
		else
			data->printers = printer;
		data->listcount++;
		publishList(data, printer, true, true, true, true, true);
		return;
	} else {
		bool jobschanged = false;
		bool newjobs = false;
		bool statuschanged = false;
		bool destchanged = false;

		printer->lastseen = data->timecode;
		if( printer->destinationlen != destinationlen ||
				strncmp(printer->destination, destination,
					destinationlen) != 0 ) {
			char *n = strdup(destination);
			if( n != NULL ) {
				free(printer->destination);
				printer->destination = n;
				printer->destinationlen = destinationlen;
				destchanged = true;
			}
		}
		if( jobs >= 0 && printer->ownjobs != jobs ) {
			jobschanged = true;
			if( printer->ownjobs < jobs )
				newjobs = true;
			printer->ownjobs = jobs;
		}
		if( printer->conditionlen != remarklen ||
				strncmp(printer->condition, remark, remarklen) != 0 ) {
			char *n = strndup(remark, remarklen);
			if( n != NULL ) {
				free(printer->condition);
				printer->condition = n;
				printer->conditionlen = remarklen;
				statuschanged = true;
			}
		}
		printer->disabled = spoolstate;
		printer->stopped = printerstate;
		publishList(data, printer, false,
				jobschanged, jobschanged,
				statuschanged, destchanged);
		return;
	}
	/* NOT REACHED */
}

static void iconify(Widget w UNUSED, XtPointer privdata, XtPointer cad UNUSED) {
	struct mainwindow *data = privdata;

	 XtVaSetValues(data->shell, XtNiconic, (XtPointer)True, NULL);
}
/*
static void listclick(Widget w UNUSED, XtPointer privdata, XtPointer calldata) {
	struct mainwindow *data = privdata;
	XawListReturnStruct *listdata = calldata;
	int printerofs = listdata->list_index / COLUMNS;
	int column = listdata->list_index % COLUMNS;

	assert(listdata->list_index >= 0);
	if( printerofs == 0 ) {
		* click into headline *
		// TODO: click on printer: 	printers to look for
		// 	 click on Jobs: 	user to look for
		return;
	}
//	fprintf(stderr, "%d: %d\n", printerofs, column);
	switch( column ) {
		case 0:
			return;
		case 1:
			logged_action(context, data->shell, "lprm", "all", NULL);
			return;
		case 2:
		case 3:
			return;
	}
	assert(0);
}
*/
static void listclick(Widget w UNUSED, XtPointer privdata, XtPointer calldata) {
	struct mainwindow *data = privdata;
	XawListReturnStruct *listdata = calldata;
	enum panes pane;
	int listindex = listdata->list_index;

	if( listindex-- <= 0 || listindex >= data->listcount ) {
		data->selected = NULL;
		data->selectedLine = 0;
		for( pane = 0 ; pane < P_COUNT ; pane++ ) {
			XawListUnhighlight(data->lists[pane]);
		}
	} else {
		data->selectedLine = listdata->list_index;
		data->selected = data->printers;
		while( listindex-- > 0 && data->selected != NULL ) {
			data->selected = data->selected->next;
		}
		for( pane = 0 ; pane < P_COUNT ; pane++ ) {
			XawListHighlight(data->lists[pane], data->selectedLine);
		}
	}
	if( data->selected == NULL ) {
		XtVaSetValues(data->details,
				XtNsensitive,	(XtPointer)False, NULL);
	} else {
		XtVaSetValues(data->details,
				XtNsensitive,	(XtPointer)True, NULL);
	}

}

static void showButton(Widget w UNUSED, XtPointer privdata, XtPointer calldata UNUSED) {
	struct mainwindow *data = privdata;

	if( data->selected == NULL )
		/* who used editres to change the sensitiveness of this button? */
		return;

	printerWindow(data->selected->name);
}

static void destroyed(Widget w UNUSED, XtPointer privdata, XtPointer cad UNUSED) {
	struct mainwindow *data = privdata;

	requestsFree(data->requests);
	printersFree(data->printers);
	data->requests = NULL;
	if( data->timer != (XtIntervalId)-1 )
		XtRemoveTimeOut(data->timer);
	free(data);
	exit(EXIT_SUCCESS);
}

static void trigger(struct mainwindow *data) {
	const char *username;

	if( data->user == None )
		return;
	XtVaGetValues(data->user, XtNlabel, &username);
	requestsTrigger(data->requests, true, data->timecode, username);
}

static void triggerSummary(XtPointer privdata, XtIntervalId *timer UNUSED) {
	struct mainwindow *data = privdata;

	data->timer = (XtIntervalId)-1;
	data->timecode++;
	trigger(data);
	if( data->resources.rescanTimeOut > 0 )
		data->timer = XtAppAddTimeOut(context,
				data->resources.rescanTimeOut,
				triggerSummary, data);
}

bool createMainWindow(Widget app) {
	struct mainwindow *data;
	enum panes pane;
	Widget w;

	data = calloc(1,sizeof(struct mainwindow));
	if( data == NULL )
		return false;

	data->shell = app;
	data->timer = (XtIntervalId)-1;

	XtVaSetValues(data->shell,	XtNiconPixmap,	bm_printer,
					XtNiconMask,	bm_printermask,
					NULL);
	data->activeIcon = False;

	data->listcount = 1;
	for( pane = 0 ; pane < P_COUNT ; pane++ ) {
		data->fields[pane] = calloc(data->listcount,sizeof(String));
		if( data->fields[pane] == NULL ) {
			free(data);
			return false;

		}
		data->fields[pane][0] = headings[pane];
	}

	data->requests = requestsCreate("-a"/*|test@pcpool00"*/, foundPrinter, finishedPrinters, data);
	if( data->requests == NULL ) {
		free(data);
		return false;
	}
	data->form = XtVaCreateManagedWidget("mainform",
			formWidgetClass, data->shell,
			NULL);
	data->paned = XtVaCreateManagedWidget("mainPaned",
			panedWidgetClass, data->form,
			XtNtop,		(XtPointer)XawChainTop,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainRight,
			XtNorientation, (XtPointer)XtorientHorizontal,
			NULL);
	for( pane = 0 ; pane < P_COUNT ; pane++ ) {
		data->lists[pane] = XtVaCreateManagedWidget(
				resourcenames[pane],
				listWidgetClass, data->paned,
				XtNdefaultColumns,
					(XtPointer)1,
				XtNforceColumns,
					(XtPointer)True,
				XtNverticalList,
					(XtPointer)False,
				XtNnumberStrings,
					(XtPointer)data->listcount,
				XtNlist,
					data->fields[pane],
				NULL);
		XtAddCallback(data->lists[pane], XtNcallback,
				listclick, data);
	}
	data->command = XtVaCreateManagedWidget("hide",
			commandWidgetClass,	data->form,
			XtNfromVert,	(XtPointer)data->paned,
			XtNfromHoriz,	(XtPointer)NULL,
			XtNtop,		(XtPointer)XawChainBottom,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainLeft,
			NULL);
	XtAddCallback(data->command, XtNcallback, iconify, data);
	data->details = XtVaCreateManagedWidget("details",
			commandWidgetClass,	data->form,
			XtNfromVert,	(XtPointer)data->paned,
			XtNfromHoriz,	(XtPointer)data->command,
			XtNtop,		(XtPointer)XawChainBottom,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainLeft,
			XtNsensitive,	(XtPointer)False,
			NULL);
	w = data->details;
	XtAddCallback(data->details, XtNcallback, showButton, data);
	w = XtVaCreateManagedWidget("user_label",
			labelWidgetClass,	data->form,
			XtNfromVert,	(XtPointer)data->paned,
			XtNfromHoriz,	(XtPointer)w,
			XtNtop,		(XtPointer)XawChainBottom,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainLeft,
			XtNborder,	(XtPointer)0,
			NULL);
	data->user = XtVaCreateManagedWidget("user",
			commandWidgetClass,	data->form,
			XtNfromVert,	(XtPointer)data->paned,
			XtNfromHoriz,	(XtPointer)w,
			XtNtop,		(XtPointer)XawChainBottom,
			XtNbottom,	(XtPointer)XawChainBottom,
			XtNleft,	(XtPointer)XawChainLeft,
			XtNright,	(XtPointer)XawChainLeft,
			XtNborder,	(XtPointer)0,
			NULL);
	w = data->user;
	// TODO:
	// XtAddCallback(data->user, XtNcallback, setUserButton, data);
	XtAddCallback(data->shell, XtNdestroyCallback, destroyed, data);
	XtInstallAllAccelerators(data->paned, data->shell);
	for( pane = 0 ; pane < P_COUNT ; pane++ ) {
		XtInstallAllAccelerators(data->lists[pane], data->shell);
	}
	XtInstallAllAccelerators(data->shell, data->shell);
	XtSetKeyboardFocus(data->shell, data->lists[P_NAME]);

	XtRealizeWidget(data->shell);
	XtGetApplicationResources(data->shell, &data->resources,
			mainwindow_resources, XtNumber(mainwindow_resources),
			NULL, ZERO);
	setDeleteHandler(data->shell, wmdDestroy);
	trigger(data);
	if( data->resources.rescanTimeOut > 0 )
		data->timer = XtAppAddTimeOut(context,
				data->resources.rescanTimeOut,
				triggerSummary, data);
	return true;
}
