/* vi:set ts=4 sw=4 et:
 *
 * main.c * xrootconsole
 * Original version: Copyright (C) 1998, 1999  Eric Youngblut
 * Current version:  Copyright (C) 1999        Eric Youngblut & Bob Galloway
 *
 * (The transparency stuff was inspired by -- pulled from the still-warm 
 * body of -- wterm.  Many thanks to wterm's "anonymous coder!")
 * 
 * $Id: main.c,v 1.4 2000/03/04 02:13:36 bob Exp $
 *
 *
 *
 * 
 **************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * The author of the original version of this software can be reached
 * by e-mail at yngblut@cs.washington.edu.  The original version of
 * this software can be found at http://www.cs.washington.edu/homes/yngblut/
 *
 * The author of the current version of this software can be reached
 * by e-mail at bgallowa@wso.williams.edu.  The latest version of this
 * software can be found at http://wso.williams.edu/~bgallowa/ 
 **************************************************************************/


#include "util.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#define USAGE \
"Usage: xrootconsole [options] [console]\n" \
"Scroll the console to a transparent window placed on the X root window.\n" \
"\n" \
"  -geometry GEO   the geometry of the window (default 80x10+0+0)\n" \
"  -fn FONT        the font to use (default fixed)\n" \
"  -fg COLOR       foreground color of the text (default white)\n" \
"  -bg COLOR       background AND-mask for shaded transparency (default clear)\n" \
"  -bd COLOR       border color (default white)\n" \
"  -bw WIDTH       border width (default 0)\n" \
"  -c COLUMNS      split window into number of text columns (default 1)\n" \
"  --solid         make background a solid color, not shaded-transparent\n" \
"  --topdown       insert lines at the top and scroll the rest down\n" \
"  --wrap          wrap long lines, instead of cutting them off\n" \
"  -h or --help    a familiar-looking help screen\n" \
"  [console]       filename to read (defaults to standard input)\n" \
"\n" \
"Report bugs to <bgallowa@wso.williams.edu>.\n"


#define DEFAULT_X        0
#define DEFAULT_Y        0
#define DEFAULT_COLS        80
#define DEFAULT_ROWS        10
#define DEFAULT_FONT        "fixed"
#define DEFAULT_COLOR        "white"
#define DEFAULT_BACKGROUND    "white"
#define DEFAULT_BORDER_WIDTH    0
#define DEFAULT_BORDER_COLOR    "white"


Display* dpy;
char* text;            /* circular buffer to store the characters */
int pos = 0;            /* where the actual first line in text is */

Window win;
Pixmap buf;            /* the main drawable, copied to win 
                   by exposures */
Pixmap bkg;            /* the background pixmap, copied to buf
                   before each drawing */

XFontStruct* fn = NULL;
GC gc;

int wrap = False;
int topdown = False;

int cols = DEFAULT_COLS;    /* size of the window (in characters) */
int rows = DEFAULT_ROWS;

int width, height;        /* size of the window (in pixels) */

int tc = 1;            /* number of text comlumns (NOT characters) */
int colwidth;            /* width of a column in characters */
int nlines;            /* number of lines to keep in memory */
int intercolgap = 0;        /* width between two columns */

/*
 * put
 */

void put(const char* s) {
    static int x = 0, y = 0;
  
    while (*s != '\0') {
        switch (*s) {
        case '\n':
            x = 0;
            if (topdown) {
                pos = (pos + nlines - 1) % nlines;
        memset(text + pos * colwidth, ' ', colwidth);
            } 
            else if (y == nlines - 1) {
        memset(text + pos * colwidth, ' ', colwidth);
        pos = (pos + 1) % nlines;
            }
            else y++;
            break;
      
        case '\t':
            x = (x / 8 + 1) * 8;
            if (x >= cols && wrap) put("\n");
            break;
      
        default:
            if (*s >= ' ' && x < colwidth) {
        text[x + ((y + pos) % nlines) * colwidth] = *s;
        if (++x == colwidth && wrap) put("\n");
            }
            break;
        }
    
        s++;
    }
}


/*
 * write the text in the pixmap
 */
void draw_pixmap(void) {
    int rowc = pos + (topdown==True);    /* row in caracters */
    int rowp = fn->ascent;        /* row in pixels */
    int i,j;

    /* setup the background */
    XCopyArea(dpy, bkg, buf, gc, 0, 0, width, height, 0, 0);
    
    /* write the string */
    for (i = rows; i>0; --i) {
        for (j = 0; j < tc; ++j) {
            XDrawString(dpy, buf, gc, 
                j * (colwidth * fn->max_bounds.width + intercolgap), 
                rowp, text + ((rowc + j*rows) % nlines) * colwidth, 
                colwidth);   
            }
        rowp += fn->ascent + fn->descent;
        ++rowc;
    }  
}


/*
 * init
 */

const char* init(char** argv) {
    XSetWindowAttributes a;
    XGCValues values;
    int geo_mask = 0;
    int x_root = DEFAULT_X, y_root = DEFAULT_Y;
    const char* fg = NULL;
    const char* bg = NULL;
    const char* bd = NULL;
    const char* console = NULL;
    int bw = 0;
    int solid_background = False;
    GC backgc;
  
    /* Connect to the X server */
    dpy = XOpenDisplay(NULL);
    if (dpy == NULL) {
        fprintf(stderr, "Cannot open display\n");
        exit(EXIT_FAILURE);
    }
     
    /* Process command-line arguments */
    while (*++argv != NULL) {
        if (!strcmp(*argv, "-geometry") || !strcmp(*argv, "-geo")) {
        geo_mask = XParseGeometry(*++argv, &x_root, &y_root,
                      (unsigned*)&cols, (unsigned*)&rows);
        }
        else if (!strcmp(*argv, "-font") || !strcmp(*argv, "-fn"))
            fn = load_font(*++argv);
        else if (!strcmp(*argv, "-foreground") || !strcmp(*argv, "-fg"))
            fg = *++argv;
        else if (!strcmp(*argv, "-background") || !strcmp(*argv, "-bg"))
            bg = *++argv;
        else if (!strcmp(*argv, "-bordercolor") || !strcmp(*argv, "-bd"))
            bd = *++argv;
        else if (!strcmp(*argv, "-borderwidth") || !strcmp(*argv, "-bw"))
            bw = atoi(*++argv);
        else if (!strcmp(*argv, "-columns") || !strcmp(*argv, "-c"))
            tc = atoi(*++argv);
        else if (!strcmp(*argv, "--solid"))
            solid_background = True;
        else if (!strcmp(*argv, "--wrap"))
            wrap = True;
        else if (!strcmp(*argv, "--topdown"))
            topdown = True;
        else if (!strcmp(*argv, "--help") || !strcmp(*argv, "-h")) {
            fprintf(stderr, "%s", USAGE);
            exit(EXIT_FAILURE);
        }
        else {
            console = *argv;
        }
    }
    
    /* Load the default font if none has already been loaded */
    if (fn == NULL) fn = load_font(DEFAULT_FONT);
  
    /* compute text column geometry */
    if (tc < 1) tc = 1;
    nlines = rows * tc + 1;
    colwidth = cols / tc;
    if (tc > 1) {
        /* ensure at least a 1 char gap between columns */
        if ((cols - colwidth*tc)/(tc-1) < 1) --colwidth; 
        intercolgap = (cols - colwidth*tc) * fn->max_bounds.width / (tc-1);
    }

    /* Create the text array. We can display "nlines-1" # of rows, and the 
       extra one is for the line being read. */
    text = (char*)malloc(colwidth * nlines);
    memset(text, ' ', colwidth * nlines);
  
    /* Calculate the position of window on the screen */
    width = cols * fn->max_bounds.width;
    height = rows * (fn->ascent + fn->descent);

    x_root = (geo_mask & XNegative) ?
        (DisplayWidth(dpy, DefaultScreen(dpy)) - width + x_root - (2*bw)) : x_root;
    y_root = (geo_mask & YNegative) ?
        (DisplayHeight(dpy, DefaultScreen(dpy)) - height + y_root - (2*bw)) : y_root;

    /* Create the window */
    a.background_pixmap = ParentRelative;
    a.border_pixel = load_color((bd != NULL) ? bd : DEFAULT_BORDER_COLOR);
    a.event_mask = ExposureMask;
    a.override_redirect = True;
     
    win = XCreateWindow(dpy, DefaultRootWindow(dpy), x_root, y_root,
                width, height, bw, CopyFromParent, CopyFromParent,
                DefaultVisual(dpy, DefaultScreen(dpy)),
                CWOverrideRedirect | CWBackPixmap | CWEventMask |
                CWBorderPixel, &a);
    
    XMapWindow(dpy, win);

    /* Create the pixmaps */
    bkg = XCreatePixmap(dpy, win, width, height,
            DefaultDepth(dpy,DefaultScreen(dpy)));
    buf = XCreatePixmap(dpy, win, width, height,
            DefaultDepth(dpy,DefaultScreen(dpy)));

    /* Create the foreground GC */
    values.font = fn->fid;
    values.foreground = load_color((fg != NULL) ? fg : DEFAULT_COLOR);
    values.graphics_exposures = 0;
    gc = XCreateGC(dpy, win, 
           GCFont | GCForeground | GCGraphicsExposures, 
           &values);
     
    /* initialize the background pixmap */
    values.background = values.foreground;
    values.foreground = load_color((bg != NULL) ? bg : DEFAULT_BACKGROUND);
    if (solid_background == True) {

        XLowerWindow(dpy, win);
        backgc = XCreateGC(dpy, win, GCForeground | GCBackground, &values);
        XFillRectangle(dpy, bkg, backgc, 0, 0, width, height);

    } else { /* shaded/transparent background */
        
        values.function = GXand;
        backgc = XCreateGC(dpy, win, 
               GCForeground | GCBackground | GCFunction,
               &values);

        /* fill the window with root pixmap */
        XClearWindow(dpy,win);

        /* backup the root pixmap */
        XCopyArea(dpy, win, bkg, gc, 0, 0, width, height,0,0);     
    /* in order to get the full pixmap we have to wait until the pixmap
       is copied before lowering the window */
        XLowerWindow(dpy, win);

        /* AND-shade the background */
        XFillRectangle(dpy, bkg, backgc, 0, 0, width, height);
    }

    /* prevent the server from redrawing the background */
    a.background_pixmap = None;
    XChangeWindowAttributes(dpy, win, CWBackPixmap, &a);

#ifdef STARTUP_MESSAGE
    /* Display a message */
    put(STARTUP_MESSAGE);
#endif
    
    draw_pixmap();

    return console;
}


/*
 * handle_expose
 */

void handle_expose(XExposeEvent* ev) {
    XCopyArea(dpy, buf, win, gc,
          ev->x, ev->y, ev->width, ev->height,
          ev->x, ev->y );
}



/*
 * open_console
 */

int open_console(const char* console) {
    int fd;
    assert(console!=NULL);
    fd = open(console, O_RDONLY|O_NONBLOCK);  
    if (fd < 0) {
        fprintf(stderr,"Console %s can't be opened!  ",console);
        perror("Error");
        exit (EXIT_FAILURE);
    }
    return fd;
}

/*
 * event_loop
 */

void event_loop(const char* console) {
    fd_set rfds;
    fd_set efds;
    int Xfd = ConnectionNumber(dpy);
    int Cfd = -1;
    int maxfd;
  
    for (;;) {
    
        if (Cfd < 0) {
            if (console == NULL) Cfd = 0;       /* STDIN special case */
            else Cfd = open_console(console);
        }
        maxfd = (Cfd > Xfd ? Cfd : Xfd) + 1;
    
        FD_ZERO(&rfds); 
        FD_ZERO(&efds);
    
        while (XPending(dpy) > 0) {
            XEvent ev;
            XMaskEvent(dpy, ExposureMask, &ev);
            handle_expose((XExposeEvent*)&ev.xexpose);
        }    
     
        FD_SET(Xfd, &rfds);
        FD_SET(Cfd, &rfds);
        FD_SET(Cfd, &efds);
        if (select(maxfd, &rfds, NULL, &efds, NULL) == -1) {
            perror("select");
            exit(EXIT_FAILURE);
        }
    
        if (FD_ISSET(Cfd, &rfds)) {
            char buf[1024];
            int n;
      
            n = read(Cfd, buf, sizeof(buf) - 1);
            if (n == -1) {
                sleep(1);
                if (errno != EAGAIN) {
                    if (Cfd == 0) exit(EXIT_SUCCESS);
                    close(Cfd);
                    Cfd = -1;
                }
            } else if (n > 0) {
                buf[n] = '\0';
                put(buf);
            draw_pixmap();
                /* make an exposure event */
                XClearArea(dpy, win, 0, 0, 0, 0, True);  
            } else {  /* n == 0 */
                sleep(1);
            }
            continue;
        }

        if (FD_ISSET(Cfd, &efds)) {
            if (Cfd == 0) exit(EXIT_SUCCESS);
            close(Cfd);
            Cfd = -1;
            sleep(1);
            continue;
        }
    }
}



/*
 * main
 */

int main(int argc, char** argv) {
    const char* console = NULL;
  
    console = init(argv);
    event_loop(console);
  
    XCloseDisplay(dpy);
    exit(EXIT_SUCCESS);
}
