// file      : xsd-frontend/xml.hxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2008 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#ifndef XSD_FRONTEND_XML_HXX
#define XSD_FRONTEND_XML_HXX

#include <xsd-frontend/types.hxx>
#include <xsd-frontend/schema-dom-parser.hxx>

#include <xercesc/dom/DOM.hpp>
#include <xercesc/util/XMLString.hpp>

#include <ostream>

namespace XSDFrontend
{
  namespace XML
  {
    namespace Xerces = xercesc;

    inline
    String
    transcode (XMLCh const* s)
    {
      String r (Xerces::XMLString::stringLen (s), L'0');

      for (Size i (0); *s != XMLCh (0); ++s, ++i)
      {
        r[i] = *s;
      }

      return r;
    }

    inline
    NarrowString
    transcode_to_narrow (XMLCh const* xs)
    {
      Char* s (Xerces::XMLString::transcode (xs));

      NarrowString r (s);

      Xerces::XMLString::release (&s);

      return r;
    }


    inline
    XMLCh*
    transcode (String const& s)
    {
      Size l (s.length ());

      XMLCh* r (new XMLCh[l + 1]);
      XMLCh* ir (r);

      for (Size i (0); i < l; ++ir, ++i)
      {
        *ir = static_cast<XMLCh>(s[i]);
        //std::wcerr << s[i] << "->" << *ir << std::endl;
      }

      *ir = XMLCh (0);

      // std::wcerr << r << std::endl;

      return r;
    }


    //@@ Need to port all use-cases below to use this string.
    //
    class XMLChString
    {
    public :
      XMLChString (String const& s)
          : s_ (transcode (s))
      {
      }

      XMLChString (WideChar const* s)
          : s_ (transcode (String (s)))
      {
      }

      ~XMLChString ()
      {
        Xerces::XMLString::release (&s_);
      }

      XMLCh const*
      c_str () const
      {
        return s_;
      }

    private:
      XMLChString (XMLChString const&);

      XMLChString&
      operator= (XMLChString const&);

    private:
      XMLCh* s_;
    };


    class Element
    {
    public:
      Element (Xerces::DOMElement* e)
          : e_ (e),
            name_ (transcode (e->getLocalName ())),
            namespace__ (transcode (e->getNamespaceURI ()))
      {
      }

      String
      name () const
      {
        return name_;
      }

      String
      namespace_ () const
      {
        return namespace__;
      }

    public:
      UnsignedLong
      line () const
      {
        //@@ cache
        //
        return reinterpret_cast<UnsignedLong> (e_->getUserData (line_key));
      }

      UnsignedLong
      column () const
      {
        //@@ cache
        //
        return reinterpret_cast<UnsignedLong> (e_->getUserData (column_key));
      }

    public:
      Element
      parent () const
      {
        return dynamic_cast<Xerces::DOMElement*>(e_->getParentNode ());
      }

    public:
      // Attribute identified by a name.
      //
      Boolean
      attribute_p (String const& name) const
      {
        return attribute_p ("", name);
        /*
        XMLCh* n (transcode (name));
        Xerces::DOMAttr* a (e_->getAttributeNode (n));
        delete[] n;

        return a != 0;
        */
      }

      String
      attribute (String const& name) const
      {
        return attribute ("", name);

        /*
        XMLCh* n (transcode (name));
        XMLCh const* value (e_->getAttribute (n));
        delete[] n;

        return transcode (value);
        */
      }

      String
      operator[] (String const& name) const
      {
        return attribute (name);
      }

      // Attribute identified by namespace and name.
      //

      Boolean
      attribute_p (String const& namespace_, String const& name) const
      {
        XMLCh* ns (transcode (namespace_));
        XMLCh* n (transcode (name));
        Xerces::DOMAttr* a (e_->getAttributeNodeNS (ns, n));
        delete[] ns;
        delete[] n;

        return a != 0;
      }

      String
      attribute (String const& namespace_, String const& name) const
      {
        XMLCh* ns (transcode (namespace_));
        XMLCh* n (transcode (name));
        XMLCh const* value (e_->getAttributeNS (ns, n));
        delete[] ns;
        delete[] n;

        return transcode (value);
      }

    public:
      Xerces::DOMElement*
      dom_element () const
      {
        return e_;
      }

    private:
      Xerces::DOMElement* e_;

      String name_;
      String namespace__;
    };

    inline
    String
    prefix (String const& n)
    {
      Size i (0);
      while (i < n.length () && n[i] != L':') ++i;

      //std::wcerr << "prefix " << n << " "
      //           << String (n, i == n.length () ? i : 0, i) << std::endl;

      return String (n, i == n.length () ? i : 0, i);
    }

    inline
    String
    uq_name (String const& n)
    {
      Size i (0);
      while (i < n.length () && n[i] != L':') ++i;

      return String (n.c_str () + (i == n.length () ? 0 : i + 1));
    }

    struct NoMapping
    {
      NoMapping (String const& prefix)
          : prefix_ (prefix)
      {
      }

      String const&
      prefix () const
      {
        return prefix_;
      }

    private:
      String prefix_;
    };

    // Throws NoMapping if there is no prefix-namespace association.
    //
    inline
    String
    ns_name (Element const& e, String const& n)
    {
      String wp (prefix (n));

      // 'xml' prefix requires special handling and Xerces folks refuse
      // to handle this in DOM so I have to do it myself.
      //
      if (wp == L"xml")
        return L"http://www.w3.org/XML/1998/namespace";


      XMLCh* p (0);

      if (!wp.empty ()) // 0 means "no prefix" to Xerces
        p = transcode (wp);

      XMLCh const* xns (e.dom_element ()->lookupNamespaceURI (p));
      delete[] p;

      if (xns == 0)
        throw NoMapping (wp);

      return transcode (xns);
    }

    class NoPrefix {};

    inline
    String
    ns_prefix (Element const& e,
               String const& wns)
    {
      XMLCh* ns (transcode (wns));

#if _XERCES_VERSION >= 30000
      XMLCh const* p (e.dom_element ()->lookupPrefix (ns));

      delete[] ns;

      if (p == 0)
      {
        // 'xml' prefix requires special handling and Xerces folks refuse
        // to handle this in DOM so I have to do it myself.
        //
        if (wns == L"http://www.w3.org/XML/1998/namespace")
          return L"xml";

        throw NoPrefix ();
      }
#else
      XMLCh const* p (e.dom_element ()->lookupNamespacePrefix (ns, false));

      if (p == 0)
      {
        Boolean r (e.dom_element ()->isDefaultNamespace (ns));

        delete[] ns;

        if (r)
          return L"";
        else
        {
          // 'xml' prefix requires special handling and Xerces folks refuse
          // to handle this in DOM so I have to do it myself.
          //
          if (wns == L"http://www.w3.org/XML/1998/namespace")
            return L"xml";

          throw NoPrefix ();
        }
      }

      delete[] ns;
#endif

      return transcode (p);
    }

    inline
    String
    fq_name (Element const& e, String const& n)
    {
      String un (uq_name (n));

      try
      {
        String ns (ns_name (e, n));
        return ns + L'#' + un;
      }
      catch (XML::NoMapping const&)
      {
        return un;
      }
    }


    // Simple auto_ptr version that calls release() instead of delete.
    //

    template <typename X>
    struct AutoPtrRef
    {
      X* x_;

      explicit
      AutoPtrRef (X* x)
          : x_ (x)
      {
      }
    };

    template <typename X>
    struct AutoPtr
    {
      ~AutoPtr ()
      {
        reset ();
      }

      explicit
      AutoPtr (X* x = 0)
          : x_ (x)
      {
      }

      AutoPtr (AutoPtr& y)
          : x_ (y.release ())
      {
      }

      AutoPtr (AutoPtrRef<X> r)
          : x_ (r.x_)
      {
      }

      AutoPtr&
      operator= (AutoPtr& y)
      {
        if (this != &y)
        {
          reset (y.release ());
        }

        return *this;
      }

      AutoPtr&
      operator= (AutoPtrRef<X> r)
      {
        if (r.x_ != x_)
        {
          reset (r.x_);
        }

        return *this;
      }

      operator AutoPtrRef<X> ()
      {
        return AutoPtrRef<X> (release ());
      }

    public:
      X&
      operator* () const
      {
        return *x_;
      }

      X*
      operator-> () const
      {
        return x_;
      }

      X*
      get () const
      {
        return x_;
      }

      X*
      release ()
      {
        X* x (x_);
        x_ = 0;
        return x;
      }

      void
      reset (X* x = 0)
      {
        if (x_)
          x_->release ();

        x_ = x;
      }

      // Conversion to bool.
      //
      typedef X* (AutoPtr::*BooleanConvertible)() const;

      operator BooleanConvertible () const throw ()
      {
        return x_ ? &AutoPtr<X>::operator-> : 0;
      }

    private:
      X* x_;
    };
  }
}

// Xerces DOoM.
//
//
inline
std::wostream&
operator<< (std::wostream& o, XMLCh const* s)
{
  return o << XSDFrontend::XML::transcode (s);
}

#endif  // XSD_FRONTEND_XML_HXX
