/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream.h>
#include <iomanip.h>

#include <stdlib.h>
#include <time.h>

#include <qobject.h>

#include "Player.h"

#include "AudioDriver.h"
#include "ConfigC.h"
#include "ConfigFile.h"
#include "MainDialog.h"
#include "PlaylistDialog.h"
#include "Playlist.h"
#include "TimeLCD.h"
#include "WaveViewDialog.h"
#include "songlendb/File.h"
#include "songlendb/MD5.h"
#include "songlendb/SidTuneMod.h"

Player::Player()
{
    pAudioDrv = new AudioDriver;
    pEmuEngine = new emuEngine;
    pSidTune = new SidTuneMod(0);  // sidtune loader
    pSongLengthDB = new SongLengthFile;
    pPlaylist = new Playlist;

    pEmuEngine->getConfig(myEmuConfigDefault);
    pEmuEngine->getConfig(myEmuConfig);
    
    state.stopped = true;
    state.playing = state.paused = state.forwarding = false;
    state.ready = false;

    playlistDriven = false;
    loopPlaylist = false;
    randPlaylist = false;
    
    driverIsOpen = false;
    buffers = 2;
    multiBufferSize = 2048;
    initAudioBufferSystem();
    
    playbackTimer.stop();
    connect(&playbackTimer,SIGNAL(timeout()),this,SLOT(timerJob()));
    
    srand(time(NULL));
}

Player::~Player()
{
    playbackTimer.stop();
    freeAudioBufferSystem();
    
    delete pPlaylist;
    delete pSongLengthDB;
    delete pSidTune;
    delete pEmuEngine;
    delete pAudioDrv;
}

void Player::readConfig(ConfigFile* config)
{
    myEmuConfig = config->getEmuConfig();
    pEmuEngine->setConfig(myEmuConfig);
    pEmuEngine->getConfig(myEmuConfig);
}

void Player::writeConfig(ConfigFile* config)
{
}

void Player::link(MainDialog* mainDlg, TimeLCD* timeLCD, 
                  PlaylistDialog* playlistDlg, WaveViewDialog* waveViewDlg)
{
    myMainDlg = mainDlg;
    myTimeLCD = timeLCD;
    myPlaylistDlg = playlistDlg;
    myWaveViewDlg = waveViewDlg;
}

SongLengthDB* Player::getSongLengthDB() const
{
    return pSongLengthDB;
}

bool Player::enableSongLengthDB(bool value)
{
    // does not do anything yet
    return value;
}

bool Player::init(const AudioConfig& audioConfig)
{
    myAudioConfig = audioConfig;
    
    if ( !openAudioDriver() )
    {  
        // Set system to inoperational state.
        return (state.ready = false);
    }
    closeAudioDriver();
    
    // Now retrieve the possibly changed config from the driver.
    myAudioConfig = pAudioDrv->getConfig();
    
    // Translate config to EMU. It is assumed that the emulator
    // engine can output in all possible audio configs.
    
    if (myAudioConfig.precision == AudioConfig::BITS_16)
        myEmuConfig.bitsPerSample = SIDEMU_16BIT;
    else
        myEmuConfig.bitsPerSample = SIDEMU_8BIT;
    
    if (myAudioConfig.encoding == AudioConfig::SIGNED_PCM)
        myEmuConfig.sampleFormat = SIDEMU_SIGNED_PCM;
    else
        myEmuConfig.sampleFormat = SIDEMU_UNSIGNED_PCM;
    
    if (myAudioConfig.channels == AudioConfig::STEREO)
        myEmuConfig.channels = SIDEMU_STEREO;
    else
    {
        myEmuConfig.channels = SIDEMU_MONO;
        myEmuConfig.autoPanning = SIDEMU_NONE;
        // Max. mono can do is VOLCONTROL mode.
        if (myEmuConfig.volumeControl==SIDEMU_FULLPANNING ||
            myEmuConfig.volumeControl==SIDEMU_STEREOSURROUND)
            myEmuConfig.volumeControl = SIDEMU_NONE;
    }
    myEmuConfig.frequency = myAudioConfig.frequency;
    // Set the emuEngine to supported settings.
    pEmuEngine->setConfig(myEmuConfig);
    pEmuEngine->getConfig(myEmuConfig);

    if (audioConfig.bufSize != 0)
        multiBufferSize = audioConfig.bufSize;
    else
        multiBufferSize = myAudioConfig.blockSize;
    
    state.ready = openAudioBufferSystem();
    return state.ready;
}

const AudioConfig& Player::getAudioConfig() const
{
    return pAudioDrv->getConfig();
}

Playlist* Player::getPlaylist() const
{
    return pPlaylist;
}

void Player::setPlaylistParams(const PlaylistItem& item)
{
    if ( &item==&curPlaylistItem )
        return;
    curPlaylistItem = item;
}

const PlaylistItem& Player::getCurPlaylistParams()
{
    return curPlaylistItem;
}

void Player::enablePlaylist(bool value)
{
    playlistDriven = value;
}

void Player::enablePlaylistLoop(bool value)
{
    loopPlaylist = value;
}

void Player::enablePlaylistRand(bool value)
{
    randPlaylist = value;
}

bool Player::playNextFromList(int delta)
{
    bool haveNext = false;
    uint count = pPlaylist->list.count();
    uint index = delta+pPlaylist->getCurrentPlayPos();
    
    if ( delta==0 )  // START
    {
        if ( randPlaylist )
        {
            float f = count;
            index = (uint)(f*rand()/(RAND_MAX+1.0));
        }
        else
        {
            index = 0;
        }
        haveNext = (index<count);
    }
    else  // PREV/NEXT
    {
        if ( randPlaylist && count>=1 )
        {
            // RANDOM
            float f = count;
            index = (uint)(f*rand()/(RAND_MAX+1.0));
            haveNext = true;
        }
        else if ( count>0 && index>=0 && index<=(count-1) )  // last is count-1
        {
            // NEXT/PREV
            haveNext = true;
        }
        else if ( loopPlaylist && count>=1 )
        {
            // LOOP
            if ( delta>0 )
                index = 0;        // TO BEGINNING
            else
                index = count-1;  // TO END
            haveNext = true;
        }
    }
    if (haveNext)
    {
        pPlaylist->setCurrentPlayPos(index);
        playFromList(index);
    }
    return haveNext;
}

void Player::playFromList(uint index)
{
    PlaylistItem* pli = pPlaylist->list.at(index);
    curPlaylistItem = *pli;
    curPlaylistItem.fileNameString.prepend(pPlaylist->baseDir);
    emit playListItem(*pli);
    emit playerPlayRequest(index);
}

void Player::initSong(int song)
{
    sidEmuInitializeSong(*pEmuEngine,*pSidTune,song);
}

void Player::initNextSong()
{
    sidEmuInitializeSong(*pEmuEngine,*pSidTune,mySidTuneInfo.currentSong+1);
    pSidTune->getInfo(mySidTuneInfo);
}

void Player::initPrevSong()
{
    sidEmuInitializeSong(*pEmuEngine,*pSidTune,mySidTuneInfo.currentSong-1);
    pSidTune->getInfo(mySidTuneInfo);
}

bool Player::haveSidTune() const
{
    return pSidTune->getStatus();
}

bool Player::havePrevSong()
{
    return ( pSidTune->getStatus() && mySidTuneInfo.currentSong>1 && 
             mySidTuneInfo.songs>1 );
}

bool Player::haveNextSong()
{
    return ( pSidTune->getStatus() &&
             mySidTuneInfo.currentSong<mySidTuneInfo.songs );
}

void Player::pause()
{
    if ( state.playing && !state.paused )
    {
        state.paused = true;
        playbackTimer.stop();
        pAudioDrv->reset();
    }
}

void Player::pauseHard()
{
    if ( state.playing && !state.paused )
    {
        state.paused = true;
        playbackTimer.stop();
        pAudioDrv->reset();
        closeAudioDriver();
    }
}

void Player::resume()
{
    if ( state.paused && !driverIsOpen && state.ready )  // pauseHard?
    {
        if ( !openAudioDriver() || !pSidTune->getStatus() )
            return;
        state.paused = false;
        playbackTimer.start(playbackTimer.interval,false);
    }
    else if ( state.paused )
    {
        state.paused = false;
        playbackTimer.start(playbackTimer.interval,false);
    }
}

void Player::stop()
{
    if ( state.playing )
    {
        state.stopped = true;
        state.playing = state.forwarding = false;
        playbackTimer.stop();
        pAudioDrv->reset();
        closeAudioDriver();
    }
}

void Player::playFastForward()
{
    extern bool sidEmuFastForwardReplay(int);
    sidEmuFastForwardReplay(10);   // calc 10%
    state.forwarding = true;
}

void Player::playNormalSpeed()
{
    extern bool sidEmuFastForwardReplay(int);
    sidEmuFastForwardReplay(100);  // calc 100%
    state.forwarding = false;
}

bool Player::isReady() const
{
    return ( state.ready && pSidTune->getStatus() );
}

bool Player::isPlaying() const
{
    return state.playing;
}

bool Player::isForwarding() const
{
    return state.forwarding;
}

bool Player::isPaused() const
{
    return state.paused;
}

bool Player::isStopped() const
{
    return state.stopped;
}

emuEngine* Player::getEmuEngine() const
{
    return pEmuEngine;
}

const emuConfig& Player::getEmuConfig()
{
    pEmuEngine->getConfig(myEmuConfig);
    return myEmuConfig;
}

const emuConfig& Player::getEmuConfigDefault() const
{
    return myEmuConfigDefault;
}

SidTuneMod* Player::getSidTune() const
{
    return pSidTune;
}

const sidTuneInfo& Player::getSidTuneInfo()
{
    pSidTune->getInfo(mySidTuneInfo);
    return mySidTuneInfo;
}

void Player::fillBuffer(ubyte* pBuffer, udword bufferSize)
{
    sidEmuFillBuffer(*pEmuEngine,*pSidTune,pBuffer,bufferSize);
    int sec = pEmuEngine->getSecondsThisSong();
//    if ( sec < (fadeInTime+1) )
//        myFadeIn.fade( pBuffer, bufferSize );
    if ( playlistDriven && curPlaylistItem.time!=0 && sec>=(curPlaylistItem.time-curPlaylistItem.fadeout) )
        myFadeOut.fade( pBuffer, bufferSize );
}

void Player::timerJob()
{
    static bool isActive = false;
    if ( !isActive )
    {
        isActive = true;
        if ( state.playing || state.forwarding )
        {
            int secs = pEmuEngine->getSecondsThisSong();

            // Multi-buffering system unavailable.
            //
            // Due to a bug in the OSS sound driver in Linux 2.0.x it is
            // impossible to determine the current playback position in an
            // accurate way. Normally, this would allow synchronization of
            // audio and graphics output.
            //
            // Let's don't care about this for some time...
            
            fillBuffer(pBuffer[currentBuffer],multiBufferSize);
            pAudioDrv->play(pBuffer[currentBuffer],multiBufferSize);
            bufferFlag[currentBuffer] = true;
            lastBufferCount += multiBufferSize;

//                bufferCount[nextBuffer] = lastBufferCount;
//                if (++nextBuffer >= buffers)
//                    nextBuffer = 0;
//                if (++currentBuffer >= buffers)
//                    currentBuffer = 0;
            
            myTimeLCD->updateDisplay(secs);
            
            if ( myWaveViewDlg->isVisible() )
            {
                if ( bufferFlag[currentBuffer] )
                {
                    bufferFlag[currentBuffer] = false;
                    myWaveViewDlg->paintWaveformFunc(pBuffer[currentBuffer]);
                }
            }
/*
            if ( secs == 5 )
            {
                cout << "playlistDriven = " << playlistDriven << endl;
                cout << "curPlaylistItem.time = " << curPlaylistItem.time << endl;
            }
*/
            if ( playlistDriven && curPlaylistItem.time && secs>curPlaylistItem.time)
            {
                if ( !playNextFromList(1) )
                {
                    stop();
                    emit stopPlaylistPlay();
                    //myMainDlg->playerButtonsForceStop();
                }
            }

        }
        isActive = false;
    }
}

bool Player::openAudioDriver()
{
    closeAudioDriver();
    driverIsOpen = pAudioDrv->open(myAudioConfig);
    return driverIsOpen;
}

void Player::closeAudioDriver()
{
    if (driverIsOpen)
    {
        pAudioDrv->close();
        driverIsOpen = false;
    }
}

void Player::start()
{
    stop();
    
    if ( !openAudioDriver() || !pSidTune->getStatus() )
        return;
    
    sidEmuInitializeSong(*pEmuEngine,*pSidTune,mySidTuneInfo.currentSong);
    playNormalSpeed();
    
//    myFadeIn.set(myAudioConfig,curPlaylistItem.fadeInTime*1000);
    myFadeOut.set(myAudioConfig,curPlaylistItem.fadeout*1000);
            
    // preFillBuffers
    currentBuffer = 0;
    nextBuffer = currentBuffer+1;
    
    lastBufferCount = 0;
    for (int i = 0; i < buffers; i++)
    {
        fillBuffer(pBuffer[i],multiBufferSize);
        bufferFlag[i] = true;
        lastBufferCount += multiBufferSize;
        bufferCount[i] = lastBufferCount;
    }
    for (int i = 0; i < buffers; i++)
    {
        pAudioDrv->play(pBuffer[i],multiBufferSize);
    }
            
    // Enable timer callback.
    playbackTimer.changeInterval(0);
    playbackTimer.start(playbackTimer.interval,false);
    
    state.stopped = state.paused = state.forwarding = false;
    state.playing = true;
}

// Multi-buffering system is not fully implemented.
// See timerJob().

void Player::initAudioBufferSystem()
{
    // Clear multi-buffering system.
    lastBufferCount = 0;
    for (int i = 0; i < maxBuffers; i++)
    {
        bufferFlag[i] = false;
        bufferCount[i] = 0;
        pBuffer[i] = 0;
    }
}

void Player::freeAudioBufferSystem()
{
    for (int i = 0; i < maxBuffers; i++)
    {
        if (pBuffer[i] != 0)
            delete[] pBuffer[i];
    }
    initAudioBufferSystem();
}

bool Player::openAudioBufferSystem()
{
    freeAudioBufferSystem();
    
    bool success = true;
    for (int i = 0; i < buffers; i++)
    {
        if ((pBuffer[i] = new ubyte[multiBufferSize]) == 0)
        {
            success = false;
            break;
        }
    }
    return success;
}

