/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <libguile.h>

#include "guile-script.h"

namespace Yehia
{

namespace Script
{

using namespace std;

namespace
{

// Helper function forward decls
bool guile_module_p(SCM obj);
std::string underscores2dashes(const string& s);
SCM guile_name2symbol(const string& s);
void *guile_constructor(SCM initargs);
size_t guile_destructor(void *);
string guile_module_name(SCM module);
void guile_module_insert(SCM module, const string& name, SCM value);
SCM guile_null_instance_p(SCM obj);
SCM guile_slot_definition(SCM sclass, SCM sname);
SCM guile_default_setter (SCM obj, SCM c);
SCM guile_ensure_accessor(SCM module, const std::string& name, 
                          SCM gf, bool getter = true);
SigC::Object *guile_get_instance(SCM sobj);
SCM guile_pointer2str(const void *p);
SCM ismob_make(SCM gf);
SCM objsmob_make(SigC::Object *obj);
SCM classprocsmob_make(SCM proc);
SCM slotsmob_make(const Slot& slot);

typedef guileObjectFactory::SmallInfo SmallInfo;
}

guileObjectFactory::guileObjectFactory()
{
  memchunk_ = g_mem_chunk_new("uC++ guile object mem chunks",
                              sizeof(guileObject),
                              sizeof(guileObject) * 256,
                              G_ALLOC_AND_FREE);

  sigc_obj_class_ = 0;
  
  gpointer mem = g_mem_chunk_alloc(memchunk_);
  null_obj_ = (new (mem) guileObject(*this));
  refcnt_++;
}



guileObjectFactory::~guileObjectFactory()
{
  g_mem_chunk_destroy(memchunk_);
  if (--refcnt_ <= 0 && smallinfo_memchunk_)
  {
    g_mem_chunk_destroy(smallinfo_memchunk_);
    smallinfo_memchunk_ = NULL;
  }
}

SCM guileObjectFactory::to_scmobj(const Any& v)
{
  SCM scmobj = SCM_UNSPECIFIED;

  try
  {
    switch (v.typecode())
    {
      case Any::TC_BOOL:
        scmobj = any_cast<bool>(v) ? SCM_BOOL_T : SCM_BOOL_F;
        break;
      case Any::TC_LONG:
        scmobj = scm_long2num(any_cast<long>(v));
        break;
      case Any::TC_ULONG:
        scmobj = scm_ulong2num(any_cast<unsigned long>(v));
        break;
      case Any::TC_REAL:
        scmobj = scm_make_real(any_cast<double>(v));
        break;
      case Any::TC_STRING:
        scmobj = scm_makfrom0str(any_cast<char *>(v));
        break;
      case Any::TC_LIST:
      {
        std::list<Any> the_list = any_cast<std::list<Any> >(v);
        std::list<Any>::const_iterator it;
        int i;
        
        scmobj = scm_make_vector(SCM_MAKINUM(the_list.size()), 
                                 SCM_UNSPECIFIED);
    
        for (i = 0, it = the_list.begin(); it != the_list.end(); ++it, i++)
          scm_vector_set_x(scmobj, SCM_MAKINUM(i), to_scmobj(*it));
        
        break;
      }
      case Any::TC_SLOT:
      {
        scmobj = SCM_UNSPECIFIED; // XXX: implement
        break;
      }
      case Any::TC_INSTANCE:
      {
        Any::InstanceValue ih = any_cast<Any::InstanceValue>(v);
        guileObject *instance;
        Object *class_object;
        
        if ((class_object = find_class(*ih.tinfo)) == 0)
        {
          if (!sigc_obj_class_)
          {
            SCM yehiamodule = scm_c_resolve_module("yehia");
            SCM objclass = SCM_MODULEP(yehiamodule) ? 
              scm_variable_ref(scm_c_module_lookup(yehiamodule, "<Object>")) : 
              0;
            sigc_obj_class_ = SCM_CLASSP(objclass) ? 
              &create_object(objclass) : 0;
          }
          instance = sigc_obj_class_ ?
            dynamic_cast<guileObject *>(
                    &wrap_instance(*sigc_obj_class_, ih.instance)) : 0;
        }
        else
          instance = dynamic_cast<guileObject *>(
                  &wrap_instance(*class_object, ih.instance));
        
        scmobj = instance ? instance->sobj() : SCM_UNSPECIFIED;
        if (instance)
          instance->unreference();
        break;
      }
      case Any::TC_VOID:
        scmobj = SCM_UNSPECIFIED;
        break;
      default:
        g_assert_not_reached();
    }
  }
  catch (BadAnyCast)
  {
    scmobj = SCM_UNSPECIFIED;
  }

  return scmobj;
}

Object& guileObjectFactory::create_value(const Any& v)
{
  return create_object(to_scmobj(v));
}

Object& guileObjectFactory::create_genfunc()
{
  return create_object(scm_make(scm_list_3(scm_class_generic, 
                                           scm_c_make_keyword("name"), 
                                           scm_makfrom0str("yehia:unnamed"))));
}

Object& guileObjectFactory::create_class(const Signature& supers)
{
  Signature::const_iterator it;
  int i;
  SCM ssupers = SCM_EOL;
  
  for (i = supers.size() - 1, it = supers.end(); i >= 0; i--)
    ssupers = scm_cons(dynamic_cast<guileObject *>(find_class(**--it))->sobj(),
                       ssupers);
  
  if (supers.size() == 0)
    ssupers = scm_cons(scm_class_object, ssupers);
  
  // We need to create a new metaclass for the class, so we can
  // implement static methods

#if 0 // this seems to cause infinite loops in goops
  SCM ensure_meta = SCM_VARIABLE_REF(scm_c_module_lookup(scm_module_goops, 
                                                         "ensure-metaclass"));
  SCM meta = scm_apply(ensure_meta, scm_list_2(ssupers, SCM_BOOL_F), 
                       SCM_EOL);
#else
  SCM meta = scm_class_class;
#endif
  meta = scm_basic_make_class(meta,
                              scm_str2symbol("yehia:meta:<unknown>"),
                              scm_list_1(meta), SCM_EOL);
  SCM slots = scm_list_1(scm_list_3(scm_str2symbol("%yehia-instance"),
                                    scm_c_make_keyword("init-value"),
                                    SCM_BOOL_F));
  SCM sclass = scm_basic_make_class(meta, scm_makfrom0str("<unknown>"), 
                                    ssupers, slots);
  return create_object(sclass);
}

Object& guileObjectFactory::create_namespace()
{
  // We indicate namespaces by the null object initially,
  // they are assigned a proper object upon insertion
  return create_object(SCM_UNSPECIFIED);
}

Object& guileObjectFactory::wrap_instance(const Object& klass,
                                          SigC::Object *obj)
{
  SCM sobj = scm_sys_allocate_instance(
          dynamic_cast<const guileObject&>(klass).sobj(), SCM_EOL);
  scm_sys_initialize_object(sobj, SCM_EOL);
  scm_slot_set_x(sobj, scm_str2symbol("%yehia-instance"), objsmob_make(obj));

  return create_object(sobj);
}

guileObject& guileObjectFactory::create_object(SCM sobj)
{
  gpointer mem = g_mem_chunk_alloc(memchunk_);
  return *(new (mem) guileObject(*this, sobj));
}

Object& guileObjectFactory::null_object()
{
  return *null_obj_;
}

Any guileObjectFactory::scm_to_any(SCM sobj)
{
  if (SCM_STRINGP(sobj))
    return Any(std::string(SCM_STRING_CHARS(sobj)));
  
  if (SCM_VECTORP(sobj))
  {
    std::list<Any> anylist;
    for (unsigned long i = 0; i < SCM_VECTOR_LENGTH(sobj); i++)
      anylist.push_back(scm_to_any(SCM_VELTS(sobj)[i]));
    return Any(anylist);
  }
  if (SCM_INUMP(sobj))
    return Any(SCM_INUM(sobj));

  if (SCM_REALP(sobj))
    return Any(SCM_REAL_VALUE(sobj));
  
  if (sobj == SCM_BOOL_T)
    return Any(true);

  if (sobj == SCM_BOOL_F)
    return Any(false);
  
  if (SCM_IS_A_P(sobj, scm_class_object))
  {
    Any::InstanceValue ih;
    
    if ((ih.instance = guile_get_instance(sobj)) != 0)
    {
      // FIXME: how do we get the tinfo? 
      // probably we need a reverse of find_class()
      //ih.class_object = create_object(SCM_CLASS_OF(sobj));
      ih.tinfo = &typeid(Any);
      return Any(ih);
    }
  }

  if (SCM_NFALSEP(scm_list_p(sobj)))
  {
    std::list<Any> anylist;
    while (SCM_CONSP(sobj))
    {
      anylist.push_back(scm_to_any(SCM_CAR(sobj)));
      sobj = SCM_CDR(sobj);
    }
    return Any(anylist);
  }

  return Any();
}

void guileObjectFactory::yehia_init_fixup()
{
  // here we add some special things to the yehia module. this should
  // be maybe moved to yehia.igd.
  
  // we define a predicate for null instances
  SCM yehiamodule = scm_c_resolve_module("yehia");
  if (SCM_MODULEP(yehiamodule))
  {
    SCM pred = scm_c_make_gsubr("null-instance?", 1, 0, 0, 
                                (SCM (*)())&guile_null_instance_p);
    guile_module_insert(yehiamodule, "null-instance?", pred);
  }
}

void guileObjectFactory::destroy(Object& obj)
{
  guileObject& sobj = dynamic_cast<guileObject&>(obj);
  sobj.~guileObject();
  g_mem_chunk_free(memchunk_, &sobj);
}

GMemChunk *guileObjectFactory::smallinfo_memchunk_ = NULL;
int guileObjectFactory::refcnt_ = 0;


#define SMALLINFO_SIZE (sizeof(SmallInfo))

void *guileObjectFactory::smallinfo_new()
{
  if (smallinfo_memchunk_ == NULL)
    smallinfo_memchunk_ = g_mem_chunk_new("yehia smallinfo chunk",
                                          SMALLINFO_SIZE, SMALLINFO_SIZE * 64,
                                          G_ALLOC_AND_FREE);
  
  gpointer mem = g_mem_chunk_alloc(smallinfo_memchunk_);
  scm_gc_register_collectable_memory(mem, SMALLINFO_SIZE, "yehia smallinfo");
  return mem;
}

size_t guileObjectFactory::smallinfo_free(void *mem)
{
  scm_gc_unregister_collectable_memory(mem, SMALLINFO_SIZE, "yehia smallinfo");
  g_mem_chunk_free(smallinfo_memchunk_, mem);
  return SMALLINFO_SIZE;
}

guileObject::guileObject(guileObjectFactory& factory, SCM sobj)
    : Script::Object(factory)
{
  sobj_ = sobj;
}

guileObject::~guileObject()
{
}

inline SCM guileObject::sobj() const
{
  return sobj_;
}

inline bool guileObject::is_null()
{
  return (sobj_ == SCM_UNSPECIFIED);
}

inline guileObject& guileObject::operator=(SCM sobj)
{
  sobj_ = sobj;
  return *this;
}

inline void guileObject::ensure(bool cond)
{
  if (!cond) throw Exception("hooo");
}

Namespace *guileObject::namespace_interface()
{
  return (is_null() || guile_module_p(sobj_)) ? this : 0;
}

Instance *guileObject::instance_interface()
{
  return 0;
}

Function *guileObject::function_interface()
{
  return 0;
}

GenFunc *guileObject::genfunc_interface()
{
  return SCM_NFALSEP(scm_procedure_p(sobj_)) ? this : 0;
}

Class *guileObject::class_interface()
{
  return (SCM_IS_A_P(sobj_, scm_class_class)) ? this : 0;
}

Any guileObject::value() const
{
  return factory().scm_to_any(sobj_);
}

SigC::Object& guileObject::object()
{
  throw 0;
}

Object& guileObject::class_object()
{
  throw 0;
}

Object *guileObject::call(const ParamList& args)
{
  throw 0;
}

Object *guileObject::member(const std::string& name)
{
  SCM var = scm_sym2var(guile_name2symbol(name),
                        scm_module_lookup_closure(sobj_),
                        SCM_BOOL_F);

  if (SCM_NFALSEP(var))
    return &factory().create_object(scm_variable_ref(var));
  
  // we have no variable in this module, but we can see if we have
  // another "nested" module
  SCM mod_ref_proc = scm_c_lookup("module-ref");
  if (SCM_FALSEP(mod_ref_proc))
    return 0;
  SCM mod = scm_apply(
          scm_variable_ref(mod_ref_proc), 
          scm_list_3(sobj_, scm_str2symbol(name.c_str()), SCM_BOOL_F), 
          SCM_EOL);
  return SCM_MODULEP(mod) ? &factory().create_object(mod) : 0;
}

void guileObject::insert(const std::string& name, Script::Object& object)
{
  guileObject *obj = &dynamic_cast<guileObject&>(object);
  
  if (is_null())
  {
    if (obj->is_null())
    {
      // new module
      *obj = scm_c_define_module(name.c_str(), NULL, NULL);
      obj->module_ = SCM_VARIABLE_REF(scm_c_lookup("the-root-module"));
      return;
    }
  }
  else if (obj->is_null())
  {
    // New module in this namespace
    string our_name = guile_module_name(sobj_);
    *obj = scm_c_define_module((our_name + " " + name).c_str(), NULL, NULL);
    obj->module_ = sobj_;
    return;
  }
  else
  {
    guile_module_insert(sobj_, name, obj->sobj_);
    obj->module_ = sobj_;
  }
}

void guileObject::add_method(const Slot& slot, const Signature& sig)
{
  SCM specializers = SCM_EOL;
  SCM arg_syms = SCM_EOL;
  char buffer[32];
  int i;
  Signature::const_iterator it;

  for (i = sig.size() - 1, it = sig.end(); i >= 0; i--)
  {
    sprintf(buffer, "arg%d", i);
    arg_syms = scm_cons(scm_str2symbol(buffer), arg_syms);
    guileObject *klass = 
      dynamic_cast<guileObject *>(factory().find_class(**--it));
    SCM sobj = klass ? klass->sobj() : SCM_UNSPECIFIED;
    if (sobj == SCM_UNSPECIFIED)
      sobj = scm_class_top;
    specializers = scm_cons(sobj, specializers);
  }

  SCM proc = slotsmob_make(slot);
  SCM procm = scm_closure(scm_list_2(arg_syms, scm_cons(proc, arg_syms)),
                          scm_top_level_env(SCM_TOP_LEVEL_LOOKUP_CLOSURE));
  SCM meth = scm_make(scm_list_5(scm_class_method,
                                 scm_c_make_keyword("specializers"), 
                                 specializers,
                                 scm_c_make_keyword("procedure"), procm));
  //scm_display(scm_list_2(sobj_, meth), scm_current_output_port());
  //scm_flush_all_ports();
  
  scm_add_method(sobj_, meth);
}

void guileObject::set_constructor(Object& genfunc)
{
  guileObject *gf = &dynamic_cast<guileObject&>(genfunc);

  SCM initialize = scm_variable_ref(
          scm_c_module_lookup(scm_module_goops, "initialize"));
  SCM sym_obj = scm_str2symbol("obj");
  SCM sym_args = scm_str2symbol("args");
  SCM constr = ismob_make(gf->sobj());
  SCM constrm = scm_closure(scm_list_2(scm_list_2(sym_obj, sym_args),
                                       scm_list_3(constr, sym_obj, sym_args)),
                            scm_top_level_env(SCM_TOP_LEVEL_LOOKUP_CLOSURE));
  SCM meth = scm_make(scm_list_5(scm_class_method,
                                scm_c_make_keyword("specializers"),
                                scm_list_2(sobj_, scm_class_top),
                                scm_c_make_keyword("procedure"),
                                constrm));

  scm_add_method(initialize, meth);
}

void guileObject::add_genfunc(const std::string& name, Object& genfunc)
{
  guile_module_insert(module_, name, 
                      dynamic_cast<guileObject&>(genfunc).sobj());
}

void guileObject::add_class_genfunc(const std::string& name, Object& genfunc)
{
  // We construct a new GF from genfunc that has all its methods
  // replaced with an additional specializer (class-of sobj_) in front
  // of the specializer list
  SCM gf = dynamic_cast<guileObject&>(genfunc).sobj();
  SCM meths = scm_generic_function_methods(gf);
  bool do_insert = false;
  SCM new_gf = scm_sym2var(guile_name2symbol(name),
                           scm_module_lookup_closure(module_),
                           SCM_BOOL_F);
  if (SCM_NFALSEP(new_gf))
    new_gf = SCM_VARIABLE_REF(new_gf);
  else
  {
    new_gf = scm_make(scm_list_3(scm_class_generic, 
                                 scm_c_make_keyword("name"), 
                                 scm_makfrom0str("yehia:unnamed")));
    do_insert = true;
  }
  
  while (SCM_CONSP(meths))
  {
    SCM meth = SCM_CAR(meths);
    SCM proc = scm_method_procedure(meth);
    SCM specs = scm_method_specializers(meth);
    SCM new_proc = classprocsmob_make(proc);
    SCM arg_syms = scm_cons(scm_str2symbol("obj"), SCM_EOL);
    SCM cons = arg_syms;
    SCM spec;
    int i;
    char buffer[32];
    
    for (spec = specs, i = 0; SCM_CONSP(spec); spec = SCM_CDR(spec), i++)
    {
      sprintf(buffer, "arg%d", i);
      SCM_SETCDR(cons, scm_cons(scm_str2symbol(buffer), SCM_EOL));
      cons = SCM_CDR(cons);
    }
    SCM new_procm = 
      scm_closure(scm_list_2(arg_syms, scm_cons(new_proc, arg_syms)), 
                  scm_top_level_env(SCM_TOP_LEVEL_LOOKUP_CLOSURE));
    SCM new_meth = scm_make(scm_list_5(scm_class_method,
                                       scm_c_make_keyword("specializers"),
                                       scm_cons(SCM_CLASS_OF(sobj_), specs),
                                       scm_c_make_keyword("procedure"),
                                       new_procm));
    scm_add_method(new_gf, new_meth);
    meths = SCM_CDR(meths);
  }
  
  if (do_insert)
    guile_module_insert(module_, name, new_gf);
}

void guileObject::add_setter(const std::string& name, Object& genfunc)
{
  SCM slot_name = guile_name2symbol(name);
  SCM setter = dynamic_cast<guileObject&>(genfunc).sobj();
  SCM accessor = guile_ensure_accessor(module_, name, setter, false);
  SCM arg_syms = scm_list_2(scm_str2symbol("obj"), scm_str2symbol("value"));
  SCM setm = scm_closure(scm_list_2(arg_syms, scm_cons(setter, arg_syms)),
                         scm_top_level_env(SCM_TOP_LEVEL_LOOKUP_CLOSURE));
  
  SCM slotdef = guile_slot_definition(sobj_, slot_name);
  if (SCM_NFALSEP(slotdef))
  {
    SCM_SETCAR(SCM_CDR(SCM_CDR(SCM_CDR(slotdef))), setm);
  }
  else
  {
    // FIXME: We don't add a slot for setters, maybe we should?
  }
  
  for (SCM meths = scm_generic_function_methods(setter); !SCM_NULLP(meths);
       meths = SCM_CDR(meths))
    scm_add_method(scm_setter(accessor), SCM_CAR(meths));
}

void guileObject::add_getter(const std::string& name, Object& genfunc)
{
  SCM slot_name = guile_name2symbol(name);
  SCM gf = dynamic_cast<guileObject&>(genfunc).sobj();
  SCM accessor = guile_ensure_accessor(module_, name, gf);
  SCM meths = scm_generic_function_methods(gf);
  
  if (SCM_NULLP(meths))
    return;
  
  SCM proc = scm_method_procedure(SCM_CAR(meths));
  SCM arg_syms = scm_list_1(scm_str2symbol("obj"));
  SCM getm = scm_closure(scm_list_2(arg_syms, scm_cons(proc, arg_syms)),
                         scm_top_level_env(SCM_TOP_LEVEL_LOOKUP_CLOSURE));
  
  SCM slotdef = guile_slot_definition(sobj_, slot_name);
  if (SCM_NFALSEP(slotdef)) 
  {
    SCM_SETCAR(SCM_CDR(SCM_CDR(slotdef)), getm); 
  }
  else
  {
    SCM slot = scm_list_5(slot_name, scm_c_make_keyword("class"), 
                          scm_class_foreign_slot,
                          scm_c_make_keyword("getter"),
                          accessor);
    SCM setproc = scm_c_make_subr("yehia:set", scm_tc7_subr_2, 
                                  (SCM (*)())guile_default_setter);
    SCM gns = scm_list_4(slot_name, SCM_BOOL_F, proc, setproc);
    
    SCM_SET_SLOT(sobj_, scm_si_slots, 
                 scm_append_x(scm_list_2(SCM_SLOT(sobj_, scm_si_slots),
                                         scm_list_1(slot))));
    SCM_SET_SLOT(sobj_, scm_si_getters_n_setters,
                 scm_append_x(scm_list_2(SCM_SLOT(sobj_, 
                                                  scm_si_getters_n_setters),
                                         scm_list_1(gns))));
  }
}

namespace
{

bool guile_module_p(SCM obj)
{
  static SCM is_module = SCM_UNDEFINED;
  
  if (is_module == SCM_UNDEFINED)
    is_module = SCM_CDR(scm_c_lookup("module?"));
  
  return SCM_NFALSEP(scm_apply(is_module, scm_list_1(obj), SCM_EOL));
}

inline std::string underscores2dashes(const std::string& s)
{
  std::string result;
  for (std::string::size_type i = 0; i < s.size(); i++)
    result.append(1, s[i] == '_' ? '-' : s[i]);
  return result;
}

SCM guile_name2symbol(const std::string& s)
{
  return scm_str2symbol(underscores2dashes(s).c_str());
}

SCM guile_pointer2str(const void *p)
{
  char buffer[32];
  sprintf(buffer, "%p", p);
  return scm_makfrom0str(buffer);
}

SigC::Object *guile_get_instance(SCM sobj)
{
  SCM s_instance = scm_str2symbol("%yehia-instance");
  if (SCM_INSTANCEP(sobj) &&
      SCM_NFALSEP(scm_slot_exists_p(sobj, s_instance)))
  {
    SCM sinst = scm_slot_ref(sobj, s_instance);
    if (SCM_NFALSEP(sinst)) // FIXME: Need better check
      return (SigC::Object *)GINT_TO_POINTER(SCM_SMOB_DATA(sinst));
  }
  return 0;
}

string guile_module_name(SCM module)
{
  SCM module_name_proc = scm_c_lookup("module-name");
  if (SCM_NFALSEP(module_name_proc))
  {
    SCM modname = scm_apply(scm_variable_ref(module_name_proc), 
                            scm_list_1(module), SCM_EOL);
    
    if (SCM_STRINGP(modname))
      return string(SCM_STRING_CHARS(modname) + 1, 
                    SCM_STRING_LENGTH(modname) - 2);
  }
  return string();
}

void guile_module_insert(SCM module, const string& name, SCM value)
{
  std::string guile_name;
  bool do_define = true;
  
  if (SCM_IS_A_P(value, scm_class_class))
  {
    // name transformation for classes
    guile_name = "<" + name + ">";
  }
  else
    guile_name = underscores2dashes(name);
  
  SCM sname = scm_str2symbol(guile_name.c_str());
  SCM oldmodule = scm_set_current_module(module);

  // Set name of inserted object
  if (SCM_CLASSP(value))
  {
    SCM_SET_SLOT(value, scm_si_name, sname);
    SCM meta = SCM_CLASS_OF(value);
    SCM metaname = scm_str2symbol(("yehia:meta:" + guile_name).c_str());
    SCM_SET_SLOT(meta, scm_si_name, metaname);
  }
  else if (SCM_GENERICP(value))
    scm_set_procedure_property_x(value, scm_sym_name, sname);      
  
  //scm_display(scm_list_2(module, value), scm_current_output_port());
  //scm_newline(scm_current_output_port());
  //scm_flush_all_ports();
  
  if (SCM_GENERICP(value))
  {
    // Check for existing binding
    SCM oldvar = 
      scm_sym2var(sname, scm_current_module_lookup_closure (), SCM_BOOL_F);
    if (SCM_NFALSEP(oldvar) && SCM_GENERICP(scm_variable_ref(oldvar)))
    {
      // In this special case (when we would re-bind a GF), we
      // actually add all methods of the new GF to the old GF
      SCM gf = scm_variable_ref(oldvar);
      SCM newmeths = scm_generic_function_methods(value);
      
      while (SCM_CONSP(newmeths))
      {
        scm_add_method(gf, SCM_CAR(newmeths));
        newmeths = SCM_CDR(newmeths);
      }
      do_define = false;
    }
  }
  
  if (do_define)
  {
    scm_define(sname, value);
    scm_c_export(guile_name.c_str(), NULL);
  }
  
  scm_set_current_module(oldmodule);
}  

SCM guile_null_instance_p(SCM obj)
{
  SCM s_instance = scm_str2symbol("%yehia-instance");
  if (SCM_INSTANCEP(obj) &&
      SCM_NFALSEP(scm_slot_exists_p(obj, s_instance)))
  {
    SCM sinst = scm_slot_ref(obj, s_instance);
    if (SCM_FALSEP(sinst) || 
        GINT_TO_POINTER(SCM_SMOB_DATA(sinst)) == NULL)
      return SCM_BOOL_T;
  }
  return SCM_BOOL_F;
}

SCM guile_slot_definition(SCM sclass, SCM sname)
{
  for (SCM gns = SCM_SLOT(sclass, scm_si_getters_n_setters); !SCM_NULLP(gns); 
       gns = SCM_CDR(gns))
    if (SCM_EQ_P(SCM_CAAR(gns), sname))
      return SCM_CAR(gns);
  
  return SCM_BOOL_F;
}

SCM guile_default_setter (SCM obj, SCM c)
{
  scm_misc_error ("slot-set!", "read-only slot", SCM_EOL);
  return 0;
}

SCM guile_make_accessor(SCM name)
{
  SCM gf = scm_make(scm_list_3(scm_class_generic, 
                               scm_c_make_keyword("name"), name));
  return scm_make(scm_list_5(scm_class_generic_with_setter,
                             scm_c_make_keyword("name"), name, 
                             scm_c_make_keyword("setter"), gf));
}

SCM guile_ensure_accessor(SCM module, const std::string& name, 
                          SCM gf, bool getter)
{
  SCM sname = guile_name2symbol(name);
  SCM var = scm_sym2var(sname, scm_module_lookup_closure(module), SCM_BOOL_F);
  SCM accessor;
  
  if (SCM_NFALSEP(var))
  {
    SCM value = SCM_VARIABLE_REF(var);
    if (!SCM_ACCESSORP(value))
    {
      accessor = guile_make_accessor(sname);
      
      if (scm_subr_p(value) && SCM_SUBR_GENERIC(value))
      {
        scm_enable_primitive_generic_x(scm_list_1(value));
        value = scm_primitive_generic_generic(value);
      }
      if (SCM_GENERICP(value))
      {
        // Add methods of old GF to the accessor
        for (SCM meths = scm_generic_function_methods(value);
             !SCM_NULLP(meths); meths = SCM_CDR(meths))
          scm_add_method(accessor, SCM_CAR(meths));
      }
      SCM_VARIABLE_SET(var, accessor);
    }
    else
      accessor = value;
  }
  else
  {
    accessor = guile_make_accessor(sname);
    guile_module_insert(module, name, accessor);
  }

  if (SCM_NFALSEP(gf))
  {
    SCM meths = scm_generic_function_methods(gf);
    while (!SCM_NULLP(meths))
    {
      if (getter)
        scm_add_method(accessor, SCM_CAR(meths));
      else
        scm_add_method(scm_setter(accessor), SCM_CAR(meths));
      
      meths = SCM_CDR(meths);
    }
  }

  return accessor;
}

//
// Initializer SMOB
//
size_t si_smob_free(SCM smob)
{
  guileObjectFactory::smallinfo_free(GINT_TO_POINTER(SCM_SMOB_DATA(smob)));
  return 0;
}

SCM si_sval_mark(SCM smob)
{
  SmallInfo *si = (SmallInfo *)GINT_TO_POINTER(SCM_SMOB_DATA(smob));
  
  return si->sval;
};

SCM ismob_apply(SCM smob, SCM obj, SCM args)
{
  // This is a method of the 'intitialize' GF. 

  // We should first invoke (next-method), but this makes no sense,
  // since the method applied would be either this function (created
  // by set_constructor for one of our ancestors) or
  // %initialize-object if we have no yehia-based ancestors.
  
  // So we can call %initialize-object directly.
  scm_sys_initialize_object(obj, SCM_EOL);
  
  SmallInfo *si = (SmallInfo *)GINT_TO_POINTER(SCM_SMOB_DATA(smob));
  
  // Now we check the passed args. If we have 'null as only argument,
  // we return a 'null' instance
  if (scm_ilength(args) == 1 &&
      SCM_EQ_P(SCM_CAR(args), scm_str2symbol("null")))
    return obj;
  
  SCM newargs = scm_cons(scm_makfrom0str(""), 
                         scm_cons(scm_makfrom0str(""), 
                                  scm_cons(scm_makfrom0str(""), args)));
  SCM result = scm_apply(si->sval, newargs, SCM_EOL);

  SigC::Object *inst = guile_get_instance(result);
  if (inst)
    scm_slot_set_x(obj, scm_str2symbol("%yehia-instance"), objsmob_make(inst));
  
  return obj;
}

SCM ismob_make(SCM gf)
{
  static scm_t_bits ismob_tag;
  static bool is_init = false;
  SmallInfo *si;
  
  if (!is_init)
  {
    ismob_tag = scm_make_smob_type("yehia:initialize", sizeof(SmallInfo));
    scm_set_smob_free(ismob_tag, &si_smob_free);
    scm_set_smob_mark(ismob_tag, &si_sval_mark);
    scm_set_smob_apply(ismob_tag, (SCM (*)())&ismob_apply, 2, 0, 0);
  }

  si = (SmallInfo *)guileObjectFactory::smallinfo_new();
  si->sval = gf;
  
  SCM_RETURN_NEWSMOB(ismob_tag, si);
}

size_t objsmob_free(SCM smob)
{
  SigC::Object *obj = (SigC::Object *)GINT_TO_POINTER(SCM_SMOB_DATA(smob));
  if (obj)
    obj->unreference();

  return 0;
}

SCM objsmob_make(SigC::Object *obj)
{
  static scm_t_bits smob_tag;
  static bool is_init = false;
  
  if (!is_init)
  {
    smob_tag = scm_make_smob_type("yehia:object", 0);
    scm_set_smob_free(smob_tag, &objsmob_free);
  }
  if (obj)
    obj->reference();

  SCM_RETURN_NEWSMOB(smob_tag, obj);
}

SCM classprocsmob_apply(SCM smob, SCM args)
{
  SmallInfo *si = (SmallInfo *)GINT_TO_POINTER(SCM_SMOB_DATA(smob));
  return scm_apply(si->sval, SCM_CDR(args), SCM_EOL);
}

SCM classprocsmob_make(SCM proc)
{
  static scm_t_bits smob_tag;
  static bool is_init = false;
  SmallInfo *si;
  
  if (!is_init)
  {
    smob_tag = scm_make_smob_type("yehia:class-method", sizeof(SmallInfo));
    scm_set_smob_free(smob_tag, &si_smob_free);
    scm_set_smob_mark(smob_tag, &si_sval_mark);
    scm_set_smob_apply(smob_tag, (SCM (*)())&classprocsmob_apply, 0, 0, 1);
  }

  si = (SmallInfo *)guileObjectFactory::smallinfo_new();
  si->sval = proc;
  
  SCM_RETURN_NEWSMOB(smob_tag, si);
}

//
// Slot SMOB
//

SCM slotsmob_apply(SCM smob, SCM args)
{
  Language *lang = LanguageManager::instance().language("guile");
  guileObjectFactory *factory = lang ? 
    &dynamic_cast<guileObjectFactory&>(lang->factory()) : 0;
  ObjectSlot *slot = (ObjectSlot *)SCM_SMOB_DATA(smob);
  ParamList params;
  Object *result = 0;

  g_return_val_if_fail(factory, NULL);

  //scm_display(args, scm_current_output_port());
  //scm_newline(scm_current_output_port());
  //scm_flush_all_ports();

  while (SCM_CONSP(args))
  {
    params.push_back(&factory->create_object(SCM_CAR(args)));
    args = SCM_CDR(args);
  }
  
  try
  {
    result = (*slot)(params);
  }
  catch (const std::exception& e)
  {
    for (ParamList::iterator it = params.begin(); it != params.end(); ++it)
      (*it)->unreference();
    scm_throw(scm_str2symbol("yehia-error"), 
              scm_list_1(factory->to_scmobj(Any(e.what()))));
  }

  for (ParamList::iterator it = params.begin(); it != params.end(); ++it)
    (*it)->unreference();
  
  guileObject *gresult = dynamic_cast<guileObject *>(result);
  SCM sresult = gresult ? gresult->sobj() : SCM_UNSPECIFIED;
  
  if (result)
    result->unreference();
  
  return sresult;
}

SCM slotsmob_make(const Slot& slot)
{
  static scm_t_bits slotsmob_tag;
  static bool is_init = false;
  Language *lang = LanguageManager::instance().language("guile");
  
  g_return_val_if_fail(lang != 0, SCM_UNSPECIFIED);
  
  if (!is_init)
  {
    slotsmob_tag = scm_make_smob_type("yehia:slot", sizeof(Slot));
    scm_set_smob_free(slotsmob_tag, &si_smob_free);
    scm_set_smob_apply(slotsmob_tag, (SCM (*)())&slotsmob_apply, 0, 0, 1);
  }
  
  void *mem = guileObjectFactory::smallinfo_new();
  new (mem) ObjectSlot(lang_convert(slot, *lang));

  SCM_RETURN_NEWSMOB(slotsmob_tag, mem);
}

} // namespace

} // namespace Script

} // namespace Yehia



