#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <gc/gc.h>

#include "tree.h"
#include "util.h"
#include "format.h"
#include "decl.h"
#include "lexer.h"
#include "parse.h"
#include "process.h"

#define INDENT_OFF 2

extern int yyparse(void);

FILE *process_outfile = NULL;

static int indent_level = 0;
static int error_count = 0;
static const char *infilename = NULL;

static void print_cpp(const char *msg, ...);
static void print_cpp_ni(const char *msg, ...);

static void process_class(Namespace *ns, Tree klass);
static void process_attribute(const char *clname, const char *clrealname,
                              Tree klass, Tree attribute);
static void process_ns(Namespace *parent, const char *parent_objname,
                       Tree ns_stmt);
static void process_meta_plugin(Tree mplugin);
static void process_output(Tree ns);
static void process_verbatim(Tree v);
static void print_lineinfo(Tree t);

static const char *attr_get_defby_name(Tree attr);

void process_toplevel_stmts(Tree stmts)
{
  Tree node;
  
  for (node = stmts; node; node = TREE_CHAIN(node))
  {
    Tree value = TREE_VALUE(node);
    switch (TREE_CODE(value))
    {
      case NAMESPACE_STMT:
      {
        const char *name =
          TREE_IDENTIFIER_STR(TREE_NAMESPACE_IDENTIFIER(value));

        process_output(value);
        
        print_cpp("//\n"
                  "// Yehia Script interface for namespace %s\n"
                  "//\n"
                  "void yehia_ns_%s_register(Yehia::Script::Language& lang)\n"
                  "{\n", name, name);
  
        indent_level++;
  
        print_cpp("using namespace Yehia::Script;\n"
                  "using SigC::retype_return;\n"
                  "using SigC::slot;\n\n"
                  "ObjectFactory& factory = lang.factory();\n"
                  "Signature bases;\n\n");

        process_ns(NULL, "lang.root_namespace()", value);

        indent_level--;

        print_cpp("}\n\n");
        break;
      }
      case META_PLUGIN:
        process_meta_plugin(value);
        break;
      case VERBATIM:
        process_verbatim(value);
        break;
      default:
        process_internal_error("invalid toplevel statement");
    }
  }
}

static void print_reg_call_tunnel(gpointer key, gpointer value, gpointer data)
{
  Member *member = (Member *)value;
  char *langvar = (char *)data;
  
  if (member->common.id == DECL_NAMESPACE)
    print_cpp("SigCX::tunnel<void, Yehia::Script::Language&>(slot(&yehia_ns_%s_register), %s, (%s).tunnel());\n", (char *)key, langvar, langvar);
}

static void print_reg_call_direct(gpointer key, gpointer value, gpointer data)
{
  Member *member = (Member *)value;

  if (member->common.id == DECL_NAMESPACE)
    print_cpp("yehia_ns_%s_register(%s);\n", (char *)key, (char *)data);
}

static void process_meta_plugin(Tree mplugin)
{
  static const char *fname = NULL;
  static int lineno = -1;
  
  Tree node;
  const char *name = TREE_IDENTIFIER_STR(TREE_META_PLUGIN_IDENTIFIER(mplugin));
  
  if (fname)
  {
    process_error("cannot use plugin meta statement more than once");
    process_error_cont(fname, lineno, "previous use here");
  }
  else
  {
    fname = infilename;
    lineno = yylineno;
  }

  // plugin class 
  print_cpp("//\n"
            "// Plugin '%s'\n"
            "//\n"
            "class %sPlugin : public Yehia::Plugin\n"
            "{\n", name, name);
  
  indent_level++;
  print_cpp("public:\n");
  indent_level++;

  /* plugin class constructor */
  print_cpp("%sPlugin(Yehia::PluginManager& mgr) : Plugin(mgr) {\n", name);
  indent_level++;
  print_cpp("using namespace Yehia::Script;\n");
  print_cpp("std::list<Language *> langs = LanguageManager::instance().languages();\n\n"
            "for (std::list<Language *>::iterator it = langs.begin();\n"
            "     it != langs.end(); ++it)\n"
            "{\n");
  indent_level++;
  g_hash_table_foreach(decl_get_root_namespace()->members,
                       &print_reg_call_tunnel, "**it");
  indent_level--;
  print_cpp("}\n");

  print_cpp("LanguageManager::instance().language_registered.connect(SigC::slot(*this, &%sPlugin::got_new_language));\n", name);
  indent_level--;
  print_cpp("}\n");

  /* plugin class description */
  print_cpp("virtual std::string description() const { return \"%s\"; }\n",
            TREE_STRING_CST_PTR(TREE_META_PLUGIN_DESCR(mplugin)));

  /* plugin class new language registrar */
  print_cpp("void got_new_language(const std::string& name) {\n");
  indent_level++;
  print_cpp("using namespace Yehia::Script;\n");
  print_cpp("Language *lang = LanguageManager::instance().language(name);\n");
  print_cpp("if (lang)\n{\n");
  indent_level++;
  g_hash_table_foreach(decl_get_root_namespace()->members,
                       &print_reg_call_tunnel, "*lang");
  indent_level--;
  print_cpp("}\n"); /* if (lang) */
  
  indent_level--;
  print_cpp("}\n"); /* got_language */
  
  indent_level--;
  indent_level--;
  print_cpp("};\n\n"); /* class */


  /* plugin initialisation function */
  print_cpp("extern \"C\" Yehia::Plugin *yehia_%s_plugin_init("
            "Yehia::PluginManager *mgr)\n", name);
  print_cpp("{\n");
  indent_level++;
  print_cpp("try\n{\n");
  indent_level++;

  /* we must first load modules we depend on */
  node = TREE_META_PLUGIN_USED_MODULES(mplugin);
  while (node != NULL_TREE)
  {
    print_cpp("mgr->load_plugin(\"%s\");\n",
              format_dot_identifier_list(TREE_VALUE(node)));
    node = TREE_CHAIN(node);
  }
  
  print_cpp("Yehia::Plugin *plugin = SigC::manage(new %sPlugin(*mgr));\n", name);
  print_cpp("plugin->reference();\n"
            "return plugin;\n");
  indent_level--;
  print_cpp("}\n");
  print_cpp("catch (...)\n"
            "{\n");
  indent_level++;
  print_cpp("mgr->set_error(\"%sPlugin initialisation failed\");\n"
            "return 0;\n", name);
  indent_level--;
  print_cpp("}\n");
  indent_level--;
  print_cpp("}\n");
}

static void process_ns(Namespace *parent, const char *parent_objname,
                       Tree ns_stmt)
{
  const char *name = TREE_IDENTIFIER_STR(TREE_NAMESPACE_IDENTIFIER(ns_stmt));
  Namespace *ns = decl_namespace(parent, name, infilename, yylineno);
  Tree node;
  
  
  print_cpp("NamespaceBuilder ns%s(factory, %s, \"%s\");\n\n",
            name, parent_objname, name);
         
  for (node = TREE_NAMESPACE_MEMBERS(ns_stmt); node; node = TREE_CHAIN(node))
  {
    Tree value = TREE_VALUE(node);
    switch (TREE_CODE(value))
    {
      case VALUE_STMT:
        print_cpp("ns%s.add(\"%s\", %s);\n", name,
                  TREE_IDENTIFIER_STR(TREE_VALUE_IDENTIFIER(value)),
                  format_value(TREE_VALUE_VAL(value)));
        break;
      case ATTR_STMT:
        if (TREE_ATTR_TYPE(value) != ATTR_METHOD)
          process_internal_error("invalid namespace member");

        process_attribute(name, NULL, NULL, value);
        break;
      case CLASS_STMT:
        process_class(ns, value);
        break;
      case NAMESPACE_STMT:
      {
        int strsz = 0, bufsz = 0;
        print_cpp("{\n");
        indent_level++;
        process_ns(ns, GC_str_concat(NULL, &strsz, &bufsz,
                                        "ns", name, NULL), value);
        indent_level--;
        print_cpp("}\n");
        break;
      }
      case CLASS_DECL:
        decl_class(ns, TREE_IDENTIFIER_STR(TREE_CLASSDECL_IDENTIFIER(value)),
                   TREE_CLASSDECL_DEFINED_BY(value), FALSE,
                   infilename, yylineno);
        break;
      default:
        process_internal_error("invalid namespace member");
    }
  }
}

static void process_class(Namespace *ns, Tree klass)
{
  Tree node;
  Tree defby = TREE_CLASS_DEFINED_BY(klass);
  const char *nsname = ns->identifier;
  const char *name = TREE_IDENTIFIER_STR(TREE_CLASS_IDENTIFIER(klass));
  const char *defbyname;

  if (defby != NULL_TREE && TREE_CODE(defby) == VERBATIM)
  {
    process_error("inline class definition not implemented");
    return;
  }
  
  defbyname = format_typename(TREE_CLASS_DEFINED_BY(klass));

  if (TREE_CLASS_SUPERS(klass) != NULL_TREE)
  {
    print_cpp("bases.erase(bases.begin(), bases.end());\n");
    for (node = TREE_CLASS_SUPERS(klass); node; node = TREE_CHAIN(node))
    {
      char *base_name;
      Class *superklass = decl_find_class(ns, TREE_VALUE(node), &base_name);
      if (!superklass)
        return; /* error was handled in find_class */

      print_cpp("bases.push_back(&typeid(%s));\n",
                format_typename(superklass->defined_by));
    }
    print_cpp("ClassBuilder<%s> cl%s(factory, ns%s, \"%s\", bases);\n",
              defbyname, name, nsname, name);
  }
  else
    print_cpp("ClassBuilder<%s> cl%s(factory, ns%s, \"%s\");\n",
              defbyname, name, nsname, name);
    
  for (node = TREE_CLASS_MEMBERS(klass); node; node = TREE_CHAIN(node))
  {
    Tree value = TREE_VALUE(node);
    
    switch (TREE_CODE(value))
    {
      case ATTR_STMT:
        process_attribute(name, defbyname, klass, value);
        break;
      case VERBATIM:
        print_cpp("{\n");
        print_cpp_ni("%s", TREE_VERBATIM_TEXT(value));
        print_cpp("}\n");
        break;
      default:
        process_internal_error("invalid class member code");
        break;
    }
  }
  decl_class(ns, name, TREE_CLASS_DEFINED_BY(klass), TRUE,
             infilename, yylineno);
}

static void process_attribute(const char *clname, const char *clrealname,
                              Tree klass, Tree a)
{
  Tree defby_node = TREE_ATTR_DEFINED_BY(a);

  if (TREE_ATTR_PURE(a))
    return;
  
  if (TREE_ATTR_TYPE(a) == ATTR_CONSTRUCTOR)
  {
    /* constructors are somewhat special */
    int len;
    Tree node1, sig = TREE_ATTR_SIGNATURE(a);
    
    len = TREE_IS_EMPTY_PARAMLIST(sig) ? 0 : tree_list_length(sig);
    if (TREE_CLASS_VPROXY_ID(klass) > 0)
      print_cpp("cl%s.constructor(%sVirtualConstructor%d<ucxxVP%d%s",
                clname, TREE_CLASS_WRAPPED(klass) ? "Wrap" : "", len,
                TREE_CLASS_VPROXY_ID(klass), clname);
    else
      print_cpp("cl%s.constructor(%sConstructor%d<%s", clname, 
                TREE_CLASS_WRAPPED(klass) ? "Wrap" : "", len,
                defby_node ? format_typename(defby_node) : clrealname);
    for (node1 = sig; node1; node1 = TREE_CHAIN(node1))
    {
      Tree type = TREE_PARAMDESCR_TYPE(TREE_VALUE(node1));
      if (type == NULL_TREE)
        break;
      print_cpp_ni(", %s", format_typedescr(type));
    }
    print_cpp_ni(">());\n");
  }
  else
  {
    Tree sig = TREE_ATTR_SIGNATURE(a);
    Tree retspec = TREE_ATTR_RET_SPEC(a);
    const char *ident = TREE_IDENTIFIER_STR(TREE_ATTR_IDENTIFIER(a));
    const char *defby = attr_get_defby_name(a);
    int bufsz = 64, strsz = 0;
    char *slotexpr = GC_malloc(bufsz);

    slotexpr[0] = '\0';

    if ((sig || retspec) && TREE_ATTR_INLINE_ID(a) == 0)
    {

      if (TREE_ATTR_TYPE(a) == ATTR_GETTER)
      {
        if (!TREE_IS_EMPTY_PARAMLIST(sig))
        {
          process_error_stmt(a, "getter must have a signature specifying "
                             "only return type");
          return;
        }
        slotexpr = GC_str_concat(slotexpr, &strsz, &bufsz,
                                 "slot((", format_typedescr(retspec),
                                 "(", clrealname, "::*)())&", NULL);
      }
      else if (sig && retspec)
        slotexpr = GC_str_concat(
                slotexpr, &strsz, &bufsz, "slot((",
                format_typedescr(retspec), " (", clrealname, "::*)(",
                format_paramlist(sig, PARAM_TYPE), "))&", NULL);
      else
      {
        process_error_stmt(a, "must specify both paramlist and return type");
        return;
      }
    }
    else if (TREE_ATTR_TYPE(a) != ATTR_SIGNAL)
      slotexpr = GC_str_append(slotexpr, "slot(&", 6, &strsz, &bufsz);
    
    if (klass && TREE_CLASS_VPROXY_ID(klass) > 0 && TREE_ATTR_VIRTUAL(a))
    {
      char buf[32];
      sprintf(buf, "%d", TREE_CLASS_VPROXY_ID(klass));
      slotexpr = GC_str_concat(slotexpr, &strsz, &bufsz, "ucxxVP", buf, clname,
                               "::ucxxdfl_", defby, NULL);
    }
    else if ((!defby_node || !TREE_CHAIN(defby_node)) && clrealname &&
             TREE_ATTR_INLINE_ID(a) == 0)
    {
      /* a single identfier as 'typename' */
      slotexpr = GC_str_concat(slotexpr, &strsz, &bufsz, clrealname, "::",
                               defby, NULL);
    }
    else
    {
      /* function used as 'method', or namespace method */
      slotexpr = GC_str_concat(slotexpr, &strsz, &bufsz, defby, NULL);
    }
    
    if (TREE_ATTR_TYPE(a) != ATTR_SIGNAL)
      slotexpr = GC_str_append(slotexpr, ")", 1, &strsz, &bufsz);
    
    switch (TREE_ATTR_TYPE(a))
    {
      case ATTR_CLASS_METHOD:
        print_cpp("cl%s.class_method(\"%s\", %s);\n", clname, ident, slotexpr);
        break;
      case ATTR_METHOD:
        if (clrealname)
          print_cpp("cl%s.method(\"%s\", %s);\n", clname, ident, slotexpr);
        else
          print_cpp("ns%s.method(\"%s\", %s);\n", clname, ident, slotexpr);
        break;
      case ATTR_GETTER:
        print_cpp("cl%s.getter(\"%s\", %s);\n", clname, ident, slotexpr);
        break;
      case ATTR_SETTER:
        print_cpp("cl%s.setter(\"%s\", %s);\n",
                  clname, ident, slotexpr);
        break;
      case ATTR_SIGNAL:
        print_cpp("cl%s.getter(\"%s\", SignalWrapper<%s>(&%s).getter());\n",
                  clname, ident, clrealname, slotexpr);
        break;
      default:
        process_internal_error("invalid attribute type");
    }
  }
}

void process_error(const char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  
  fprintf(stderr, "%s:%d: ", infilename, yylineno);
  vfprintf(stderr, fmt, args);
  fputs("\n", stderr);
  
  va_end(args);
  
  error_count++;
}

void process_error_stmt(Tree stmt, const char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  
  fprintf(stderr, "%s:%d: ", TREE_FILENAME(stmt), TREE_LINENO(stmt));
  vfprintf(stderr, fmt, args);
  fputs("\n", stderr);
  
  va_end(args);
}

void process_error_cont(const char *fname, int lineno, const char *fmt, ...)
{
  va_list args;

  va_start(args, fmt);
  
  fprintf(stderr, "%s:%d: ", fname, lineno);
  vfprintf(stderr, fmt, args);
  fputs("\n", stderr);
  
  va_end(args);
}

void process_internal_error(const char *fmt, ...)
{
  va_list args;
  
  if (error_count > 0)
  {
    fprintf(stderr, "%s:%d: confused by earlier errors, bailing out\n",
            infilename, yylineno);
    exit(EXIT_FAILURE);
  }
  va_start(args, fmt);
  
  fprintf(stderr, "Internal error: ");
  vfprintf(stderr, fmt, args);
  fputs("\n", stderr);
  
  va_end(args);
  
  exit(EXIT_FAILURE);
}

int process_file(FILE *infile, const char *infname,
                 FILE *outfile)
{
  yyin = infile;
  infilename = infname;
  process_outfile = outfile;
  error_count = 0;
  
  yyparse();

  return error_count == 0;
}

const char *process_get_infile_name(void)
{
  return infilename;
}

int process_get_infile_lineno(void)
{
  return yylineno;
}


static void output_vtable(Tree klass);
static void output_inline_meths(Tree klass);
static void output_inline_meth(Tree klass, Tree meth);

static void process_output(Tree ns)
{
  Tree node;
  
  for (node = TREE_NAMESPACE_MEMBERS(ns); node; node = TREE_CHAIN(node))
  {
    Tree value = TREE_VALUE(node);

    switch (TREE_CODE(value))
    {
      case CLASS_STMT:
        output_vtable(value);
        output_inline_meths(value);
        break;
      case ATTR_STMT:
        output_inline_meth(NULL_TREE, value);
        break;
      case NAMESPACE_STMT:
        process_output(value);
        break;
      default:
        /* ignore */
        break;
    }
  }
}

static void process_verbatim(Tree v)
{
  /* print_lineinfo(v); */
  print_cpp_ni("%s", TREE_VERBATIM_TEXT(v));
}

static void output_vtable(Tree klass)
{
  static int vproxy_cnt = 1;
  Tree node;
  int i;
  const char *clname = TREE_IDENTIFIER_STR(TREE_CLASS_IDENTIFIER(klass));
  const char *defbyname = format_typename(TREE_CLASS_DEFINED_BY(klass));
  gboolean found_virtual = FALSE;

  for (node = TREE_CLASS_MEMBERS(klass); node; node = TREE_CHAIN(node))
  {
    Tree value = TREE_VALUE(node);
    
    if (TREE_CODE(value) == ATTR_STMT &&
        TREE_ATTR_TYPE(value) == ATTR_METHOD && TREE_ATTR_VIRTUAL(value))
    {
      found_virtual = TRUE;
      break;
    }
  }

  if (!found_virtual)
    return;

  print_cpp("//\n"
            "// Virtual method proxy for class '%s'\n"
            "//\n", clname);
  print_cpp("class ucxxVP%d%s : public %s\n"
            "{\n", vproxy_cnt, clname, defbyname);
  indent_level++;
  print_cpp("public:\n");
  indent_level++;
  
  for (node = TREE_CLASS_MEMBERS(klass); node; node = TREE_CHAIN(node))
  {
    Tree value = TREE_VALUE(node);
    
    if (TREE_ATTR_TYPE(value) == ATTR_CONSTRUCTOR)
    {
      Tree sig = TREE_ATTR_SIGNATURE(value);
      
      print_cpp("ucxxVP%d%s(Yehia::Script::Language *lang, "
                "Yehia::Script::Namespace *scope, "
                "Yehia::Script::Object *instance",
                vproxy_cnt, clname);
      if (!TREE_IS_EMPTY_PARAMLIST(sig))
      {
        print_cpp_ni(", %s) : %s(%s",
                     format_paramlist(sig, PARAM_BOTH | PARAM_NUMBER_NAME),
                     defbyname,
                     format_paramlist(sig, PARAM_NAME | PARAM_NUMBER_NAME));
      }
      print_cpp_ni(") {\n");
      indent_level++;
      print_cpp("inst_ = instance;\n"
                "inst_->reference();\n"
                "scope_ = scope;\n"
                "lang_ = lang;\n",
                "//scope_->reference();\n", defbyname);
      indent_level--;
      print_cpp("}\n");
    }
    else if (TREE_ATTR_TYPE(value) == ATTR_METHOD && TREE_ATTR_VIRTUAL(value))
    {
      Tree node1;
      Tree sig = TREE_ATTR_SIGNATURE(value);
      const char *retspec = format_typedescr(TREE_ATTR_RET_SPEC(value));
      const char *methname = TREE_IDENTIFIER_STR(TREE_ATTR_IDENTIFIER(value));
      
      print_cpp("virtual %s %s(%s) %s {\n", retspec, methname,
                format_paramlist(sig, PARAM_BOTH | PARAM_NUMBER_NAME),
                TREE_ATTR_CONST(value) ? "const" : "");
      indent_level++;
      print_cpp("using namespace Yehia;\n"
                "using namespace Yehia::Script;\n\n");
      print_cpp("std::list<Any> args;\n"
                "args.push_back(inst_->value());\n");
      if (!TREE_IS_EMPTY_PARAMLIST(sig))
        for (i = 1, node1 = sig; node1; node1 = TREE_CHAIN(node1), i++)
          print_cpp("args.push_back(MarshalTraits<%s>::marshal(p%d));\n",
                    format_typedescr(
                            TREE_PARAMDESCR_TYPE(TREE_VALUE(node1))), i);

      if (strcmp(retspec, "void") != 0)
      {
        print_cpp("Any result = call_script_function(lang_, "
                "scope_, \"%s\", args);\n", methname);
        print_cpp("return MarshalTraits<%s>::unmarshal(result);\n", retspec);
      }
      else
        print_cpp("call_script_function(lang_, scope_, \"%s\", args);\n",
                  methname);
      indent_level--; /* virtual method */
      print_cpp("}\n");

      /* default implementation */
      print_cpp("%s ucxxdfl_%s(%s) {\n", retspec, methname,
                format_paramlist(sig, PARAM_BOTH | PARAM_NUMBER_NAME));
      indent_level++;
      if (strcmp(retspec, "void") == 0)
        print_cpp("%s::%s(", defbyname, methname);
      else
        print_cpp("return %s::%s(", defbyname, methname);
      print_cpp_ni("%s);\n", format_paramlist(sig, PARAM_NAME | PARAM_NUMBER_NAME));
      indent_level--;
      print_cpp("}\n");
    }
  }
  /* destructor */
  print_cpp("~ucxxVP%d%s() {\n", vproxy_cnt, clname);
  indent_level++;
  print_cpp("inst_->unreference();\n"
            "//scope_->unreference();\n");
  indent_level--;
  print_cpp("}\n");
  indent_level--;
  print_cpp("private:\n");
  indent_level++;
  print_cpp("Yehia::Script::Object *inst_;\n"
            "%s *cppinst_;\n"
            "Yehia::Script::Namespace *scope_;\n"
            "Yehia::Script::Language *lang_;\n", defbyname);
  indent_level -= 2;
  print_cpp("};\n");

  TREE_CLASS_VPROXY_ID(klass) = vproxy_cnt++;
}

static void output_inline_meths(Tree klass)
{
  Tree node;
  
  for (node = TREE_CLASS_MEMBERS(klass); node; node = TREE_CHAIN(node))
  {
    Tree value = TREE_VALUE(node);
    if (TREE_CODE(value) != ATTR_STMT)
      continue;
    
    output_inline_meth(klass, value);
  }    
}

static void output_inline_meth(Tree klass, Tree meth)
{
  static int inline_cnt = 1;

  Tree sig = TREE_ATTR_SIGNATURE(meth);
  Tree retspec = TREE_ATTR_RET_SPEC(meth);
  Tree code = TREE_ATTR_DEFINED_BY(meth);
  
  if (code == NULL_TREE || TREE_CODE(code) != VERBATIM)
    return;
  
  if (retspec == NULL_TREE)
  {
    process_error_stmt(meth,
                       "missing return type specification for inline method");
    return;
  }

  TREE_ATTR_INLINE_ID(meth) = inline_cnt++;

  if (klass && TREE_ATTR_TYPE(meth) != ATTR_CLASS_METHOD)
  {
    Tree typname = tree_build_typename(TREE_CLASS_IDENTIFIER(klass),
                                       NULL_TREE);
    Tree typespec = tree_cons(typname, NULL_TREE);
    Tree classtype = tree_build_typedescr(FALSE, typespec, TYPEDESCR_REF);
    if (TREE_IS_EMPTY_PARAMLIST(sig))
      sig = NULL_TREE;
    sig = tree_cons(tree_build_paramdescr(classtype, tree_build_identifier("self")), sig);
  }
  print_cpp("%s ucxxinl%d__(%s)\n", format_typedescr(retspec),
            TREE_ATTR_INLINE_ID(meth), format_paramlist(sig, PARAM_BOTH));
  print_cpp("{\n");
  indent_level++;
  process_verbatim(code);
  indent_level--;
  print_cpp("}\n");
}

static const char *attr_get_defby_name(Tree attr)
{
  Tree defby_node = TREE_ATTR_DEFINED_BY(attr);
  
  if (!defby_node)
    return TREE_IDENTIFIER_STR(TREE_ATTR_IDENTIFIER(attr));
  else if (TREE_ATTR_INLINE_ID(attr) == 0)
    return format_typename(defby_node);
  else
  {
    char *name = GC_malloc(64);
    snprintf(name, 64, "ucxxinl%d__", TREE_ATTR_INLINE_ID(attr));
    return name;
  }
}

static void print_cpp(const char *msg, ...)
{
  va_list args;
  const char *cp;
  int i, j, bufsz = strlen(msg) + 5 * INDENT_OFF;
  char *buffer = GC_malloc(bufsz);

  for (i = 0, cp = msg; *cp; i++, cp++)
  {
    if (i >= bufsz)
    {
      buffer = GC_realloc(buffer, bufsz + 5 * INDENT_OFF);
      bufsz += 5 * INDENT_OFF;
    }
    buffer[i] = *cp;
    if (*cp == '\n'&& cp[1] != '\0')
      for (j = indent_level * INDENT_OFF; j > 0; j--)
        buffer[++i] = ' ';
  }
  buffer[i] = '\0';

  for (i = indent_level * INDENT_OFF; i > 0; i--)
    fputc(' ', process_outfile);
  
  va_start(args, msg);
  vfprintf(process_outfile, buffer, args);
  va_end(args);

  GC_free(buffer);
}

static void print_cpp_ni(const char *msg, ...)
{
  va_list args;

  va_start(args, msg);
  vfprintf(process_outfile, msg, args);
  va_end(args);
}

static void print_lineinfo(Tree t)
{
  if (TREE_LINENO(t) > 0 && TREE_FILENAME(t) != NULL)
    print_cpp_ni("#line %d \"%s\"\n", TREE_LINENO(t), TREE_FILENAME(t));
}

