/**
 * @file    ygraph.h
 * @brief   Main header file.
 *
 *          Defines useful global macros.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __YGRAPH_H__
#define __YGRAPH_H__

#ifndef __G_LIB_H__
#  include <glib.h>
#endif

#ifndef __GDK_H__
#  include <gdk/gdk.h>
#endif

#ifndef __GTK_H__
#  include <gtk/gtk.h>
#endif

#ifndef __GDK_IMLIB_H__
#  include <gdk_imlib.h>
#endif

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>

/* #define YDEBUG */
#ifdef YDEBUG
#define YDEB(x) printf(x)
#else
#define YDEB(x) /*(x)*/
#endif

/*
 * Constants.
 */

/* window labels */
#define CONTROL_PANEL_NAME              "ygraph-control"
#define SCALE_PANEL_NAME                "ygraph function rescale"
#define SCALE_TITLE_STRING              "re-scale factor"
#define FILE_SELECTION_TITLE            "read file"
#define EXPORT_TO_FILE_TITLE            "export file"
#define EXPORT_TO_DIR_TITLE             "export directory"
#define ABOUT_TITLE                     "about"
#define MESSAGE_TITLE                   "message ..."
#define KEYBINDING_TITLE                "ygraph keybindings"
#define SUBTRACT_DIALOG_TITLE           "subtract: A - B"
#define SUBTRACT_MESSAGE                "Subtract files: A - B"
#define CONVERGENCE_DIALOG_TITLE        "convergence test"
#define CONVERGENCE_MESSAGE             "Convergence test"
#define EXPORT_FILE_SELECT_MESSAGE      "Export current frame to file"
#define EXPORT_FILE_LABEL               "File:"
#define EXPORT_DIR_SELECT_MESSAGE       "Export multiple frames to directory"
#define EXPORT_DIR_LABEL                "Directory:"
#define FILE_A_LABEL                    "File A:"
#define FILE_B_LABEL                    "File B:"
#define FILE_1_LABEL                    "File 1 (1xres):"
#define FILE_2_LABEL                    "File 2 (2xres):"
#define FILE_4_LABEL                    "File 3 (4xres):"
#define NEW_WINDOW_LABEL                "plot in new window"
#define RANGE_DIALOG_TITLE              "xy range"
#define XMIN_LABEL                      "x min"
#define XMAX_LABEL                      "x max"
#define YMIN_LABEL                      "y min"
#define YMAX_LABEL                      "y max"
#define SAVE_FILE_BASE_NAME             "frame"
#define X_AXIS_TYPE_LABEL               "x-axis"
#define Y_AXIS_TYPE_LABEL               "y-axis"
#define AXIS_LINEAR_TYPE_LABEL          "linear"
#define AXIS_LOG_TYPE_LABEL             "log"

/* button labels */
#define STEP_BACK_BUTTON_LABEL          "<Step"
#define STEP_FWD_BUTTON_LABEL           "Step>"
#define START_BUTTON_LABEL              "<<Start"
#define END_BUTTON_LABEL                "End>>"
#define PLAY_BUTTON_LABEL               "Play"
#define PAUSE_BUTTON_LABEL              "Pause"
#define TIME_LABEL                      "Time"
#define DELAY_LABEL                     "Delay"
#define SHOW_ALL_BUTTON_LABEL           "Show All"
#define RELOAD_BUTTON_LABEL             "Reload"
#define NEW_BUTTON_LABEL                "New plot"
#define QUIT_BUTTON_LABEL               "Quit"
#define OKAY_BUTTON_LABEL               "Okay"
#define APPLY_BUTTON_LABEL              "Apply"
#define CANCEL_BUTTON_LABEL             "Cancel"
#define SELECT_LABEL                    "Select ..."

#define TIME_STR_SIZE                   10
#define DELAY_STR_SIZE                  6
#define FILE_STR_SIZE                   255
#define FILE_ENTRY_LENGTH               120
#define RANGE_VAL_SIZE                  12
#define MAX_PIXELS                      10000

#define FIRST_FRAME                     0
#define TIME_EPSILON                    1e-10   

#define FONT_FONT "-adobe-helvetica-bold-r-normal-*-*-%d-*-*-*-*-*-*"

#define FONT_NORMALSIZE      get_font(FONT_FONT, 100)
#define FONT_Large           get_font(FONT_FONT, 120)
#define FONT_LARGE           get_font(FONT_FONT, 140)
#define FONT_Huge            get_font(FONT_FONT, 170)
#define FONT_HUGE            get_font(FONT_FONT, 200)

#define X_PADDING                       20
#define Y_PADDING                       20
#define DEFAULT_PLOT_FONT               FONT_NORMALSIZE
#define GRID_INTENSITY                  40000

#define OPTIMUM_TICK_MARK_NUMBER        6
#define SHORT_TICK_LENGTH               5
#define LONG_TICK_LENGTH                8
#define MAX_AXIS_LABEL_LENGTH           40
#define AXIS_EPSILON                    1e-6
#define DEFAULT_AXIS_FONT               FONT_NORMALSIZE

#define TEXT_X_WIDTH_PADDING            2
#define TEXT_X_HEIGHT_PADDING           2
#define TEXT_Y_WIDTH_PADDING            2
#define TEXT_Y_HEIGHT_PADDING           2

#define MESSAGE_X_PADDING               20
#define MESSAGE_Y_PADDING               20

#define DEFAULT_LEGEND_WIDTH            80
#define LEGEND_WIDTH_PADDING            5
#define LEGEND_HEIGHT_PADDING           2
#define LEGEND_LINE_DIVISIONS           20
#define LEGEND_LINE_THICKNESS           4
#define LEGEND_OFF                      0
#define LEGEND_RIGHT                    1
#define LEGEND_TOP                      2
#define LEGEND_OVER                     3
#define DEFAULT_LEGEND_FONT             FONT_NORMALSIZE
#define ALPHABET "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYX0123456789"

#define MAX_LINE_LENGTH                 255
#define TIME_STR_MAX_SIZE               20
#define RANGE_STR_MAX_SIZE              80

#define DEFAULT_I_SIZE                  400
#define DEFAULT_J_SIZE                  400
#define DEFAULT_LINE_WIDTH              1
#define DOT_SIZE                        4
#define CROSS_SIZE                      2

#define NAN_STRING                      "nan nanq inf"
#define COMMENT_DELIMETERS              "\"#"
#define CMD_LINE_RANGE_DELIMETER        ","
#define TIME_INDICATOR_STRING           "time"
#define LABEL_INDICATOR_STRING          "label"
#define EQUAL_SIGN                      '='
#define UNINITIALISED_TIME              -G_MAXDOUBLE

#define FILENAME_SIZE                   255
#define SCALE_ENTRY_SIZE                8

#define NULLC                           '\0'

#define NCOLORS                         7

#define NO_ZOOM                         -1

#define INIT_DATA                       0
#define APPEND_DATA                     1
#define RESCALE_DATA                    2

#define DEFAULT_FONT                    FONT_NORMALSIZE
#define BIG_FONT                        FONT_LARGE

#define DEFAULT_ANIMATE_DELAY           100 /* milliseconds */
#define DEFAULT_SKIP_STEP               1

#define DEFAULT_DATA_SCALE_FACTOR       1.0

#define READ_COMMENT                    0
#define READ_DATA                       1

#define NEW_DATA_SET                    -1
#define FAIL                            -1
#define SUCCESS                          0

#define INTERP_ORDER                     1
#define MAX_INTERP_ORDER                 1

/* Special plot types */

#define YG_DATAFILE                      0
#define YG_SUBTRACT                      1
#define YG_DERIVATIVE                    3
#define YG_FRAME_INT                     4
#define YG_TIME_INT                      5
#define YG_INT                           6

#define INTERNAL_STDIN_STR "ygraph#internal#stdin#string"

#define CONVERGENCE_LABEL                "convergence"

#define HELP_MSG                         "Type 'ygraph --help' for help"
#define COPYRIGHT_MSG             "(c) 2002-2003 Denis Pollney <denis.pollney@aei.mpg.de>"

#define UNUSED(x) x=x

/**
 * @defgroup macros Macros.
 * @{
 */

/**
 * A macro to set individual array elements is missing from glib, but this
 * macro does the job.
 */
#define array_index_set_val(a,t,i,v)     (((t*) (a)->data) [(i)]) = v

/** @} */

/**
 * @defgroup enumerations Enumerations.
 * @{
 */

/**
 * Possible modes for displaying the plot area.
 */
typedef enum
  {
    SHOW_ALL_MODE,                   /**< show all frames of all datasets    */
    ANIMATE_MODE,                    /**< loop through the frames            */
    PAUSE_MODE                       /**< show a single frame of each dataset*/
  } DISPLAY_MODE;

/**
 * Flags to distinguish between the two axis types.
 */
typedef enum
  {
    X_AXIS,                          /**< x-axis orientation                 */
    Y_AXIS                           /**< y-axis orientation                 */
  } AXIS_ORIENTATION;

/**
 * Flags to distinguish between the two axis types.
 */
typedef enum
  {
    AXIS_LINEAR,                     /**< linear scale                       */
    AXIS_LOG                         /**< log scale                          */
  } AXIS_TYPE;

/** @} */

/**
 * @defgroup structs Data structures.
 * @{
 */

/**
 * @brief    Describes an individual tick-mark on an axis.
 */
struct _Tick
{
  gdouble val;                       /**< the coordinate value of the tick   */
  gchar* str;                        /**< same value, converted to a string  */
};
typedef struct _Tick Tick;

struct _Point_Double
{
  gdouble x;
  gdouble y;
};
typedef struct _Point_Double Point_Double;

/**
 * @brief    Describes the x and y axes of a plot.
 */
struct _Axis
{
  AXIS_ORIENTATION orientation;      /**< either X_AXIS or Y_AXIS            */
  AXIS_TYPE type;                    /**< axis scale type (linear or log)    */
  GdkFont* font;                     /**< the plot axis font                 */
  GdkGC* gc;                         /**< graphics context for the axis      */
  GArray* ticks;                     /**< a list of tick mark locations      */
  gint width;                        /**< width needed by the axis (pixels)  */
  gint height;                       /**< height needed by the axis(pixels)  */
};
typedef struct _Axis Axis;

/**
 * @brief    Describes the contents of the plot legend.
 */
struct _Legend
{
  GdkFont* font;                     /**< font for the legend                */
  GdkGC* gc;                         /**< graphics context for the legend    */
  GdkPixmap* pixmap;                 /**< holds an image of the legend       */
  GArray* names;                     /**< list of data set names to be drawn */
  gint width;                        /**< width of the legend in pixels      */
  gint height;                       /**< height of the legend in pixels     */
  gint position;                     /**< position of the legend             */
  gboolean rebuild;                  /**< rebuild the legend flag            */
};
typedef struct _Legend Legend;

/**
 * @brief    Describes an individual frame of xy-data.
 */
struct _Frame
{
  gdouble time;                      /**< the frame time                     */
  gdouble x_range[2];                /**< range along the x axis [min,max]   */
  gdouble y_range[2];                /**< range along the x axis [min,max]   */
  gint npoints;                      /**< number of points in the frame      */
  GArray *xy_data;                   /**< list of xy data points             */
};
typedef struct _Frame Frame;

/**
 * @brief    Describes a collection of frame as a single set of data.
 */
struct _DataSet
{
  gint type;                         /**< one of the YG_... types            */
  gchar* name;                       /**< a name to be display in the legend */
  gchar* fname;                      /**< full filename (including directory)*/
  gchar* dir;                        /**< the directory containing the file  */
  gint nframes;                      /**< number of frames in the data set   */
  gdouble x_range[2];                /**< max and min of all frame x values  */
  gdouble y_range[2];                /**< max and min of all frame y values  */
  gdouble scale;                     /**< value by which to scale y values   */
  GArray* frame;                     /**< a list of frames                   */
  GArray* cmpt_set;                  /**< a list of dataset indices used to
                                        build this set (eg. for special plot
                                        types)                               */
  GtkEntry* scale_entry_field;       /**< a widget for entering scale vals   */
};
typedef struct _DataSet DataSet;

/**
 * @brief    Describes a frame (line) in terms of window dependent ij to be
 *           plotted.
 */
struct _PlotLine
{
  GdkGC* gc;                         /**< a graphics context for the line    */
  Frame* frame;                      /**< pointer to the corresponding frame */
  GdkPoint* ij_data;                 /**< ij coordinates of the line points  */
  GdkColor* start_color;             /**< color for the first line of the set*/
  GdkColor* end_color;               /**< color of the last line of the set  */
  gint npoints;                      /**< number of points in the frame      */
};
typedef struct _PlotLine PlotLine;

/**
 * @brief    Describes all of the data contained in a single plot window.
 */
struct _Plot
{
  gint plot_nbr;                     /**< a unique identifier for the window */

  GtkWidget* window;                 /**< pointer to the plot's window widget*/
  GtkWidget* plot_area;              /**< the drawing area of the window     */
  GdkPixmap* pixmap;                 /**< the pixmap in which data is drawn  */
  GtkItemFactory* menu;              /**< the menu bar along the top         */
  GdkGC* gc;                         /**< a graphics context for the window  */
  GdkFont* font;                     /**< default font for text in the window*/
  gint font_height;                  /**< approx of default font height      */
  GdkColormap* colormap;             /**< a colormap for the window          */

  GArray* data;                      /**< xy-data (array of DataSet*)        */
  GArray* plot_data;                 /**< ij-data (array of PlotLine*)       */
  GArray* current_frame;             /**< identify the current frame for each*/
                                     /**< data set (gint)                    */
  Legend* legend;                    /**< the plot legend                    */
  Axis* x_axis;                      /**< the plot x-axis                    */
  Axis* y_axis;                      /**< the plot y-axis                    */

  gdouble x_range[2];                /**< x-axis range to be plotted         */
  gdouble y_range[2];                /**< y-axis range to be plotted         */
  gint i_origin;                     /**< plot i-origin within the window    */
  gint j_origin;                     /**< plot j-origin within the window    */
  gint i_size;                       /**< length (in pixels) of the x-axis   */
  gint j_size;                       /**< length (in pixels) of the y-axis   */
  gint zoom_x_start;                 /**< starting x coord of a zoom region  */
  gint zoom_y_start;                 /**< starting y coord of a zoom region  */
  GArray* zoom_history_start;        /**< Array of previous zooms            */
  GArray* zoom_history_end;          /**< Array of previous zooms            */

  gboolean draw_points;              /**< draw dotss at data values          */
  gboolean fixed_range;              /**< fix the range, ie. no zoom         */
  gboolean draw_grid;                /**< draw a background grid             */
  gboolean draw_range;               /**< print range values on the plot     */

  gchar* current_directory;          /**< directory from which to load files */
};
typedef struct _Plot Plot;

/**
 * @brief    Describes the widgets contained in the control panel.
 */
struct _ControlPanel
{
  GtkWidget* window;                 /**< the control panel window           */
  GtkWidget* play_button;            /**< the play button                    */
  GtkWidget* time_entry;             /**< the time entry field               */
  GtkWidget* delay_entry;            /**< the animation delay entry field    */
};
typedef struct _ControlPanel ControlPanel;

/** @} */

/*
 * Global variables;
 */

/* A set of colors from which to choose defaults for plot lines. */
extern gulong global_base_color[NCOLORS][3];

/* The plot window is in one of three states given by the DISPLAY_MODE enum.
 * This variable holds the current state so that it can be conveniently
 * accessed when updating windows. Note that all windows are assumed to
 * have the same state.
 */
extern DISPLAY_MODE global_display_mode;     

/* An array of Plot* plots. Each entry corresponds to a unique plot window. */
extern GArray* global_plot_window;

/* A list of all of the data sets that have been loaded. These are pointers
 * to the data sets which are also contained in the Plot* structures, and
 * is used so that we can easily process the data sets without having to
 * go through the main structure.
 * One issue is that when data sets are freed, both the Plot* pointer and
 * the global_data_set_list entry need to be removed.
 */
extern GArray* global_data_set_list;

/* A (currently unused) integer which is used to assign unique identifiers to
 * plot windows.
 */
extern gint global_plot_nbr;

/* A global pointer to the control panel so that it's easy to get at. */
extern ControlPanel* global_control_panel;

/* A global pointer to the scale panel so that it's easy to get at. */
extern GtkWidget* global_scale_panel;

/* A convenience pointer holding the about window data. */
extern GdkPixmap* global_about_pixmap;

/* The current frame, common to all windows, used for animation. */
extern gint global_current_frame;

/* A list holding the unique times contained in each of the frames of each
 * of the datasets. Animation works like this: Starting from frame zero,
 * times from the global_time_list are read one at a time, defining the
 * 'current time'. Frames from each of the datasets are compared with this
 * value. The nearest frame whose time is <= the current time is displayed
 * for each dataset.
 */
extern GArray* global_time_list;

/* The total number of frames that are loaded, corresponding to the number of
 * entries in the time list.
 */
extern gint global_last_frame;

/* Intensity of the background grid. Maybe this should be an option_ instead */
extern gulong global_grid_intensity;

/* Current working directory (directory in which the program was started). */
extern gchar* global_working_directory;

/* User-set values for the x and y axis ranges. */
extern gdouble* global_x_range;
extern gdouble* global_y_range;

/* User-set values for the x and y columns */
extern gint global_x_column;
extern gint global_y_column;
extern gchar *global_column_format_string;

/*
 * Command line options.
 */

/* Don't create a control panel on startup. */
extern gboolean option_start_without_control_panel;

/* Print a version message and quit. */
extern gboolean option_version_msg;

/* Create a separate window for each dataset listed on the command line. */
extern gboolean option_separate_start_windows;

/* Draw dots at the data points by default. */
extern gboolean option_draw_points;

/* Draw a background grid on the plots by default. */
extern gboolean option_draw_grid;

/* Draw range values on the corners of the plot by default. */
extern gboolean option_draw_range;

/* Width of the lines in a plot. */
extern gint option_line_width;

/* Time delay between individual frames, measured in milliseconds. */
extern gint option_animate_delay;

/* Read only every nth frame from input files */
extern gint option_read_skip_step;

/* Position of the legend relative to the plot data */
extern gchar* option_legend_position;

/* List of potential NaN strings */
extern gchar** nan_string;

/* The actual used font, only used as placeholder */
extern gchar* actual_font;

/* steerable by command line */
extern gdouble font_size_factor;


/* ************** function definitions *************** */

/* about.c */
void about_window_create(void);
void help_window_create(void);
void keybinding_window_create(void);

/* axis.c */
gdouble linear2log(const gdouble value);
gdouble axis_transformation(const gdouble value, const Axis* axis);
void plot_range_set(Plot* plot, gint mode);
GArray* axis_tick_append(GArray* ticks, gdouble val, Axis* axis,
                         gboolean exp_notation);
GArray* axis_calc_ticks(gdouble min, gdouble max, Axis* axis);
gint axis_get_text_width(GdkFont* font, gchar* label, gboolean axis_type);
gint axis_get_text_height(GdkFont* font, gchar* label, gboolean axis_type);
gint axis_calc_width (Axis* axis);
gint axis_calc_height (Axis* axis);
gint axis_calc_tick_sep (Plot* plot, Axis* axis, guint i);
void axis_draw_x_text(Plot* plot);
void axis_draw_y_text(Plot* plot);
Axis* axis_create(Plot* plot, gint orientation, AXIS_TYPE type);
void axis_free (Axis* axis);
void axis_draw_lines(Plot* plot);
void axis_draw_log_indicator(Plot* plot, Axis* axis);
void axis_draw_log_indicator_xy(Plot* plot);
void axis_draw(Plot* plot);
void plot_axes_create(Plot* plot);
void axis_type_set_toggle(Axis* axis, GtkToggleButton* toggle_linear,
                          GtkToggleButton* toggle_log);
void range_entry_value_set(Plot* plot);
void range_dialog_create(Plot* plot, gint action, GtkItem* range_set_button);
gboolean range_reset(Plot* plot);
gboolean axis_type_reset(Plot* plot);
gboolean axis_set_from_dialog(GtkObject* button, Plot* plot);
void axis_toggle_log_state(Plot* plot);

/* callbacks.c */
gboolean plot_window_configure(GtkWidget* plot_area, GdkEventConfigure* event,
                               Plot* plot);
gboolean plot_window_expose(GtkWidget* plot_area, GdkEventExpose* event,
                            Plot* plot);
gboolean plot_window_destroy(GtkWidget* plot_area, GdkEventConfigure* event,
                             Plot* plot);
void plot_window_close(Plot* plot, gint action, GtkItem* file_select_button);
gboolean plot_area_motion_notify_event(GtkWidget* plot_area,
                                       GdkEventMotion* event, Plot* plot);
gboolean plot_area_button_release_event(GtkWidget* plot_area,
                                        GdkEventButton* event, Plot* plot);
gboolean plot_area_button_press_event(GtkWidget* plot_area,
                                      GdkEventButton* event, Plot* plot);
void plot_window_unzoom(Plot* plot);
void plot_window_empty_new(void);
gboolean file_select_read(GtkObject* ok_button, GtkFileSelection* fs);
void file_select(Plot* plot, gint action, GtkItem* file_select_button);
void file_select_new_window(Plot* plot, gint action,
                            GtkItem* file_select_button);
void plot_window_draw_points_toggle(Plot* plot, gint action,
                                    GtkItem* draw_points_toggle);
void plot_window_draw_grid_toggle(Plot* plot, gint action,
                                  GtkItem* draw_grid_toggle);
void plot_window_draw_range_toggle(Plot* plot, gint action,
                                   GtkItem* draw_range_toggle);
void plot_window_derivative(Plot* plot, gint action, GtkItem* button);
gboolean pause_play_button_click(GtkWidget* play_button, GdkEventExpose* event,
                                 gpointer* data);
gboolean start_button_click(GtkWidget* start_button, GdkEventExpose* event,
                            gpointer* data);
gboolean end_button_click(GtkWidget* end_button, GdkEventExpose* event,
                          gpointer* data);
gboolean step_fwd_button_click(GtkWidget* fwd_button, GdkEventExpose* event,
                               gpointer* data);
gboolean step_back_button_click(GtkWidget* back_button, GdkEventExpose* event,
                                gpointer* data);
gboolean show_all_button_click(GtkWidget* showall_button, GdkEventExpose* event,
                               gpointer* data);
gboolean plot_window_key_press_event(GtkWidget* plot_area, GdkEventKey* event,
                            Plot* plot);
gboolean delay_set(GtkEditable* delay_entry, gpointer data);
gboolean current_time_set(GtkEditable* time_entry, gpointer data);
gboolean data_reload_all(GtkWidget* reload_button, GdkEvent* event);
void data_reload_plot(Plot* plot, gint action, GtkItem* reload_button);
gboolean file_select_export(GtkObject* ok_button, GtkFileSelection* fs);
gboolean export_file_set(GtkEditable* file_entry, Plot* plot);
gboolean export_dir_set(GtkEditable* file_entry, Plot* plot);
void image_export_file_select(GtkWidget* select_button, GdkEvent* event,
                              Plot* plot);
void image_export_dialog(Plot* plot, gint action, GtkItem* export_button);
void image_dir_export_dialog(Plot* plot, gint action, GtkItem* export_button);
gboolean about_expose(GtkWidget* drawing_area, GdkEventExpose* event,
                      gpointer data);
gboolean help_expose(GtkWidget* text_area, GdkEventExpose* event,
                      gpointer data);
/* color.c */
GdkColor* color_set_rgb(GdkColormap* colormap,
                        gulong red, gulong green, gulong blue);
GdkGC* line_set_gc(Plot* plot, PlotLine* plot_data, gint frame_nbr,
                   gint nframes);
void plotline_set_colors(Plot* plot, PlotLine* plotline, gint i);

/* control_panel.c */
void current_frame_set(gint val);
void current_time_display_value(void);
void current_frame_increment(gint incr);
void control_panel_raise(void);
  
/* export.c */
void export_to_file(Plot* plot, gchar* filename);
void export_to_dir(Plot* plot, gchar* dirname);
  
/* legend.c */
void legend_free(Legend* legend);
void legend_create(Plot* plot);
void legend_draw(Plot* plot);

/* message.c */
void message_dialog(gchar* msg);

/* plot_area.c */
void plot_window_draw_time(Plot* plot);
void plot_area_draw(Plot* plot);
void plot_window_display_data(Plot* plot);
void all_windows_draw(void);
gint frame_draw_next(GtkObject* play_button);
void plot_area_write_location(Plot* plot, gint i, gint j);
void plot_area_zoom_start(Plot* plot, gint x, gint y);
void plot_area_zoom_finish(Plot* plot, gint x, gint y);
void plot_area_zoom_rectangle_draw(Plot* plot, gint x, gint y);

/* plot_window.c */
void plot_free(Plot* plot);
Plot* plot_data_init(GArray* data);
void plot_data_append(Plot* plot, gint data_set_idx);
void plot_window_reconfigure(Plot* plot);
void plot_window_build(Plot* plot);
void plot_window_expunge(Plot* plot);
void plot_window_display_all(Plot* plot);
void plot_window_show(void);

/* read_data.c */
DataSet* data_set_init(gchar* fname);
Frame* frame_init(void);
void frame_free(Frame* frame);
void dataset_append_frame(DataSet* data_set, Frame* frame);
void frame_append_data_point(Frame* frame, gdouble x_val, gdouble y_val);
gint global_data_set_list_append(DataSet* data_set);
gint dataset_read_from_file (gchar* fname, gint skip);
void time_list_merge(DataSet* data_set);
void time_list_build(void);
int check_for_nan(gchar* str);

/* scape_panel.c */
void scale_panel_create(Plot* plot, gint action, GtkItem* range_set_button);

/* special_fn.c */
void dataset_recalc(DataSet* data_set);
gint dataset_read_subtract_files(gchar* file0, gchar* file1);
void dataset_calc_rescale(DataSet* data_set, gdouble scale);
gint dataset_read_frame_integrate_file(gchar* file, gint option_read_skip_step,
                                                    gint option_t_int );
gint dataset_read_integrate_file(gchar* file, gint option_read_skip_step);
DataSet* dataset_derivative(gint idx);
void subtract_select_dialog_create(Plot* plot, gint action, GtkItem* button);
void file_select_A(GtkWidget* select_button, GdkEvent* event, Plot* plot);
void file_select_B(GtkWidget* select_button, GdkEvent* event, Plot* plot);
gboolean file_subtract(GtkObject* ok_button, Plot* plot);
  
/* util.c */
gchar* double_to_str(gdouble val, gboolean exp_notation);
gdouble str_to_double(gchar* s);
gdouble set_fac(const gdouble x, const gdouble y, const gint window_size);
gint x_to_i(Plot* plot, gdouble x);
gint y_to_j(Plot* plot, gdouble y);
gdouble i_to_x(Plot* plot, gint i);
gdouble j_to_y(Plot* plot, gint j);
void zoom_x_range_set(Plot* plot, gint i1, gint i2);
void zoom_y_range_set(Plot* plot, gint i1, gint i2);
gint current_frame_calc(DataSet* data_set);
DataSet* plot_get_data_index(Plot* plot, gint idx);
gdouble interp(GArray* xy_data, gdouble x_val, gint order);
void string_to_columns(gchar* range_string);
gdouble* string_to_range(gchar* range_string);
gchar* get_font(const char* font, gint font_size);

#endif /* __YGRAPH_H__ */

