#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../include/Y2/Y.h"
#include "../include/Y2/Ylib.h"
#include "../include/Y2/Ymixercodes.h"

#include "../include/string.h"


static void print_help(void);


/*
 *	Prints help message.
 */
static void print_help(void)
{
	printf("\
Usage: ymixer [mixer] [value1] [value2] [options]\n\
\n\
    [mixer] specifies the mixer channel device name, running without\n\
    any arguments prints a list of available mixer names.
\n\
    [value1] and [value2] specify the percent values in the domain\n\
    of [0.0%%, 100.0%%] to be set (decimals allowed). If neither is\n\
    specified then the current value is printed.\n\
\n\
    [options] can be any of the following:\n\
\n\
        --recorder <address:port>    Specify which Y server to connect to.\n\
\n\
    Return values:\n\
\n\
        0       Success.\n\
        1       General error.\n\
        2       Cannot connect to Y server error.\n\
        3       Systems error.\n\
\n"
	);

	return;
}


int main(int argc, char *argv[])
{
	int i, n;
	char *strptr;
	const char *con_arg = NULL;

	YConnection *con = NULL;

	int mixer_code = 0;
	int mixer_num = 0;

	int got_values = 0;
	Coefficient value1 = 0.0, value2 = 0.0;
	char *mixer_name[] = YMixerConicalNames;


	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Help. */
	    if(strcasepfx(argv[i], "--h") ||
               strcasepfx(argv[i], "-h") ||
               !strcmp(argv[i], "?")
	    )
	    {
		print_help();
		return(0);
	    }
            /* Connect address. */
            else if(strcasepfx(argv[i], "--rec") || 
                    strcasepfx(argv[i], "-rec")
            )
            {
                i++;
                if(i < argc)
                {
                    con_arg = argv[i];
                }
                else
                {
                    fprintf(stderr,
                        "%s: Requires argument.\n",
                        argv[i - 1]
                    );
                }
            }
	    /* Mixer code and index number, check if this is
	     * the first argument.
	     */
	    else if(i == 1)
	    {
		strptr = argv[i];

		/* Go through mixer names list. */
		for(n = 0; n < YTotalMixers; n++)
		{
		    /* Mixer names match? */
		    if(!strcmp(strptr, mixer_name[n]))
		    {
			/* Got match, but remember to add base offset
			 * to get correct mixer code, the mixer code
			 * is different from the mixer number.
			 */
			mixer_code = n + YMixerCodeBaseOffset;
			mixer_num = n;
		    }
		}
	    }
            /* Value 1. */
            else if(i == 2)
            {
		value1 = atof(argv[i]);
		got_values = 1;		/* Mark that we got value(s). */
            }
            /* Value 2. */
            else if(i == 3)
            {
                value2 = atof(argv[i]);
		got_values = 1;		/* Mark that we got value(s). */
            }
	}

	/* Connect to Y server. */
	con = YOpenConnection(
	    NULL,		/* No start argument. */
	    con_arg
	);
	if(con == NULL)
	{
	    fprintf(stderr, "Cannot connect to Y server");
	    if(con_arg == NULL)
		con_arg = getenv("RECORDER");
	    if(con_arg == NULL)
		fprintf(stderr, ".\n");
	    else
		fprintf(stderr, ": %s\n", con_arg);
	    return(2);
	}

	/* Check if we should get or set mixer values. If no mixer
	 * values were given on the command line then got_values
	 * will be false.
	 */
	if(got_values)
	{
	    /* We have values, so set them. */
	    if(YSetMixerChannel(
		con,
		mixer_code,
		(Coefficient)(value1 / 100),
		(Coefficient)(value2 / 100)
	    ))
		fprintf(
		    stderr,
		    "%s channel: Cannot set values.\n",
		    mixer_name[mixer_num]
		);
	}
	else
	{
	    /* No values specified, so print mixer channel value(s). */

	    if(mixer_code == 0)
	    {
		/* Mixer code is 0, that implies that we print all mixer
		 * channel values.
		 */

		/* Go through mixer numbers. */
		for(i = 0; i < YTotalMixers; i++)
		{
		    /* Calculate mixer code, remember i is mixer
		     * number. To get the mixer code we need to add
		     * YMixerCodeBaseOffset to the mixer number.
		     */
		    mixer_code = i + YMixerCodeBaseOffset;

		    /* Set mixer index number. */
		    mixer_num = i;

		    /* Get mixer values for this mixer code. */
		    if(YGetMixerChannel(
			con, mixer_code,
			&value1, &value2
		    ))
			printf(
			    "%s: Cannot obtain values.\n",
			    mixer_name[mixer_num]
			);
		    else
			printf(
			    "%s: %.1f%% %.1f%%\n",
			    mixer_name[mixer_num],
			    (float)(value1 * 100), (float)(value2 * 100)
			);
		}
	    }
	    else
	    {
		/* Print specific mixer channel. */

		/* Sanitize mixer index number. */
		if(mixer_num >= YTotalMixers)
		    mixer_num = YTotalMixers - 1;
		if(mixer_num < 0)
		    mixer_num = 0;

                /* Get mixer values. */
                if(YGetMixerChannel(
                    con, mixer_code,
                    &value1, &value2
                ))
                    printf(
                        "%s: Cannot obtain values.\n",
                        mixer_name[mixer_num]
                    );
                else
                    printf(
                        "%s: %.1f%% %.1f%%\n",
                        mixer_name[mixer_num],
                        (float)(value1 * 100), (float)(value2 * 100)
                    );
	    }
	}

	/* Disconnect from Y server. */
	YCloseConnection(con, False);
	con = NULL;

	return(0);
}
