#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

#include "../include/Y2/Y.h"
#include "../include/Y2/Ylib.h"

#include "../include/string.h"


static void print_help(void);


#define MIN(a,b)        ((a) < (b) ? (a) : (b))
#define MAX(a,b)        ((a) > (b) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)     (((x) < 0) ? ((x) * -1) : (x))


/*
 *	Print help.
 */
static void print_help(void)
{
	printf("\
Usage: yrecinfo [options]\n\
\n\
    [options] can be any of the following:\n\
\n\
        --modes                      Print list of Audio modes.\n\
        --recorder <address:port>    Specify which Y server to connect to.\n\
\n\
    Return values:\n\
\n\
        0       Success.\n\
        1       General error.\n\
        2       Unable to connect to the Y server.\n\
        3       Systems error.\n\
\n"
	);
}


int main(int argc, char *argv[])
{
	int i, total;
	int print_audio_modes = 0;
	const char *con_arg = NULL;
	unsigned long cur_time;
	YConnection *con = NULL;
	YEventAudioStats astats;
	YEventServerStats sstats;
	YAudioModeValuesStruct **audio_mode, *audio_mode_ptr;

	tzset();

	/* Parse arguments. */
	for(i = 1; i < argc; i++)
	{
	    if(argv[i] == NULL)
		continue;

	    /* Help. */
	    if(strcasepfx(argv[i], "--h") ||
               strcasepfx(argv[i], "-h") ||
               !strcmp(argv[i], "?")
	    )
	    {
		print_help();
		return(0);
	    }
	    /* Print Audio modes list. */
	    else if(strcasepfx(argv[i], "--m") ||
                    strcasepfx(argv[i], "-m")
            )
	    {
		print_audio_modes = 1;
	    }
	    /* Connect address. */
	    else if(strcasepfx(argv[i], "--rec") ||
                    strcasepfx(argv[i], "-rec")
	    )
	    {
		i++;
		if(i < argc)
		{
		    con_arg = argv[i];
		}
		else
		{
		    fprintf(stderr,
			"%s: Requires argument.\n",
			argv[i - 1]
		    );
		}
	    }
	}


        /* Connect to Y server. */
        con = YOpenConnection(
            NULL,               /* No start argument. */ 
            con_arg
        );
        if(con == NULL)
        {
            fprintf(stderr, "Unable to connect to the Y server");
            if(con_arg == NULL)
                con_arg = getenv("RECORDER");
            if(con_arg == NULL)
                fprintf(stderr, ".\n");
            else
                fprintf(stderr, ": %s\n", con_arg);
            return(2);
        }


	/* Get current time. */
	cur_time = time(NULL);


	/* Begin requesting and printing stats. */

	/* Print recorder address. */
	if(con_arg == NULL)
	    con_arg = getenv("RECORDER");
        printf("recorder: %s\n",
            (con_arg == NULL) ? "127.0.0.1:9433" : con_arg
        );

	/* Server stats. */
	if(!YGetServerStats(con, &sstats))
	{
	    unsigned long uptime = cur_time - sstats.start_time;

            printf("vendor name: \"%s\"\n",
                sstats.vendor_name
            );

            printf("server name: \"%s\"\n",
                sstats.server_name
            );

            printf("sound name: \"%s\"\n",
                sstats.sound_name
            );

            printf("version: %i.%i\n",
                sstats.protocol_version_major,
		sstats.protocol_version_minor
            );

            printf("cycle load: %.1f%%\n",
                (float)(sstats.cycle_load * 100)
            );

            printf("uptime: %.2i:%.2i:%.2i\n",
                    (int)(uptime / 3600),
                    (int)((uptime / 60) % 60),
                    (int)(uptime % 60)
	    );

            printf("maximum protocol statement: %i bytes\n",
                YNetRecvBufLen
            );
	}

	/* Audio mode values. */
	if(!YGetAudioStats(con, &astats))
	{
	    printf("cycle set: %i\n",
		astats.cycle_set
	    );

            printf("cycle: %ld microseconds\n",
		astats.cycle_us
	    );
            printf("compensated cycle: %ld microseconds\n",
                astats.compensated_cycle_us
            );

            printf("write ahead: %ld microseconds\n",
                astats.write_ahead_us
            );
            printf("cumulative latency: %ld microseconds\n",
                astats.cumulative_latency_us
            );

            printf("sample rate: %i Hz\n",
                astats.sample_rate
            );
            printf("channels: %i (%s)\n",
                astats.channels,
		(astats.channels == 4) ? "quad" :
		    ((astats.channels == 2) ? "stereo" : "mono")
            );
            printf("sample size: %i bits\n",
                astats.sample_size
            );
            printf("bytes per second: %i bytes\n",
                astats.bytes_per_sec
            );

            printf("allow sound buffer fragments: %s\n",  
                ((astats.allow_fragments) ? "yes" : "no")
            );
            printf("sound buffer fragments: %i\n",
		astats.num_fragments
	    );
	    printf("sound buffer size: %i bytes\n",
		astats.fragment_size
	    );

            printf("flip stereo: %s\n",
                ((astats.flip_stereo) ? "yes" : "no")
            );

            printf("direction: %i\n",
                astats.direction
            );
	}

	/* Print audio modes. */
	if(print_audio_modes)
	{
	    printf("\n");
	    audio_mode = YGetAudioModes(con, &total);
	    for(i = 0; i < total; i++)
	    {
		audio_mode_ptr = audio_mode[i];
		if(audio_mode_ptr == NULL)
		    continue;

		printf(
		    "audio: %s\n",
		    audio_mode_ptr->name
		);
		printf("  sample rate: %i Hz\n",
		    audio_mode_ptr->sample_rate
		);
		printf("  channels: %i\n",
		    audio_mode_ptr->channels
		);
		printf("  sample size: %i bits\n",
		    audio_mode_ptr->sample_size
		);
		if(audio_mode_ptr->allow_fragmenting)
		{
		    printf("  sound buffer size: %i bytes\n",
			audio_mode_ptr->fragment_size_bytes
		    );
		    printf("  sound buffer fragments: %i\n",
			audio_mode_ptr->num_fragments
		    );
		}
		printf("\n");
	    }
	    YFreeAudioModesList(audio_mode, total);
	}

        /* Disconnect from Y server. */
        YCloseConnection(con, False);
        con = NULL;

	return(0);
}
