/*
  SOY 1.2 (2005-Apr-11) Sparse Operations with Yorick
  Author: Ralf Flicker (rflicker@mac.com)
  Web site: http://homepage.mac.com/rflicker/soy.htm
  
  This work free software; you may redistribute and modify
  it under the terms of the GNU General Public License.

  Revision history:

  2004/11/14: SOY v1.0
  - Wrappers translated from IDL to Yorick (v1.5.15)

  2004/11/18: SOY v1.1
  - Adapted as a plugin for Yorick v1.6.01
  - Memory management from Yorick scripting level

  2005/04/11: SOY v1.2
  - Updated for Yorick v1.6.02
  - Ported for 64-bit OS compatibility
*/


/* MAKE-INSTRUCTIONS
SRCS = soy.c
LIB = soy
*/

//==================================================================

if (!is_void(plug_in)) plug_in, "soy";
write,"SOY v1.2.01 (COW-development version)";

struct rco
{
  int r;
  int c;
  int n;
  pointer ix;
  pointer jx;
  pointer xn;
  float t;
}

struct rco_d
{
  int r;
  int c;
  int n;
  pointer ix;
  pointer jx;
  pointer xn;
  double t;
}

struct ruo
{
  int r;
  int n;
  pointer ix;
  pointer jx;
  pointer xn;
  pointer xd;
  float t;
}

struct ruo_d
{
  int r;
  int n;
  pointer ix;
  pointer jx;
  pointer xn;
  pointer xd;
  double t;
}

//==================================================================
func rcoinf(a)
  /* DOCUMENT func rcoinf(a)
     Inflates RCO compressed matrix to full form.
  */
{
  if (typeof(*a.xn) == "float") x = array(float,a.c,a.r);
  else if (typeof(*a.xn) == "double") x = array(double,a.c,a.r);
  else error,"Unsupported Data Type";
  for (i=1; i<=a.r; i++) {
    if ((*a.ix)(i+1) > (*a.ix)(i)) {
      for (j=(*a.ix)(i)+1; j<=(*a.ix)(i+1); j++) {
        x((*a.jx)(j)+1,i)=(*a.xn)(j);}}}
  return x;
}

//==================================================================
func ruoinf(a)
  /* DOCUMENT func ruoinf(a)
     Inflates RUO compressed matrix to full form.
  */
{
  if (typeof(*a.xn) == "float") x = array(float,a.r,a.r);
  else if (typeof(*a.xn) == "double") x = array(double,a.r,a.r);
  else error,"Unsupported Data Type";  
  for (i=1; i<=a.r; i++) x(i,i) = (*a.xd)(i);
  for (i=1; i<a.r; i++) {
    if ((*a.ix)(i+1) > (*a.ix)(i)) {
      //k++;
      //write,format="%i  %i  %i  %i\n",k,(*a.ix)(i),(*a.ix)(i+1),(*a.ix)(i+1) > (*a.ix)(i);
      for (j=(*a.ix)(i)+1; j<=(*a.ix)(i+1); j++) {
        x(i,(*a.jx)(j)+1)=x((*a.jx)(j)+1,i)=(*a.xn)(j);}}}
  return x;
}

//==================================================================
func rcox(a,c)
  /* DOCUMENT func rcox(a,c)
     Multiplies RCO compressed matrix by a scalar.
  */
{
  if (typeof(*a.xn) != typeof(c)) error,"Mixed Data Types";
  (*a.xn)(1:a.n) = c*(*a.xn)(1:a.n);
}

//==================================================================
func ruox(a,c)
  /* DOCUMENT func ruox(a,c)
     Multiplies RUO matrix "a" by a scalar "c".
  */
{
  if (typeof(*a.xn) != typeof(c)) error,"Mixed Data Types";
  (*a.xn)(1:a.n) = c*(*a.xn)(1:a.n);
  (*a.xd)(1:a.r) = c*(*a.xd)(1:a.r);
}

//==================================================================
func rcos(a)
  /* DOCUMENT func rcos(a)
     Computes the sparseness (or rather, the "fill") of a matrix on RCO format.
  */
{
  if (a.r != 0) {xfill = float(a.n)/float(a.c)/float(a.r);}
  else {xfill = 0;}
  return xfill;
}

//==================================================================
func ruos(a)
  /* DOCUMENT func ruos(a)
     Computes the sparseness (or rather, the "fill") of a matrix on RUO format.
  */
{
  if (a.r != 0) {xfill = (float(a.n)*2.0f+float(a.r))/float(a.r)^2.;}
  else {xfill = 0;}
  return xfill;
}

//==================================================================
func spinfo(A)
  /* DOCUMENT func spinfo(a)
     Prints information about a sparse matrix in RCO or RUO format.
  */
{
  nsx = dimsof(A)(0);
  if (nsx == 0 && (typeof(A(1)) == "struct_instance")) nsx = 1;
  for (nx=1; nx<=nsx; nx++) {
    a = A(nx);
    if (nsx > 1) write,format="Block # %i\n",nx;
    if (typeof(a) != "struct_instance") {error,"Argument not a RCO or RUO structure!";}
    members= strtok(strtok(print(structof(a))(2:-1))(2,)," (;")(1,);
    if (numberof(members) != 7) {error,"Argument not a RCO or RUO structure!";}
    if (members(2) == "c") {
      sptype = "RCO";
      if (a.n != 0) {xfill = rcos(a)*100.;}
      else {error,"empty structure";}
    } else {
      sptype = "RUO";
      if (a.n != 0) {xfill = ruos(a)*100.;}
      else {error,"empty structure";}
    }
    ur = numberof(*a.ix);
    un = numberof(*a.xn);
    sfilln = float(a.n)/float(un)*100.;
    sfillr = float(a.r)/float(ur)*100.;
    ptype = typeof((*a.xn)(1));
    write,sptype+", "+ptype+"  stored      max.   usage";
    write,format="no. rows : %8i %9i   %4.2f%%\n",a.r,ur,sfillr;
    write,format="elements : %8i %9i   %4.2f%%\n",a.n,un,sfilln;
    if (sptype == "RCO") write,format="   (cols : %8i)\n",a.c;
    write,format="matrix fill : %4.2f%%\n",xfill;
  }
}
//==================================================================
func sprco(x,t=,ur=,un=)
  /* DOCUMENT func sprco(x,t=,ur=,un=)
     Compress a 2D matrix on sparse RCO format.
     SEE ALSO: sprco_float (soy.c)
  */
{
  argc = 7;
  if (ur == []) ur = MR;
  if (un == []) un = MN;
  c = int((dimsof(x))(2));
  r = int((dimsof(x))(3));
  ix = array(int,ur);
  jx = array(int,un);
  if (typeof(x) == "float") {
    if (t != []) {t = float(t);}
    else {t = float(0);}
    xn = array(float,un);
    a = [&x,&r,&c,&t,&ix,&jx,&xn];
    n = sprco_float(argc,&a);
    s = rco();
  } else if (typeof(x) == "double") {
    if (t != []) {t = double(t);}
    else {t = double(0);}
    xn = array(double,un);
    a = [&x,&r,&c,&t,&ix,&jx,&xn];
    n = sprco_double(argc,&a);
    s = rco_d();
  } else error,"Unsupported Data Type";
  s.r = r;
  s.c = c;
  s.n = n;
  s.t = t;
  s.ix = &ix;
  s.jx = &jx;
  s.xn = &xn;
  return s;
}

extern sprco_float;
/* PROTOTYPE
   int sprco_float(int argc, pointer a)
*/
extern sprco_double;
/* PROTOTYPE
   int sprco_double(int argc, pointer a)
*/

//==================================================================
func spruo(x,t=,ur=,un=)
  /* DOCUMENT func spruo(x,t=,ur=,un=)
     Compress a 2D matrix on sparse RUO format.
     SEE ALSO: spruo_float (soy.c)
  */
{
  argc = 7;
  if (ur == []) ur = MR;
  if (un == []) un = MN;
  r = int((dimsof(x))(3));
  ix = array(int,ur);
  jx = array(int,un);
  if (typeof(x) == "float") {
    if (t != []) {t = float(t);}
    else {t = float(0);}
    xn = array(float,un);
    xd = array(float,ur);
    a = [&x,&r,&t,&ix,&jx,&xn,&xd];
    n = spruo_float(argc,&a);
    s = ruo();
  } else if (typeof(x) == "double") {
    if (t != []) {t = double(t);}
    else {t = double(0);}
    xn = array(double,un);
    xd = array(double,ur);
    a = [&x,&r,&t,&ix,&jx,&xn,&xd];
    n = spruo_double(argc,&a);
    s = ruo_d();
  } else error,"Unsupported Data Type";
  s.r = r;
  s.n = n;
  s.t = t;
  s.ix = &ix;
  s.jx = &jx;
  s.xn = &xn;
  s.xd = &xd;
  return s;
}

extern spruo_float;
/* PROTOTYPE
   int spruo_float(int argc, pointer a)
*/
extern spruo_double;
/* PROTOTYPE
   int spruo_double(int argc, pointer a)
*/

//==================================================================
func rcoxv(a,v)
  /* DOCUMENT func rcoxv(a,v)
     Sparse matrix-vector multiplication of RCO and real vector.
     SEE ALSO: rcoxv_float (soy.c)
  */
{
  argc = 6;
  if (typeof(v) == "float" && typeof(*a.xn) == "float") {
    u = array(float,a.r);
    s = [&v,&u,&a.r,a.ix,a.jx,a.xn];
    tmp = rcoxv_float(argc,&s);
  } else if (typeof(v) == "double" && typeof(*a.xn) == "double") {
    u = array(double,a.r);
    s = [&v,&u,&a.r,a.ix,a.jx,a.xn];
    tmp = rcoxv_double(argc,&s);
  }
  return u;
}

extern rcoxv_float;
/* PROTOTYPE
   int rcoxv_float(int argc, pointer s)
*/
extern rcoxv_double;
/* PROTOTYPE
   int rcoxv_double(int argc, pointer s)
*/

//==================================================================
func ruoxv(a,v)
  /* DOCUMENT func ruoxv(a,v)
     Sparse matrix-vector multiplication of RUO and real vector.
     SEE ALSO: ruoxv_float (soy.c)
  */
{
  argc = 8;
  if (typeof(v) == "float" && typeof(*a.xn) == "float") {
    u = array(float,a.r);
    w = array(float,a.r);
    s = [&v,&u,&w,&a.r,a.ix,a.jx,a.xn,a.xd];
    tmp = ruoxv_float(argc,&s);
  } else if (typeof(v) == "double" && typeof(*a.xn) == "double") {
    u = array(double,a.r);
    w = array(double,a.r);
    s = [&v,&u,&w,&a.r,a.ix,a.jx,a.xn,a.xd];
    tmp = ruoxv_double(argc,&s);
  }
  return u;
}

extern ruoxv_float;
/* PROTOTYPE
   int ruoxv_float(int argc, pointer s)
*/
extern ruoxv_double;
/* PROTOTYPE
   int ruoxv_double(int argc, pointer s)
*/
//==================================================================
func rcoadd(a,b,ur=,un=)
  /* DOCUMENT func rcoadd(a,b,ur=,un=)
     Sparse addition of two RCO matrices.
     SEE ALSO: rcoadd_float (soy.c)
  */
{
  if (typeof(*a.xn) != typeof(*b.xn)) error,"Mixed Data Types";
  if (a.r != b.r || a.c != b.c) error,"Matrices have incompatible dimensions!";
  argc = 17;
  if (ur == []) {ur = 2*(a.r+b.r);}
  if (un == []) {un = 2*(a.n+b.n);}
  ss = array(int,ur);
  if (typeof(*a.xn) == "float") {
    c = rco();
    c.r = a.r;
    c.c = a.c;
    c.t = a.t;
    c.ix = &(array(int,ur));
    c.jx = &(array(int,un));
    c.xn = &(array(float,un));
    t = array(float,ur);
    s = [&a.r,&a.c,a.ix,a.jx,a.xn,&b.r,&b.c,b.ix,b.jx,b.xn,&c.r,&c.c,c.ix,c.jx,c.xn,&t,&ss];
    n = rcoadd_float(argc, &s);
  }
  else if (typeof(*a.xn) == "double") {
    c = rco_d();
    c.r = a.r;
    c.c = a.c;
    c.t = a.t;
    c.ix = &(array(int,ur));
    c.jx = &(array(int,un));
    c.xn = &(array(double,un));
    t = array(double,ur);
    s = [&a.r,&a.c,a.ix,a.jx,a.xn,&b.r,&b.c,b.ix,b.jx,b.xn,&c.r,&c.c,c.ix,c.jx,c.xn,&t,&ss];
    n = rcoadd_double(argc,&s);
  }
  else error,"Unsupported Data Type";
  c.n = n;
  return c;
}

extern rcoadd_float;
/* PROTOTYPE
   int rcoadd_float(int argc, pointer s)
*/
extern rcoadd_double;
/* PROTOTYPE
   int rcoadd_double(int argc, pointer s)
*/
//==================================================================
func ruoadd(a,b,ur=,un=)
  /* DOCUMENT func ruoadd(a,b,ur=,un=)
     Sparse addition of two RUO matrices.
     SEE ALSO: ruoadd_float (soy.c)
  */
{
  if (typeof(*a.xn) != typeof(*b.xn)) error,"Mixed Data Types";
  if (a.r != b.r) error,"Matrices have incompatible dimensions!";
  argc = 17;
  if (ur == []) {ur = a.r+b.r;}
  if (un == []) {un = 2*(a.n+b.n);}
  ss = array(int,ur);
  if (typeof(*a.xn) == "float") {
    c = ruo();
    c.r = a.r;
    c.t = a.t;
    c.ix = &(array(int,ur));
    c.jx = &(array(int,un));
    c.xn = &(array(float,un));
    c.xd = &(array(float,ur));
    tt = array(float,ur);
    s = [&a.r,a.xd,a.ix,a.jx,a.xn,&b.r,b.xd,b.ix,b.jx,b.xn,&c.r,c.xd,c.ix,c.jx,c.xn,&tt,&ss];
    n = ruoadd_float(argc, &s);
  }
  else if (typeof(*a.xn) == "double") {
    c = ruo_d();
    c.r = a.r;
    c.t = a.t;
    c.ix = &(array(int,ur));
    c.jx = &(array(int,un));
    c.xn = &(array(double,un));
    c.xd = &(array(double,ur));
    tt = array(double,ur);
    s = [&a.r,a.xd,a.ix,a.jx,a.xn,&b.r,b.xd,b.ix,b.jx,b.xn,&c.r,c.xd,c.ix,c.jx,c.xn,&tt,&ss];
    n = ruoadd_double(argc,&s);
  }
  else error,"Unsupported Data Type";
  c.n = n;
  return c;
}

extern ruoadd_float;
/* PROTOTYPE
   int ruoadd_float(int argc, pointer s)
*/
extern ruoadd_double;
/* PROTOTYPE
   int ruoadd_double(int argc, pointer s)
*/
//==================================================================
func rcoata(a,ur=,un=,t=)
  /* DOCUMENT func rcoata(a,ur=,un=,t=)
     Sparse mutiplication of an RCO matrix with its transpose from
     the left, i.e. transpose(a)##a
     SEE ALSO: rcoata_float (soy.c)
  */
{
  argc = 9;
  if (ur == [] && MR != []) ur = MR;
  if (ur == [] && MR == []) {
    ur = int(a.r+2);
  }
  if (un == [] && MN != []) un = MN;
  if (un == [] && MN == []) {
    un = int(a.n*5);
  }
  if (typeof(*a.xn) == "float") {
    b = ruo();
    b.r = a.r;
    if (is_set(t)) b.t = t;
    else b.t = float(a.t^2.);
    b.ix = &(array(int,ur));
    b.jx = &(array(int,un));
    b.xn = &(array(float,un));
    b.xd = &(array(float,ur));
    s = [&a.r,a.ix,a.jx,a.xn,&b.t,b.ix,b.jx,b.xn,b.xd];
    n = rcoata_float(argc,&s);
  }
  else if (typeof(*a.xn) == "double") {
    b = ruo_d();
    b.r = a.r;
    if (is_set(t)) b.t = t;
    else b.t = double(a.t^2.);
    b.ix = &(array(int,ur));
    b.jx = &(array(int,un));
    b.xn = &(array(double,un));
    b.xd = &(array(double,ur));
    s = [&a.r,a.ix,a.jx,a.xn,&b.t,b.ix,b.jx,b.xn,b.xd];
    n = rcoata_double(argc, &s);
  }
  else error,"Unsupported Data Type";
  b.n = n;
  return b;
}

extern rcoata_float;
/* PROTOTYPE
   int rcoata_float(int argc, pointer s)
*/
extern rcoata_double;
/* PROTOTYPE
   int rcoata_double(int argc, pointer s)
*/
//==================================================================
func rcoatb(a,b,ur=,un=,t=,u=)
  /* DOCUMENT func rcoatb(a,b,ur=,un=,t=,u=)
     Sparse mutiplication of two RCO matrices, a'.b. Setting u=1
     ("upper") computes only the upper triangular and diagonal
     elements of the matrix product, and returns an RUO matrix.
     Use this when computing e.g. the final step of a 3-matrix
     product of the type M = A'.W.A, where A is RCO, W is RUO
     and you know that the final result must be RUO.
     SEE ALSO: rcoatb_float (soy.c)
  */
{
  if (ur == []) ur = MR;
  if (un == []) un = MN;
  if (typeof(*a.xn) == "float" && typeof(*b.xn) == "float") {
    if (u == 1) {
      argc = 13;
      c = ruo();
      c.r = a.r;
      if (is_set(t)) c.t = t;
      else c.t = min([a.t,b.t]);
      c.ix = &(array(int,ur));
      c.jx = &(array(int,un));
      c.xn = &(array(float,un));
      c.xd = &(array(float,ur));
      s = [&a.r,a.ix,a.jx,a.xn,&b.r,b.ix,b.jx,b.xn,&c.t,c.ix,c.jx,c.xn,c.xd];
      n = rcoatb2_float(argc,&s);
    }
    else {
      argc = 12;
      c = rco();
      c.r = a.r;
      c.c = b.r;
      if (is_set(t)) c.t = t;
      else c.t = min([a.t,b.t]);
      c.ix = &(array(int,ur));
      c.jx = &(array(int,un));
      c.xn = &(array(float,un));
      s = [&a.r,a.ix,a.jx,a.xn,&b.r,b.ix,b.jx,b.xn,&c.t,c.ix,c.jx,c.xn];
      n = rcoatb_float(argc,&s);
    }
  }
  else if (typeof(*a.xn) == "double" && typeof(*b.xn) == "double") {
    argc = 12;
    c = rco_d();
    c.r = a.r;
    c.c = b.r;
    if (is_set(t)) c.t = t;
    else c.t = min([a.t,b.t]);
    c.ix = &(array(int,ur));
    c.jx = &(array(int,un));
    c.xn = &(array(double,un));
    s = [&a.r,a.ix,a.jx,a.xn,&b.r,b.ix,b.jx,b.xn,&c.t,c.ix,c.jx,c.xn];
    n = rcoatb_double(argc,&s);
  }
  else error,"Unsupported or mixed data type(s)";
  c.n = n;
  return c;
}

extern rcoatb_float;
/* PROTOTYPE
   int rcoatb_float(int argc, pointer s)
*/
extern rcoatb2_float;
/* PROTOTYPE
   int rcoatb2_float(int argc, pointer s)
*/
extern rcoatb_double;
/* PROTOTYPE
   int rcoatb_double(int argc, pointer s)
*/
//==================================================================
func rcotr(a)
  /* DOCUMENT func rcotr(arg)
     Transposes an RCO matrix. Uses one builtin Yorick function
     (sort) to make the rest a little bit easier.
  */
{
  argc = 8;
  ur = dimsof(*a.ix);
  un = dimsof(*a.xn);
  if (typeof(*a.xn) == "float") {
    at = rco();
    at.xn = &(array(float,un));
  }
  else if (typeof(*a.xn) == "double") {
    at = rco_d();
    at.xn = &(array(double,un));
  }
  else error,"Unsupported data type";
  at.n = a.n;
  at.r = a.c;
  at.c = a.r;
  at.t = a.t;
  at.ix = &(array(int,a.c+2));
  at.jx = &(array(int,un));
  sjx = int(sort((*a.jx)(1:a.n)));
  hjx = (*a.jx)(sjx);
  ax = array(int,a.c);
  acx = array(int,a.c+2);
  rind = array(int,at.n);
  s = [&ax,&acx,&hjx,&rind,&a.r,&a.c,&a.n,a.ix];
  tmp = rcotr_fix(argc, &s);
  (*at.ix)(1:at.r+1) = acx(1:at.r+1);
  (*at.jx)(1:at.n) = rind(sjx);
  (*at.xn)(1:at.n) = (*a.xn)(sjx);
  return at;
}

extern rcotr_fix;
/* PROTOTYPE
   int rcotr_fix(int argc, pointer s)
*/
//==================================================================
func ruopcg(a,b,x0,&nit,tol=,itmax=,sgs=)
  /* DOCUMENT func ruopcg(a,b,x0,&nit,tol=,itmax=,sgs=)
     Preconditioned conjugate gradient solver for a symmetric positive
     definite sparse linear system, with Jacobi preconditioner. This
     algorithm is implemented straight out of Numerical Recipes, with
     the matrix-vector multiplications carried out sparsely by the
     ruoxv(a,v) function.
     Optionally one may invoke symmetric Gauss-Seidel iterations upon
     the Jacobi preconditioning, by setting the keyword sgs=#iters.
     (at least 1). SGS requires the U, D and L to be externally defined.
     SEE ALSO: ruoxv, aotest
  */
{
  prec = typeof(*a.xn);
  if (sum(prec == [typeof(b),typeof(x0)]) != 2) {
    error,"Inconsistent Data Types a/b/x0";}
  if (itmax == []) itmax = a.r;
  if (tol == []) tol = 1.e-4;
  x = array(double,a.r);
  bnrm = sum(b^2.);
  u = ruoxv(a,x0(1:a.r));
  r = b-u;
  //tmp = float(sqrt((*a.xd)(1:a.r)));
  tmp = (*a.xd)(1:a.r);
  //if (sgs == []) {z = r/(*a.xd)(1:a.r);}  // Jacobi preconditioner
  if (sgs == []) {z = r/tmp;}  // Jacobi preconditioner
  //  else {z = ruosgs_Y(x0,r,sgs,U,D,L);}  // Gauss-Seidel iterations
  else {z = ruosgs(U,L,D,sgs,x0*0.f,r);}  // Gauss-Seidel iterations
  k = 0;
  err = 1.;
  while ((k <= itmax) && (err > tol)) {
    k += 1;
    bknum = sum(z*r);
    if (k > 1) {
      bk = bknum/bkden;
      p = bk*p+z;}
    else p = z;
    bkden = bknum;
    if (prec == "float") u = ruoxv(a,float(p(1:a.r)));
    else u = ruoxv(a,double(p(1:a.r)));
    z = u;
    akden = sum(z*p);
    ak = bknum/akden;
    x += ak*p;
    r -= ak*z;
    //  if (sgs == []) {z = r/(*a.xd)(1:a.r);}  // Jacobi preconditioner
    if (sgs == []) {z = r/tmp;}  // Jacobi preconditioner
    //    else {z = ruosgs(x0,r,sgs,U,D,L);}  // Gauss-Seidel iterations
    else {z = ruosgs(U,L,D,sgs,x0*0.f,r);}  // Gauss-Seidel iterations
    err = sum(r^2.)/bnrm;
  }
  nit = k;
  if (prec == "float") return float(x);
  else return double(x);
}

//==================================================================
func rcobuild(&a,v,t,ur=,un=)
  /* DOCUMENT func rcobuild(a,v,t,ur=,un=)
     Appends a row-vector v to an RCO matrix a, at threshold t.
  */
{
  if (ur == []) ur = MR;
  if (un == []) un = MN;
  if (a == [] && typeof(v) == "float") a = rco();
  if (a == [] && typeof(v) == "double") a = rco_d();
  if (*a.xn == [] && typeof(v) == "float") a.xn = &(array(float,un));
  if (*a.xn == [] && typeof(v) == "double") a.xn = &(array(float,un));
  if (*a.ix == []) a.ix = &(array(int,ur));
  if (*a.jx == []) a.jx = &(array(int,un));
  if (a.c == 0) {
    a.c = int(numberof(v));
    a.r = int(0);
    a.t = t;
    a.n = int(0);
  }
  a.r += int(1);
  tmp0 = (abs(v) > t);
  if (anyof(tmp0)) {
    tmp = where(tmp0);
    n = numberof(tmp);
    (*a.jx)(1+a.n:a.n+n) = int(tmp-1);
    (*a.xn)(1+a.n:a.n+n) = v(tmp);
    (*a.ix)(a.r+1) = int((*a.ix)(a.r)+n);
    a.n += int(n);
  } else {
    (*a.ix)(a.r+1) = (*a.ix)(a.r);
  }
}

//==================================================================
func Laplace_FDA(nact,aind,ur=,un=)
  /* DOCUMENT Laplace_FDA(CS,nact,aind)
     Builds the discrete 2D Laplacian operator directly on RCO
     format. Requires a bit of pre-processing (computing "aind"
     - see aotest.i for how to do that).
  */
{
  if (ur == []) ur = nact+2;
  if (un == []) un = nact*5+1;
  w = array(int,5);
  v = [&float([-1.,0.25,0.25,0.25,0.25]),\
       &float([-0.75,0.25,0.25,0.25]),\
       &float([-2./3.,1./3.,1./3.])];
  c = rco();
  c.r = int(nact);
  c.c = int(nact);
  c.ix = &(array(int,ur));
  c.jx = &(array(int,un));
  c.xn = &(array(float,un));
  c.t = 0.;
  cp = 0;
  for (i=1; i<=nact; i++) {
    w *=0;
    w(1) = i; // center grid point
    actr = (aind(i,1)+1 == aind(,1)) & (aind(i,2) == aind(,2));
    actu = (aind(i,1) == aind(,1)) & (aind(i,2)+1 == aind(,2));
    actl = (aind(i,1)-1 == aind(,1)) & (aind(i,2) == aind(,2));
    actd = (aind(i,1) == aind(,1)) & (aind(i,2)-1 == aind(,2));
    if (anyof(actr)) w(2) = where(actr); // neighbor right
    if (anyof(actu)) w(3) = where(actu); // neighbor up
    if (anyof(actl)) w(4) = where(actl); // neighbor left
    if (anyof(actd)) w(5) = where(actd); // neighbor down
    z = where(w > 0);
    nz = numberof(z);
    (*c.xn)(cp+1:cp+nz) = *v(6-nz);
    (*c.jx)(cp+1:cp+nz) = int(w(z)-1);
    cp += nz;
    (*c.ix)(i+1) = int(cp);
  }
  c.n = int(cp);
  return c;
}

//==================================================================
func Laplace_FDA2(ap,ur=,un=)
  /* DOCUMENT Laplace_FDA2(ap,ur=,un=)
     Same as Laplace_FDA, but MUCH faster. In fact, so fast that I
     don't need to bother with going to a C implementation.
     Input is a [0,1]-valued aperture function.
  */
{
  dim = dimsof(ap)(0);
  nx = sum(ap);
  ind = where(ap);
  x = span(1,dim,dim)(,-:1:dim);
  y = transpose(x);
  aind = array(long,nx,2);
  aind(,1) = x(ind);
  aind(,2) = y(ind);
  inds = int(ap*0);
  inds(ind) = int(span(1,nx,nx));

  //Symbolic build
  validx = array(int,nx*4);
  for (i=1; i<=nx; i++) {
    xc = int(x(ind(i)));
    yc = int(y(ind(i)));
    if (xc < dim) validx((i-1)*4+1) = inds(xc+1,yc)>0; //right
    if (yc < dim) validx((i-1)*4+2) = inds(xc,yc+1)>0; //up
    if (xc > 1) validx((i-1)*4+3) = inds(xc-1,yc)>0;   //left
    if (yc > 1) validx((i-1)*4+4) = inds(xc,yc-1)>0;   //down
  }

  if ((ur == []) || (ur < nx)) ur = nx+1;
  if ((un == []) || (un < nx*5)) un = nx*5+1;
  w = array(int,5);
  v = [&float([-1.,0.25,0.25,0.25,0.25]),\
       &float([-0.75,0.25,0.25,0.25]),\
       &float([-2./3.,1./3.,1./3.])];
  c = rco();
  c.r = int(nx);
  c.c = int(nx);
  c.ix = &(array(int,ur));
  c.jx = &(array(int,un));
  c.xn = &(array(float,un));
  c.t = float(0.);
  cp = int(0);
  
  //Numeric build
  for (i=1; i<=nx; i++) {
    w *= int(0);
    cn = int(1);
    w(1) = i;
    if (validx((i-1)*4+1)) {
      cn++;
      w(cn) = w(1)+1;
    }
    if (validx((i-1)*4+2)) {
      cn++;
      w(cn) = inds(aind(i,1),aind(i,2)+1);
    }
    if (validx((i-1)*4+3)) {
      cn++;
      w(cn) = w(1)-1;
    }
    if (validx((i-1)*4+4)) {
      cn++;
      w(cn) = inds(aind(i,1),aind(i,2)-1);
    }
    (*c.xn)(cp+1:cp+cn) = *v(6-cn);
    (*c.jx)(cp+1:cp+cn) = int(w(1:cn)-1);
    cp += cn;
    (*c.ix)(i+1) = int(cp);
  }
  c.n = int(cp);
  return c;
}

//==================================================================
func Laplace_FDA3(ap,ur=,un=)
  /* DOCUMENT Laplace_FDA3(ap,ur=,un=)
     Same as Laplace_FDA2, but with C implementation (faster).
     Input is a [0,1]-valued aperture function.
  */
{
  /*
    dim = 64;
    ap = dist(dim)<dim/2;
    CS = Laplace_FDA3(ap);
  */
  dim = int(dimsof(ap)(0));
  nx = int(sum(ap));
  ind = int(where(ap));
  x = int(span(1,dim,dim)(,-:1:dim));
  y = transpose(x);
  aind = array(int,nx*2);
  aind(1:nx) = x(ind);
  aind(nx+1:) = y(ind);
  inds = int(ap*0);
  inds(ind) = int(span(1,nx,nx));
  validx = array(int,nx*4);
  
  //Symbolic build
  s = [&x,&y,&ind,&inds,&validx,&nx,&dim];
  argc = 7; 
  tmp = laplace1_float(argc,&s);
  
  if ((ur == []) || (ur < nx)) ur = nx+1;
  if ((un == []) || (un < nx*5)) un = nx*5+1;
  c = rco();
  c.r = int(nx);
  c.c = int(nx);
  c.ix = &(array(int,ur));
  c.jx = &(array(int,un));
  c.xn = &(array(float,un));
  c.t = float(0.);

  //Numerical build
  argc = 11;
  s = [c.ix,c.jx,c.xn,&validx,&nx,&dim,&inds,&aind,\
       &float([-1.,0.25,0.25,0.25,0.25]),\
       &float([-0.75,0.25,0.25,0.25]),\
       &float([-2./3.,1./3.,1./3.])]; 
  c.n = laplace2_float(argc,&s);
  return c;
}

extern laplace1_float;
/* PROTOTYPE
   int laplace1_float(int argc, pointer s)
*/

extern laplace2_float;
/* PROTOTYPE
   int laplace2_float(int argc, pointer s)
*/

//==================================================================
func save_rco(A,fn,bin=)
  /* DOCUMENT save_rco(a,fn,bin=)
     Saves an RCO structure a to the binary file "fnX.bin" by converting
     all of its elements to float (double) and putting them into a
     single vector.
  */
{
  nsx = dimsof(A)(0);
  if (nsx == 0 && (typeof(A(1)) == "struct_instance")) {nsx = nsx0 = 1;}
  else {nsx0 = 0;}
  for (nx=1; nx<=nsx; nx++) {
    a = A(nx); 
    r = a.r;
    c = a.c;
    n = a.n;
    t = a.t;
    if (typeof(*a.xn) == "float") {
      v = array(float,n*2+r+6);
      v(1:4) = float([n,r,c,t]);
      v(5:n+4) = (*a.xn)(1:n);
      v(n+5:2*n+4) = float((*a.jx)(1:n));
      v(2*n+5:2*n+r+5) = float((*a.ix)(1:r+1));
    }
    else if (typeof(*a.xn) == "double") {
      v = array(double,n*2+r+6);
      v(1:4) = double([n,r,c,t]);
      v(5:n+4) = (*a.xn)(1:n);
      v(n+5:2*n+4) = double((*a.jx)(1:n));
      v(2*n+5:2*n+r+5) = double((*a.ix)(1:r+1));
    }
    else error,"Unsupported data type";
    if (is_set(bin)) {
      if (nsx0 == 0) {fname = fn+swrite(format="%i",int(nx))+".bin";}
      else {fname = fn+".bin";}
      save,createb(fname),v;
    } else {
      if (nsx0 == 0) {fname = fn+swrite(format="%i",int(nx))+".fits";}
      else {fname = fn+".fits";}
      img_write,v,fname,type="fits";
    }
  }
}

//==================================================================
func save_ruo(A,fn,bin=)
  /* DOCUMENT save_ruo(a,fn,bin=)
     Saves an RUO structure a to the binary file fn by converting
     all of its elements to float (double) and putting them into a
     single vector.
  */
{
  nsx = dimsof(A)(0);
  if (nsx == 0 && (typeof(A(1)) == "struct_instance")) {nsx = nsx0 = 1;}
  else {nsx0 = 0;}
  for (nx=1; nx<=nsx; nx++) {
    a = A(nx); 
    r = a.r;
    n = a.n;
    t = a.t;
    if (typeof(*a.xn) == "float") {
      v = array(float,n*2+r*2+6);
      v(1:3) = float([n,r,t]);
      v(4:n+3) = (*a.xn)(1:n);
      v(n+4:2*n+3) = float((*a.jx)(1:n));
      v(2*n+4:2*n+r+3) = float((*a.ix)(1:r));
      v(2*n+r+4:2*n+2*r+3) = (*a.xd)(1:r);
    }
    else if (typeof(*a.xn) == "double") {
      v = array(double,n*2+r*2+6);
      v(1:3) = double([n,r,t]);
      v(4:n+3) = (*a.xn)(1:n);
      v(n+4:2*n+3) = double((*a.jx)(1:n));
      v(2*n+4:2*n+r+3) = double((*a.ix)(1:r));
      v(2*n+r+4:2*n+2*r+3) = (*a.xd)(1:r);
    }
    else error,"Unsupported data type";
    if (is_set(bin)) {
      if (nsx0 == 0) {fname = fn+swrite(format="%i",int(nx))+".bin";}
      else {fname = fn+".bin";}
      save,createb(fname),v;
    } else {
      if (nsx0 == 0) {fname = fn+swrite(format="%i",int(nx))+".fits";}
      else {fname = fn+".fits";}
      img_write,v,fname,type="fits";
    }
  }
}

//==================================================================
func float_rco(&a)
  /* DOCUMENT float_rco(&a)
     Converts the double rco_d structure to float rco.
  */
{
  b = rco();
  b.r = a.r;
  b.c = a.c;
  b.n = a.n;
  b.t = float(a.t);
  b.ix = a.ix;
  b.jx = a.jx;
  b.xn = &float(*a.xn);
  a = b;
}
//==================================================================
func float_ruo(&a)
  /* DOCUMENT float_ruo(&a)
     Converts the double ruo_d structure to float ruo.
  */
{
  b = ruo();
  b.r = a.r;
  b.n = a.n;
  b.t = float(a.t);
  b.ix = a.ix;
  b.jx = a.jx;
  b.xn = &float(*a.xn);
  b.xd = &float(*a.xd);
  a = b;
}
//==================================================================
func double_rco(&a)
  /* DOCUMENT double_rco(&a)
     Converts the float rco structure to double rco_d.
  */
{
  b = rco_d();
  b.r = a.r;
  b.c = a.c;
  b.n = a.n;
  b.t = double(a.t);
  b.ix = a.ix;
  b.jx = a.jx;
  b.xn = &double(*a.xn);
  a = b;
}
//==================================================================
func double_ruo(&a)
  /* DOCUMENT double_ruo(&a)
     Converts the float ruo structure to double ruo_d.
  */
{
  b = ruo_d();
  b.r = a.r;
  b.n = a.n;
  b.t = double(a.t);
  b.ix = a.ix;
  b.jx = a.jx;
  b.xn = &double(*a.xn);
  b.xd = &double(*a.xd);
  a = b;
}
//==================================================================
func restore_rco(fn,bin=)
  /* DOCUMENT restore_rco(fn,bin=)
     Returns the RCO structure saved in the file fn by save_rco.
  */
{
  if (is_set(bin)) restore,openb(fn),v;
  else v = img_read(fn);
  if (typeof(v) == "float") {a = rco();}
  else if (typeof(v) == "double") {a = rco_d();}
  else error,"Unsupported data type";
  a.n = int(v(1));
  a.r = int(v(2));
  a.c = int(v(3));
  a.t = v(4);
  a.xn = &(v(5:a.n+4));
  a.jx = &(int(v(a.n+5:2*a.n+4)));
  a.ix = &(int(v(2*a.n+5:2*a.n+a.r+5)));
  return a;
}

//==================================================================
func restore_ruo(fn,bin=)
  /* DOCUMENT restore_ruo(fn,bin=)
     Returns the RUO structure saved in the file fn by save_rco.
  */
{
  if (is_set(bin)) restore,openb(fn),v;
  else v = img_read(fn);
  if (typeof(v) == "float") a = ruo();
  else if (typeof(v) == "double") a = ruo_d();
  else error,"Unsupported data type in";
  a.n = int(v(1));
  a.r = int(v(2));
  a.t = v(3);
  a.xn = &(v(4:a.n+3));
  a.jx = &(int(v(a.n+4:2*a.n+3)));
  a.ix = &(int(v(2*a.n+4:2*a.n+a.r+3)));
  a.xd = &(v(2*a.n+a.r+4:2*a.n+2*a.r+3));
  return a;
}

//==================================================================
func rcodr(&a,r)
  /* DOCUMENT rcodr(a,r)
     Delete a specific row from an RCO structure.
  */
{
  nel = (*a.ix)(r+1)-(*a.ix)(r);
  /*
  if (nel == 0) {

    if (r == 1) {
      (*a.ix)(2:a.r) = (*a.ix)(3:a.r+1);
    } else if (r < a.r) {
      (*a.ix)(r+1:a.r) = (*a.ix)(r+2:a.r+1);
    }

  } else {

    if (r == a.r) {
      (*a.jx)(a.n-nel+1:a.n) *= 0;
      (*a.xn)(a.n-nel+1:a.n) *= 0.0f;

    } else if (r == 1) {
      (*a.jx)(1:a.n-nel) = (*a.jx)(nel+1:a.n);
      (*a.xn)(1:a.n-nel) = (*a.xn)(nel+1:a.n);
      (*a.ix)(2:a.r) = (*a.ix)(3:a.r+1)-nel;

    } else {
      if ((*a.ix)(r) > 0) {

        (*a.jx)((*a.ix)(r):a.n-nel-1) = (*a.jx)((*a.ix)(r+1):a.n-1);
        (*a.xn)((*a.ix)(r):a.n-nel-1) = (*a.xn)((*a.ix)(r+1):a.n-1);
        (*a.ix)(r+1:a.r) = (*a.ix)(r+2:a.r+1)-nel;

      } else {
        (*a.jx)(1:a.n-nel) = (*a.jx)((*a.ix)(r+1):a.n-1);
        (*a.xn)(1:a.n-nel) = (*a.xn)((*a.ix)(r+1):a.n-1);
        (*a.ix)(r+1:a.r) = (*a.ix)(r+2:a.r+1)-nel;
      }
    }
  }
  (*a.ix)(a.r+1) = 0;
  */

  if (nel == 0) {

    if (r == 1) {
      /*
      ix1 = (*a.ix)(1);
      ix2 = (*a.ix)(3:a.r+1);
      ix = [];
      grow,ix,ix1;
      grow,ix,ix2;
      */
      (*a.ix)(2:a.r) = (*a.ix)(3:a.r+1);
    } else if (r < a.r) {
      /*
      ix1 = (*a.ix)(1:r);
      ix2 = (*a.ix)(r+2:a.r+1);
      ix = [];
      grow,ix,ix1;
      grow,ix,ix2;
      */
      (*a.ix)(r+1:a.r) = (*a.ix)(r+2:a.r+1);
    }

    a.r = a.r-1n;
    
    /*
    if (r == a.r) {
      ix = (*a.ix)(1:a.r+1);
      ix(a.r+1) = ix(a.r);
    } else if (r == 1) {
      ix = (*a.ix)(2:a.r+1)
    } else if (r < a.r) {
      ix1 = (*a.ix)(1:r-1);
      ix2 = (*a.ix)(r+1:a.r+1);
      ix = [];
      grow,ix,ix1;
      grow,ix,ix2;
    }
    */
    
    /*

    Example 1: row 3 empty
    ----------------------

    #  [1] [2] [3] [4] [5] [6]
    ix  0   5   7   7   12  15  ...
           /   /   /   /   /  
    row   1   2   3   4   5
    nel   5   2   0   5   3

    After operation:

    #  [1] [2] [3] [4] [5]
    ix  0   5   7   12  15  ...
           /   /   /   /  
    row   1   2   3   4
    nel   5   2   5   3

    What happened: ix = [ ix(1:r-1) , ix(r+1:) ]

    Example 2: last row empty
    -------------------------

    ix  0   5   7   9   12  12  ...
           /   /   /   /   /  
    row   1   2   3   4   5
    nel   5   2   2   3   0

    After operation:

    ix  0   5   7   9  12  ...
           /   /   /   /  
    row   1   2   3   4
    nel   5   2   2   3
    
    */
    
  } else {

    if (r == a.r) {
      //write,"case 1";
      ix = (*a.ix)(1:r);
      jx = (*a.jx)(1:a.n-nel);
      xn = (*a.xn)(1:a.n-nel);
    } else if (r == 1) {
      //write,"case 2";
      ix = (*a.ix)(2:a.r+1)-nel;
      jx = (*a.jx)((*a.ix)(2)+1:(*a.ix)(a.r+1));
      xn = (*a.xn)((*a.ix)(2)+1:(*a.ix)(a.r+1));
    } else {
      //write,"case 3";
      jx1 = (*a.jx)((*a.ix)(1)+1:(*a.ix)(r));
      jx2 = (*a.jx)((*a.ix)(r+1)+1:);
      xn1 = (*a.xn)((*a.ix)(1)+1:(*a.ix)(r));
      xn2 = (*a.xn)((*a.ix)(r+1)+1:);
      jx = xn = [];
      grow,jx,jx1;
      grow,jx,jx2;
      grow,xn,xn1;
      grow,xn,xn2;
      ix1 = (*a.ix)(1:r-1);
      ix2 = (*a.ix)(r+1:a.r+1)-nel;
      ix = [];
      grow,ix,ix1;
      grow,ix,ix2;
    }
    a.n = a.n-nel;
    a.ix = &(ix);
    a.jx = &(jx);
    a.xn = &(xn);
    a.r = a.r-1n;
  }
  //return a;
}

//==================================================================
func spcon(&a,b,diag=,ruo=)
  /* DOCUMENT spcon(&a,b,diag=,ruo=)
     Concatennate two RCO matrices row-wise (default), diagonally
     (diag=1), or two RUO matrices diagonally (ruo=1).
  */
{
  (*a.jx)(a.n+1:a.n+b.n) = (*b.jx)(1:b.n);
  if (diag == 1 && ruo != 1) (*a.jx)(a.n+1:a.n+b.n) += a.c-1;
  if (ruo == 1) (*a.jx)(a.n+1:a.n+b.n) += a.r-1;
  (*a.xn)(a.n+1:a.n+b.n) = (*b.xn)(1:b.n);
  if (ruo == 1) (*a.xd)(a.r+1:a.r+b.r) = (*b.xd)(1:b.r);
  if (ruo == 1) (*a.ix)(a.r+1:a.r+b.r-1) = (*b.ix)(2:b.r)+(*a.ix)(a.r);
  else (*a.ix)(a.r+2:a.r+b.r+1) = (*b.ix)(2:b.r+1)+(*a.ix)(a.r+1);
  a.r += b.r;
  a.n += b.n;
  if (diag == 1 && ruo != 1) a.c += b.c;
  else if (diag != 1 && ruo != 1) a.c = max([a.c,b.c]);
  // it is possible for two RCO matrices of different widths to
  // be concatennated row-wise - the narrower one is automatically
  // padded with zeroes to the right.
}

//==================================================================
func rcocc(&a,b)
  /* DOCUMENT rcocc(&a,b)
     Concatennate two RCO matrices row-wise
  */

{
  bix = (*b.ix)(2:b.r+1);
  bjx = (*b.jx)(1:b.n);
  bxn = (*b.xn)(1:b.n);
  if (sizeof(*a.ix) == 0 || a == []) {
    a = rco();
    a.ix = &([0n]);
    aix = 0n;
    ajx = axn = [];
  } else {
    aix = (*a.ix)(1:a.r+1);
    ajx = (*a.jx)(1:a.n);
    axn = (*a.xn)(1:a.n);
  }
  grow,aix,bix+(*a.ix)(a.r+1);
  grow,ajx,bjx
  grow,axn,bxn;
  if (a.c == 0 && b.c == 0) {
    error,"matrix width unknown (specify #columns)";
  } else {
    a.c = b.c;
  }
  a.r += b.r;
  a.n += b.n;
  a.ix = &aix;
  a.jx = &ajx;
  a.xn = &axn;
}

//==================================================================
func ruocc(&a,b)
  /* DOCUMENT ruocc(&a,b)
     Concatennate two RUO matrices block-diagonally
  */
{
  bjx = (*b.jx)(1:b.n);
  bxn = (*b.xn)(1:b.n);
  bxd = (*b.xd)(1:b.r);
  if (sizeof(*a.ix) == 0) {
    bix = (*b.ix)(2:b.r+1);
    a.ix = &([0n]);
    aix = 0n;
    ajx = axn = axd = [];
    grow,aix,bix+a.n;
  } else {
    bix = (*b.ix)(1:b.r);
    aix = (*a.ix)(1:a.r);
    ajx = (*a.jx)(1:a.n);
    axn = (*a.xn)(1:a.n);
    axd = (*a.xd)(1:a.r);
    grow,aix,bix+a.n;
  }
  grow,ajx,bjx+a.r;
  grow,axn,bxn;
  grow,axd,bxd;
  a.r += b.r;
  a.n += b.n;
  a.ix = &aix;
  a.jx = &ajx;
  a.xn = &axn;
  a.xd = &axd;
}

//==================================================================
func ruo2rco(a)
  /* DOCUMENT func ruo2rco(a)
     Converts an RUO matrix into RCO. Calls both rcotr and rcoadd.
     This is not very efficient, and should be used sparingly.
  */
{
  xd = (*a.xd)(1:a.r);
  d = rco();
  d.c = a.r;
  d.r = a.r;
  d.n = a.r;
  d.xn = &xd;
  dix = int(span(0,a.r,a.r+1));
  d.ix = &dix;
  djx = int(span(0,a.r-1,a.r));
  d.jx = &djx;
  u = rco();
  uix = *a.ix;
  ujx = *a.jx;
  uxn = *a.xn;
  u.jx = &ujx;
  u.xn = &uxn;
  if (dimsof(uix)(0) == a.r) {
    grow,uix,[(*a.ix)(a.r),(*a.ix)(a.r)];
  } else if (dimsof(uix)(0) > a.r) {
    uix = uix(1:a.r);
    grow,uix,[(*a.ix)(a.r),(*a.ix)(a.r)];
  } else {
    error,"Something wrong - vector IX has too few elements";
  } 
  u.ix = &uix;
  u.r = a.r;
  u.c = u.r;
  u.n = a.n;
  tmp = rcoadd(u,d,ur=a.r+10n,un=int(a.n*2+a.r*2));
  l = rcotr(u);   
  b = rcoadd(tmp,l,ur=a.r+10n,un=int(a.n*2+a.r*2));
  
  return b;
}
//==================================================================
func intop(dims)
  /* DOCUMENT int(dims)
     Interpolating operators implemented as sparse matrices.
     "dims" is a dimension list with 2^i entries., example:
     test = binop([64,32,16]) will return a 2-element pointer
     array, each pointing to a 3-element array of RCO structures.
     SEE ALSO:
  */
{
  nd = numberof(dims);
  a = array(rco,nd);
  for (i=1;i<=nd;i++) {
    ur = dims(i)*dims(i)+1;
    un = int(ur*2.25);
    a(i).ix = &(array(int,ur));
    a(i).jx = &(array(int,un));
    a(i).xn = &(array(float,un));
  }
  v1 = array(0.5f,2);
  v2 = array(0.25f,4);

  /* // debugging the truth table...
  for (i=1;i<=dims(k);i++) {
    for (j=1;j<=dims(k);j++) {
      //ind = (i-1)*dims(k)+j-1;
      ind = (int((i+1)/2)-1)*dims(k)/2+int((j+1)/2)-1;
      i2 = i%2; j2 = j%2;
      write,format="%2d %2d %2d %d %2d %d %d %d %4d\n",j,i,j2,i2,\
        ((1-j2) && i2),((1-i2) && j2),((1-i2) && (1-j2)),(i2 && j2),ind;
    }
  }
  */

  for (k=1;k<=nd;k++) {
    ii = 1;
    n = 1;
    d1 = dims(k)/2;
    d2 = (dims(k)/2)^2;
    for (i=1;i<=dims(k);i++) {
      for (j=1;j<=dims(k);j++) {
        //        ind = (i-1)*dims(k)+j-1;
        ind = int((int((i+1)/2)-1)*d1+int((j+1)/2)-1);
        i2 = i%2; j2 = j%2;
        if ((1-j2) && i2) {             // x-interpolation
          (*a(k).xn)(n:n+1) = v1;
          inds = [ind,ind+1];
          if (j == dims(k)) inds(2) -= d1;
          (*a(k).jx)(n:n+1) = inds;
          (*a(k).ix)(ii+1) = (*a(k).ix)(ii)+2;
          n += 2;
        } else if ((1-i2) && j2) {      // y-interpolation
          (*a(k).xn)(n:n+1) = v1;
          inds = [ind,ind+d1];
          if (i == dims(k)) inds(2) -= d2;
          (*a(k).jx)(n:n+1) = inds;
          (*a(k).ix)(ii+1) = (*a(k).ix)(ii)+2;
          n += 2;
        } else if ((1-i2) && (1-j2)) {    // xy-interpolation
          (*a(k).xn)(n:n+3) = v2;
          inds = [ind,ind+1,ind+d1,ind+d1+1];
          if (j == dims(k)) {
            inds(2) -= d1;
            inds(4) -= d1;
              }
          if (i == dims(k)) {
            inds(3) -= d2;
            inds(4) -= d2;
          }
          (*a(k).jx)(n:n+3) = inds;
          (*a(k).ix)(ii+1) = (*a(k).ix)(ii)+4;
          n += 4;
        } else if (i2 && j2) {        //existing point
          (*a(k).xn)(n) = 1.f;
          (*a(k).jx)(n) = ind;
          (*a(k).ix)(ii+1) = (*a(k).ix)(ii)+1;
          n += 1;
        }
        ii++;
      }
    }
    a(k).r = int(dims(k)^2);
    a(k).c = int(dims(k)^2/4);
    a(k).n = int(n-1);
  }
  b = array(rco,nd);
  for (k=1;k<=nd;k++) {
    b(k) = rcotr(a(k));
  }
  return [&a,&b];
}
//==================================================================
