/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "swindow/SFontTTF.h"

#include "stoolkit/SString.h"
#include "stoolkit/SExcept.h"
/*
 * ntohl 
 */
#ifndef USE_WINAPI
#include <netinet/in.h>
#else
#include <winsock.h>
#endif

#include <stdio.h>
#include <ctype.h>

#define BYTE unsigned char
#define CHAR signed char
#define USHORT unsigned short
#define SHORT signed short
#define ULONG unsigned int
#define LONG signed int
#define FWORD SHORT
#define UFWORD USHORT

/* OTF */
static const SString SS_TB_GSUB("GSUB");

/*----------------------------------------------------------------------------
 * Open Type Font handling routines
 *--------------------------------------------------------------------------*/
typedef struct _OTF_Feature
{
  USHORT offset;
  USHORT count;
  USHORT record[1];
} OTF_Feature;

typedef struct _OTF_FeatureRecord
{
  char    tag[4];
  USHORT  offset;
} OTF_FeatureRecord;

typedef struct _OTF_FeatureList
{
  USHORT count;
  OTF_FeatureRecord record[1];
} OTF_FeatureList;

typedef struct _OTF_Lookup
{
  USHORT type;
  USHORT flag;
  USHORT count;
  USHORT subtable[1];
} OTF_Lookup;

typedef struct _OTF_LookupList
{
  USHORT count;
  USHORT record[1];
} OTF_LookupList;

typedef struct _OTF_RangeRecord
{
  USHORT start;
  USHORT end;
  USHORT index;
} OTF_RangeRecord;

typedef struct _OTF_CoverageFormat2
{
  USHORT format;
  USHORT count;
  OTF_RangeRecord record[1];
} OTF_CoverageFormat2;

typedef struct _OTF_CoverageFormat1
{
  USHORT format;
  USHORT count;
  USHORT id[1]; /* numerical order */
} OTF_CoverageFormat1;

/*
 * IF I have more time I will debug this 
 */
typedef struct _OTF_LangSys
{
  USHORT  lookupOrder; /* 0 for 1.0 */
  USHORT  reqFeatureIndex;
  USHORT  featureCount;
  USHORT  featureIndex[1]; /* or more  - index in feature */
} OTF_LangSys;

typedef struct _OTF_LangSysRecord
{
  char        tag[4];
  USHORT      offsetFromScript;
} OTF_LangSysRecord;

typedef struct  _OTF_Script
{
  USHORT             defaultLangSys; /* may be 0 */
  USHORT             langSysCount;
  OTF_LangSysRecord  langSysRecord[1]; /* or more */
} OTF_Script;

typedef struct _OTF_ScriptRecord
{
  char               tag[4];
  USHORT             offset;
} OTF_ScriptRecord;

typedef struct _OTF_ScriptList 
{
  USHORT              scriptCount;
  OTF_ScriptRecord    records[1];
} OTF_ScriptList;

/* Why1? I dont know - MS terminology...*/
typedef struct _OTF_LigatureSusbstFormat1
{
  USHORT format;
  USHORT coverage;
  USHORT count;
  USHORT offset[1];
} OTF_LigatureSusbstFormat1;

typedef struct _OTF_Ligature
{
  USHORT glyph;
  USHORT count;
  USHORT component[1];
} OTF_Ligature;

typedef struct _OTF_LigatureSet
{
  USHORT count;
  USHORT offset[1];
} OTF_LigatureSet;

typedef struct gsub_head {
	ULONG   version;
	USHORT 	scriptList;
	USHORT	featureList;
	USHORT	lookupList;
} GSUB_HEAD;


static SS_GlyphIndex getOTFFeature (OTF_Feature* feat, 
  OTF_LookupList* lookuplist, unsigned int substtype,
  const SString& fontname, const SS_GlyphIndex* chars,
   unsigned int length, SS_GlyphIndex* out);

static OTF_LangSys* getNextOTFLanguageSystem (const SString fontname, 
  GSUB_HEAD* gsubh, const SString& _script, unsigned int* from);

/**
 * return true if this font can do OTF ligatures
 */
bool
SFontTTF::hasOTFLigatures()
{
  return (tables[SS_TB_GSUB] != 0);
}

SString debugTag ;
/**
 * Go thrgough the GSUB table and get the ligature.
 * GSUB table may exists only for an Open Type Font.
 * Currently we just get a ligature substitution. We could get 
 * shape substitution too - with type 1.
 * param featurelist is a comma separated list of features.
 * if some featrue needs to be skipped it should start with '!'
 * @return 0 on fail.
 * Comment: probably they could not have designed a more convoluted
 *  format for OTF ligatures....
 */
unsigned int 
SFontTTF::getOTFLigature (const char* _script, const char* _featurelist,
    const SS_GlyphIndex* chars, unsigned int liglen, SS_GlyphIndex* out)
{
  *out = 0;
  GSUB_HEAD* gsubh = (GSUB_HEAD*) tables[SS_TB_GSUB];
  if (ntohl (gsubh->version) != 0x00010000)
  {
//    fprintf (stderr, "SFontTTF: BAD GSUB Table in TTF file [%*.*s].\n", 
//        SSARGS(name));
    return 0;
  }
  bool nonfeature = false;
  SBinHashtable<int> features;
  if (_featurelist != 0 && _featurelist[0] != 0 && _featurelist[1] != 0)
  {
    SString f = SString(_featurelist);
    if (f[0] == '!')
    {
      nonfeature = true;
      f.remove (0);
    }
    SStringVector v(f);
    for (unsigned int i=0; i<v.size(); i++)
    {
      SString s(v[i]);
      if (s.size()==4)
      {
        features.put (s, 1);
      }
    }
  }

  int ofeat = ntohs(gsubh->featureList);
  OTF_FeatureList*  featureList = (OTF_FeatureList*) ((BYTE*)gsubh + ofeat);
  USHORT fcount = ntohs (featureList->count);

  int olookup = ntohs(gsubh->lookupList);
  OTF_LookupList*  lookupList = (OTF_LookupList*) ((BYTE*)gsubh + olookup);
  //USHORT lcount = ntohs (lookupList->count);

  /*  find feature 4 - ligature */
  /* 'init' 'medi' 'fina' 'liga' will be supported by yudit */
  for (unsigned int i=0; i< fcount; i++)
  {
     SString tag (featureList->record[i].tag, 4);
     debugTag = tag;
     USHORT loffset = ntohs (featureList->record[i].offset);
     /* got or not omitted */
     if (_featurelist)
     {
       if (nonfeature)
       {
          if (features.get(tag)) continue;
       }
       else
       {
         if (features.get(tag))
         {
           OTF_Feature* feat = (OTF_Feature*) ((char*)featureList+loffset);
           unsigned int ind = getOTFFeature (feat, lookupList, 
               4, name, chars, liglen, out);
           if (ind) return ind;
         }
       }
     }
     /* process ligatures anyway */
     if (tag==SString("liga"))
     {
        OTF_Feature* feat = (OTF_Feature*) ((char*)featureList+loffset);
        unsigned int ind = getOTFFeature (feat, lookupList, 
          4, name, chars, liglen, out);
        if (ind) return ind;
     }
     //fprintf (stderr, "feature[%u]=%*.*s\n", i, SSARGS(tag));
  }
  unsigned int next = 0; /* iterator - just in case we have same lang twice */
  SString script (_script);
  OTF_LangSys* lsys = 0;
  while ((lsys = getNextOTFLanguageSystem (name, gsubh, script, &next))!=0)
  {
    USHORT fcount = ntohs (lsys->featureCount);
    /* index lookupList through lsys->featureIndex */
    USHORT lorder = ntohs (lsys->lookupOrder);
    if (lorder != 0)
    {
      static bool warned = false;
      if (!warned)
      {
        fprintf (stderr, 
          "LanguageSystem lookup order %u not supported",
          (unsigned int) lorder);
        fprintf (stderr, " in %*.*s.\n", SSARGS(name));
        warned = true;
      }
      continue;
    }
    for (unsigned int i=0; i< fcount; i++)
    {
      unsigned int index = ntohs (lsys->featureIndex[i]) ;
      SString tag (featureList->record[index].tag, 4);
//fprintf (stderr, "feature[%u]=%*.*s\n", i, SSARGS(tag));
      if (_featurelist)
      {
        if (nonfeature)
        {
          if (features.get(tag)) continue;
        }
        else
        {
          if (!features.get(tag)) continue;
        }
      }
      //fprintf (stderr, "lang feature[%u]=%*.*s\n", i, SSARGS(tag));

      USHORT loffset = ntohs (featureList->record[index].offset);
      /* tags will have mistic ligature names and stuff like that - 
         dont check */
      OTF_Feature* feat = (OTF_Feature*) ((char*)featureList+loffset);
      unsigned int ind = getOTFFeature (feat, lookupList, 
         4, name, chars, liglen, out);
      if (ind) return ind;
    }
  }
  return 0;
}

/**
 * @param feat is the feature to go through
 * @param lookuplist is the list of lookups.
 * @param substtype is 4 for ligature substituion.
 * @param name is the fontname - used in error printouts.
 * @param chars is the glyphindeces  in a ligature
 * @param liglen is the glyphindeces count
 * @return glyph index or 0
 */
static SS_GlyphIndex
getOTFFeature (OTF_Feature* feat, OTF_LookupList* lookupList,
  unsigned int substtype, const SString& name, 
  const SS_GlyphIndex* chars, unsigned int liglen, SS_GlyphIndex* outchar)
{
  USHORT nfcount = ntohs (feat->count);
  for (unsigned int j=0; j<nfcount; j++)
  {
    USHORT rec = ntohs (feat->record[j]);
    USHORT lrec = ntohs (lookupList->record[rec]);
    OTF_Lookup * ltable = (OTF_Lookup*)((char*)lookupList + lrec); 
    USHORT type = ntohs (ltable->type);
    USHORT ltcount = ntohs (ltable->count);
    USHORT flag = ntohs (ltable->flag);
    /*
     * Substitution 
     * TODO: take flag into consideration
     */
    if (type == substtype && (flag & 0x0e)==0) /* ignore masks 'ignored' */
    //if (type == substtype) /* ignore masks 'ignored' */
    {
      for (unsigned int k=0; k<ltcount; k++)
      {
        USHORT offset = ntohs (ltable->subtable[k]);
        OTF_LigatureSusbstFormat1 *lformat =  
            (OTF_LigatureSusbstFormat1*) ((char*)ltable +  offset);
        USHORT cformat = ntohs (lformat->format);
        if (cformat != 1) 
        {
          fprintf (stderr, "Broken font 'liga' '%*.*s'\n", SSARGS(name));
          return 0;
        }
        USHORT coffset = ntohs (lformat->coverage);
        /* get the coverage */
        USHORT ccount = ntohs (lformat->count);
        USHORT coverageIndex = ccount; 
        /* This should be less than ccount after coverage is found */

        /* go through coverage tables - they should be sorted */
        USHORT format = ntohs (*(USHORT*)((char*)lformat +  coffset));
        if (format==1)
        {
          OTF_CoverageFormat1* format1 = (OTF_CoverageFormat1*) 
                 ((char*)lformat +  coffset);
          USHORT gcount = ntohs (format1->count);
          /* Do binary search here   set coverageIndex */
          unsigned int    top, bottom, mid;
          top = gcount; bottom = 0;
          unsigned int c0 = chars[0];
          USHORT id = c0 +1;
          while (top > bottom)
          {
            mid = (top+bottom)/2;
            id = ntohs (format1->id[mid]);
            if (c0 == id) { top = mid; break; }
            if (c0 < id) { top = mid; continue; }
            bottom = mid + 1;
          }
          if (top < gcount && c0 == id) coverageIndex = top;
          /* I was looging for a glyph and I could not find it... eyelash.*/
#ifdef LOOKING_FOR_EYELASH
          USHORT glyphtofind = 0x6B59; //findGlyph(0xE97D);
          for (unsigned int i=0; i<gcount; i++)
          {
            id = ntohs (format1->id[i]);
            /* dont  loop - get coverageIndex */
            USHORT cinoffset = ntohs (lformat->offset[i]);
            OTF_LigatureSet *lset = (OTF_LigatureSet *)  
            ((BYTE*) lformat + cinoffset);
            USHORT count = ntohs (lset->count);
            for (unsigned int m=0; m<count; m++)
            {
              USHORT offset = ntohs (lset->offset[m]);
              OTF_Ligature* lig = (OTF_Ligature*) 
              ((BYTE*) lset + offset);
            
              USHORT glyph = ntohs (lig->glyph);
              if (glyph != glyphtofind) continue;

              USHORT compcount = ntohs (lig->count);
              fprintf (stderr, "Found glyph %04X in '%*.*s' %04X", 
                  glyphtofind, SSARGS(debugTag), id);
              for (unsigned int n=0; n+1<compcount; n++)
              {
                 USHORT component = ntohs (lig->component[n]);
                 fprintf (stderr, " %04X",  component);
              }
              fprintf (stderr, "\n");
            }
          }
#endif // LOOKING_FOR_EYELASH
        }
        else if (format==2)
        {
          OTF_CoverageFormat2* format2 = (OTF_CoverageFormat2*) 
             ((char*)lformat +  coffset);
          /* I could not find any format2 fonts-this part is untested */
          USHORT gcount = ntohs (format2->count);
          /* Do binary search here. */
          for (unsigned int l=0; l<gcount; l++)
          {
            USHORT index = ntohs (format2->record[l].index);
            USHORT start = ntohs (format2->record[l].start);
            USHORT end = ntohs (format2->record[l].end);
            if (chars[0] >= start && chars[0] <= end)
            {
              coverageIndex = index + chars[0] - start;
              break;
            }
          }
        }
        if (ccount == coverageIndex)
        {
           continue; /* no coverage */
        }
        /* dont  loop - get coverageIndex */
        USHORT cinoffset = ntohs (lformat->offset[coverageIndex]);
        OTF_LigatureSet *lset = (OTF_LigatureSet *)  
         ((BYTE*) lformat + cinoffset);
        USHORT count = ntohs (lset->count);
        for (unsigned int m=0; m<count; m++)
        {
          USHORT offset = ntohs (lset->offset[m]);
          OTF_Ligature* lig = (OTF_Ligature*) 
            ((BYTE*) lset + offset);
          USHORT compcount = ntohs (lig->count);
          if (compcount > liglen)
          {
            continue;
          }
          bool found = true;
          for (unsigned int n=0; n+1<compcount; n++)
          {
             USHORT component = ntohs (lig->component[n]);
             if (chars[n+1] != component) found = false;
          }
          if (found)
          {
             USHORT glyph = ntohs (lig->glyph);
             *outchar = (SS_GlyphIndex) glyph;
             return (unsigned int) compcount;
          }
        }
      }
    }
    /* We checked for type 4 - above. TODO: otehr types, like non-liga. */
  }
  return 0;
}

/**
 * Iterate to the next language system.
 * @param gsubh is the GSUB_HEAD header
 * @param _script is the OTF standard name of script.
 *  like "taml"
 * @patam from is the input output iterator counter 
 *       - first it is needed to be set to zero.
 */
static OTF_LangSys*
getNextOTFLanguageSystem (const SString fontname, GSUB_HEAD* gsubh, 
  const SString& script, unsigned int* from)
{
  /* Look for script. */
  int offset = ntohs(gsubh->scriptList);
  OTF_ScriptList*  scriptList = (OTF_ScriptList*) ((BYTE*)gsubh + offset);
  USHORT numscripts = ntohs (scriptList->scriptCount);
  
  for (unsigned int i=*from; i<numscripts; i++)
  {
    SString tag (scriptList->records[i].tag);
    USHORT sl = ntohs (scriptList->records[i].offset);
    /* we found the scipt */
    if (tag==script)
    {
      OTF_Script* stable = (OTF_Script*) ((char*)scriptList + sl);
      USHORT defaultLangsys = ntohs (stable->defaultLangSys);
      USHORT langSysCount = ntohs (stable->langSysCount);
      OTF_LangSysRecord* langsysrec = 0;
      OTF_LangSys *lsys = 0 ;
      /* attention count includes default */
      if (defaultLangsys == 0 && langSysCount > 0)
      {
        langsysrec = &stable->langSysRecord[0];
      }
      else if (langSysCount > 0 && defaultLangsys != 0)
      {
        langsysrec = (OTF_LangSysRecord*) ((char*) stable + defaultLangsys);
      }
      /* Locate the langsys table */
      if (langsysrec)
      {
        SString ltag (langsysrec->tag, 4);
        USHORT lsysoffset = ntohs (langsysrec->offsetFromScript);
        lsys =  (OTF_LangSys*) ((char*) stable + lsysoffset);
      } 
      else  if (defaultLangsys != 0) 
      {
        lsys = (OTF_LangSys*) ((char*) stable + defaultLangsys) ;
      }
      USHORT lorder = ntohs (lsys->lookupOrder);
      if (lorder != 0)
      {
        static bool fixed = false;
        /* get a langsys record taht works */
        for (unsigned int sg = 0; sg<langSysCount; sg++)
        {
          OTF_LangSysRecord* lr = &stable->langSysRecord[sg];
          USHORT lsysoffset = ntohs (lr->offsetFromScript);
          OTF_LangSys* ls =  (OTF_LangSys*) ((char*) stable + lsysoffset);
          USHORT lorder = ntohs (ls->lookupOrder);
          if (lorder == 0)
          {
            lsys = ls;
            if (!fixed)
            { 
              fprintf (stderr, 
               "Fixed broken default langsys table in  %*.*s. (%4.4s %u/%u)\n", 
                SSARGS(fontname), lr->tag, (unsigned int) sg, 
                (unsigned int) langSysCount);
            }
          }
          fixed = true;
        }
      }
      *from = i+1;
      return lsys;
    }
  }
  *from = numscripts;
  return 0;
}
