/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include "swindow/sx11/SXInputMethod.h"
#include "swindow/sx11/SX11Color.h"
#include "stoolkit/STypes.h"
#include "stoolkit/SEncoder.h"


/**
 * This is out X11 input method. Currently only kinput2 is here, so
 * we have one class only.
 */
SXInputMethod::SXInputMethod (SX11Impl* _impl, long _id)
{
  impl = _impl;
  id = _id;
  name = "";
  kproperty = None;
  conversionOwner = None;
}

SXInputMethod::~SXInputMethod()
{
  impl->removeXEventHandler (id, PropertyNotify);
  impl->removeXEventHandler (id, ClientMessage);
}

/**
 * Start kinput2 conversion.
 * @param _name is the conversion name - only "_JAPANESE_CONVERSION" is OK now.
 * @param properties contains the properties for input.
 */
bool
SXInputMethod::start (const SString& _name, const SProperties& properties)
{
  if (_name != ATOM_KINPUT2) return false;


  name = _name;
  conversionOwner = getOwner (toAtom (name));
  if (conversionOwner == None)
  { 
    //name = ""; 
    return false;
  }

  /**
   * Check if we can use this '_JAPANESE_CONVERSION' thingy.
   */
  
  Atom            expectedAttribueType = toAtom ("_CONVERSION_ATTRIBUTE_TYPE");

  Atom            attributeType; 
  int             format;
  unsigned long   nitems;
  unsigned long   bytesafter;
  unsigned long   *data;

  XGetWindowProperty (
    impl->display, conversionOwner, toAtom ("_CONVERSION_PROFILE"),
    0L, 100L, False, toAtom ("_CONVERSION_ATTRIBUTE_TYPE"),
    &attributeType, &format, &nitems, &bytesafter, (unsigned char **)&data);

  if (data == 0) return false;

  if (format != 32 || attributeType != expectedAttribueType)
  {
    XFree ((char*) data);
    name = "";
    return false;
  }

  bool isOK = false;

  for (unsigned i=0; i<nitems; i++)
  {
    int code = CODE_OF_ATTR (data[i]);
    int len = LENGTH_OF_ATTR (data[i]);
    if (len+i > nitems) break;

    switch (code)
    {
    case CONVPROF_PROTOCOL_VERSION:
      if (data[i+1] == toAtom ("PROTOCOL-2.0"))
      {
        isOK = true;
      }
    /* TODO: */
    case CONVPROF_SUPPORTED_STYLES:
      break;
    }
    i+=len;
  }

  XFree (data);
  if (!isOK)
  {
    conversionOwner = None;
    name = "";
    return false;
  }

  /* set attributes */
  if (!setKinputAttributes (properties))
  {
    conversionOwner = None;
    name = "";
    return false;
  }
      
  impl->setXEventHandler (id, ClientMessage, this);
  return sendEvent ("CONVERSION_REQUEST", conversionOwner,
     toAtom (name), id, toAtom ("COMPOUND_TEXT"), toAtom (name), 
     toAtom ("CONVERSION_ATTRIBUTE"));
}

/**
 * Stop the conversion.
 */
void
SXInputMethod::stop ()
{
  if (name == "") return;
  if (conversionOwner == None) return;


  Window owner = getOwner (toAtom (name));
  if (owner == None || conversionOwner != owner)
  {
    name = "";
    conversionOwner = None;
    impl->removeXEventHandler (id, PropertyNotify);
    impl->removeXEventHandler (id, ClientMessage);
    kproperty = None;
    return;
  }

  Atom an = toAtom(name);
  name = "";
  sendEvent ("CONVERSION_END_REQUEST", conversionOwner, an, id);
  conversionOwner = None; name = "";

  impl->removeXEventHandler (id, PropertyNotify);
  impl->removeXEventHandler (id, ClientMessage);
  kproperty = None;
}

/**
 * Chech if it is running and return name or null
 */
const SString&
SXInputMethod::getName()
{
  if (name == "") return name;
  if (conversionOwner == None)
  {
    name = "";
    impl->removeXEventHandler (id, PropertyNotify);
    impl->removeXEventHandler (id, ClientMessage);
    kproperty = None;
    return name;
  }

  Window owner = getOwner (toAtom (name));
  if (owner == None || conversionOwner != owner)
  {
    name = "";
    conversionOwner = None;
    impl->removeXEventHandler (id, PropertyNotify);
    impl->removeXEventHandler (id, ClientMessage);
    kproperty = None;
  }
  return name;
}

void
SXInputMethod::setProperties ( const SProperties& props)
{
  if (name == "") return;
  if (conversionOwner == None) return;

  Window owner = getOwner (toAtom (name));
  if (owner == None || conversionOwner != owner 
   || !setKinputAttributes (props))
  {
    name = "";
    conversionOwner = None;
    impl->removeXEventHandler (id, PropertyNotify);
    impl->removeXEventHandler (id, ClientMessage);
    kproperty = None;
    return;
  }

  long attr = CONV_ATTR(CONVATTR_INDIRECT, 1);
  sendEvent ("CONVERSION_ATTRIBUTE_NOTIFY", conversionOwner,
     toAtom (name), id, attr, toAtom ("CONVERSION_ATTRIBUTE")); 
  return;
}

/**
 * Handle event. return false if no more call is needed.
 * @return false if we can not process this.
 */
bool
SXInputMethod::handleEvent (const XEvent& event)
{
  if (conversionOwner == None) return false;

  /* Check if we got the right event. */
  switch (event.xany.type)
  {
  case PropertyNotify:
    //fprintf (stderr,"SXInputMethod: Property Notify.\n");
    return handlePropertyNotify (event);

  /* response to conversion start */
  case ClientMessage:
    if (event.xclient.format != 32) return false;

    if (event.xclient.message_type == toAtom ("CONVERSION_END")
        && event.xclient.data.l[0] == (long) toAtom (name))
    {
      fprintf (stderr, "SXInputMethod: conversion ended.\n");
      name = "";
      conversionOwner = None;
      impl->removeXEventHandler (id, PropertyNotify);
      impl->removeXEventHandler (id, ClientMessage);
      kproperty = None;
      return true;
    }
    /* we take notify only */
    if (event.xclient.message_type != toAtom ("CONVERSION_NOTIFY")
      || (Atom) event.xclient.data.l[0] != toAtom (name))
    {
      name = "";
      conversionOwner = None;
      impl->removeXEventHandler (id, PropertyNotify);
      impl->removeXEventHandler (id, ClientMessage);
      kproperty = None;
      fprintf (stderr, "SXInputMethod: conversion request failed.\n");
      return false;
    }
    if (event.xclient.data.l[2] == None)
    {
      name = "";
      conversionOwner = None;
      impl->removeXEventHandler (id, PropertyNotify);
      impl->removeXEventHandler (id, ClientMessage);
      kproperty = None;
      fprintf (stderr,"SXInputMethod: request can not be performed.\n");
      return true;
    }
    /* Why do I need this ? */
    kproperty = (Atom) event.xclient.data.l[2];
    //fprintf (stderr,"SXInputMethod: request OK.\n");
    impl->setXEventHandler (id, PropertyNotify, this);
    return true;
  }
  /* remove this handler */
  return false;
}

/**
 * Handle property notify event. return false if no more call is needed.
 * @return true if we want to propagate this.
 */
bool
SXInputMethod::handlePropertyNotify (const XEvent& event)
{
  if (name.size() == 0) return true;
  if (conversionOwner == None) return true;
  if (kproperty == None)
  {
    return false;
  }
  if ((long) event.xproperty.window != id 
   || event.xproperty.atom != kproperty
   || event.xproperty.state != PropertyNewValue)
  {
    return false;
  }
  Atom            proptype;
  int             propformat;
  unsigned long   propsize;
  unsigned long   rest;
  unsigned char*  propvalue=0;
 
  /* Kinput just hang some property on our window. */
  XGetWindowProperty (impl->display, (Window)id, kproperty, 
    0L, 100000L, True, AnyPropertyType, 
    &proptype, &propformat, &propsize, &rest, &propvalue);

  /**
   * this happens if accumulated property change
   * appened, and we already have read the data.
   */
  if (proptype == None)
  {
    if (propvalue) XFree (propvalue);
    return true;
  }
  // Should not happen
  if (proptype != toAtom ("COMPOUND_TEXT"))
  {
    if (propvalue) XFree (propvalue);
    fprintf (stderr, "Expected COMPUND_TEXT\n");
    return true;
  }
  // Should not happen
  if (propformat != 8)
  {
    if (propvalue) XFree (propvalue);
    return true;
  }
  SString in ((char*)propvalue, propsize);
  if (propvalue) XFree (propvalue);

  SEncoder utf8Encoder;
  SEncoder ctextJP ("iso-2022-x11");
  if (!ctextJP.isOK())
  {
    fprintf (stderr, "could not load encoder for iso-2022-jp\n");
    return true;
  }
  impl->sendString (id, utf8Encoder.encode (ctextJP.decode (in)));
  return true;
}

/**
 * Changes the attributes on this window by changing the window properties
 * to whatever we have in properties.
 */
bool
SXInputMethod::setKinputAttributes (const SProperties& properties)
{
  /* pad with 4 zero bytes on top for buggy kinput on alpha */
  SString xprop;

  if (properties.get ("InputStyle")==0)
  {
    fprintf (stderr, "InputStyle is not present in properties.\n");
    return false;
  }
  int style = CONVARG_OVERTHESPOT;
  SString s = properties["InputStyle"];
  if (s=="root")
  {
    style = CONVARG_ROOTWINDOW;
  }
  else if (s =="off-the-spot")
  {
    style = CONVARG_OFFTHESPOT;
  }
  else if (s == "over-the-spot")
  {
    style = CONVARG_OVERTHESPOT;
  }
  else
  {
    fprintf (stderr, "InputStyle is bad - %*.*s.\n", SSARGS(s));
    return false;
  }
  xprop.append (SString ((long) CONV_ATTR(CONVATTR_INPUT_STYLE, 1)));
  xprop.append (SString((long) style));

  // CONVARG_NONE - not supported
  // CONVARG_SELECT_FOCUS_WINDOW - key events that happen
  //     during coversion and dont have SendEvent flags need to be
  //     ignored.
  // CONVARG_CREATE_INPUTONLY - create an invisible window in front of
  //     the real one - does not work with all window managers-
  //     click to type. - CURRENTLY YUDIT CAN DO ONLY THIS.
  xprop.append (SString ((long) CONV_ATTR(CONVATTR_EVENT_CAPTURE_METHOD, 1)));
  xprop.append (SString((long) CONVARG_SELECT_FOCUS_WINDOW));

  if (style == CONVARG_OVERTHESPOT 
     && properties.get ("InputSpot") && properties.get ("InputStyle"))
  {
    SString spotLocation = properties["InputSpot"];
    spotLocation.append ((char)0);
    int x, y;
    sscanf (spotLocation.array(), "%d,%d", &x, &y);
    xprop.append (SString ((long) CONV_ATTR(CONVATTR_SPOT_LOCATION, 1)));
    xprop.append (SString((long) (x << 16) | (y & 0xffff)));
  }

  if (properties.get ("LineSpacing"))
  {
    SString lsp = properties["LineSpacing"];
    lsp.append ((char)0);
    int spacing;
    sscanf (lsp.array(), "%d", &spacing);
    xprop.append (SString ((long) CONV_ATTR(CONVATTR_LINE_SPACING, 1)));
    xprop.append (SString((long)spacing));
  }

  if (properties.get ("Color"))
  {
    SString col = properties["Color"];
    col.append ((char)0);
    unsigned long bg, fg;
    sscanf (col.array(), "%lu,%lu", &bg, &fg);
    SX11Color xbg = SX11Color(impl, (SS_WORD32)bg);
    SX11Color xfg = SX11Color(impl, (SS_WORD32)fg);
    //fprintf (stderr, "Colors (fg): %u %u %u\n", (unsigned int) xfg.red,
     //  (unsigned int) xfg.green, (unsigned int) xfg.blue);

    xprop.append (SString ((long) CONV_ATTR(CONVATTR_COLOR, 2)));
    xprop.append (SString ((long)(xfg.getPixelValue())));
    xprop.append (SString ((long)(xbg.getPixelValue())));
  }

  Atom attributeAtom = toAtom ("CONVERSION_ATTRIBUTE");
  unsigned int psize = xprop.size();
  /* I have to deal here with bugs all over X11/Kinput on Alpha AXP */
  for (unsigned int i=0; i<psize; i++)
  {
    xprop.append (SString((long)0));
  }
  XChangeProperty (impl->display, (Window)id, 
     attributeAtom, attributeAtom, 
     32, PropModeReplace, 
     (unsigned char *)xprop.array(), 
     /* tricky len calculation for buggy kinput2 */
     (sizeof(long)/sizeof(SS_WORD32)) * psize  / sizeof(SS_WORD32));

  XFlush (impl->display);
  return true;
}

/**
 * Convert string to atom.
 * @param str is the string name of atom.
 */
Atom
SXInputMethod::toAtom (const SString& str)
{
  SString a=str;
  a.append ((char)0);
  return XInternAtom (impl->display, a.array(), False);
}

/**
 * Get the window that is the selection owner of atom.
 */
Window
SXInputMethod::getOwner (Atom atom)
{
  if (atom == None) return None;
  return XGetSelectionOwner (impl->display, atom);
}

/**
 * Send a client message to window. 
 * Mostly used to communicate with kinput2 Window.
 * TODO: This is the most dangerous part. If window disappears while 
 * getting the id and sending the event, X error handles needs to be implemented
 * to prevent app from exiting.
 * @param _type is the message type - atom string.
 * @param v contains all the elements in the message 
 * @return false if anything is wrong.
 */
bool
SXInputMethod::sendEvent (const SString& _type, Window window,
  long p0 ,long p1, long p2, long p3, long p4)
{
  if (window == None) return false;
  Atom type = toAtom (_type);
  if (type == None) return false;

  XEvent event;
  event.xclient.type = ClientMessage;
  event.xclient.window = window;
  event.xclient.display = impl->display;
  event.xclient.message_type = type; 
  event.xclient.format = 32;
  event.xclient.data.l[0] = p0;
  event.xclient.data.l[1] = p1;
  event.xclient.data.l[2] = p2;
  event.xclient.data.l[3] = p3;
  event.xclient.data.l[4] = p4;
  XSendEvent (impl->display, window,  False, NoEventMask, &event);
  return true;
}

