/* -*- c++ -*-_______________________________________________________________

        Zinf - Zinf Is Not FreeA*p (The Free MP3 Player)

        Portions Copyright (C) 1999 EMusic.com

        This program is free software; you can redistribute it and/or modify
        it under the terms of the GNU General Public License as published by
        the Free Software Foundation; either version 2 of the License, or
        (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program; if not, write to the Free Software
        Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

        $Id: metadatadb.h,v 1.2 2003/09/29 02:29:46 kgk Exp $
____________________________________________________________________________*/

#ifndef _METADATADB_H
#define _METADATADB_H 1

#include <stdint.h>

#include <string>
#include <map>
#include <vector>
#include <boost/format.hpp>


#include "errors.h"
#include "config.h"

/** 
 *   Metadata table for stream,playlist and file metadata.  Acts
 *   somewhat like a map for simple manipulation.
 *
 *   Metadata keys are always ASCII - ISO 8859-1 lower case.
 *   Values should be UTF8 for printable strings.
 *   FUTURE:  We might need to track bitmap etc.. and change
 *            the stored datatype to something more generic.
 *
 */
class Metadata {
public:
    typedef std::map<std::string,std::string> map_t;
    typedef map_t::iterator iterator;
    typedef map_t::const_iterator const_iterator;
    typedef map_t::size_type size_type;
    typedef map_t::key_type key_type;
    typedef map_t::value_type value_type;

    /**
     *  Standard tag constants for backward compatibility.
     *  New tags should just be some printable string
     */
    static const std::string kUrl; // url of item
    static const std::string kType; //  'F': file, 'S': stream, 'P':playlist
    static const std::string kTitle;
    static const std::string kArtist;
    static const std::string kAlbum;
    static const std::string kGenre;
    static const std::string kComment;
    static const std::string kTrack;
    static const std::string kYear;

    static const std::string kGuid;
    static const std::string kTime;
    static const std::string kSize;
    static const std::string kExt;

    static std::vector<std::string> kStandardTags;

    operator std::string();

public:
    Metadata() : m_pairs() {}

    void setTag(const std::string&tag, const std::string&val);
    void setTag(const std::string&tag, const int32_t&val);
    void getTag(const std::string&tag, std::string&val) const;
    void getTag(const std::string&tag, int32_t&val)const;

    //Optional 
    bool hasKey (const std::string&key) const {
        return m_pairs.count(key) > 0;
    }

    /** Initialize required tags .. used for searching required tags */
    void initialize_tags();

    //////////////////////////////////////
    // Deprecated -- Use k<name> instead

    void setArtist(const std::string &artist) { }
    const std::string& getArtist() const { return getStrVal(kArtist); }

    void setAlbum(const std::string &album) {  }
    const std::string& getAlbum() const { return getStrVal(kAlbum); }

    void setTitle(const std::string &title) {  }
    const std::string& getTitle() const { return getStrVal(kTitle); }

    void setComment(const std::string &comment) {  }
    const std::string& Comment() const { return getStrVal(kComment); }

    void setGenre(const std::string &genre) {  }
    const std::string& getGenre() const { return getStrVal(kGenre); }

    void setFormatExtension(const std::string &ext) {  }
    const std::string& getFormatExtension()const{ return getStrVal(kExt); }

    void setYear(uint32_t year) { }
    uint32_t getYear() const { return 0; }

    void setTrack(uint32_t track) { }
    uint32_t Track() const { return 0; }

    void setTime(uint32_t time){ }
    uint32_t Time() const { return 0; }

    void setSize(uint32_t bytes){ }
    uint32_t Size() const { return 0; }

    void setGUID(const std::string &guid) { }
    const std::string& GUID() const { return getStrVal(kGuid); }

    void setPlayCount(uint32_t playcount) {  }
    uint32_t PlayCount() const { return 0; }



    ///////////////////////////////
    // Priority functionality
    // Each element has a set prioity
    // which determines whether it may overwritten 
    // with by some other element when metadata is combined.
    // The entire structure has a default priority if
    // individual members are not set.
    void     setDefaultPrioirty(int32_t);
    void     setPriority(const std::string&key);
    int32_t  getPriority(const std::string&key);

    ////////////////////////////////
    // Map functionality
    std::string& operator[] (std::string k)  {return m_pairs[k];}
    iterator begin() { return m_pairs.begin();}
    iterator end() { return m_pairs.end();}
    iterator find(const key_type&k)  { return m_pairs.find(k);}
    void     erase(iterator pos) { m_pairs.erase(pos);}
    size_type size() { return m_pairs.size(); }

    std::pair<iterator,bool> insert(const map_t::value_type&t) 
        { return m_pairs.insert(t); }
    iterator insert(iterator hint, const map_t::value_type&t) 
        { return m_pairs.insert(hint, t); }
    void clear() { m_pairs.clear(); }

private:
    const std::string& getStrVal(const std::string& k) const;
    int getIntVal(std::string k) const;
    map_t m_pairs;
};


/** 
 *   MetadataDB base is a source of Metadata.  This is a
 *   base class for actual metadata implementations.
 */
class MetadataDB {
public:
    typedef std::string url_t;
    typedef std::pair<url_t,Metadata> element_t;
    typedef std::map<url_t, Metadata> map_t;
    typedef std::map<std::string, std::string> params_t;
    typedef std::vector<std::string> result_t;

//     typedef map_t::iterator iterator;
//     typedef map_t::const_iterator const_iterator;

public:
    MetadataDB();
    virtual ~MetadataDB();


    // Low level manipulations
    virtual bool add(const url_t& url, const Metadata&);
    virtual bool remove(const url_t& url);
    virtual bool contains(const url_t&);
    virtual bool getMetadata(const url_t&url, Metadata&m);
    virtual bool setMetadata(const url_t&url, const Metadata&m);

    /** 
     *  Query the database for a target with know constraints.
     *
     * @param target: Metadata tag field that i.e. artist,album,url, etc
     * @param params: A dictionary of contrained metadata fields
     * @param results:  A list of results
     */
    virtual Error query(const std::string& target, const params_t& params, 
                        result_t& results);

    virtual void commit();

    /** This database will save it's metadata */
    bool permitWrites(bool );

    /** Higher priority metadata is used in preference to lower  */
    void     setPriority(int spot) { m_priority = spot; }
    int32_t  getPriority() { return m_priority; }

    //iterator begin();
    //iterator end();

protected:
    void log(const std::string&msg);
    void log(boost::format &f);


    bool        m_writesOK;
    int32_t     m_priority;
};



/**-----------------------------------------------------------------
 * A MetadataCollection represents a group of accessible metadata.
 *------------------------------------------------------------------*/

class ZINF_EXPORT MetadataCollection
{
public:
    typedef std::string url_t;
    typedef std::map<std::string, std::string> params_t;
    typedef std::vector<std::string> result_t;

public:
    MetadataCollection(const std::string&name);
    ~MetadataCollection();

    bool add(const url_t&, const Metadata&);
    bool remove(const url_t&);
    bool contains(const url_t&url);
    /** 
     * Read the metadata of the url from all available
     * sources to this DB.  The metadata structure should 
     * be initialize to the required metadata tags
     */
    bool readMetadata(const url_t&, Metadata&);
    bool saveMetadata(const url_t&, Metadata&);

    Error query(const std::string& target, 
                const params_t& params, 
                result_t& results);

    void append(MetadataDB*);

    void commit();
private:
    typedef std::vector<MetadataDB*>  metapath_type;


    std::string m_name;
    metapath_type  m_path;
};


#endif /* _METADATADB_H */

/* arch-tag: ceb63127-cdf8-4e34-a453-707151b5e215 */
